/************************************************************************
* FILE:        hc_tclCfgMgr.cpp
* PROJECT:
* SW-COMPONENT:fc_heatctrl
*----------------------------------------------------------------------
*
* DESCRIPTION: common configuration manager class
*
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2014 Robert Bosch GmbH, Hildesheim
* HISTORY:
* Date      | Author             | Modification
* 24.06.2015| CM-AI/EPB2 Bernard | init
*
*************************************************************************/

/*******************************************************************************
| includes: system- and project- includes
|-----------------------------------------------------------------------------*/
// Basic OSAL includes
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define AHL_S_IMPORT_INTERFACE_GENERIC
#include "ahl_if.h"         // use Application Help Library

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

/*******************************************************************************
| set own module id
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| includes: needed interfaces from external components
|-----------------------------------------------------------------------------*/


/*******************************************************************************
| includes: internal and external interfaces from this component
|-----------------------------------------------------------------------------*/
// datapool access
#ifndef DP_S_IMPORT_INTERFACE_FI
#define DP_S_IMPORT_INTERFACE_FI
#include "dp_generic_if.h"
#include "dp_fc_heatctrl_if.h"
#endif //#ifndef DP_S_IMPORT_INTERFACE_FI

#include "hc_tclCfgMgr.h"
#include "hc_AppMain.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_HEATCTRL_APPLICATION
#include "trcGenProj/Header/hc_tclCfgMgr.cpp.trc.h"
#endif

/*******************************************************************************
| defines and macros (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| typedefs (scope: module-local)
|-----------------------------------------------------------------------------*/


/*******************************************************************************
| variable definition (scope: global)
|-----------------------------------------------------------------------------*/


/*******************************************************************************
| variable definition (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| function prototype (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| function implementation (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
*
* FUNCTION:    hc_tclCfgMgr()
*
* DESCRIPTION: constructor
*
* PARAMETER:   const hc_tclAppMain*
*
* RETURNVALUE: none
*
*******************************************************************************/
hc_tclCfgMgr::hc_tclCfgMgr(const hc_tclAppMain* poMainAppl)
: I_hc_tclCfgMgr(poMainAppl)
{
   ETG_TRACE_USR4(("hc_tclCfgMgr() entered."));

   // coverity_19125
   for(tU16 _u16Idx = 0; HC_C_U16_ARRAYELEMENTS(m_aenSensorIdx) > _u16Idx; _u16Idx++)
   {
	   m_aenSensorIdx[_u16Idx] = (tenKDSDataSensor)OSAL_C_S32_MAX;
   }
   for(tU16 _u16Idx = 0; HC_C_U16_ARRAYELEMENTS(m_au8KDSSensorEnable) > _u16Idx; _u16Idx++)
   {
	   m_au8KDSSensorEnable[_u16Idx] = OSAL_C_U8_MAX;
   }
   m_u8KDSFANEnable = OSAL_C_U8_MAX;

   // loading of all configuration data
   vLoadHWConfig();
   vMirrorPDDTHM();
}

/*******************************************************************************
*
* FUNCTION:    ~hc_tclCfgMgr()
*
* DESCRIPTION: destructor
*
* PARAMETER:   none
*
* RETURNVALUE: none
*
*******************************************************************************/
hc_tclCfgMgr::~hc_tclCfgMgr()
{
   ETG_TRACE_USR4(("~hc_tclCfgMgr() entered."));
}

/*************************************************************************
*
* FUNCTION:    vHandleMessage
*
* DESCRIPTION: handle internal message
*
* PARAMETER:   hc_tclBaseIf::TMsg* pMsg : reference of received internal message
*
* RETURNVALUE: none
*
*************************************************************************/
tVoid hc_tclCfgMgr::vHandleMessage(hc_tclBaseIf::TMsg* pMsg)
{
	HC_NULL_POINTER_CHECK(pMsg);
	ETG_TRACE_USR4(("vHandleMessage() entered %u -> data: %d.", ETG_ENUM(HC_INT_MSG , (tU32)pMsg->eCmd), pMsg->u.u32Data));
	switch (pMsg->eCmd)
	{
	case eTriggerKDSChanged:
	   {
		   vOnLoadSettings();
	   }
	   break;
	default:
	   {

	   }
	   break;
	}
}

/*******************************************************************************
*
* FUNCTION: 	vHandleTraceMessage()
*
* DESCRIPTION: 	handler for trace command
*
* PARAMETER:   	const tUChar* puchData: reference of received message
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclCfgMgr::vHandleTraceMessage(const tUChar* puchData)
{
	ETG_TRACE_USR4(("vHandleTraceMessage() entered (data: 0x%08x).", puchData ));
	HC_NULL_POINTER_CHECK(puchData);

	tU32 u32MsgCode = (puchData[1]<<8) | puchData[2];
	switch (u32MsgCode) {
	case FC_HEATCTRL_HC_GETPDDTHMSTATE:
	   {
		   ETG_TRACE_FATAL(("%26s: HC_GetPddThmState ",
				   "HC.APPL.CFG_.CMD__.START" ));
		   vTraceInfo();
		   ETG_TRACE_FATAL(("%26s: finished without errors",
				   "HC.APPL.CFG_.CMD__.RES__" ));
	   }
	   break;

	case FC_HEATCTRL_HC_UPDATEPDDTHMSTATE:
	   {
		   ETG_TRACE_FATAL(("%26s: HC_GetPddThmState ",
				   "HC.APPL.CFG_.CMD__.START" ));
		   if (TRUE == m_oPDDTHM.bUpdatePDD())
		   {
			   ETG_TRACE_FATAL(("%26s: finished without errors",
					   "HC.APPL.CFG_.CMD__.RES__" ));
		   }
		   else
		   {
			   ETG_TRACE_FATAL(("%26s: finished with errors",
					   "HC.APPL.CFG_.CMD__.RES__" ));
		   }
	   }
	   break;

	default:
	   ETG_TRACE_USR4(("vHandleTraceMessage(): unhandled command %d.", u32MsgCode));
	   break;
	}
}

/*******************************************************************************
*
* FUNCTION: 	vGetReferences(tVoid)
*
* DESCRIPTION: 	Function to get all reference needed by this class.
* 				A reference should always be the Interface class of the object
*
* PARAMETER: 	None.
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclCfgMgr::vGetReferences(tVoid)
{
   ETG_TRACE_USR4(("vGetReferences() entered."));
   vTraceInfo();
}

/*******************************************************************************
*
* FUNCTION: 	tVoid vStartCommunication()
*
* DESCRIPTION: 	Function to start all dynamic objects e.g. threads, ...
*
* PARAMETER: 	None.
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclCfgMgr::vStartCommunication(tVoid)
{
   ETG_TRACE_USR4(("vStartCommunication() entered."));
}

/*******************************************************************************
*
* FUNCTION: 	vTraceInfo()
*
* DESCRIPTION: 	Trace information
*
* PARAMETER: 	None.
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclCfgMgr::vTraceInfo()
{
	ETG_TRACE_USR4(("vTraceInfo() entered."));
	tChar _strIdentStart[30] = "HC.APPL.CFG_.MGR__.START";
	tChar _strIdentAttr[30]  = "HC.APPL.CFG_.MGR__.ATTR_";
	tChar _strIdentEnd[30]   = "HC.APPL.CFG_.MGR__.END__";

	ETG_TRACE_USR2(("%026s: ", _strIdentStart ));
	ETG_TRACE_USR2(("%26s: <<< hardware configuration >>> ", _strIdentAttr ));
	vTraceInfo_HWConfig(_strIdentAttr);
	ETG_TRACE_USR2(("%26s: <<< PDD-THM configuration >>> ", _strIdentAttr ));
	vTraceInfo_PDDTHM(_strIdentAttr);
	ETG_TRACE_USR2(("%026s: ", _strIdentEnd ));
}

/*******************************************************************************
*
* FUNCTION: 	u8GetKDSSensorEnable
*
* DESCRIPTION: 	return PDD sensor availability
*
* PARAMETER: 	tenThermalSensorID enSensorID
*
* RETURNVALUE: 	tU8 ... TRUE = enable | FALSE = disable | else not configured
*
*******************************************************************************/
tU8 hc_tclCfgMgr::u8GetKDSSensorEnable(tenThermalSensorID enSensorID) const
{
   ETG_TRACE_USR4(("u8GetKDSSensorEnable(sensor id: %d) entered.", enSensorID ));
   if (enThermalSensorID_MAX > enSensorID)
	   return m_au8KDSSensorEnable[enSensorID];
   return OSAL_C_U8_MAX;
}

/*******************************************************************************
*
* FUNCTION: 	s16GetKDSSensorThreshold
*
* DESCRIPTION: 	return PDD sensor temperature threshold
*
* PARAMETER: 	tenThermalSensorID enSensorID
*               tenKDSDataSensorThreshold enThreshold
*               tBool bHysteresisRequired
*
* RETURNVALUE: 	tS16 ... >-2730 = valid temperature | else not configured
*
*******************************************************************************/
tS16 hc_tclCfgMgr::s16GetPDDSensorThreshold(tenThermalSensorID enSensorID, tenKDSDataSensorThreshold enThreshold, tBool bHysteresisRequired) const
{
   ETG_TRACE_USR4(("s16GetPDDSensorThreshold(sensor id: %d, threshold: %d, hysteresis required: %d) entered.", enSensorID, enThreshold, bHysteresisRequired ));
   tS16 _s16Return = HC_TEMP_INVALID_VALUE;
   if (enThermalSensorID_MAX > enSensorID && enKDSDataSensorThreshold_MAX > enThreshold)
   {
	   tenKDSDataSensor _enSensorIdx = enGetSensorIdx(enSensorID);
	   if (enKDSDataSensor_01 <= _enSensorIdx && enKDSDataSensor_MAX > _enSensorIdx)
	   {
		   hc_tclCfgMgr_PDDSensorConfigBase* _poSensorCfgData = m_oPDDTHM.poGetSensor(_enSensorIdx);
		   if (OSAL_NULL != _poSensorCfgData)
		   {
			   hc_tclCfgMgr_PDDThresholdBase* _poThreshold = _poSensorCfgData->poGetThreshold(enThreshold);
			   if (OSAL_NULL != _poThreshold)
			   {
				   if (FALSE == bHysteresisRequired)
				   {
					   _s16Return = _poThreshold->s16GetPDDThreshold();
				   }
				   else if (enKDSDataSensorThreshold_Low > enThreshold)
				   {
					   _s16Return = (_poThreshold->s16GetPDDThreshold() - (tS16)_poThreshold->u8GetPDDHysteresis());
				   }
				   else
				   {
					   _s16Return = (_poThreshold->s16GetPDDThreshold() + (tS16)_poThreshold->u8GetPDDHysteresis());
				   }
			   }
		   }
	   }
   }
   return _s16Return;
}

/*******************************************************************************
*
* FUNCTION: 	s16GetKDSSensorMaxTemp
*
* DESCRIPTION: 	return KDS maximum temperature of specified sensor
*
* PARAMETER: 	tenThermalSensorID enSensorID
*
* RETURNVALUE: 	tS16 ... >-2730 = valid temperature | else not configured
*
*******************************************************************************/
tS16 hc_tclCfgMgr::s16GetPDDSensorMaxTemp(tenThermalSensorID enSensorID) const
{
   ETG_TRACE_USR4(("s16GetPDDSensorMaxTemp(sensor id: %d) entered.", enSensorID ));
   tS16 _s16Return = HC_TEMP_INVALID_VALUE;
   tenKDSDataSensor _enSensorIdx = enGetSensorIdx(enSensorID);
   if (enKDSDataSensor_01 <= _enSensorIdx && enKDSDataSensor_MAX > _enSensorIdx)
   {
	   hc_tclCfgMgr_PDDSensorConfigBase* _poSensorCfgData = m_oPDDTHM.poGetSensor(enGetSensorIdx(enSensorID));
	   if (OSAL_NULL != _poSensorCfgData)
	   {
		   _s16Return = _poSensorCfgData->s16GetPDDMaxTemp();
	   }
   }
   return _s16Return;
}

/*******************************************************************************
*
* FUNCTION: 	s16GetKDSSensorMinTemp
*
* DESCRIPTION: 	return KDS minimum temperature of specified sensor
*
* PARAMETER: 	tenThermalSensorID enSensorID
*
* RETURNVALUE: 	tS16 ... >-2730 = valid temperature | else not configured
*
*******************************************************************************/
tS16 hc_tclCfgMgr::s16GetPDDSensorMinTemp(tenThermalSensorID enSensorID) const
{
   ETG_TRACE_USR4(("s16GetPDDSensorMinTemp(sensor id: %d) entered.", enSensorID ));
   tS16 _s16Return = HC_TEMP_INVALID_VALUE;
   tenKDSDataSensor _enSensorIdx = enGetSensorIdx(enSensorID);
   if (enKDSDataSensor_01 <= _enSensorIdx && enKDSDataSensor_MAX > _enSensorIdx)
   {
	   hc_tclCfgMgr_PDDSensorConfigBase* _poSensorCfgData = m_oPDDTHM.poGetSensor(enGetSensorIdx(enSensorID));
	   if (OSAL_NULL != _poSensorCfgData)
	   {
		   _s16Return = _poSensorCfgData->s16GetPDDMinTemp();
	   }
   }
   return _s16Return;
}

/*******************************************************************************
*
* FUNCTION: 	u8GetKDSFANEnable
*
* DESCRIPTION: 	return KDS FAN availability
*
* PARAMETER: 	None
*
* RETURNVALUE: 	tU8 ... TRUE = enable | FALSE = disable | else not configured
*
*******************************************************************************/
tU8 hc_tclCfgMgr::u8GetKDSFANEnable(tVoid) const
{
   ETG_TRACE_USR4(("u8GetKDSFANEnable() entered." ));
   return m_u8KDSFANEnable;
}

/*******************************************************************************
*
* FUNCTION: 	vLoadHWConfig
*
* DESCRIPTION: 	loads hardware configuration data from KDS
*
* PARAMETER: 	None.
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclCfgMgr::vLoadHWConfig(tVoid)
{
   ETG_TRACE_USR4(("vLoadVariantCoding() entered."));

   dp_tclKdsCMVariantCoding _oVariantCoding;
   tU8 _u8Data = FALSE;
   // CD drive mounted
   tU8 _u8Status = _oVariantCoding.u8GetCD(_u8Data);
   if (DP_U8_ELEM_STATUS_VALID == _u8Status)
   {
	   m_au8KDSSensorEnable[enThermalSensorID_CDDRIVE] = (((0x01 == _u8Data) || (0x02 == _u8Data)) ? (TRUE == _u8Data) : OSAL_C_U8_MAX);
   }
   else
   {
	   m_au8KDSSensorEnable[enThermalSensorID_CDDRIVE] = OSAL_C_U8_MAX;
   }
   // DVD drive mounted
   _u8Status = _oVariantCoding.u8GetDVD(_u8Data);
   if (DP_U8_ELEM_STATUS_VALID == _u8Status)
   {
	   m_au8KDSSensorEnable[enThermalSensorID_DVDDRIVE] = (((0x01 == _u8Data) || (0x02 == _u8Data)) ? (TRUE == _u8Data) : OSAL_C_U8_MAX);
   }
   else
   {
	   m_au8KDSSensorEnable[enThermalSensorID_DVDDRIVE] = OSAL_C_U8_MAX;
   }
   // TODO: BlueRay drive mounted
   _u8Status = _oVariantCoding.u8GetBDDrive(_u8Data);
   if (DP_U8_ELEM_STATUS_VALID == _u8Status)
   {
	   m_au8KDSSensorEnable[enThermalSensorID_BLUERAY] = (((0x01 == _u8Data) || (0x02 == _u8Data)) ? (TRUE == _u8Data) : OSAL_C_U8_MAX);
   }
   else
   {
	   m_au8KDSSensorEnable[enThermalSensorID_BLUERAY] = OSAL_C_U8_MAX;
   }
   // GNSS/GPS module mounted
   _u8Status = _oVariantCoding.u8GetGNSS(_u8Data);
   if (DP_U8_ELEM_STATUS_VALID == _u8Status)
   {
	   m_au8KDSSensorEnable[enThermalSensorID_GPS] = (((0x01 == _u8Data) || (0x02 == _u8Data)) ? (TRUE == _u8Data) : OSAL_C_U8_MAX);
   }
   else
   {
	   m_au8KDSSensorEnable[enThermalSensorID_GPS] = OSAL_C_U8_MAX;
   }
   // GYRO module mounted
   _u8Status = _oVariantCoding.u8GetGYRO(_u8Data);
   if (DP_U8_ELEM_STATUS_VALID == _u8Status)
   {
	   m_au8KDSSensorEnable[enThermalSensorID_GYRO] = (((0x01 == _u8Data) || (0x02 == _u8Data)) ? (TRUE == _u8Data) : OSAL_C_U8_MAX);
   }
   else
   {
	   m_au8KDSSensorEnable[enThermalSensorID_GYRO] = OSAL_C_U8_MAX;
   }
   // FAN mounted
   _u8Status = _oVariantCoding.u8GetFAN(_u8Data);
   if (DP_U8_ELEM_STATUS_VALID == _u8Status)
   {
	   m_u8KDSFANEnable = (((0x01 == _u8Data) || (0x02 == _u8Data)) ? (TRUE == _u8Data) : OSAL_C_U8_MAX);
   }
   else
   {
	   m_u8KDSFANEnable = OSAL_C_U8_MAX;
   }
   // XM module
   _u8Status = _oVariantCoding.u8GetXMTuner(_u8Data);
   if (DP_U8_ELEM_STATUS_VALID == _u8Status)
   {
	   m_au8KDSSensorEnable[enThermalSensorID_XMModule] = (((0x01 == _u8Data) || (0x02 == _u8Data)) ? (TRUE == _u8Data) : OSAL_C_U8_MAX);
   }
   else
   {
	   m_au8KDSSensorEnable[enThermalSensorID_XMModule] = OSAL_C_U8_MAX;
   }
   // TV module
   _u8Status = _oVariantCoding.u8GetDTV(_u8Data);
   if (DP_U8_ELEM_STATUS_VALID == _u8Status)
   {
	   m_au8KDSSensorEnable[enThermalSensorID_TVModule] = (((0x01 == _u8Data) || (0x02 == _u8Data)) ? (TRUE == _u8Data) : OSAL_C_U8_MAX);
   }
   else
   {
	   m_au8KDSSensorEnable[enThermalSensorID_TVModule] = OSAL_C_U8_MAX;
   }
   // thermal sensor of second display
#ifdef VARIANT_S_FTR_ENABLE_SENSOR_DISPLAY2
   m_au8KDSSensorEnable[enThermalSensorID_Panel2] = TRUE;
#else
   m_au8KDSSensorEnable[enThermalSensorID_Panel2] = OSAL_C_U8_MAX;
#endif //#ifdef VARIANT_S_FTR_ENABLE_SENSOR_DISPLAY2
}

/*******************************************************************************
*
* FUNCTION: 	vMirrorPDDTHM
*
* DESCRIPTION: 	mirror configuration data between KDS and PDD for thermal management
*
* PARAMETER: 	None.
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclCfgMgr::vMirrorPDDTHM(tVoid)
{
   ETG_TRACE_USR4(("bMirrorPDDTHM() entered."));
   tU8  au8KDSRawDataBuffer[DP_U8_KDSLEN_THERMALMANAGMENTTEMPERATURETHRESHOLDS_COMPLETE]={0};

   // read complete KDS raw data
   tS32 s32ReturnCode = DP_s32GetConfigItem( "ThermalManagmentTemperatureThresholds",
		   "", au8KDSRawDataBuffer, sizeof(au8KDSRawDataBuffer));
   if (DP_S32_NO_ERR == s32ReturnCode)
   {
	   // prepare PDD data
	   tU8 _u8DataSize = m_oPDDTHM.u8Deserialize(au8KDSRawDataBuffer,sizeof(au8KDSRawDataBuffer),0);
	   if (DP_U8_KDSLEN_THERMALMANAGMENTTEMPERATURETHRESHOLDS_COMPLETE != _u8DataSize)
	   {
		   ETG_TRACE_FATAL(("bMirrorPDDTHM: invalid data size %d", _u8DataSize));
	   }
	   // update PDD data
	   else
	   {
		   ETG_TRACE_USR1(("bMirrorPDDTHM: PDD data (size: %d, KDS loaded: %d, changed: %d, PDD initialized: %d)  prepared",
				   _u8DataSize, m_oPDDTHM.bIsKDSLoaded(), m_oPDDTHM.bIsConfigChanged(), m_oPDDTHM.bIsPDDInitialized() ));
		   // trace config data and status
		   //m_oPDDTHM.vTraceInfo();
		   // synchronize PDD of V850
#ifdef VARIANT_S_FTR_ENABLE_PDDTHM_AUTO_UPDATE
		   if (FALSE == m_oPDDTHM.bUpdatePDD())
		   {
			   ETG_TRACE_FATAL(("bMirrorPDDTHM: PDD couldn't update!" ));
		   }
		   // trace config data and status
		   //m_oPDDTHM.vTraceInfo();
#else
		   ETG_TRACE_USR1(("bMirrorPDDTHM: PDD will not update automatically!" ));
#endif //#ifdef VARIANT_S_FTR_ENABLE_PDDTHM_AUTO_UPDATE
	   }
   }
   else
   {
	   ETG_TRACE_FATAL(("bMirrorPDDTHM: KDS data couldn't read! (error code: %d)", s32ReturnCode));
   }
   // assign sensor index to sensor id
   (tVoid)memset((tVoid*)m_aenSensorIdx, -1, sizeof(m_aenSensorIdx));
   for (tU32 _u32SensorIdx = 0; (tU32)enKDSDataSensor_MAX > _u32SensorIdx; ++_u32SensorIdx)
   {
	   hc_tclCfgMgr_PDDSensorConfigBase* _poSensorData = m_oPDDTHM.poGetSensor((tenKDSDataSensor)_u32SensorIdx);
	   if ((OSAL_NULL != _poSensorData) && (0 < _poSensorData->u8GetPDDModulID()))
	   {
		   tenKDSDataSensorID _enKDSSensorID = m_oPDDTHM.enConvert2KDSSensorID(_poSensorData->u8GetPDDModulID());
		   tenThermalSensorID _enSensorID = m_oPDDTHM.enConvert2SensorID(_enKDSSensorID);
		   if (FALSE == bSetSensorIdx(_enSensorID, (tenKDSDataSensor)_u32SensorIdx))
		   {
			   ETG_TRACE_USR1(("bMirrorPDDTHM: Saving of sensor data index %d failed for SensorID %d!", _u32SensorIdx, _enSensorID ));
		   }
	   }
   }
}

/*******************************************************************************
*
* FUNCTION: 	vTraceInfo_HWConfig()
*
* DESCRIPTION: 	Trace hardware configuration information
*
* PARAMETER:   	const char* strIdentifier		trace identifier
*                                               like "HC_APPL.TSEN.GYRO_.ATTR_"
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclCfgMgr::vTraceInfo_HWConfig(const char* strIdentifier) const
{
	ETG_TRACE_USR4(("vTraceInfo_HWConfig() entered."));

	ETG_TRACE_USR2(("%26s: %30s=%u ", strIdentifier, "      KDS_FAN_ENABLE",
			ETG_ENUM(HC_Bool, u8GetKDSFANEnable()) ));
	for (tU32 _u32Idx = 0; (tU32)enThermalSensorID_MAX > _u32Idx; ++_u32Idx)
	{
		tChar _strAttruteName[30] = "";
		sprintf(_strAttruteName, " KDS_TS_%5s_ENABLE", strGetThermalSensorAbbreviation((tenThermalSensorID)_u32Idx));
		ETG_TRACE_USR2(("%26s: %30s=%u ", strIdentifier, _strAttruteName,
			   ETG_ENUM(HC_Bool, u8GetKDSSensorEnable((tenThermalSensorID)_u32Idx)) ));
	}
}

/*******************************************************************************
*
* FUNCTION: 	vTraceInfo_PDDTHM()
*
* DESCRIPTION: 	Trace PDD-THM configuration information
*
* PARAMETER:   	const char* strIdentifier		trace identifier
*                                               like "HC_APPL.TSEN.GYRO_.ATTR_"
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclCfgMgr::vTraceInfo_PDDTHM(const char* strIdentifier) const
{
	ETG_TRACE_USR4(("vTraceInfo_PDDTHM() entered."));

	for (tU32 _u32Idx = 0; (tU32)enThermalSensorID_MAX > _u32Idx; ++_u32Idx)
	{
		tChar _strAttruteName[30] = "";
		sprintf(_strAttruteName, "  KDS_SENSORID_%5s", strGetThermalSensorAbbreviation((tenThermalSensorID)_u32Idx));
		ETG_TRACE_USR2(("%26s: %30s=%d ", strIdentifier, _strAttruteName,
				enGetSensorIdx((tenThermalSensorID)_u32Idx) ));
	}
	m_oPDDTHM.vTraceInfo(strIdentifier);
}

/*******************************************************************************
*
* FUNCTION:    vOnLoadSettings()
*
* DESCRIPTION: loading of all settings
*
* PARAMETER:   none
*
* RETURNVALUE: none
*
*******************************************************************************/
tVoid hc_tclCfgMgr::vOnLoadSettings(tVoid)
{
   ETG_TRACE_USR4(("vOnLoadSettings() entered."));

   vLoadHWConfig();
   vMirrorPDDTHM();
   if (OSAL_NULL != hc_tclAppMain::theServer())
   {
	   hc_tclBaseIf::TMsg _oMsg;
	   (tVoid)memset((tVoid*)&_oMsg, OSAL_NULL, sizeof(_oMsg));
	   _oMsg.eCmd = hc_tclBaseIf::eTriggerLoadSettings;
	   hc_tclAppMain::theServer()->vPostInternalMessage(&_oMsg);
   }
}
