/***************************************************************************
 *
 * Copyright 2010,2011 BMW Car IT GmbH
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ****************************************************************************/

#include "configurator.h"
#include "ICommandExecutor.h"
#include "CommitCommand.h"
#include "Configuration.h"
#include "Log.h"
#include <unistd.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include <dirent.h>

#include "pdd.h"
#include "pdd_config_nor_user.h"
#include "dispvidctrl_datapool.h"
#include "VD_EARLYNorData.h"

AIVIConfigurator::AIVIConfigurator(ICommandExecutor& executor, Configuration& config)
: IConfigurator(&executor)
, mExecutor(executor)
, mConfiguration(config)
{
}

/* has to be the same for all Configurator plugins*/
PluginApi AIVIConfigurator::pluginGetApi() const
{
    return Configurator_Api_v1;
}

/* can be the same for all Scene plugins
 * this plugin is not running all the time
 * nobody should use this function
 * but we have to fulfill the plugin interface*/
HealthCondition AIVIConfigurator::pluginGetHealth()
{
    return HealthRunning;
}

/*Core function of the Scene plugin
 * this will be executed when plugin is loaded
 * all Scene setup has to be defined here*/
bool AIVIConfigurator::configure()
{
    /*read out the configuration from NOR early*/
    bool success = vReadEarlyConfig();

    if(success == true) {
        /*set new path for input device configuration file*/
        mConfiguration.setInputDevConfigPath((const char*)tU8TrFlushDesignConfigFile);

        /*set new mapping for display id's, always provide a mapping for 4 possible display
        * even they are not used, duplicated are not allowed*/
        mConfiguration.setDisplayIDmapping(tU8TrDisplayOrder[0], tU8TrDisplayOrder[1], tU8TrDisplayOrder[2], tU8TrDisplayOrder[3]);

        updateScreenLayoutsDefinition();
    }

    LOG_INFO("AIVIConfigurator", "CONFIGURATOR PLUGIN COMPLETE");

    return true;
}

/* should be the name of a class!!!*/
t_ilm_const_string AIVIConfigurator::pluginGetName() const
{
    return "AIVIConfigurator";
}

int AIVIConfigurator::getIterationCounter()
{
    // this plugin is one-shot and is not monitored.
    static int dummyValue = 0;
    return ++dummyValue;
}

bool AIVIConfigurator::vReadEarlyConfig() {

    // Set default values
    tU8EarlyBrand = 0;
    tU8EarlyDisplayOrientation = 0;
    tU8EarlyResolution[0] = '\0';
    tU8TrDisplayOrder[0] = 0;
    tU8TrDisplayOrder[1] = 2;
    tU8TrDisplayOrder[2] = 1;
    tU8TrDisplayOrder[3] = 3;
    tU8TrFlushDesignConfigFile[0] = '\0';

    char str[200];

    //Get settings data from NOR data
    int s32StreamLength = 800;

    bool bSuccess = true;

    TEarlyNorForceConfiguration tEarlyNorForce;

    //Allocate buffer
    unsigned char* pu8DataStreamBuffer = (unsigned char*) malloc((unsigned int)s32StreamLength);
    tU8            Vu8Info             = PDD_READ_INFO_NORMAL_FILE;
    int s32Result = pdd_read_datastream_early_from_nor((const char*)PDD_NOR_USER_DATASTREAM_NAME_DATASET_VD_EARLY_FORCE_CONFIG, pu8DataStreamBuffer, s32StreamLength, 1, &Vu8Info);
    if (s32Result > 0) {
        if(pdd_helper_get_element_from_stream((const char *)"EarlyNorConfigForce",pu8DataStreamBuffer,(size_t)s32Result, (void*) &tEarlyNorForce, sizeof(TEarlyNorForceConfiguration),0)==0) { //lint !e605
            sprintf(str, "CONFIGURATOR PLUGIN - AIVIConfigurator::vReadEarlyConfigScreenBroker() Brand READ FROM NOR %d", tEarlyNorForce.u8EarlyBrand);
            LOG_INFO("AIVIConfigurator", str);
            sprintf(str, "CONFIGURATOR PLUGIN - AIVIConfigurator::vReadEarlyConfigScreenBroker() DisplayOrientation READ FROM NOR %d", tEarlyNorForce.u8EarlyDisplayOrientation);
            LOG_INFO("AIVIConfigurator", str);
            sprintf(str, "CONFIGURATOR PLUGIN - AIVIConfigurator::vReadEarlyConfigScreenBroker() Resolution READ FROM NOR %s", tEarlyNorForce.au8EarlyResolution);
            LOG_INFO("AIVIConfigurator", str);
            sprintf(str, "CONFIGURATOR PLUGIN - AIVIConfigurator::vReadEarlyConfigScreenBroker() TrDisplayOrder READ FROM NOR %d, %d, %d, %d", tEarlyNorForce.au8EarlyDisplayOrder[0], tEarlyNorForce.au8EarlyDisplayOrder[1], tEarlyNorForce.au8EarlyDisplayOrder[2], tEarlyNorForce.au8EarlyDisplayOrder[3]);
            LOG_INFO("AIVIConfigurator", str);
            sprintf(str, "CONFIGURATOR PLUGIN - AIVIConfigurator::vReadEarlyConfigScreenBroker() TrFlushDesignConfigFile READ FROM NOR %s", tEarlyNorForce.au8EarlyFlushDesignConfigFile);
            LOG_INFO("AIVIConfigurator", str);
            tU8EarlyBrand = tEarlyNorForce.u8EarlyBrand;
            tU8EarlyDisplayOrientation = tEarlyNorForce.u8EarlyDisplayOrientation;
            memcpy(&tU8EarlyResolution[0], &tEarlyNorForce.au8EarlyResolution[0], sizeof tU8EarlyResolution);
			tU8TrDisplayOrder[0] = tEarlyNorForce.au8EarlyDisplayOrder[0];
			tU8TrDisplayOrder[1] = tEarlyNorForce.au8EarlyDisplayOrder[1];
			tU8TrDisplayOrder[2] = tEarlyNorForce.au8EarlyDisplayOrder[2];
			tU8TrDisplayOrder[3] = tEarlyNorForce.au8EarlyDisplayOrder[3];
            memcpy(&tU8TrFlushDesignConfigFile[0], &tEarlyNorForce.au8EarlyFlushDesignConfigFile[0], sizeof tU8TrFlushDesignConfigFile);
        }
        else {
            bSuccess = false;
            LOG_ERROR("AIVIConfigurator", "CONFIGURATOR PLUGIN - AIVIConfigurator::vReadEarlyConfigScreenBroker() EarlyNorConfigForce READ FROM NOR Failure");
        }
    }
    else {
        bSuccess = false;
        sprintf(str, "CONFIGURATOR PLUGIN - AIVIConfigurator::vReadEarlyConfigScreenBroker() pdd_read_datastream_early_from_nor Failure %d", s32Result);
        LOG_WARNING("AIVIConfigurator", str);
    }

    free(pu8DataStreamBuffer);
    return bSuccess;
}

void AIVIConfigurator::updateScreenLayoutsDefinition()
{
   if(createScreenLayoutsDir())
   {
       ::std::string name = "/opt/bosch/hmibase/" + getScreenLayoutsName();
	   
       char str[200];
       sprintf(str, "CONFIGURATOR PLUGIN - AIVIConfigurator::updateScreenLayoutsDefinition() Screen Layout file name %s", name.c_str());
       LOG_INFO("AIVIConfigurator", str);
	   
       if (isFileExistsInSystem(name))
       {
           mode_t lastMode   = umask(0);
           ::std::string cmd = "ln -sf ";
           cmd.append(name);
           cmd.append(" /tmp/hmi/ScreenLayouts/ScreenLayouts.xml");
           if (system(cmd.c_str()) != 0)
           {
              LOG_INFO("AIVIConfigurator", "CONFIGURATOR PLUGIN - AIVIConfigurator::updateScreenLayoutsDefinition() Screen layout symlink update failed: Symlink CMD is not executed");
           }
           umask(lastMode);
       }
       else
       {
           LOG_INFO("AIVIConfigurator", "CONFIGURATOR PLUGIN - AIVIConfigurator::updateScreenLayoutsDefinition() Screen layout symlink update failed: Invalid File");
       }
   }
   else
   {
      LOG_INFO("AIVIConfigurator", "CONFIGURATOR PLUGIN - AIVIConfigurator::updateScreenLayoutsDefinition() Screen layout symlink update failed: Invalid ScreenLayout Dir");
   }
}

bool AIVIConfigurator::createScreenLayoutsDir() const
{
    bool isValid = false;
    DIR* pDir = opendir("/tmp/hmi/ScreenLayouts");
    if (NULL == pDir)
    {
        if (system("mkdir -p /tmp/hmi/ScreenLayouts") == 0)
        {
           isValid = true;
        }
    }
    else
    {
       isValid = true;
       closedir(pDir);
    }
    return isValid;
}

bool AIVIConfigurator::isFileExistsInSystem(const ::std::string& name) const
{
    bool isValid = false;
    if (!name.empty())
    {
        struct stat buffer;
        if ((stat(name.c_str(), &buffer) == 0) && (S_ISREG(buffer.st_mode)))
        {
            isValid = true;
        }
    }
    return isValid;
}

::std::string AIVIConfigurator::getScreenLayoutsName() const
{
    ::std::string name = "ScreenLayouts_rnaivi.xml";
    switch (tU8EarlyBrand)
    {
        case DISPVIDCTRL_AllianceOpeningAnimation_Type_Renault:
		case 0x04:
        {
            name = getScreenLayoutsNameForRenault();
            break;
        }
        case DISPVIDCTRL_AllianceOpeningAnimation_Type_Infiniti:
        {
            name = "ScreenLayouts_npivi.xml";
            break;
        }
        case DISPVIDCTRL_AllianceOpeningAnimation_Type_Nissan:
        {
            if (strcmp((const char*)tU8EarlyResolution, "jdi_lam080gx") == 0)
            {
               name = "ScreenLayouts_rnaivi_ST3_HighRes.xml";
               break;
            }
        }
        default:
            break;
    }
    return name;
}

::std::string AIVIConfigurator::getScreenLayoutsNameForRenault() const
{
    ::std::string name = "ScreenLayouts_rivie.xml";
    switch (tU8EarlyDisplayOrientation)
    {
        case DISPVIDCTRL_AllianceDisplayOrientation_Portrait:
        {
            name = "ScreenLayouts_rivieP.xml";
            break;
        }
        default:
            break;
    }
    return name;
}

/* The name in brakes has to be the same as name of a class!!!
 * otherwise plugin will not be accepted*/
DECLARE_LAYERMANAGEMENT_PLUGIN(AIVIConfigurator)
