/**
 * @file   : cfgmgr_ClientDatabase.cpp
 * @author : mrr8kor
 * @date   : Aug 24, 2017
 * @brief  : Client Configtable to store registrations from application
 * @copyright (c) 2017 Robert Bosch Car Multimedia Gmbh
 * @addgroup : cfgmgr
 * @{
 */
 
#ifndef CFGMGR_CLIENTDATABASE_H_
#include "cfgmgr_ClientDatabase.h"
#endif

#ifndef CFGMGR_ASF_APPCLIENT_H_
#include "cfgmgr_ASFAppClient.h"
#endif

#ifndef CFGMGR_ASF_APPSERVER_H_
#include "cfgmgr_ASFAppServer.h"
#endif

using namespace std;
using namespace asf;
using namespace de::bosch::cm::ConfigManager::ConfigData;

/*Initialization of static self-referential object*/
cfgmgr_ClientDatabase* cfgmgr_ClientDatabase::m_poClientDatabaseObj=NULL;

/********************************************************************************/
/* 						cfgmgr_ClientDatabase Constructor			            */
/********************************************************************************/
cfgmgr_ClientDatabase::cfgmgr_ClientDatabase()
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_ClientDatabase::cfgmgr_ClientDatabase constructor");

	this->fillCfgMgrPrivateItems();
	m_poPublicProxyList.clear();
}

/********************************************************************************/
/* 			cfgmgr_ClientDatabase findCaseInsensitive sub-string	            */
/********************************************************************************/
tbool cfgmgr_ClientDatabase::compareItemName(tStr RegItemName, tStr ItemName)
{
	tbool result = false;

	if(!(trim(RegItemName).empty()) && !(trim(ItemName).empty()))
	{
		size_t pos1 =  RegItemName.find('*');

		if(pos1 != string::npos)
		{
			RegItemName = RegItemName.substr(0,pos1);
		}

		size_t pos2 =  ItemName.find('*');

		if(pos2 != string::npos)
		{
			ItemName = ItemName.substr(0,pos2);
		}

		if((pos1 != string::npos) || (pos2 != string::npos))
		{
			if(RegItemName.length() > ItemName.length())
			{
				tUInt8 count = ItemName.length();

				if(findCaseInsensitive(RegItemName.substr(0,count),ItemName) != string::npos)
				{
					result = true;
				}
			}
			else if(RegItemName.length() < ItemName.length())
			{

				tUInt8 count = RegItemName.length();

				if(findCaseInsensitive(ItemName.substr(0,count),RegItemName) != string::npos)
				{
					result = true;
				}
			}
			else
			{
				result = CompareStrings(trim(RegItemName),trim(ItemName));
			}
		}
		else
		{
			result = CompareStrings(trim(RegItemName),trim(ItemName));
		}
	}
	return result;
}

/***********************************************************************************************/
/* 						Function to register for Publi Items by Config Manager   	 		   */
/*					    Please use DBUS Name as "cfgmgr" for all items for easy understanding  */
/***********************************************************************************************/
tVoid cfgmgr_ClientDatabase::fillCfgMgrPrivateItems()
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_ClientDatabase::fillCfgMgrPublicItems ");

	ProxyList_T cfgmgrPublicTable[CFGMGR_PUBLIC_ITEM_TABLE_SIZE] =
	{
			{"CSA.Connectivity_and_services_status","cfgmgr",DEFAULT_FORMAT,DP_CSA,"Status"},
			//Add Public Items received from FOTA (Remote Public Items)

			{"URL","cfgmgr",DEFAULT_FORMAT,DP_URL,"Base_URL"},
			{"partID","cfgmgr",DEFAULT_FORMAT,DP_PARTID,"partID"}
			//Add Public Items received from Alliance telematics and other components (Local items)
	};

	m_poPrivateProxyList.clear();

	for(tUInt16 count = 0; count < CFGMGR_PUBLIC_ITEM_TABLE_SIZE;count++)
	{
		if(count < CFGMGR_DATAPOOL_SIZE)
		{
			m_poPrivateProxyList.push_back(cfgmgrPublicTable[count]);
		}
		m_PublicItemTable.push_back(cfgmgrPublicTable[count]);
	}
}

/***********************************************************************************************/
/* Search in the entry table to check if application is registered for the configItem			*/
/***********************************************************************************************/
Reg_Status_T cfgmgr_ClientDatabase::searchForConfigItemName(tStr configItem, tStr Dbus, ListType type)
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_ClientDatabase::searchForConfigItemName called with configItem - ", configItem.c_str(), ",Dbus Name -", Dbus.c_str(), ",List type -", type);

	Reg_Status_T regStatus = REG_STATUS_FAILED;
	tbool validPublicItem = false;
	vector<ProxyList_T>::iterator listIterator;

	switch(type)
	{
	case PRIVATE_LIST:
	{
		for(listIterator = m_poPrivateProxyList.begin() ; listIterator != m_poPrivateProxyList.end();listIterator ++)
		{
			if(compareItemName(listIterator->configItemName, configItem))
			{
				log(&CFM0,DLT_LOG_DEBUG,"cfgmgr_ClientDatabase::searchForConfigItemName configItem is present in private list");
				regStatus = REG_STATUS_ALREADY_REGISTERED;
				break;
			}
		}
		break;
	}
	case PUBLIC_LIST:
	{
		for(tUInt16 count = 0; count < CFGMGR_PUBLIC_ITEM_TABLE_SIZE;count ++)
		{
			if(CompareStrings(m_PublicItemTable[count].configItemName,configItem))
			{
				validPublicItem = true;
				break;
			}
		}

		if(validPublicItem)
		{
			for(listIterator = m_poPublicProxyList.begin() ; listIterator != m_poPublicProxyList.end();listIterator ++)
			{
				if((CompareStrings(listIterator->configItemName, configItem)) && (CompareStrings(listIterator->DBUSName, Dbus)))
				{
					log(&CFM0,DLT_LOG_DEBUG,"cfgmgr_ClientDatabase::searchForConfigItemName configItem is present in PUBLIC_LIST");
					regStatus = REG_STATUS_ALREADY_REGISTERED;
					break;
				}
			}
		}
		else
		{
			regStatus = REG_STATUS_UNKNOWN_ITEM;
		}

		break;
	}
	default:
		break;
	}

	return regStatus;
}

/***********************************************************************************************/
/* Check for Format XML (private), Default (public)												*/
/***********************************************************************************************/
tbool cfgmgr_ClientDatabase::getFormatIdType(tStr configItem, tbool &isCfgMgrRegistered, FormatIdentifier &formatFlag)
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_ClientDatabase::getFormatIdType called with configItem - ", configItem.c_str());

	tbool namePresent = false;
	std::vector<ProxyList_T> tempList;
	tempList.clear();

	tempList.assign(m_poPrivateProxyList.begin(),m_poPrivateProxyList.end());

	vector<ProxyList_T>::iterator listIterator;
	for(listIterator = tempList.begin() ; listIterator != tempList.end();listIterator ++)
	{
		if(compareItemName(listIterator->configItemName, configItem))
		{
			namePresent = true;

			if(listIterator->DBUSName == "cfgmgr")
			{
				isCfgMgrRegistered = true;
			}
			formatFlag = listIterator->formatFlag;
			break;
		}
	}
	return namePresent;
}

/***********************************************************************************************/
/* 	Get proxy for applications registered for private items										*/
/***********************************************************************************************/
tbool cfgmgr_ClientDatabase::getProxy(tStr profileName,::boost::shared_ptr<ConfigDataProxy> &proxy,ListType type)
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_ClientDatabase::getProxy called with profileName - ",profileName.c_str(), ",ListType - ", type);

	tbool namePresent = false;
	std::vector<ProxyList_T> tempList;
	tempList.clear();

	if(PRIVATE_LIST == type)
	{
		tempList.assign(m_poPrivateProxyList.begin(),m_poPrivateProxyList.end());
	}
	else if(PUBLIC_LIST == type)
	{
		tempList.assign(m_poPublicProxyList.begin(),m_poPublicProxyList.end());
	}

	vector<ProxyList_T>::iterator listIterator;
	for(listIterator = tempList.begin() ; listIterator != tempList.end();listIterator ++)
	{
		if(compareItemName(listIterator->configItemName,profileName))
		{
			namePresent = true;
			proxy = listIterator->proxy;
			break;
		}
	}


	return namePresent;
}

/***********************************************************************************************/
/* 	Get proxy for applications registered for public items										*/
/***********************************************************************************************/
tbool cfgmgr_ClientDatabase::getProxyforRegisteredPublicItem(tStr profileName, vector< ::boost::shared_ptr<ConfigDataProxy> >  &proxylist)
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_ClientDatabase::getProxyforRegisteredPublicItem with profileName - ", profileName.c_str());

	::boost::shared_ptr<ConfigDataProxy> proxy;

	tbool matchFound = false;

	std::vector<ProxyList_T> tempList;
	tempList.clear();

	tempList.assign(m_poPublicProxyList.begin(),m_poPublicProxyList.end());
	vector<ProxyList_T>::iterator listIterator;

	for(listIterator = tempList.begin() ; listIterator != tempList.end();listIterator ++)
	{
		if(listIterator->configItemName == profileName)
		{
			matchFound = true;
			proxy = listIterator->proxy;

			proxylist.push_back(proxy);
		}
	}

	if(!matchFound)
	{
		log(&CFM0,DLT_LOG_ERROR,"cfgmgr_ClientDatabase::getProxyforRegisteredPublicItem - match NOT found!");
	}

	return matchFound;
}

/***********************************************************************************************/
/* 	Store application details for private item													*/
/***********************************************************************************************/
tVoid cfgmgr_ClientDatabase::storePrivateAppInfo(tStr configItemName,tStr DbusName,::boost::shared_ptr<ConfigDataProxy > proxy,FormatIdentifier formatFlag)
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_ClientDatabase::storePrivateAppInfo", configItemName.c_str(), ",DbusName - ", DbusName.c_str(), ",formatFlag - ", formatFlag);

	ProxyList_T temp;

	temp.configItemName = configItemName;
	temp.DBUSName = DbusName;
	temp.formatFlag = formatFlag;
	temp.proxy = proxy;

	m_poPrivateProxyList.push_back(temp);

	log(&CFM0,DLT_LOG_DEBUG,"Size of Private List - ",m_poPrivateProxyList.size());
}

/***********************************************************************************************/
/* Store application details for public item - Register Applications							*/
/***********************************************************************************************/
tVoid cfgmgr_ClientDatabase::storePublicAppInfo(tStr configItemName,tStr DbusName,::boost::shared_ptr<ConfigDataProxy > proxy,FormatIdentifier formatFlag)
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_ClientDatabase::storePublicAppInfo with configItemName - ", configItemName.c_str(), ",DbusName - ", DbusName.c_str(), ",formatFlag - ", formatFlag);

	ProxyList_T temp;

	temp.configItemName = configItemName;
	temp.DBUSName = DbusName;
	temp.formatFlag = formatFlag;
	temp.proxy = proxy;

	if(temp.proxy != NULL)
	{
		m_poPublicProxyList.push_back(temp);
	}
	log(&CFM0,DLT_LOG_DEBUG,"Size of Public List - ", m_poPublicProxyList.size());
}

/***********************************************************************************************/
/* 	Unregister Application for Private item														*/
/***********************************************************************************************/  
Reg_Status_T cfgmgr_ClientDatabase::deletePrivateAppInfo(tStr configItemName, tStr DbusName)
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_ClientDatabase::deletePrivateAppInfo with configItemName - ", configItemName.c_str(), ",DbusName - ", DbusName.c_str());
	
	Reg_Status_T status = REG_STATUS_ALREADY_UNREGISTERED;

	for(tUInt16 count = CFGMGR_ZERO; count < m_poPrivateProxyList.size();count++)
	{
		if((configItemName == m_poPrivateProxyList.at(count).configItemName) && ((DbusName == m_poPrivateProxyList.at(count).DBUSName)))
		{
			m_poPrivateProxyList.erase(m_poPrivateProxyList.begin()+ count);
			log(&CFM0,DLT_LOG_DEBUG,"cfgmgr_ClientDatabase::deletePrivateAppInfo - unregistered Application");
			status = REG_STATUS_SUCCESS;
			break;
		}
	}

	return status;
}

/***********************************************************************************************/
/* 	Unregister Application for Public item							*/
/***********************************************************************************************/
Reg_Status_T cfgmgr_ClientDatabase::deletePublicAppInfo(tStr configItem, tStr DbusName)
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_ClientDatabase::deletePublicAppInfo with configItemName - ", configItem.c_str(), ", DbusName - ", DbusName.c_str());
	
	Reg_Status_T status = REG_STATUS_ALREADY_UNREGISTERED;
	tbool validPublicItem = false;

	for(tUInt16 count = 0; count < CFGMGR_PUBLIC_ITEM_TABLE_SIZE;count ++)
	{
		if(CompareStrings(m_PublicItemTable[count].configItemName,configItem))
		{
			validPublicItem = true;
			break;
		}
	}

	if(validPublicItem)
	{
		for(tUInt16 count = CFGMGR_ZERO; count < m_poPublicProxyList.size();count++)
		{
			if((configItem == m_poPublicProxyList.at(count).configItemName) && ((DbusName == m_poPublicProxyList.at(count).DBUSName)))
			{
				m_poPublicProxyList.erase(m_poPublicProxyList.begin()+ count);
				log(&CFM0,DLT_LOG_DEBUG,"cfgmgr_ClientDatabase::deletePublicAppInfo - unregistered Application");
				status = REG_STATUS_SUCCESS;
				break;
			}
		}
	}
	else
	{
		status = REG_STATUS_UNKNOWN_ITEM;
	}

	return status;
}

/***********************************************************************************************/
/* 								Get List of All the Public Item Names						   */
/***********************************************************************************************/
tbool cfgmgr_ClientDatabase::getListofAllPublicItemNames(map<tStr,DatapoolType> &publicItemList)
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_ClientDatabase::getListofAllPublicItemNames");

	tStr item;
	DatapoolType type;
	std::vector<ProxyList_T> tempList;
	tempList.clear();

	tempList.assign(m_PublicItemTable.begin(),m_PublicItemTable.end());
	vector<ProxyList_T>::iterator listIterator;

	for(listIterator = tempList.begin() ; listIterator != tempList.end();listIterator ++)
	{
		item = listIterator->configItemName;
		type = listIterator->dpType;

		publicItemList[item] = type;
	}

	if (CFGMGR_ZERO != publicItemList.size())
	{
		log(&CFM0,DLT_LOG_INFO,"cfgmgr_ClientDatabase::getListofAllPublicItemNames,public items = ",publicItemList.size());
		return true;
	}
	else
	{
		log(&CFM0,DLT_LOG_ERROR,"cfgmgr_ClientDatabase::getListofAllPublicItemNames, NO public items ");
		return false;
	}
}

tVoid cfgmgr_ClientDatabase::getConfigItemList(tStr itemType)
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_ClientDatabase::getConfigItemList, ItemType - ", itemType);
	std::vector<ProxyList_T> tempList;
	tempList.clear();

	if(itemType == "PRIVATE")
	{
		tempList.assign(m_poPrivateProxyList.begin(),m_poPrivateProxyList.end());
	}
	else if(itemType == "PUBLIC")
	{
		tempList.assign(m_poPublicProxyList.begin(),m_poPublicProxyList.end());
	}
	else
	{
		log(&CFM0,DLT_LOG_ERROR,"ItemType not matched");
	}

	log(&CMIT,DLT_LOG_DEBUG,"ItemList size - ", tempList.size());

	if(CFGMGR_ZERO != tempList.size())
	{
		vector<ProxyList_T>::iterator listIterator;
		log(&CMIT,DLT_LOG_INFO," ************************************* ");
		for(listIterator = tempList.begin(); listIterator != tempList.end(); listIterator++)
		{
			log(&CMIT,DLT_LOG_INFO,"CfgMgr Registration: ", listIterator->configItemName, " : ", listIterator->DBUSName, " : ", listIterator->formatFlag);
		}
		log(&CMIT,DLT_LOG_INFO," ************************************* ");
	}

	return;
}


tVoid cfgmgr_ClientDatabase::getProxyRegItem(tStr configItem,tStr dbusname,::boost::shared_ptr<ConfigDataProxy> &proxy)
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_ClientDatabase::getProxyRegItem called with configItem - ",configItem.c_str(), ",dbusname - ", dbusname);

	tbool namePresent = false;
	std::vector<ProxyList_T> tempList;
	tempList.clear();

	tempList.assign(m_poPublicProxyList.begin(),m_poPublicProxyList.end());

	vector<ProxyList_T>::iterator listIterator;
	for(listIterator = tempList.begin() ; listIterator != tempList.end();listIterator ++)
	{
		if((CompareStrings(listIterator->configItemName, configItem)) && (CompareStrings(listIterator->DBUSName, dbusname)))
		{
			proxy = listIterator->proxy;
			break;
		}
	}
	return;
}

DatapoolType cfgmgr_ClientDatabase::getDpType(tStr ItemName,tStr &storedKey)
{
	DatapoolType dp_Type = DP_UNKNOWN;
	vector<ProxyList_T>::iterator listIterator;
	for(listIterator = m_PublicItemTable.begin() ; listIterator != m_PublicItemTable.end();listIterator ++)
	{
		if(CompareStrings(listIterator->configItemName, ItemName))
		{
			dp_Type = listIterator->dpType ;
			storedKey = listIterator->pKey;
			break;
		}
	}
	return dp_Type;
}
