/**
 * @file   : cfgmgr_ASFAppClient.h
 * @author : mrr8kor
 * @date   : Aug 24, 2017
 * @brief  : ASF Native App Client implementation for cfgmgr
 * @copyright (c) 2017 Robert Bosch Car Multimedia Gmbh
 * @addgroup : cfgmgr
 * @{
 */

#ifndef CFGMGR_ASF_APPCLIENT_H_
#include "cfgmgr_ASFAppClient.h"
#endif

namespace asf{

using namespace de::bosch::cm::ConfigManager::ConfigData;

/***********************************************************************************************/
/* 						Function to get the instance of Update manager    	    			   */
/***********************************************************************************************/
cfgmgr_UpdateManager* cfgmgr_ASFAppClient::getInstanceOfUpdateManager()
{
	return cfgmgr_UpdateManager::getInstanceofUpdateManager();
}

/***********************************************************************************************/
/* 						cfgmgr_ASFAppClient Constructor         	    			           */
/***********************************************************************************************/
cfgmgr_ASFAppClient::cfgmgr_ASFAppClient() :BaseComponent(),m_pUpdateManager(NULL)
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_ASFAppClient object is created successfully");

	m_pUpdateManager = getInstanceOfUpdateManager();
	if(m_pUpdateManager)
	{
		m_pUpdateManager->registerClientHandler(this);
	}
}

/***********************************************************************************************/
/* 						cfgmgr_ASFAppClient Destructor            	    			           */
/***********************************************************************************************/
cfgmgr_ASFAppClient::~cfgmgr_ASFAppClient()
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_ASFAppClient Destructor");

	m_pUpdateManager = NULL;
}

/***********************************************************************************************/
/* 						Function called once Applications are available for Communication      */
/***********************************************************************************************/
tVoid cfgmgr_ASFAppClient::onAvailable(const boost::shared_ptr<asf::core::Proxy>& proxy, const ServiceStateChange &stateChange)
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_ASFAppClient::onAvailable");
}

/***********************************************************************************************/
/* 						Function called once Applications are unavailable for Communication    */
/***********************************************************************************************/
tVoid cfgmgr_ASFAppClient::onUnavailable(const boost::shared_ptr<asf::core::Proxy>& proxy, const ServiceStateChange &stateChange)
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_ASFAppClient::onUnavailable");
}

/***********************************************************************************************/
/* 						Function called on request of set Private Item Error cases             */
/***********************************************************************************************/
tVoid cfgmgr_ASFAppClient::onSetPrivateItemError(const ::boost::shared_ptr< ConfigDataProxy >& proxy, const ::boost::shared_ptr< SetPrivateItemError >& error)
{
	log(&CFM0,DLT_LOG_ERROR,"cfgmgr_ASFAppClient::onSetPrivateItemError");

	if (m_Item.ActID == error->getAct())
	{
		if(NULL != m_pUpdateManager)
		{
			m_pUpdateManager->onSetPrivateItem(STATUS_FAILED, m_Item.configItem, m_Item.actionType);
		}
		else
		{
			log(&CFM0,DLT_LOG_ERROR,"cfgmgr_ASFAppClient::onSetPrivateItemError -- m_pUpdateManager pointer is NULL");
		}
	}
	else
	{
		log(&CFM0,DLT_LOG_ERROR,"cfgmgr_ASFAppClient::onSetPrivateItemError -- ACTID mismatch");
	}
}

/***********************************************************************************************/
/* 						Function called on request of set Private Item Success cases           */
/***********************************************************************************************/
tVoid cfgmgr_ASFAppClient::onSetPrivateItemResponse(const ::boost::shared_ptr< ConfigDataProxy >& proxy, const ::boost::shared_ptr< SetPrivateItemResponse >& response)
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_ASFAppClient::onSetPrivateItemResponse");

	Cfgmgr_ItemStatus_T status = (Cfgmgr_ItemStatus_T) response->getStatus();

	if (m_Item.ActID == response->getAct())
	{
		if(NULL != m_pUpdateManager)
		{
			m_pUpdateManager->onSetPrivateItem(status, m_Item.configItem, m_Item.actionType);
		}
		else
		{
			log(&CFM0,DLT_LOG_ERROR,"cfgmgr_ASFAppClient::onSetPrivateItemResponse -- m_pUpdateManager pointer is NULL");
		}
	}
	else
	{
		log(&CFM0,DLT_LOG_ERROR,"cfgmgr_ASFAppClient::onSetPrivateItemResponse -- ACTID mismatch");
	}
}

/***********************************************************************************************/
/* 						Function called on request of get Private Item Error cases             */
/***********************************************************************************************/
tVoid cfgmgr_ASFAppClient::onGetPrivateItemError(const ::boost::shared_ptr< ConfigDataProxy >& proxy, const ::boost::shared_ptr< GetPrivateItemError >& error)
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_ASFAppClient::onGetPrivateItemError");

	ConfigData_T tempData;
	ConfigItem_T data;

	tempData.key = "";
	tempData.value = "";
	data.push_back(tempData);

	if (m_Item.ActID == error->getAct())
	{
		if(NULL != m_pUpdateManager)
		{
			m_pUpdateManager->onGetPrivateItem(STATUS_FAILED, data, m_Item.configItem, m_Item.actionType);
		}
		else
		{
			log(&CFM0,DLT_LOG_ERROR,"cfgmgr_ASFAppClient::onGetPrivateItemError -- m_pUpdateManager pointer is NULL");
		}
	}
	else
	{
		log(&CFM0,DLT_LOG_ERROR,"cfgmgr_ASFAppClient::onGetPrivateItemError -- ACTID mismatch");
	}
}

/***********************************************************************************************/
/* 						Function called on request of get Private Item Success cases           */
/***********************************************************************************************/
tVoid cfgmgr_ASFAppClient::onGetPrivateItemResponse(const ::boost::shared_ptr< ConfigDataProxy >& proxy, const ::boost::shared_ptr< GetPrivateItemResponse >& response)
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_ASFAppClient::onGetPrivateItemResponse");

	Cfgmgr_ItemStatus_T status = (Cfgmgr_ItemStatus_T) response->getStatus();

	if (m_Item.ActID == response->getAct() )
	{
		if(NULL != m_pUpdateManager)
		{
			::std::vector< ConfigInfo > configData = response->getPrivateItem();

			ConfigItem_T itemList;
			ConfigData_T data;

			for(tUInt16 count = CFGMGR_ZERO; count < configData.size(); count++)
			{
				data.key  = configData.at(count).getKey();
				data.value = configData.at(count).getValue();

				itemList.push_back(data);
			}

			m_pUpdateManager->onGetPrivateItem(status, itemList, m_Item.configItem, m_Item.actionType);
		}
		else
		{
			log(&CFM0,DLT_LOG_ERROR,"cfgmgr_ASFAppClient::onGetPrivateItemResponse -- m_pUpdateManager pointer is NULL");
		}
	}
	else
	{
		log(&CFM0,DLT_LOG_ERROR,"cfgmgr_ASFAppClient::onGetPrivateItemResponse -- ACTID mismatch");
	}
}

/***********************************************************************************************/
/* 						Function called on request of Exchange Descmo --  Error cases          */
/***********************************************************************************************/
tVoid cfgmgr_ASFAppClient::onExchangeDESCMOError(const ::boost::shared_ptr< ConfigDataProxy >& proxy, const ::boost::shared_ptr< ExchangeDESCMOError >& error)
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_ASFAppClient::onExchangeDESCMOError");

	if (m_Item.ActID == error->getAct() )
	{
		if(NULL != m_pUpdateManager)
		{
			m_pUpdateManager->onExchangeDESCMO(STATUS_FAILED, "STATUS_FAILED",m_Item.configItem); //TODO: "STATUS_FAILED" To be handled in a better way
		}
		else
		{
			log(&CFM0,DLT_LOG_ERROR,"cfgmgr_ASFAppClient::onExchangeDESCMOError -- m_pUpdateManager pointer is NULL");
		}
	}
	else
	{
		log(&CFM0,DLT_LOG_ERROR,"cfgmgr_ASFAppClient::onExchangeDESCMOError -- ACTID mismatch");
	}
}

/***********************************************************************************************/
/* 						Function called on request of Exchange Descmo --  Success cases        */
/***********************************************************************************************/
tVoid cfgmgr_ASFAppClient::onExchangeDESCMOResponse(const ::boost::shared_ptr< ConfigDataProxy >& proxy, const ::boost::shared_ptr< ExchangeDESCMOResponse >& response)
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_ASFAppClient::onExchangeDESCMOResponse called");

	if (m_Item.ActID == response->getAct() )
	{
		Cfgmgr_ItemStatus_T status = (Cfgmgr_ItemStatus_T) response->getStatus();

		tStr xmlData = response->getOutPrivateItemXML();

		if(NULL != m_pUpdateManager)
		{
			m_pUpdateManager->onExchangeDESCMO(status, xmlData, m_Item.configItem);
		}
		else
		{
			log(&CFM0,DLT_LOG_ERROR,"cfgmgr_ASFAppClient::onExchangeDESCMOResponse -- m_pUpdateManager pointer is NULL");
		}
	}
	else
	{
		log(&CFM0,DLT_LOG_ERROR,"cfgmgr_ASFAppClient::onExchangeDESCMOResponse -- ACTID mismatch");
	}
}	 

/***********************************************************************************************/
/* 						Function called on to create the Proxy for dbus communiation           */
/***********************************************************************************************/
::boost::shared_ptr<ConfigDataProxy > cfgmgr_ASFAppClient::createProxy(tStr Dbus_Name)
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_ASFAppClient::createProxy with Dbus_Name - ", Dbus_Name.c_str());

	::boost::shared_ptr<ConfigDataProxy > proxy = (ConfigDataProxy::createProxy("configDataPort",
			Dbus_Name,
			"/de/bosch/cm/ConfigManager/ConfigDataInterface",
			DBUS_BUS_SESSION,
			*this));

	return proxy;
}

/***********************************************************************************************/
/* To send updates of private data to Native Application								    	*/
/***********************************************************************************************/
tVoid cfgmgr_ASFAppClient::updatePrivateData(DescmoState_T configItem, ConfigItem_T data, ::boost::shared_ptr<ConfigDataProxy > proxy)
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_ASFAppClient::updatePrivateData - Send Private Data with itemName - ", configItem.profileName.c_str());

	::std::vector< ConfigInfo > configData;

	ConfigInfo configInfo;

	for(tUInt16 count = CFGMGR_ZERO; count < data.size(); count++)
	{
		configInfo.setKey(data.at(count).key);
		configInfo.setValue(data.at(count).value);

		configData.push_back(configInfo);
	}

	if(NULL != proxy)
	{
		log(&CFM0,DLT_LOG_DEBUG,"cfgmgr_ASFAppClient::updatePrivateData, sendSetPrivateItemRequest called");
		m_Item.ActID = proxy->sendSetPrivateItemRequest(*this, configItem.profileName, configData);
		m_Item.configItem = configItem;
		m_Item.actionType = "SET";
	}
	else
	{
		log(&CFM0,DLT_LOG_ERROR,"cfgmgr_ASFAppClient::updatePrivateData -- proxy pointer is NULL");
	}

}

/***********************************************************************************************/
/* 	To get the private data from the Native application												*/
/***********************************************************************************************/
tVoid cfgmgr_ASFAppClient::getPrivateData(DescmoState_T configItem, ::boost::shared_ptr<ConfigDataProxy > proxy)
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_ASFAppClient::getPrivateData - Get Private Data with itemName - ", configItem.profileName.c_str());

	if(NULL != proxy )
	{
		log(&CFM0,DLT_LOG_DEBUG,"cfgmgr_ASFAppClient::getPrivateData, sendGetPrivateItemRequest called");
		m_Item.ActID = proxy->sendGetPrivateItemRequest(*this, configItem.profileName);
		m_Item.configItem = configItem;
		m_Item.actionType = "GET";
	}
	else
	{
		log(&CFM0,DLT_LOG_ERROR,"cfgmgr_ASFAppClient::getPrivateData -- proxy pointer is NULL");
	}
}

/***********************************************************************************************/
/* Send updates of public data to the application												*/
/***********************************************************************************************/
tVoid cfgmgr_ASFAppClient::updatePublicData(tStr itemName, ConfigItem_T data, ::boost::shared_ptr<ConfigDataProxy > proxy)
{
	log(&CFM0,DLT_LOG_INFO,"cfgmgr_ASFAppClient::updatePublicData - Send Public Data with itemName - ", itemName.c_str());

	::std::vector< ConfigInfo > configData;

	ConfigInfo configInfo;

	EnItemsStatus status = EnItemsStatus__SUCCESS;

	DescmoState_T configItem;

	for(tUInt16 count = CFGMGR_ZERO; count < data.size(); count++)
	{
		configInfo.setKey(data.at(count).key);
		configInfo.setValue(data.at(count).value);

		configData.push_back(configInfo);
	}

	if(NULL != proxy)
	{
		log(&CFM0,DLT_LOG_DEBUG,"cfgmgr_ASFAppClient::updatePublicData, sendUpdatePublicConfigItemRequest called");
		proxy->sendUpdatePublicConfigItemRequest(itemName, configData, status);
	}
	else
	{
		log(&CFM0,DLT_LOG_ERROR,"cfgmgr_ASFAppClient::updatePublicData -- proxy pointer is NULL");
	}
}

/***********************************************************************************************/
/* To send XML data to Managed Applications via AGW												*/
/***********************************************************************************************/
tVoid cfgmgr_ASFAppClient::SendXMLData(DescmoState_T configItem, ::boost::shared_ptr<ConfigDataProxy > proxy)
{

	log(&CFM0,DLT_LOG_INFO,"cfgmgr_ASFAppClient::SendXMLData  - Send XML Data with itemName - ",configItem.profileName.c_str(), ",xmlData - ", configItem.xmlData.c_str(), ",xmlSignature - ", configItem.xmlDataSignature.c_str());

	if(NULL != proxy)
	{
		log(&CFM0,DLT_LOG_DEBUG,"cfgmgr_ASFAppClient::SendXMLData, sendExchangeDESCMORequest called");
		m_Item.ActID = proxy->sendExchangeDESCMORequest(*this, configItem.profileName, configItem.xmlData, configItem.xmlDataSignature);
		m_Item.configItem = configItem;
		m_Item.actionType = "";
	}
	else
	{
		log(&CFM0,DLT_LOG_ERROR,"cfgmgr_ASFAppClient::SendXMLData -- proxy pointer is NULL");
	}
}

}
