/* ***************************************************************************************
	* FILE:          clRouterPlugin.cpp
	* SW-COMPONENT:  avdecc_appl_plugins
	* DESCRIPTION:  clRouterPlugin.cpp is part of avdecc_appl_plugins library
	* COPYRIGHT:  (c) 2020-21 Robert Bosch Car Multimedia GmbH
	* HISTORY: 
	* AUTHOR:  
	* REVISION: 
	*
	* The reproduction, distribution and utilization of this file as well as the
	* communication of its contents to others without express authorization is
	* prohibited. Offenders will be held liable for the payment of damages.
	* All rights reserved in the event of the grant of a patent, utility model or design.
	*
	*************************************************************************************** */
	/*****************************************************************
	| includes
	|----------------------------------------------------------------*/
	#include <clRouterPlugin.h>
	#include "plugin_trace.h"
    #include "MediaRouter.h"
	#include "clControlMapping.h"
	#include "clPluginDataProvider.h"
	#include "PluginDataType.h"
	#include <sstream>
	#include "PluginServerHandler.h"
	
 	#ifdef DP_DATAPOOL_ID
	#define DP_S_IMPORT_INTERFACE_FI
	#include "dp_generic_if.h"
	#include "dp_hmi_11_if.h"
	#endif
	
	#include "PluginEvtListenerBase.h"
    #include "PluginEventBase.h"
	#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
	#define ETG_DEFAULT_TRACE_CLASS  TR_CLASS_PLUGIN_ROUTER
	#include "trcGenProj/Header/clRouterPlugin.cpp.trc.h"
	#endif // VARIANT_S_FTR_ENABLE_TRC_GEN

#define CONTROL_ID 0
#define CONTROL_VAL 1
#define USB_CONNECTED 1

#define CTRLBLOCK_STREAM1_ROUTER_TOTALPLAYTIME "RouterSteam1TotalPlayTime"
#define CTRLBLOCK_STREAM1_ROUTER_ELAPSEDTIME "RouterStream1ElapsedTime"
#define CTRLBLOCK_ROUTER_DATABASE "RouterMediaDbStatus"
#define CTRLBLOCK_ROUTER_USBSTATUS "RouterUSBStatus"
#define CTRLBLOCK_STREAM1_ROUTER_VIDEONAME "RouterStream1VideoName"
#define CTRLBLOCK_STREAM1_ROUTER_VIDEOEPISODE "RouterStream1VideoEpisode"
#define CTRLBLOCK_ROUTER_STREAM1_PLAYBACKACTION "RouterStream1PlayBackAction"
#define CTRLBLOCK_ROUTER_MEDIA_CHECKSUM_STATUS "RouterUSBChecksumStatus"
#define CTRLBLOCK_ROUTER_MEDIA_DB_STATUS "RouterDBStatus"
#define CTRLBLOCK_ROUTER_VIDEO_REPEAT_MODE "RouterUSBVideoRepeatMode"
#define CTRLBLOCK_ROUTER_SIGNAL_SOURCE_CHANGE "RouterSignalSourceChange"

#define CTRLBLOCK_ROUTER_STREAM3_PLAYBACKACTION "RouterStream3PlayBackAction"
#define CTRLBLOCK_STREAM3_ROUTER_TOTALPLAYTIME "RouterSteam3TotalPlayTime"
#define CTRLBLOCK_STREAM3_ROUTER_ELAPSEDTIME "RouterStream3ElapsedTime"
#define CTRLBLOCK_STREAM3_ROUTER_TRACKNAME "RouterStream3TrackName"
#define CTRLBLOCK_STREAM3_ROUTER_ARTISTNAME "RouterStream3ArtistName"


clRouterPlugin* clRouterPlugin::poSelf = NULL;

clRouterPlugin::clRouterPlugin()
{
	//Communucation Protocol Object creation logic
	m_pCommControl = new clCommunicationProtocol();
	if(m_pCommControl != NULL)
	{
		ETG_TRACE_USR4(("clRouterPlugin::clCommunicationProtocol is created"));
	}

            m_stPrimCMR.u64EntityID = 0;
            m_stPrimCMR.u32EntityIndex = 0;
            m_stPrimCMR.strEndStationName = "";
            m_stPrimCMR.strEntityName = "";
            m_stPrimCMR.end_station = 0;
            m_stPrimCMR.entity = 0;
            m_stPrimCMR.configuration = 0;
	pValue= new uint8_t;
	m_strCurrentCheckSum = "";
	m_strPrevCheckSum = "";
	m_bPlayAction = false;
	m_PlayState = MEDIA_PAUSE;
    	m_bvideosourcerestore = false;
    	m_u8TrackIndex_HighByte = 0;
    	m_u8TRackIndex_LowByte = 0;
    	m_PlayControlId =0;
    	m_dbstatus = 0;
    	m_busbStatus = 0;
			m_CabinTrackInfo.u8Region = 0;
			m_CabinTrackInfo.u8ListType = MEDIA_LISTTYPE_INVALID;
            m_CabinTrackInfo.u16TrackIndex = 0;
			m_CabinTrackInfo.AuidoPLaybackMode = Repeat_List;
			m_CabinTrackInfo.VideoPLaybackMode = Repeat_List;
			m_CabinTrackInfo.u16Filter1 = 0;
			m_CabinTrackInfo.u16Filter2 = 0;
			m_CabinTrackInfo.FolderPath =  "root";
			m_CabinTrackInfo.u16Filetype = entFileType_Video;
			m_CabinTrackInfo.SubSourceType = entSubSourceType_CMR_VIDEO;
        //vRegisterResponseCallbacks();
    
}

clRouterPlugin::~clRouterPlugin(){
	delete pValue;
	delete m_pCommControl;
}

clRouterPlugin* clRouterPlugin::getInstance()
{
	ETG_TRACE_USR4(("clRouterPlugin::getInstance"));
	if(poSelf == NULL)
	{
		ETG_TRACE_USR4(("Creating object clRouterPlugin"));
		poSelf = new clRouterPlugin();
	}
	ETG_TRACE_USR4(("Return object clRouterPlugin %d", poSelf));
	return poSelf;
	
}
void clRouterPlugin::deleteInstance()
{
	if(NULL != poSelf)
	{
		delete poSelf;
	}
}

void clRouterPlugin::vRegisterResponseCallbacks()
{
    //register for callbacks
    clControlMapping *p_controlmap = clControlMapping::getInstance();
    if(NULL != p_controlmap)
    {
        p_controlmap->setRespCallBack(CMR, MEDIA_DATA_BASE_STATUS, this, (FUNCPTR)(&clRouterPlugin::vRouterDbAvailableResponse));
        p_controlmap->setRespCallBack(CMR, STREAM1_TOTAL_PLAY_TIME, this, (FUNCPTR)(&clRouterPlugin::vStream1TotalPlayTime));
        p_controlmap->setRespCallBack(CMR, STREAM1_CURRENT_PLAY_TIME, this, (FUNCPTR)(&clRouterPlugin::vStream1CurrentPlayTime));
        p_controlmap->setRespCallBack(CMR, USB_DEVICE_STATUS, this, (FUNCPTR)(&clRouterPlugin::vRouterUsbStatusResponse));
        p_controlmap->setRespCallBack(CMR, STREAM1_VIDEO_NAME, this, (FUNCPTR)(&clRouterPlugin::vRouterStream1VideoNameResponse));
        p_controlmap->setRespCallBack(CMR, STREAM1_VIDEO_EPISODE, this, (FUNCPTR)(&clRouterPlugin::vRouterStream1VideoEpisodeResponse));
        p_controlmap->setRespCallBack(CMR, STREAM1_PLAY_ACTION, this, (FUNCPTR)(&clRouterPlugin::vRouterStream1PlayBackActionResponse));
        p_controlmap->setRespCallBack(CMR, STREAM1_SEEK_TO, this, (FUNCPTR)(&clRouterPlugin::vRouterStream1SeekToResponse));
        p_controlmap->setRespCallBack(CMR, MEDIA_CHECKSUM_STATUS, this, (FUNCPTR)(&clRouterPlugin::vCheckSumStatusResponse));
        p_controlmap->setRespCallBack(CMR, STREAM1_CURRENT_TRACK_INDEX_HIGH_BYTE, this, (FUNCPTR)(&clRouterPlugin::vStream1CurTrackIndexHighByteResponse));
        p_controlmap->setRespCallBack(CMR, STREAM1_CURRENT_TRACK_INDEX_LOW_BYTE, this, (FUNCPTR)(&clRouterPlugin::vStream1CurTrackIndexLowByteResponse));
        p_controlmap->setRespCallBack(CMR, STREAM3_TOTAL_PLAY_TIME, this, (FUNCPTR)(&clRouterPlugin::vStream3TotalPlayTime));
        p_controlmap->setRespCallBack(CMR, STREAM3_CURRENT_PLAY_TIME, this, (FUNCPTR)(&clRouterPlugin::vStream3CurrentPlayTime));

        p_controlmap->setRespCallBack(CMR, STREAM3_TRACK_NAME, this, (FUNCPTR)(&clRouterPlugin::vRouterStream3TrackNameResponse));

        p_controlmap->setRespCallBack(CMR, STREAM3_ARTIST_NAME, this, (FUNCPTR)(&clRouterPlugin::vRouterStream3ArtistNameResponse));

        p_controlmap->setRespCallBack(CMR, STREAM3_PLAY_ACTION, this, (FUNCPTR)(&clRouterPlugin::vRouterStream3PlayBackActionResponse));

        p_controlmap->setRespCallBack(CMR, STREAM3_SEEK_TO, this, (FUNCPTR)(&clRouterPlugin::vRouterStream3SeekToResponse));

        p_controlmap->setRespCallBack(CMR, STREAM3_CURRENT_TRACK_INDEX_HIGH_BYTE, this, (FUNCPTR)(&clRouterPlugin::vStream3CurTrackIndexHighByteResponse));

        p_controlmap->setRespCallBack(CMR, STREAM3_CURRENT_TRACK_INDEX_LOW_BYTE, this, (FUNCPTR)(&clRouterPlugin::vStream3CurTrackIndexLowByteResponse));


    }

}
void clRouterPlugin::vEventESReadComplete()
{
	ETG_TRACE_USR4(("clRouterPlugin::vEventESReadComplete()"));
	//extract the details of the entities required
	clPluginDataProvider *ptrMain = clPluginDataProvider::getInstance();
	std::vector<stEntityDesc> vecCmR;
	if(NULL != ptrMain)
	{
		stEntityDesc stEntityDetails;
		bool CMRfound = ptrMain->getEndStationDetails(CMR, vecCmR);
		/*for(auto itr = vecCmR.begin(); itr != vecCmR.end(); itr++)
		{
			if(!isEntityDescriptorAvailable(*itr)) // Finds in m_CMREntity
			{
				m_stPrimCMR = *itr;
				vRegisterCmr(*itr);
			}
		};*/
        if(vecCmR.size()!= m_CMREntity.size())
        {
			m_CMREntity.clear();
			m_CMREntity=vecCmR;
        }
		if(true == CMRfound)
		{
			ETG_TRACE_USR4(("clRouterPlugin::vEventESReadComplete()::CMR found, Copying Entity details"));
			for_each(m_CMREntity.begin(), m_CMREntity.end(), [](stEntityDesc Ent){
	            ETG_TRACE_USR4(("clRouterPlugin::vEventESReadComplete() CMR EntityID is %d", Ent.u64EntityID));
	            ETG_TRACE_USR4(("clRouterPlugin::vEventESReadComplete() CMR EntityIndex is %d", Ent.u32EntityIndex));
	            ETG_TRACE_USR4(("clRouterPlugin::vEventESReadComplete() CMR EndStationName is %s", Ent.strEndStationName.c_str()));
	            ETG_TRACE_USR4(("clRouterPlugin::vEventESReadComplete() CMR EntityName is %s", Ent.strEntityName.c_str()));
	            ETG_TRACE_USR4(("clRouterPlugin::vEventESReadComplete() CMR end_station is %d", Ent.end_station));
	            ETG_TRACE_USR4(("clRouterPlugin::vEventESReadComplete() CMR entity is %d", Ent.entity));
	            ETG_TRACE_USR4(("clRouterPlugin::vEventESReadComplete() CMR configuration is %d", Ent.configuration));
			});

           
		}

	}

}

/******************************************************************************
*NAME		  : isEntityDescriptorAvailable
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Finds if Entity is already available in m_CMREntity
******************************************************************************/
bool clRouterPlugin::isEntityDescriptorAvailable(stEntityDesc EntityDescriptor)
{
	ETG_TRACE_USR4(("clRouterPlugin::isEntityDescriptorAvailable()"));
	for(auto itr : m_CMREntity)
	{
		if((itr.u64EntityID == EntityDescriptor.u64EntityID) && (itr.configuration != 0))
		{
			ETG_TRACE_USR4(("clRouterPlugin::isEntityDescriptorAvailable() Entity already present and registered"));
			return true;
		}
	}
	return false;
}


/******************************************************************************
*NAME		  : vRegisterCmr
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Registers for UnSolicitedResponse and CreateGetControlCommand
******************************************************************************/
void clRouterPlugin::vRegisterCmr(stEntityDesc EntityDescriptor)
{
	ETG_TRACE_USR4(("clRouterPlugin::vRegisterCmr()"));
	//register for unsolicited response for router.
	if(m_pCommControl != NULL)
	{
		m_pCommControl->RegisterUnSolicitedResponse(EntityDescriptor.end_station);
	}	
	//read control at start up
	ETG_TRACE_USR4(("Read control at start"));
        vCreateGetControlCommand(m_stPrimCMR,USB_DEVICE_STATUS);
	vCreateGetControlCommand(m_stPrimCMR,MEDIA_DATA_BASE_STATUS);

	
}

/******************************************************************************
*NAME          : vCheckSumStatusResponse
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Function for vCheckSumStatusResponse
******************************************************************************/
void clRouterPlugin::vCheckSumStatusResponse(uint16_t cmd_type,uint64_t entity_id)
{
	ETG_TRACE_USR4(("clRouterPlugin::vCheckSumStatusResponse"));
	
	if((m_stPrimCMR.configuration == 0) || (m_stPrimCMR.u64EntityID != entity_id))
	{
		ETG_TRACE_USR4(("clRouterPlugin::vCheckSumStatusResponse, ERROR! Configuration %d, EntityID %d", m_stPrimCMR.configuration, entity_id));
		return;
	}
	
	ControlDataElement controlElement;
	controlElement.configuration = m_stPrimCMR.configuration;
	controlElement.u32ControlDescIdx = MEDIA_CHECKSUM_STATUS;
	controlElement.strGetValue = "";
	if(m_pCommControl != NULL)
	{
		if(AEM_CMD_GET_CONTROL /*25*/ == cmd_type)
		{
			controlElement.puintGetValue = pValue;
			controlElement.u16Readlen = 512;
			m_pCommControl->GetControlValue(controlElement);
			ETG_TRACE_USR4(("vCheckSumStatusResponse from get control== controlData.strGetValue=%s  ",controlElement.strGetValue.c_str()));
		}
		else if (AEM_CMD_SET_CONTROL /*24*/ == cmd_type)
		{
			m_pCommControl->GetUnsolicitedValue(controlElement);
			ETG_TRACE_USR4(("vCheckSumStatusResponse == controlData.strGetValue=%s  ",controlElement.strGetValue.c_str()));
		}
	}
	m_strCurrentCheckSum =  controlElement.strGetValue;
	dp_tclCMRDpCRC _oDpCRC;
	char dpcrcread[255] = {0};
	_oDpCRC.s32GetData(dpcrcread,255);
	m_strPrevCheckSum.assign(dpcrcread);
	ETG_TRACE_USR4(("DP read CRC = %s ",dpcrcread));
	bool bCRCStatus = false;
	ETG_TRACE_USR4(("vCheckSumStatusResponse DP stored checksum  = %s  ",m_strPrevCheckSum.c_str())); 
	if(m_strCurrentCheckSum == m_strPrevCheckSum)
	{
		bCRCStatus = true;
	}
	else if (m_strCurrentCheckSum != "0")
	{
		ETG_TRACE_USR4(("vCheckSumStatusResponse new CheckSum is not equal to Prev CheckSum"));
	}	
	else
	ETG_TRACE_USR4(("CRC Update is Zero "));		
	ETG_TRACE_USR4(("CRCStatus = %d  ", bCRCStatus));
	if(m_busbStatus == USB_CONNECTED)
	{
		PluginServerHandler::getInstance()->sendsig_CRCCheckStatusUpdate(bCRCStatus,entSourceType_CMR);
		vSendPluginResponseToHmi(controlElement.u32ControlDescIdx,bCRCStatus);
	}
	else
	ETG_TRACE_USR4(("CRC Update:Device not connected"));
}


void clRouterPlugin::vStream1TotalPlayTime(uint16_t cmd_type,uint64_t entity_id)
{
   ETG_TRACE_USR4(("clRouterPlugin::vStream1TotalPlayTime"));
	
	if((m_stPrimCMR.configuration == 0) || (m_stPrimCMR.u64EntityID != entity_id))
	{
		ETG_TRACE_USR4(("clRouterPlugin::vStream1TotalPlayTime, ERROR! Configuration %d, EntityID %d", m_stPrimCMR.configuration, entity_id));
		return;
	}
	
	ControlDataElement controlElement;
	controlElement.configuration = m_stPrimCMR.configuration;
	controlElement.u32ControlDescIdx = STREAM1_TOTAL_PLAY_TIME;
	//for string redlen = 255
	//Get value of total time for the current track, this is received only once when a track is changed
	if(m_pCommControl != NULL)
	{
		m_pCommControl->GetUnsolicitedValue(controlElement);
	}
	ETG_TRACE_USR4(("vStream1TotalPlayTime == controlData.strGetValue=%s  ",controlElement.strGetValue.c_str()));
	
	double TotalTimevalue = vConvertTimeInSecs(controlElement.strGetValue);
	if (TotalTimevalue != 0)
	{
            //send the string data to HMI
            vSendPluginResponseToHmi(controlElement.u32ControlDescIdx,controlElement.strGetValue);
	    if(m_bvideosourcerestore)
            {
                vCreateSetControlCommand(STREAM1_PLAY_ACTION,MEDIA_PAUSE);
                m_bvideosourcerestore = false;
                vCreateGetControlCommand(m_stPrimCMR,STREAM1_CURRENT_PLAY_TIME);
            }
	}
	else 
	{	
                if ((m_bPlayAction == false)&& (m_PlayState == MEDIA_PLAY) && (m_CabinTrackInfo.u16TrackIndex != TRACKINDEX_UNAVAILABLE))
		{
			
			if (m_CabinTrackInfo.VideoPLaybackMode != Repeat_Track)
			{
				ETG_TRACE_USR4(("vStream1TotalPlayTime : Zero Update to fetch the next track index for continous playback \n"));
				PluginServerHandler::getInstance()->sendsig_ReqTrackIndex(MEDIA_NEXTTRACKINDEXREQ,entSourceType_CMR, m_CabinTrackInfo.u8ListType, m_CabinTrackInfo.u16Filter1,m_CabinTrackInfo.u16Filter2,m_CabinTrackInfo.u16TrackIndex,m_CabinTrackInfo.u8Region,m_CabinTrackInfo.FolderPath);
			}
			else
			PluginServerHandler::getInstance()->sendsig_ReqTrackIndex(MEDIA_CURRENTTRACKINDEXREQ,entSourceType_CMR, m_CabinTrackInfo.u8ListType, m_CabinTrackInfo.u16Filter1,m_CabinTrackInfo.u16Filter2,m_CabinTrackInfo.u16TrackIndex,m_CabinTrackInfo.u8Region,m_CabinTrackInfo.FolderPath);
		}
		else
		{
			ETG_TRACE_USR4(("vStream1TotalPlayTime : Ignore the update of the total play time \n"));
		}
	}	
}

/************************************************************************
 *FUNCTION: 		vConvertTimeInSecs
 *DESCRIPTION: 		Time string(hh:mm:ss) is passed as parameter to function getSlidertime,extracting hh,mm,ss separately to string and
                        converting to double using atof and converting hours(*3600) and mins(*60) to seconds and summing up those to totaltime and returning the totaltime
 *REQUIREMENT ID:	SWFL-2891,SWFL-2889
 *PARAMETER:		string
 *RETURNVALUE: 		double
 ************************************************************************/
double clRouterPlugin::vConvertTimeInSecs(std::string timevalue)
{
	ETG_TRACE_USR4(("clRouterPlugin::vConvertTimeInSecs hh:mm:ss : %s", timevalue.c_str()));

	   vector <string> timetokens;
	   string hr = "";
	   string min = "";
	   string sec = "";
	   double hr1 = 0;
	   double hour = 0;
	   double min1 = 0;
	   double minute = 0;
	   double sec1 = 0;
	   double second = 0;
	   double totaltime = 0;
	   stringstream check1(timevalue);
	   string intermediate = "";
	   while (getline(check1, intermediate, ':'))
	   {
	      timetokens.push_back(intermediate);
	   }
	   ETG_TRACE_USR4(("clRouterPlugin::vConvertTimeInSecs timetokens.size() : %d", timetokens.size()));
	   if (timetokens.size() >= 1)
	   {
	      hr = timetokens[0];
	      hr1 = atof(hr.c_str());
	      hour = hr1 * 3600;
	      min = timetokens[1];
	      min1 = atof(min.c_str());
	      minute = min1 * 60;
	      sec = timetokens[2];
	      sec1 = atof(sec.c_str());
	      second = sec1;
	      totaltime = hour + minute + second;
	      ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vConvertTimeInSecs totaltime : %d", totaltime));
	      timetokens.clear();//clearing the vector
	   }
	   return totaltime;
}




void clRouterPlugin::vStream1CurrentPlayTime(uint16_t cmd_type,uint64_t entity_id)
{
    ETG_TRACE_USR4(("clRouterPlugin::vStream1CurrentPlayTime"));
	if((m_stPrimCMR.configuration == 0) || (m_stPrimCMR.u64EntityID != entity_id))
	{
		ETG_TRACE_USR4(("clRouterPlugin::vStream1CurrentPlayTime, ERROR! Configuration %d, EntityID %d", m_stPrimCMR.configuration, entity_id));
		return;
	}
	
	ControlDataElement controlElement;
	controlElement.configuration = m_stPrimCMR.configuration;
	controlElement.u32ControlDescIdx = STREAM1_CURRENT_PLAY_TIME;
	
	//Get value of current time for the current track	
	if(m_pCommControl != NULL)
	{
            if(AEM_CMD_GET_CONTROL /*25*/ == cmd_type)
            {
                    controlElement.puintGetValue = pValue;
                    controlElement.u16Readlen = 512;
                    m_pCommControl->GetControlValue(controlElement);
                    ETG_TRACE_USR4(("vStream1CurrentPlayTime  get control== controlData.strGetValue=%s  ",controlElement.strGetValue.c_str()));
            }
            else if (AEM_CMD_SET_CONTROL /*24*/ == cmd_type)
            {
                    m_pCommControl->GetUnsolicitedValue(controlElement);
                    ETG_TRACE_USR4(("vStream1CurrentPlayTime == controlData.strGetValue=%s  ",controlElement.strGetValue.c_str()));
            }
	}

	//send the string data to HMI
	vSendPluginResponseToHmi(controlElement.u32ControlDescIdx,controlElement.strGetValue);

}


void clRouterPlugin::vRouterDbAvailableResponse(uint16_t cmd_type,uint64_t entity_id)
{
    ETG_TRACE_USR4(("clRouterPlugin::vRouterDbAvailableResponse"));
    if((m_stPrimCMR.configuration == 0) || (m_stPrimCMR.u64EntityID != entity_id))
	{
		ETG_TRACE_USR4(("clRouterPlugin::vRouterDbAvailableResponse, ERROR! Configuration %d, EntityID %d", m_stPrimCMR.configuration, entity_id));
		return;
	}
	
	ControlDataElement controlElement;
	controlElement.configuration = m_stPrimCMR.configuration;
	controlElement.u32ControlDescIdx = MEDIA_DATA_BASE_STATUS;
	controlElement.uint8Value = 0;
	
	if(m_pCommControl != NULL)
	{
		if (AEM_CMD_GET_CONTROL /*25*/ == cmd_type)
		{
			controlElement.puintGetValue = pValue;
			controlElement.u16Readlen = 1;
			m_pCommControl->GetControlValue(controlElement);
			ETG_TRACE_USR4(("clRouterPlugin::vRouterDbAvailableResponse from Get Control  = controlElement.uint8Value %d  ", controlElement.uint8Value));
			uint16_t u16_dbState = controlElement.uint8Value;
			m_dbstatus = controlElement.uint8Value;
			ETG_TRACE_USR4(("Db State from Get Control = controlElement.uint8Value =%d  ",controlElement.uint8Value));
			DBStateStruct DBStruct(u16_dbState,entSourceType_CMR);
			PluginServerHandler::getInstance()->setDBState(DBStruct);
			//PluginServerHandler::getInstance()->sendSig_DBStatusSignalUpdate(controlElement.uint8Value ,entSourceType_CMR);
			vSendPluginResponseToHmi(controlElement.u32ControlDescIdx, controlElement.uint8Value);
		}
		else if (AEM_CMD_SET_CONTROL /*24*/ == cmd_type)
		{
			m_pCommControl->GetUnsolicitedValue(controlElement);
			ETG_TRACE_USR4(("clRouterPlugin::vRouterDbAvailableResponse from Unsolicited response = controlElement.uint8Value %d   ", controlElement.uint8Value));
			uint16_t u16_dbState = controlElement.uint8Value;
			ETG_TRACE_USR4(("Db State from Unsolicited response = controlElement.uint8Value =%d  ",controlElement.uint8Value));
			DBStateStruct DBStruct(u16_dbState,entSourceType_CMR);
			PluginServerHandler::getInstance()->setDBState(DBStruct);
			//PluginServerHandler::getInstance()->sendSig_DBStatusSignalUpdate(controlElement.uint8Value ,entSourceType_CMR);
			vSendPluginResponseToHmi(controlElement.u32ControlDescIdx, controlElement.uint8Value);
		}
		else
		{
			ETG_TRACE_USR4(("clRouterPlugin::vRouterDbAvailableResponse, Command type is not matching"));
		}
                if((controlElement.uint8Value == 0)) //DB state invalide
                m_bPlayAction = false;
	}
}

void clRouterPlugin::vRouterUsbStatusResponse(uint16_t cmd_type,uint64_t entity_id)
{
	ETG_TRACE_USR4(("clRouterPlugin::vRouterUsbStatusResponse"));
   if((m_stPrimCMR.configuration == 0) || (m_stPrimCMR.u64EntityID != entity_id))
	{
		ETG_TRACE_USR4(("clRouterPlugin::vRouterUsbStatusResponse, ERROR! Configuration %d, EntityID %d", m_stPrimCMR.configuration, entity_id));
		return;
	}
	
	ControlDataElement controlElement;
	controlElement.configuration = m_stPrimCMR.configuration;
	controlElement.u32ControlDescIdx = USB_DEVICE_STATUS;
	controlElement.uint8Value = 0;
	if(m_pCommControl != NULL)
	{
		if (AEM_CMD_GET_CONTROL /*25*/ == cmd_type)
		{
			controlElement.puintGetValue = pValue;
			controlElement.u16Readlen = 1;
			m_pCommControl->GetControlValue(controlElement);
			m_busbStatus = controlElement.uint8Value;
			ETG_TRACE_USR4(("clRouterPlugin::vRouterUsbStatusResponse from Get Control = controlElement.uint8Value %d  ", controlElement.uint8Value));
			vSendPluginResponseToHmi(controlElement.u32ControlDescIdx,controlElement.uint8Value);
		}
		else if (AEM_CMD_SET_CONTROL /*24*/ == cmd_type)
		{
			m_pCommControl->GetUnsolicitedValue(controlElement);
			m_busbStatus = controlElement.uint8Value;
			ETG_TRACE_USR4(("clRouterPlugin::vRouterUsbStatusResponse from Unsolicited response = controlElement.uint8Value %d  ", controlElement.uint8Value));
			vSendPluginResponseToHmi(controlElement.u32ControlDescIdx,controlElement.uint8Value);
		}
		else
		{
			ETG_TRACE_USR4(("clRouterPlugin::vRouterUsbStatusResponse, Command type is not matching"));
			return;
		}
	}
    if(controlElement.uint8Value != USB_CONNECTED)
    {
        m_CabinTrackInfo.AuidoPLaybackMode = Repeat_List;
        m_CabinTrackInfo.VideoPLaybackMode = Repeat_List;
    }
}

void clRouterPlugin::vRouterStream1VideoNameResponse(uint16_t cmd_type,uint64_t entity_id)
{
	ETG_TRACE_USR4(("clRouterPlugin::vRouterStream1VideoNameResponse"));
   if((m_stPrimCMR.configuration == 0) || (m_stPrimCMR.u64EntityID != entity_id))
	{
		ETG_TRACE_USR4(("clRouterPlugin::vRouterStream1VideoNameResponse, ERROR! Configuration %d, EntityID %d", m_stPrimCMR.configuration, entity_id));
		return;
	}
	
	ControlDataElement controlElement;
	controlElement.configuration = m_stPrimCMR.configuration;
	controlElement.u32ControlDescIdx = STREAM1_VIDEO_NAME;
	
	if(m_pCommControl != NULL)
	{
		if(AEM_CMD_GET_CONTROL /*25*/ == cmd_type)
		{
			controlElement.puintGetValue = pValue;
			controlElement.u16Readlen = 512;
			m_pCommControl->GetControlValue(controlElement);
			ETG_TRACE_USR4(("vRouterStream1VideoNameResponse from get control== controlData.strGetValue=%s  ",controlElement.strGetValue.c_str()));
            vRenameifInvalidUTF8isFound(controlElement.strGetValue);
			vSendPluginResponseToHmi(controlElement.u32ControlDescIdx, controlElement.strGetValue);
		}
		else if (AEM_CMD_SET_CONTROL /*24*/ == cmd_type)
		{
			m_pCommControl->GetUnsolicitedValue(controlElement);
			ETG_TRACE_USR4(("vRouterStream1VideoNameResponse == controlData.strGetValue=%s  ",controlElement.strGetValue.c_str()));
            vRenameifInvalidUTF8isFound(controlElement.strGetValue);
			vSendPluginResponseToHmi(controlElement.u32ControlDescIdx, controlElement.strGetValue);
		}
		else
		{
			ETG_TRACE_USR4(("clRouterPlugin::vRouterStream1VideoNameResponse, Command type is not matching"));
		}
	}
	
}

void clRouterPlugin::vRouterStream1VideoEpisodeResponse(uint16_t cmd_type,uint64_t entity_id)
{
	ETG_TRACE_USR4(("clRouterPlugin::vRouterStream1VideoEpisodeNameResponse"));
   if((m_stPrimCMR.configuration == 0) || (m_stPrimCMR.u64EntityID != entity_id))
	{
		ETG_TRACE_USR4(("clRouterPlugin::vRouterStream1VideoEpisodeResponse, ERROR! Configuration %d, EntityID %d", m_stPrimCMR.configuration, entity_id));
		return;
	}
	
	ControlDataElement controlElement;
	controlElement.configuration = m_stPrimCMR.configuration;
	controlElement.u32ControlDescIdx = STREAM1_VIDEO_EPISODE;
	if(m_pCommControl != NULL)
	{
		if(AEM_CMD_GET_CONTROL /*25*/ == cmd_type)
		{
			controlElement.puintGetValue = pValue;
			controlElement.u16Readlen = 512;
			m_pCommControl->GetControlValue(controlElement);
			ETG_TRACE_USR4(("vRouterStream1VideoEpisodeNameResponse from get control== controlData.strGetValue=%s  ",controlElement.strGetValue.c_str()));
            vRenameifInvalidUTF8isFound(controlElement.strGetValue);
			vSendPluginResponseToHmi(controlElement.u32ControlDescIdx, controlElement.strGetValue);
		}
		else if (AEM_CMD_SET_CONTROL /*24*/ == cmd_type)
		{
			m_pCommControl->GetUnsolicitedValue(controlElement);
			ETG_TRACE_USR4(("vRouterStream1VideoEpisodeNameResponse == controlData.strGetValue=%s  ",controlElement.strGetValue.c_str()));
            vRenameifInvalidUTF8isFound(controlElement.strGetValue);
			vSendPluginResponseToHmi(controlElement.u32ControlDescIdx, controlElement.strGetValue);
		}
		else
		{
			ETG_TRACE_USR4(("clRouterPlugin::vRouterStream1VideoNameResponse, Command type is not matching"));
		}
	}
}

void clRouterPlugin::vRouterStream1PlayBackActionResponse(uint16_t cmd_type,uint64_t entity_id)
{
	ETG_TRACE_USR4(("clRouterPlugin::vRouterStream1PlayBackActionResponse"));
	if((m_stPrimCMR.configuration == 0) || (m_stPrimCMR.u64EntityID != entity_id))
	{
		ETG_TRACE_USR4(("clRouterPlugin::vRouterStream1PlayBackActionResponse, ERROR! Configuration %d, EntityID %d", m_stPrimCMR.configuration, entity_id));
		return;
	}
	
	ControlDataElement controlElement;
	controlElement.configuration = m_stPrimCMR.configuration;
	controlElement.u32ControlDescIdx = STREAM1_PLAY_ACTION;
	controlElement.uint8Value = 0;
	
	if(m_pCommControl != NULL)
	{
		m_pCommControl->GetUnsolicitedValue(controlElement);
	}
	ETG_TRACE_USR4(("clRouterPlugin::vRouterStream1PlayBackActionResponse = controlElement.uint8Value %d  ", controlElement.uint8Value));

        vSendPluginResponseToHmi(controlElement.u32ControlDescIdx,controlElement.uint8Value);
	
	if(controlElement.uint8Value == MEDIA_PLAY)
	{
                m_PlayState = controlElement.uint8Value;
		//explicit metadata update to hmi
		vCreateGetControlCommand(m_stPrimCMR,STREAM1_VIDEO_NAME);
		//write to datapool
		VWritedatapool();
	}
}

void clRouterPlugin::vRouterStream1SeekToResponse(uint16_t cmd_type,uint64_t entity_id)
{
	ETG_TRACE_USR4(("clRouterPlugin::vRouterStream1SeekToResponse"));
	if((m_stPrimCMR.configuration == 0) || (m_stPrimCMR.u64EntityID != entity_id))
	{
		ETG_TRACE_USR4(("clRouterPlugin::vRouterStream1SeekToResponse, ERROR! Configuration %d, EntityID %d", m_stPrimCMR.configuration, entity_id));
		return;
	}
	
	ControlDataElement controlElement;
	controlElement.configuration = m_stPrimCMR.configuration;
	controlElement.u32ControlDescIdx = STREAM1_SEEK_TO;
	controlElement.uint8Value = 0;
	controlElement.puintGetValue = pValue;
	controlElement.u16Readlen = 1;
	
	if(m_pCommControl != NULL)
	{
		m_pCommControl->GetControlValue(controlElement); //getcontrolval CHANGE
	}
	ETG_TRACE_USR4(("clRouterPlugin::vRouterUsbStatusResponse = controlElement.uint8Value %d  ", controlElement.uint8Value));
}

void clRouterPlugin::vSendPluginResponseToHmi(uint32_t u32DescIndex,std::string strData){
	ETG_TRACE_USR4(("vSendPluginResponseToHmi  =u32DescIndex=%d strData=%s  ",u32DescIndex,strData.c_str()));
	boost::shared_ptr<EventDataUtility> pluginData(EventDataUtility::newEventDataUtility());
	if(NULL != pluginData.get())
	{
		pluginData->addEventData(u32DescIndex);
		pluginData->addEventData(strData.c_str());
	}
	switch(u32DescIndex){
		case STREAM1_TOTAL_PLAY_TIME: 
				ETG_TRACE_USR4(("clRouterPlugin::vSendPluginResponseToHmi  CTRLBLOCK_STREAM1_ROUTER_TOTALPLAYTIME"));
				RecvMsgFromPlugin(PLUGIN_NAME_ROUTER,CTRLBLOCK_STREAM1_ROUTER_TOTALPLAYTIME,pluginData);
				break;
		case STREAM1_CURRENT_PLAY_TIME:
				ETG_TRACE_USR4(("clRouterPlugin::vSendPluginResponseToHmi  CTRLBLOCK_STREAM1_ROUTER_ELAPSEDTIME"));
				RecvMsgFromPlugin(PLUGIN_NAME_ROUTER,CTRLBLOCK_STREAM1_ROUTER_ELAPSEDTIME,pluginData);
				break;
		case STREAM1_VIDEO_NAME:
				ETG_TRACE_USR4(("clRouterPlugin::vSendPluginResponseToHmi  CTRLBLOCK_STREAM1_ROUTER_VIDEONAME"));
				RecvMsgFromPlugin(PLUGIN_NAME_ROUTER,CTRLBLOCK_STREAM1_ROUTER_VIDEONAME,pluginData);
				break;
		case STREAM1_VIDEO_EPISODE:
				ETG_TRACE_USR4(("clRouterPlugin::vSendPluginResponseToHmi  CTRLBLOCK_STREAM1_ROUTER_VIDEOEPISODE"));
				RecvMsgFromPlugin(PLUGIN_NAME_ROUTER,CTRLBLOCK_STREAM1_ROUTER_VIDEOEPISODE,pluginData);
				break;
				
		case STREAM3_TOTAL_PLAY_TIME: 
				ETG_TRACE_USR4(("clRouterPlugin::vSendPluginResponseToHmi  CTRLBLOCK_STREAM3_ROUTER_TOTALPLAYTIME"));
				RecvMsgFromPlugin(PLUGIN_NAME_ROUTER,CTRLBLOCK_STREAM3_ROUTER_TOTALPLAYTIME,pluginData);
				break;
		case STREAM3_CURRENT_PLAY_TIME:
				ETG_TRACE_USR4(("clRouterPlugin::vSendPluginResponseToHmi  CTRLBLOCK_STREAM3_ROUTER_ELAPSEDTIME"));
				RecvMsgFromPlugin(PLUGIN_NAME_ROUTER,CTRLBLOCK_STREAM3_ROUTER_ELAPSEDTIME,pluginData);
				break;
		case STREAM3_ARTIST_NAME:
				ETG_TRACE_USR4(("clRouterPlugin::vSendPluginResponseToHmi  CTRLBLOCK_STREAM3_ROUTER_ARTISTNAME"));
				RecvMsgFromPlugin(PLUGIN_NAME_ROUTER,CTRLBLOCK_STREAM3_ROUTER_ARTISTNAME,pluginData);
				break;
		case STREAM3_TRACK_NAME:
				ETG_TRACE_USR4(("clRouterPlugin::vSendPluginResponseToHmi  STREAM3_TRACK_NAME"));
				RecvMsgFromPlugin(PLUGIN_NAME_ROUTER,CTRLBLOCK_STREAM3_ROUTER_TRACKNAME,pluginData);
				break;
		default:
				ETG_TRACE_USR4(("clRouterPlugin::vSendPluginResponseToHmi  default case"));
				break;
	}

}

void clRouterPlugin::vSendPluginResponseToHmi(uint32_t u32DescIndex, uint8_t uint8Value)
{
	ETG_TRACE_USR4(("vSendPluginResponseToHmi  =u32DescIndex=%d uint8Value=%d  ",u32DescIndex,uint8Value));
	boost::shared_ptr<EventDataUtility> pluginData(EventDataUtility::newEventDataUtility());
	if(NULL != pluginData.get())
	{
		pluginData->addEventData(u32DescIndex);
		pluginData->addEventData(uint8Value);
	}
	switch(u32DescIndex)
	{
		case USB_DEVICE_STATUS:
			ETG_TRACE_USR4(("clRouterPlugin::vSendPluginResponseToHmi  USB_DEVICE_STATUS"));
			RecvMsgFromPlugin(PLUGIN_NAME_ROUTER,CTRLBLOCK_ROUTER_USBSTATUS,pluginData);
			break;
		case STREAM1_PLAY_ACTION:
			ETG_TRACE_USR4(("clRouterPlugin::vSendPluginResponseToHmi  CTRLBLOCK_ROUTER_STREAM1_PLAYBACKACTION"));
			RecvMsgFromPlugin(PLUGIN_NAME_ROUTER,CTRLBLOCK_ROUTER_STREAM1_PLAYBACKACTION,pluginData);
			break;
		case MEDIA_CHECKSUM_STATUS:
			ETG_TRACE_USR4(("clRouterPlugin::vSendPluginResponseToHmi  MEDIA_CHECKSUM_STATUS"));
			RecvMsgFromPlugin(PLUGIN_NAME_ROUTER,CTRLBLOCK_ROUTER_MEDIA_CHECKSUM_STATUS,pluginData);
			break;
		case MEDIA_DATA_BASE_STATUS:
			ETG_TRACE_USR4(("clRouterPlugin::vSendPluginResponseToHmi  MEDIA_DATA_BASE_STATUS"));
			RecvMsgFromPlugin(PLUGIN_NAME_ROUTER,CTRLBLOCK_ROUTER_MEDIA_DB_STATUS,pluginData);
			break;
		case STREAM3_PLAY_ACTION:
			ETG_TRACE_USR4(("clRouterPlugin::vSendPluginResponseToHmi  CTRLBLOCK_ROUTER_STREAM3_PLAYBACKACTION"));
			RecvMsgFromPlugin(PLUGIN_NAME_ROUTER,CTRLBLOCK_ROUTER_STREAM3_PLAYBACKACTION,pluginData);
			break;
		case CMRSTREAM1_PLAYBACKMODE:
            ETG_TRACE_USR4(("clRouterPlugin::vSendPluginResponseToHmi  CMRSTREAM1_PLAYBACKMODE"));
			RecvMsgFromPlugin(PLUGIN_NAME_ROUTER,CTRLBLOCK_ROUTER_VIDEO_REPEAT_MODE,pluginData);
            break;
         case CMRSTREAM3_PLAYBACKMODE:
            ETG_TRACE_USR4(("clRouterPlugin::vSendPluginResponseToHmi  CMRSTREAM3_PLAYBACKMODE"));
            RecvMsgFromPlugin(PLUGIN_NAME_ROUTER,CTRLBLOCK_ROUTER_VIDEO_REPEAT_MODE,pluginData);
            break;
		case MEDIA_SIGNAL_SOURCE_CHANGE:
			ETG_TRACE_USR4(("clRouterPlugin::vSendPluginResponseToHmi  MEDIA_SIGNAL_SOURCE_CHANGE"));
			if(NULL != pluginData.get())
			{
				pluginData->addEventData(m_CabinTrackInfo.u16TrackIndex);
				pluginData->addEventData(m_CabinTrackInfo.FolderPath);
				pluginData->addEventData(m_CabinTrackInfo.u8ListType);
				pluginData->addEventData(m_CabinTrackInfo.SubSourceType);
				pluginData->addEventData(m_CabinTrackInfo.u8Region);
			}
			RecvMsgFromPlugin(PLUGIN_NAME_ROUTER,CTRLBLOCK_ROUTER_SIGNAL_SOURCE_CHANGE,pluginData);
			break;
		default:
			break;
	}
}

/******************************************************************************
*NAME          : addPluginEventHandler
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Function to get plugin event handler, part of evenbus mechanism
******************************************************************************/
std::unique_ptr<HandlerRegistration> clRouterPlugin::addPluginEventHandler()
{
    ETG_TRACE_USR4(("clRouterPlugin::addPluginEventHandler"));
    PluginEvtListenerBase<clRouterPlugin>* RouterPluginEventListener = PluginEvtListenerBase<clRouterPlugin>::getInstance();
    std::unique_ptr<HandlerRegistration> handle (EventBus::AddHandler<PluginEventBase<clRouterPlugin>>(RouterPluginEventListener));
    return std::move(handle);
}

/******************************************************************************185
*NAME          : addPluginEventHandler
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Function to get plugin event, part of evenbus mechanism
******************************************************************************/
std::unique_ptr<Event> clRouterPlugin::getPluginEvent(Object* sender, const std::string msg, boost::shared_ptr<EventDataUtility>evalue)
{
    ETG_TRACE_USR4(("clRouterPlugin::getPluginEvent"));
    std::unique_ptr<PluginEventBase<clRouterPlugin>> RouterPluginEvent(new PluginEventBase<clRouterPlugin>(sender,this,msg,evalue));
    return std::move(RouterPluginEvent);
}

/******************************************************************************
*NAME          : addPluginEventHandler
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Function to get plugin list event, part of evenbus mechanism
******************************************************************************/
std::unique_ptr<Event> clRouterPlugin::getPluginListEvent(Object* sender, const std::string msg, boost::shared_ptr<EventDataUtility>eData,
                                                                const boost::shared_ptr<EventListDataUtility>& eListData)
{
    ETG_TRACE_USR4(("clRouterPlugin::getPluginListEvent"));
    std::unique_ptr<PluginEventBase<clRouterPlugin>> RouterPluginEvent(new PluginEventBase<clRouterPlugin>(sender,this,msg,eData,eListData));
    return std::move(RouterPluginEvent);
}

/******************************************************************************
*NAME          : setPluginData
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Based on the key, the corresponding function is invoked
******************************************************************************/
void clRouterPlugin::setPluginData(std::string key, const boost::shared_ptr<EventDataUtility>& eValue)
{	
	if(NULL != eValue.get())
	{
		uint8_t u8ControlVal = 0;
		uint8_t u8ControlId = 0;
		std::vector<EventDataItem*> dataItems = eValue->getData();
		if(dataItems.size()>1)
		{
			EventDataItem* eventDataItemControl =  dataItems.at(CONTROL_ID);
			EventDataItem* eventDataItemControlVal =  dataItems.at(CONTROL_VAL);
			if(NULL != eventDataItemControl)
			{
				EventDataItem::Data eventData = eventDataItemControl->getData();
				if(EventDataItem::UINT8 == eventData._type)
				{
					u8ControlId = eventData._value._uint8Value;
				}
			}
			if(NULL != eventDataItemControlVal)
			{
				EventDataItem::Data eventData = eventDataItemControlVal->getData();
				if(EventDataItem::UINT8 == eventData._type)
				{
					u8ControlVal = eventData._value._uint8Value;
				}
			}
			
			ETG_TRACE_USR4(("setPluginData  =u8ControlId=%d  u16ControlVal=%d  ",u8ControlId,u8ControlVal));
			if ((u8ControlId == STREAM1_PLAY_ACTION) || (u8ControlId == STREAM3_PLAY_ACTION))
			{
				if(!m_bPlayAction)
				{
					if ((u8ControlVal == MEDIA_PAUSE)||(u8ControlVal == MEDIA_PLAY))
                    {
						 m_PlayState = u8ControlVal;
						 if ((m_CabinTrackInfo.u16TrackIndex != TRACKINDEX_UNAVAILABLE))
						 {
							 vCreateSetControlCommand(u8ControlId,u8ControlVal);							 
						 }
						 else
						 {
							 vCheckCRCStatus();
                         }                  
                    }
					else
					{
						m_PlayControlId = u8ControlId;
						m_bPlayAction = true;
						vHandlePlaybackAction(dataItems);
					}
				}
				else
				{
					ETG_TRACE_USR4(("====set_PluginData  ignore already one playaction under process =u16ControlId=%d  u16ControlVal=%d  ",u8ControlId,u8ControlVal));
				}
			}				
			else if ((u8ControlId == CMRSTREAM1_PLAYBACKMODE)||(u8ControlId == CMRSTREAM3_PLAYBACKMODE))
			{
				//Call the function which updates the current play back mode setting property
				vHandlePlaybackMode(dataItems);							
			}
			else if(u8ControlId == CMR_USB_DB_CRC_UPDATE)
			{
				//Call the function which update the USB entities to HMI
				vUsbentitiesupdate();
			}
			else			
			vCreateSetControlCommand(u8ControlId,u8ControlVal);//call to setcontrol			
		}

	}
}
void clRouterPlugin::vHandlePlaybackMode(std::vector<EventDataItem*> dataItems)
{
	
	ETG_TRACE_USR4(("clRouterPlugin::vHandlePlaybackMode"));
	uint8_t selectionValues[3] = {0,0,0},audiovideo_playbackmode;
	uint16_t u16ItemPosition = 0;
	if(dataItems.size()>1)
	{
	  for(uint8_t index = 0; (index<3)&&(index < (dataItems.size())); index++)
      {
		 EventDataItem* eventDataItem =  dataItems.at(index);
         if(NULL != eventDataItem)
         {
			const EventDataItem::Data& eventData = eventDataItem->getData();
            if(EventDataItem::UINT8 == eventData._type)
			{
				selectionValues[index] = eventData._value._uint8Value;
			}
		 }
	  }	
	}
	
	//playbackmode,sourcetype,regionid,subsourcetype
	ETG_TRACE_USR4(("clRouterPlugin::PlaybackMode value:%d", selectionValues[1]));
	
	if(selectionValues[0] == CMRSTREAM1_PLAYBACKMODE || m_CabinTrackInfo.u8ListType == MEDIA_LISTTYPE_FOLDER)
	{
		m_CabinTrackInfo.VideoPLaybackMode = selectionValues[1];
		PlayBackModeStruct ModeStruct(selectionValues[1],entSourceType_CMR,selectionValues[2],entSubSourceType_CMR_VIDEO);
    	PluginServerHandler::getInstance()->setPlayBackMode(ModeStruct);
		vSendPluginResponseToHmi(CMRSTREAM1_PLAYBACKMODE,selectionValues[1]);
	}
	else
	{
		m_CabinTrackInfo.AuidoPLaybackMode = selectionValues[1];
		PlayBackModeStruct ModeStruct(selectionValues[1],entSourceType_CMR,selectionValues[2],entSubSourceType_CMR_AUDIO);
    	PluginServerHandler::getInstance()->setPlayBackMode(ModeStruct);
		vSendPluginResponseToHmi(CMRSTREAM3_PLAYBACKMODE,selectionValues[1]);
	}
	
	//uint8_t Higher 4 bits for audio and Loewr 4 bits for video playback mode store to datapool
    audiovideo_playbackmode = (m_CabinTrackInfo.AuidoPLaybackMode & 0x0f)|((m_CabinTrackInfo.VideoPLaybackMode<<4) & 0xf0);
	//write to datapool
	dp_tclCMRDpPLaybackMode _oDpPLaybackMode;
	_oDpPLaybackMode.s32SetData(audiovideo_playbackmode);
	ETG_TRACE_USR4(("DP write audiovideo_playbackmode = %d ", audiovideo_playbackmode));
	
	
}
void clRouterPlugin::vHandlePlaybackAction(std::vector<EventDataItem*> dataItems)
{
	ETG_TRACE_USR4(("clRouterPlugin::HandlePlaybackAction"));
	ETG_TRACE_USR4(("clRouterPlugin::HandlePlaybackAction  dataitems.size():%d", dataItems.size()));
	uint8_t selectionValues[5] = {0,0,0,0,0};
	uint16_t selectionValues1[4] = {0,0,0,0};
	std::string Sfolderpath = "";
	if(dataItems.size()>1)
	{	
		for(uint8_t index = 0,index1=0,index2=0; index < dataItems.size() ; index++)
		{
			 EventDataItem* eventDataItem = dataItems.at(index);
			 if(NULL != eventDataItem)
			 {
				const EventDataItem::Data& eventData = eventDataItem->getData();
				switch(eventData._type){
					case EventDataItem::UINT8:
						selectionValues[index1] = eventData._value._uint8Value;
						ETG_TRACE_USR4(("clRouterPlugin::HandlePlaybackAction  eventData UINT8[%d] = %d",index1,selectionValues[index1]));
						index1++;
						break;
					case EventDataItem::UINT16:
						selectionValues1[index2] = eventData._value._uint16Value;
						ETG_TRACE_USR4(("clRouterPlugin::HandlePlaybackAction  eventData UINT16[%d] = %d",index2,selectionValues[index2]));
						index2++;
						break;
					case EventDataItem::STRING:
						Sfolderpath = eventData._value._stringValue->c_str();
						ETG_TRACE_USR4(("clRouterPlugin::HandlePlaybackAction  eventData string = %s",Sfolderpath.c_str()));
						break;
					default:
						ETG_TRACE_USR4(("clRouterPlugin::HandlePlaybackAction Wrong data Update"));
				}
			 }
		}	
		//UINT8
		m_CabinTrackInfo.u8Region = selectionValues[2];
		m_CabinTrackInfo.u8ListType = selectionValues[3];
		if (m_PlayControlId == STREAM1_PLAY_ACTION )
		{
			m_CabinTrackInfo.SubSourceType = entSubSourceType_CMR_VIDEO;
		}
		else
		{
			m_CabinTrackInfo.SubSourceType = entSubSourceType_CMR_AUDIO;;
		}
		//UINT16
		m_CabinTrackInfo.u16TrackIndex = selectionValues1[0];
		m_CabinTrackInfo.u16Filetype = selectionValues1[1];
		//m_CabinTrackInfo.u16Filter1 = selectionValues1[2];
		//m_CabinTrackInfo.u16Filter2 = selectionValues1[3];
		//string
		m_CabinTrackInfo.FolderPath = Sfolderpath;
			
		ETG_TRACE_USR4(("clRouterPlugin::HandlePlaybackAction Control Id :%d Control Value:%d List Type : %d Position = %d SubSourceType =%d ", selectionValues[0],selectionValues[1],selectionValues[3],selectionValues1[0],selectionValues[4]));
		ETG_TRACE_USR4(("clRouterPlugin::HandlePlaybackAction   u16Filetype=%d FolderPath=%s",selectionValues1[1],Sfolderpath.c_str()));
		vHandleAudioVideoPlayBackAction(selectionValues[0],selectionValues[1]);
	}
}

void clRouterPlugin::vHandleAudioVideoPlayBackAction(uint8_t u8ControlId, uint8_t u8PlayAction)
{
	ETG_TRACE_USR4(("clRouterPlugin::HandleAudioVideoPlayBackAction controlid = %d ",u8ControlId));
	switch (u8PlayAction)
	{        
            case MEDIA_NEXT :
			ETG_TRACE_USR4(("clRouterPlugin::Request for nexttrackIndex"));
			PluginServerHandler::getInstance()->sendsig_ReqTrackIndex(MEDIA_NEXTTRACKINDEXREQ,entSourceType_CMR, m_CabinTrackInfo.u8ListType, m_CabinTrackInfo.u16Filter1,m_CabinTrackInfo.u16Filter2,m_CabinTrackInfo.u16TrackIndex,m_CabinTrackInfo.u8Region,m_CabinTrackInfo.FolderPath); 
			break;
            case MEDIA_PREVIOUS :
                            ETG_TRACE_USR4(("clRouterPlugin::Request for prevtrackIndex"));
                            PluginServerHandler::getInstance()->sendsig_ReqTrackIndex(MEDIA_PREVTRACKINDEXREQ,entSourceType_CMR, m_CabinTrackInfo.u8ListType, m_CabinTrackInfo.u16Filter1,m_CabinTrackInfo.u16Filter2,m_CabinTrackInfo.u16TrackIndex,m_CabinTrackInfo.u8Region,m_CabinTrackInfo.FolderPath);
                            break;
            case MEDIA_SOURCEACTIVE :
                            vCheckCRCStatus();
                            break;
            case MEDIA_BROWSE:
                            PluginServerHandler::getInstance()->sendsig_ReqTrackIndex(MEDIA_SELECTTRACKINDEXREQ,entSourceType_CMR, m_CabinTrackInfo.u8ListType, m_CabinTrackInfo.u16Filter1,m_CabinTrackInfo.u16Filter2,m_CabinTrackInfo.u16TrackIndex,m_CabinTrackInfo.u8Region,m_CabinTrackInfo.FolderPath);
                            m_PlayState = u8PlayAction;
                            vSetStreamCurrentTrackIndex(m_CabinTrackInfo.u16TrackIndex,m_CabinTrackInfo.u16Filetype);
                            //Playback mode updates to HMI
							if(m_CabinTrackInfo.u8ListType == MEDIA_LISTTYPE_SONG)
							{
								vSendPluginResponseToHmi(CMRSTREAM3_PLAYBACKMODE,m_CabinTrackInfo.AuidoPLaybackMode);
							}
							else
							{
								vSendPluginResponseToHmi(CMRSTREAM1_PLAYBACKMODE,m_CabinTrackInfo.VideoPLaybackMode);
							}
							break;
            case MEDIA_VIDEOSOURCERESTORE :
                            //calling function for video source restoration in pause state
                            vVideosourcerestoreinpoause();
                            break;
            default:
                            vCreateSetControlCommand(u8ControlId,u8PlayAction);
                            break;
	}
}

void clRouterPlugin::vCheckCRCStatus()
{
	ETG_TRACE_USR4(("clRouterPlugin::vCheckCRCStatus "));
	if (m_strCurrentCheckSum == m_strPrevCheckSum)
	{
		//Dapool reading
		VReadCMRdatafromdatapool();
		
		/** set the track index read from DP****/
		if (m_CabinTrackInfo.u16TrackIndex != TRACKINDEX_UNAVAILABLE)
		{
			ETG_TRACE_USR4(("clRouterPlugin::Request for current track index"));
		PluginServerHandler::getInstance()->sendsig_ReqTrackIndex(MEDIA_SELECTTRACKINDEXREQ,entSourceType_CMR, m_CabinTrackInfo.u8ListType, m_CabinTrackInfo.u16Filter1,m_CabinTrackInfo.u16Filter2,m_CabinTrackInfo.u16TrackIndex,m_CabinTrackInfo.u8Region,m_CabinTrackInfo.FolderPath);
		vSetStreamCurrentTrackIndex(m_CabinTrackInfo.u16TrackIndex,m_CabinTrackInfo.u16Filetype);
		}
		else
		{
			ETG_TRACE_USR4(("clRouterPlugin::Request for First track index in case of track index not available "));
			PluginServerHandler::getInstance()->sendsig_ReqTrackIndex(MEDIA_FIRSTTRACKINDEXREQ,entSourceType_CMR,m_CabinTrackInfo.u8ListType, m_CabinTrackInfo.u16Filter1,m_CabinTrackInfo.u16Filter2,m_CabinTrackInfo.u16TrackIndex,m_CabinTrackInfo.u8Region,m_CabinTrackInfo.FolderPath);
		}
	}
	else
	{
		ETG_TRACE_USR4(("clRouterPlugin::Request for First track index"));
		if(m_CabinTrackInfo.u8ListType == MEDIA_LISTTYPE_INVALID)
		{
			if (m_PlayControlId == STREAM1_PLAY_ACTION )
			{
				m_CabinTrackInfo.u8ListType = MEDIA_LISTTYPE_VIDEO; 
			}
			else
			{
				m_CabinTrackInfo.u8ListType = MEDIA_LISTTYPE_SONG;
			}
		}
		
		PluginServerHandler::getInstance()->sendsig_ReqTrackIndex(MEDIA_FIRSTTRACKINDEXREQ,entSourceType_CMR, m_CabinTrackInfo.u8ListType, m_CabinTrackInfo.u16Filter1,m_CabinTrackInfo.u16Filter2,m_CabinTrackInfo.u16TrackIndex,m_CabinTrackInfo.u8Region,m_CabinTrackInfo.FolderPath); 
        //Update default source playback mode to HMI
        vSendPluginResponseToHmi(CMRSTREAM1_PLAYBACKMODE,m_CabinTrackInfo.VideoPLaybackMode);
        //DP CRC write
		dp_tclCMRDpCRC _oDpCRC;
		uint8_t crclen = m_strCurrentCheckSum.length();
		char dpcrcwrite[crclen+1] ={0};
 		strncpy(dpcrcwrite,m_strCurrentCheckSum.c_str(),crclen);
		dpcrcwrite[crclen] = '\0';
		_oDpCRC.s32SetData(dpcrcwrite);
		ETG_TRACE_USR4(("DP write CRC = %s ", dpcrcwrite));
		
		m_strPrevCheckSum = m_strCurrentCheckSum;
		VclearCMRdatapool();
	}
}
/******************************************************************************
*NAME          : setPluginListData
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Based on the key, the corresponding function is invoked
******************************************************************************/
void clRouterPlugin::setPluginListData(std::string key, const boost::shared_ptr<EventDataUtility>& eData,
        const boost::shared_ptr<EventListDataUtility>& eListData){
   ETG_TRACE_USR4(("clRouterPlugin::setPluginListData"));
    
}

/************************************************************************
 *FUNCTION: 		vCreateGetControlCommand
 *DESCRIPTION: 		for creating command object
 *REQUIREMENT ID:	SWFL-2891,SWFL-2889
 *PARAMETER:		uint16_t
 *RETURNVALUE: 		void
 ************************************************************************/
void clRouterPlugin::vCreateGetControlCommand(stEntityDesc EntityDescriptor,uint8_t u8ControlId)
{
	ETG_TRACE_USR4(("====vCreateGetControlCommand  =u8ControlId=%d  ",u8ControlId));
	
	ControlDataElement controlElement;
	controlElement.configuration = EntityDescriptor.configuration;
	controlElement.u32ControlDescIdx = u8ControlId;
		
	if(m_pCommControl != NULL)
	{
		m_pCommControl->getControl(controlElement);
	}
}
/******************************************************************************
*NAME          : vCreateSetControlCommand
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Send the SetControl command to the Pluginbase by Communication control
******************************************************************************/
void clRouterPlugin::vCreateSetControlCommand(uint8_t u8ControlId,uint8_t u8ControlVal)
{	
	ControlDataElement controlElement;
	controlElement.configuration = m_stPrimCMR.configuration;
	controlElement.u32ControlDescIdx = u8ControlId;
	controlElement.uint8Value = u8ControlVal;
	ETG_TRACE_USR4(("clRouterPlugin::vCreateSetControlCommand , for setting track index u8ControlId= %d  u16ControlVal=%d  ",controlElement.u32ControlDescIdx,controlElement.uint8Value));	
	if(m_pCommControl != NULL)
	{
		m_pCommControl->setControl(controlElement);
	}
}
/************************************************************************
 *FUNCTION: 		vSetStreamCurrentTrackIndex
 *DESCRIPTION: 		for creating command object to set the STREAM1CURRENTTRACKINDEX
 *REQUIREMENT ID:	SWFL-2891,SWFL-2889
 *PARAMETER:		uint32_t, uint16_t
 *RETURNVALUE: 		void
 ************************************************************************/
 
void clRouterPlugin::vSetStreamCurrentTrackIndex(uint32_t streamTrackIndex,uint16_t Filetype)
{
	ETG_TRACE_USR4(("====vSetStreamCurrentTrackIndex  =streamTrackIndex=%d  ",streamTrackIndex));
	ETG_TRACE_USR4(("====vSetStreamCurrentTrackIndex  =m_PlayControlId=%d  ",m_PlayControlId));
	ETG_TRACE_USR4(("====vSetStreamCurrentTrackIndex  =Filetype=%d  ",Filetype));
	if (streamTrackIndex != TRACKINDEX_UNAVAILABLE)
	{
		if(m_CabinTrackInfo.u8ListType!=MEDIA_LISTTYPE_FOLDER)
		{
			if (m_PlayControlId == STREAM1_PLAY_ACTION )
			{
				vSetStream1CurrentTrackIndex (streamTrackIndex);
			}
			else if ( m_PlayControlId == STREAM3_PLAY_ACTION )
			{
				vSetStream3CurrentTrackIndex ( streamTrackIndex ) ;
			}
			else 
			{
				ETG_TRACE_USR4(("====vSetStreamCurrentTrackIndex nothing to be done "));
			}
		}
		else
		{
			vSetStreamFolderCurrentTrackIndex(streamTrackIndex,Filetype);
		}
	}
	else
	{
		ETG_TRACE_USR4(("====Invalid TRack Index Update  "));
		m_CabinTrackInfo.u16TrackIndex = streamTrackIndex;
		vCreateSetControlCommand(m_PlayControlId,MEDIA_PAUSE);
		dp_tclCMRDpListtrackindex _oDpListTrackIndex;
		_oDpListTrackIndex.s32SetData(streamTrackIndex);
		ETG_TRACE_USR4(("DP write Track Index = %d ", streamTrackIndex));
		m_bPlayAction = false;
	}
}

void clRouterPlugin::vSetStreamFolderCurrentTrackIndex(uint32_t FolderstreamTrackIndex,uint16_t FolderFiletype)
{
	ETG_TRACE_USR4(("clRouterPlugin::vSetStreamFolderCurrentTrackIndex"));

	if(FolderFiletype == m_CabinTrackInfo.u16Filetype)
	{
		if (m_PlayControlId == STREAM1_PLAY_ACTION )
		{
			vSetStream1CurrentTrackIndex (FolderstreamTrackIndex);
		}
		else if ( m_PlayControlId == STREAM3_PLAY_ACTION )
		{
			vSetStream3CurrentTrackIndex (FolderstreamTrackIndex );
		}
		else 
		{
			ETG_TRACE_USR4(("====vSetStreamCurrentTrackIndex nothing to be done "));
		}
	}
	else
	{
		m_bPlayAction = false;
		m_CabinTrackInfo.u16TrackIndex = FolderstreamTrackIndex;
		if(FolderFiletype == entFileType_Video)
		{
			m_CabinTrackInfo.SubSourceType = entSubSourceType_CMR_VIDEO;
		}
		else if(FolderFiletype == entFileType_Audio)
		{
			m_CabinTrackInfo.SubSourceType = entSubSourceType_CMR_AUDIO;
		}
		else
		{
			ETG_TRACE_USR4(("Wrong FolderFiletype type update"));
			return;
		}
		VWritedatapool();
		vSendPluginResponseToHmi(MEDIA_SIGNAL_SOURCE_CHANGE, FolderFiletype);
	}
}

void clRouterPlugin::vSetStream1CurrentTrackIndex(uint32_t stream1TrackIndex)
{
	ETG_TRACE_USR4(("====vSetStream1CurrentTrackIndex  =stream1TrackIndex=%d  ",stream1TrackIndex));
	uint32_t sendTrackIndex = stream1TrackIndex;
	ETG_TRACE_USR4(("====vSetStream1CurrentTrackIndex  =sendTrackIndex=%d  ",sendTrackIndex));
		//set the stream1currenttack index to play the recived track index from listhandler.
		uint8_t u8HighByte = (stream1TrackIndex>>8) & 0X00FF;
		m_u8TRackIndex_LowByte = (stream1TrackIndex) & 0x00FF; 	

		ETG_TRACE_USR4(("====vSetStream1CurrentTrackIndex  high byte =%d lower byte = %d ",u8HighByte,m_u8TRackIndex_LowByte));

		//set the stream1currenttack index to play the recived track index from listhandler.	
		vCreateSetControlCommand(STREAM1_CURRENT_TRACK_INDEX_HIGH_BYTE,u8HighByte);
}


/******************************************************************************
*NAME          : vStream1CurTrackIndexHighByteResponse
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Function for sending Track Index for vStream1CurTrackIndexHighByteResponse
******************************************************************************/
void clRouterPlugin::vStream1CurTrackIndexHighByteResponse(uint16_t cmd_type,uint64_t entity_id)
{
	ETG_TRACE_USR4(("clRouterPlugin::vStream1CurTrackIndexHighByteResponse"));
	
	if((m_stPrimCMR.configuration == 0) || (m_stPrimCMR.u64EntityID != entity_id))
	{
		ETG_TRACE_USR4(("clRouterPlugin::, ERROR! Configuration %d, EntityID %d", m_stPrimCMR.configuration, entity_id));
		return;
	}
	
	ControlDataElement controlElement;
	controlElement.configuration = m_stPrimCMR.configuration;
	controlElement.u32ControlDescIdx = STREAM1_CURRENT_TRACK_INDEX_HIGH_BYTE;
	controlElement.uint8Value = 0;
	
	//Getvalue for the current track index
	if(m_pCommControl != NULL)
	{
		m_pCommControl->GetUnsolicitedValue(controlElement);
	}	
	m_u8TrackIndex_HighByte = controlElement.uint8Value;
	//set control for low byte once the response of high byte is recieved 
	vCreateSetControlCommand(STREAM1_CURRENT_TRACK_INDEX_LOW_BYTE,m_u8TRackIndex_LowByte);
        vCreateSetControlCommand(STREAM1_PLAY_ACTION,MEDIA_PLAY);
	ETG_TRACE_USR4(("Track Index = High Byte  = %d  ", m_u8TrackIndex_HighByte));
}

/******************************************************************************
*NAME          : vStream1CurTrackIndexLowByteResponse
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Function for sending Track Index for vStream1CurTrackIndexHighByteResponse
******************************************************************************/
void clRouterPlugin::vStream1CurTrackIndexLowByteResponse(uint16_t cmd_type,uint64_t entity_id)
{
	ETG_TRACE_USR4(("clRouterPlugin::vStream1CurTrackIndexResponse"));

	if((m_stPrimCMR.configuration == 0) || (m_stPrimCMR.u64EntityID != entity_id))
	{
		ETG_TRACE_USR4(("clRouterPlugin::vStream1CurTrackIndexLowByteResponse, ERROR! Configuration %d, EntityID %d", m_stPrimCMR.configuration, entity_id));
		return;
	}
	
	ControlDataElement controlElement;
	controlElement.configuration = m_stPrimCMR.configuration;
	controlElement.u32ControlDescIdx = STREAM1_CURRENT_TRACK_INDEX_LOW_BYTE;
	controlElement.uint8Value = 0;
	
	//Getvalue for the current track index
	if(m_pCommControl != NULL)
	{
		m_pCommControl->GetUnsolicitedValue(controlElement);
	}
	ETG_TRACE_USR4(("Track Index Lower byte =  %d  ", controlElement.uint8Value));

	uint8_t u8LowByte = controlElement.uint8Value; 			
	uint16_t SongID = (m_u8TrackIndex_HighByte<<8)|(u8LowByte & 0X00FF);
	m_CabinTrackInfo.u16TrackIndex = SongID;
   	bool playState = entPlayState_PlaySuccess;
	uint16_t sourceType = entSourceType_CMR;
	PluginServerHandler::getInstance()->sendSig_PlayStatusSignalUpdate(playState,sourceType,SongID,m_CabinTrackInfo.u16Filetype,m_CabinTrackInfo.u8ListType,m_CabinTrackInfo.u16Filter1,m_CabinTrackInfo.u16Filter2,m_CabinTrackInfo.FolderPath);
	m_bPlayAction = false;
}
/******************************************************************************
*NAME          : vStream3TotalPlayTime
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Function for handling the total play time update for Audio
******************************************************************************/
void clRouterPlugin::vStream3TotalPlayTime(uint16_t cmd_type,uint64_t entity_id)
{
   ETG_TRACE_USR4(("clRouterPlugin::vStream3TotalPlayTime"));
	
	if((m_stPrimCMR.configuration == 0) || (m_stPrimCMR.u64EntityID != entity_id))
	{
		ETG_TRACE_USR4(("clRouterPlugin::vStream3TotalPlayTime, ERROR! Configuration %d, EntityID %d", m_stPrimCMR.configuration, entity_id));
		return;
	}
	
	ControlDataElement controlElement;
	controlElement.configuration = m_stPrimCMR.configuration;
	controlElement.u32ControlDescIdx = STREAM3_TOTAL_PLAY_TIME;
	//for string redlen = 255
	//Get value of total time for the current track, this is received only once when a track is changed
	if(m_pCommControl != NULL)
	{
		m_pCommControl->GetUnsolicitedValue(controlElement);
	}
	ETG_TRACE_USR4(("vStream3TotalPlayTime == controlData.strGetValue=%s  ",controlElement.strGetValue.c_str()));
	
	double TotalTimevalue = vConvertTimeInSecs(controlElement.strGetValue);
	if (TotalTimevalue != 0)
	//send the string data to HMI
	vSendPluginResponseToHmi(controlElement.u32ControlDescIdx,controlElement.strGetValue);
	else 
	{	
        if ((m_bPlayAction == false)&& (m_PlayState == MEDIA_PLAY) && (m_CabinTrackInfo.u16TrackIndex != TRACKINDEX_UNAVAILABLE))
		{
			if (m_CabinTrackInfo.u8ListType != MEDIA_LISTTYPE_FOLDER)
			{
				if (m_CabinTrackInfo.AuidoPLaybackMode  != Repeat_Track)
				{
					ETG_TRACE_USR4(("vStream3TotalPlayTime : Zero Update to fetch the next track index for continous playback \n"));
					PluginServerHandler::getInstance()->sendsig_ReqTrackIndex(MEDIA_NEXTTRACKINDEXREQ,entSourceType_CMR, m_CabinTrackInfo.u8ListType, m_CabinTrackInfo.u16Filter1,m_CabinTrackInfo.u16Filter2,m_CabinTrackInfo.u16TrackIndex,m_CabinTrackInfo.u8Region,m_CabinTrackInfo.FolderPath);
				}
				else
					PluginServerHandler::getInstance()->sendsig_ReqTrackIndex(MEDIA_CURRENTTRACKINDEXREQ,entSourceType_CMR, m_CabinTrackInfo.u8ListType, m_CabinTrackInfo.u16Filter1,m_CabinTrackInfo.u16Filter2,m_CabinTrackInfo.u16TrackIndex,m_CabinTrackInfo.u8Region,m_CabinTrackInfo.FolderPath);
			}
			else
			{
				if (m_CabinTrackInfo.VideoPLaybackMode != Repeat_Track)
				{
					ETG_TRACE_USR4(("vStream3TotalPlayTime : Zero Update to fetch the next track index for continous playback for Folder track\n"));
					PluginServerHandler::getInstance()->sendsig_ReqTrackIndex(MEDIA_NEXTTRACKINDEXREQ,entSourceType_CMR, m_CabinTrackInfo.u8ListType, m_CabinTrackInfo.u16Filter1,m_CabinTrackInfo.u16Filter2,m_CabinTrackInfo.u16TrackIndex,m_CabinTrackInfo.u8Region,m_CabinTrackInfo.FolderPath);
				}
				else
					PluginServerHandler::getInstance()->sendsig_ReqTrackIndex(MEDIA_CURRENTTRACKINDEXREQ,entSourceType_CMR, m_CabinTrackInfo.u8ListType, m_CabinTrackInfo.u16Filter1,m_CabinTrackInfo.u16Filter2,m_CabinTrackInfo.u16TrackIndex,m_CabinTrackInfo.u8Region,m_CabinTrackInfo.FolderPath);
			}
		}
		else
		{
			ETG_TRACE_USR4(("vStream3TotalPlayTime : Ignore the update of the tpotal play time \n"));
		}
	}
}

void clRouterPlugin::vStream3CurrentPlayTime(uint16_t cmd_type,uint64_t entity_id)
{
   ETG_TRACE_USR4(("clRouterPlugin::vStream3CurrentPlayTime"));
	
	if((m_stPrimCMR.configuration == 0) || (m_stPrimCMR.u64EntityID != entity_id))
	{
		ETG_TRACE_USR4(("clRouterPlugin::vStream3CurrentPlayTime, ERROR! Configuration %d, EntityID %d", m_stPrimCMR.configuration, entity_id));
		return;
	}
	
	ControlDataElement controlElement;
	controlElement.configuration = m_stPrimCMR.configuration;
	controlElement.u32ControlDescIdx = STREAM3_CURRENT_PLAY_TIME;
	
	//Get value of current time for the current track	
	if(m_pCommControl != NULL)
	{
		m_pCommControl->GetUnsolicitedValue(controlElement);
	}
	ETG_TRACE_USR4(("vStream3CurrentPlayTime == controlData.strGetValue=%s  ",controlElement.strGetValue.c_str()));
	
	//send the string data to HMI
	vSendPluginResponseToHmi(controlElement.u32ControlDescIdx,controlElement.strGetValue);

}

void clRouterPlugin::vRouterStream3TrackNameResponse(uint16_t cmd_type,uint64_t entity_id)
{
	ETG_TRACE_USR4(("clRouterPlugin::vRouterStream3TrackNameResponse"));
   if((m_stPrimCMR.configuration == 0) || (m_stPrimCMR.u64EntityID != entity_id))
	{
		ETG_TRACE_USR4(("clRouterPlugin::vRouterStream3TrackNameResponse, ERROR! Configuration %d, EntityID %d", m_stPrimCMR.configuration, entity_id));
		return;
	}
	
	ControlDataElement controlElement;
	controlElement.configuration = m_stPrimCMR.configuration;
	controlElement.u32ControlDescIdx = STREAM3_TRACK_NAME;
	if(m_pCommControl != NULL)
	{
		if(AEM_CMD_GET_CONTROL /*25*/ == cmd_type)
		{
			controlElement.puintGetValue = pValue;
			controlElement.u16Readlen = 512;
			m_pCommControl->GetControlValue(controlElement);
			ETG_TRACE_USR4(("vRouterStream3TrackNameResponse from get control controlData.strGetValue=%s  ",controlElement.strGetValue.c_str()));
                        vRenameifInvalidUTF8isFound(controlElement.strGetValue);
			vSendPluginResponseToHmi(controlElement.u32ControlDescIdx, controlElement.strGetValue);
		}
		else if (AEM_CMD_SET_CONTROL /*24*/ == cmd_type)
		{
			m_pCommControl->GetUnsolicitedValue(controlElement);
			ETG_TRACE_USR4(("vRouterStream3TrackNameResponse from Unsolicited response controlData.strGetValue=%s  ",controlElement.strGetValue.c_str()));
                        vRenameifInvalidUTF8isFound(controlElement.strGetValue);
			vSendPluginResponseToHmi(controlElement.u32ControlDescIdx, controlElement.strGetValue);
		}
		else
		{
			ETG_TRACE_USR4(("clRouterPlugin::vRouterStream3TrackNameResponse, Command type is not matching"));
		}
	}
}

void clRouterPlugin::vRouterStream3PlayBackActionResponse(uint16_t cmd_type,uint64_t entity_id)
{
	ETG_TRACE_USR4(("clRouterPlugin::vRouterStream3PlayBackActionResponse"));
	if((m_stPrimCMR.configuration == 0) || (m_stPrimCMR.u64EntityID != entity_id))
	{
		ETG_TRACE_USR4(("clRouterPlugin::vRouterStream3PlayBackActionResponse, ERROR! Configuration %d, EntityID %d", m_stPrimCMR.configuration, entity_id));
		return;
	}
	
	ControlDataElement controlElement;
	controlElement.configuration = m_stPrimCMR.configuration;
	controlElement.u32ControlDescIdx = STREAM3_PLAY_ACTION;	
	controlElement.uint8Value = 0;
	
	if(m_pCommControl != NULL)
	{
		m_pCommControl->GetUnsolicitedValue(controlElement);
	}

	ETG_TRACE_USR4(("clRouterPlugin::vRouterStream3PlayBackActionResponse = controlElement.uint8Value %d  ", controlElement.uint8Value));	
	vSendPluginResponseToHmi(controlElement.u32ControlDescIdx,controlElement.uint8Value);
	
	if(controlElement.uint8Value == MEDIA_PLAY)
	{
                m_PlayState = controlElement.uint8Value;
		//explicit metadata update to hmi
		vCreateGetControlCommand(m_stPrimCMR,STREAM3_TRACK_NAME);
		//write to datapool
		VWritedatapool();
	}
}

void clRouterPlugin::vRouterStream3SeekToResponse(uint16_t cmd_type,uint64_t entity_id)
{
	ETG_TRACE_USR4(("clRouterPlugin::vRouterStream3SeekToResponse"));
	if((m_stPrimCMR.configuration == 0) || (m_stPrimCMR.u64EntityID != entity_id))
	{
		ETG_TRACE_USR4(("clRouterPlugin::vRouterStream3SeekToResponse, ERROR! Configuration %d, EntityID %d", m_stPrimCMR.configuration, entity_id));
		return;
	}
	
	ControlDataElement controlElement;
	controlElement.configuration = m_stPrimCMR.configuration;
	controlElement.u32ControlDescIdx = STREAM3_SEEK_TO;
	controlElement.uint8Value = 0;
	controlElement.puintGetValue = pValue;
	controlElement.u16Readlen = 1;
	
	if(m_pCommControl != NULL)
	{
		m_pCommControl->GetControlValue(controlElement); //getcontrolval CHANGE
	}
	ETG_TRACE_USR4(("clRouterPlugin::vRouterStream3SeekToResponse = controlElement.uint8Value %d  ", controlElement.uint8Value));
}

void clRouterPlugin::vSetStream3CurrentTrackIndex(uint32_t stream3TrackIndex)
{
	ETG_TRACE_USR4(("====vSetStream3CurrentTrackIndex  =stream1TrackIndex=%d  ",stream3TrackIndex));
	uint32_t sendTrackIndex = stream3TrackIndex;
	ETG_TRACE_USR4(("====vSetStream3CurrentTrackIndex  =sendTrackIndex=%d  ",sendTrackIndex));
	//set the stream1currenttack index to play the recived track index from listhandler.
	uint8_t u8HighByte = (stream3TrackIndex>>8) & 0X00FF;
	m_u8TRackIndex_LowByte = (stream3TrackIndex) & 0x00FF; 	

	ETG_TRACE_USR4(("====vSetStream3CurrentTrackIndex  high byte =%d lower byte = %d ",u8HighByte,m_u8TRackIndex_LowByte));

	//set the stream1currenttack index to play the recived track index from listhandler.	
	vCreateSetControlCommand(STREAM3_CURRENT_TRACK_INDEX_HIGH_BYTE,u8HighByte);
}

/******************************************************************************
*NAME          : vStream3CurTrackIndexLowByteResponse
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Function for sending Track Index for vStream1CurTrackIndexHighByteResponse
******************************************************************************/
void clRouterPlugin::vStream3CurTrackIndexLowByteResponse(uint16_t cmd_type,uint64_t entity_id)
{
	ETG_TRACE_USR4(("clRouterPlugin::vStream3CurTrackIndexResponse"));

	if((m_stPrimCMR.configuration == 0) || (m_stPrimCMR.u64EntityID != entity_id))
	{
		ETG_TRACE_USR4(("clRouterPlugin::vStream3CurTrackIndexResponse, ERROR! Configuration %d, EntityID %d", m_stPrimCMR.configuration, entity_id));
		return;
	}
	
	ControlDataElement controlElement;
	controlElement.configuration = m_stPrimCMR.configuration;
	controlElement.u32ControlDescIdx = STREAM3_CURRENT_TRACK_INDEX_LOW_BYTE;
	controlElement.uint8Value = 0;
	
	//Getvalue for the current track index
	if(m_pCommControl != NULL)
	{
		m_pCommControl->GetUnsolicitedValue(controlElement);
	}
	ETG_TRACE_USR4(("Track Index Lower byte =  %d  ", controlElement.uint8Value));
	
	uint8_t u8LowByte = controlElement.uint8Value; 			
	uint16_t SongID = (m_u8TrackIndex_HighByte<<8)|(u8LowByte & 0X00FF);
        m_CabinTrackInfo.u16TrackIndex = SongID;
   	bool playState = entPlayState_PlaySuccess;
	uint16_t sourceType = entSourceType_CMR;
	PluginServerHandler::getInstance()->sendSig_PlayStatusSignalUpdate(playState,sourceType,SongID,m_CabinTrackInfo.u16Filetype,m_CabinTrackInfo.u8ListType,m_CabinTrackInfo.u16Filter1,m_CabinTrackInfo.u16Filter2,m_CabinTrackInfo.FolderPath);
	m_bPlayAction = false;
}

/******************************************************************************
*NAME          : vStream3CurTrackIndexHighByteResponse
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Function for sending Track Index for vStream1CurTrackIndexHighByteResponse
******************************************************************************/
void clRouterPlugin::vStream3CurTrackIndexHighByteResponse(uint16_t cmd_type,uint64_t entity_id)
{
	ETG_TRACE_USR4(("clRouterPlugin::vStream3CurTrackIndexHighByteResponse"));
	
	if((m_stPrimCMR.configuration == 0) || (m_stPrimCMR.u64EntityID != entity_id))
	{
		ETG_TRACE_USR4(("clRouterPlugin::vStream3CurTrackIndexHighByteResponse, ERROR! Configuration %d, EntityID %d", m_stPrimCMR.configuration, entity_id));
		return;
	}
	
	ControlDataElement controlElement;
	controlElement.configuration = m_stPrimCMR.configuration;
	controlElement.u32ControlDescIdx = STREAM3_CURRENT_TRACK_INDEX_HIGH_BYTE;
	controlElement.uint8Value = 0;
	
	//Getvalue for the current track index
	if(m_pCommControl != NULL)
	{
		m_pCommControl->GetUnsolicitedValue(controlElement);
	}	
	m_u8TrackIndex_HighByte = controlElement.uint8Value;
	//set control for low byte once the response of high byte is recieved 
	vCreateSetControlCommand(STREAM3_CURRENT_TRACK_INDEX_LOW_BYTE,m_u8TRackIndex_LowByte);
	vCreateSetControlCommand(STREAM3_PLAY_ACTION,MEDIA_PLAY);
	ETG_TRACE_USR4(("Track Index = High Byte  = %d  ", m_u8TrackIndex_HighByte));
}

void clRouterPlugin::vRouterStream3ArtistNameResponse(uint16_t cmd_type,uint64_t entity_id)
{
	ETG_TRACE_USR4(("clRouterPlugin::vRouterStream3ArtistNameResponse"));
    if((m_stPrimCMR.configuration == 0) || (m_stPrimCMR.u64EntityID != entity_id))
	{
		ETG_TRACE_USR4(("clRouterPlugin::vRouterStream3ArtistNameResponse, ERROR! Configuration %d, EntityID %d", m_stPrimCMR.configuration, entity_id));
		return;
	}
	
	ControlDataElement controlElement;
	controlElement.configuration = m_stPrimCMR.configuration;
	controlElement.u32ControlDescIdx = STREAM3_ARTIST_NAME;
	
	if(m_pCommControl != NULL)
	{
		m_pCommControl->GetUnsolicitedValue(controlElement);
	}
	ETG_TRACE_USR4(("vRouterStream3ArtistNameResponse == controlData.strGetValue=%s  ",controlElement.strGetValue.c_str()));
        vRenameifInvalidUTF8isFound(controlElement.strGetValue);
	vSendPluginResponseToHmi(controlElement.u32ControlDescIdx, controlElement.strGetValue);
}

/******************************************************************************
*NAME          : VReadCMRdatafromdatapool
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Function to read last CMR data(CRC, Track index, List type, Playback mode, Rigion) from datapool
******************************************************************************/
void clRouterPlugin::VReadCMRdatafromdatapool()
{
	ETG_TRACE_USR4(("clRouterPlugin::VReadCMRdatafromdatapool"));
	
	dp_tclCMRDpListtrackindex _oDpListTrackIndex;
	m_CabinTrackInfo.u16TrackIndex = _oDpListTrackIndex.tGetData();
   	ETG_TRACE_USR4(("DP read Track Index = %d ", m_CabinTrackInfo.u16TrackIndex));
	
	dp_tclCMRDpListtype _oDpListtype;
   	m_CabinTrackInfo.u8ListType = _oDpListtype.tGetData();
  	ETG_TRACE_USR4(("DP read List type = %d ", m_CabinTrackInfo.u8ListType));
	
	uint8_t audiovideo_playbackmode;
	dp_tclCMRDpPLaybackMode _oDpPLaybackMode;
	audiovideo_playbackmode = _oDpPLaybackMode.tGetData();
	m_CabinTrackInfo.AuidoPLaybackMode = audiovideo_playbackmode & 0x0f;
	ETG_TRACE_USR4(("DP read Playback mode = %d ", audiovideo_playbackmode));
        m_CabinTrackInfo.VideoPLaybackMode = (audiovideo_playbackmode >> 4) & 0x0f;
	ETG_TRACE_USR4(("DP read Video PlaybackMode %d",m_CabinTrackInfo.VideoPLaybackMode));
	ETG_TRACE_USR4(("DP read Audio PlaybackMode %d",m_CabinTrackInfo.AuidoPLaybackMode));

	dp_tclCMRDpRegion _oDpRegion;
	m_CabinTrackInfo.u8Region =  _oDpRegion.tGetData();
	ETG_TRACE_USR4(("DP read Region = %d ", m_CabinTrackInfo.u8Region));
	
	dp_tclCMRDpSubSourceType _oDpSubSourceType;
	m_CabinTrackInfo.SubSourceType = _oDpSubSourceType.tGetData();
	ETG_TRACE_USR4(("DP read SubsourceType = %d ", m_CabinTrackInfo.SubSourceType));
	
	if(m_CabinTrackInfo.SubSourceType == entSubSourceType_CMR_AUDIO)
	{
		m_CabinTrackInfo.u16Filetype = entFileType_Audio;
	}
	else
	{
		m_CabinTrackInfo.u16Filetype = entFileType_Video;
	}	
	/*
	dp_tclCMRDpFilter1 _oDpFilter1;
	m_CabinTrackInfo.u16Filter1 = _oDpFilter1.tGetData();
	ETG_TRACE_USR4(("DP read Filter1 = %d ", m_CabinTrackInfo.u16Filter1));
	
	dp_tclCMRDpFilter2 _oDpFilter2;
	m_CabinTrackInfo.u16Filter2 = _oDpFilter2.tGetData();
	ETG_TRACE_USR4(("DP read Filter2 = %d ", m_CabinTrackInfo.u16Filter2));
	*/
	dp_tclCMRDpFolderPath _oDpFolderPath;
	char dpfpathread[255] = {0};
	_oDpFolderPath.s32GetData(dpfpathread,255);
	m_CabinTrackInfo.FolderPath.assign(dpfpathread);
	ETG_TRACE_USR4(("DP read Folder path = %s ",m_CabinTrackInfo.FolderPath));
		
	//Playback mode updates to HMI
	if(m_CabinTrackInfo.u8ListType == MEDIA_LISTTYPE_SONG)
	{
		vSendPluginResponseToHmi(CMRSTREAM3_PLAYBACKMODE,m_CabinTrackInfo.AuidoPLaybackMode);
	}
	else
	{
		vSendPluginResponseToHmi(CMRSTREAM1_PLAYBACKMODE,m_CabinTrackInfo.VideoPLaybackMode);
	}
    
	//Playback mode updates to List Handler
	PlayBackModeStruct ModeStructAudio(m_CabinTrackInfo.AuidoPLaybackMode,entSourceType_CMR,m_CabinTrackInfo.u8Region,entSubSourceType_CMR_AUDIO), 
	ModeStructVideo(m_CabinTrackInfo.VideoPLaybackMode,entSourceType_CMR,m_CabinTrackInfo.u8Region,entSubSourceType_CMR_VIDEO);
    PluginServerHandler::getInstance()->setPlayBackMode(ModeStructAudio);
	PluginServerHandler::getInstance()->setPlayBackMode(ModeStructVideo);
	//updating the valid list type if stored listtype is 0
	if(m_CabinTrackInfo.u8ListType == 0)
	{
		if (m_PlayControlId == STREAM1_PLAY_ACTION )
		{
			m_CabinTrackInfo.u8ListType = MEDIA_LISTTYPE_VIDEO; 
		}
		else
		{
			m_CabinTrackInfo.u8ListType = MEDIA_LISTTYPE_SONG;
		}
	}
	else
		ETG_TRACE_USR4(("VReadCMPdatafromdatapool listtype = %d ", m_CabinTrackInfo.u8ListType));
}	

/******************************************************************************
*NAME          : VeraseCMRdatapool
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Function to erase CMR datapool(Track index, List type, Playback mode, Rigion, subsource type)
******************************************************************************/
void clRouterPlugin::VclearCMRdatapool()
{
	ETG_TRACE_USR4(("clRouterPlugin::VclearCMRdatapool"));
	uint8_t u8Defaultvalue = 0;
	uint8_t u8DefaultPlaybackMode = MEDIA_DATAPOOL_DEFAULT_AUDIOVIDEO_PLAYBACKMODE;
	uint16_t u16Defaultvalue = 0;
	uint16_t u16DefaultTrackIndex = TRACKINDEX_UNAVAILABLE;
	
	dp_tclCMRDpListtrackindex _oDpListTrackIndex;
	_oDpListTrackIndex.s32SetData(u16DefaultTrackIndex);
	
	dp_tclCMRDpListtype _oDpListtype;
	_oDpListtype.s32SetData(u16Defaultvalue);
	
	dp_tclCMRDpRegion _oDpRegion;
	_oDpRegion.s32SetData(u16Defaultvalue);
	
	dp_tclCMRDpSubSourceType _oDpSubSourceType;
	_oDpSubSourceType.s32SetData(u8Defaultvalue);
	
	dp_tclCMRDpPLaybackMode _oDpPLaybackMode;
	_oDpPLaybackMode.s32SetData(u8DefaultPlaybackMode);
	/*
	dp_tclCMRDpFilter1 _oDpFilter1;
	_oDpFilter1.s32SetData(u16Defaultvalue);
	
	dp_tclCMRDpFilter2 _oDpFilter2;
	_oDpFilter2.s32SetData(u16Defaultvalue);
    */
	dp_tclCMRDpFolderPath _oDpFolderPath;
    char fpath_Defaultvalue[5] = {'r','o','o','t','\0'};
	_oDpFolderPath.s32SetData(fpath_Defaultvalue);
}

/******************************************************************************
*NAME          : vUsbentitiesupdate
*DESIGN SECTION: 6.1.3
*PARAMETER:		 void
*RETURNVALUE: 	 void
*DESCRIPTION   : Function to update USB status, DB status, CRC status to HMI on HMI request
******************************************************************************/
void clRouterPlugin::vUsbentitiesupdate()
{
	ETG_TRACE_USR4(("clRouterPlugin::vUsbentitiesupdate"));
	if(m_busbStatus == USB_CONNECTED)
	{
		uint8_t u8_Value;
		uint32_t u32_ControlDescIdx;
		//USB status update
		u32_ControlDescIdx = USB_DEVICE_STATUS;
		u8_Value = USB_CONNECTED;
		vSendPluginResponseToHmi(u32_ControlDescIdx,u8_Value);
		//DB status update
		u32_ControlDescIdx = MEDIA_DATA_BASE_STATUS;
		u8_Value = m_dbstatus;
		vSendPluginResponseToHmi(u32_ControlDescIdx,u8_Value);
		//CRC status update
		vCreateGetControlCommand(m_stPrimCMR,MEDIA_CHECKSUM_STATUS);
	}
	else
	{
		ETG_TRACE_USR4(("clRouterPlugin::vUsbentitiesupdate USB not connected, ignore update"));
	} 
}

/******************************************************************************
*NAME          : vRenameifInvalidUTF8isFound
*DESIGN SECTION: 6.1.3
*PARAMETER:	 string &
*RETURNVALUE: 	 void
*DESCRIPTION   : Function to validate string is in UTF8 formate or not
******************************************************************************/
void clRouterPlugin::vRenameifInvalidUTF8isFound(std::string &Name)
{
    ETG_TRACE_USR4(("clRouterPlugin::vRenameifInvalidUTF8isFound"));
    if (Name.length() > 0)
    {
        if (!Utf8String::isValid(Name))
        {
            ETG_TRACE_USR4(("clRouterPlugin::vRenameifInvalidUTF8isFound Invalid"));
            Name.assign("Unknown");
        }
    }
}

/******************************************************************************
*NAME          : vVideosourcerestoreinpoause
*DESIGN SECTION: 6.1.3
*PARAMETER:	     void
*RETURNVALUE: 	 void
*DESCRIPTION   : Function to restore video source in pause state
******************************************************************************/
void clRouterPlugin::vVideosourcerestoreinpoause()
{
	ETG_TRACE_USR4(("clRouterPlugin::vVideosourcerestoreinpoause"));
        m_bvideosourcerestore = true;
	VReadCMRdatafromdatapool();
	PluginServerHandler::getInstance()->sendsig_ReqTrackIndex(MEDIA_SELECTTRACKINDEXREQ,entSourceType_CMR, m_CabinTrackInfo.u8ListType, m_CabinTrackInfo.u16Filter1,m_CabinTrackInfo.u16Filter2,m_CabinTrackInfo.u16TrackIndex,m_CabinTrackInfo.u8Region,m_CabinTrackInfo.FolderPath);
        vSetStreamCurrentTrackIndex(m_CabinTrackInfo.u16TrackIndex,m_CabinTrackInfo.u16Filetype);
}

/******************************************************************************
*NAME          : Vwritedatapool
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Function to write CMR data to datapool
******************************************************************************/
void clRouterPlugin::VWritedatapool()
{
	ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::Vwritedatapool"));
	
	dp_tclCMRDpListtrackindex _oDpListTrackIndex;
	if(m_CabinTrackInfo.u16TrackIndex != _oDpListTrackIndex.tGetData())
	{
		_oDpListTrackIndex.s32SetData(m_CabinTrackInfo.u16TrackIndex);
		ETG_TRACE_USR4(("DP write Track Index = %d ", m_CabinTrackInfo.u16TrackIndex));
	}
	
	dp_tclCMRDpListtype _oDpListtype;
	if(m_CabinTrackInfo.u8ListType != _oDpListtype.tGetData())
	{
		_oDpListtype.s32SetData(m_CabinTrackInfo.u8ListType);
		ETG_TRACE_USR4(("DP Write List type = %d ", m_CabinTrackInfo.u8ListType));
	}
	
	dp_tclCMRDpRegion _oDpRegion;
	if(m_CabinTrackInfo.u8Region != _oDpRegion.tGetData())
	{
		_oDpRegion.s32SetData(m_CabinTrackInfo.u8Region);
		ETG_TRACE_USR4(("DP write Region = %d ", m_CabinTrackInfo.u8Region));
	}
	
	dp_tclCMRDpSubSourceType _oDpSubSourceType;
	if(m_CabinTrackInfo.SubSourceType != _oDpSubSourceType.tGetData())
	{
		_oDpSubSourceType.s32SetData(m_CabinTrackInfo.SubSourceType);
		ETG_TRACE_USR4(("DP write SubSource = %d ", m_CabinTrackInfo.SubSourceType));
	}
	/*
	dp_tclCMRDpFilter1 _oDpFilter1;
	if(m_CabinTrackInfo.u16Filter1 != _oDpFilter1.tGetData())
	{
		_oDpFilter1.s32SetData(m_CabinTrackInfo.u16Filter1);
		ETG_TRACE_USR4(("DP write Track Index = %d ", m_CabinTrackInfo.u16Filter1));
	}
	
	dp_tclCMRDpFilter2 _oDpFilter2;
	if(m_CabinTrackInfo.u16Filter2 != _oDpFilter2.tGetData())
	{
		_oDpFilter2.s32SetData(m_CabinTrackInfo.u16Filter2);
		ETG_TRACE_USR4(("DP write Track Index = %d ", m_CabinTrackInfo.u16Filter2));
	}
	*/
        dp_tclCMRDpFolderPath _oDpFolderPath;
        std::string dpfpath;
        char dpfpathread[255] = {0};
        _oDpFolderPath.s32GetData(dpfpathread,255);
        dpfpath.assign(dpfpathread);
        if(m_CabinTrackInfo.FolderPath != dpfpath)
        {
                uint8_t fpathlen = m_CabinTrackInfo.FolderPath.length();
                char dpfpathwrite[fpathlen+1] ={0};
                strncpy(dpfpathwrite,m_CabinTrackInfo.FolderPath.c_str(),fpathlen);
                dpfpathwrite[fpathlen] = '\0';
                _oDpFolderPath.s32SetData(dpfpathwrite);
                ETG_TRACE_USR4(("DP write FolderPath = %s ", dpfpathwrite));
        }
}

