/* ***************************************************************************************
	* FILE:		  clExternalMediaPlayerPlugin.cpp
	* SW-COMPONENT:  avdecc_appl_plugins
	* DESCRIPTION:  clExternalMediaPlayerPlugin.cpp is part of avdecc_appl_plugins library
	* COPYRIGHT:  (c) 2020-21 Robert Bosch Car Multimedia GmbH
	* HISTORY: 
	* AUTHOR:  
	* REVISION: 
	*
	* The reproduction, distribution and utilization of this file as well as the
	* communication of its contents to others without express authorization is
	* prohibited. Offenders will be held liable for the payment of damages.
	* All rights reserved in the event of the grant of a patent, utility model or design.
	*
	*************************************************************************************** */
	/*****************************************************************
	| includes
	|----------------------------------------------------------------*/
	#include <clExternalMediaPlayerPlugin.h>
	#include "plugin_trace.h"
	#include "MediaCameraPort.h"
	#include "clControlMapping.h"
	#include "clPluginDataProvider.h"
	#include "PluginDataType.h"
	#include <sstream>
	#include "PluginServerHandler.h"
	
	#include "PluginEvtListenerBase.h"
	#include "PluginEventBase.h"
	//#include "XMLRead.h"
	#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
	#define ETG_DEFAULT_TRACE_CLASS  TR_CLASS_PLUGIN_EXTERNALMEDIAPLAYER
	#include "trcGenProj/Header/clExternalMediaPlayerPlugin.cpp.trc.h"
	#endif // VARIANT_S_FTR_ENABLE_TRC_GEN

#define CONTROL_ID 0
#define CONTROL_VAL 1

#define PLUGIN_NAME_MEDIA "Media"
#define CTRLBLOCK_NAME_MEDIA_PLAYBACKACTION  "PlaybackAction"
#define CTRLBLOCK_NAME_MEDIA_USBAUDIOSTATUS  "USBAudioStatus"
#define CTRLBLOCK_NAME_MEDIA_HDMISTATUS "HDMIStatus"
#define CTRLBLOCK_NAME_MEDIA_SIGNAL_SOURCE_CHANGE "SignalSourceChange"


clExternalMediaPlayerPlugin* clExternalMediaPlayerPlugin::poSelf = NULL;
clExternalMediaPlayerPlugin::clExternalMediaPlayerPlugin()
{
	m_CMPEntityDetails.clear();
	//Communucation Protocol Object creation logic
	m_pCommControl = new clCommunicationProtocol();
	if(m_pCommControl != NULL)
	{
		ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::clCommunicationProtocol is created"));
	}
	m_u16LastExecutedSetCommand = 0;
		m_stPrimCMP.u64EntityID = 0;
		m_stPrimCMP.u32EntityIndex = 0;
		m_stPrimCMP.strEndStationName = "";
		m_stPrimCMP.strEntityName = "";
		m_stPrimCMP.end_station = 0;
		m_stPrimCMP.entity = 0;
		m_stPrimCMP.configuration = 0;
		m_PlayControlId = 0;
	pValue= new uint8_t;
	m_u8TrackIndex_HighByte =0;
	m_u8TRackIndex_LowByte = 0;
	m_u8TrackIndex_Audio_HighByte = 0;
	m_u8TRackIndex_Audio_LowByte = 0;
	m_strCurrentCheckSum = "";
	m_strPrevCheckSum = "";
	m_u8audioStatus = 0;
	m_bPlayAction = false;
	m_bvideosourcerestore = false;
	m_TotalTimevalue = 0;
		m_CabinTrackInfo.u8Region = 0;
		m_CabinTrackInfo.u8ListType = MEDIA_LISTTYPE_INVALID;
                m_CabinTrackInfo.u16TrackIndex = 0;
		m_CabinTrackInfo.AuidoPLaybackMode = Repeat_List;
		m_CabinTrackInfo.VideoPLaybackMode = Repeat_List;
		m_CabinTrackInfo.u16Filter1 = 0;
		m_CabinTrackInfo.u16Filter2 = 0;
		m_CabinTrackInfo.FolderPath = "root";
		m_CabinTrackInfo.u16Filetype = entFileType_Video;
		m_CabinTrackInfo.SubSourceType = entSubSourceType_CMP_VIDEO;
    vRegisterResponseCallbacks();
}

clExternalMediaPlayerPlugin::~clExternalMediaPlayerPlugin(){
	delete pValue;
	delete m_pCommControl;
}

clExternalMediaPlayerPlugin* clExternalMediaPlayerPlugin::getInstance()
{
	ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::getInstance"));
	if(poSelf == NULL)
	{
		ETG_TRACE_USR4(("Creating object clExternalMediaPlayerPlugin"));
		poSelf = new clExternalMediaPlayerPlugin();
	}
	ETG_TRACE_USR4(("Return object clExternalMediaPlayerPlugin %d", poSelf));
	return poSelf;
	
}
void clExternalMediaPlayerPlugin::deleteInstance()
{
	if(NULL != poSelf)
	{
		delete poSelf;
	}
}

void clExternalMediaPlayerPlugin::vEventESReadComplete()
{ 
	ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vEventESReadComplete()"));
	//extract the details of the entities required
	clPluginDataProvider *ptrMain = clPluginDataProvider::getInstance();
	std::vector<stEntityDesc> vecCmpDetails;
	if(NULL != ptrMain)
	{
		m_vecCamDesc.clear();
		m_vecCamDesc = ptrMain->vecGetCamDesc();
        //for easy access during response.
        m_stPrimCMP = getDetailsOfCMP(PRIMARY_CAMPORT);
        m_stSecCMP = getDetailsOfCMP(SECOND_CAMPORT);
        m_stTerCMP = getDetailsOfCMP(THIRD_CAMPORT);
		bool found = ptrMain->getEndStationDetails(CMP, vecCmpDetails);
      	ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vEventESReadComplete() found = %d", found));		
		for(auto itr = vecCmpDetails.begin(); itr != vecCmpDetails.end(); itr++)
		{
			if(!isEntityDescriptorAvailable(*itr)) // Finds in m_CMPEntityDetails
			{
				vRegisterCamPort(*itr);
			}
		};
		m_CMPEntityDetails.clear();
		m_CMPEntityDetails=vecCmpDetails;
		//print CMP Details
		if(true == found)
		{
		  ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vEventESReadComplete()::logging Entity details"));
		  for_each(m_CMPEntityDetails.begin(), m_CMPEntityDetails.end(), [](stEntityDesc Ent){
				ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vEventESReadComplete() u64EntityID is %d", Ent.u64EntityID));
				ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vEventESReadComplete() strEndStationName is %s", Ent.strEndStationName.c_str()));
				ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vEventESReadComplete() strEntityName is %s", Ent.strEntityName.c_str()));
				ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vEventESReadComplete() configuration is %d", Ent.configuration));
		  });
		}
	}
}

/******************************************************************************
*NAME		  : isEntityDescriptorAvailable
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Finds if Entity is already available in m_CMPEntityDetails
******************************************************************************/
bool clExternalMediaPlayerPlugin::isEntityDescriptorAvailable(stEntityDesc EntityDescriptor)
{
	ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::isEntityDescriptorAvailable()"));
	for(auto itr : m_CMPEntityDetails)
	{
		if((itr.u64EntityID == EntityDescriptor.u64EntityID) && (itr.configuration != 0))
		{
			ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::isEntityDescriptorAvailable() Entity already present and registered"));
			return true;
		}
	}
	return false;
}

/******************************************************************************
*NAME		  : vRegisterCamPort
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Registers for UnSolicitedResponse and CreateGetControlCommand
******************************************************************************/
void clExternalMediaPlayerPlugin::vRegisterCamPort(stEntityDesc EntityDescriptor)
{
	ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vRegisterCamPort()"));

	for(auto itr : m_vecCamDesc) // vec<pos, stentitydesc>   
	{
		if(itr.CamEntityDesc.u64EntityID == EntityDescriptor.u64EntityID)
		{
			if((itr.CamEntityDesc.configuration != 0) && (m_pCommControl != NULL))
			{
				m_pCommControl->RegisterUnSolicitedResponse(itr.CamEntityDesc.end_station);
				vCreateGetControlCommand(itr.CamEntityDesc, HDMISTATUS);
				if(itr.position == PRIMARY_CAMPORT)
				{
					vCreateGetControlCommand(itr.CamEntityDesc, USBAUDIOSTATUS);
					vCreateGetControlCommand(itr.CamEntityDesc, DBSTATUS);

				}
			}
		}
	}
}

/******************************************************************************
*NAME		  : getDetailsOfCMP
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : This function  extracts the entity details structure based on Position
******************************************************************************/
stEntityDesc clExternalMediaPlayerPlugin::getDetailsOfCMP(uint16_t u16CMP_Pos)
{
	ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::getDetailsOfCMP()"));
	stEntityDesc stCMPDetails;
	stCMPDetails.u64EntityID = 0;
	stCMPDetails.u32EntityIndex = 0;
	stCMPDetails.strEndStationName = "";
	stCMPDetails.strEntityName = "";
	stCMPDetails.end_station = 0;
	stCMPDetails.entity = 0;
	stCMPDetails.configuration = 0;
	for(auto itr : m_vecCamDesc) // vec<pos, stentitydesc>   
	{
		if(itr.position == u16CMP_Pos)
		{
			stCMPDetails= itr.CamEntityDesc;
			ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::getDetailsOfCMP(), for PrimaryCMP u64EntityID is %d", stCMPDetails.u64EntityID));
		}
	}
	return stCMPDetails;
}

/******************************************************************************
*NAME		  : vRegisterResponseCallbacks
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Registers callbacks for all controls
******************************************************************************/
void clExternalMediaPlayerPlugin::vRegisterResponseCallbacks()
{
	ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vRegisterResponseCallbacks"));

	//register for callbacks
	clControlMapping *p_controlmap = clControlMapping::getInstance();
	
	if(NULL != p_controlmap)
	{
		p_controlmap->setRespCallBack(CMP, STREAM1PLAYACTION, this, (FUNCPTR)(&clExternalMediaPlayerPlugin::vStream1PlayactionResponse));
		p_controlmap->setRespCallBack(CMP, STREAM3PLAYACTION, this, (FUNCPTR)(&clExternalMediaPlayerPlugin::vStream3PlayactionResponse));
		p_controlmap->setRespCallBack(CMP, STREAM1SEEKTO, this, (FUNCPTR)(&clExternalMediaPlayerPlugin::vStream1SeekToResponse));
		p_controlmap->setRespCallBack(CMP, STREAM1CURPLAYTIME, this, (FUNCPTR)(&clExternalMediaPlayerPlugin::vStream1CurPlayTimeResponse));
		p_controlmap->setRespCallBack(CMP, STREAM1TOTPLAYTIME, this, (FUNCPTR)(&clExternalMediaPlayerPlugin::vStream1TotPlayTimeResponse));
		p_controlmap->setRespCallBack(CMP, VIDNAME, this, (FUNCPTR)(&clExternalMediaPlayerPlugin::vVidNameResponse));
		p_controlmap->setRespCallBack(CMP, VIDEPISODE, this, (FUNCPTR)(&clExternalMediaPlayerPlugin::vVidEpisodeResponse));		
		p_controlmap->setRespCallBack(CMP, STREAM1CURRENTTRACKINDEXHIGH, this, (FUNCPTR)(&clExternalMediaPlayerPlugin::vStream1CurTrackIndexHighByteResponse));
		p_controlmap->setRespCallBack(CMP, STREAM1CURRENTTRACKINDEXLOW, this, (FUNCPTR)(&clExternalMediaPlayerPlugin::vStream1CurTrackIndexLowByteResponse));
		p_controlmap->setRespCallBack(CMP, DBSTATUS, this, (FUNCPTR)(&clExternalMediaPlayerPlugin::vDBStatusResponse));			
		p_controlmap->setRespCallBack(CMP, USBAUDIOSTATUS, this, (FUNCPTR)(&clExternalMediaPlayerPlugin::vUSBAudioStatusResponse));	
		p_controlmap->setRespCallBack(CMP, CHECKSUM, this, (FUNCPTR)(&clExternalMediaPlayerPlugin::vCheckSumStatusResponse));
		p_controlmap->setRespCallBack(CMP, STREAM3SEEKTO, this, (FUNCPTR)(&clExternalMediaPlayerPlugin::vStream3SeekToResponse));
		p_controlmap->setRespCallBack(CMP, STREAM3CURPLAYTIME, this, (FUNCPTR)(&clExternalMediaPlayerPlugin::vStream3CurPlayTimeResponse));
		p_controlmap->setRespCallBack(CMP, STREAM3TOTPLAYTIME, this, (FUNCPTR)(&clExternalMediaPlayerPlugin::vStream3TotPlayTimeResponse));
		p_controlmap->setRespCallBack(CMP, STREAM3CURRENTTRACKINDEXHIGH, this, (FUNCPTR)(&clExternalMediaPlayerPlugin::vStream3CurTrackIndexHighByteResponse));
		p_controlmap->setRespCallBack(CMP, STREAM3CURRENTTRACKINDEXLOW, this, (FUNCPTR)(&clExternalMediaPlayerPlugin::vStream3CurTrackIndexLowByteResponse));
		p_controlmap->setRespCallBack(CMP, TRACK, this, (FUNCPTR)(&clExternalMediaPlayerPlugin::vTrackNameResponse));
		p_controlmap->setRespCallBack(CMP, HDMISTATUS, this, (FUNCPTR)(&clExternalMediaPlayerPlugin::vHDMIStatusResponse));
	}
}

/******************************************************************************
*NAME		  : vStream1PlayactionResponse
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Function for sending command response to HMI for stream1plyaction(for CMP Video source)
******************************************************************************/
void clExternalMediaPlayerPlugin::vStream1PlayactionResponse(uint16_t cmd_type,uint64_t entity_id)
{
	ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vStream1PlayactionResponse"));

	if((m_stPrimCMP.configuration == 0) || (m_stPrimCMP.u64EntityID != entity_id))
	{
		ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vStream1PlayactionResponse, ERROR! Configuration %d, EntityID %d", m_stPrimCMP.configuration, entity_id));
		return;
	}
	
	
	ControlDataElement controlElement;
	controlElement.configuration = m_stPrimCMP.configuration;
	controlElement.u32ControlDescIdx = STREAM1PLAYACTION;
	controlElement.uint8Value = 0;
	
	if(m_pCommControl != NULL)
	{
		m_pCommControl->GetUnsolicitedValue(controlElement);
	}
	ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vStream1PlayactionResponse = controlElement.uint8Value %d  ", controlElement.uint8Value));

	//Update to HMI
	vSendPluginResponseToHmi(controlElement.u32ControlDescIdx,controlElement.uint8Value);
	//write to datapool
	if(controlElement.uint8Value == MEDIA_PLAY)
	{
		VWritedatapool();
	}
}

/******************************************************************************
*NAME		  : vStream3PlayactionResponse
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Function for sending command response to HMI for stream1plyaction(for CMP Video source)
******************************************************************************/
void clExternalMediaPlayerPlugin::vStream3PlayactionResponse(uint16_t cmd_typec,uint64_t entity_id)
{
	ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vStream3PlayactionResponse"));
	if((m_stPrimCMP.configuration == 0) || (m_stPrimCMP.u64EntityID != entity_id))
	{
		ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vStream3PlayactionResponse, ERROR! Configuration %d, EntityID %d", m_stPrimCMP.configuration, entity_id));
		return;
	}
	
	ControlDataElement controlElement;
	controlElement.configuration = m_stPrimCMP.configuration;
	controlElement.u32ControlDescIdx = STREAM3PLAYACTION;
	controlElement.uint8Value = 0;
	
	if(m_pCommControl != NULL)
	{
		m_pCommControl->GetUnsolicitedValue(controlElement);
	}
	ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vStream1PlayactionResponse = controlElement.uint8Value %d  ", controlElement.uint8Value));


	vSendPluginResponseToHmi(controlElement.u32ControlDescIdx,controlElement.uint8Value);
	
	//write to datapool
	if(controlElement.uint8Value == MEDIA_PLAY)
	{
		VWritedatapool();
	}
}
/******************************************************************************
*NAME		  : vStream3SeekToResponse
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Function for sending command response to HMI for vStream3SeekToResponse(for CMP Audio source)
******************************************************************************/
void clExternalMediaPlayerPlugin::vStream3SeekToResponse(uint16_t cmd_type,uint64_t entity_id)
{
	ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vStream3SeekToResponse"));
	
	if((m_stPrimCMP.configuration == 0) || (m_stPrimCMP.u64EntityID != entity_id))
	{
		ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vStream3SeekToResponse, ERROR! Configuration %d, EntityID %d", m_stPrimCMP.configuration, entity_id));
		return;
	}
	
	ControlDataElement controlElement;
	controlElement.configuration = m_stPrimCMP.configuration;
	controlElement.u32ControlDescIdx = STREAM3SEEKTO;
	controlElement.uint8Value = 0;
	controlElement.puintGetValue = pValue;
	controlElement.u16Readlen = 1;
	
	if(m_pCommControl != NULL)
	{
		m_pCommControl->GetControlValue(controlElement); 
	}
	ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vStream3SeekToResponse = controlElement.uint8Value %d  ", controlElement.uint8Value));
}

/******************************************************************************
*NAME		  : vStream1SeekToResponse
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Function for sending command response to HMI for vStream1SeekToResponse(for CMP Video source)
******************************************************************************/
void clExternalMediaPlayerPlugin::vStream1SeekToResponse(uint16_t cmd_type,uint64_t entity_id)
{
	ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vStream1SeekToResponse"));
	
	if((m_stPrimCMP.configuration == 0) || (m_stPrimCMP.u64EntityID != entity_id))
	{
		ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vStream1SeekToResponse, ERROR! Configuration %d, EntityID %d", m_stPrimCMP.configuration, entity_id));
		return;
	}
	
	ControlDataElement controlElement;
	controlElement.configuration = m_stPrimCMP.configuration;
	controlElement.u32ControlDescIdx = STREAM1SEEKTO;
	controlElement.uint8Value = 0;
	controlElement.puintGetValue = pValue;
	controlElement.u16Readlen = 1;
	
	if(m_pCommControl != NULL)
	{
		m_pCommControl->GetControlValue(controlElement); 
	}
	ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vStream1SeekToResponse = controlElement.uint8Value %d  ", controlElement.uint8Value));
}
/******************************************************************************
*NAME		  : vStream1CurPlayTimeResponse
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Function for sending command response to HMI for vStream1CurPlayTimeResponse(for CMP Video source)
******************************************************************************/
void clExternalMediaPlayerPlugin::vStream1CurPlayTimeResponse(uint16_t cmd_type,uint64_t entity_id)
{
	ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vStream1CurPlayTimeResponse"));
	
	ControlDataElement controlElement;
	if((m_stPrimCMP.configuration == 0) || (m_stPrimCMP.u64EntityID != entity_id))
	{
		ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vStream1CurPlayTimeResponse, ERROR! Configuration %d, EntityID %d", m_stPrimCMP.configuration, entity_id));
		return;
	}
	
	controlElement.configuration = m_stPrimCMP.configuration;
	controlElement.u32ControlDescIdx = STREAM1CURPLAYTIME;
	
	//Get value of current time for the current track	
	if(m_pCommControl != NULL)
	{
		m_pCommControl->GetUnsolicitedValue(controlElement);
	}
	ETG_TRACE_USR4(("vStream1CurPlayTimeResponse == controlData.strGetValue=%s  ",controlElement.strGetValue.c_str()));
	
	//send the string data to HMI
	vSendStringResponseToHmi(controlElement.u32ControlDescIdx,controlElement.strGetValue);
	
	//Handling of playing the next track automatically.Elapsed time to be compared with the TotalTime and send absolutetrackindex request signal to listhandler
	double ElapsedTimevalue = vConvertTimeInSecs(controlElement.strGetValue);
	ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vStream1CurPlayTimeResponse ElapsedTimevalue %f", ElapsedTimevalue));
	ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vStream1CurPlayTimeResponse TotalTimevalue %f", m_TotalTimevalue));
	
        if((ElapsedTimevalue == m_TotalTimevalue) && (m_TotalTimevalue != 0))
	{
		ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vStream1CurPlayTimeResponse End of song"));
			
		if (m_CabinTrackInfo.VideoPLaybackMode != Repeat_Track)//No repeat track request for next else request for same track
		{
			ETG_TRACE_USR4(("vStream1TotalPlayTime : Zero Update to fetch the next track index for continous playback \n"));
			PluginServerHandler::getInstance()->sendsig_ReqTrackIndex(MEDIA_NEXTTRACKINDEXREQ,entSourceType_CMP,m_CabinTrackInfo.u8ListType, m_CabinTrackInfo.u16Filter1,m_CabinTrackInfo.u16Filter2,m_CabinTrackInfo.u16TrackIndex,m_CabinTrackInfo.u8Region,m_CabinTrackInfo.FolderPath);
		}
		else
		{
			PluginServerHandler::getInstance()->sendsig_ReqTrackIndex(MEDIA_CURRENTTRACKINDEXREQ,entSourceType_CMP,m_CabinTrackInfo.u8ListType, m_CabinTrackInfo.u16Filter1,m_CabinTrackInfo.u16Filter2,m_CabinTrackInfo.u16TrackIndex,m_CabinTrackInfo.u8Region,m_CabinTrackInfo.FolderPath);
			ETG_TRACE_USR4(("vStream1TotalPlayTime : Repeat Track request \n"));
		}		
	}
}
/******************************************************************************
*NAME		  : vStream3CurPlayTimeResponse
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Function for sending command response to HMI for vStream3CurPlayTimeResponse(for CMP audio source)
******************************************************************************/
void clExternalMediaPlayerPlugin::vStream3CurPlayTimeResponse(uint16_t cmd_type,uint64_t entity_id)
{
	ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vStream3CurPlayTimeResponse"));
	
	ControlDataElement controlElement;
	if((m_stPrimCMP.configuration == 0) || (m_stPrimCMP.u64EntityID != entity_id))
	{
		ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vStream3CurPlayTimeResponse, ERROR! Configuration %d, EntityID %d", m_stPrimCMP.configuration, entity_id));
		return;
	}
	controlElement.configuration = m_stPrimCMP.configuration;
	controlElement.u32ControlDescIdx = STREAM3CURPLAYTIME;
	controlElement.strGetValue = "";
	
	//Get value of current time for the current track	
	if(m_pCommControl != NULL)
	{
		m_pCommControl->GetUnsolicitedValue(controlElement);
	}
	ETG_TRACE_USR4(("vStream3CurPlayTimeResponse == controlData.strGetValue=%s  ",controlElement.strGetValue.c_str()));
	
	//send the string data to HMI
	vSendStringResponseToHmi(controlElement.u32ControlDescIdx,controlElement.strGetValue);
	
	//Handling of playing the next track automatically.Elapsed time to be compared with the TotalTime and send absolutetrackindex request signal to listhandler
	double ElapsedTimevalue = vConvertTimeInSecs(controlElement.strGetValue);
	ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vStream3CurPlayTimeResponse ElapsedTimevalue %f", ElapsedTimevalue));
	ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vStream1CurPlayTimeResponse TotalTimevalue %f", m_TotalTimevalue));
	
        if((ElapsedTimevalue == m_TotalTimevalue) && (m_TotalTimevalue != 0))
	{
		ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vStream3CurPlayTimeResponse End of song"));
		
		if (m_CabinTrackInfo.u8ListType != MEDIA_LISTTYPE_FOLDER)
		{
			if (m_CabinTrackInfo.AuidoPLaybackMode != Repeat_Track)//No repeat track request for next else request for same track
			{
				ETG_TRACE_USR4(("vStream3TotalPlayTime : Zero Update to fetch the next track index for continous playback \n"));
				PluginServerHandler::getInstance()->sendsig_ReqTrackIndex(MEDIA_NEXTTRACKINDEXREQ,entSourceType_CMP,m_CabinTrackInfo.u8ListType, m_CabinTrackInfo.u16Filter1,m_CabinTrackInfo.u16Filter2,m_CabinTrackInfo.u16TrackIndex,m_CabinTrackInfo.u8Region,m_CabinTrackInfo.FolderPath);
			}
			else
			{
				PluginServerHandler::getInstance()->sendsig_ReqTrackIndex(MEDIA_CURRENTTRACKINDEXREQ,entSourceType_CMP,m_CabinTrackInfo.u8ListType, m_CabinTrackInfo.u16Filter1,m_CabinTrackInfo.u16Filter2,m_CabinTrackInfo.u16TrackIndex,m_CabinTrackInfo.u8Region,m_CabinTrackInfo.FolderPath);
				ETG_TRACE_USR4(("vStream3TotalPlayTime : Repeat Track request \n"));
			}
		}
		else
		{
			if (m_CabinTrackInfo.VideoPLaybackMode != Repeat_Track)//No repeat track request for next else request for same track
			{
				ETG_TRACE_USR4(("vStream3TotalPlayTime : Zero Update to fetch the next track index for continous playback for Folder track\n"));
				PluginServerHandler::getInstance()->sendsig_ReqTrackIndex(MEDIA_NEXTTRACKINDEXREQ,entSourceType_CMP,m_CabinTrackInfo.u8ListType, m_CabinTrackInfo.u16Filter1,m_CabinTrackInfo.u16Filter2,m_CabinTrackInfo.u16TrackIndex,m_CabinTrackInfo.u8Region,m_CabinTrackInfo.FolderPath);
			}
			else
			{
				PluginServerHandler::getInstance()->sendsig_ReqTrackIndex(MEDIA_CURRENTTRACKINDEXREQ,entSourceType_CMP,m_CabinTrackInfo.u8ListType, m_CabinTrackInfo.u16Filter1,m_CabinTrackInfo.u16Filter2,m_CabinTrackInfo.u16TrackIndex,m_CabinTrackInfo.u8Region,m_CabinTrackInfo.FolderPath);
				ETG_TRACE_USR4(("vStream3TotalPlayTime : Repeat Track request for Folder track\n"));
			}
		}
	}
}

/******************************************************************************
*NAME		  : vStream1TotPlayTimeResponse
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Function for sending command response to HMI for vStream1TotPlayTimeResponse(for CMP Video source)
******************************************************************************/
void clExternalMediaPlayerPlugin::vStream1TotPlayTimeResponse(uint16_t cmd_type,uint64_t entity_id)
{
	ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vStream1TotPlayTimeResponse"));
	
	ControlDataElement controlElement;
	if((m_stPrimCMP.configuration == 0) || (m_stPrimCMP.u64EntityID != entity_id))
	{
		ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vStream1TotPlayTimeResponse, ERROR! Configuration %d, EntityID %d", m_stPrimCMP.configuration, entity_id));
		return;
	}
	controlElement.configuration = m_stPrimCMP.configuration;
	controlElement.u32ControlDescIdx = STREAM1TOTPLAYTIME;
	
	//Get value of total time for the current track, this is received only once when a track is changed
	if(m_pCommControl != NULL)
	{
		m_pCommControl->GetUnsolicitedValue(controlElement);
	}
	ETG_TRACE_USR4(("vStream1TotPlayTimeResponse == controlData.strGetValue=%s  ",controlElement.strGetValue.c_str()));
	
	m_TotalTimevalue = vConvertTimeInSecs(controlElement.strGetValue);
	ETG_TRACE_USR4(("vStream1TotPlayTimeResponse == m_TotalTimevalue=%f  ",m_TotalTimevalue));

	
        vSendStringResponseToHmi(controlElement.u32ControlDescIdx,controlElement.strGetValue);
        //Send a getcontrol request to fetch the Metadata For the current video track
        vCreateGetControlCommand(m_stPrimCMP, VIDNAME);
        vCreateGetControlCommand(m_stPrimCMP, VIDEPISODE);
        if(m_bvideosourcerestore && (vConvertTimeInSecs(controlElement.strGetValue) != 0))
        {
                vCreateSetControlCommand(m_PlayControlId,MEDIA_PAUSE);
                m_bvideosourcerestore = false;
        }
	
}
/******************************************************************************
*NAME		  : vStream3TotPlayTimeResponse
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Function for sending command response to HMI for vStream3TotPlayTimeResponse(for CMP Audio source)
******************************************************************************/
void clExternalMediaPlayerPlugin::vStream3TotPlayTimeResponse(uint16_t cmd_type,uint64_t entity_id)
{
	ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vStream3TotPlayTimeResponse"));
	
	ControlDataElement controlElement;
	if((m_stPrimCMP.configuration == 0) || (m_stPrimCMP.u64EntityID != entity_id))
	{
		ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vStream3TotPlayTimeResponse, ERROR! Configuration %d, EntityID %d", m_stPrimCMP.configuration, entity_id));
		return;
	}
	controlElement.configuration = m_stPrimCMP.configuration;
	controlElement.u32ControlDescIdx = STREAM3TOTPLAYTIME;
	
	//Get value of total time for the current track, this is received only once when a track is changed
	if(m_pCommControl != NULL)
	{
		m_pCommControl->GetUnsolicitedValue(controlElement);
	}
	

	vCreateGetControlCommand(m_stPrimCMP, TRACK);
	
	m_TotalTimevalue = vConvertTimeInSecs(controlElement.strGetValue);
	ETG_TRACE_USR4(("vStream3TotPlayTimeResponse == m_TotalTimevalue=%f ",m_TotalTimevalue));
	
	vSendStringResponseToHmi(controlElement.u32ControlDescIdx,controlElement.strGetValue);
}
/******************************************************************************
*NAME		  : vVidNameResponse
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Function for sending command response to HMI for vVidNameResponse(for CMP Video source)
******************************************************************************/
void clExternalMediaPlayerPlugin::vVidNameResponse(uint16_t cmd_type,uint64_t entity_id)
{
	ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vVidNameResponse"));
	
	if((m_stPrimCMP.configuration == 0) || (m_stPrimCMP.u64EntityID != entity_id))
	{
		ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vVidNameResponse, ERROR! Configuration %d, EntityID %d", m_stPrimCMP.configuration, entity_id));
		return;
	}

	ControlDataElement controlElement;
	controlElement.configuration = m_stPrimCMP.configuration;
	controlElement.u32ControlDescIdx = VIDNAME;

	if(m_pCommControl != NULL)
	{
		if(AEM_CMD_GET_CONTROL /*25*/ == cmd_type)
		{
			controlElement.puintGetValue = pValue;
			controlElement.u16Readlen = 512;
			m_pCommControl->GetControlValue(controlElement);
			ETG_TRACE_USR4(("vVidNameResponse from get control== controlData.strGetValue=%s  ",controlElement.strGetValue.c_str()));
            vRenameifInvalidUTF8isFound(controlElement.strGetValue);
			vSendStringResponseToHmi(controlElement.u32ControlDescIdx, controlElement.strGetValue);
		}
		else if (AEM_CMD_SET_CONTROL /*24*/ == cmd_type)
		{
			m_pCommControl->GetUnsolicitedValue(controlElement);
			ETG_TRACE_USR4(("vVidNameResponse == controlData.strGetValue=%s  ",controlElement.strGetValue.c_str()));
            vRenameifInvalidUTF8isFound(controlElement.strGetValue);
			vSendStringResponseToHmi(controlElement.u32ControlDescIdx, controlElement.strGetValue);
		}
		else
		{
			ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vVidNameResponse, Command type is not matching"));
		}
	}
}
/******************************************************************************
*NAME		  : vVidEpisodeResponse
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Function for sending command response to HMI for vVidEpisodeResponse(for CMP Video source)
******************************************************************************/
void clExternalMediaPlayerPlugin::vVidEpisodeResponse(uint16_t cmd_type,uint64_t entity_id)
{
	ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vVidEpisodeResponse"));
	
	ControlDataElement controlElement;
	if((m_stPrimCMP.configuration == 0) || (m_stPrimCMP.u64EntityID != entity_id))
	{
		ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vVidEpisodeResponse, ERROR! Configuration %d, EntityID %d", m_stPrimCMP.configuration, entity_id));
		return;
	}
	controlElement.configuration = m_stPrimCMP.configuration;
	controlElement.u32ControlDescIdx = VIDEPISODE;


	if(m_pCommControl != NULL)
	{
		if(AEM_CMD_GET_CONTROL /*25*/ == cmd_type)
		{
			controlElement.puintGetValue = pValue;
			controlElement.u16Readlen = 512;
			m_pCommControl->GetControlValue(controlElement);
			ETG_TRACE_USR4(("vVidEpisodeResponse from get control== controlData.strGetValue=%s  ",controlElement.strGetValue.c_str()));
            vRenameifInvalidUTF8isFound(controlElement.strGetValue);
			vSendStringResponseToHmi(controlElement.u32ControlDescIdx, controlElement.strGetValue);
		}
		else if (AEM_CMD_SET_CONTROL /*24*/ == cmd_type)
		{
			m_pCommControl->GetUnsolicitedValue(controlElement);
			ETG_TRACE_USR4(("vVidEpisodeResponse == controlData.strGetValue=%s  ",controlElement.strGetValue.c_str()));
            vRenameifInvalidUTF8isFound(controlElement.strGetValue);
			vSendStringResponseToHmi(controlElement.u32ControlDescIdx, controlElement.strGetValue);
		}
		else
		{
			ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vVidEpisodeResponse, Command type is not matching"));
		}
	}
}

/******************************************************************************
*NAME		  : vStream1CurTrackIndexHighByteResponse
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Function for sending Track Index for vStream1CurTrackIndexHighByteResponse
******************************************************************************/
void clExternalMediaPlayerPlugin::vStream1CurTrackIndexHighByteResponse(uint16_t cmd_type,uint64_t entity_id)
{
	ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vStream1CurTrackIndexHighByteResponse"));
	
	ControlDataElement controlElement;
	if((m_stPrimCMP.configuration == 0) || (m_stPrimCMP.u64EntityID != entity_id))
	{
		ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vStream1CurTrackIndexHighByteResponse, ERROR! Configuration %d, EntityID %d", m_stPrimCMP.configuration, entity_id));
		return;
	}
	controlElement.configuration = m_stPrimCMP.configuration;
	controlElement.u32ControlDescIdx = STREAM1CURRENTTRACKINDEXHIGH;
	controlElement.uint8Value = 0;
	
	//Getvalue for the current track index
	if(m_pCommControl != NULL)
	{
		m_pCommControl->GetUnsolicitedValue(controlElement);
	}	
	m_u8TrackIndex_HighByte = controlElement.uint8Value;
	//set control for low byte once the response of high byte is recieved 
	vCreateSetControlCommand(STREAM1CURRENTTRACKINDEXLOW,m_u8TRackIndex_LowByte);
	vCreateSetControlCommand(STREAM1PLAYACTION,MEDIA_PLAY);
        m_bPlayAction = false;
	ETG_TRACE_USR4(("Track Index = High Byte  = %d  ", m_u8TrackIndex_HighByte));
}

/******************************************************************************
*NAME		  : vStream1CurTrackIndexLowByteResponse
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Function for sending Track Index for vStream1CurTrackIndexHighByteResponse
******************************************************************************/
void clExternalMediaPlayerPlugin::vStream1CurTrackIndexLowByteResponse(uint16_t cmd_type,uint64_t entity_id)
{
	ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vStream1CurTrackIndexLowByteResponse"));
		   // 00000011
	ControlDataElement controlElement;
	if((m_stPrimCMP.configuration == 0) || (m_stPrimCMP.u64EntityID != entity_id))
	{
		ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vStream1CurTrackIndexLowByteResponse, ERROR! Configuration %d, EntityID %d", m_stPrimCMP.configuration, entity_id));
		return;
	}
	controlElement.configuration = m_stPrimCMP.configuration;
	controlElement.u32ControlDescIdx = STREAM1CURRENTTRACKINDEXLOW;
	controlElement.uint8Value = 0;
	//Getvalue for the current track index
	if(m_pCommControl != NULL)
	{
		m_pCommControl->GetUnsolicitedValue(controlElement);
	}
	ETG_TRACE_USR4(("Track Index Lower byte =  %d  ", controlElement.uint8Value));
	
	uint8_t u8LowByte = controlElement.uint8Value; 			
	uint16_t SongID = (m_u8TrackIndex_HighByte<<8)|(u8LowByte & 0X00FF);
        m_CabinTrackInfo.u16TrackIndex = SongID;
   	bool playState = entPlayState_PlaySuccess;
	uint16_t sourceType = entSourceType_CMP;
	PluginServerHandler::getInstance()->sendSig_PlayStatusSignalUpdate(playState, sourceType, SongID, m_CabinTrackInfo.u16Filetype, m_CabinTrackInfo.u8ListType, m_CabinTrackInfo.u16Filter1, m_CabinTrackInfo.u16Filter2, m_CabinTrackInfo.FolderPath);
}

/******************************************************************************
*NAME		  : vDBStatusResponse
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Function for sending Track Index for vDBStatusResponse
******************************************************************************/
void clExternalMediaPlayerPlugin::vDBStatusResponse(uint16_t cmd_type,uint64_t entity_id)
{
	ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vDBStatusResponse"));
	
	ControlDataElement controlElement;
	if((m_stPrimCMP.configuration == 0) || (m_stPrimCMP.u64EntityID != entity_id))
	{
		ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vDBStatusResponse, ERROR! Configuration %d, EntityID %d", m_stPrimCMP.configuration, entity_id));
		return;
	}
	controlElement.configuration = m_stPrimCMP.configuration;
	controlElement.u32ControlDescIdx = DBSTATUS;
	controlElement.uint8Value = 0;
    uint16_t u16CAMPposition = PRIMARY_CAMPORT;
	controlElement.uint8Value = 0;
	uint16_t u16_dbState = 0;//28
	if (m_pCommControl != NULL)
	{
		if (AEM_CMD_GET_CONTROL /*25*/ == cmd_type)
		{
			controlElement.puintGetValue = pValue;
			controlElement.u16Readlen = 1;
			m_pCommControl->GetControlValue(controlElement);
            u16_dbState = controlElement.uint8Value;
			ETG_TRACE_USR4(("DB Status from Get Control = controlElement.uint8Value =%d  ", u16_dbState));
			DBStateStruct DBStruct(u16_dbState, entSourceType_CMP);
			PluginServerHandler::getInstance()->setDBState(DBStruct);
                        vSendPluginResponseToHmi(controlElement.u32ControlDescIdx, controlElement.uint8Value, u16CAMPposition); //3rd parameter wont be of use here
		}
		else if (AEM_CMD_SET_CONTROL /*24*/ == cmd_type)
		{
			m_pCommControl->GetUnsolicitedValue(controlElement);
			u16_dbState = controlElement.uint8Value;
			ETG_TRACE_USR4(("DB Status from Unsolicited response = controlElement.uint8Value =%d  ", u16_dbState));
			DBStateStruct DBStruct(u16_dbState, entSourceType_CMP);
			PluginServerHandler::getInstance()->setDBState(DBStruct);
                        vSendPluginResponseToHmi(controlElement.u32ControlDescIdx, controlElement.uint8Value, u16CAMPposition); //3rd parameter wont be of use here
		}
		else
		{
			ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vUSBAudioStatusResponse, Command type is not matching"));
		}
                if(u16_dbState==0)
                {
                    m_bPlayAction = false;
                    m_CabinTrackInfo.AuidoPLaybackMode = Repeat_List;
                    m_CabinTrackInfo.VideoPLaybackMode = Repeat_List;
                }
	}

	// Read the CRC if the DB State is 5 "DB No Change"
	if (u16_dbState == 5)
	vCreateGetControlCommand(m_stPrimCMP, CHECKSUM);
	
}

/******************************************************************************
*NAME		  : vUSBAudioStatusResponse
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Function for vUSBAudioStatusResponse
******************************************************************************/
void clExternalMediaPlayerPlugin::vUSBAudioStatusResponse(uint16_t cmd_type,uint64_t entity_id)
{
	ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vUSBAudioStatusResponse"));
	
	ControlDataElement controlElement;
	if((m_stPrimCMP.configuration == 0) || (m_stPrimCMP.u64EntityID != entity_id))
	{
		ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vUSBAudioStatusResponse, ERROR! Configuration %d, EntityID %d", m_stPrimCMP.configuration, entity_id));
		return;
	}
	controlElement.configuration = m_stPrimCMP.configuration;
	controlElement.u32ControlDescIdx = USBAUDIOSTATUS;
	
	//Getvalue of USB audio status

	if(m_pCommControl != NULL)
	{
		if(AEM_CMD_GET_CONTROL /*25*/ == cmd_type)
		{
			controlElement.puintGetValue = pValue;
			controlElement.u16Readlen = 1;
			m_pCommControl->GetControlValue(controlElement);
			m_u8audioStatus = controlElement.uint8Value;
			ETG_TRACE_USR4(("USB Audio Status from Get Control = controlElement.uint8Value =%d  ",m_u8audioStatus));
			//send the value to HMI to update
			uint16_t u16CAMPposition = PRIMARY_CAMPORT;
			vSendPluginResponseToHmi(controlElement.u32ControlDescIdx, controlElement.uint8Value, u16CAMPposition); //3rd parameter wont be of use here
		}
		else if (AEM_CMD_SET_CONTROL /*24*/ == cmd_type)
		{
			controlElement.uint8Value = 0;
			m_pCommControl->GetUnsolicitedValue(controlElement);
			m_u8audioStatus = controlElement.uint8Value;
			ETG_TRACE_USR4(("USB Audio Status from Unsolicited Response = controlElement.uint8Value =%d  ",m_u8audioStatus));
			uint16_t u16CAMPposition = PRIMARY_CAMPORT;
			//send the value to HMI to update
			vSendPluginResponseToHmi(controlElement.u32ControlDescIdx, controlElement.uint8Value, u16CAMPposition); //3rd parameter wont be of use here
		}
		else
		{
			ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vUSBAudioStatusResponse, Command type is not matching"));
		}
	}
}


/******************************************************************************
*NAME		  : vHDMIStatusResponse
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Function for vHDMIStatusResponse
******************************************************************************/
void clExternalMediaPlayerPlugin::vHDMIStatusResponse(uint16_t cmd_type,uint64_t entity_id)
{
	ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vHDMIStatusResponse, commandtype is %d ", cmd_type));
	
	ControlDataElement controlElement;
	// controlElement.configuration = m_stPrimCMP.configuration;
	uint16_t u16CMPPosition = 0;
	controlElement.configuration = getConfigurationAndPositon(entity_id, u16CMPPosition); //function also updates the corresponding position
	if(controlElement.configuration == 0 || u16CMPPosition == 0) {
		ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vHDMIStatusResponse configuration 0, CMP is not available"));
		return;
	}
	controlElement.u32ControlDescIdx = HDMISTATUS;

	if(m_pCommControl != NULL)
	{
		uint8_t u8_hdmiStatus = 0;//17
		if(AEM_CMD_GET_CONTROL /*25*/ == cmd_type)
		{
			controlElement.puintGetValue = pValue;
			controlElement.u16Readlen = 1;
			m_pCommControl->GetControlValue(controlElement);
			u8_hdmiStatus = controlElement.uint8Value;
			ETG_TRACE_USR4(("HDMI Status from Get Control = controlElement.uint8Value =%d , position = %d ",u8_hdmiStatus,u16CMPPosition));
			vSendPluginResponseToHmi(controlElement.u32ControlDescIdx, controlElement.uint8Value, u16CMPPosition);
		}
		else if (AEM_CMD_SET_CONTROL /*24*/ == cmd_type)
		{
			controlElement.uint8Value = 0;
			m_pCommControl->GetUnsolicitedValue(controlElement);
			u8_hdmiStatus = controlElement.uint8Value;
			ETG_TRACE_USR4(("HDMI Status from Unsolicited Response = controlElement.uint8Value =%d , position = %d ",u8_hdmiStatus,u16CMPPosition));
			vSendPluginResponseToHmi(controlElement.u32ControlDescIdx, controlElement.uint8Value, u16CMPPosition);
		}
		else
		{
			ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vHDMIStatusResponse, Command type is not matching"));
		}
	}	
}
/******************************************************************************
*NAME		  : getConfigurationAndPositon
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Function to return configuration and Position for corresponding entity id
******************************************************************************/
avdecc_lib::configuration_descriptor* clExternalMediaPlayerPlugin::getConfigurationAndPositon(uint64_t u64Entity_id, uint16_t& u16CMPPosition)
{
	ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::getConfigurationAndPositon Entity_id=%d", u64Entity_id));
	if(m_stPrimCMP.u64EntityID == u64Entity_id){
		u16CMPPosition = PRIMARY_CAMPORT;
		return m_stPrimCMP.configuration;
	}
	if(m_stSecCMP.u64EntityID == u64Entity_id){
		u16CMPPosition = SECOND_CAMPORT;
		return m_stSecCMP.configuration;
	}
	if(m_stTerCMP.u64EntityID == u64Entity_id){
		u16CMPPosition = THIRD_CAMPORT;
		return m_stTerCMP.configuration;
	}
	u16CMPPosition = 0;
	return m_stTerCMP.configuration;
}

/******************************************************************************
*NAME		  : vCheckSumStatusResponse
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Function for vCheckSumStatusResponse
******************************************************************************/
void clExternalMediaPlayerPlugin::vCheckSumStatusResponse(uint16_t cmd_type,uint64_t entity_id)
{
	ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vCheckSumStatusResponse"));
	if((m_stPrimCMP.configuration == 0) || (m_stPrimCMP.u64EntityID != entity_id))
	{
		ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vCheckSumStatusResponse, ERROR! Configuration %d, EntityID %d", m_stPrimCMP.configuration, entity_id));
		return;
	}
	ControlDataElement controlElement;
	controlElement.configuration = m_stPrimCMP.configuration;
	controlElement.u32ControlDescIdx = CHECKSUM;
	std::string NewCheckSum="";
	
	if(m_pCommControl != NULL)
	{
		if(AEM_CMD_GET_CONTROL /*25*/ == cmd_type)
		{
			controlElement.puintGetValue = pValue;
			controlElement.u16Readlen = 512;
			m_pCommControl->GetControlValue(controlElement);
			NewCheckSum = controlElement.strGetValue;
			ETG_TRACE_USR4(("Checksum from Get Control =%s  ",NewCheckSum.c_str()));
		}
		else if (AEM_CMD_SET_CONTROL /*24*/ == cmd_type)
		{
			m_pCommControl->GetUnsolicitedValue(controlElement);
			NewCheckSum = controlElement.strGetValue;
			ETG_TRACE_USR4(("Checksum from Unsolicited Response = %s  ",NewCheckSum.c_str()));
		}
		else
		{
			ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::Checksum resposne, Command type is not matching"));
		}	
	}	
	dp_tclCMRDpCMP_CRC _oDpCRC;
	char dpcrcread[255] = {0};
	_oDpCRC.s32GetData(dpcrcread,255);
	m_strPrevCheckSum.assign(dpcrcread);
	ETG_TRACE_USR4(("DP read CMP CRC = %s ",dpcrcread));
	m_strCurrentCheckSum =  NewCheckSum;
	
	bool bCRCStatus = false; 
	ETG_TRACE_USR4(("vCheckSumStatusResponse previous stored CMP checksum  =%s  ",m_strPrevCheckSum.c_str() )); 
	ETG_TRACE_USR4(("vCheckSumStatusResponse new CMP CheckSum %s ", m_strCurrentCheckSum.c_str())); 
	
	if(m_strCurrentCheckSum == m_strPrevCheckSum)
	{
		bCRCStatus = true;
	}
	else if (m_strCurrentCheckSum != "0")
	{
		ETG_TRACE_USR4(("vCheckSumStatusResponse new CheckSum is not equal to Prev CheckSum"));
	}	
	else
	ETG_TRACE_USR4(("CRC Update is Zero "));	
	
	ETG_TRACE_USR4(("CRCStatus = %d  ", bCRCStatus));
	
	//check if USB is connected, then send status to HMI and LH
	if(m_u8audioStatus)
	{
		PluginServerHandler::getInstance()->sendsig_CRCCheckStatusUpdate(bCRCStatus,entSourceType_CMP);
	
		//Send CRC Status to HMI
		vSendPluginResponseToHmi(controlElement.u32ControlDescIdx, bCRCStatus);
	}
	else
		ETG_TRACE_USR4(("CRC Update:Device not connected"));	
}


/******************************************************************************
*NAME		  : vStream3CurTrackIndexHighByteResponse
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Function for sending Track Index for audio file for vStream3CurTrackIndexHighByteResponse
******************************************************************************/
void clExternalMediaPlayerPlugin::vStream3CurTrackIndexHighByteResponse(uint16_t cmd_type,uint64_t entity_id)
{
	ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vStream3CurTrackIndexHighByteResponse"));

	ControlDataElement controlElement;
	if((m_stPrimCMP.configuration == 0) || (m_stPrimCMP.u64EntityID != entity_id))
	{
		ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vStream3CurTrackIndexHighByteResponse, ERROR! Configuration %d, EntityID %d", m_stPrimCMP.configuration, entity_id));
		return;
	}
	controlElement.configuration = m_stPrimCMP.configuration;
	controlElement.u32ControlDescIdx = STREAM3CURRENTTRACKINDEXHIGH;
	controlElement.uint8Value = 0;

	//Getvalue for the current track index
	if (m_pCommControl != NULL)
	{
		m_pCommControl->GetUnsolicitedValue(controlElement);
	}
	m_u8TrackIndex_Audio_HighByte = controlElement.uint8Value;
	//set control for low byte once the response of high byte is recieved 
	vCreateSetControlCommand(STREAM3CURRENTTRACKINDEXLOW, m_u8TRackIndex_Audio_LowByte);
	vCreateSetControlCommand(STREAM3PLAYACTION, MEDIA_PLAY);
        m_bPlayAction = false;
	ETG_TRACE_USR4(("Audio Track Index = High Byte  = %d  ", m_u8TrackIndex_Audio_HighByte));
}

/******************************************************************************
*NAME		  : vStream3CurTrackIndexLowByteResponse
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Function for sending Track Index for Audio file for vStream3CurTrackIndexLowByteResponse
******************************************************************************/
void clExternalMediaPlayerPlugin::vStream3CurTrackIndexLowByteResponse(uint16_t cmd_type,uint64_t entity_id)
{
	ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vStream3CurTrackIndexLowByteResponse"));

	ControlDataElement controlElement;
	if((m_stPrimCMP.configuration == 0) || (m_stPrimCMP.u64EntityID != entity_id))
	{
		ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vStream3CurTrackIndexLowByteResponse, ERROR! Configuration %d, EntityID %d", m_stPrimCMP.configuration, entity_id));
		return;
	}
	controlElement.configuration = m_stPrimCMP.configuration;
	controlElement.u32ControlDescIdx = STREAM3CURRENTTRACKINDEXLOW;
	controlElement.uint8Value = 0;

	//Getvalue for the current track index
	if (m_pCommControl != NULL)
	{
		m_pCommControl->GetUnsolicitedValue(controlElement);
	}
	ETG_TRACE_USR4(("Audio Track Index Lower byte =  %d  ", controlElement.uint8Value));

	uint8_t u8LowByte = 0;
	u8LowByte = controlElement.uint8Value;
	uint16_t SongID = 0; 
	SongID =(m_u8TrackIndex_Audio_HighByte << 8) | (u8LowByte & 0X00FF);
        m_CabinTrackInfo.u16TrackIndex = SongID;

	PluginServerHandler::getInstance()->sendSig_PlayStatusSignalUpdate(entPlayState_PlaySuccess, entSourceType_CMP, SongID, m_CabinTrackInfo.u16Filetype, m_CabinTrackInfo.u8ListType, m_CabinTrackInfo.u16Filter1, m_CabinTrackInfo.u16Filter2, m_CabinTrackInfo.FolderPath);
}

/************************************************************************
*FUNCTION: 		vSetStream3CurrentTrackIndex
*DESCRIPTION: 		for creating command object to set the STREAM3CURRENTTRACKINDEX
*REQUIREMENT ID:	SWFL-2891,SWFL-2889
*PARAMETER:		uint32_t
*RETURNVALUE: 		void
************************************************************************/
void clExternalMediaPlayerPlugin::vSetStream3CurrentTrackIndex(uint16_t stream3TrackIndex)
{
	ETG_TRACE_USR4(("====vSetStream3CurrentTrackIndex  =stream3TrackIndex=%d  ", stream3TrackIndex));
	uint32_t sendTrackIndex = stream3TrackIndex;
	ETG_TRACE_USR4(("====vSetStream3CurrentTrackIndex  =sendTrackIndex=%d  ", sendTrackIndex));
	//convert 16 bit track index to 2 8 bit values
	uint8_t u8HighByte = (stream3TrackIndex >> 8) & 0X00FF;
	m_u8TRackIndex_Audio_LowByte = (stream3TrackIndex)& 0x00FF;

	ETG_TRACE_USR4(("====vSetStream3CurrentTrackIndex  high byte =%d lower byte = %d ", u8HighByte, m_u8TRackIndex_Audio_LowByte));

	//set the high byte of index.	
	vCreateSetControlCommand(STREAM3CURRENTTRACKINDEXHIGH, u8HighByte);
}
/******************************************************************************
*NAME		  : addPluginEventHandler
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Function to get plugin event handler, part of evenbus mechanism
******************************************************************************/
std::unique_ptr<HandlerRegistration> clExternalMediaPlayerPlugin::addPluginEventHandler()
{
	ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::addPluginEventHandler"));
	PluginEvtListenerBase<clExternalMediaPlayerPlugin>* EMPPluginEventListener = PluginEvtListenerBase<clExternalMediaPlayerPlugin>::getInstance();
	std::unique_ptr<HandlerRegistration> handle (EventBus::AddHandler<PluginEventBase<clExternalMediaPlayerPlugin>>(EMPPluginEventListener));
	return std::move(handle);
}

/******************************************************************************185
*NAME		  : addPluginEventHandler
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Function to get plugin event, part of evenbus mechanism
******************************************************************************/
std::unique_ptr<Event> clExternalMediaPlayerPlugin::getPluginEvent(Object* sender, const std::string msg, boost::shared_ptr<EventDataUtility>evalue)
{
	ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::getPluginEvent"));
	std::unique_ptr<PluginEventBase<clExternalMediaPlayerPlugin>> EMPPluginEvent(new PluginEventBase<clExternalMediaPlayerPlugin>(sender,this,msg,evalue));
	return std::move(EMPPluginEvent);
}

/******************************************************************************
*NAME		  : addPluginEventHandler
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Function to get plugin list event, part of evenbus mechanism
******************************************************************************/
std::unique_ptr<Event> clExternalMediaPlayerPlugin::getPluginListEvent(Object* sender, const std::string msg, boost::shared_ptr<EventDataUtility>eData,
																const boost::shared_ptr<EventListDataUtility>& eListData)
{
	ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::getPluginListEvent"));
	std::unique_ptr<PluginEventBase<clExternalMediaPlayerPlugin>> EMPPluginEvent(new PluginEventBase<clExternalMediaPlayerPlugin>(sender,this,msg,eData,eListData));
	return std::move(EMPPluginEvent);
}

/******************************************************************************
*NAME		  : setPluginData
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Based on the key, the corresponding function is invoked
******************************************************************************/
void clExternalMediaPlayerPlugin::setPluginData(std::string key, const boost::shared_ptr<EventDataUtility>& eValue)
{
	ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::setPluginData map size %d ", m_MemberFunctionMap.size()));

	if(NULL != eValue.get())
	{
		uint16_t u16ControlVal = 0;
		uint16_t u16ControlId = 0;
		std::vector<EventDataItem*> dataItems = eValue->getData();
		if(dataItems.size()>1)
		{
			EventDataItem* eventDataItemControl =  dataItems.at(CONTROL_ID);
			EventDataItem* eventDataItemControlVal =  dataItems.at(CONTROL_VAL);
			if(NULL != eventDataItemControl)
			{
				EventDataItem::Data eventData = eventDataItemControl->getData();
				if(EventDataItem::UINT16 == eventData._type)
				{
					u16ControlId = eventData._value._uint16Value;
				}
			}
			if(NULL != eventDataItemControlVal)
			{
				EventDataItem::Data eventData = eventDataItemControlVal->getData();
				if(EventDataItem::UINT16 == eventData._type)
				{
					u16ControlVal = eventData._value._uint16Value;
				}
			}
			
			ETG_TRACE_USR4(("====setPluginData  =u16ControlId=%d  u16ControlVal=%d  ",u16ControlId,u16ControlVal));
			if ((u16ControlId == STREAM3PLAYACTION) || (u16ControlId == STREAM1PLAYACTION))
			{
				if(!m_bPlayAction)
				{
                    if ((u16ControlVal == MEDIA_PAUSE)||(u16ControlVal == MEDIA_PLAY))
					{
						if (m_CabinTrackInfo.u16TrackIndex != TRACKINDEX_UNAVAILABLE)
							vCreateSetControlCommand(u16ControlId,u16ControlVal);
						else
							vCheckCRCStatus();
					}
					else
					{
						m_PlayControlId = u16ControlId;
						m_bPlayAction = true;
						vHandlePlaybackAction(dataItems);
					}
				}
				else
				{
					ETG_TRACE_USR4(("====set_PluginData  ignore already one playaction under process =u16ControlId=%d  u16ControlVal=%d  ",u16ControlId,u16ControlVal));
				}
			}
			else if ((u16ControlId == STREAM1REPEATMODE) || (u16ControlId == STREAM3REPEATMODE))
			{
				//Call the function which updates the current play back mode setting property
				vHandlePlaybackMode(dataItems);	
			}
			else if(u16ControlId == MEDIA_USB_DB_CRC_UPDATE)
			{
				//Call the function which update the USB entities to HMI
				vUsbentitiesupdate();
			}	
			else		
			vCreateSetControlCommand(u16ControlId,u16ControlVal);
		}

	}
}

void clExternalMediaPlayerPlugin::vHandlePlaybackMode(std::vector<EventDataItem*> dataItems)
{
	ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vHandlePlaybackMode"));
	uint8_t selectionValues[3] = {0,0,0},audiovideo_playbackmode;
	uint16_t u16CAMPposition = PRIMARY_CAMPORT;
	if(dataItems.size()>1)
	{
	  for(uint8_t index = 0; (index<3)&&(index < (dataItems.size())); index++)
      {
		 EventDataItem* eventDataItem =  dataItems.at(index);
         if(NULL != eventDataItem)
         {
			const EventDataItem::Data& eventData = eventDataItem->getData();
            if(EventDataItem::UINT8 == eventData._type || EventDataItem::UINT16 == eventData._type)
			{
				selectionValues[index] = eventData._value._uint8Value;
				ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vHandlePlaybackMode , selectionValues[%d]:%d", index, selectionValues[index]));
			}
		 }
	  }	
	}
	
	//playbackmode,sourcetype,regionid,subsourcetype
	ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::PlaybackMode value:%d", selectionValues[1]));	
	
	
	if(selectionValues[0] == STREAM1REPEATMODE || m_CabinTrackInfo.u8ListType == MEDIA_LISTTYPE_FOLDER)
	{
		m_CabinTrackInfo.VideoPLaybackMode = selectionValues[1];
		PlayBackModeStruct ModeStruct(selectionValues[1],entSourceType_CMP,selectionValues[2],entSubSourceType_CMP_VIDEO);
                PluginServerHandler::getInstance()->setPlayBackMode(ModeStruct);
		vSendPluginResponseToHmi(STREAM1REPEATMODE,selectionValues[1],u16CAMPposition);
		ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::CMP video_playback mode = %d ", m_CabinTrackInfo.VideoPLaybackMode));
	}
	else
	{
		m_CabinTrackInfo.AuidoPLaybackMode = selectionValues[1];
		PlayBackModeStruct ModeStruct(selectionValues[1],entSourceType_CMP,selectionValues[2],entSubSourceType_CMP_AUDIO);
                PluginServerHandler::getInstance()->setPlayBackMode(ModeStruct);
		vSendPluginResponseToHmi(STREAM3REPEATMODE,selectionValues[1],u16CAMPposition);
		ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::CMP audio playback mode = %d ", m_CabinTrackInfo.AuidoPLaybackMode));
	}	

	//uint8_t Higher 4 bits for audio and Lower 4 bits for video playback mode store to datapool
        audiovideo_playbackmode = (m_CabinTrackInfo.AuidoPLaybackMode & 0x0f)|((m_CabinTrackInfo.VideoPLaybackMode<<4) & 0xf0);
	//write to datapool
	dp_tclCMRDpCMP_PLaybackMode _oDpPLaybackMode;
	_oDpPLaybackMode.s32SetData(audiovideo_playbackmode);
	ETG_TRACE_USR4(("DP write CMP Playback mode = %d ", audiovideo_playbackmode));
	
}

void clExternalMediaPlayerPlugin::vHandlePlaybackAction(std::vector<EventDataItem*> dataItems)
{
		ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::HandlePlaybackAction"));
	ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::HandlePlaybackAction dataitem size = %d ", dataItems.size()));
        uint8_t selectionValues[3] = {0,0,0};
        uint16_t selectionValues1[6] = {0,0,0,0,0,0};
	std::string Sfolderpath = "";
	if(dataItems.size()>1)
	{	
		for(uint8_t index = 0,index1=0,index2=0; index < dataItems.size(); index++)
		{
			 EventDataItem* eventDataItem =  dataItems.at(index);
			 if(NULL != eventDataItem)
			 {
				const EventDataItem::Data& eventData = eventDataItem->getData();
				switch(eventData._type){
					case EventDataItem::UINT8:
						selectionValues[index1] = eventData._value._uint8Value;
						ETG_TRACE_USR4(("clRouterPlugin::HandlePlaybackAction  eventData UINT8[%d] = %d",index1,selectionValues[index1]));
						index1++;
						break;
					case EventDataItem::UINT16:
						selectionValues1[index2] = eventData._value._uint16Value;
						ETG_TRACE_USR4(("clRouterPlugin::HandlePlaybackAction  eventData UINT16[%d] = %d",index2,selectionValues[index2]));
						index2++;
						break;
					case EventDataItem::STRING:
						Sfolderpath = eventData._value._stringValue->c_str();
						ETG_TRACE_USR4(("clRouterPlugin::HandlePlaybackAction  eventData string = %s",Sfolderpath.c_str()));
						break;
					default:
						ETG_TRACE_USR4(("clRouterPlugin::HandlePlaybackAction Wrong data Update"));
				}
			 }
		}	
		//UINT8
                m_CabinTrackInfo.u8Region = selectionValues[0];
                m_CabinTrackInfo.u8ListType = selectionValues[1];
                //m_CabinTrackInfo.SubSourceType = selectionValues[2];
		if(m_PlayControlId == STREAM1PLAYACTION)
		{
			m_CabinTrackInfo.SubSourceType = entSubSourceType_CMP_VIDEO;
		}
		else
		{
			m_CabinTrackInfo.SubSourceType = entSubSourceType_CMP_AUDIO;
		}
		//UINT16
                m_CabinTrackInfo.u16TrackIndex = selectionValues1[2];
                m_CabinTrackInfo.u16Filetype = selectionValues1[3];
               // m_CabinTrackInfo.u16Filter1 = selectionValues1[4];
                //m_CabinTrackInfo.u16Filter2 = selectionValues1[5];
		//string
		m_CabinTrackInfo.FolderPath = Sfolderpath;
			
                ETG_TRACE_USR4(("clRouterPlugin::HandlePlaybackAction Control Id :%d Control Value:%d List Type : %d Position = %d SubSourceType =%d ",selectionValues1[0],selectionValues1[1],selectionValues[1],selectionValues1[2],selectionValues[2]));
                ETG_TRACE_USR4(("clRouterPlugin::HandlePlaybackAction   u16Filetype=%d FolderPath=%s",selectionValues1[3],Sfolderpath.c_str()));
                vHandleAudioVideoPlayBackAction(selectionValues1[0],selectionValues1[1]);
	}
}

void clExternalMediaPlayerPlugin::vHandleAudioVideoPlayBackAction(uint8_t u8ControlId, uint8_t u8PlayAction)
{
	ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::HandleAudioVideoPlayBackAction controlid = %d ",u8ControlId));
	switch (u8PlayAction)
	{
	    case MEDIA_NEXT : 
			ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::Request for nexttrackIndex"));
			PluginServerHandler::getInstance()->sendsig_ReqTrackIndex(MEDIA_NEXTTRACKINDEXREQ,entSourceType_CMP,m_CabinTrackInfo.u8ListType, m_CabinTrackInfo.u16Filter1,m_CabinTrackInfo.u16Filter2,m_CabinTrackInfo.u16TrackIndex,m_CabinTrackInfo.u8Region,m_CabinTrackInfo.FolderPath); 
			break;
		case MEDIA_PREVIOUS : 
				ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::Request for prevtrackIndex"));
				PluginServerHandler::getInstance()->sendsig_ReqTrackIndex(MEDIA_PREVTRACKINDEXREQ,entSourceType_CMP,m_CabinTrackInfo.u8ListType, m_CabinTrackInfo.u16Filter1,m_CabinTrackInfo.u16Filter2,m_CabinTrackInfo.u16TrackIndex,m_CabinTrackInfo.u8Region,m_CabinTrackInfo.FolderPath);
				break;
		case MEDIA_SOURCEACTIVE :
				ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::MEDIA_PLAY"));
				//check CRC status
				vCheckCRCStatus();
				break;
		case MEDIA_BROWSE:
				ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::MEDIA_BROWSE u16TrackIndex = %d ",m_CabinTrackInfo.u16TrackIndex));
				PluginServerHandler::getInstance()->sendsig_ReqTrackIndex(MEDIA_SELECTTRACKINDEXREQ,entSourceType_CMP,m_CabinTrackInfo.u8ListType, m_CabinTrackInfo.u16Filter1,m_CabinTrackInfo.u16Filter2,m_CabinTrackInfo.u16TrackIndex,m_CabinTrackInfo.u8Region,m_CabinTrackInfo.FolderPath);
            	vSetStreamCurrentTrackIndex(m_CabinTrackInfo.u16TrackIndex,m_CabinTrackInfo.u16Filetype);
				//Playback mode updates to HMI
				if(m_CabinTrackInfo.u8ListType == MEDIA_LISTTYPE_SONG)
				{
					vSendPluginResponseToHmi(STREAM3REPEATMODE,m_CabinTrackInfo.AuidoPLaybackMode);
				}
				else
				{
					vSendPluginResponseToHmi(STREAM1REPEATMODE,m_CabinTrackInfo.VideoPLaybackMode);
				}
				break;
		case MEDIA_VIDEOSOURCERESTORE :
				//calling function for video source restoration in pause state
				vVideosourcerestoreinpoause();
				break;
		default: 
				ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::setplayaction : Pause"));
				vCreateSetControlCommand(u8ControlId,u8PlayAction);
				break;
	}
}
/******************************************************************************
*NAME		  : vCreateSetControlCommand
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Send the SetControl command to the Pluginbase by Communication control
******************************************************************************/
void clExternalMediaPlayerPlugin::vCreateSetControlCommand(uint16_t u16ControlId,uint16_t u16ControlVal)
{
	ETG_TRACE_USR4(("====vCreateSetControlCommand  =u16ControlId=%d  u16ControlVal=%d  ",u16ControlId,u16ControlVal));
	
	m_u16LastExecutedSetCommand = u16ControlId;//Set value received from HMI
	
	ControlDataElement controlElement;
	controlElement.configuration = m_stPrimCMP.configuration;
	controlElement.u32ControlDescIdx = u16ControlId;
	controlElement.uint8Value = u16ControlVal;
		
	if((m_pCommControl != NULL) && (m_stPrimCMP.configuration != 0))
	{
		m_pCommControl->setControl(controlElement);
	}
}
/******************************************************************************
*NAME		  : setPluginListData
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Based on the key, the corresponding function is invoked
******************************************************************************/
void clExternalMediaPlayerPlugin::setPluginListData(std::string key, const boost::shared_ptr<EventDataUtility>& eData,
		const boost::shared_ptr<EventListDataUtility>& eListData){
   ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::setPluginListData"));
	
}

/************************************************************************
 *FUNCTION: 		vSendStringResponseToHmi
 *DESCRIPTION: 		or sending command response to HMI
 *REQUIREMENT ID:	SWFL-2957,SWFL-2969
 *PARAMETER:		uint16_t,std::string
 *RETURNVALUE: 		void
 ************************************************************************/
void clExternalMediaPlayerPlugin::vSendStringResponseToHmi(uint16_t u16DescIndex, std::string strData, uint16_t u16CMPPosition){
	ETG_TRACE_USR4(("==== vSendStringResponseToHmi  =u16DescIndex=%d strData=%s  ",u16DescIndex,strData.c_str()));
	boost::shared_ptr<EventDataUtility> pluginData(EventDataUtility::newEventDataUtility());
	if(NULL != pluginData.get())
	{
		pluginData->addEventData(u16DescIndex);
		pluginData->addEventData(strData.c_str());
		pluginData->addEventData(u16CMPPosition); //Sending PRIMARY_CAMPORT as the default 3rd parameter.
	}
	RecvMsgFromPlugin(PLUGIN_NAME_MEDIA,CTRLBLOCK_NAME_MEDIA_PLAYBACKACTION,pluginData);

}

/************************************************************************
*FUNCTION: 		vSendPluginResponseToHmi
*DESCRIPTION: 		sending command response to HMI with uint8_t value
*REQUIREMENT ID:	SWFL-2957,SWFL-2969
*PARAMETER:		uint16_t,uint8_t
*RETURNVALUE: 		void
************************************************************************/
void clExternalMediaPlayerPlugin::vSendPluginResponseToHmi(uint32_t u32DescIndex, uint8_t uint8Value, uint16_t u16CMPPosition)
{
	ETG_TRACE_USR4(("vSendPluginResponseToHmi  =u32DescIndex=%d uint8Value=%d  ", u32DescIndex, uint8Value));
	boost::shared_ptr<EventDataUtility> pluginData(EventDataUtility::newEventDataUtility());
	if (NULL != pluginData.get())
	{
		pluginData->addEventData(u32DescIndex);
		pluginData->addEventData(uint8Value);
		pluginData->addEventData(u16CMPPosition);
	}
	switch (u32DescIndex)
	{
	case USBAUDIOSTATUS:
		ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vSendPluginResponseToHmi  USBAUDIOSTATUS"));
		RecvMsgFromPlugin(PLUGIN_NAME_MEDIA, CTRLBLOCK_NAME_MEDIA_USBAUDIOSTATUS, pluginData);
		break;		
	case HDMISTATUS:
		ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vSendPluginResponseToHmi  HDMISTATUS"));
		RecvMsgFromPlugin(PLUGIN_NAME_MEDIA, CTRLBLOCK_NAME_MEDIA_HDMISTATUS, pluginData);
		break;
	case SIGNAL_SOURCE_CHANGE:
			ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vSendPluginResponseToHmi  SIGNAL_SOURCE_CHANGE"));
			if(NULL != pluginData.get())
			{
				pluginData->addEventData(m_CabinTrackInfo.u16TrackIndex);
				pluginData->addEventData(m_CabinTrackInfo.FolderPath);
				pluginData->addEventData(m_CabinTrackInfo.u8ListType);
				pluginData->addEventData(m_CabinTrackInfo.SubSourceType);
				pluginData->addEventData(m_CabinTrackInfo.u8Region);
			}
			RecvMsgFromPlugin(PLUGIN_NAME_MEDIA,CTRLBLOCK_NAME_MEDIA_SIGNAL_SOURCE_CHANGE,pluginData);
			break;
	default:
		ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vSendPluginResponseToHmi  DEFAULT"));
		RecvMsgFromPlugin(PLUGIN_NAME_MEDIA,CTRLBLOCK_NAME_MEDIA_PLAYBACKACTION, pluginData);
		break;
	}
}

/************************************************************************
 *FUNCTION: 		vCreateGetControlCommand
 *DESCRIPTION: 		for creating command object
 *REQUIREMENT ID:	SWFL-2891,SWFL-2889
 *PARAMETER:		uint16_t
 *RETURNVALUE: 		void
 ************************************************************************/
void clExternalMediaPlayerPlugin::vCreateGetControlCommand(stEntityDesc EntityDescriptor, uint16_t u16ControlId)
{
	ETG_TRACE_USR4(("====vCreateGetControlCommand  =u16ControlId=%d  ",u16ControlId));
	
	ControlDataElement controlElement;
	if(EntityDescriptor.configuration == 0) {
		ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vCreateGetControlCommand, Primary CMP unavailable"));
		return;
	}
	controlElement.configuration = EntityDescriptor.configuration;
	controlElement.u32ControlDescIdx = u16ControlId;
		
	if(m_pCommControl != NULL)
	{
		m_pCommControl->getControl(controlElement);
	}
}
/************************************************************************
 *FUNCTION: 		vSetStreamCurrentTrackIndex
 *DESCRIPTION: 		for creating command object to set the STREAM1CURRENTTRACKINDEX
 *REQUIREMENT ID:	SWFL-2891,SWFL-2889
 *PARAMETER:		uint32_t, uint16_t
 *RETURNVALUE: 		void
 ************************************************************************/
 
void clExternalMediaPlayerPlugin::vSetStreamCurrentTrackIndex(uint16_t streamTrackIndex,uint16_t Filetype)
{
	ETG_TRACE_USR4(("====vSetStreamCurrentTrackIndex  =streamTrackIndex=%d  ",streamTrackIndex));
	ETG_TRACE_USR4(("====vSetStreamCurrentTrackIndex  =m_PlayControlId=%d  ",m_PlayControlId));
	ETG_TRACE_USR4(("====vSetStreamCurrentTrackIndex  =Filetype=%d  ",Filetype));
	if (streamTrackIndex != TRACKINDEX_UNAVAILABLE )
	{
		if(m_CabinTrackInfo.u8ListType!=MEDIA_LISTTYPE_FOLDER)
		{
			if (m_PlayControlId == STREAM1PLAYACTION)
			{
				vSetStream1CurrentTrackIndex (streamTrackIndex);
			}
			else if(m_PlayControlId == STREAM3PLAYACTION)
			{
				vSetStream3CurrentTrackIndex(streamTrackIndex);
			}
			else
				ETG_TRACE_USR4(("====vSetStreamCurrentTrackIndex nothing to be done "));
		}
		else
		{
			vSetStreamFolderCurrentTrackIndex(streamTrackIndex,Filetype);
		}
	}	
	else
	{
		ETG_TRACE_USR4(("====Invalid TRack Index Update===="));
		m_CabinTrackInfo.u16TrackIndex = streamTrackIndex;
		vCreateSetControlCommand(m_PlayControlId,MEDIA_PAUSE);
		dp_tclCMRDpCMP_Listtrackindex _oDpListTrackIndex;
		_oDpListTrackIndex.s32SetData(streamTrackIndex);
        m_bPlayAction = false;
		ETG_TRACE_USR4(("DP write Track Index = %d ", streamTrackIndex));
	}
}

void clExternalMediaPlayerPlugin::vSetStreamFolderCurrentTrackIndex(uint32_t FolderstreamTrackIndex,uint16_t FolderFiletype)
{
	 ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vSetStreamFolderCurrentTrackIndex"));

	if(FolderFiletype == m_CabinTrackInfo.u16Filetype)
	{
		if (m_PlayControlId == STREAM1PLAYACTION)
		{
			vSetStream1CurrentTrackIndex(FolderstreamTrackIndex);
		}
		else if(m_PlayControlId == STREAM3PLAYACTION)
		{
			vSetStream3CurrentTrackIndex(FolderstreamTrackIndex);
		}
		else
			ETG_TRACE_USR4(("====vSetStreamCurrentTrackIndex nothing to be done "));
	}
	else
	{
		m_bPlayAction = false;
		m_CabinTrackInfo.u16TrackIndex = FolderstreamTrackIndex;
		if(FolderFiletype == entFileType_Video)
		{
			m_CabinTrackInfo.SubSourceType = entSubSourceType_CMP_VIDEO;
		}
		else if(FolderFiletype == entFileType_Audio)
		{
			m_CabinTrackInfo.SubSourceType = entSubSourceType_CMP_AUDIO;
		}
		else
		{
			ETG_TRACE_USR4(("Wrong FolderFiletype type update"));
			return;
		}
		VWritedatapool();
		vSendPluginResponseToHmi(SIGNAL_SOURCE_CHANGE, FolderFiletype);
	}
}
 
void clExternalMediaPlayerPlugin::vSetStream1CurrentTrackIndex(uint16_t stream1TrackIndex)
{
	ETG_TRACE_USR4(("====vSetStream1CurrentTrackIndex  =stream1TrackIndex=%d  ",stream1TrackIndex));
	uint32_t sendTrackIndex = stream1TrackIndex;
	ETG_TRACE_USR4(("====vSetStream1CurrentTrackIndex  =sendTrackIndex=%d  ",sendTrackIndex));
	//set the stream1currenttack index to play the recived track index from listhandler.
	uint8_t u8HighByte = (stream1TrackIndex>>8) & 0X00FF;
	m_u8TRackIndex_LowByte = (stream1TrackIndex) & 0x00FF; 	

	ETG_TRACE_USR4(("====vSetStream1CurrentTrackIndex  high byte =%d lower byte = %d ",u8HighByte,m_u8TRackIndex_LowByte));

	//set the stream1currenttack index to play the recived track index from listhandler.	
	vCreateSetControlCommand(STREAM1CURRENTTRACKINDEXHIGH,u8HighByte);
}
/************************************************************************
 *FUNCTION: 		vConvertTimeInSecs
 *DESCRIPTION: 		Time string(hh:mm:ss) is passed as parameter to function getSlidertime,extracting hh,mm,ss separately to string and
						converting to double using atof and converting hours(*3600) and mins(*60) to seconds and summing up those to totaltime and returning the totaltime
 *REQUIREMENT ID:	SWFL-2891,SWFL-2889
 *PARAMETER:		string
 *RETURNVALUE: 		double
 ************************************************************************/
double clExternalMediaPlayerPlugin::vConvertTimeInSecs(std::string timevalue)
{
	ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vConvertTimeInSecs hh:mm:ss : %s", timevalue.c_str()));

	   vector <string> timetokens;
	   string hr = "";
	   string min = "";
	   string sec = "";
	   double hr1 = 0;
	   double hour = 0;
	   double min1 = 0;
	   double minute = 0;
	   double sec1 = 0;
	   double second = 0;
	   double totaltime = 0;
	   stringstream check1(timevalue);
	   string intermediate = "";
	   while (getline(check1, intermediate, ':'))
	   {
		  timetokens.push_back(intermediate);
	   }
	   ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vConvertTimeInSecs timetokens.size() : %d", timetokens.size()));
	   if (timetokens.size() >= 1)
	   {
		  hr = timetokens[0];
		  hr1 = atof(hr.c_str());
		  hour = hr1 * 3600;
		  min = timetokens[1];
		  min1 = atof(min.c_str());
		  minute = min1 * 60;
		  sec = timetokens[2];
		  sec1 = atof(sec.c_str());
		  second = sec1;
		  totaltime = hour + minute + second;
		  ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vConvertTimeInSecs totaltime : %d", totaltime));
		  timetokens.clear();//clearing the vector
	   }
	   return totaltime;
}

/******************************************************************************
*NAME		  : vTrackNameResponse
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Function for sending command response to HMI for CMP Audio source with playing track name
******************************************************************************/
void clExternalMediaPlayerPlugin::vTrackNameResponse(uint16_t cmd_type,uint64_t entity_id)
{
	ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vTrackNameResponse"));

	ControlDataElement controlElement;
	if(m_stPrimCMP.configuration == 0) {
		ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vTrackNameResponse, Primary CMP unavailable"));
		return;
	}
	controlElement.configuration = m_stPrimCMP.configuration;
	controlElement.u32ControlDescIdx = TRACK;


	if(m_pCommControl != NULL)
	{
		if(AEM_CMD_GET_CONTROL /*25*/ == cmd_type)
		{
			controlElement.puintGetValue = pValue;
			controlElement.u16Readlen = 512;
			m_pCommControl->GetControlValue(controlElement);
			ETG_TRACE_USR4(("Track Name == from get control== controlData.strGetValue=%s  ",controlElement.strGetValue.c_str()));
            vRenameifInvalidUTF8isFound(controlElement.strGetValue);
			vSendStringResponseToHmi(controlElement.u32ControlDescIdx, controlElement.strGetValue);
		}
		else if (AEM_CMD_SET_CONTROL /*24*/ == cmd_type)
		{
			m_pCommControl->GetUnsolicitedValue(controlElement);
			ETG_TRACE_USR4(("Track Name == controlData.strGetValue=%s  ",controlElement.strGetValue.c_str()));
            vRenameifInvalidUTF8isFound(controlElement.strGetValue);
			vSendStringResponseToHmi(controlElement.u32ControlDescIdx, controlElement.strGetValue);
		}
		else
		{
			ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vTrackNameResponse, Command type is not matching"));
		}
	}
}

void clExternalMediaPlayerPlugin::vCheckCRCStatus()
{
	ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vCheckCRCStatus "));
	if (m_strCurrentCheckSum == m_strPrevCheckSum)
	{
		//Datapool read
		VReadCMPdatafromdatapool();
		
		/** set the track index ****/
		if (m_CabinTrackInfo.u16TrackIndex != TRACKINDEX_UNAVAILABLE)
		{
			ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::Request for current track index"));
			PluginServerHandler::getInstance()->sendsig_ReqTrackIndex(MEDIA_SELECTTRACKINDEXREQ,entSourceType_CMP,m_CabinTrackInfo.u8ListType, m_CabinTrackInfo.u16Filter1,m_CabinTrackInfo.u16Filter2,m_CabinTrackInfo.u16TrackIndex,m_CabinTrackInfo.u8Region,m_CabinTrackInfo.FolderPath);
			vSetStreamCurrentTrackIndex(m_CabinTrackInfo.u16TrackIndex, m_CabinTrackInfo.u16Filetype);
		}
		else
		{
			ETG_TRACE_USR4(("clRouterPlugin::Request for First track index in case of track index not available "));
			PluginServerHandler::getInstance()->sendsig_ReqTrackIndex(MEDIA_FIRSTTRACKINDEXREQ,entSourceType_CMP,m_CabinTrackInfo.u8ListType, m_CabinTrackInfo.u16Filter1,m_CabinTrackInfo.u16Filter2,m_CabinTrackInfo.u16TrackIndex,m_CabinTrackInfo.u8Region,m_CabinTrackInfo.FolderPath);
		}
	}
	else
	{
		ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::Request for First track index"));
		
		if(m_CabinTrackInfo.u8ListType == MEDIA_LISTTYPE_INVALID)
		{
			if (m_PlayControlId == STREAM1PLAYACTION)
			{
				m_CabinTrackInfo.u8ListType = MEDIA_LISTTYPE_VIDEO; 
			}
			else
			{
				m_CabinTrackInfo.u8ListType = MEDIA_LISTTYPE_SONG;
			}
		}
		
		PluginServerHandler::getInstance()->sendsig_ReqTrackIndex(MEDIA_FIRSTTRACKINDEXREQ,entSourceType_CMP,m_CabinTrackInfo.u8ListType, m_CabinTrackInfo.u16Filter1,m_CabinTrackInfo.u16Filter2,m_CabinTrackInfo.u16TrackIndex,m_CabinTrackInfo.u8Region,m_CabinTrackInfo.FolderPath); 
        //Update default source playback mode
		vSendPluginResponseToHmi(STREAM1REPEATMODE,m_CabinTrackInfo.VideoPLaybackMode);
                
		dp_tclCMRDpCMP_CRC _oDpCRC;
		uint8_t crclen = m_strCurrentCheckSum.length();
		char dpcrcwrite[crclen+1] ={0};
 		strncpy(dpcrcwrite,m_strCurrentCheckSum.c_str(),crclen);
		dpcrcwrite[crclen] = '\0';
		_oDpCRC.s32SetData(dpcrcwrite);
		ETG_TRACE_USR4(("DP write CMP CRC = %s ", dpcrcwrite));
		
		//datapool erase
		vClearCMPdatapool();
		m_strPrevCheckSum = m_strCurrentCheckSum;
	}
}
void clExternalMediaPlayerPlugin::VReadCMPdatafromdatapool()
{
	ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::VReadCMPdatafromdatapool"));
	
	uint16_t u16CAMPposition = PRIMARY_CAMPORT;
	dp_tclCMRDpCMP_Listtrackindex _oDpListTrackIndex;
	m_CabinTrackInfo.u16TrackIndex = _oDpListTrackIndex.tGetData();
   	ETG_TRACE_USR4(("CMP DP read Track Index = %d ", m_CabinTrackInfo.u16TrackIndex));
	
	dp_tclCMRDpCMP_Listtype _oDpListtype;
   	m_CabinTrackInfo.u8ListType = _oDpListtype.tGetData();
  	ETG_TRACE_USR4(("CMP DP read List type = %d ", m_CabinTrackInfo.u8ListType));
	
	uint8_t audiovideo_playbackmode;
	dp_tclCMRDpCMP_PLaybackMode _oDpPLaybackMode;
	audiovideo_playbackmode = _oDpPLaybackMode.tGetData();
	ETG_TRACE_USR4(("CMP DP read Playback mode = %d ", audiovideo_playbackmode));
	m_CabinTrackInfo.AuidoPLaybackMode = audiovideo_playbackmode & 0x0f;
	ETG_TRACE_USR4(("CMP DP read Audio PlaybackMode %d",m_CabinTrackInfo.AuidoPLaybackMode));
        m_CabinTrackInfo.VideoPLaybackMode = (audiovideo_playbackmode >> 4) & 0x0f;
	ETG_TRACE_USR4(("CMP DP read Video PlaybackMode %d",m_CabinTrackInfo.VideoPLaybackMode));
	
	dp_tclCMRDpRegion _oDpRegion;
	m_CabinTrackInfo.u8Region =  _oDpRegion.tGetData();
	ETG_TRACE_USR4(("DP read Region = %d ", m_CabinTrackInfo.u8Region));
	
	dp_tclCMRDpCMP_SubSourceType _oDpSubSourceType;
	m_CabinTrackInfo.SubSourceType = _oDpSubSourceType.tGetData();
	ETG_TRACE_USR4(("CMP DP read SubsourceType = %d ", m_CabinTrackInfo.SubSourceType));
	
	if(m_CabinTrackInfo.SubSourceType == entSubSourceType_CMP_AUDIO)
	{
		m_CabinTrackInfo.u16Filetype = entFileType_Audio;
	}
	else
	{
		m_CabinTrackInfo.u16Filetype = entFileType_Video;
	}	
	/*
	dp_tclCMRDpCMP_Filter1 _oDpFilter1;
	m_CabinTrackInfo.u16Filter1 = _oDpFilter1.tGetData();
	ETG_TRACE_USR4(("DP read Filter1 = %d ", m_CabinTrackInfo.u16Filter1));
	
	dp_tclCMRDpCMP_Filter2 _oDpFilter2;
	m_CabinTrackInfo.u16Filter2 = _oDpFilter2.tGetData();
	ETG_TRACE_USR4(("DP read Filter2 = %d ", m_CabinTrackInfo.u16Filter2));
	*/
	dp_tclCMRDpCMP_FolderPath _oDpFolderPath;
	char dpfpathread[255] = {0};
	_oDpFolderPath.s32GetData(dpfpathread,255);
	m_CabinTrackInfo.FolderPath.assign(dpfpathread);
	ETG_TRACE_USR4(("DP read Folder path = %s ",m_CabinTrackInfo.FolderPath.c_str()));
	
	//Playback mode updates to HMI
	if(m_CabinTrackInfo.u8ListType == MEDIA_LISTTYPE_SONG)
	{
		vSendPluginResponseToHmi(STREAM3REPEATMODE,m_CabinTrackInfo.AuidoPLaybackMode);
	}
	else
	{
		vSendPluginResponseToHmi(STREAM1REPEATMODE,m_CabinTrackInfo.VideoPLaybackMode);
	}
	//Playback mode updates to List Handler
	PlayBackModeStruct ModeStructAudio(m_CabinTrackInfo.AuidoPLaybackMode,entSourceType_CMP,m_CabinTrackInfo.u8Region,entSubSourceType_CMP_AUDIO), 
					   ModeStructVideo(m_CabinTrackInfo.VideoPLaybackMode,entSourceType_CMP,m_CabinTrackInfo.u8Region,entSubSourceType_CMP_VIDEO);
        PluginServerHandler::getInstance()->setPlayBackMode(ModeStructAudio);
	PluginServerHandler::getInstance()->setPlayBackMode(ModeStructVideo);
	
	//updating the valid list type if stored listtype is 0
	if(m_CabinTrackInfo.u8ListType == 0)
	{
		if (m_PlayControlId == STREAM1PLAYACTION)
		{
			m_CabinTrackInfo.u8ListType = MEDIA_LISTTYPE_VIDEO; 
		}
		else
		{
			m_CabinTrackInfo.u8ListType = MEDIA_LISTTYPE_SONG;
		}	
	}
	else
		ETG_TRACE_USR4(("VReadCMPdatafromdatapool listtype = %d ", m_CabinTrackInfo.u8ListType));
}

/******************************************************************************
*NAME          : vClearCMPdatapool
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Function to erase CMP datapool(Track index, List type, Playback mode, Rigion, subsource type ...)
******************************************************************************/
void clExternalMediaPlayerPlugin::vClearCMPdatapool()
{
	ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vClearCMPdatapool"));
	uint8_t u8Defaultvalue = 0;
	uint8_t u8DefaultPlaybackMode = MEDIA_DATAPOOL_DEFAULT_AUDIOVIDEO_PLAYBACKMODE;
	uint16_t u16Defaultvalue = 0;
	uint16_t u16DefaultTrackIndex = TRACKINDEX_UNAVAILABLE;
	
	dp_tclCMRDpCMP_Listtrackindex _oDpListTrackIndex;
	_oDpListTrackIndex.s32SetData(u16DefaultTrackIndex);
	
	dp_tclCMRDpCMP_Listtype _oDpListtype;
	_oDpListtype.s32SetData(u16Defaultvalue);
	
	dp_tclCMRDpRegion _oDpRegion;
	_oDpRegion.s32SetData(u16Defaultvalue);
	
	dp_tclCMRDpCMP_SubSourceType _oDpSubSourceType;
	_oDpSubSourceType.s32SetData(u8Defaultvalue);
	
	dp_tclCMRDpCMP_PLaybackMode _oDpPLaybackMode;
	_oDpPLaybackMode.s32SetData(u8DefaultPlaybackMode);
	/*	
	dp_tclCMRDpCMP_Filter1 _oDpFilter1;
	_oDpFilter1.s32SetData(u16Defaultvalue);
	
	dp_tclCMRDpCMP_Filter2 _oDpFilter2;
	_oDpFilter2.s32SetData(u16Defaultvalue);
	*/	
	dp_tclCMRDpCMP_FolderPath _oDpFolderPath;
        char fpath_Defaultvalue[5] = {'r','o','o','t','\0'};
	_oDpFolderPath.s32SetData(fpath_Defaultvalue);

}

/******************************************************************************
*NAME          : vUsbentitiesupdate
*DESIGN SECTION: 6.1.3
*PARAMETER:		 void
*RETURNVALUE: 	 void
*DESCRIPTION   : Function to update USB status, DB status, CRC status to HMI on HMI request
******************************************************************************/
void clExternalMediaPlayerPlugin::vUsbentitiesupdate()
{
	ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vUsbentitiesupdate"));
	if(m_u8audioStatus)
	{
		uint32_t u32_ControlDescIdx;
		uint8_t u8_Value;
		bool b_crcstatus = false; 
		//USB status update
		u32_ControlDescIdx = USBAUDIOSTATUS;
		u8_Value = m_u8audioStatus;
		vSendPluginResponseToHmi(u32_ControlDescIdx, u8_Value);
		//CRC status
		if (m_strCurrentCheckSum == m_strPrevCheckSum)
		b_crcstatus = true;
		u32_ControlDescIdx = CHECKSUM;
		vSendPluginResponseToHmi(u32_ControlDescIdx, b_crcstatus);
                if(m_stPrimCMP.configuration != 0)
                vCreateGetControlCommand(m_stPrimCMP, DBSTATUS);
	}
	else
	{
		ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vUsbentitiesupdate USB not connected, ignore update"));
	}
        //HDMI status
        if(m_stPrimCMP.configuration != 0)
            vCreateGetControlCommand(m_stPrimCMP, HDMISTATUS);

        if(m_stSecCMP.configuration != 0)
            vCreateGetControlCommand(m_stSecCMP, HDMISTATUS);

        if(m_stTerCMP.configuration != 0)
            vCreateGetControlCommand(m_stTerCMP, HDMISTATUS);
}

/******************************************************************************
*NAME          : vRenameifInvalidUTF8isFound
*DESIGN SECTION: 6.1.3
*PARAMETER:	 string &
*RETURNVALUE: 	 void
*DESCRIPTION   : Function to validate string is in UTF8 formate or not
******************************************************************************/
void clExternalMediaPlayerPlugin::vRenameifInvalidUTF8isFound(std::string &Name)
{
    ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vRenameifInvalidUTF8isFound"));
    if (Name.length() > 0)
    {
        if (!Utf8String::isValid(Name))
        {
            ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vRenameifInvalidUTF8isFound Invalid"));
            Name.assign("Unknown");
        }
    }
}

/******************************************************************************
*NAME          : vVideosourcerestoreinpoause
*DESIGN SECTION: 6.1.3
*PARAMETER:	     void
*RETURNVALUE: 	 void
*DESCRIPTION   : Function to restore video source in pause state
******************************************************************************/
void clExternalMediaPlayerPlugin::vVideosourcerestoreinpoause()
{
	ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::vVideosourcerestoreinpoause"));
	m_bvideosourcerestore = true;
	VReadCMPdatafromdatapool();
	PluginServerHandler::getInstance()->sendsig_ReqTrackIndex(MEDIA_SELECTTRACKINDEXREQ,entSourceType_CMP,m_CabinTrackInfo.u8ListType, m_CabinTrackInfo.u16Filter1,m_CabinTrackInfo.u16Filter2,m_CabinTrackInfo.u16TrackIndex,m_CabinTrackInfo.u8Region,m_CabinTrackInfo.FolderPath);
	vSetStreamCurrentTrackIndex(m_CabinTrackInfo.u16TrackIndex, m_CabinTrackInfo.u16Filetype);
}

/******************************************************************************
*NAME          : Vwritedatapool
*DESIGN SECTION: 6.1.3
*DESCRIPTION   : Function to write CMP data to datapool
******************************************************************************/
void clExternalMediaPlayerPlugin::VWritedatapool()
{
	ETG_TRACE_USR4(("clExternalMediaPlayerPlugin::Vwritedatapool"));
	
	dp_tclCMRDpCMP_Listtrackindex _oDpListTrackIndex;
	if(m_CabinTrackInfo.u16TrackIndex != _oDpListTrackIndex.tGetData())
	{
		_oDpListTrackIndex.s32SetData(m_CabinTrackInfo.u16TrackIndex);
		ETG_TRACE_USR4(("DP write Track Index = %d ", m_CabinTrackInfo.u16TrackIndex));
	}
		
	dp_tclCMRDpCMP_Listtype _oDpListtype;
	if(m_CabinTrackInfo.u8ListType != _oDpListtype.tGetData())
	{
		_oDpListtype.s32SetData(m_CabinTrackInfo.u8ListType);
		ETG_TRACE_USR4(("DP Write List type = %d ", m_CabinTrackInfo.u8ListType));
	}
	
	dp_tclCMRDpRegion _oDpRegion;
	if(m_CabinTrackInfo.u8Region != _oDpRegion.tGetData())
	{
		_oDpRegion.s32SetData(m_CabinTrackInfo.u8Region);
		ETG_TRACE_USR4(("DP write Region = %d ", m_CabinTrackInfo.u8Region));
	}
	
	dp_tclCMRDpCMP_SubSourceType _oDpSubSourceType;
	if(m_CabinTrackInfo.SubSourceType != _oDpSubSourceType.tGetData())
	{
		_oDpSubSourceType.s32SetData(m_CabinTrackInfo.SubSourceType);
		ETG_TRACE_USR4(("DP write CMP SubSource = %d ", m_CabinTrackInfo.SubSourceType));
	}
	/*	
	dp_tclCMRDpCMP_Filter1 _oDpFilter1;
	if(m_CabinTrackInfo.u16Filter1 != _oDpFilter1.tGetData())
	{
		_oDpFilter1.s32SetData(m_CabinTrackInfo.u16Filter1);
		ETG_TRACE_USR4(("DP write Track Index = %d ", m_CabinTrackInfo.u16Filter1));
	}
	
	dp_tclCMRDpCMP_Filter2 _oDpFilter2;
	if(m_CabinTrackInfo.u16Filter2 != _oDpFilter2.tGetData())
	{
		_oDpFilter2.s32SetData(m_CabinTrackInfo.u16Filter2);
		ETG_TRACE_USR4(("DP write Track Index = %d ", m_CabinTrackInfo.u16Filter2));
	}
     */
        dp_tclCMRDpCMP_FolderPath _oDpFolderPath;
        std::string dpfpath;
        char dpfpathread[255] = {0};
        _oDpFolderPath.s32GetData(dpfpathread,255);
        dpfpath.assign(dpfpathread);
        if(m_CabinTrackInfo.FolderPath != dpfpath)
        {
                uint8_t fpathlen = m_CabinTrackInfo.FolderPath.length();
                char dpfpathwrite[fpathlen+1] ={0};
                strncpy(dpfpathwrite,m_CabinTrackInfo.FolderPath.c_str(),fpathlen);
                dpfpathwrite[fpathlen] = '\0';
                _oDpFolderPath.s32SetData(dpfpathwrite);
                ETG_TRACE_USR4(("DP write FolderPath = %s ", dpfpathwrite));
        }
}
