/************************************************************************
*FILE: 			 TTFisCmdHandler.cpp
*SW-COMPONENT:  HMI
*DESCRIPTION: 	 Defines the class TTFisCmdHandler which
*               handles TTFis input commands (that simulates Clients and NanoMsg Components actions)
*AUTHOR:        Supriya Seshadri(RBEI/ECG5)
*COPYRIGHT:     (c) 2018 - 2019 Robert Bosch Car Multimedia GmbH
*HISTORY:
*20.02.2020 Rev. 0.1 Supriya Seshadri
* 		    Initial Revision.
*31.03.2020 Rev 0.2 Supriya Seshadri
*			Added the Thermal and Voltage Request TTFis commands
************************************************************************/
/*****************************************************************
| includes
|----------------------------------------------------------------*/

#include "TTFisCmdHandler/TTFisCmdHandler.h"
#include "plugin_trace.h"
#include "../../../../di_trace/components/etg/etg.h"	
#include <unistd.h>
#include "bosch/cm/di/NanoMsgLCMInterface.h"
#include "AsfLCMServiceAdaptor.h"
#include "AsfGenericServiceAdaptor.h"
#include "AsfIntercomServiceAdaptor.h"
#include "Constants.h"
#include <iostream>
#include <arpa/inet.h> 

/* Header files required for Tracing out log statements */
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_NANOMSGMASTERASF
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_NAVRES6
#define ETG_I_TTFIS_CMD_PREFIX            "NANOMSG_"
#define ETG_I_FILE_PREFIX                 NanoMsg_MasterApp::NanoMsgServer::TTFisCmdHandler::
#include "trcGenProj/Header/TTFisCmdHandler.cpp.trc.h"
#endif


//using namespace ::App::Core;
using namespace bosch::cm::di::NanoMsgLCMInterface;

/*****************************************************************
| function implementation (scope: global)
|----------------------------------------------------------------*/

/************************************************************************
*FUNCTION:      TTFisCmdHandler
*DESCRIPTION:   Constructor
*PARAMETER:     None
*RETURNVALUE:   NA
*
*HISTORY:
*Rev 0.1 Initial Revision.
*HISTORY:
*revision 0.1	Supriya Seshadri	02/02/2020
************************************************************************/
namespace NanoMsg_MasterApp{
	namespace NanoMsgServer{

TTFisCmdHandler::TTFisCmdHandler()
{
	ETG_TRACE_USR4(("TTFisCmdHandler"));
}

/************************************************************************
*FUNCTION:      ~TTFisCmdHandler
*DESCRIPTION:   Destructor
*PARAMETER:     None
*RETURNVALUE:   NA
*
*HISTORY:
*Rev 0.1 Initial Revision.
*HISTORY:
*revision 0.1	Supriya Seshadri	02/02/2020
************************************************************************/

TTFisCmdHandler::~TTFisCmdHandler()
{
	ETG_TRACE_USR4(("~TTFisCmdHandler()"));
}

/************************************************************************
*FUNCTION:      initialize
*DESCRIPTION:   Initialise fnction of tracehandler
*PARAMETER:     None
*RETURNVALUE:   NA
*
*HISTORY:
*revision 0.1	Supriya S(RBEI/ECG5)	02/02/2020
************************************************************************/
void TTFisCmdHandler::initialize()
{
	ETG_TRACE_USR4(("TraceCommandHandler::initialize() "));
	ETG_I_REGISTER_CHN(TR_TTFIS_NAVRES6);
	ETG_I_REGISTER_FILE();
}


/************************************************************************
*FUNCTION:      traceCmd_RspGetPoweModeSignal
*DESCRIPTION:   Handles TTFis command for RspGetPoweModeSignal
*PARAMETER:		uint16_t tPowerMode, uint8_t devId, uint8_t devNr, tCString strIP
*
*RETURNVALUE:   void
*
*HISTORY:
*revision 0.2	DRM2KOR(RBEI/ECG5)	29/01/2021
************************************************************************/
ETG_I_CMD_DEFINE((traceCmd_RspGetPoweModeSignal, "RspGetPoweModeSignal %u %u %u %s",uint16_t, uint8_t, uint8_t, ETG_I_STRING))
void TTFisCmdHandler::traceCmd_RspGetPoweModeSignal(uint16_t tPowerMode, uint8_t devId, uint8_t devNr, tCString strIP)
{
	ETG_TRACE_USR4(("traceCmd_RspGetPoweModeSignal()"));
	clAsfLCMServiceAdaptor *instance = clAsfLCMServiceAdaptor::getInstance();

	//Data Stub
	stNanoMsgData data;
	data.DevID = devId;
	data.DevNr = devNr;
	data.strURL= strIP;
	data.MsgID = 0x0006;
	data.data[0] = 0x00FF & tPowerMode;
	data.data[1] = (0xFF00 & tPowerMode) >> 8;

	//Call  the LCM adaptor stub which will broadcast the RspGetPoweModeSignal signal
	if(NULL != instance)
	instance->sendLCMRspGetPowerModeSignal(&data);
}


/************************************************************************
*FUNCTION:      traceCmd_ReqSetChainModeRequest
*DESCRIPTION:   Handles TTFis command for ReqSetChainModeRequest 
*PARAMETER:		(uint8_t data1, uint8_t data2, uint8_t data3, uint8_t data4)
*				Data Need to construct powermodedata , chaindata
*RETURNVALUE:   void
*
*HISTORY:
*revision 0.1	Supriya S(RBEI/ECG5)	02/02/2020
************************************************************************/
ETG_I_CMD_DEFINE((traceCmd_ReqSetChainModeRequest, "ReqSetChainModeRequest %u %u %u %u", uint8_t, uint8_t, uint8_t, uint8_t ))
void TTFisCmdHandler::traceCmd_ReqSetChainModeRequest(uint8_t data1, uint8_t data2, uint8_t data3, uint8_t data4)
{
	ETG_TRACE_USR4(("traceCmd_ReqSetChainModeRequest()"));
	clAsfLCMServiceAdaptor *instance = clAsfLCMServiceAdaptor::getInstance();
	boost::shared_ptr<ReqSetChainModeRequest> p1{new ReqSetChainModeRequest()};
	p1->setDev_Nr(1);
	p1->setMsg_ID(0x005);
	enDevID Devid = enDevID::enDevID__enDevID_Media_Router;
	p1->setDev_ID(Devid);
	
	uint16_t powermodedata = data1 | (data2 << 8) ;
	uint16_t chaindata = data3 | (data4 << 8) ;
	entPowerMode powermode = static_cast<entPowerMode>(powermodedata);
	
	p1->setPowerMode(powermode);
	p1->setTChain(chaindata);
	if(NULL != instance)
	instance->onReqSetChainModeRequest(p1);
}

/************************************************************************
*FUNCTION:      traceCmd_ReqSetPowerMode
*DESCRIPTION:   Handles TTFis command for ReqSetPowerMode 
*PARAMETER:		(uint8_t DEVID, uint8_t DEVNO, uint16_t POWERMODE, tCString strIP)
*				
*RETURNVALUE:   void
*
*HISTORY:
*revision 0.2	DRM2KOR(RBEI/ECG5)	29/01/2021
************************************************************************/
ETG_I_CMD_DEFINE((traceCmd_ReqSetPowerMode, "ReqSetPowerMode %u %u %u %s", uint8_t, uint8_t, uint16_t, ETG_I_STRING))
void TTFisCmdHandler::traceCmd_ReqSetPowerMode(uint8_t DEVID, uint8_t DEVNO, uint16_t POWERMODE, tCString strIP)
{
	ETG_TRACE_USR4(("traceCmd_ReqSetPowerMode()"));
	clAsfLCMServiceAdaptor *instance = clAsfLCMServiceAdaptor::getInstance();
	boost::shared_ptr<ReqSetPowerModeRequest> p1{ new ReqSetPowerModeRequest() };
	p1->setDev_Nr(DEVNO);
	p1->setMsg_ID(0x0003);
	std::string DestIPValue(strIP);
	p1->setDestIP(DestIPValue);
	enDevID devid = static_cast<enDevID>(DEVID);
	p1->setDev_ID(devid);
	
	entPowerMode powermode = static_cast<entPowerMode>(POWERMODE);
	p1->setPowerMode(powermode);

	if(NULL != instance)
	instance->onReqSetPowerModeRequest(p1);
}


/************************************************************************
*FUNCTION:      traceCmd_ReqGetPowerMode
*DESCRIPTION:   Handles TTFis command for ReqGetPowerMode 
*PARAMETER:		(uint8_t DEVID, uint8_t DEVNO, tCString strIP)
*				
*RETURNVALUE:   void
*
*HISTORY:
*revision 0.1	Supriya S(RBEI/ECG5)	01/03/2020
************************************************************************/
ETG_I_CMD_DEFINE((traceCmd_ReqGetPowerMode, "ReqGetPowerMode %u %u %s", uint8_t, uint8_t, ETG_I_STRING))
void TTFisCmdHandler::traceCmd_ReqGetPowerMode(uint8_t DEVID, uint8_t DEVNO, tCString strIP)
{
	ETG_TRACE_USR4(("traceCmd_ReqGetPowerMode()"));
	clAsfLCMServiceAdaptor *instance = clAsfLCMServiceAdaptor::getInstance();
	boost::shared_ptr<ReqGetPowerModeRequest> p1{ new ReqGetPowerModeRequest() };
	p1->setDev_Nr(DEVNO);
	p1->setMsg_ID(0x0005);
    std::string DestIPValue(strIP);
	p1->setDestIP(DestIPValue);
	enDevID devid = static_cast<enDevID>(DEVID);
	p1->setDev_ID(devid);

	if(NULL != instance)
	instance->onReqGetPowerModeRequest(p1);
}


/************************************************************************
*FUNCTION:      traceCmd_ReqGetWakeupReason
*DESCRIPTION:   Handles TTFis command for ReqGetWakeupReason 
*PARAMETER:		(uint8_t DEVID, uint8_t DEVNO, tCString strIP)
*				
*RETURNVALUE:   void
*
*HISTORY:
*revision 0.1	Supriya S(RBEI/ECG5)	01/03/2020
************************************************************************/
ETG_I_CMD_DEFINE((traceCmd_ReqGetWakeupReason, "ReqGetWakeupReason %u %u %s", uint8_t, uint8_t, ETG_I_STRING))
void TTFisCmdHandler::traceCmd_ReqGetWakeupReason(uint8_t DEVID, uint8_t DEVNO, tCString strIP)
{
	ETG_TRACE_USR4(("traceCmd_ReqGetWakeupReason()"));
	clAsfLCMServiceAdaptor *instance = clAsfLCMServiceAdaptor::getInstance();
	boost::shared_ptr<ReqGetWakeupReasonRequest> p1{ new ReqGetWakeupReasonRequest() };
	p1->setDev_Nr(DEVNO);
	p1->setMsg_ID(0x0008);
    std::string DestIPValue(strIP);
	p1->setDestIP(DestIPValue);
	enDevID devid = static_cast<enDevID>(DEVID);
	p1->setDev_ID(devid);

	if(NULL != instance)
	instance->onReqGetWakeupReasonRequest(p1);
}


/************************************************************************
*FUNCTION:      traceCmd_ReqShutdown
*DESCRIPTION:   Handles TTFis command for ReqShutdown 
*PARAMETER:		(uint8_t DEVID, uint8_t DEVNO, uint16_t TIMEOUT, tCString strIP)
*				
*RETURNVALUE:   void
*
*HISTORY:
*revision 0.1	Supriya S(RBEI/ECG5)	01/03/2020
************************************************************************/
ETG_I_CMD_DEFINE((traceCmd_ReqShutdown, "ReqShutdown %u %u %u %s", uint8_t, uint8_t, uint16_t, ETG_I_STRING))
void TTFisCmdHandler::traceCmd_ReqShutdown(uint8_t DEVID, uint8_t DEVNO, uint16_t TIMEOUT, tCString strIP)
{
	ETG_TRACE_COMP(("traceCmd_ReqShutdown()"));
	clAsfLCMServiceAdaptor *instance = clAsfLCMServiceAdaptor::getInstance();
	boost::shared_ptr<ReqShutdownRequest> p1{ new ReqShutdownRequest() };
	p1->setDev_Nr(DEVNO);
	p1->setMsg_ID(0x0010);
     std::string DestIPValue(strIP);
	p1->setDestIP(DestIPValue);
	enDevID devid = static_cast<enDevID>(DEVID);
	p1->setDev_ID(devid);

	p1->setTimeout(TIMEOUT);

	if(NULL != instance)
	instance->onReqShutdownRequest(p1);
}


/************************************************************************
*FUNCTION:      traceCmd_EventDigitalInput
*DESCRIPTION:   Handles TTFis command for EventDigitalInput 
*PARAMETER:		(uint32_t DigitalInput, uint32_t Response,  tCString strIP)
*				
*RETURNVALUE:   void
*
*HISTORY:
*revision 0.1	Supriya S(RBEI/ECG5)	01/03/2020
************************************************************************/
ETG_I_CMD_DEFINE((traceCmd_EventDigitalInput, "EventDigitalInput %u %u %u %u %s", uint8_t, uint8_t, uint32_t, uint32_t, ETG_I_STRING))
void TTFisCmdHandler::traceCmd_EventDigitalInput(uint8_t DEVID, uint8_t DEVNO, uint32_t DigitalInput, uint32_t Response,  tCString strIP)
{
	ETG_TRACE_USR4(("traceCmd_EventDigitalInput()"));
	clAsfGenericServiceAdaptor *instance = clAsfGenericServiceAdaptor::getInstance();
	

	//Data Stub
	stNanoMsgData data;
	data.DevNr = DEVNO;
	data.DevID = DEVID;
	data.MsgID = 0x0015;
	data.strURL= strIP;
	DigitalInput = htonl(DigitalInput);
	data.data[0] = DigitalInput | 0x000000FF;
	data.data[1] = DigitalInput | 0x0000FF00;
	data.data[2] = DigitalInput | 0x00FF0000;
	data.data[3] = DigitalInput | 0xFF000000;
	data.data[4] = Response | 0x000000FF;
	
	if(NULL != instance)
	//Call  the Generic adaptor stub which will broadcast the EventDigitalInput signal
	instance->sendGenericEventDigitalInput(&data);
}

/************************************************************************
*FUNCTION:      traceCmd_EventIntercomCall
*DESCRIPTION:   Handles TTFis command for EventIntercomCall 
*PARAMETER:		(uint32_t tPhoneNumber)
*				
*RETURNVALUE:   void
*
*HISTORY:
*revision 0.1	Supriya S(RBEI/ECG5)	01/03/2020
************************************************************************/
ETG_I_CMD_DEFINE((traceCmd_EventIntercomCall, "EventIntercomCall %u %s", uint32_t, ETG_I_STRING))
void TTFisCmdHandler::traceCmd_EventIntercomCall(uint32_t tPhoneNumber, tCString strIP)
{
	ETG_TRACE_USR4(("traceCmd_EventIntercomCall()"));
	clAsfIntercomServiceAdaptor *instance = clAsfIntercomServiceAdaptor::getInstance();
	

	//Data Stub
	stNanoMsgData data;
        data.strURL= strIP;
	data.data[0] = 0x000000FF | tPhoneNumber;
	
	if(NULL != instance)
	//Call  the Generic adaptor stub which will broadcast the EventIntercomCall signal
	instance->sendIntercomEventIntercomCall(&data);
	
}

/************************************************************************
*FUNCTION:      traceCmd_EventPhoneStatusChange
*DESCRIPTION:   Handles TTFis command for EventPhoneStatusChange
*PARAMETER:		(uint8_t tPhoneNumber, uint8_t phone_Status, tCString strIP)
*				
*RETURNVALUE:   void
*
*HISTORY:
*revision 0.1	DRM2KOR(RBEI/ECG5)	11/01/2021
************************************************************************/
ETG_I_CMD_DEFINE((traceCmd_EventPhoneStatusChange, "EventPhoneStatusChange %u %u %s", uint8_t, uint8_t, ETG_I_STRING))
void TTFisCmdHandler::traceCmd_EventPhoneStatusChange(uint8_t tPhoneNumber, uint8_t phone_Status, tCString strIP)
{
	ETG_TRACE_USR4(("traceCmd_EventPhoneStatusChange()"));
	clAsfIntercomServiceAdaptor *instance = clAsfIntercomServiceAdaptor::getInstance();
	

	//Data Stub
	stNanoMsgData data;
        
	data.data[0] = tPhoneNumber;
	data.data[1] = phone_Status;
	data.strURL= strIP;
	if(NULL != instance)
	//Call  the Generic adaptor stub which will broadcast the EventPhoneStatusChange signal
	instance->sendIntercomEventPhoneStatusChanged(&data);

}

/************************************************************************
*FUNCTION:      traceCmd_ReqGetIntercomAvailable
*DESCRIPTION:   Handles TTFis command for ReqGetIntercomAvailable
*PARAMETER:		(uint16_t MSGID, uint8_t DEVID, uint8_t DEVNO)
*				
*RETURNVALUE:   void
*
*HISTORY:
*revision 0.1	Madhuree	26/10/2020
************************************************************************/
ETG_I_CMD_DEFINE((traceCmd_ReqGetIntercomAvailable, "ReqGetIntercomAvailable  %s", ETG_I_STRING))
void TTFisCmdHandler::traceCmd_ReqGetIntercomAvailable( tCString strIP)
{
	ETG_TRACE_USR4(("traceCmd_ReqGetIntercomAvailable()"));
	clAsfIntercomServiceAdaptor *instance = clAsfIntercomServiceAdaptor::getInstance();
	boost::shared_ptr<ReqGetIntercomAvailableRequest>request{ new ReqGetIntercomAvailableRequest() };
	//fill the stNanoMsgData structure
        request->setDev_ID(0);
	request->setDev_Nr(0);
	request->setMsg_ID(0x0114);
	std::string DestIPValue(strIP);
	request->setDestIP(DestIPValue);

	if(NULL != instance)
	instance->onReqGetIntercomAvailableRequest(request);
}

/************************************************************************
*FUNCTION:      traceCmd_EventThermalState
*DESCRIPTION:   Handles TTFis command for EventThermalState
*PARAMETER:		(uint8_t DEVID, uint8_t DEVNO, uint16_t ThermalState, uint16_t ThermalSensorID, uint16_t Temperature, tCString strIP)
*				
*RETURNVALUE:   void
*
*HISTORY:
*revision 0.1	DRM2KOR(RBEI/ECG5)	11/01/2021
************************************************************************/
ETG_I_CMD_DEFINE((traceCmd_EventThermalState, "EventThermalState %u %u %u %u %u %s", uint8_t, uint8_t, uint16_t, uint16_t, uint16_t, ETG_I_STRING))
void TTFisCmdHandler::traceCmd_EventThermalState(uint8_t DEVID, uint8_t DEVNO, uint16_t ThermalState, uint16_t ThermalSensorID, uint16_t Temperature, tCString strIP)
{
	ETG_TRACE_USR4(("traceCmd_EventThermalState()"));
		clAsfGenericServiceAdaptor *instance = clAsfGenericServiceAdaptor::getInstance();
	

	//Data Stub
	stNanoMsgData data;
        data.DevNr = DEVNO;
	data.DevID = DEVID;
	data.MsgID = 0x0043;
	data.data[0] = (0xFF00 & ThermalState) >> 8;
        data.data[1] = 0x00FF & ThermalState;
	data.data[2] = (0xFF00 & ThermalSensorID) >> 8;
	data.data[3] = 0x00FF & ThermalSensorID;
	data.data[4] = (0xFF00 & Temperature) >> 8;
	data.data[5] = 0x00FF & Temperature;

	data.strURL= strIP;
	if(NULL != instance)
	//Call  the Generic adaptor stub which will broadcast the EventThermalState signal
	instance->sendGenericEventThermalSignal(&data);

}

/************************************************************************
*FUNCTION:      traceCmd_ReqGetThermalState
*DESCRIPTION:   Handles TTFis command for ReqGetThermalState 
*PARAMETER:		(uint8_t DEVID, uint8_t DEVNO, tCString strIP)
*				URL IP of the receiver
*RETURNVALUE:   void
*
*HISTORY:
*revision 0.2	DRM2KOR(RBEI/ECG5)	02.02.2021
************************************************************************/
ETG_I_CMD_DEFINE((traceCmd_ReqGetThermalState, "ReqGetThermalState %u %u %s", uint8_t, uint8_t, ETG_I_STRING))
void TTFisCmdHandler::traceCmd_ReqGetThermalState(uint8_t DEVID, uint8_t DEVNO, tCString strIP)
{
	ETG_TRACE_USR4(("traceCmd_ReqGetThermalState"));
	boost::shared_ptr<ReqGetThermalStateRequest> request{ new ReqGetThermalStateRequest() };
	//fill the stNanoMsgData structure
        enDevID devid = static_cast<enDevID>(DEVID);
	request->setDev_ID(devid);
	request->setDev_Nr(DEVNO);
	request->setMsg_ID(0x0044);
	std::string DestIPValue(strIP);
	request->setDestIP(DestIPValue);
	clAsfGenericServiceAdaptor *instance = clAsfGenericServiceAdaptor::getInstance();
	if(NULL != instance)
	instance->onReqGetThermalStateRequest(request);
}

/************************************************************************
*FUNCTION:      traceCmd_ReqHealthCare
*DESCRIPTION:   Handles TTFis command for ReqHealthCare 
*PARAMETER:		(uint8_t DEVID, uint8_t DEVNO, tCString strIP)
*				
*RETURNVALUE:   void
*
*HISTORY:
*revision 0.1	DRM2KOR(RBEI/ECG5)	05/03/2021
************************************************************************/
ETG_I_CMD_DEFINE((traceCmd_ReqHealthCare, "ReqHealthCare %u %u %s", uint8_t, uint8_t, ETG_I_STRING))
void TTFisCmdHandler::traceCmd_ReqHealthCare(uint8_t DEVID, uint8_t DEVNO, tCString strIP)
{
    ETG_TRACE_USR4(("TTFisCmdHandler::traceCmd_ReqHealthCare()"));
    clAsfLCMServiceAdaptor *instance = clAsfLCMServiceAdaptor::getInstance();
    boost::shared_ptr<ReqHealthCareRequest> req{ new ReqHealthCareRequest() };
    req->setDev_Nr(DEVNO);
    req->setMsg_ID(0x0030);
    req->setDestIP(strIP);
    enDevID devid = static_cast<enDevID>(DEVID);
    req->setDev_ID(devid);

    if(NULL != instance)
    {
        instance->onReqHealthCareRequest(req);
    }
}

/************************************************************************
*FUNCTION:      traceCmd_EventDtcOccurance
*DESCRIPTION:   Handles TTFis command for traceCmd_EventDtcOccurance 
*PARAMETER:		(uint8_t DEVID, uint8_t DEVNO, tCString strIP)
*				
*RETURNVALUE:   void
*
*HISTORY:
*revision 0.1	Ndw5kor(RBEI/ECG5)	29/10/2021
************************************************************************/
ETG_I_CMD_DEFINE((traceCmd_EventDtcOccurance, "EventDtcOccurance %u %u %s", uint8_t, uint8_t, ETG_I_STRING))
void TTFisCmdHandler::traceCmd_EventDtcOccurance(uint8_t DEVID, uint8_t DEVNO, tCString strIP)
{
    ETG_TRACE_USR4(("TTFisCmdHandler::traceCmd_EventDtcOccurance()"));
	
    clAsfLCMServiceAdaptor *instance = clAsfLCMServiceAdaptor::getInstance();
	
  //Data Stub
  stNanoMsgData data;
  data.DevID = DEVID;
  data.DevNr = DEVNO;
  data.strURL= strIP;
  data.MsgID = 0x0032;
  
  // DTC data .
  data.data[0] = 0x003;
  data.data[1] = 0x00FF; //255
  data.data[2] = 0x0015; //21
  data.data[3] = 0x118; //276
  data.data[4] = 0x003;
  data.data[5] = 0x00A; //10
  data.data[6] = 0x00B; //12
  data.data[7] = 0x00C; //13


  //Call  the LCM adaptor stub which will broadcast the EventDtcOccurance signal
  if(NULL != instance)
  {
	  
     instance->sendLCMEventDtcOccuranceSignal(&data);
	 ETG_TRACE_USR4(("TTFisCmdHandler::Successfull"));
  }
}

/************************************************************************
*FUNCTION:      traceCmd_ReqCMGSvivelPosition
*DESCRIPTION:   Handles TTFis command for ReqCMGSvivelPosition 
*PARAMETER:		(tCString strIP)
*				URL IP of the receiver
*RETURNVALUE:   void
*
*HISTORY:
*revision 0.1	Madhuree	26.10.2020
************************************************************************/   
ETG_I_CMD_DEFINE(( traceCmd_ReqCMGSvivelPosition, "ReqCMGSvivelPosition %u %u %s", uint8_t, uint8_t, ETG_I_STRING))
void TTFisCmdHandler::traceCmd_ReqCMGSvivelPosition(uint8_t DEVID, uint8_t DEVNO, tCString strIP)
{
	ETG_TRACE_USR4(("TTFisCmdHandler::traceCmd_ReqCMGSvivelPosition"));

	boost::shared_ptr<ReqCMGSvivelPositionRequest> request{ new ReqCMGSvivelPositionRequest() };
	//fill the stNanoMsgData structure
        enDevID devid = static_cast<enDevID>(DEVID);
	request->setDev_ID(devid);
	request->setDev_Nr(DEVNO);
	request->setMsg_ID(0x0064);
	std::string DestIPValue(strIP);
	request->setDestIP(DestIPValue);
        clAsfGenericServiceAdaptor *instance = clAsfGenericServiceAdaptor::getInstance();
	if(NULL != instance)
	instance->onReqCMGSvivelPositionRequest(request);
}


/************************************************************************
*FUNCTION:      traceCmd_EventVoltage
*DESCRIPTION:   Handles TTFis command for EventVoltage
*PARAMETER:		(uint8_t DEVID, uint8_t DEVNO, uint16_t VoltageLevel, uint16_t VoltageSensor, uint16_t VoltageValue, tCString strIP)
*				
*RETURNVALUE:   void
*
*HISTORY:
*revision 0.1	DRM2KOR(RBEI/ECG5)	11/01/2021
************************************************************************/
ETG_I_CMD_DEFINE((traceCmd_EventVoltage, "EventVoltage %u %u %u %u %u %s", uint8_t, uint8_t, uint16_t, uint16_t, uint16_t, ETG_I_STRING))
void TTFisCmdHandler::traceCmd_EventVoltage(uint8_t DEVID, uint8_t DEVNO, uint16_t VoltageLevel, uint16_t VoltageSensor, uint16_t VoltageValue, tCString strIP)
{
	ETG_TRACE_COMP(("traceCmd_EventVoltage()"));
		clAsfGenericServiceAdaptor *instance = clAsfGenericServiceAdaptor::getInstance();
	

	//Data Stub
	stNanoMsgData data;
        data.DevNr = DEVNO;
	data.DevID = DEVID;
	data.MsgID = 0x0012;
	data.data[0] = (0xFF00 & VoltageLevel) >> 8;
        data.data[1] = 0x00FF & VoltageLevel;
	data.data[2] = (0xFF00 & VoltageSensor) >> 8;
	data.data[3] = 0x00FF & VoltageSensor;
	data.data[4] = (0xFF00 & VoltageValue) >> 8;
	data.data[5] = 0x00FF & VoltageValue;
	data.strURL= strIP;
	if(NULL != instance)
	//Call  the Generic adaptor stub which will broadcast the EventVoltage signal
	instance->sendGenericEventVoltageSignal(&data);

}

/************************************************************************
*FUNCTION:      traceCmd_ReqGetVoltageState
*DESCRIPTION:   Handles TTFis command for ReqGetVoltageState 
*PARAMETER:		(uint8_t DEVID, uint8_t DEVNO, uint16_t voltageval, tCString strIP)
*				URL IP of the receiver
*RETURNVALUE:   void
*
*HISTORY:
*revision 0.2	DRM2KOR(RBEI/ECG5)	02.02.2021
************************************************************************/
ETG_I_CMD_DEFINE((traceCmd_ReqGetVoltageState, "ReqGetVoltageState %u %u %u %s", uint8_t, uint8_t, uint16_t, ETG_I_STRING))
void TTFisCmdHandler::traceCmd_ReqGetVoltageState(uint8_t DEVID, uint8_t DEVNO, uint16_t voltageval, tCString strIP)
{
	ETG_TRACE_USR4(("traceCmd_ReqGetVoltageState"));

	boost::shared_ptr<ReqGetVoltageRequest> request{ new ReqGetVoltageRequest() };
	//fill the stNanoMsgData structure
	enDevID devid = static_cast<enDevID>(DEVID);
	request->setDev_ID(devid);
	request->setDev_Nr(DEVNO);
	request->setMsg_ID(0x0013);
	std::string DestIPValue(strIP);
	request->setDestIP(DestIPValue);
	
	entVoltageSensor VoltageSensorValue = static_cast<entVoltageSensor>(voltageval);
        request->setVoltageSensor(VoltageSensorValue);
	
	clAsfGenericServiceAdaptor *instance = clAsfGenericServiceAdaptor::getInstance();
	if(NULL != instance)
	instance->onReqGetVoltageRequest(request);
}

}}




/******************************************************************************
| EOF
|----------------------------------------------------------------------------*/

