/* ***************************************************************************************
 * FILE:          NanoMsgSender.cpp
 * SW-COMPONENT:  NanoMsg_Master_Application
 * DESCRIPTION:  NanoMsgSender.cpp is part of NanoMsg_MasterApplication
 * 				 This is the protocol part of the application, has the socket Sender implementation
 * COPYRIGHT:  (c) 2020-21 	Robert Bosch Car Multimedia GmbH
 * HISTORY: 
 * AUTHOR:  Shivakumar J(RBEI/ECH2)  02.02.2020
 * REVISION: Supriya S 18.03.2020
 *		Added the extraction of IP URL in the recieve logic, Data Length correction after receive
 *
 * The reproduction, distribution and utilization of this file as well as the
 * communication of its contents to others without express authorization is
 * prohibited. Offenders will be held liable for the payment of damages.
 * All rights reserved in the event of the grant of a patent, utility model or design.
 *
 *************************************************************************************** */
#include "NanoMsgSender.h"
#include "plugin_trace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_NANOMSGMASTERPROTOCOL
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_NAVRES6
#include "trcGenProj/Header/NanoMsgSender.cpp.trc.h"
#endif

/************************************************************************
*FUNCTION: 		clNanoMsgSender()
*DESCRIPTION:   Constructor of the clNanoMsgSender class
*PARAMETER:		void *
*RETURNVALUE: 	None
*History:		Shiva Kumar J (RBEI/ECH2)   Initial version
*************************************************************************/
clNanoMsgSender::clNanoMsgSender():m_pIDMappingInstance(NULL),m_pCoreReceiver(NULL)
{
	ETG_TRACE_USR4(("clNanoMsgSender::clNanoMsgSender"));
	m_pCoreReceiver = clReceiverThread::getInstance();
	m_pIDMappingInstance = clMsgIDMapping::getInstance();
	//m_pXmlReader = XMLReader::GetInstance();  
	init();	
}


/************************************************************************
*FUNCTION: 		clNanoMsgSender()
*DESCRIPTION:   Destrcutor of the clNanoMsgSender class
*PARAMETER:		void *
*RETURNVALUE: 	None
*History:		Shiva Kumar J (RBEI/ECH2)   Initial version
*************************************************************************/
clNanoMsgSender::~clNanoMsgSender(){
	//m_pIDMappingInstance is a Core class, should be deleted by the Core or main service
	m_pIDMappingInstance = NULL;
	//m_pXmlReader = NULL;
	m_pCoreReceiver = NULL;
	//todo: delete instance of XML reader needs implementation
}

/************************************************************************
*FUNCTION: 		clNanoMsgSender()
*DESCRIPTION:   Initialization implementation of t class
*PARAMETER:		None
*RETURNVALUE: 	None
*History:		Shiva Kumar J (RBEI/ECH2)   Initial version
*************************************************************************/
void clNanoMsgSender::init()
{
	ETG_TRACE_USR4(("clNanoMsgSender::init()"));

	//Keep the Header for Sending data ready as the Sender info remains
	//Same throughtout the cycle
	clDataPacket::initCMCheader(this->tsenderheader);

}

/************************************************************************
*FUNCTION: 		processRecvMsg()
*DESCRIPTION:   the receieved socket message is processed and the structure stNanoMsgData is filled
*PARAMETER:		unsigned char* buffer, int recv_bytes
*RETURNVALUE: 	None
*History:		Shiva Kumar J(RBEI/ECH2)   Initial version
*Revision:		Supriya S(RBEI/ECG5) Added the Data Length correction , IP extarction
*Revision:		Supriya S(RBEI/ECG5) Added the recv_bytes and logic to copy payload data only when sufficient
				data is available.
*Revision:      Shiva Kumar J(RBEI/ECH2) Added error case if event cannot be added to queue and added few more checks
*************************************************************************/
/************************************************************************
Packet structure according to the specification:
Field Name              Byte(s)				Description
 Version				 2                  Protocol version(Not used) 
 SourceIP				 4					IP of sending station(CM-A, CM-P etc..)
 MAC					 6                  MAC adress of sending station (Not used) 
 Length					 2                  MsgID + DevID + DevNr + Data
 DevID					 1					Source DevID
 DevNr					 1                  Source DevNr
 MsgID					 2                  Specific command or event
 Data					 xx(Max 8)	        command specific data
************************************************************************/
//processRecvMsg function is also present in NanoMsgReceiver.
// todo: put this as a helper function in datapacket.cpp
void clNanoMsgSender::processRecvMsg(unsigned char* buffer, int recv_bytes){
      ETG_TRACE_USR4((" clNanoMsgSender::processRecvMsg() "));
      stNanoMsgData recvBufferData;
      if (NULL != buffer && recv_bytes > 0) {  			  
	
			//if we have come to this point, it means recv_bytes >= INDEX_RAW_DATA
			//So memcpy can be done till buffer[INDEX_RAW_DATA]
			
			/*****copy source IP to recvBufferData***********************************/
			uint8_t URL[4];
			memcpy(URL, (buffer + INDEX_RAW_IP), DATA_SIZE_FOUR_BYTE);
			clDataPacket::vPackURL(URL, recvBufferData.strURL);
			ETG_TRACE_USR4(("Msg received IP is %s", recvBufferData.strURL.c_str())); 
			/*************************************************************************/
			
			/*****copy MSGID to recvBufferData***********************************/
            memcpy(&recvBufferData.MsgID, (buffer + INDEX_RAW_MSGID), DATA_SIZE_TWO_BYTE);
			recvBufferData.MsgID = ntohs(recvBufferData.MsgID);
			ETG_TRACE_USR4(("Msg received MsgID is %d", recvBufferData.MsgID )); 
			/*************************************************************************/
			
			/*****copy DevID and DevNr to recvBufferData***********************************/
            recvBufferData.DevID = buffer[INDEX_DEV_ID];
            recvBufferData.DevNr = buffer[INDEX_DEV_NUMBER];
			/*************************************************************************/
			
			/*****copy DataLength to recvBufferData***********************************/
			memcpy(&recvBufferData.DataLength, (buffer + INDEX_RAW_LENGTH), DATA_SIZE_TWO_BYTE);
            recvBufferData.DataLength = ntohs(recvBufferData.DataLength);
			ETG_TRACE_USR4(("Msg received DataLength is after ntohs %d", recvBufferData.DataLength )); 
			/*************************************************************************/
			
			/*****Data length correction if not received as expected****************/
			if(recvBufferData.DataLength >= DATA_HEADER_LENGTH)
			recvBufferData.DataLength = recvBufferData.DataLength - DATA_HEADER_LENGTH;
						
			if((recvBufferData.DataLength > MAX_PAYLOAD_LENGTH) || (recv_bytes < (INDEX_RAW_DATA+ recvBufferData.DataLength))){
			ETG_TRACE_USR4(("Recieved payload is not supported %d", recvBufferData.DataLength ));
			return;			
			}
			/*************************************************************************/
			
			/*****copy Data to recvBufferData***********************************/
			if((recvBufferData.DataLength > 0) && (recv_bytes >= INDEX_RAW_DATA+ recvBufferData.DataLength))
			memcpy(recvBufferData.data, &buffer[INDEX_RAW_DATA], recvBufferData.DataLength);
            /*************************************************************************/

            /*************************Print the whole buffer**************************/
			for(int itr = 0; itr < recv_bytes ; itr++)
                  ETG_TRACE_USR4(("value at buffer[%d] is %d", itr, buffer[itr] ));
			/*************************************************************************/
			
            //Based on the need and traffic response thread would be created and used instead of Rx.
			bool beventSuccess = false;
			if(NULL != m_pCoreReceiver)
			beventSuccess = m_pCoreReceiver->setNanoMsgEvent(recvBufferData);
		
			if (!beventSuccess){
				ETG_TRACE_USR4(("Event not added to the queue retry after some time" ));
				//Sleep(50); // current version no retry, Method error to be posted, will not be done for SOP1
			}
            
      }
  }

/************************************************************************
*FUNCTION: 		vSendDummyRejectResponse()
*DESCRIPTION:   The dummy reponse is created for Requested msg, when we do not get reponse from subsystem
* The IP, DevID, DevNr is same as Req msg. Response MsgID for request is used for MsgID. Status value used is
* 0xFF for No response from subsystems
*PARAMETER:		stNanoMsgData data (Msg which had to be sent, but reponse for the send was not received)
*RETURNVALUE: 	None
*History:		Dhanya Elza Sunny (MS/ECG5-XC) 02/06/2022  Initial version
*************************************************************************/
void clNanoMsgSender::vSendDummyRejectResponse(stNanoMsgData data){
	
    stNanoMsgData recvBufferData;
    recvBufferData.strURL = data.strURL;
	recvBufferData.MsgID = 0xFFFF;
	if (NULL != m_pIDMappingInstance)
	{
		recvBufferData.MsgID = m_pIDMappingInstance->u16GetRspMsgID(data.MsgID);
	}
	if(0xFFFF == recvBufferData.MsgID)
    {
        ETG_TRACE_USR4(("clNanoMsgSender::vSendDummyRejectResponse msgID %d is not part of map",data.MsgID));
        return;
	}		
	recvBufferData.DevID = data.DevID;
	recvBufferData.DevNr = data.DevNr;
	recvBufferData.DataLength = 5;
    recvBufferData.data[0] = 0xFF; //NO Response
	
	//Based on the need and traffic response thread would be created and used instead of Rx.
	bool beventSuccess = false;
	if(NULL != m_pCoreReceiver)
	beventSuccess = m_pCoreReceiver->setNanoMsgEvent(recvBufferData);

	if (!beventSuccess){
        ETG_TRACE_USR4(("clNanoMsgSender::vSendDummyRejectResponse Event not added to the queue retry after some time" ));
		//Sleep(50); // current version no retry, Method error to be posted, will not be done for SOP1
	}

}
/************************************************************************
*FUNCTION: 		sendDataToEntity()
*DESCRIPTION:   Send Socket implementation 
*PARAMETER:		stNanoMsgData data
*RETURNVALUE: 	bool
*History:		Shiva Kumar J(RBEI/ECH2)   Initial version
*Revision:		Supriya S(RBEI/ECG5) Added the IP URL correction 
*Revision: 		Shiva Kumar J(RBEI/ECH2) Added checks for NULL and send returing bool 
*Revision:      Dhanya Elza Sunny (MS/ECG5-XC) 06/02/2022
*               Added single retry for send failure
*               Added dummy reponse if response is not obtained from subsystems
*************************************************************************/
bool clNanoMsgSender::sendDataToEntity(stNanoMsgData data){
	ETG_TRACE_USR4(("clNanoMsgSender::sendDataToEntity"));
	bool bEventResponse = false;

	if (NULL != m_pIDMappingInstance)
	{
		ETG_TRACE_USR4((" clNanoMsgSender::sendDataToEntity checking if ACK is needed"));
		//check if ACK is needed
		bEventResponse = m_pIDMappingInstance->bcheckforEvent(data.MsgID);
	}
		
	std::string urlString = data.strURL.c_str();
	if (urlString.empty()){
		return false;
	}
    uint8_t u8retrycount = 0;
    bool bRetryRequired = TRUE;
    while((u8retrycount < 2)&&(TRUE == bRetryRequired))
	{
        ETG_TRACE_USR4((" clNanoMsgSender::sendDataToEntity Executing while loop retrycount: %d", u8retrycount));
        clSocket sendsocket(NN_PAIR);
		std::string end_point = "tcp://" + urlString + ":2227";
		ETG_TRACE_USR4((" clNanoMsgSender::sendDataToEntity  end_point : %s",end_point.c_str()));
			// Connect socket
		bool bsocketconnect  = sendsocket.connect(end_point);
		//if Socket connection fails then just exit from this function
		//as there is no point in continuing to process the message
		if(!bsocketconnect)
		{
			ETG_TRACE_FATAL((" clNanoMsgSender::sendDataToEntity  Socket Connect Failed"	));
            bRetryRequired = TRUE;
            u8retrycount++;
            sendsocket.disconnect();
			continue;
			//return false;
		}
		
		// Socket options
		if (sendsocket.setSocketOptionSender() < 0) {
			//fatal("n_setsockopt");
            bRetryRequired = TRUE;
            u8retrycount++;
            sendsocket.disconnect();
			continue;
			ETG_TRACE_USR4(("clNanoMsgSender::nn_setsockopt error"));
		} 
		// Create Nano Packet
		tNanoDataPack nanoPacket;//need to construct this with static header
		nanoPacket = clDataPacket::vCreateNanoPacket(this->tsenderheader, data);
			
		uint16_t payloadsize = data.DataLength;
		int ssize = SIZE_OF_HEADER + payloadsize;
		int send_bytes = 0;
		int recv_bytes = 0;
		//send the data, should be nanoPacket = header + data
		if(!sendsocket.send(nanoPacket, ssize, send_bytes)){
            bRetryRequired = TRUE;
            u8retrycount++;
            sendsocket.disconnect();
            //vSendDummyRejectResponse(data); //added for testing. Remove this.
			continue;
			//return false; // if false no retry for SOP1
		}
        else{
            bRetryRequired = FALSE;
		}
		
		unsigned char* recv_packet;
		recv_packet = sendsocket.receive(recv_bytes);
		
		//Process the received message only for requests
		if(false == bEventResponse && NULL != recv_packet){
			processRecvMsg(recv_packet, recv_bytes);
		}
		//free buffer after data read
		if(NULL != recv_packet)
		{		
			if(!sendsocket.freeBuffer(recv_packet)){
			//could be buffer is NULL! cannot be the case
				ETG_TRACE_USR4(("clNanoMsgSender::Buffer NULL"));
			}
		}
        if((false == bEventResponse) && (NULL == recv_packet)) {
		
		    ETG_TRACE_USR4(("clNanoMsgSender::Send Successfull, Response Not received MsgID: %d", data.MsgID));
            vSendDummyRejectResponse(data);
		}			
		sendsocket.disconnect();
    }
    if ((TRUE == bRetryRequired) && (u8retrycount > 1))
    {
        ETG_TRACE_USR4(("clNanoMsgSender::retry done. send not successfull"));
        return false;
    }
	return true;
}

