/* ***************************************************************************************
 * FILE:          clNanoMsgCoreThread.cpp
 * SW-COMPONENT:  NanoMsg_Master_Application
 * DESCRIPTION:  clNanoMsgCoreThread.cpp is part of NanoMsg_MasterApplication
 * 				 This is thread base class with queue and wait logic, class that 
 *				 inherits from this would have to implement _nanoMsgCallBack(std::shared_ptr<stNanoMsgData> data)
 *				 function to get a msg update. 
 * COPYRIGHT:  (c) 2020-21 	Robert Bosch Car Multimedia GmbH
 * HISTORY: 
 * AUTHOR:  Shivakumar J(RBEI/ECH2)  29.01.2020
 *				Initial version
 * The reproduction, distribution and utilization of this file as well as the
 * communication of its contents to others without express authorization is
 * prohibited. Offenders will be held liable for the payment of damages.
 * All rights reserved in the event of the grant of a patent, utility model or design.
 *
 *************************************************************************************** */

#include "clNanoMsgCoreThread.h"
#include "plugin_trace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS  TR_CLASS_NANOMSGMASTERCMPCORE
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_NAVRES6
#include "trcGenProj/Header/clNanoMsgCoreThread.cpp.trc.h"
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN



void msSleep( unsigned int msec )
{
	usleep(msec*1000);
}
/************************************************************************
*FUNCTION: 		clNanoMsgCoreThread()
*DESCRIPTION:   Constructor, initializes queue for adding msg's
*PARAMETER:		None
*RETURNVALUE: 	None
*History:		Shiva Kumar J(RBEI/ECH2)   Initial version
*************************************************************************/
clNanoMsgCoreThread::clNanoMsgCoreThread(void)
:m_thRunStatus(FALSE)
,m_threadID(0L)
,m_thCurrentState(enmThStateDown)
,m_thNanoMsgQueue(NULL)
,m_thNanoMsgCurrentContainer(NULL)
,m_thStackSize(0)
,m_thStopTimer(30)
,m_thStatusTime(100)
{
m_thNanoMsgQueue = new vector<std::shared_ptr<stNanoMsgData>>(0);
	if( !m_thNanoMsgQueue )
	{
		setThreadState(enmThStateFault);
		ETG_TRACE_FATAL(("clNanoMsgCoreThread::Queue creation failed, hence thread not created!!"));
		//sd_notify(0, "ERRNO=2"); // No tested so far, so commented out.
		return; //to be commented out when we enable sd_notify
	}
	pthread_mutexattr_t mattr;
	pthread_mutexattr_init(&mattr);
	pthread_mutex_init(&m_thMutexLock,&mattr);
	pthread_cond_init(&m_thWaitCond,NULL);
	pThreadStart();
}

/************************************************************************
*FUNCTION: 		~clNanoMsgCoreThread()
*DESCRIPTION:   Destructor
*PARAMETER:		None
*RETURNVALUE: 	None
*History:		Shiva Kumar J(RBEI/ECH2)   Initial version
*************************************************************************/
clNanoMsgCoreThread::~clNanoMsgCoreThread(void)
{
	if( getThreadRunStatus() ) // gracefull termination
	{
		try
		{
			if( !pThreadStop() )
			{
				ETG_TRACE_FATAL(("clNanoMsgCoreThread::Cannot stop thread!!"));
			}
		}
		catch( char *s )
		{
			//FATAL trace:
		}
	}
	m_thNanoMsgQueue->clear();
}

/************************************************************************
*FUNCTION: 		thStartFunction(void *arg);
*DESCRIPTION:   Thread routine function, core threads(Receiver & Sender) thread entry point
*PARAMETER:		void *
*RETURNVALUE: 	None
*History:		Shiva Kumar J(RBEI/ECH2)   Initial version
*************************************************************************/
VOIDPTR clNanoMsgCoreThread::thStartRoutine( VOIDPTR NanoPacket )
{
	ETG_TRACE_USR4(("clNanoMsgCoreThread::thStartRoutine"));
	pthread_mutex_lock(&m_thMutexLock);

	setThreadState(enmThStateWaiting);
	setThreadRunStatus(TRUE);

	pthread_mutex_unlock(&m_thMutexLock);
	while( getThreadRunStatus() )
	{
		pthread_mutex_lock(&m_thMutexLock);
		//Wait for the message through signal
		if (0 != pthread_cond_wait(&m_thWaitCond, &m_thMutexLock))
		{
			break;
		}
		pthread_mutex_unlock(&m_thMutexLock);
		if(!processNanoMsg() )
		{
			//if(!getThreadRunStatus()){
			   ETG_TRACE_FATAL(("clNanoMsgCoreThread::processNanoMsg retured a FALSE"));
			  // break;
			//}
		}
	}
	pthread_mutex_lock(&m_thMutexLock);
	setThreadState(enmThStateDown);
	setThreadRunStatus(FALSE);
	pthread_mutex_unlock(&m_thMutexLock);

	return (VOIDPTR)0;
}

/************************************************************************
*FUNCTION: 		setNanoMsgEvent
*DESCRIPTION:   struct msg would be added to the queue and signal wait thread
*				Called by Dbus server and protocol threads
*PARAMETER:		stNanoMsgData struct
*RETURNVALUE: 	bool
*History:		Shiva Kumar J(RBEI/ECH2)   Initial version
*************************************************************************/
bool clNanoMsgCoreThread::setNanoMsgEvent(stNanoMsgData nanoData)
{
	ETG_TRACE_USR4(("clNanoMsgCoreThread::setNanoMsgEvent"));
	pthread_mutex_lock(&m_thMutexLock);
 // return FALSE if thread is not running;
	if( !getThreadRunStatus() )
	{
		pthread_mutex_unlock(&m_thMutexLock);
		return FALSE;
	}
/* 	create shared pointer of stNanoMsgData struct and copy data 
	to ensure data is not lost between threads */
	std::shared_ptr<stNanoMsgData> sPa(new stNanoMsgData);
	sPa->MsgID = nanoData.MsgID;
	sPa->DevID = nanoData.DevID;
	sPa->DevNr = nanoData.DevNr;
	sPa->DataLength = nanoData.DataLength;
	sPa->strURL = nanoData.strURL.c_str();
	memcpy(sPa->data, &nanoData.data, sizeof(nanoData.data));

	pthread_mutex_unlock(&m_thMutexLock);
	// Add Msg to queue
	if (!addNanoMsgToQueue(sPa))
	{
		return FALSE;
	}
	//signal waiting thread
	pthread_cond_signal(&m_thWaitCond);

	return TRUE;
}

/************************************************************************
*FUNCTION: 		processNanoMsg
*DESCRIPTION:   After getting signal to waiting thread, this function would
*				would get called to process the message added to queue
*PARAMETER:		void
*RETURNVALUE: 	bool
*History:		Shiva Kumar J(RBEI/ECH2)   Initial version
*************************************************************************/
bool clNanoMsgCoreThread::processNanoMsg()
{
	ETG_TRACE_USR4(("clNanoMsgCoreThread::processNanoMsg()"));
	pthread_mutex_lock(&m_thMutexLock);
	setThreadState(enmThStateBusy);
	if( !getThreadRunStatus() )
	{
		setThreadState(enmThStateShuttingDown);
		pthread_mutex_unlock(&m_thMutexLock);
		return FALSE;
	}

	pthread_mutex_unlock(&m_thMutexLock);
	if( !isQueueEmpty() )
	{
		while( !isQueueEmpty() )
		{
			getNanoMsgFromQueue();
			if(NULL != m_thNanoMsgCurrentContainer)
			{
				if( !_nanoMsgCallBack(m_thNanoMsgCurrentContainer) )
				{
					pthread_mutex_lock(&m_thMutexLock);
					m_thNanoMsgCurrentContainer.reset();
					//setThreadState(enmThStateShuttingDown);
					pthread_mutex_unlock(&m_thMutexLock);
					ETG_TRACE_FATAL(("clNanoMsgCoreThread::_nanoMsgCallBack retured a FALSE"));
					//Todo error handling in SOP2
					//return FALSE;
				}
			}
		}
		pthread_mutex_lock(&m_thMutexLock);
		if(m_thNanoMsgCurrentContainer != NULL)
		{
		 m_thNanoMsgCurrentContainer.reset();// delete managed object
		}
		setThreadState(enmThStateWaiting);
		pthread_mutex_unlock(&m_thMutexLock);
	}
	return TRUE;
}

/************************************************************************
*FUNCTION: 		isQueueEmpty
*DESCRIPTION:   Checks the size of the queue and sends true if >0 or false
*PARAMETER:		void
*RETURNVALUE: 	bool
*History:		Shiva Kumar J(RBEI/ECH2)   Initial version
*************************************************************************/
bool clNanoMsgCoreThread::isQueueEmpty()
{
	pthread_mutex_lock(&m_thMutexLock);
	if( m_thNanoMsgQueue->size() <= 0 )
	{
		pthread_mutex_unlock(&m_thMutexLock);
		return TRUE;
	}
	pthread_mutex_unlock(&m_thMutexLock);
	return FALSE;
}

/************************************************************************
*FUNCTION: 		addNanoMsgToQueue
*DESCRIPTION:   adds msg's to the queue and checks if size> 100.
*				current size of the queue is kept to 100 per thread
*PARAMETER:		void
*RETURNVALUE: 	bool
*History:		Shiva Kumar J(RBEI/ECH2)   Initial version
*************************************************************************/
bool clNanoMsgCoreThread::addNanoMsgToQueue( std::shared_ptr<stNanoMsgData> nanoData )
{
	ETG_TRACE_USR4(("clNanoMsgCoreThread::addNanoMsgToQueue"));
	if( !nanoData ) return TRUE;

	pthread_mutex_lock(&m_thMutexLock);
	if( m_thNanoMsgQueue->size() >= NANO_MSG_QUEUE_SIZE )
	{
		pthread_mutex_unlock(&m_thMutexLock);
		ETG_TRACE_FATAL(("clNanoMsgCoreThread::addNanoMsgToQueue,size() >= NANO_MSG_QUEUE_SIZE"));
		return FALSE;
	}
	//Push into vector
	m_thNanoMsgQueue->push_back(nanoData);
	pthread_mutex_unlock(&m_thMutexLock);
	if( m_thNanoMsgQueue->size() >= NANO_MSG_QUEUE_SIZE )
	{
		ETG_TRACE_FATAL(("clNanoMsgCoreThread::addNanoMsgToQueue, stack full"));
	}
	return TRUE;
}

/************************************************************************
*FUNCTION: 		getNanoMsgFromQueue
*DESCRIPTION:   Picks the 1st item from queue and puts on current container
*				and removes item from queue
*PARAMETER:		void
*RETURNVALUE: 	bool
*History:		Shiva Kumar J(RBEI/ECH2)   Initial version
*************************************************************************/
bool clNanoMsgCoreThread::getNanoMsgFromQueue()
{
	ETG_TRACE_USR4(("clNanoMsgCoreThread::getNanoMsgFromQueue()"));
	pthread_mutex_lock(&m_thMutexLock);
	if( m_thNanoMsgQueue->empty())
	{
		pthread_mutex_unlock(&m_thMutexLock);
		return FALSE;
	}

	if(m_thNanoMsgQueue->size() > 0)
	{
	  m_thNanoMsgCurrentContainer = m_thNanoMsgQueue->at(0);
	  m_thNanoMsgQueue->erase(m_thNanoMsgQueue->begin());
	}
	pthread_mutex_unlock(&m_thMutexLock);
	return TRUE;
}

/************************************************************************
*FUNCTION: 		pThreadStop
*DESCRIPTION:   Makes thread run status to false to exit from while and stops thread
*PARAMETER:		void
*RETURNVALUE: 	bool
*History:		Shiva Kumar J(RBEI/ECH2)   Initial version
*************************************************************************/
bool clNanoMsgCoreThread::pThreadStop()
{
	try 
	{
		pthread_mutex_lock(&m_thMutexLock);
		setThreadRunStatus(FALSE);
		pthread_mutex_unlock(&m_thMutexLock);

		pthread_cond_signal(&m_thWaitCond);

		int ticks = (m_thStopTimer*1000)/100;
		for( int i=0; i<ticks; i++ )
		{
			msSleep(100);// Need to find a solution, currently sleep is used until signal is processed
			pthread_mutex_lock(&m_thMutexLock);
			if( getThreadState() == enmThStateDown )
			{
				pthread_mutex_unlock(&m_thMutexLock);
				return TRUE;
			}
			pthread_mutex_unlock(&m_thMutexLock);
		} 
	}
	catch (char *s)
	{
		ETG_TRACE_FATAL(("clNanoMsgCoreThread::Thread Exit failed!!"));
	}
	return FALSE;
}

/************************************************************************
*FUNCTION: 		pThreadStop
*DESCRIPTION:   thread create function 
*PARAMETER:		void
*RETURNVALUE: 	bool
*History:		Shiva Kumar J(RBEI/ECH2)   Initial version
*************************************************************************/
bool clNanoMsgCoreThread::pThreadStart()
{
	try 
	{
		pthread_mutex_lock(&m_thMutexLock);
		if( getThreadRunStatus() )
		{
			pthread_mutex_unlock(&m_thMutexLock);
			return TRUE;
		}

		pthread_mutex_unlock(&m_thMutexLock);

		pthread_attr_t attr;
		pthread_attr_init(&attr);
		if( m_thStackSize != 0 )
		{
			pthread_attr_setstacksize(&attr,m_thStackSize);
		}

		int error = pthread_create(&m_thHandle,&attr,(THREADFUNCPTR)&clNanoMsgCoreThread::thStartRoutine,(VOIDPTR)this);
		
		if( error != 0 )
		{
			setThreadState(enmThStateFault);
			ETG_TRACE_FATAL(("clNanoMsgCoreThread::Thread creation failed!!"));
			return FALSE;	
		}
		
/****************** Thread real time priority setting********************
      // struct sched_param is used to store the scheduling priority
     struct sched_param params;
 
     // We'll set the priority to the maximum.
     params.sched_priority = sched_get_priority_max(SCHED_FIFO);
	 
	   // Attempt to set thread real-time priority to the SCHED_FIFO policy
     ret = pthread_setschedparam(m_thHandle, SCHED_FIFO, &params);
     if (ret != 0) {
         ETG_TRACE_FATAL(("Unsuccessful in setting thread realtime priority!!"));
      } 
****************************No tested so far, so commented out**********/
	}
	catch (char *s)
	{
		exit(-1); //to be commented out when we enable sd_notify
		//sd_notify(0, "ERRNO=2"); // No tested so far, so commented out.
	}
	return TRUE;
}

/************************************************************************
*FUNCTION: 		checkThreadRunStatus
*DESCRIPTION:   In case if any msg comes before thread creation, this function 
*				waits for some time for thread to get created.
*				This function will be introduced on need basis
*PARAMETER:		LONG
*RETURNVALUE: 	bool
*History:		Shiva Kumar J(RBEI/ECH2)   Initial version
*************************************************************************/
/* bool clNanoMsgCoreThread::checkThreadRunStatus(LONG timeout )
{
	LONG totalTime = 0;
    ETG_TRACE_USR4((" clNanoMsgCoreThread::checkThreadRunStatus %d",getThreadRunStatus()));
	while(TRUE)
	{
		if( totalTime > timeout && timeout > 0 )
			return FALSE;

		pthread_mutex_lock(&m_thMutexLock);
		if( getThreadRunStatus() )
		{
			pthread_mutex_unlock(&m_thMutexLock);
			return TRUE;
		}
		totalTime += m_thStatusTime;

		pthread_mutex_unlock(&m_thMutexLock);
		Sleep(m_thStatusTime);
	}
	return FALSE;
} */

