/*
 * Copyright (c) 2010 Gracenote.
 *
 * This software may not be used in any way or distributed without
 * permission. All rights reserved.
 *
 * Some code herein may be covered by US and international patents.
 */

/*
 * xsampa_to_lnh_conversion.c
 *
 * Is the heart of the conversion and it demonstrates the X-SAMPA 
 * to L&H+ conversion as well as the L&H+ language specific 
 * post-processing.  It will output the conversion data to the 
 * valid or invalid output file depending on its success.
 */

#include "xsampa_to_lnh_conversion.h"
#include "gn_phonetic_conversion.h"
#include "gn_phonetic_conversion_data.h"

/* EPAL Headers */
#include "gnepal_string.h"
#include "gnepal_memory.h"

/*
 * Globals and typedefs
 */

/*
 * Language Identifier Strings
 */
gn_uchar_t* g_speech_langs[SPEECH_LANG_COUNT] =
{
	(gn_uchar_t*)"USA_eng",
	(gn_uchar_t*)"MEX_spa",
	(gn_uchar_t*)"CAN_fre",
	(gn_uchar_t*)"GBR_eng",
	(gn_uchar_t*)"FRA_fre",
	(gn_uchar_t*)"ESP_spa",
	(gn_uchar_t*)"DEU_ger",
	(gn_uchar_t*)"ITA_ita",
	(gn_uchar_t*)"JPN_jpn",
	(gn_uchar_t*)"CHN_qad",
	(gn_uchar_t*)"RUS_rus",
	(gn_uchar_t*)"NLD_dut",
	(gn_uchar_t*)"KOR_kor",
	(gn_uchar_t*)"BRA_por",
	(gn_uchar_t*)"PRT_por",
	(gn_uchar_t*)"TUR_tur",
	(gn_uchar_t*)"AUS_eng"
};

/*
 * Static Prototypes
 */
static gnex_error_t _find_language_index_by_name(
	const gn_uchar_t* name,
	gn_uint32_t* lang_index
	);

/*
 * xsampa_to_lnh_conversion
 *
 * Converts the xsampa transcription to lnh and returns the output. 
 */
gnex_error_t xsampa_to_lnh_conversion(
    const gn_uchar_t* lang,				/* language identifier */
    const gn_uchar_t* xsampa,			/* X-SAMPA transcription */
    gn_uchar_t** lnh_conversion,		/* lnh converted transcription */
    gn_uchar_t** lnh_final_conversion	/* lnh final converted transcription */
    )
{
    gnex_error_t error = GNEX_PHOCVRT_NoError;
    gn_uchar_t* lnh_cnv = GN_NULL;
    gn_uchar_t* lnh_final_cnv = GN_NULL;
    gn_uint32_t lang_idx = 0;
    
    gn_phonetic_conversion_map_t* lnh_conversion_map = GN_NULL;
    gn_phonetic_conversion_map_t* lnh_post_process_conversion_map = GN_NULL;
    
	if (	(lang == GN_NULL)
		||	(xsampa == GN_NULL)
		||	(lnh_final_conversion == GN_NULL)
		||	(*lnh_final_conversion != GN_NULL)
		)
    {
        return GNEX_PHOCVRT_InvalidArg;
    }

    error = _find_language_index_by_name(lang, &lang_idx);
    if (error == GNEX_PHOCVRT_NoError)
    {
        /*
         * Retrieve conversion maps and perform xsampa to lnh conversion
         */
        error = gn_phonetic_conversion_lnh_map_get(&lnh_conversion_map, lang_idx);
        if (error == GNEX_PHOCVRT_NoError)
        {
            error = gn_phonetic_conversion_map_convert(
				lnh_conversion_map, 
				xsampa, 
				&lnh_cnv
			);
            if (error == GNEX_PHOCVRT_NoError)
            {
                error = gn_phonetic_conversion_lnh_post_process_map_get(
					&lnh_post_process_conversion_map,
					lang_idx
				);
                if (error == GNEX_PHOCVRT_NoError)
                {
					/*
                     * Not all languages need a post-processing step, so duplicate string
                     */
                    if (	(lnh_post_process_conversion_map != GN_NULL)
						&&	(lnh_post_process_conversion_map->rules != GN_NULL)
						&&	(lnh_post_process_conversion_map->algorithm_id != PHONETIC_MAP_ALGORITHM_NONE)
						)
                    {
                        error = gn_phonetic_conversion_map_convert(
							lnh_post_process_conversion_map,
							lnh_cnv, 
							&lnh_final_cnv
						);
                    }
                    else
                    {
                        lnh_final_cnv = gnepal_strdup(lnh_cnv);
						if (lnh_final_cnv == GN_NULL)
						{
							error = GNEX_PHOCVRT_NoMemory;
						}
                    }
				}
            }
        }
    }

    /*
     * Verify conversion and report success or failure
     */
    if (error != GNEX_PHOCVRT_NoError)
    {
        /* Error returned from conversion */
		gnepal_smart_free(&lnh_cnv);
        gnepal_smart_free(&lnh_final_cnv);
    }
    else
    {
        *lnh_final_conversion = lnh_final_cnv;
		if ((lnh_conversion != GN_NULL) && (*lnh_conversion == GN_NULL))
		{
			*lnh_conversion = lnh_cnv;
		}
		else
		{
			gnepal_smart_free(&lnh_cnv);
		}
    }

    return error;
}

/*
 * Static API Implementation
 */

/*
 * _find_language_data_by_name
 *
 * From the language string find the corresponding zero based index.
 */
static gnex_error_t _find_language_index_by_name(
	const gn_uchar_t* name,
	gn_uint32_t* lang_index
	)
{
	gnex_error_t error = GNEX_PHOCVRT_InvalidLanguage;
	gn_uint32_t i = 0;
	
	if (name == GN_NULL)
	{
		return GNEX_PHOCVRT_InvalidArg;
	}
	
	for (i = 0; i < SPEECH_LANG_COUNT; i++)
	{
		if (0 == gnepal_strncmp(name, g_speech_langs[i], gnepal_strlen(g_speech_langs[i])))
		{
			error = GNEX_PHOCVRT_NoError;
			if (lang_index != GN_NULL)
			{
				*lang_index = i;
			}
			break;
		}
	}
	
	return error;
}
