/*
 * Copyright (c) 2010 Gracenote.
 *
 * This software may not be used in any way or distributed without
 * permission. All rights reserved.
 *
 * Some code herein may be covered by US and international patents.
 */

/*
 * gn_phonetic_conversion_utility.h - 
 *
 * Is a dynamic string implementation that grows the memory of the
 * character pointer when adding a new character or character string.
 * It only grows memory when needed.  This was created to encapsulate
 * this behavior to make the conversion algorithm easier to understand.
 */

#ifndef _GN_PHONETIC_CONVERSION_UTILITY_H_
#define _GN_PHONETIC_CONVERSION_UTILITY_H_

#include "gn_abs_types.h"

/* Error-Code Headers */
#include "gnex_errors.h"
#include "gnex_error_values.h"

#ifdef __cplusplus
extern "C"{
#endif

/*
 * Type Declarations
 */

/* Dynamic String Struct */
typedef struct gn_dynamic_string_s
{
	gn_uchar_t* buffer;				/* Buffer for character array */
	gn_uint32_t size;				/* Size of buffer */
	gn_uint32_t used;				/* Amount used of buffer */
} gn_dynamic_string_t;

/*
 * API Declarations
 */

/* gn_dynamic_string_create
 *
 * Create and initialize a Dynamic String object.
 *
 * Memory is allocated by this call. The gn_dynamic_string_free function
 * should be called once the Dynamic String object is no longer needed.
 *
 * Parameters:
 *  IN/OUT	: dynamic_string	: An un-initialized Dynamic String object.
 *  IN		: size				: initial size of Dynamic String.
 *
 * Return Values:
 *  On success:
 *   gnex_error_t				: GNEX_PHOCVRT_NoError
 *   dynamic_string				: An initialized Dynamic String object.
 *
 *	On error:
 *   gnex_error_t				: Gracenote Extension Error Value
 *   dynamic_string				: Will not have been modified.
 *
 *  Errors include:
 *	 GNEX_PHOCVRT_NoMemory		: memory allocation
 *   GNEX_PHOCVRT_InvalidArg	: already initialized
 */
gnex_error_t
gn_dynamic_string_create(
	gn_dynamic_string_t** dynamic_string,
	gn_uint32_t size
	);

/* gn_dynamic_string_copy
 *
 * Initialize and copy a Dynamic String object.
 *
 * Memory is allocated by this call. The gn_dynamic_string_free function
 * should be called once the Dynamic String object is no longer needed.
 *
 * Parameters:
 *  IN/OUT	: dst_dynamic_string	: An un-initialized Dynamic String object.
 *  IN		: src_dynamic_string	: An initialized Dynamic String object.
 *
 * Return Values:
 *  On success:
 *   gnex_error_t				: GNEX_PHOCVRT_NoError
 *   src_dynamic_string			: An initialized Dynamic String object.
 *
 *	On error:
 *   gnex_error_t				: Gracenote Extension Error Value
 *   src_dynamic_string			: Will not have been modified.
 *
 *  Errors include:
 *	 GNEX_PHOCVRT_NoMemory		: memory allocation
 *   GNEX_PHOCVRT_InvalidArg	: already initialized
 */
gnex_error_t
gn_dynamic_string_copy(
	gn_dynamic_string_t** dst_dynamic_string,
	const gn_dynamic_string_t* src_dynamic_string
	);

/* gn_dynamic_string_free
 *
 * Free a Dynamic String object created by gn_dynamic_string_create.
 *
 * Parameters:
 *  IN/OUT	: dynamic_string	: An initialized Dynamic String object.
 *
 * Return Values:
 *  On success:
 *   gnex_error_t				: GNEX_PHOCVRT_NoError
 *   dynamic_string				: An uinitialized Dynamic String object,
 *								freed and set to GN_NULL.
 *
 *	On error:
 *   gnex_error_t				: Gracenote Extension Error Value
 *   dynamic_string				: Will not have been modified.
 *
 *  Errors include:
 *   GNEX_PHOCVRT_InvalidArg	: not initialized
 */
gnex_error_t
gn_dynamic_string_free(
	gn_dynamic_string_t** dynamic_string
	);

/* gn_dynamic_string_append_segment
 *
 * Append string segment to a Dynamic String object.
 *
 * Memory may be allocated by this call. The gn_dynamic_string_free function
 * should be called once the Phoneme Map object is no longer needed.
 *
 * Parameters:
 *  IN/OUT	: dynamic_string	: An initialized Dynamic String object.
 *	IN		: append			: String to append to Dynamic String.
 *	IN		: size				: Size of string to append to Dynamic String.
 *
 * Return Values:
 *  On success:
 *   gnex_error_t				: GNEX_PHOCVRT_NoError
 *   dynamic_string				: Dynamic String with string segement appended.
 *
 *	On error:
 *   gnex_error_t				: Gracenote Extension Error Value
 *   dynamic_string				: Will not have been modified.
 *
 *  Errors include:
 *	 GNEX_PHOCVRT_NoMemory		: memory allocation
 *   GNEX_PHOCVRT_InvalidArg	: not initialized
 */
gnex_error_t
gn_dynamic_string_append_segment(
	gn_dynamic_string_t* dynamic_string,
	const gn_uchar_t* append,
	gn_uint32_t size
	);

/* gn_dynamic_string_append_string
 *
 * Append string to a Dynamic String object.
 *
 * Memory may be allocated by this call. The gn_dynamic_string_free function
 * should be called once the Phoneme Map object is no longer needed.
 *
 * Parameters:
 *  IN/OUT	: dynamic_string	: An initialized Dynamic String object.
 *	IN		: append			: String to append to Dynamic String.
 *
 * Return Values:
 *  On success:
 *   gnex_error_t				: GNEX_PHOCVRT_NoError
 *   dynamic_string				: Dynamic String with string appended.
 *
 *	On error:
 *   gnex_error_t				: Gracenote Extension Error Value
 *   dynamic_string				: Will not have been modified.
 *
 *  Errors include:
 *	 GNEX_PHOCVRT_NoMemory		: memory allocation
 *   GNEX_PHOCVRT_InvalidArg	: not initialized
 */
gnex_error_t
gn_dynamic_string_append_string(
	gn_dynamic_string_t* dynamic_string,
	const gn_uchar_t* append
	);
	
/* gn_dynamic_string_append_char
 *
 * Append character to a Dynamic String object.
 *
 * Memory may be allocated by this call. The gn_dynamic_string_free function
 * should be called once the Phoneme Map object is no longer needed.
 *
 * Parameters:
 *  IN/OUT	: dynamic_string	: An initialized Dynamic String object.
 *	IN		: append			: Character to append to Dynamic String.
 *
 * Return Values:
 *  On success:
 *   gnex_error_t				: GNEX_PHOCVRT_NoError
 *   dynamic_string				: Dynamic String with character appended.
 *
 *	On error:
 *   gnex_error_t				: Gracenote Extension Error Value
 *   dynamic_string				: Will not have been modified.
 *
 *  Errors include:
 *	 GNEX_PHOCVRT_NoMemory		: memory allocation
 *   GNEX_PHOCVRT_InvalidArg	: not initialized
 */
gnex_error_t
gn_dynamic_string_append_char(
	gn_dynamic_string_t* dynamic_string,
	gn_uchar_t append
	);

/* gn_dynamic_string_clear
 *
 * Clear a Dynamic String object.
 *
 * Parameters:
 *  IN/OUT	: dynamic_string	: An initialized Dynamic String object.
 *
 * Return Values:
 *  On success:
 *   gnex_error_t				: GNEX_PHOCVRT_NoError
 *   dynamic_string				: Initialized Dynamic String with data cleared.
 *
 *	On error:
 *   gnex_error_t				: Gracenote Extension Error Value
 *   dynamic_string				: Will not have been modified.
 *
 *  Errors include:
 *   GNEX_PHOCVRT_InvalidArg	: not initialized
 */
gnex_error_t
gn_dynamic_string_clear(
	gn_dynamic_string_t* dynamic_string
	);

#ifdef __cplusplus
};
#endif

#endif
