/// \file GracenoteLanguageId.cpp
///
/// This provides helper classes to convert sds_fi_lang language codes to
/// Gracenote spoken language identifieres and back
///
/// Copyright: (c) 2011 Robert Bosch GmbH
///
/// \author Ingo Reise CM-AI/PJ-GM28 (external.ingo.reise@de.bosch.com)
///
/// \version 05.04.2011 Initial version
/// \version 13.06.2016 Upgraded ISO 639-1 two letter to ISO 639-2/639-3 three letter codes (based on GM changes from 30.03.2015)
///
/// \section Example Example
/// \code
/// tclLanguageId german(sds_fi_tcl_e16_ISOLanguageCode::FI_EN_ISO_639_1_DEU,
///                   sds_fi_tcl_e16_ISOCountryCode::FI_EN_ISO_ALPHA_3_DEU)
/// \endcode

using namespace std;

#define __PLACEMENT_NEW_INLINE
#include "GracenoteLanguageId.h"
#include <sstream>
#include <iostream>
#include <string>
#include <map>


static map<string, string> ISO639Helper;
/* **************************************************************************
 * Function initISO639HelperStruct 
 * *************************************************************************/ 
/**
* Fill ISO639HelperStruct
* 
* Content taken from
* http://en.wikipedia.org/wiki/List_of_ISO_639-1_codes#Partial_ISO_639_table
* 
*/
void tclLanguageId::initISO639HelperStruct(void)
{
	if (ISO639Helper.size() > 0)
		return ;
	
	////maps ISO639-2/T+ISO639-3(partial) to gracenote 'style' ISO639-2/B
	//ISO639-2/T section
	ISO639Helper["aar"] = "aar";
	ISO639Helper["abk"] = "abk";
	ISO639Helper["ave"] = "ave";
	ISO639Helper["afr"] = "afr";
	ISO639Helper["aka"] = "aka";
	ISO639Helper["amh"] = "amh";
	ISO639Helper["arg"] = "arg";
	ISO639Helper["ara"] = "ara";
	ISO639Helper["asm"] = "asm";
	ISO639Helper["ava"] = "ava";
	ISO639Helper["aym"] = "aym";
	ISO639Helper["aze"] = "aze";
	ISO639Helper["bak"] = "bak";
	ISO639Helper["bel"] = "bel";
	ISO639Helper["bul"] = "bul";
	ISO639Helper["bih"] = "bih";
	ISO639Helper["bis"] = "bis";
	ISO639Helper["bam"] = "bam";
	ISO639Helper["ben"] = "ben";
	ISO639Helper["bod"] = "tib";
	ISO639Helper["bre"] = "bre";
	ISO639Helper["bos"] = "bos";
	ISO639Helper["cat"] = "cat";
	ISO639Helper["che"] = "che";
	ISO639Helper["cha"] = "cha";
	ISO639Helper["cos"] = "cos";
	ISO639Helper["cre"] = "cre";
	ISO639Helper["ces"] = "cze";
	ISO639Helper["chu"] = "chu";
	ISO639Helper["chv"] = "chv";
	ISO639Helper["cym"] = "wel";
	ISO639Helper["dan"] = "dan";
	ISO639Helper["deu"] = "ger";
	ISO639Helper["div"] = "div";
	ISO639Helper["dzo"] = "dzo";
	ISO639Helper["ewe"] = "ewe";
	ISO639Helper["ell"] = "gre";
	ISO639Helper["eng"] = "eng";
	ISO639Helper["epo"] = "epo";
	ISO639Helper["spa"] = "spa";
	ISO639Helper["est"] = "est";
	ISO639Helper["eus"] = "baq";
	ISO639Helper["fas"] = "per";
	ISO639Helper["ful"] = "ful";
	ISO639Helper["fin"] = "fin";
	ISO639Helper["fij"] = "fij";
	ISO639Helper["fao"] = "fao";
	ISO639Helper["fra"] = "fre";
	ISO639Helper["fry"] = "fry";
	ISO639Helper["gle"] = "gle";
	ISO639Helper["gla"] = "gla";
	ISO639Helper["glg"] = "glg";
	ISO639Helper["grn"] = "grn";
	ISO639Helper["guj"] = "guj";
	ISO639Helper["glv"] = "glv";
	ISO639Helper["hau"] = "hau";
	ISO639Helper["heb"] = "heb";
	ISO639Helper["hin"] = "hin";
	ISO639Helper["hmo"] = "hmo";
	ISO639Helper["hrv"] = "hrv";
	ISO639Helper["hat"] = "hat";
	ISO639Helper["hun"] = "hun";
	ISO639Helper["hye"] = "arm";
	ISO639Helper["her"] = "her";
	ISO639Helper["ina"] = "ina";
	ISO639Helper["ind"] = "ind";
	ISO639Helper["ile"] = "ile";
	ISO639Helper["ibo"] = "ibo";
	ISO639Helper["iii"] = "iii";
	ISO639Helper["ipk"] = "ipk";
	ISO639Helper["ido"] = "ido";
	ISO639Helper["isl"] = "ice";
	ISO639Helper["ita"] = "ita";
	ISO639Helper["iku"] = "iku";
	ISO639Helper["jpn"] = "jpn";
	ISO639Helper["jav"] = "jav";
	ISO639Helper["kat"] = "geo";
	ISO639Helper["kon"] = "kon";
	ISO639Helper["kik"] = "kik";
	ISO639Helper["kua"] = "kua";
	ISO639Helper["kaz"] = "kaz";
	ISO639Helper["kal"] = "kal";
	ISO639Helper["khm"] = "khm";
	ISO639Helper["kan"] = "kan";
	ISO639Helper["kor"] = "kor";
	ISO639Helper["kau"] = "kau";
	ISO639Helper["kas"] = "kas";
	ISO639Helper["kur"] = "kur";
	ISO639Helper["kom"] = "kom";
	ISO639Helper["cor"] = "cor";
	ISO639Helper["kir"] = "kir";
	ISO639Helper["lat"] = "lat";
	ISO639Helper["ltz"] = "ltz";
	ISO639Helper["lug"] = "lug";
	ISO639Helper["lim"] = "lim";
	ISO639Helper["lin"] = "lin";
	ISO639Helper["lao"] = "lao";
	ISO639Helper["lit"] = "lit";
	ISO639Helper["lub"] = "lub";
	ISO639Helper["lav"] = "lav";
	ISO639Helper["mlg"] = "mlg";
	ISO639Helper["mah"] = "mah";
	ISO639Helper["mri"] = "mao";
	ISO639Helper["mkd"] = "mac";
	ISO639Helper["mal"] = "mal";
	ISO639Helper["mon"] = "mon";
	ISO639Helper["mar"] = "mar";
	ISO639Helper["msa"] = "may";
	ISO639Helper["mlt"] = "mlt";
	ISO639Helper["mya"] = "bur";
	ISO639Helper["nau"] = "nau";
	ISO639Helper["nob"] = "nob";
	ISO639Helper["nde"] = "nde";
	ISO639Helper["nep"] = "nep";
	ISO639Helper["ndo"] = "ndo";
	ISO639Helper["nld"] = "dut";
	ISO639Helper["nno"] = "nno";
	ISO639Helper["nor"] = "nor";
	ISO639Helper["nbl"] = "nbl";
	ISO639Helper["nav"] = "nav";
	ISO639Helper["nya"] = "nya";
	ISO639Helper["oci"] = "oci";
	ISO639Helper["oji"] = "oji";
	ISO639Helper["orm"] = "orm";
	ISO639Helper["ori"] = "ori";
	ISO639Helper["oss"] = "oss";
	ISO639Helper["pan"] = "pan";
	ISO639Helper["pli"] = "pli";
	ISO639Helper["pol"] = "pol";
	ISO639Helper["pus"] = "pus";
	ISO639Helper["por"] = "por";
	ISO639Helper["que"] = "que";
	ISO639Helper["roh"] = "roh";
	ISO639Helper["run"] = "run";
	ISO639Helper["ron"] = "rum";
	ISO639Helper["rus"] = "rus";
	ISO639Helper["kin"] = "kin";
	ISO639Helper["san"] = "san";
	ISO639Helper["srd"] = "srd";
	ISO639Helper["snd"] = "snd";
	ISO639Helper["sme"] = "sme";
	ISO639Helper["sag"] = "sag";
	ISO639Helper["sin"] = "sin";
	ISO639Helper["slk"] = "slo";
	ISO639Helper["slv"] = "slv";
	ISO639Helper["smo"] = "smo";
	ISO639Helper["sna"] = "sna";
	ISO639Helper["som"] = "som";
	ISO639Helper["sqi"] = "alb";
	ISO639Helper["srp"] = "srp";
	ISO639Helper["ssw"] = "ssw";
	ISO639Helper["sot"] = "sot";
	ISO639Helper["sun"] = "sun";
	ISO639Helper["swe"] = "swe";
	ISO639Helper["swa"] = "swa";
	ISO639Helper["tam"] = "tam";
	ISO639Helper["tel"] = "tel";
	ISO639Helper["tgk"] = "tgk";
	ISO639Helper["tha"] = "tha";
	ISO639Helper["tir"] = "tir";
	ISO639Helper["tuk"] = "tuk";
	ISO639Helper["tgl"] = "tgl";
	ISO639Helper["tsn"] = "tsn";
	ISO639Helper["ton"] = "ton";
	ISO639Helper["tur"] = "tur";
	ISO639Helper["tso"] = "tso";
	ISO639Helper["tat"] = "tat";
	ISO639Helper["twi"] = "twi";
	ISO639Helper["tah"] = "tah";
	ISO639Helper["uig"] = "uig";
	ISO639Helper["ukr"] = "ukr";
	ISO639Helper["urd"] = "urd";
	ISO639Helper["uzb"] = "uzb";
	ISO639Helper["ven"] = "ven";
	ISO639Helper["vie"] = "vie";
	ISO639Helper["vol"] = "vol";
	ISO639Helper["wln"] = "wln";
	ISO639Helper["wol"] = "wol";
	ISO639Helper["xho"] = "xho";
	ISO639Helper["yid"] = "yid";
	ISO639Helper["yor"] = "yor";
	ISO639Helper["zha"] = "zha"; 
	ISO639Helper["zho"] = "qad"; //Exception: use 'qad' instead of 'chi'
	ISO639Helper["zul"] = "zul";
	
	// ISO639-3 section (partial support only)
	ISO639Helper["cmn"] = "qad"; //mapping ISO639-3 Mandarin 'cmn' to 'qad'
	
	// Unknown not supported language
	ISO639Helper["???"] = "???";

	/*ISO639Helper["aa"] = "aar";
	ISO639Helper["ab"] = "abk";
	ISO639Helper["ae"] = "ave";
	ISO639Helper["af"] = "afr";
	ISO639Helper["ak"] = "aka";
	ISO639Helper["am"] = "amh";
	ISO639Helper["an"] = "arg";
	ISO639Helper["ar"] = "ara";
	ISO639Helper["as"] = "asm";
	ISO639Helper["av"] = "ava";
	ISO639Helper["ay"] = "aym";
	ISO639Helper["az"] = "aze";
	ISO639Helper["ba"] = "bak";
	ISO639Helper["be"] = "bel";
	ISO639Helper["bg"] = "bul";
	ISO639Helper["bh"] = "bih";
	ISO639Helper["bi"] = "bis";
	ISO639Helper["bm"] = "bam";
	ISO639Helper["bn"] = "ben";
	ISO639Helper["bo"] = "tib";
	ISO639Helper["br"] = "bre";
	ISO639Helper["bs"] = "bos";
	ISO639Helper["ca"] = "cat";
	ISO639Helper["ce"] = "che";
	ISO639Helper["ch"] = "cha";
	ISO639Helper["co"] = "cos";
	ISO639Helper["cr"] = "cre";
	ISO639Helper["cs"] = "cze";
	ISO639Helper["cu"] = "chu";
	ISO639Helper["cv"] = "chv";
	ISO639Helper["cy"] = "wel";
	ISO639Helper["da"] = "dan";
	ISO639Helper["de"] = "ger";
	ISO639Helper["dv"] = "div";
	ISO639Helper["dz"] = "dzo";
	ISO639Helper["ee"] = "ewe";
	ISO639Helper["el"] = "gre";
	ISO639Helper["en"] = "eng";
	ISO639Helper["eo"] = "epo";
	ISO639Helper["es"] = "spa";
	ISO639Helper["et"] = "est";
	ISO639Helper["eu"] = "baq";
	ISO639Helper["fa"] = "per";
	ISO639Helper["ff"] = "ful";
	ISO639Helper["fi"] = "fin";
	ISO639Helper["fj"] = "fij";
	ISO639Helper["fo"] = "fao";
	ISO639Helper["fr"] = "fre";
	ISO639Helper["fy"] = "fry";
	ISO639Helper["ga"] = "gle";
	ISO639Helper["gd"] = "gla";
	ISO639Helper["gl"] = "glg";
	ISO639Helper["gn"] = "grn";
	ISO639Helper["gu"] = "guj";
	ISO639Helper["gv"] = "glv";
	ISO639Helper["ha"] = "hau";
	ISO639Helper["he"] = "heb";
	ISO639Helper["hi"] = "hin";
	ISO639Helper["ho"] = "hmo";
	ISO639Helper["hr"] = "hrv";
	ISO639Helper["ht"] = "hat";
	ISO639Helper["hu"] = "hun";
	ISO639Helper["hy"] = "arm";
	ISO639Helper["hz"] = "her";
	ISO639Helper["ia"] = "ina";
	ISO639Helper["id"] = "ind";
	ISO639Helper["ie"] = "ile";
	ISO639Helper["ig"] = "ibo";
	ISO639Helper["ii"] = "iii";
	ISO639Helper["ik"] = "ipk";
	ISO639Helper["io"] = "ido";
	ISO639Helper["is"] = "ice";
	ISO639Helper["it"] = "ita";
	ISO639Helper["iu"] = "iku";
	ISO639Helper["ja"] = "jpn";
	ISO639Helper["jv"] = "jav";
	ISO639Helper["ka"] = "geo";
	ISO639Helper["kg"] = "kon";
	ISO639Helper["ki"] = "kik";
	ISO639Helper["kj"] = "kua";
	ISO639Helper["kk"] = "kaz";
	ISO639Helper["kl"] = "kal";
	ISO639Helper["km"] = "khm";
	ISO639Helper["kn"] = "kan";
	ISO639Helper["ko"] = "kor";
	ISO639Helper["kr"] = "kau";
	ISO639Helper["ks"] = "kas";
	ISO639Helper["ku"] = "kur";
	ISO639Helper["kv"] = "kom";
	ISO639Helper["kw"] = "cor";
	ISO639Helper["ky"] = "kir";
	ISO639Helper["la"] = "lat";
	ISO639Helper["lb"] = "ltz";
	ISO639Helper["lg"] = "lug";
	ISO639Helper["li"] = "lim";
	ISO639Helper["ln"] = "lin";
	ISO639Helper["lo"] = "lao";
	ISO639Helper["lt"] = "lit";
	ISO639Helper["lu"] = "lub";
	ISO639Helper["lv"] = "lav";
	ISO639Helper["mg"] = "mlg";
	ISO639Helper["mh"] = "mah";
	ISO639Helper["mi"] = "mao";
	ISO639Helper["mk"] = "mac";
	ISO639Helper["ml"] = "mal";
	ISO639Helper["mn"] = "mon";
	ISO639Helper["mr"] = "mar";
	ISO639Helper["ms"] = "may";
	ISO639Helper["mt"] = "mlt";
	ISO639Helper["my"] = "bur";
	ISO639Helper["na"] = "nau";
	ISO639Helper["nb"] = "nob";
	ISO639Helper["nd"] = "nde";
	ISO639Helper["ne"] = "nep";
	ISO639Helper["ng"] = "ndo";
	ISO639Helper["nl"] = "dut";
	ISO639Helper["nn"] = "nno";
	ISO639Helper["no"] = "nor";
	ISO639Helper["nr"] = "nbl";
	ISO639Helper["nv"] = "nav";
	ISO639Helper["ny"] = "nya";
	ISO639Helper["oc"] = "oci";
	ISO639Helper["oj"] = "oji";
	ISO639Helper["om"] = "orm";
	ISO639Helper["or"] = "ori";
	ISO639Helper["os"] = "oss";
	ISO639Helper["pa"] = "pan";
	ISO639Helper["pi"] = "pli";
	ISO639Helper["pl"] = "pol";
	ISO639Helper["ps"] = "pus";
	ISO639Helper["pt"] = "por";
	ISO639Helper["qu"] = "que";
	ISO639Helper["rm"] = "roh";
	ISO639Helper["rn"] = "run";
	ISO639Helper["ro"] = "rum";
	ISO639Helper["ru"] = "rus";
	ISO639Helper["rw"] = "kin";
	ISO639Helper["sa"] = "san";
	ISO639Helper["sc"] = "srd";
	ISO639Helper["sd"] = "snd";
	ISO639Helper["se"] = "sme";
	ISO639Helper["sg"] = "sag";
	ISO639Helper["si"] = "sin";
	ISO639Helper["sk"] = "slo";
	ISO639Helper["sl"] = "slv";
	ISO639Helper["sm"] = "smo";
	ISO639Helper["sn"] = "sna";
	ISO639Helper["so"] = "som";
	ISO639Helper["sq"] = "alb";
	ISO639Helper["sr"] = "srp";
	ISO639Helper["ss"] = "ssw";
	ISO639Helper["st"] = "sot";
	ISO639Helper["su"] = "sun";
	ISO639Helper["sv"] = "swe";
	ISO639Helper["sw"] = "swa";
	ISO639Helper["ta"] = "tam";
	ISO639Helper["te"] = "tel";
	ISO639Helper["tg"] = "tgk";
	ISO639Helper["th"] = "tha";
	ISO639Helper["ti"] = "tir";
	ISO639Helper["tk"] = "tuk";
	ISO639Helper["tl"] = "tgl";
	ISO639Helper["tn"] = "tsn";
	ISO639Helper["to"] = "ton";
	ISO639Helper["tr"] = "tur";
	ISO639Helper["ts"] = "tso";
	ISO639Helper["tt"] = "tat";
	ISO639Helper["tw"] = "twi";
	ISO639Helper["ty"] = "tah";
	ISO639Helper["ug"] = "uig";
	ISO639Helper["uk"] = "ukr";
	ISO639Helper["ur"] = "urd";
	ISO639Helper["uz"] = "uzb";
	ISO639Helper["ve"] = "ven";
	ISO639Helper["vi"] = "vie";
	ISO639Helper["vo"] = "vol";
	ISO639Helper["wa"] = "wln";
	ISO639Helper["wo"] = "wol";
	ISO639Helper["xh"] = "xho";
	ISO639Helper["yi"] = "yid";
	ISO639Helper["yo"] = "yor";
	ISO639Helper["za"] = "zha";
	//    ISO639Helper["zh"]="chi"; Exception: use qad instead
	ISO639Helper["zh"] = "qad";
	ISO639Helper["zu"] = "zul";
	// Unknown not supported language
	ISO639Helper["??"] = "???";*/
}

// tclLanguageId::ISO639HelperStruct tclLanguageId::ISO639Helper;


void tclLanguageId::updateGNCode(void)
{
	ostringstream GNCode;
	
	//calculate the ISO639-3 character sequence...
	char ISO6393[4] = { (char) (0x60 + (int) (language / 1024)), (char) (0x60 + (int) ((language % 1024) / 32)), (char) (0x60 + (language % 32)),'\0' };

	if (language == 0)
	{
		ISO6393[0] = '\0';
	}

	GNCode << (char) (64 + (int) (country / 1024))
	<< (char) (64 + (int) ((country % 1024) / 32))
	<< (char) (64 + (country % 32))
	<< "_";
	if (ISO639Helper.find(ISO6393) != ISO639Helper.end())
	{
		GNCode << ISO639Helper[ISO6393];
	}
	else
	{
		GNCode << ISO639Helper["???"];
	}

   GNCode.str().copy((char *)GNCodeArray, GNCode.str().size());
   GNCodeArray[7] = '\0';
}


/* **************************************************************************
 * Function cast to  gn_spklangid_t
 * *************************************************************************/ 
/**
 * get Gracenote spoken language identifier
 * 
 * This is useful also for output in streams 
 * 
 * \return Gracenote spoken language identifier
 * 
 */
tclLanguageId::operator unsigned char * const() const
{
   return (unsigned char *) GNCodeArray;
}


/* **************************************************************************
 * Function Constructor 
 * *************************************************************************/ 
/**
 * Constructor specifying both codes directly
 * 
 * Example:
 * \code
 * tclLanguageId german(sds_fi_tcl_e16_ISOLanguageCode::FI_EN_ISO_639_3_DEU,
 *                   sds_fi_tcl_e16_ISOCountryCode::FI_EN_ISO_ALPHA_3_DEU);
 * \endcode
 */
tclLanguageId::tclLanguageId(tclLanguageCode LC, tclCountryCode CC) : language(LC),
      country(CC)
{
   initISO639HelperStruct();
   updateGNCode();
}

/* **************************************************************************
 * Function Constructor 
 * *************************************************************************/ 
/**
 * Default constructor initializing to 0, this should be a not existing 
 * langauage
 */
tclLanguageId::tclLanguageId() : language((short)0), country((short)0)
{
   initISO639HelperStruct();
   updateGNCode();
}

/* **************************************************************************
 * Function Constructor 
 * *************************************************************************/ 
/**
 * Constructor specifying a spoken language identifier from Gracenote
 * 
 * Example:
 * \code
 * tclLanguageId english("USA_eng");
 * \endcode
 */
tclLanguageId::tclLanguageId(char *gnId) : language((short) 0),
      country((char *) string(gnId).substr(0, 3).c_str())
{
   initISO639HelperStruct();
   string lang3 = string(gnId).substr(4);
   language = tclLanguageCode("???");
   for (map<string,string>::iterator ihsi = ISO639Helper.begin();
         ihsi != ISO639Helper.end();
         ++ihsi)
      if (ihsi->second == lang3)
      {
         language = tclLanguageCode((char *) ihsi->first.c_str());
         break;
      }

   updateGNCode();
   if (string(GNCodeArray) != gnId)
      std::cerr << "tclLanguageId: Invalid language code: " << gnId << endl;
}

/* **************************************************************************
 * Function Less operator 
 * *************************************************************************/ 
/**
 * Less operator
 */
bool tclLanguageId::operator<(const tclLanguageId &li2) const
{
   return (this->language < li2.language
           || (this->language == li2.language && this->country < li2.country));
}

/* **************************************************************************
 * Function Equality operator 
 * *************************************************************************/ 
/**
 * Equality operator
 */
bool tclLanguageId::operator==(const tclLanguageId &li2) const
{
   return (this->language == li2.language
           && this->country == li2.country);
}

