/// \file GracenoteLanguageCode.cpp
///
/// This provides base class to handle numeric (short) representations of ISO language codes
///
/// Copyright: (c) 2011 Robert Bosch GmbH
///
/// \author Ingo Reise CM-AI/PJ-GM28 (external.ingo.reise@de.bosch.com)
///
/// \version 05.04.2011 Initial version
/// \version 13.06.2016 Upgraded ISO 639-1 two letter to ISO 639-2/639-3 three letter codes (based on GM changes from 30.03.2015)
///
#include "GracenoteLanguageCode.h"
#include "string.h"

/* **************************************************************************
 * Function setCode
 * *************************************************************************/ 
/**
 * \param[in] lc 3-char language code (ISO 639-2/639-3)
 */
void tclLanguageCode::setCode(char* const lc)
{
	//ISO 639-3 codes each character in 5-bit, starting with 1 for A...
	if (strlen(lc)==3)
	{
		//ISO 639-3 defines lower case characters, but we are supporting upper and lower case (but no mixtures)
		if (lc[0] >= 0x61 && lc[0] <= 0x7A
			&& lc[1] >= 0x61 && lc[1] <= 0x7A
			&& lc[2] >= 0x61 && lc[2] <= 0x7A)
		{
			code = (short) ( (lc[0]-0x60)*1024 + (lc[1]-0x60)*32 + (lc[2]-0x60));
		}
		else if (lc[0] >= 0x41 && lc[0] <= 0x5A
			&& lc[1] >= 0x41 && lc[1] <= 0x5A
			&& lc[2] >= 0x41 && lc[2] <= 0x5A)
		{
			code = (short) ( (lc[0]-0x40)*1024 + (lc[1]-0x40)*32 + (lc[2]-0x40));
		}
	}
}

/*! \cond ALL */
/* **************************************************************************
 * Function Constructor 
 * *************************************************************************/ 
/**
 * Constructor just setting code directly
 * 
 * \param[in] lc tclLanguageCode
 */
tclLanguageCode::tclLanguageCode(short lc)
{
    code = lc;
}

/* **************************************************************************
 * Function Constructor  
 * *************************************************************************/ 
/**
 * Constructor converting two characters 
 * 
 * \param[in] lc 3-char language code (ISO 639-2/639-3)
 */
tclLanguageCode::tclLanguageCode(char *lc)
{
    code = 0;
	
	setCode(lc);
}

/*! \endcond */

/// Constructor with sds_fi_tcl_e16_ISOLanguageCode
/// \details Example:
/// \code tclLanguageCode(sds_fi_tcl_e16_ISOLanguageCode::FI_EN_ISO_639_1_DE); \endcode
/* **************************************************************************
 * Function Constructor
 * *************************************************************************/ 
/**
 * Constructor converting sds_fi_tcl_e16_ISOLanguageCode
 *
 * \param[in] lc sds_fi_tcl_e16_ISOLanguageCode
 * 
 */
tclLanguageCode::tclLanguageCode(sds_fi_tcl_e16_ISO639_3_SDSLanguageCode lc)
{
    code = (short) lc.enType;
}
