/*
 * Copyright (C) Robert Bosch Car Multimedia GmbH, 2020
 * This software is property of Robert Bosch GmbH. Unauthorized
 * duplication and disclosure to third parties is prohibited.
 *
 * FILE                : HealthcareMonParser.cpp
 * COMPONENT Name      : HealthcareMonParser
 * @details            :Parsing HelthcareMon application doipcmcerrmemdb.xml to get the DTC Info of the devices.
 * AUTHOR              : Piyush Priye
 * Date                : 11.05.2021
 * Revision History    : 1.0
 */
 #include "HealthMontrace.h"
 #include "HealthcareMonParser.h"
 #include <iostream>
#include <bits/stdc++.h>
using namespace std;

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_HEALTHCAREMONASF
#define ETG_I_TRACE_CHANNEL               TR_COMP_FC_TELEMATIC
#define ETG_I_TTFIS_CMD_PREFIX            "HEALTHCAREMONAPP"
#define ETG_I_FILE_PREFIX                 HEALTHCAREMONAPP::SERVICE
#include "trcGenProj/Header/HealthcareMonParser.cpp.trc.h"
#endif


#ifdef TEST_XML
	tCString  XML_PATH = "./doipcmcerrmemdb.xml";
#else
	tCString  XML_PATH = "/opt/bosch/coach/doipcmcerrmemdb.xml";
#endif

#define ATTR_MATCH 0
#define NODE_TAG "node"
#define ATTR_NAME "name"
#define ATTR_CODE "code"
#define ATTR_SEVERITY "severity"
#define ATTR_TEXTID "textid"
#define ATTR_ERR "err"
#define ATTR_LANG "lang"
#define DEV_NAME_CMC "CMC01B"
#define DEV_NAME_CMA "Amplifier"
#define DEV_NAME_CMG "Glasses"
#define DEV_NAME_CMP "CameraPort"
#define DEV_NAME_CMR "Router"
#define ROOT_TAG "errmemdb"
#define HEXADECIMAL_BASE 16
#define MANDATORY_ATTR_COUNT 5

tclHealthcareMonParser* tclHealthcareMonParser::m_pInstance = nullptr;

/*! ***********************************************************************
 *FUNCTION: 		tclHealthcareMonParser
 *@details: 		Constr
 *REQUIREMENT ID:	NA
 *@param:		NA
 *@return: 		NA
 *@details
 *HISTORY:
 *05.28.2020		1.0		Piyush Priye		Intial Version
 ************************************************************************/
tclHealthcareMonParser::tclHealthcareMonParser():m_bParseState(false)
{
	ETG_TRACE_USR4(("tclHealthcareMonParser: Constructor"));
}

/*! ***********************************************************************
 *FUNCTION: 		tclHealthcareMonParser
 *@details: 		Destr
 *REQUIREMENT ID:	NA
 *@param:		NA
 *@return: 		NA
 *@details
 *HISTORY:
 *05.28.2020		1.0		Piyush Priye		Intial Version
 ************************************************************************/
tclHealthcareMonParser::~tclHealthcareMonParser()
{
	ETG_TRACE_USR4(("tclHealthcareMonParser: Destructor"));
}

/*! ***********************************************************************
 *FUNCTION: 		pGetInstance
 *@details: 		Getting self instance
 *REQUIREMENT ID:	NA(Utility Function)
 *@param:		NA
 *@return: 		tclAvRoutingParser ptr
 *@details
 *HISTORY:
 *05.28.2020		1.0		Piyush Priye		Intial Version
 ************************************************************************/
tclHealthcareMonParser* tclHealthcareMonParser::pGetInstance()
{
	vInitPlatformEtg();	
	if( nullptr == m_pInstance ) {
		m_pInstance = new tclHealthcareMonParser();
	}
	ETG_TRACE_USR4(("tclHealthcareMonParser: Parser Instance address = %p",m_pInstance));
	return m_pInstance;
}

/*! ***********************************************************************
 *FUNCTION: 		bParseEntity
 *@details: 		Parsing Entity Tag
 *REQUIREMENT ID:	NA(Utility Function)
 *@param TiXmlElement ptr
 *@return: 		tBool
 *@details
 *HISTORY:
 *05.28.2020		1.0		Piyush priye		Intial Version
 ************************************************************************/
tBool tclHealthcareMonParser::bParseEntity(TiXmlElement* pElement)
{
	ETG_TRACE_USR4(("tclHealthcareMonParser: bParseEntity entered"));
   	if ( !pElement ) 
	{
		ETG_TRACE_ERR(("tclHealthcareMonParser:: XML EntityParse Failed"));
		return false;
	}
	TiXmlAttribute* pXmlAttribute = pElement->FirstAttribute();

	for ( ; pXmlAttribute;pXmlAttribute = pXmlAttribute->Next())
	{
		
		tCString  pAttribName = pXmlAttribute->Name();
		tCString  pAttribValue = pXmlAttribute->Value();
		ETG_TRACE_USR4(("tclHealthcareMonParser: pAttribName %s",pAttribName));
		ETG_TRACE_USR4(("tclHealthcareMonParser: pAttribValue %s",pAttribValue));
		
		if(ATTR_MATCH == strcmp(ATTR_NAME, pAttribName))
		{
			for( pElement = pElement->FirstChildElement(); pElement; pElement=pElement->NextSiblingElement())
			{
				TiXmlAttribute* pXmlAttribute = pElement->FirstAttribute();
			
				if( ATTR_MATCH == strcmp(DEV_NAME_CMC, pAttribValue))
				{
					tDeviceDTCInfo DeviceDTCInfo = {};
					DeviceDTCInfo = ParseDeviceDtcInfo(pXmlAttribute,pElement);
					m_CMCDTCInfo.push_back(DeviceDTCInfo);
				}
				if( ATTR_MATCH == strcmp(DEV_NAME_CMA, pAttribValue))
				{
					tDeviceDTCInfo DeviceDTCInfo = {};
					DeviceDTCInfo = ParseDeviceDtcInfo(pXmlAttribute,pElement);
					m_CMADTCInfo.push_back(DeviceDTCInfo);
				}
				if( ATTR_MATCH == strcmp(DEV_NAME_CMG, pAttribValue))
				{
					tDeviceDTCInfo DeviceDTCInfo = {};
					DeviceDTCInfo = ParseDeviceDtcInfo(pXmlAttribute,pElement);
					m_CMGDTCInfo.push_back(DeviceDTCInfo);
				}
				if( ATTR_MATCH == strcmp(DEV_NAME_CMP, pAttribValue))
				{
					tDeviceDTCInfo DeviceDTCInfo = {};
					DeviceDTCInfo = ParseDeviceDtcInfo(pXmlAttribute,pElement);
					m_CMPDTCInfo.push_back(DeviceDTCInfo);
				}
				if( ATTR_MATCH == strcmp(DEV_NAME_CMR, pAttribValue))
				{
					tDeviceDTCInfo DeviceDTCInfo = {};
					DeviceDTCInfo = ParseDeviceDtcInfo(pXmlAttribute,pElement);
					m_CMRDTCInfo.push_back(DeviceDTCInfo);
				}
			
			}
		}
		
	}
	return true;
}

/*! ***********************************************************************
 *FUNCTION: 		ParseDeviceDtcInfo
 *@details: 		Parsing Entity Tag
 *REQUIREMENT ID:	NA(Utility Function)
 *@param TiXmlElement ptr
 *@return: 		tBool
 *@details
 *HISTORY:
 *05.28.2020		1.0		Piyush priye		Intial Version
 ************************************************************************/
tDeviceDTCInfo tclHealthcareMonParser::ParseDeviceDtcInfo(TiXmlAttribute* pXmlAttribute,TiXmlElement* pElement)
{
	char* pEndString = nullptr;
	tDeviceDTCInfo DeviceDTCInfo;
	DeviceDTCInfo.m_sCode = 0xFFFFFFFFFF;
	DeviceDTCInfo.m_severity = 0xFF;
	DeviceDTCInfo.m_textid = 0xFFFF;
	DeviceDTCInfo.m_errname ="";
	DeviceDTCInfo.m_lang ="en";
	for ( ; pXmlAttribute;pXmlAttribute = pXmlAttribute->Next())
	{
		tCString  pAttribName = pXmlAttribute->Name();
		tCString  pAttribValue = pXmlAttribute->Value();
		if (ATTR_MATCH == strcmp(ATTR_CODE, pAttribName))
		{
			DeviceDTCInfo.m_sCode = std::strtoull(pAttribValue, &pEndString, HEXADECIMAL_BASE);
		}
		if (ATTR_MATCH == strcmp(ATTR_SEVERITY, pAttribName))
		{
			DeviceDTCInfo.m_severity = std::strtoull(pAttribValue, &pEndString, HEXADECIMAL_BASE);
		}
		if (ATTR_MATCH == strcmp(ATTR_TEXTID, pAttribName))
		{
			DeviceDTCInfo.m_textid = std::strtoull(pAttribValue, &pEndString, HEXADECIMAL_BASE);
		}
	}
	if(pElement != nullptr)
	{
		TiXmlElement * pElement1 = pElement->FirstChildElement();
		TiXmlAttribute* pXmlAttributeErrname = pElement1->FirstAttribute();
		for ( ; pXmlAttributeErrname;pXmlAttributeErrname = pXmlAttributeErrname->Next())
		{
			tCString  pAttribName = pXmlAttributeErrname->Name();
			tCString  pAttribValue = pXmlAttributeErrname->Value();
			if (ATTR_MATCH == strcmp(ATTR_ERR, pAttribName))
			{
				DeviceDTCInfo.m_errname = pAttribValue;
			}
			if (ATTR_MATCH == strcmp(ATTR_LANG, pAttribName))
			{
				DeviceDTCInfo.m_lang = pAttribValue;
			}
		}
	}
	
	ETG_TRACE_USR4(("tclHealthcareMonParser: ParseDeviceDtcInfo  DeviceDTCInfo.m_sCode= 0x%x",DeviceDTCInfo.m_sCode));
	ETG_TRACE_USR4(("tclHealthcareMonParser: ParseDeviceDtcInfo  DeviceDTCInfo.m_severity= %d",DeviceDTCInfo.m_severity));
	ETG_TRACE_USR4(("tclHealthcareMonParser: ParseDeviceDtcInfo  DeviceDTCInfo.m_textid = %d",DeviceDTCInfo.m_textid));
	ETG_TRACE_USR4(("tclHealthcareMonParser: ParseDeviceDtcInfo  DeviceDTCInfo.m_errname = %s",(DeviceDTCInfo.m_errname).c_str()));
	ETG_TRACE_USR4(("tclHealthcareMonParser: ParseDeviceDtcInfo  DeviceDTCInfo.m_lang = %s",(DeviceDTCInfo.m_lang).c_str()));
	return DeviceDTCInfo;
}

/*! ***********************************************************************
 *FUNCTION: 		bParseXml
 *@details: 		Parsing XML
 *REQUIREMENT ID:	NA(Utility Function)
 *@param:		NA
 *@return: 		tBool
 *@details
 *HISTORY:
 *05.28.2020		1.0		Piyush Priye		Intial Version
 ************************************************************************/
tBool tclHealthcareMonParser::bParseXml()
{
	ETG_TRACE_USR4(("tclHealthcareMonParser::bParseXml() entered"));
	if ( m_bParseState ) 
	{
		ETG_TRACE_USR4(("tclHealthcareMonParser: Xml already parsed"));
		return true;
	}
	TiXmlDocument xmlDoc(XML_PATH);
	if( xmlDoc.LoadFile())
	{
		TiXmlHandle xmlHandler(&xmlDoc);
		TiXmlElement* pElement = xmlHandler.FirstChild(ROOT_TAG).FirstChild(NODE_TAG).ToElement();
		if( pElement ) 
		{
			tCString  pTagName = pElement->Value();
			ETG_TRACE_USR4(("tclHealthcareMonParser: Tag Name is %s ",pTagName));	
			tBool bParseState =false;
			for( ; pElement; pElement=pElement->NextSiblingElement())
			{
				bParseState = bParseEntity(pElement);
			}
			ETG_TRACE_USR4(("tclHealthcareMonParser: CMC size is %d",m_CMCDTCInfo.size()));
			ETG_TRACE_USR4(("tclHealthcareMonParser: CMA size is %d",m_CMADTCInfo.size()));
			ETG_TRACE_USR4(("tclHealthcareMonParser: CMP size is %d",m_CMPDTCInfo.size()));
			ETG_TRACE_USR4(("tclHealthcareMonParser: CMG size is %d",m_CMGDTCInfo.size()));
			ETG_TRACE_USR4(("tclHealthcareMonParser: CMR size is %d",m_CMRDTCInfo.size()));
			m_bParseState = bParseState;
		}	
	}
	else 
	{
		//Failed to load file
		ETG_TRACE_ERRMEM(("tclHealthcareMonParser::bParseXml() - doipcmcerrmemdb.xml load Failed - Error: %s", xmlDoc.ErrorDesc()));
		m_parsingErrorObj.vSetErrorCode(xmlDoc.ErrorId());
		m_parsingErrorObj.vSetErrorDescription(xmlDoc.ErrorDesc());
		m_bParseState = false;
	}
	return m_bParseState;
}
/*! ***********************************************************************
 *FUNCTION: 		u8GetErrorCode
 *@details: 		GetErrorCode
 *REQUIREMENT ID:	NA(Utility Function)
 *@param:		NA
 *@return: 		Error code
 *@details
 *HISTORY:
 *08.03.2020		0.1		Piyush Priye		Initial Version
 ************************************************************************/
tU8 tclHealthcareMonParser::u8GetErrorCode()
{
	return m_parsingErrorObj.u8GetErrorCode();
}

/*! ***********************************************************************
 *FUNCTION: 		sGetErrorDescription
 *@details: 		GetErrorDescription
 *REQUIREMENT ID:	NA(Utility Function)
 *@param:		NA
 *@return: 		Error code
 *@details
 *HISTORY:
 *08.03.2020		0.1		Piyush Priye		Initial Version
 ************************************************************************/
std::string tclHealthcareMonParser::sGetErrorDescription()
{
	return m_parsingErrorObj.sGetErrorDescription();
}

/*! ***********************************************************************
 *FUNCTION: 		vectorGetCMCDTCCodeList
 *@details: 		Device DTC list in xml
 *REQUIREMENT ID:	NA(Utility Function)
 *@param:		NA
 *@return: 	        List the DTC Info of Devices.
 *@details
 *HISTORY:
 *05.22.2020		1.0		Piyush Priye		Initial Version
 ************************************************************************/
tDTCInfo  tclHealthcareMonParser::vectorGetDTCListInfo(tCString pDevName)
{
	ETG_TRACE_USR4(("tclHealthcareMonParser: vectorGetDTCListInfo"));
	
	if( ATTR_MATCH == strcmp(pDevName,"CMC01B"))
	{
		return 	m_CMCDTCInfo;
	}
	if( ATTR_MATCH == strcmp(pDevName,"Amplifier"))
	{
		return 	m_CMADTCInfo;
	}
	if( ATTR_MATCH == strcmp(pDevName,"Glasses"))
	{
		return 	m_CMGDTCInfo;
	}
	if( ATTR_MATCH == strcmp(pDevName,"CameraPort"))
	{
		return 	m_CMPDTCInfo;
	}
	if( ATTR_MATCH == strcmp(pDevName,"Router"))
	{
		return 	m_CMRDTCInfo;
	}
	return static_cast<tDTCInfo>(0);	
}



