/****************************************************************************
* Copyright (C) Robert Bosch Car Multimedia GmbH, 2018 - 2019
* This software is property of Robert Bosch GmbH. Unauthorized
* duplication and disclosure to third parties is prohibited.
*
* FILE                : PluginDataConverter.cpp
* COMPONENT Name      : PluginDataConverter
* DESCRIPTION         : Data coversion class used by PluginClientHandler and PluginServerHandler
*						      Converts data packed in EventDataUtility into Dbus variant data and vice versa
* AUTHOR              : Preethi Alagappan
* Date                : 15.10.2018
* Revision History    : 0.1
* Date 15.10.2018     : Initial version
* Revision History    : 0.2
* Date 15.11.2018     : Implementation for list data conversion
****************************************************************************/

/*****************************************************************
| includes
|----------------------------------------------------------------*/
#include "PluginDataConverter.h"
#include "../plugin_trace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS  TR_CLASS_APPUTILITIES_PLUGINDATACONVERTER
#include "trcGenProj/Header/PluginDataConverter.cpp.trc.h"
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN

/************************************************************************
*NAME        : convertToVariant
*DESCRIPTION : Design section 8.2.3.1 - Conversion from EventDataUtility to D-Bus variant
************************************************************************/
bool PluginDataConverter::convertToVariant(const boost::shared_ptr<EventDataUtility>& dataUtility, ::asf::dbus::DBusVariant& variant)
{
   ETG_TRACE_USR4(("PluginDataConverter::convertToVariant()"));
   bool isSuccess = false;

   if(NULL != dataUtility.get())
   {
      VariantWriter writer(variant);
      isSuccess = writeStructData(dataUtility.get(), &writer);
   }
   else  // EventDataUtility pointer is NULL
   {
      isSuccess = true;
      ETG_TRACE_USR4(("PluginDataConverter::convertToVariant(), EventDataUtility NULL pointer. No data to be converted"));
   }
   ETG_TRACE_USR4(("PluginDataConverter::convertToVariant(), isSuccess : %d", isSuccess));
   return isSuccess;
}

/************************************************************************
*NAME        : convertToListVariant
*DESCRIPTION : Design section 8.2.3.3 - Conversion from EventListDataUtility to D-Bus variant
************************************************************************/
bool PluginDataConverter::convertToListVariant(const boost::shared_ptr<EventListDataUtility>& listDataUtility, ::asf::dbus::DBusVariant& variant)
{
   ETG_TRACE_USR4(("PluginDataConverter::convertToListVariant()"));
   bool isSuccess = false;

   if(NULL != listDataUtility.get())
   {
      const std::vector<const EventDataUtility*>& listDataItems = listDataUtility->getData();
      if(0 < listDataItems.size())
      {
         std::string arraySignature;
         isSuccess = getArraySignature(listDataItems.at(0), arraySignature);

         if(true == isSuccess)
         {
            VariantWriter writer(variant);
            isSuccess = writer.open_array(arraySignature.c_str());
            for(int index = 0; (true == isSuccess) && (index < listDataItems.size()); index++)
            {
               isSuccess = writeArrayStructData(listDataItems.at(index), &writer, arraySignature);
            }
            isSuccess = isSuccess && writer.close_array();
         }
      }
      else  // No list entries
      {
         isSuccess = true;
         ETG_TRACE_USR4(("PluginDataConverter::convertToListVariant(), No items in EventListDataUtility"));
      }
   }
   else  // EventListDataUtility pointer is NULL
   {
      isSuccess = true;
      ETG_TRACE_USR4(("PluginDataConverter::convertToListVariant(), EventListDataUtility NULL pointer. No data to be converted"));
   }
   ETG_TRACE_USR4(("PluginDataConverter::convertToListVariant(), isSuccess : %d", isSuccess));
   return isSuccess;
}

/************************************************************************
*NAME        : convertToEventDataUtility
*DESCRIPTION : Design section 8.2.3.2 - Conversion from D-Bus variant to EventDataUtility
************************************************************************/
bool PluginDataConverter::convertToEventDataUtility(::asf::dbus::DBusVariant variant, boost::shared_ptr<EventDataUtility>& dataUtility)
{
   ETG_TRACE_USR4(("PluginDataConverter::convertToEventDataUtility()"));
   bool isSuccess = false;

   std::string typeSpecifier;
   variant.getSignature(typeSpecifier);
   ETG_TRACE_USR4(("PluginDataConverter::convertToEventDataUtility(), Type specifier : %s", typeSpecifier.c_str()));

   if(0 < typeSpecifier.length())
   {
      VariantReader reader(variant);
      dataUtility = boost::shared_ptr<EventDataUtility>(EventDataUtility::newEventDataUtility());

      isSuccess = readStructData(&reader, typeSpecifier, dataUtility.get());
   }
   else
   {
      isSuccess = true;
      ETG_TRACE_USR4(("PluginDataConverter::convertToEventDataUtility(), No data to be converted"));
   }

   ETG_TRACE_USR4(("PluginDataConverter::convertToEventDataUtility(), isSuccess : %d", isSuccess));
   return isSuccess;
}

/************************************************************************
*NAME        : convertToEventListDataUtility
*DESCRIPTION : Design section 8.2.3.4 - Conversion from D-Bus variant to EventListDataUtility
************************************************************************/
bool PluginDataConverter::convertToEventListDataUtility(::asf::dbus::DBusVariant variant, boost::shared_ptr<EventListDataUtility>& listDataUtility)
{
   ETG_TRACE_USR4(("PluginDataConverter::convertToEventListDataUtility()"));
   bool isSuccess = false;

   std::string typeSpecifier;
   variant.getSignature(typeSpecifier);
   ETG_TRACE_USR4(("PluginDataConverter::convertToEventListDataUtility(), Type specifier : %s", typeSpecifier.c_str()));

   if(0 < typeSpecifier.length())
   {
      if(DBUS_TYPE_ARRAY == typeSpecifier.at(0))
      {
         listDataUtility = boost::shared_ptr<EventListDataUtility>(EventListDataUtility::newEventListDataUtility());
         if(NULL != listDataUtility)
         {
            std::string arraySignature = typeSpecifier.substr(1);
            VariantReader reader(variant);

            isSuccess = reader.open_array();
            while((true == isSuccess) && (false == reader.at_end()))
            {
               EventDataUtility* dataUtility = EventDataUtility::newEventDataUtility();
               listDataUtility->addEventListData(dataUtility);
               isSuccess = readStructData(&reader, arraySignature, dataUtility);
            }
            if(true == isSuccess)
            {
               reader.close_array();
            }
         }
      }
   }
   else
   {
      isSuccess = true;
      ETG_TRACE_USR4(("PluginDataConverter::convertToEventListDataUtility(), No data to be converted"));
   }

   ETG_TRACE_USR4(("PluginDataConverter::convertToEventListDataUtility(), isSuccess : %d", isSuccess));
   return isSuccess;
}

/************************************************************************
*NAME        : getArraySignature
*DESCRIPTION : Design section 8.2.3.3 - Conversion from EventListDataUtility to D-Bus variant
************************************************************************/
bool PluginDataConverter::getArraySignature(const EventDataUtility* dataUtility, std::string& typeSpecifier)
{
   ETG_TRACE_USR4(("PluginDataConverter::getArraySignature()"));
   bool isSuccess = false;

   if(NULL != dataUtility)
   {
      const std::vector<EventDataItem*>& dataItems = dataUtility->getData();
      if(0 < dataItems.size())
      {
         isSuccess = true;
         typeSpecifier.append(DBUS_STRUCT_BEGIN_CHAR_AS_STRING);

         int index = 0;
         for(; (index < dataItems.size()) && (true == isSuccess) && (typeSpecifier.length() < (DBUS_MAXIMUM_SIGNATURE_LENGTH - 1)); index++)
         {
            EventDataItem* dataItem = dataItems.at(index);
            if(NULL != dataItem)
            {
               const EventDataItem::Data& data = dataItem->getData();
               switch (data._type)
               {
                  case EventDataItem::BOOL:
                  {
                     typeSpecifier.append(DBUS_TYPE_BOOLEAN_AS_STRING);
                     break;
                  }

                  case EventDataItem::UINT8:
                  {
                     typeSpecifier.append(DBUS_TYPE_BYTE_AS_STRING);
                     break;
                  }

                  case EventDataItem::UINT16:
                  {
                     typeSpecifier.append(DBUS_TYPE_UINT16_AS_STRING);
                     break;
                  }

                  case EventDataItem::UINT32:
                  {
                     typeSpecifier.append(DBUS_TYPE_UINT32_AS_STRING);
                     break;
                  }

                  case EventDataItem::UINT64:
                  {
                     typeSpecifier.append(DBUS_TYPE_UINT64_AS_STRING);
                     break;
                  }

                  case EventDataItem::DOUBLE:
                  {
                     typeSpecifier.append(DBUS_TYPE_DOUBLE_AS_STRING);
                     break;
                  }

                  case EventDataItem::STRING:
                  {
                     if(NULL != data._value._stringValue)
                     {
                        typeSpecifier.append(DBUS_TYPE_STRING_AS_STRING);
                     }
                     else
                     {
                        isSuccess = false;
                     }
                     break;
                  }

                  case EventDataItem::BYTEBUFFER:
                  {
                     if(NULL != data._value._byteBufferValue)
                     {
                        typeSpecifier.append(DBUS_TYPE_ARRAY_AS_STRING);
                        typeSpecifier.append(DBUS_TYPE_BYTE_AS_STRING);
                     }
                     else
                     {
                        isSuccess = false;
                     }
                     break;
                  }

                  default:
                     isSuccess = false;
                     break;
               }
            }
         }
         if((index == dataItems.size()) && (true == isSuccess))
         {
            typeSpecifier.append(DBUS_STRUCT_END_CHAR_AS_STRING);
         }
         else
         {
            isSuccess = false;
         }
         ETG_TRACE_USR4(("PluginDataConverter::getArraySignature(), Type specifier : %s", typeSpecifier.c_str()));
      }
      else
      {
         ETG_TRACE_USR4(("PluginDataConverter::getArraySignature(), No items in EventDataUtility"));
      }
   }
   return isSuccess;
}

/************************************************************************
*NAME        : writeArrayStructData
*DESCRIPTION : Design section 8.2.3.3 - Conversion from EventListDataUtility to D-Bus variant
************************************************************************/
bool PluginDataConverter::writeArrayStructData(const EventDataUtility* dataUtility, VariantWriter *writer, const std::string& typeSpecifier)
{
   ETG_TRACE_USR4(("PluginDataConverter::writeArrayStructData(), Type specifier : %s", typeSpecifier.c_str()));
   bool isSuccess = false;

   size_t signatureLength = typeSpecifier.length();
   if((NULL != dataUtility) && (NULL != writer) && (DBUS_MAXIMUM_SIGNATURE_LENGTH >= signatureLength))
   {
      int typeIndex = 0;
      const std::vector<EventDataItem*>& dataItems = dataUtility->getData();

      if((0 < dataItems.size()) && (DBUS_STRUCT_BEGIN_CHAR == typeSpecifier.at(typeIndex)))
      {
         typeIndex++;
         isSuccess = writer->open_struct();

         for(int index = 0; (index < dataItems.size()) && (true == isSuccess); index++, typeIndex++)
         {
            isSuccess = false;
            EventDataItem* dataItem = dataItems.at(index);
            if(NULL != dataItem)
            {
               const EventDataItem::Data& data = dataItem->getData();
               char dataType = typeSpecifier.at(typeIndex);
               switch (data._type)
               {
                  case EventDataItem::BOOL:
                  {
                     isSuccess = (DBUS_TYPE_BOOLEAN == dataType) ? writer->insert_bool(data._value._boolValue) : false;
                     ETG_TRACE_USR4(("PluginDataConverter::writeArrayStructData(), Index : %d, bool value : %d", index, data._value._boolValue));
                     break;
                  }

                  case EventDataItem::UINT8:
                  {
                     isSuccess = (DBUS_TYPE_BYTE == dataType) ? writer->insert_uint8(data._value._uint8Value) : false;
                     ETG_TRACE_USR4(("PluginDataConverter::writeArrayStructData(), Index : %d, uint8 value : %d", index, data._value._uint8Value));
                     break;
                  }

                  case EventDataItem::UINT16:
                  {
                     isSuccess = (DBUS_TYPE_UINT16 == dataType) ? writer->insert_uint16(data._value._uint16Value) : false;
                     ETG_TRACE_USR4(("PluginDataConverter::writeArrayStructData(), Index : %d, uint16 value : %d", index, data._value._uint16Value));
                     break;
                  }

                  case EventDataItem::UINT32:
                  {
                     isSuccess = (DBUS_TYPE_UINT32 == dataType) ? writer->insert_uint32(data._value._uint32Value) : false;
                     ETG_TRACE_USR4(("PluginDataConverter::writeArrayStructData(), Index : %d, uint32 value : %d", index, data._value._uint32Value));
                     break;
                  }

                  case EventDataItem::UINT64:
                  {
                     isSuccess = (DBUS_TYPE_UINT64 == dataType) ? writer->insert_uint64(data._value._uint64Value) : false;
                     ETG_TRACE_USR4(("PluginDataConverter::writeArrayStructData(), Index : %d, uint64 value : %d", index, data._value._uint64Value));
                     break;
                  }

                  case EventDataItem::DOUBLE:
                  {
                     isSuccess = (DBUS_TYPE_DOUBLE == dataType) ? writer->insert_double(data._value._doubleValue) : false;
                     ETG_TRACE_USR4(("PluginDataConverter::writeArrayStructData(), Index : %d, double value : %f", index, data._value._doubleValue));
                     break;
                  }

                  case EventDataItem::STRING:
                  {
                     if(NULL != data._value._stringValue)
                     {
                        isSuccess = (DBUS_TYPE_STRING == dataType) ? writer->insert_string(*data._value._stringValue) : false;
                        ETG_TRACE_USR4(("PluginDataConverter::writeArrayStructData(), Index : %d", index));
                        ETG_TRACE_USR4(("PluginDataConverter::writeArrayStructData(), String value : %s", data._value._stringValue->c_str()));
                     }
                     break;
                  }

                  case EventDataItem::BYTEBUFFER:
                  {
                     if((NULL != data._value._byteBufferValue) &&
                        (DBUS_TYPE_ARRAY == dataType) &&
                        (DBUS_TYPE_BYTE == typeSpecifier.at(typeIndex + 1)))
                     {
                        isSuccess = writer->insert_array_uint8(*data._value._byteBufferValue);
                        ETG_TRACE_USR4(("PluginDataConverter::writeArrayStructData(), Index : %d, Size of byte buffer : %d", index, data._value._byteBufferValue->size()));
                        typeIndex++;
                     }
                     break;
                  }

                  default:
                     break;
               }
            }
         }
         if((true == isSuccess) && (DBUS_STRUCT_END_CHAR == typeSpecifier.at(typeIndex)))
         {
            isSuccess = writer->close_struct();
         }
         else
         {
            isSuccess = false;
         }
      }
   }
   ETG_TRACE_USR4(("PluginDataConverter::writeArrayStructData(), isSuccess : %d", isSuccess));
   return isSuccess;
}

/************************************************************************
*NAME        : writeStructData
*DESCRIPTION : Design section 8.2.3.1 - Conversion from EventDataUtility to D-Bus variant
************************************************************************/
bool PluginDataConverter::writeStructData(const EventDataUtility* dataUtility, VariantWriter *writer)
{
   ETG_TRACE_USR4(("PluginDataConverter::writeStructData()"));
   bool isSuccess = false;

   if((NULL != dataUtility) && (NULL != writer))
   {
      const std::vector<EventDataItem*>& dataItems = dataUtility->getData();
      if(0 < dataItems.size())
      {
         int index = 0;
         int signatureLength = 0;
         isSuccess = writer->open_struct();

         for(signatureLength++;
             (index < dataItems.size()) && (true == isSuccess) && (signatureLength < (DBUS_MAXIMUM_SIGNATURE_LENGTH - 1));
             index++, signatureLength++)
         {
            isSuccess = false;
            EventDataItem* dataItem = dataItems.at(index);
            if(NULL != dataItem)
            {
               const EventDataItem::Data& data = dataItem->getData();
               switch (data._type)
               {
                  case EventDataItem::BOOL:
                  {
                     bool value = data._value._boolValue;
                     ETG_TRACE_USR4(("PluginDataConverter::writeStructData(), Index : %d, bool value : %d", index, value));

                     isSuccess = writer->insert_bool(value);
                     break;
                  }

                  case EventDataItem::UINT8:
                  {
                     uint8 value = data._value._uint8Value;
                     ETG_TRACE_USR4(("PluginDataConverter::writeStructData(), Index : %d, uint8 value : %d", index, value));

                     isSuccess = writer->insert_uint8(value);
                     break;
                  }

                  case EventDataItem::UINT16:
                  {
                     uint16 value = data._value._uint16Value;
                     ETG_TRACE_USR4(("PluginDataConverter::writeStructData(), Index : %d, uint16 value : %d", index, value));

                     isSuccess = writer->insert_uint16(value);
                     break;
                  }

                  case EventDataItem::UINT32:
                  {
                     uint32 value = data._value._uint32Value;
                     ETG_TRACE_USR4(("PluginDataConverter::writeStructData(), Index : %d, uint32 value : %d", index, value));

                     isSuccess = writer->insert_uint32(value);
                     break;
                  }

                  case EventDataItem::UINT64:
                  {
                     uint64 value = data._value._uint64Value;
                     ETG_TRACE_USR4(("PluginDataConverter::writeStructData(), Index : %d, uint64 value : %d", index, value));

                     isSuccess = writer->insert_uint64(value);
                     break;
                  }

                  case EventDataItem::DOUBLE:
                  {
                     double value = data._value._doubleValue;
                     ETG_TRACE_USR4(("PluginDataConverter::writeStructData(), Index : %d, double value : %f", index, value));

                     isSuccess = writer->insert_double(value);
                     break;
                  }


                  case EventDataItem::STRING:
                  {
                     ETG_TRACE_USR4(("PluginDataConverter::writeStructData(), Index : %d", index));
                     if(NULL != data._value._stringValue)
                     {
                        ETG_TRACE_USR4(("PluginDataConverter::writeStructData(), String value : %s", data._value._stringValue->c_str()));

                        isSuccess = writer->insert_string(*data._value._stringValue);

                     }
                     break;
                  }

                  case EventDataItem::BYTEBUFFER:
                  {
                     ETG_TRACE_USR4(("PluginDataConverter::writeStructData(), Index : %d", index));
                     if(NULL != data._value._byteBufferValue)
                     {
                        ETG_TRACE_USR4(("PluginDataConverter::writeStructData(), Size of byte buffer : %d", data._value._byteBufferValue->size()));

                        isSuccess = writer->insert_array_uint8(*data._value._byteBufferValue);
                        signatureLength++;
                     }
                     break;
                  }

                  default:
                     break;
               }
            }
         }
         if((index == dataItems.size()) && (true == isSuccess))
         {
            isSuccess = writer->close_struct();
         }
         else
         {
            isSuccess = false;
            ETG_TRACE_USR4(("PluginDataConverter::writeStructData(), Error in data conversion"));
         }
      }
      else  // No data items added to EventDataUtility
      {
         isSuccess = true;
         ETG_TRACE_USR4(("PluginDataConverter::writeStructData(), No items in EventDataUtility"));
      }
   }
   ETG_TRACE_USR4(("PluginDataConverter::writeStructData(), isSuccess : %d", isSuccess));
   return isSuccess;
}

/************************************************************************
*NAME        : readStructData
*DESCRIPTION : Design section 8.2.3.2 - Conversion from D-Bus variant to EventDataUtility
*              Design section 8.2.3.4 - Conversion from D-Bus variant to EventListDataUtility
************************************************************************/
bool PluginDataConverter::readStructData(VariantReader *reader, const std::string& typeSpecifier, EventDataUtility* dataUtility)
{
   ETG_TRACE_USR4(("PluginDataConverter::readStructData()"));
   bool isSuccess = false;

   int typeIndex = 0;
   if((NULL != dataUtility) && (NULL != reader) && (DBUS_STRUCT_BEGIN_CHAR == typeSpecifier.at(typeIndex)))
   {
      typeIndex++;
      isSuccess = reader->open_struct();

      for(;(DBUS_STRUCT_END_CHAR != typeSpecifier.at(typeIndex)) && (true == isSuccess); typeIndex++)
      {
         switch(typeSpecifier.at(typeIndex))
         {
            case DBUS_TYPE_BOOLEAN:
            {
               bool value = false;
               isSuccess = reader->get_bool(&value) ? (dataUtility->addEventData(value), true) : false;
               ETG_TRACE_USR4(("PluginDataConverter::readStructData(), Index : %d, bool value : %d", typeIndex, value));
               break;
            }

            case DBUS_TYPE_BYTE:
            {
               uint8 value = 0;
               isSuccess = reader->get_uint8(&value) ? (dataUtility->addEventData(value), true) : false;
               ETG_TRACE_USR4(("PluginDataConverter::readStructData(), Index : %d, uint8 value : %d", typeIndex, value));
               break;
            }

            case DBUS_TYPE_UINT16:
            {
               uint16 value = 0;
               isSuccess = reader->get_uint16(&value) ? (dataUtility->addEventData(value), true) : false;
               ETG_TRACE_USR4(("PluginDataConverter::readStructData(), Index : %d, uint16 value : %d", typeIndex, value));
               break;
            }

            case DBUS_TYPE_UINT32:
            {
               uint32 value = 0;
               isSuccess = reader->get_uint32(&value) ? (dataUtility->addEventData(value), true) : false;
               ETG_TRACE_USR4(("PluginDataConverter::readStructData(), Index : %d, uint32 value : %d", typeIndex, value));
               break;
            }

            case DBUS_TYPE_UINT64:
            {
               uint64 value = 0;
               isSuccess = reader->get_uint64(&value) ? (dataUtility->addEventData(value), true) : false;
               ETG_TRACE_USR4(("PluginDataConverter::readStructData(), Index : %d, uint64 value : %d", typeIndex, value));
               break;
            }

            case DBUS_TYPE_DOUBLE:
            {
               double value = 0;
               isSuccess = reader->get_double(&value) ? (dataUtility->addEventData(value), true) : false;
               ETG_TRACE_USR4(("PluginDataConverter::readStructData(), Index : %d, double value : %f", typeIndex, value));
               break;
            }

            case DBUS_TYPE_STRING:
            {
               string value;
               isSuccess = reader->get_string(&value) ? (dataUtility->addEventData(value), true) : false;
               ETG_TRACE_USR4(("PluginDataConverter::readStructData(), Index : %d, String value : %s", typeIndex, value.c_str()));
               break;
            }

            case DBUS_TYPE_ARRAY:
            {
               typeIndex++;
               if(DBUS_TYPE_BYTE == typeSpecifier.at(typeIndex))
               {
                  std::vector<uint8> value;
                  isSuccess = reader->get_array_uint8(&value) ? (dataUtility->addEventData(value), true) : false;
                  ETG_TRACE_USR4(("PluginDataConverter::readStructData(), Index : %d, Byte buffer size : %d", typeIndex, value.size()));
               }
               break;
            }

            default:
               isSuccess = false;
               break;
         } // end of switch
      } // end of for
      if(true == isSuccess)
      {
         reader->close_struct();
      }
   } // end of condition check for begin char

   ETG_TRACE_USR4(("PluginDataConverter::readStructData(), isSuccess : %d", isSuccess));
   return isSuccess;
}
