/****************************************************************************
* Copyright (C) Robert Bosch Car Multimedia GmbH, 2018
* This software is property of Robert Bosch GmbH. Unauthorized
* duplication and disclosure to third parties is prohibited.
*
* FILE                : VariantWriter.cpp
* COMPONENT Name      : dbus_variant_helper
* DESCRIPTION         : Wrapper class for MessageIterWriter. Writes data into variant
* AUTHOR              : Preethi Alagappan
* Date                : 10.09.2018
* Revision History    : 0.1
* Date 10.09.2018     : Initial version
* Revision History    : 0.2
* Date 25.10.2018     : Dynamic memory allocation for sub iterator
*                     : Modified method signature for container related functions
*                     : Added new method insert_array_uint8() and insert_double()
****************************************************************************/

#include "VariantWriter.h"
#include "../plugin_trace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS  TR_CLASS_APPUTILITIES_DBUSVARIANTHANDLER
#include "trcGenProj/Header/VariantWriter.cpp.trc.h"
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN
/************************************************************************
*NAME        : VariantWriter
*DESCRIPTION : Design section 8.2.2.2 - VariantWriter
************************************************************************/
VariantWriter::VariantWriter(::asf::dbus::DBusVariant& variant)
{
   ETG_TRACE_USR4(("VariantWriter::VariantWriter()"));
   DBusMessageIter *writeIter = variant.getWriteIterator();
   MessageIterWriter *iter = new MessageIterWriter(writeIter);
   stackIter.push(iter);
}

/************************************************************************
*NAME        : ~VariantWriter
*DESCRIPTION : Design section 8.2.2.2 - VariantWriter
************************************************************************/
VariantWriter::~VariantWriter()
{
    ETG_TRACE_USR4(("VariantWriter::~VariantWriter()"));
	while(false == stackIter.empty())
	{
		MessageIterWriter* iter = stackIter.top();
		stackIter.pop();
		if(NULL != iter)
		{
			delete iter;
		}
	}
}

/************************************************************************
*NAME        : insert_bool
*DESCRIPTION : Design section 8.2.2.2 - VariantWriter
************************************************************************/
bool VariantWriter::insert_bool(const bool& value)
{
    ETG_TRACE_USR4(("VariantWriter::insert_bool()"));
	bool retVal = false;
	MessageIterWriter* iter = stackIter.top();
	if(NULL != iter)
	{
		retVal = iter->append_bool(value);
	}
	return retVal;
}

/************************************************************************
*NAME        : insert_uint8
*DESCRIPTION : Design section 8.2.2.2 - VariantWriter
************************************************************************/
bool VariantWriter::insert_uint8(const uint8& value)
{
    ETG_TRACE_USR4(("VariantWriter::insert_uint8()"));
	bool retVal = false;
	MessageIterWriter* iter = stackIter.top();
	if(NULL != iter)
	{
		retVal = iter->append_uint8(value);
	}
	return retVal;
}

/************************************************************************
*NAME        : insert_uint16
*DESCRIPTION : Design section 8.2.2.2 - VariantWriter
************************************************************************/
bool VariantWriter::insert_uint16(const uint16& value)
{
    ETG_TRACE_USR4(("VariantWriter::insert_uint16()"));
	bool retVal = false;
	MessageIterWriter* iter = stackIter.top();
	if(NULL != iter)
	{
		retVal = iter->append_uint16(value);
	}
	return retVal;
}

/************************************************************************
*NAME        : insert_uint32
*DESCRIPTION : Design section 8.2.2.2 - VariantWriter
************************************************************************/
bool VariantWriter::insert_uint32(const uint32& value)
{
   ETG_TRACE_USR4(("VariantWriter::insert_uint32()"));
	bool retVal = false;
	MessageIterWriter* iter = stackIter.top();
	if(NULL != iter)
	{
		retVal = iter->append_uint32(value);
	}
	return retVal;
}

/************************************************************************
*NAME        : insert_uint64
*DESCRIPTION : Design section 8.2.2.2 - VariantWriter
************************************************************************/
bool VariantWriter::insert_uint64(const uint64& value)
{
   ETG_TRACE_USR4(("VariantWriter::insert_uint64()"));
   bool retVal = false;
   MessageIterWriter* iter = stackIter.top();
   if(NULL != iter)
   {
      retVal = iter->append_uint64(value);
   }
   return retVal;
}

/************************************************************************
*NAME        : insert_double
*DESCRIPTION : Design section 8.2.2.2 - VariantWriter
************************************************************************/
bool VariantWriter::insert_double(const double& value)
{
   ETG_TRACE_USR4(("VariantWriter::insert_double()"));
   bool retVal = false;
   MessageIterWriter* iter = stackIter.top();
   if(NULL != iter)
   {
      retVal = iter->append_double(value);
   }
   return retVal;
}

/************************************************************************
*NAME        : insert_string
*DESCRIPTION : Design section 8.2.2.2 - VariantWriter
************************************************************************/
bool VariantWriter::insert_string(const std::string& value)
{
    ETG_TRACE_USR4(("VariantWriter::insert_string()"));
	bool retVal = false;
	MessageIterWriter* iter = stackIter.top();
	if(NULL != iter)
	{
		retVal = iter->append_string(value);
	}
	return retVal;
}

/************************************************************************
*NAME        : insert_array_uint8
*DESCRIPTION : Design section 8.2.2.2 - VariantWriter
************************************************************************/
bool VariantWriter::insert_array_uint8(const uint8** values, int numElems)
{
    ETG_TRACE_USR4(("VariantWriter::insert_array_uint8()"));
	bool retVal = false;
	MessageIterWriter* iter = stackIter.top();
	if(NULL != iter)
	{
		retVal = iter->append_array_uint8(values, numElems);
	}
	return retVal;
}

/************************************************************************
*NAME        : insert_array_uint16
*DESCRIPTION : Design section 8.2.2.2 - VariantWriter
************************************************************************/
bool VariantWriter::insert_array_uint16(const uint16** values, int numElems)
{
   ETG_TRACE_USR4(("VariantWriter::insert_array_uint16()"));
   bool retVal = false;
   MessageIterWriter* iter = stackIter.top();
   if(NULL != iter)
   {
      retVal = iter->append_array_uint16(values, numElems);
   }
   return retVal;
}

/************************************************************************
*NAME        : insert_array_uint32
*DESCRIPTION : Design section 8.2.2.2 - VariantWriter
************************************************************************/
bool VariantWriter::insert_array_uint32(const uint32** values, int numElems)
{
    ETG_TRACE_USR4(("VariantWriter::insert_array_uint32()"));
	bool retVal = false;
	MessageIterWriter* iter = stackIter.top();
	if(NULL != iter)
	{
		retVal = iter->append_array_uint32(values, numElems);
	}
	return retVal;
}

/************************************************************************
*NAME        : insert_array_uint8
*DESCRIPTION : Design section 8.2.2.2 - VariantWriter
************************************************************************/
bool VariantWriter::insert_array_uint8(const std::vector<uint8>& values)
{
   ETG_TRACE_USR4(("VariantWriter::insert_array_uint8()"));
   bool retVal = false;
   MessageIterWriter* iter = stackIter.top();
   if(NULL != iter)
   {
      retVal = iter->append_array_uint8(values);
   }
   return retVal;
}

/************************************************************************
*NAME        : insert_array_uint16
*DESCRIPTION : Design section 3.1.1.2.2   VariantWriter
************************************************************************/
bool VariantWriter::insert_array_uint16(const std::vector<uint16>& values)
{
   ETG_TRACE_USR4(("VariantWriter::insert_array_uint16()"));
   bool retVal = false;
   MessageIterWriter* iter = stackIter.top();
   if(NULL != iter)
   {
      retVal = iter->append_array_uint16(values);
   }
   return retVal;
}

/************************************************************************
*NAME        : insert_array_uint32
*DESCRIPTION : Design section 8.2.2.2 - VariantWriter
************************************************************************/
bool VariantWriter::insert_array_uint32(const std::vector<uint32>& values)
{
   ETG_TRACE_USR4(("VariantWriter::insert_array_uint32()"));
   bool retVal = false;
   MessageIterWriter* iter = stackIter.top();
   if(NULL != iter)
   {
      retVal = iter->append_array_uint32(values);
   }
   return retVal;
}

/************************************************************************
*NAME        : insert_array_string
*DESCRIPTION : Design section 8.2.2.2 - VariantWriter
************************************************************************/
bool VariantWriter::insert_array_string(const std::vector<std::string>& values)
{
   ETG_TRACE_USR4(("VariantWriter::insert_array_string()"));
	bool retVal = false;
	MessageIterWriter* iter = stackIter.top();
	if(NULL != iter)
	{
		retVal = iter->append_array_string(values);
	}
	return retVal;
}

/************************************************************************
*NAME        : open_array
*DESCRIPTION : Design section 8.2.2.2 - VariantWriter
************************************************************************/
bool VariantWriter::open_array(const char* signature)
{
    ETG_TRACE_USR4(("VariantWriter::open_array(), Signature: %s", signature));
   bool retVal = false;
	MessageIterWriter* iter = stackIter.top();
	if(NULL != iter)
	{
		MessageIterWriter* subIter = new MessageIterWriter();
		if(NULL != subIter)
		{
		   retVal = iter->new_array(signature, subIter);
		   if(true == retVal)
		   {
		      stackIter.push(subIter);
		   }
		   else
		   {
		      delete subIter;
		      subIter = NULL;
		   }
		}
	}
	return retVal;
}

/************************************************************************
*NAME        : open_variant
*DESCRIPTION : Design section 8.2.2.2 - VariantWriter
************************************************************************/
bool VariantWriter::open_variant(const char* signature)
{
   ETG_TRACE_USR4(("VariantWriter::open_variant(), Signature: %s", signature));
   bool retVal = false;
	MessageIterWriter* iter = stackIter.top();
	if(NULL != iter)
   {
      MessageIterWriter* subIter = new MessageIterWriter();
      if(NULL != subIter)
      {
         retVal = iter->new_variant(signature, subIter);
         if(true == retVal)
         {
            stackIter.push(subIter);
         }
         else
         {
            delete subIter;
            subIter = NULL;
         }
      }
   }
   return retVal;
}

/************************************************************************
*NAME        : open_struct
*DESCRIPTION : Design section 8.2.2.2 - VariantWriter
************************************************************************/
bool VariantWriter::open_struct()
{
   ETG_TRACE_USR4(("VariantWriter::open_struct()"));
   bool retVal = false;
   MessageIterWriter* iter = stackIter.top();
   if(NULL != iter)
   {
      MessageIterWriter* subIter = new MessageIterWriter();
      if(NULL != subIter)
      {
         retVal = iter->new_struct(subIter);
         if(true == retVal)
         {
            stackIter.push(subIter);
         }
         else
         {
            delete subIter;
            subIter = NULL;
         }
      }
   }
   return retVal;
}

/************************************************************************
*NAME        : open_dict_entry
*DESCRIPTION : Design section 8.2.2.2 - VariantWriter
************************************************************************/
bool VariantWriter::open_dict_entry()
{
   ETG_TRACE_USR4(("VariantWriter::open_dict_entry()"));
   bool retVal = false;
   MessageIterWriter* iter = stackIter.top();
   if(NULL != iter)
   {
      MessageIterWriter* subIter = new MessageIterWriter();
      if(NULL != subIter)
      {
         retVal = iter->new_dict_entry(subIter);
         if(true == retVal)
         {
            stackIter.push(subIter);
         }
         else
         {
            delete subIter;
            subIter = NULL;
         }
      }
   }
   return retVal;
}

/************************************************************************
*NAME        : close_array
*DESCRIPTION : Design section 8.2.2.2 - VariantWriter
************************************************************************/
bool VariantWriter::close_array()
{
   ETG_TRACE_USR4(("VariantWriter::close_array()"));
	return close_container();
}

/************************************************************************
*NAME        : close_variant
*DESCRIPTION : Design section 8.2.2.2 - VariantWriter
************************************************************************/
bool VariantWriter::close_variant()
{
   ETG_TRACE_USR4(("VariantWriter::close_variant()"));
   return close_container();
}

/************************************************************************
*NAME        : close_struct
*DESCRIPTION : Design section 8.2.2.2 - VariantWriter
************************************************************************/
bool VariantWriter::close_struct()
{
   ETG_TRACE_USR4(("VariantWriter::close_struct()"));
   return close_container();
}

/************************************************************************
*NAME        : close_dict_entry
*DESCRIPTION : Design section 8.2.2.2 - VariantWriter
************************************************************************/
bool VariantWriter::close_dict_entry()
{
   ETG_TRACE_USR4(("VariantWriter::close_dict_entry()"));
   return close_container();
}

/************************************************************************
*NAME        : close_container
*DESCRIPTION : Design section 8.2.2.2 - VariantWriter
************************************************************************/	
bool VariantWriter::close_container()
{
   ETG_TRACE_USR4(("VariantWriter::close_container()"));
   bool retVal = false;

	MessageIterWriter* subIter = stackIter.top();
	stackIter.pop();

	MessageIterWriter* iter = stackIter.top();
	if(NULL != iter)
	{
		retVal = iter->delete_container(subIter);
	}
   if(NULL != subIter)
   {
      delete subIter;
      subIter = NULL;
   }
	return retVal;
}
