/****************************************************************************
* Copyright (C) Robert Bosch Car Multimedia GmbH, 2018
* This software is property of Robert Bosch GmbH. Unauthorized
* duplication and disclosure to third parties is prohibited.
*
* FILE                : MessageIterReader.cpp
* COMPONENT Name      : dbus_variant_helper
* DESCRIPTION         : Reads data from variant
* AUTHOR              : Preethi Alagappan
* Date                : 10.09.2018
* Revision History    : 0.1
* Date 10.09.2018     : Initial version
* Revision History    : 0.2
* Date 25.10.2018     : Moved dynamic memory allocation for sub iterator to VariantReader
*                     : Modified method signature for container related functions
*                     : Added new method get_array_uint8() and get_double()
****************************************************************************/

#include "MessageIterReader.h"
#include "../plugin_trace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS  TR_CLASS_APPUTILITIES_DBUSVARIANTHANDLER
#include "trcGenProj/Header/MessageIterReader.cpp.trc.h"
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN

/************************************************************************
*NAME        : MessageIterReader
*DESCRIPTION : Design section 8.2.2.1 - MessageIterReader
************************************************************************/
MessageIterReader::MessageIterReader()
{
	ETG_TRACE_USR4(("MessageIterReader::MessageIterReader()"));
	memset(&memberIter, 0, sizeof(DBusMessageIter));
	iter = &memberIter;
}

/************************************************************************
*NAME        : MessageIterReader
*DESCRIPTION : Design section 8.2.2.1 - MessageIterReader
************************************************************************/
MessageIterReader::MessageIterReader(DBusMessageIter *iter)
{
	ETG_TRACE_USR4(("MessageIterReader::MessageIterReader(DBusMessageIter *)"));
	memset(&memberIter, 0, sizeof(DBusMessageIter));
	this->iter = iter;
}

/************************************************************************
*NAME        : ~MessageIterReader
*DESCRIPTION : Design section 8.2.2.1 - MessageIterReader
************************************************************************/
MessageIterReader::~MessageIterReader()
{
	ETG_TRACE_USR4(("MessageIterReader::~MessageIterReader()"));
}

/************************************************************************
*NAME        : getIterator
*DESCRIPTION : Design section 8.2.2.1 - MessageIterReader
************************************************************************/
DBusMessageIter* MessageIterReader::getIterator()
{
	ETG_TRACE_USR4(("MessageIterReader::getIterator()"));
	return iter;
}

/************************************************************************
*NAME        : get_type
*DESCRIPTION : Design section 8.2.2.1 - MessageIterReader
************************************************************************/
int MessageIterReader::get_type()
{
   ETG_TRACE_USR4(("MessageIterReader::get_type()"));

	int type_id = dbus_message_iter_get_arg_type(iter);
	ETG_TRACE_USR4(("MessageIterReader::get_type(), Type id : %d", type_id));
	return type_id;
}

/************************************************************************
*NAME        : get_array_type
*DESCRIPTION : Design section 8.2.2.1 - MessageIterReader
************************************************************************/
int MessageIterReader::get_array_type()
{
   ETG_TRACE_USR4(("MessageIterReader::get_array_type()"));

   int type_id = dbus_message_iter_get_element_type(iter);
	ETG_TRACE_USR4(("MessageIterReader::get_array_type(), Type id : %d", type_id));
	return type_id;
}

/************************************************************************
*NAME        : at_end
*DESCRIPTION : Design section 8.2.2.1 - MessageIterReader
************************************************************************/
bool MessageIterReader::at_end()
{
	ETG_TRACE_USR4(("MessageIterReader::at_end()"));

	bool isEnd = (DBUS_TYPE_INVALID == get_type()) ? true : false;
	ETG_TRACE_USR4(("MessageIterReader::at_end(), End reached : %d", isEnd));
	return isEnd;
}

/************************************************************************
*NAME        : has_next
*DESCRIPTION : Design section 8.2.2.1 - MessageIterReader
************************************************************************/
bool MessageIterReader::has_next()
{
	ETG_TRACE_USR4(("MessageIterReader::has_next()"));

	bool retVal = dbus_message_iter_has_next(iter);
	ETG_TRACE_USR4(("MessageIterReader::has_next() Next data present : %d", retVal));
	return retVal;
}

/************************************************************************
*NAME        : goto_next
*DESCRIPTION : Design section 8.2.2.1 - MessageIterReader
************************************************************************/
bool MessageIterReader::goto_next()
{
	ETG_TRACE_USR4(("MessageIterReader::goto_next()"));
	return dbus_message_iter_next(iter);
}

/************************************************************************
*NAME        : get_bool
*DESCRIPTION : Design section 8.2.2.1 - MessageIterReader
************************************************************************/
bool MessageIterReader::get_bool(bool* value)
{
	ETG_TRACE_USR4(("MessageIterReader::get_bool()"));
	bool retVal = false;

	dbus_bool_t dbusValue = false;
	retVal = get_basic(DBUS_TYPE_BOOLEAN, &dbusValue);
	*value = static_cast<bool>(dbusValue);
	return retVal;
}

/************************************************************************
*NAME        : get_uint8
*DESCRIPTION : Design section 8.2.2.1 - MessageIterReader
************************************************************************/
bool MessageIterReader::get_uint8(uint8* value)
{
	ETG_TRACE_USR4(("MessageIterReader::get_uint8()"));
	return get_basic(DBUS_TYPE_BYTE, value);
}

/************************************************************************
*NAME        : get_uint16
*DESCRIPTION : Design section 8.2.2.1 - MessageIterReader
************************************************************************/
bool MessageIterReader::get_uint16(uint16* value)
{
	ETG_TRACE_USR4(("MessageIterReader::get_uint16()"));
	return get_basic(DBUS_TYPE_UINT16, value);
}

/************************************************************************
*NAME        : get_uint32
*DESCRIPTION : Design section 8.2.2.1 - MessageIterReader
************************************************************************/
bool MessageIterReader::get_uint32(uint32* value)
{
	ETG_TRACE_USR4(("MessageIterReader::get_uint32()"));
	return get_basic(DBUS_TYPE_UINT32, value);
}

/************************************************************************
*NAME        : get_uint64
*DESCRIPTION : Design section 8.2.2.1 - MessageIterReader
************************************************************************/
bool MessageIterReader::get_uint64(uint64* value)
{
   ETG_TRACE_USR4(("MessageIterReader::get_uint64()"));
   return get_basic(DBUS_TYPE_UINT64, value);
}

/************************************************************************
*NAME        : get_double
*DESCRIPTION : Design section 8.2.2.1 - MessageIterReader
************************************************************************/
bool MessageIterReader::get_double(double* value)
{
   ETG_TRACE_USR4(("MessageIterReader::get_double()"));
   return get_basic(DBUS_TYPE_DOUBLE, value);
}

/************************************************************************
*NAME        : get_string
*DESCRIPTION : Design section 8.2.2.1 - MessageIterReader
************************************************************************/
bool MessageIterReader::get_string(std::string* value)
{
	ETG_TRACE_USR4(("MessageIterReader::get_string()"));
	bool retVal = false;

	if(NULL != value)
	{
      char* tempValue = NULL;
      retVal = get_basic(DBUS_TYPE_STRING, &tempValue);
      if(true == retVal)
      {
         value->assign(tempValue);
      }
	}
	return retVal;
}

/************************************************************************
*NAME        : get_array_uint8
*DESCRIPTION : Design section 8.2.2.1 - MessageIterReader
************************************************************************/
bool MessageIterReader::get_array_uint8(uint8** values, int* numElems)
{
	ETG_TRACE_USR4(("MessageIterReader::get_array_uint8()"));
	bool retVal = false;

	MessageIterReader subIter;
	retVal = open_array(&subIter);
	retVal = retVal && subIter.get_array_basic(DBUS_TYPE_BYTE, values, numElems);
	return retVal;
}

/************************************************************************
*NAME        : get_array_uint16
*DESCRIPTION : Design section 8.2.2.1 - MessageIterReader
************************************************************************/
bool MessageIterReader::get_array_uint16(uint16** values, int* numElems)
{
	ETG_TRACE_USR4(("MessageIterReader::get_array_uint16()"));
	bool retVal = false;

   MessageIterReader subIter;
   retVal = open_array(&subIter);
   retVal = retVal && subIter.get_array_basic(DBUS_TYPE_UINT16, values, numElems);
   return retVal;
}

/************************************************************************
*NAME        : get_array_uint32
*DESCRIPTION : Design section 8.2.2.1 - MessageIterReader
************************************************************************/
bool MessageIterReader::get_array_uint32(uint32** values, int* numElems)
{
	ETG_TRACE_USR4(("MessageIterReader::get_array_uint32()"));
	bool retVal = false;

   MessageIterReader subIter;
   retVal = open_array(&subIter);
   retVal = retVal && subIter.get_array_basic(DBUS_TYPE_UINT32, values, numElems);
   return retVal;
}

/************************************************************************
*NAME        : get_array_uint8
*DESCRIPTION : Design section 8.2.2.1 - MessageIterReader
************************************************************************/
bool MessageIterReader::get_array_uint8(vector<uint8>* values)
{
   ETG_TRACE_USR4(("MessageIterReader::get_array_uint8()"));
   bool retVal = false;

   if(NULL != values)
   {
      MessageIterReader subIter;
      retVal = open_array(&subIter);
      while(retVal && (false == subIter.at_end()))
      {
         uint8 u8Value;
         retVal = subIter.get_uint8(&u8Value);
         if(true == retVal)
         {
            values->push_back(u8Value);
         }
         subIter.goto_next();
      }
   }
   return retVal;
}

/************************************************************************
*NAME        : get_array_uint16
*DESCRIPTION : Design section 8.2.2.1 - MessageIterReader
************************************************************************/
bool MessageIterReader::get_array_uint16(vector<uint16>* values)
{
   ETG_TRACE_USR4(("MessageIterReader::get_array_uint16()"));
   bool retVal = false;

   if(NULL != values)
   {
      MessageIterReader subIter;
      retVal = open_array(&subIter);
      while(retVal && (false == subIter.at_end()))
      {
         uint16 u16Value;
         retVal = subIter.get_uint16(&u16Value);
         if((true == retVal) && (NULL != values))
         {
            values->push_back(u16Value);
         }
         subIter.goto_next();
      }
   }
   return retVal;
}

/************************************************************************
*NAME        : get_array_uint32
*DESCRIPTION : Design section 8.2.2.1 - MessageIterReader
************************************************************************/
bool MessageIterReader::get_array_uint32(vector<uint32>* values)
{
   ETG_TRACE_USR4(("MessageIterReader::get_array_string()"));
   bool retVal = false;

   if(NULL != values)
   {
      MessageIterReader subIter;
      retVal = open_array(&subIter);
      while(retVal && (false == subIter.at_end()))
      {
         uint32 u32Value;
         retVal = subIter.get_uint32(&u32Value);
         if(true == retVal)
         {
            values->push_back(u32Value);
         }
         subIter.goto_next();
      }
   }
   return retVal;
}

/************************************************************************
*NAME        : get_array_string
*DESCRIPTION : Design section 8.2.2.1 - MessageIterReader
************************************************************************/
bool MessageIterReader::get_array_string(std::vector<std::string>* values)
{
	ETG_TRACE_USR4(("MessageIterReader::get_array_string()"));
	bool retVal = false;

   if(NULL != values)
   {
      MessageIterReader subIter;
      retVal = open_array(&subIter);
      while(retVal && (false == subIter.at_end()))
      {
         string strValue;
         retVal = subIter.get_string(&strValue);
         if(true == retVal)
         {
            values->push_back(strValue);
         }
         subIter.goto_next();
      }
   }
   return retVal;
}

/************************************************************************
*NAME        : open_array
*DESCRIPTION : Design section 8.2.2.1 - MessageIterReader
************************************************************************/
bool MessageIterReader::open_array(MessageIterReader* subIter)
{
	ETG_TRACE_USR4(("MessageIterReader::open_array()"));
	bool retVal = false;
	if((DBUS_TYPE_ARRAY == get_type()) && (NULL != subIter))
	{
	   recurse(subIter->getIterator());
	   retVal = true;
	}
	return retVal;
}

/************************************************************************
*NAME        : open_variant
*DESCRIPTION : Design section 8.2.2.1 - MessageIterReader
************************************************************************/
bool MessageIterReader::open_variant(MessageIterReader* subIter)
{
	ETG_TRACE_USR4(("MessageIterReader::open_variant()"));
	bool retVal = false;
	if((DBUS_TYPE_VARIANT == get_type()) && (NULL != subIter))
	{
	   recurse(subIter->getIterator());
	   retVal = true;
	}
	return retVal;
}

/************************************************************************
*NAME        : open_struct
*DESCRIPTION : Design section 8.2.2.1 - MessageIterReader
************************************************************************/
bool MessageIterReader::open_struct(MessageIterReader* subIter)
{
	ETG_TRACE_USR4(("MessageIterReader::open_struct()"));
	bool retVal = false;
	if((DBUS_TYPE_STRUCT == get_type()) && (NULL != subIter))
	{
	   recurse(subIter->getIterator());
	   retVal = true;
	}
	return retVal;
}

/************************************************************************
*NAME        : open_dict_entry
*DESCRIPTION : Design section 8.2.2.1 - MessageIterReader
************************************************************************/
bool MessageIterReader::open_dict_entry(MessageIterReader* subIter)
{
	ETG_TRACE_USR4(("MessageIterReader::open_dict_entry()"));
	bool retVal = false;
	if((DBUS_TYPE_DICT_ENTRY == get_type()) && (NULL != subIter))
	{
	   recurse(subIter->getIterator());
	   retVal = true;
	}
	return retVal;
}

/************************************************************************
*NAME        : is_array_type_fixed
*DESCRIPTION : Design section 8.2.2.1 - MessageIterReader
************************************************************************/
bool MessageIterReader::is_array_type_fixed(int type_id)
{
   ETG_TRACE_USR4(("MessageIterReader::is_array_type_fixed(), type id : %d", type_id));
   return dbus_type_is_fixed(type_id);
}

/************************************************************************
*NAME        : get_basic
*DESCRIPTION : Design section 8.2.2.1 - MessageIterReader
************************************************************************/
bool MessageIterReader::get_basic(int type_id, void* value)
{
	ETG_TRACE_USR4(("MessageIterReader::get_basic()"));
	bool retVal = false;
	if(type_id == get_type())
	{
		dbus_message_iter_get_basic(iter, value);
		retVal = true;
	}
	return retVal;
}

/************************************************************************
*NAME        : get_array_basic
*DESCRIPTION : Design section 8.2.2.1 - MessageIterReader
************************************************************************/
bool MessageIterReader::get_array_basic(int type_id, void* value, int* numElems)
{
	ETG_TRACE_USR4(("MessageIterReader::get_array_basic()"));
	bool retVal = false;
	if((type_id == get_type()) && (true == is_array_type_fixed(type_id)))
	{
		dbus_message_iter_get_fixed_array(iter, value, numElems);
		retVal = true;
	}
	return retVal;
}

/************************************************************************
*NAME        : recurse
*DESCRIPTION : Design section 8.2.2.1 - MessageIterReader
************************************************************************/
void MessageIterReader::recurse(DBusMessageIter* subIter)
{
	ETG_TRACE_USR4(("MessageIterReader::recurse()"));
	dbus_message_iter_recurse(iter, subIter);
}
