/************************************************************************
 * FILE:         scd.cpp
 * SW-COMPONENT:
 * DESCRIPTION:
 * AUTHOR:       K7/EFT21-Kuhn
 * COPYRIGHT:    (c) 2000 Robert Bosch GmbH, Hildesheim
 * HISTORY:
 * 30.11.2000 Rev.1.0 K7/EFT21-Kuhn
 *            initial revision
 * 25.05.2001 Rev.1.2 K7/EFG32-Kuhn
 *            some definitions changed to confirm with cca-paper
 * 28.05.2001 Rev.1.3 K7/EFG32-Kuhn
 *            debug zones added
 *            make use of the reg component instead of win32 api calls
 *	          to access the system registry
 * 19.06.2001 Rev.1.3 K7/EFG32-Kuhn
 *            osal_trace added
 * 09.03.2011 CM-AI/PJ-GM54 Creux
 *            new parameter for bGetAppIdFromServiceId()/scd_bGetAppIdFromServiceId()
 *
 *************************************************************************/

/*************************************************************************
| includes:
|   system- and project- includes
|*************************************************************************/

/*************************************************************************
| includes:
|   needed interfaces from external components
|*************************************************************************/
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"
#define SYSTEM_S_IMPORT_INTERFACE_REGISTRY
#include "system_pif.h"
#define REG_S_IMPORT_INTERFACE_GENERIC
#include "reg_if.h"
#define SCD_S_IMPORT_INTERFACE_GENERIC
#include "scd_if.h"

#include <map>


// define for enabling timing of some long running functions
// #define SCD_TRACE_TIMING

/* 
   SIMPLE_SCD uses cache inside the dev_registry instead of cache inside the each
   process, so we can use a single cache inside the registry for that instead of 
   complete parsing the registry at startup
*/

// define for manual enable these funtionality for testing
// #define SIMPLE_SCD

// check if OSAL provides these funtionality
#ifdef OSAL_C_S32_IOCTRL_REGLOOKUPAPPPATH
#ifdef OSAL_C_S32_IOCTRL_REGLOOKUPSRVPATH
#define SIMPLE_SCD
#endif
#endif

#ifdef SIMPLE_SCD
   // this is only for test purpose
   #ifndef OSAL_C_S32_IOCTRL_REGLOOKUPAPPPATH
   #define OSAL_C_S32_IOCTRL_REGLOOKUPAPPPATH (OSAL_C_S32_IOCTRL_REGDELETEVALUE + 1)
   #endif
   #ifndef OSAL_C_S32_IOCTRL_REGLOOKUPSRVPATH
   #define OSAL_C_S32_IOCTRL_REGLOOKUPSRVPATH (OSAL_C_S32_IOCTRL_REGDELETEVALUE + 2)
   #endif
#endif

/*************************************************************************
| defines and macros (scope: modul-local)
|*************************************************************************/
#define SCD_MAX_POOL_SIZE_VALUE_NAME      "MAXPOOLSIZE"

#define SCD_MAX_STRING_LENGTH              255
#ifdef SIMPLE_SCD
static OSAL_tIODescriptor fd = OSAL_ERROR;
#endif
static tBool bLookupIoControlAvailable = FALSE;

/*************************************************************************
| typedefs (scope: modul-local)
|*************************************************************************/

/**
 * This class holds the mapping between service id and application id.
 * It is used to speed up the registry access.
 * 
 * @author CM-DI/PJ-CF11 Wiedemann
 */
class scd_tclServiceAppMap
{
public:
   //-- gets the appID for a given serviceID - only one
   tBool bGetAppID(tU16 u16ServiceID, tU16& rfu16AppID);
   //-- gets the appIDs for a given serviceID - one or more.
   tBool bGetAppIdFromServiceId(tU16 u16ServiceId, tU32 *pu32NbrOfAppInfos, trScdAppInfo *prScdAppInfo, tBool bReadRegistry);

   //-- initialize the object (read information from registry)
   tBool bInit();

   void vFree();

   //-- request if initialized
   tBool bIsInitialized() const {return bInitialized;};

   //-- get singleton
   static scd_tclServiceAppMap& rfoGetServiceAppMap();

private:  
   //-- constructor
   scd_tclServiceAppMap()
      : bInitialized(FALSE)
      , poServiceAppMap(NULL)
      , m_hSem(OSAL_C_INVALID_HANDLE)
      , m_u32ClientCntr(0)
   {
      m_sSemName[0] = 0;
   }

   tBool bReadData(); //lint !e1704 Constructor has private access specification -> use signleton !

   //-- some constants for this class
   enum
   {
      C_SEM_NAME_SIZE = 32
   };

   tBool bInitialized;
   typedef std::multimap<tU16, tU16>  serviceMap;
   serviceMap * poServiceAppMap;
   OSAL_tSemHandle m_hSem;            //-- semaphore handle to synchronize data access
   tU32 m_u32ClientCntr;              //-- Counts how many clients called the initialization
   tChar m_sSemName[C_SEM_NAME_SIZE]; //-- semaphore name
};

tBool scd_tclServiceAppMap::bInit()
{
   tBool bResult = TRUE;
   tChar sTempSemName[C_SEM_NAME_SIZE]; //-- temporary semaphore name
   sTempSemName[0] = 0;

   if(m_hSem == OSAL_C_INVALID_HANDLE)
   {
      poServiceAppMap = OSAL_NEW serviceMap;

      if (poServiceAppMap == NULL)
      {
         scd_vTraceMsg(TR_LEVEL_COMPONENT,"scd_tclServiceAppMap::bInit: create service/app map failed");
         return FALSE;
      }

      //-- create system unique semaphore name (each binary will create its own
      //   semaphore)
      if (OSALUTIL_s32SaveNPrintFormat(sTempSemName, 
                                       C_SEM_NAME_SIZE, 
                                       "S%x%p", 
                                       OSAL_ProcessWhoAmI(), 
                                       this) != OSAL_ERROR)
      {
        //-- create semaphore (if semaphore can not be created this object simply
        //   does not manage any entries)
        if(OSAL_OK == OSAL_s32SemaphoreCreate( sTempSemName, &m_hSem, 0))
        {
           //-- Semaphore created
           (void)OSALUTIL_szSaveStringNCopy(m_sSemName, sTempSemName, sizeof(m_sSemName));
           scd_vTraceMsg(TR_LEVEL_COMPONENT,"scd_tclServiceAppMap::bInit: create Semaphore %s", m_sSemName);

           if (!bLookupIoControlAvailable)
           {
              //-- read all application entries from registry and save them 
              //   for further access
              (tVoid)bReadData();
           }

           bInitialized = TRUE;

           //-- post semaphore
           (void)OSAL_s32SemaphorePost(m_hSem);
        }
        else
        {
           //-- suppose two threads enter the check "m_hSem == OSAL_C_INVALID_HANDLE"
           //   at the same time and both think they have to create the semaphore.
           //   Then, both try to create the semaphore but only one Thread will succeed.
           //   Both Threads are serialized at the create semaphore call but only the
           //   first one will succeed. The other one can check m_hSem for a second Time
           //   if it is now valid, then the function can return success. Otherwise the
           //   initialisation failed
           if(m_hSem != OSAL_C_INVALID_HANDLE)
           {
              //-- Wait and Post semaphore to ensure, the semaphore name to be copied
              (void)OSAL_s32SemaphoreWait(m_hSem, OSAL_C_TIMEOUT_FOREVER);
              (void)OSAL_s32SemaphorePost(m_hSem);
           }
           else
           {
              bResult = FALSE;
           }
        }
      }
      else
      {
        NORMAL_M_ASSERT_ALWAYS();
        bResult = FALSE;
      }
   }
   else
   {
      //-- Wait and Post semaphore to ensure, the semaphore name to be copied
      (void)OSAL_s32SemaphoreWait(m_hSem, OSAL_C_TIMEOUT_FOREVER);
      (void)OSAL_s32SemaphorePost(m_hSem);
   }

   if(bResult == FALSE)
   {
      scd_vTraceMsg(TR_LEVEL_ERRORS,"scd_tclServiceAppMap::bInit: Semaphore %s creation failed", sTempSemName);
   }

   return bResult;
}

/**
 * read serviceID/appID mapping
 *
 * @return TRUE  if initialization succeeded otherwise FALSE
 *
 * @author CM-DI/PJ-CF11 Wiedemann
 */
tBool scd_tclServiceAppMap::bReadData()
{
   // read all service ids and save them with the belonging AppID

   // think positive, the initialisation will not fail
   tBool bResult = TRUE;

   // iterate processes
   if(poServiceAppMap)
   {
      reg_tclProcessIterator oProcIter;
      while(oProcIter.bIsDone() == FALSE)
      {
         reg_tclProcess oProc;
         oProc = oProcIter.oItem();

         if(oProc.bIsValid())
         {
            // iterate application
            reg_tclApplicationIterator oAppIter(oProc);
            while(oAppIter.bIsDone() == FALSE)
            {
               reg_tclApplication oApp;
               oApp = oAppIter.oItem();
               if(oApp.bIsValid())
               {
                  // iterate all service entries
                  reg_tclServiceIterator oSvcIter(oApp);
                  tU16 u16AppID;
                  tBool bValidAppID = oApp.bGetID(&u16AppID);

                  while(    (bValidAppID == TRUE) 
                     && (oSvcIter.bIsDone() == FALSE) )
                  {
                     reg_tclService oSvc;
                     oSvc = oSvcIter.oItem();
                     if(oSvc.bIsValid())
                     {
                        tU32 u32SvcID;
                        if(oSvc.bGetID(&u32SvcID) == TRUE)
                        {
                           // add pair of service id and app id to map
                           try
                           {
                              (tVoid)poServiceAppMap->insert(std::pair<tU16, tU16>((tU16)u32SvcID, u16AppID));
                           }
                           catch (...)
                           {
                              bResult = FALSE;
                              scd_vTraceMsg(TR_LEVEL_ERRORS,"scd_tclServiceAppMap::bReadData: can't insert service id to map");
                           }
                           
                        }/*if*/
                        else
                        {
                           scd_vTraceMsg(TR_LEVEL_ERRORS,"scd_tclServiceAppMap::bReadData: can't retrieve service id for AppId 0x%04x.", u16AppID);
                        }/*else*/
                     }/*if*/
                     else
                     {
                        scd_vTraceMsg(TR_LEVEL_ERRORS,"scd_tclServiceAppMap::bReadData: service key invalid for AppId 0x%04x.", u16AppID);
                     }/*else*/
                     oSvcIter.vNext();
                  }/*while*/
               }/*if*/
               else
               {
                  scd_vTraceMsg(TR_LEVEL_ERRORS,"scd_tclServiceAppMap::bReadData: app key invalid");
               }/*else*/
               oAppIter.vNext();
            }/*while*/
         }/*if*/
         else
         {
            scd_vTraceMsg(TR_LEVEL_ERRORS,"scd_tclServiceAppMap::bReadData: process key invalid");
         }/*else*/
         oProcIter.vNext();
      }/*while*/
   }
   else
   {
      scd_vTraceMsg(TR_LEVEL_ERRORS,"scd_tclServiceAppMap::bReadData: ApplicationMap not initialized");
      bResult = FALSE;
   }

   return bResult;
}


/**
 * gets the appID for a given serviceID. Note: this method works only for the 
 * next best appID. If you expect to find more than one AppID for your service
 * you cannot use this method but bGetAppIdFromServiceId().
 *
 * @param  u16ServiceID      requested service
 * @param  rfu16AppID        return value for AppID
 * @return TRUE              if AppID was found otherwise FALSE
 *
 * @author CM-DI/PJ-CF11 Wiedemann
 */
tBool scd_tclServiceAppMap::bGetAppID(tU16 u16ServiceID, tU16& rfu16AppID)
{
   tBool bResult = FALSE;

   // successfully initialized try to find the application entry
   if(bIsInitialized() && poServiceAppMap)
   {
      // get exclusive access
      (void)OSAL_s32SemaphoreWait(m_hSem, OSAL_C_TIMEOUT_FOREVER);

      // find entry
      serviceMap::const_iterator oIterator = poServiceAppMap->find(u16ServiceID);
   
      if( oIterator != poServiceAppMap->end())
      {
         // if entry found return it
         rfu16AppID = oIterator->second;
         bResult = TRUE;
      }
      else
      {
         if (!bLookupIoControlAvailable)
         {
            // if the entry was not found in the map re-read the map
            poServiceAppMap->clear();
            (tVoid)bReadData();
            // try again to find the entry
            oIterator = poServiceAppMap->find(u16ServiceID);
   
            if(oIterator != poServiceAppMap->end())
            {
               rfu16AppID = oIterator->second;
               bResult = TRUE;
            }
         }
      }

      // release exclusive access
      (void)OSAL_s32SemaphorePost(m_hSem);
   } else {
      FATAL_M_ASSERT_ALWAYS();
   }

   // return search result
   return bResult;
}

/**
 * gets the appIDs for a given serviceID.
 *
 * We are searching the map for at max pu32NbrOfAppInfos mapping for the requested u16ServiceId.
 *  bReadRegistry  pu32NbrOfAppInfos  Description
 *   TRUE           >  1              read registry/update the map, search in the map
 *   TRUE           == 1              search in the map, if not in the map: read registry/update the map and search again in the map
 *   FALSE          all values        search in the map
 *
 * @param  u16ServiceID        [in]  requested service
 * @param  pu32NbrOfAppInfos   [in]  max number of ScdAppInfo to be stored in prScdAppInfo, [out] real number of ScdAppInfo stored in prScdAppInfo
 * @param  prScdAppInfo        [out] pointer where to store ScdAppInfo
 * @param  bReadRegistry       [in]  boolean to indicates to read the registry/update the map or not
 * 
 * @return FALSE               if parameters were incorrect or searching was not possible, no out-parameters were modified
 * @return TRUE                search was executed and out-parameters have been updated - If the number of elements found is less than the mex requested, only the found number part of prScdAppInfo have been updated
 *
 * @author CM-AI/PJ-GM54 Creux
 * @date   02.03.2011
 *         Initial version
 * @date   09.03.2011
 *         add new parameter bReadRegistry
 */
tBool scd_tclServiceAppMap::bGetAppIdFromServiceId (tU16 u16ServiceId, tU32 *pu32NbrOfAppInfos, trScdAppInfo *prScdAppInfo, tBool bReadRegistry)
{
   tBool bResult = FALSE;

   //first of all, check if the pointers pu32NbrOfAppInfos and prScdAppInfo are legal
   if ( (prScdAppInfo == NULL) ||
        (pu32NbrOfAppInfos == NULL) )
   {
      return bResult;
   }

   // successfully initialized try to find the application entry
   if ( (bIsInitialized()) &&
        (poServiceAppMap != NULL) )
   {
      // get exclusive access
      (void)OSAL_s32SemaphoreWait(m_hSem, OSAL_C_TIMEOUT_FOREVER);

     if (!bLookupIoControlAvailable)
     {
        // shall we read the registry/update the map
        if ( (bReadRegistry) &&
              (*pu32NbrOfAppInfos > 1) )
        {
            poServiceAppMap->clear();
            (tVoid)bReadData();
        }
     }

      tU32 u32ItemsFoundCount = 0;
      trScdAppInfo *prAppInfo = prScdAppInfo;
      // search entry
      serviceMap::const_iterator oIterator = poServiceAppMap->find(u16ServiceId);
      while ( (oIterator != poServiceAppMap->end()) &&
              (oIterator->first == u16ServiceId)  &&
              (u32ItemsFoundCount < *pu32NbrOfAppInfos) )
      {
         prAppInfo->u16AppId = oIterator->second;
         ++ prAppInfo;
         ++ u32ItemsFoundCount;
         ++ oIterator;
      }

     if ( (u32ItemsFoundCount == 0) &&
           (bReadRegistry)           &&
           (*pu32NbrOfAppInfos == 1) )
     {
         if (!bLookupIoControlAvailable)
         {
            // if the single entry was not found in the map re-read the map
            poServiceAppMap->clear();
            (tVoid)bReadData();
         }

         u32ItemsFoundCount = 0;
         prAppInfo = prScdAppInfo;
         // search single entry again
         oIterator = poServiceAppMap->find(u16ServiceId);
         if (oIterator != poServiceAppMap->end())
         {
            prAppInfo->u16AppId = oIterator->second;
            ++ prAppInfo;
            ++ u32ItemsFoundCount;
         }
     }
      bResult = TRUE;
      * pu32NbrOfAppInfos = u32ItemsFoundCount;

      // release exclusive access
      (void)OSAL_s32SemaphorePost(m_hSem);
   } else {
      FATAL_M_ASSERT_ALWAYS();
   }

   // return search result
   return bResult;
}

/**
 * get singleton object
 *
 * @return   reference to access object
 *
 * @author CM-DI/PJ-CF11 Wiedemann
 */
scd_tclServiceAppMap& scd_tclServiceAppMap::rfoGetServiceAppMap()
{
   static scd_tclServiceAppMap oTheServiceMap;
   return oTheServiceMap;
}


void scd_tclServiceAppMap::vFree()
{
   OSAL_DELETE poServiceAppMap;
   poServiceAppMap = NULL;

   //-- check for valid semaphore
   if(m_hSem != OSAL_C_INVALID_HANDLE)
   {
      //-- close and delete semaphore
      (void)OSAL_s32SemaphoreClose(m_hSem);
      (void)OSAL_s32SemaphoreDelete(m_sSemName);
      m_sSemName[0]=0;
   }
}

/*---------------------------------------------------------------------*//**
 * Class definition and implementation. The intention of this class is to
 * manage reg_tclApplication objects for a faster access to the registry.
 * The problem is, that the registry keys are organized in a data tree
 * with the application specific data somewhere down that tree.
 * To access application specific data the applications identify the
 * applications root by their AppID and give the wanted registry key as a
 * relative path to their root. To prevent a search to the application root
 * path every time an application requests a registry key the entry points
 * are stored in an object of this class. So the applications entry point
 * has to be searched only once via registry. Note: Searching the registry
 * is very inperformant since their are lots of device open / close
 * necessary.
 *
 * @author  CM-DI/ESP4-Wiedemann
 * @date    16.11.06
 *//*---------------------------------------------------------------------*/
class scd_tclAppRegHandleMngr
{
public:
   //-- default constructor
   scd_tclAppRegHandleMngr();

   //-- destructor
   ~scd_tclAppRegHandleMngr();

   //-- method to find and get the entry point for a certain AppID
   reg_tclApplication* poGetEntry(tU16 u16AppID);

   //-- method to add a new entry to the manager. The manager uses the given
   //   entry point as a prototype and adds a clone of it to its list
   reg_tclApplication* poAddClone(tU16 u16AppID, const reg_tclApplication& roNewEntry);

   tBool bInit();
   tBool bIsInitialized() const {return ((m_u32ClientCntr > 0)? TRUE : FALSE);};
   //-- removes all entries of the manged list
   void                vFree(tBool bLastCall = FALSE);

private:
   //-- read all application entry points
   void                vReadAppEntries();
   reg_tclApplication* poIntAddClone(tU16 u16AppID, const reg_tclApplication& roNewEntry);

   //avoid copy
   scd_tclAppRegHandleMngr(const scd_tclAppRegHandleMngr &corfoAppRegHandleMngr);
   //avoid assingment
   scd_tclAppRegHandleMngr & operator = (const scd_tclAppRegHandleMngr &corfoAppRegHandleMngr);

   //-- data structure to handle the list entries
   typedef struct
   {
      tU16 u16AppID;
      reg_tclApplication* poAppReg;
   } tDataElement;

   //-- some constants for this class
   enum
   {
      C_INCREASE = 31,
      C_SEM_NAME_SIZE = 32
   };

   //-- resize the internal list
   tBool               bResize();

   tU16 m_u16Size;                    //-- current size of list
   tU16 m_u16Entries;                 //-- current number of entries
   tDataElement* m_poAppRegHandler;   //-- entry point to the internal list
   OSAL_tSemHandle m_hSem;            //-- semaphore handle to synchronize data access
   tU32 m_u32ClientCntr;              //-- Counts how many clients called the initialization
   tChar m_sSemName[C_SEM_NAME_SIZE]; //-- semaphore name
};

/*---------------------------------------------------------------------*//**
 * Default constructor
 * creates an empty list and a system unique semaphore.
 *
 * @author  CM-DI/ESP4-Wiedemann
 * @date    16.11.06
 *//*---------------------------------------------------------------------*/
scd_tclAppRegHandleMngr::scd_tclAppRegHandleMngr()
   : m_u16Size(0)
   , m_u16Entries(0)
   , m_poAppRegHandler(NULL)
   , m_hSem(OSAL_C_INVALID_HANDLE)
   , m_u32ClientCntr(0)
{
   m_sSemName[0] = 0;
}

/*---------------------------------------------------------------------*//**
 * Initialises the object
 *
 * @return  TRUE or FALSE if initialisation failed
 * @author  CM-DI/ESP4-Wiedemann
 * @date    16.11.06
 *//*---------------------------------------------------------------------*/
tBool scd_tclAppRegHandleMngr::bInit()
{
   tChar sTempSemName[C_SEM_NAME_SIZE]; //-- temporary semaphore name
   tBool bResult = TRUE;
   OSAL_tProcessID tProcessId = OSAL_ProcessWhoAmI();

   if(m_hSem == OSAL_C_INVALID_HANDLE)
   {
      //-- create system unique semaphore name (each binary will create its own
      //   semaphore)
      if (OSALUTIL_s32SaveNPrintFormat(sTempSemName, 
                                       sizeof(sTempSemName),
                                       "S%d%p", 
                                       tProcessId, 
                                       this) != OSAL_ERROR)
      {
        //-- create semaphore (if semaphore can not be created this object simply
        //   does not manage any entries)
        if(OSAL_OK == OSAL_s32SemaphoreCreate( sTempSemName, &m_hSem, 0))
        {
           //-- Semaphore created
           ++m_u32ClientCntr;
           (void)OSALUTIL_szSaveStringNCopy(m_sSemName, sTempSemName, sizeof(m_sSemName));
           scd_vTraceMsg(TR_LEVEL_COMPONENT,"SCD create Semaphore %s from process %d", m_sSemName, tProcessId);

           //-- read all application entries from registry and save them 
           //   for further access
            
           if (!bLookupIoControlAvailable)
              vReadAppEntries();

           //-- post semaphore
           (void)OSAL_s32SemaphorePost(m_hSem);
        }
        else
        {
           //-- suppose two threads enter the check "m_hSem == OSAL_C_INVALID_HANDLE"
           //   at the same time and both think they have to create the semaphore.
           //   Then, both try to create the semaphore but only one Thread will succeed.
           //   Both Threads are serialized at the create semaphore call but only the
           //   first one will succeed. The other one can check m_hSem for a second Time
           //   if it is now valid, then the function can return success. Otherwise the
           //   initialisation failed
           if(m_hSem != OSAL_C_INVALID_HANDLE)
           {
              //-- Wait and Post semaphore to ensure, the semaphore name to be copied
              (void)OSAL_s32SemaphoreWait(m_hSem, OSAL_C_TIMEOUT_FOREVER);
              ++m_u32ClientCntr;
              (void)OSAL_s32SemaphorePost(m_hSem);

              //-- creation "succeeded". another thread has already created the semaphore
              scd_vTraceMsg(TR_LEVEL_ERRORS,"Late initialization for SCD from process %d. Forgotten to call scd_init() once at process startup? (sem: %s)", tProcessId, m_sSemName);
           }
           else
           {
              bResult = FALSE;
           }
        }
      }
      else
      {
        NORMAL_M_ASSERT_ALWAYS();
        bResult = FALSE;
      }
   }
   else
   {
      //-- Wait and Post semaphore to ensure, the semaphore name to be copied
      (void)OSAL_s32SemaphoreWait(m_hSem, OSAL_C_TIMEOUT_FOREVER);
      ++m_u32ClientCntr;
      (void)OSAL_s32SemaphorePost(m_hSem);

      scd_vTraceMsg(TR_LEVEL_ERRORS,"SCD scd_init already called from process %d (sem: %s)", tProcessId, m_sSemName);
   }

   if(bResult == FALSE)
   {
      scd_vTraceMsg(TR_LEVEL_FATAL,"SCD Semaphore %s creation failed from process %d", sTempSemName, tProcessId);
   }

   return bResult;
}

/*---------------------------------------------------------------------*//**
 * destructor
 *
 * @author  CM-DI/ESP4-Wiedemann
 * @date    16.11.06
 *//*---------------------------------------------------------------------*/
scd_tclAppRegHandleMngr::~scd_tclAppRegHandleMngr()
{
   //-- free all elements
   vFree(TRUE); //lint !e1551 Warning 1551: Function may throw exception.
            //            vFree throws no exeption!
} //lint !e1740 !e1579 Info 1740: pointer member 'm_poAppRegHandler' not directly
  //            freed or zeroed by destructor. Is done by vFree!

/*---------------------------------------------------------------------*//**
 * "destroys" the object. Only to destroy all contents without calling the
 * destructor. Necessary for destruction of a global object.
 *
 * @author  CM-DI/ESP4-Wiedemann
 * @date    16.11.06
 *//*---------------------------------------------------------------------*/
void scd_tclAppRegHandleMngr::vFree(tBool bLastCall)
{
   tDataElement*   poCurrent = NULL;
   OSAL_tSemHandle hSem = OSAL_C_INVALID_HANDLE;

   //-- check for valid semaphore
   if(m_hSem != OSAL_C_INVALID_HANDLE)
   {
      //-- wait semaphore
      (void)OSAL_s32SemaphoreWait(m_hSem, OSAL_C_TIMEOUT_FOREVER);
      
      //-- decrease client counter
      --m_u32ClientCntr;

      hSem = m_hSem;

      if(m_u32ClientCntr == 0)
      {
         //-- if last client calling prepare for memory and 
         //   semaphore destruction 
         m_hSem = OSAL_C_INVALID_HANDLE;
         bLastCall = TRUE; 
      }

      //-- post semaphore
      (void)OSAL_s32SemaphorePost(hSem);
   }

   if(bLastCall == TRUE)
   {
      poCurrent = m_poAppRegHandler;

      if(poCurrent != NULL)
      {
         //-- iterate all entries and delete the application root objects
         while(m_u16Entries > 0)
         {
            OSAL_DELETE poCurrent->poAppReg;
            poCurrent->poAppReg = NULL;

            ++poCurrent;
            --m_u16Entries;
         }

         //-- remove list memory
         OSAL_vMemoryFree(m_poAppRegHandler);
         m_poAppRegHandler = NULL;
         m_u16Size = 0;
      }
   }

   //-- if last call and semaphore valid then close it
   if(    (TRUE == bLastCall)
       && (hSem != OSAL_C_INVALID_HANDLE))
   {
      scd_vTraceMsg(TR_LEVEL_COMPONENT,"SCD destroy Semaphore %s", m_sSemName);
      //-- close and delete semaphore
      (void)OSAL_s32SemaphoreClose(hSem);
      (void)OSAL_s32SemaphoreDelete(m_sSemName);
      m_sSemName[0]=0;
   }
}

/*---------------------------------------------------------------------*//**
 * look up the application ID in the managed objects. Returns a pointer
 * to that object if found.
 *
 * @param   u16AppID     Application ID of the wanted entry
 * @return  NULL or valid pointer
 *
 * @author  CM-DI/ESP4-Wiedemann
 * @date    16.11.06
 *//*---------------------------------------------------------------------*/
reg_tclApplication* scd_tclAppRegHandleMngr::poGetEntry(tU16 u16AppID) 
{
   reg_tclApplication* poRet = NULL;
   OSAL_tProcessID tProcessId = OSAL_ProcessWhoAmI();


   //-- for those who forgot to call scd_init
   if(bIsInitialized() == FALSE)
   {
      scd_vInitTrace();
      scd_vTraceMsg(TR_LEVEL_FATAL, "SCD call scd_init once per process before initializing your App (AppID=0x%04X, ProcessID=%d)", u16AppID, tProcessId);
      if (!bInit())
         scd_vTraceMsg(TR_LEVEL_FATAL, "SCD bInit returns false (AppID=0x%04X, ProcessID=%d)", u16AppID, tProcessId);
   }

   //-- if semaphore exists lock it --> skip search if no semaphore available
   if(    (m_hSem != OSAL_C_INVALID_HANDLE)
       && (OSAL_s32SemaphoreWait(m_hSem, OSAL_C_TIMEOUT_FOREVER) == OSAL_OK))
   {
      tDataElement* poCurrent = m_poAppRegHandler;

      if(poCurrent != NULL)
      {
         //-- iterate all entries until valid entry found
         tU16 u16Cnt = m_u16Entries;

         while(    (u16Cnt > 0)
                && (poRet == NULL))
         {
            if(poCurrent->u16AppID == u16AppID)
            {
               poRet = poCurrent->poAppReg;
            }
            --u16Cnt;
            ++poCurrent;
         }

      }

      //-- release semaphore
      (void)OSAL_s32SemaphorePost(m_hSem);
   }

   return poRet;
}

/*---------------------------------------------------------------------*//**
 * adds a clone of the given entry to the internal mangement table.
 *
 * @param   u16AppID     Application ID of the new entry
 * @param   roNewEntry   Prototype for the new entry
 *
 * @return  NULL or valid pointer if entry succeeds
 *
 * @author  CM-DI/ESP4-Wiedemann
 * @date    16.11.06
 *//*---------------------------------------------------------------------*/
reg_tclApplication* scd_tclAppRegHandleMngr::poAddClone(tU16 u16AppID, const reg_tclApplication& roNewEntry)
{
   reg_tclApplication* poNewEntry = NULL;

   //-- lock semaphore if exists
   if(    (m_hSem != OSAL_C_INVALID_HANDLE)
       && (OSAL_s32SemaphoreWait(m_hSem, OSAL_C_TIMEOUT_FOREVER) == OSAL_OK))
   {
      //-- create clone of object to add it to the list
      poNewEntry = poIntAddClone(u16AppID, roNewEntry);

      //-- release semaphore
      (void)OSAL_s32SemaphorePost(m_hSem);
   }

   //-- return pointer of clone
   return poNewEntry;
}

/*---------------------------------------------------------------------*//**
 * adds a clone of the given entry to the internal mangement table.
 *
 * @param   u16AppID     Application ID of the new entry
 * @param   roNewEntry   Prototype for the new entry
 *
 * @return  NULL or valid pointer if entry succeeds
 *
 * @author  CM-DI/ESP4-Wiedemann
 * @date    16.11.06
 *//*---------------------------------------------------------------------*/
reg_tclApplication* scd_tclAppRegHandleMngr::poIntAddClone(tU16 u16AppID, const reg_tclApplication& roNewEntry)
{
   reg_tclApplication* poNewEntry;

   //-- create clone of object to add it to the list
   poNewEntry = OSAL_NEW reg_tclApplication(roNewEntry);

   if(poNewEntry != NULL)
   {
      if(    (m_u16Entries < m_u16Size)
          && (m_poAppRegHandler != NULL))
      {
         //-- add entry if it fits
         m_poAppRegHandler[m_u16Entries].poAppReg = poNewEntry;
         m_poAppRegHandler[m_u16Entries].u16AppID = u16AppID;
         ++m_u16Entries;
      }
      else
      {
         //-- resize memory for list
         if(bResize())
         {
            //-- and add entry if resize succeeded
            m_poAppRegHandler[m_u16Entries].poAppReg = poNewEntry;
            m_poAppRegHandler[m_u16Entries].u16AppID = u16AppID;
            ++m_u16Entries;
         }
         else
         {
            //-- it is not possible to add a new entry therefore
            //   delete the new object
            OSAL_DELETE poNewEntry;
            poNewEntry = NULL;
         }
      }
   }

   if(poNewEntry == NULL)
   {
      scd_vTraceMsg(TR_LEVEL_ERRORS,"SCD failed to add App 0x%04X (%s)", u16AppID, m_sSemName);
   }
   else
   {
      scd_vTraceMsg(TR_LEVEL_COMPONENT,"SCD App 0x%04X added (%s)", u16AppID, m_sSemName);
   }

   //-- return pointer of clone
   return poNewEntry;
}

/*---------------------------------------------------------------------*//**
 * resizes the management table
 *
 * @return  TRUE or FALSE dependent on success. If it returns FALSE, the
 *                        former table is still valid.
 *
 * @author  CM-DI/ESP4-Wiedemann
 * @date    16.11.06
 *//*---------------------------------------------------------------------*/
tBool scd_tclAppRegHandleMngr::bResize()
{
   tBool bResult = FALSE;

   //-- save old management table
   tDataElement* poTemp = m_poAppRegHandler;

   //-- allocate bigger memory for management table
   m_poAppRegHandler = (tDataElement*)OSAL_pvMemoryAllocate((m_u16Size + C_INCREASE) * sizeof(tDataElement));

   if(m_poAppRegHandler != NULL)
   {
      if(poTemp != NULL)
      {
         //-- copy content of old table to new memory
         (void)OSAL_pvMemoryCopy(m_poAppRegHandler, poTemp, m_u16Size * sizeof(tDataElement));

         //-- delete old table
         OSAL_vMemoryFree(poTemp);
      }

      //-- set size to new size
      m_u16Size = (tU16)(m_u16Size + C_INCREASE);
      scd_vTraceMsg(TR_LEVEL_COMPONENT,"SCD resize list Semaphore %s %d", m_sSemName, m_u16Size);

      bResult = TRUE;
   }
   else
   {
      //-- leave everything unchanged
      m_poAppRegHandler = poTemp;
      scd_vTraceMsg(TR_LEVEL_ERRORS,"SCD resize failed Semaphore %s %d", m_sSemName, m_u16Size);
   }

   return bResult;
}

/*---------------------------------------------------------------------*//**
 * This method iterates through all processes and applications which are
 * in the registry and memorizes the application entry points. This will
 * be done at initialization time. 
 *
 * @author  CM-DI/ESP4-Wiedemann
 * @date    16.11.06
 *//*---------------------------------------------------------------------*/
void scd_tclAppRegHandleMngr::vReadAppEntries()
{
   tU16 u16AppID;

   scd_vTraceMsg(TR_LEVEL_COMPONENT,"SCD init app entries (%s)", m_sSemName);

   //-- iterate all processes
   reg_tclProcessIterator oProcIter;
   while (oProcIter.bIsDone() == FALSE)
   {
      reg_tclProcess oProc;
      oProc = oProcIter.oItem();
      if(oProc.bIsValid())
      {
         // iterate applications
         reg_tclApplicationIterator oAppIter(oProc);
         while(oAppIter.bIsDone() == FALSE)
         {
            reg_tclApplication oApp;
            oApp = oAppIter.oItem();
            if(oApp.bIsValid())
            {
               if(oApp.bGetID (&u16AppID) == TRUE)
               {
                  (tVoid)poIntAddClone(u16AppID, oApp);
               }

            }/*if*/
            else
            {
               scd_vTraceMsg(TR_LEVEL_ERRORS,"SCD cannot open application");
            }/*else*/
            oAppIter.vNext();
         }/*while*/
      }/*if*/
      else
      {
         scd_vTraceMsg(TR_LEVEL_ERRORS,"SCD cannot open process");
      }/*else*/
      oProcIter.vNext();
   }/*while*/
}

/*************************************************************************
| variable definition (scope: global)
|*************************************************************************/
/*none*/

/*************************************************************************
| variable definition (scope: modul-local)
|*************************************************************************/
static tBool bMsgPoolCreatedByMe = FALSE;

/*************************************************************************
| function prototype (scope: modul-local)
|*************************************************************************/

/*************************************************************************
| function implementation (scope: modul-local)
|*************************************************************************/

/*---------------------------------------------------------------------*//**
 * get pointer to the global management table object for application
 * entry points to the registry
 *
 * @return  pointer to the object
 *
 * @author  CM-DI/ESP4-Wiedemann
 * @date    16.11.06
 *//*---------------------------------------------------------------------*/
static scd_tclAppRegHandleMngr& rfoGetTheAppRegMngr()
{
   static scd_tclAppRegHandleMngr oTheAppRegHandleMngr;

   return oTheAppRegHandleMngr;
}

/*---------------------------------------------------------------------*//**
 * SCD internal function to get the registry entry point for a certain
 * application.
 *
 * This function tries to find the applications entry point in the registry
 * management table first. If that does not succeed it iterates the registry
 * for the application and enters the entry point to the management table if
 * found
 *
 * @param   u16AppID     Application ID of the wanted application
 * @return  NULL or valid pointer to a reg_tclApplication object
 *
 * @author  CM-DI/ESP4-Wiedemann
 * @date    16.11.06
 *//*---------------------------------------------------------------------*/
static reg_tclApplication* scd_poGetApp(tU16 u16AppID)
{
   //-- lookup AppID in management table
   reg_tclApplication* poApp = rfoGetTheAppRegMngr().poGetEntry(u16AppID);
   tBool bFound = FALSE;
   tU16  u16LocAppID;

   if (poApp != NULL)
   {
      scd_vTraceMsg(TR_LEVEL_USER_1,"SCD reuse access AppID 0x%04X", u16AppID);
   }
   else
   {
      if (!bLookupIoControlAvailable)
      {
         scd_vTraceMsg(TR_LEVEL_USER_1,"SCD iterate for AppID 0x%04X", u16AppID);

         //-- if not found iterate processes
         reg_tclProcessIterator oProcIter;
         while ((oProcIter.bIsDone() == FALSE)&&(bFound == FALSE))
         {
            reg_tclProcess oProc;
            oProc = oProcIter.oItem();
            if (oProc.bIsValid())
            {
               // iterate applications
               reg_tclApplicationIterator oAppIter(oProc);
               while ((oAppIter.bIsDone() == FALSE)&&(bFound == FALSE))
               {
                  reg_tclApplication oApp;
                  oApp = oAppIter.oItem();
                  if (oApp.bIsValid())
                  {
                     tBool bAppIDValid = oApp.bGetID (&u16LocAppID);
                     if (    bAppIDValid 
                             && (u16LocAppID == u16AppID))
                     {
                        //-- if application found add it to the management table
                        poApp = rfoGetTheAppRegMngr().poAddClone(u16AppID, oApp);
                        bFound = TRUE;
                     }/*if*/
                  }/*if*/
                  else
                  {
                     scd_vTraceMsg(TR_LEVEL_ERRORS,"scd_bGetAppConfigurationValue: key invalid");
                  }/*else*/
                  oAppIter.vNext();
               }/*while*/
            }/*if*/
            else
            {
               scd_vTraceMsg(TR_LEVEL_ERRORS,"scd_bGetAppConfigurationValue: invalid handle");
            }/*else*/
            oProcIter.vNext();
         }/*while*/
      }
      #ifdef SIMPLE_SCD
      else
      {
         if (fd != OSAL_ERROR)
         {
            OSAL_trIOCtrlRegistryValue srReg;

            srReg.s32Type = u16AppID;

            if (OSAL_s32IOControl(fd, OSAL_C_S32_IOCTRL_REGLOOKUPAPPPATH, (intptr_t)&srReg) != OSAL_ERROR)
            {
               tChar szTempName[REG_C_U32_KEYNAME_MAXLEN];

               if (OSALUTIL_s32SaveNPrintFormat((tString)szTempName,
                                                sizeof(szTempName),
                                                "%s/%s", 
                                                OSAL_C_STRING_DEVICE_REGISTRY, 
                                                srReg.s8Name) != OSAL_ERROR)
               {
                 reg_tclRegKey reg(szTempName);
                 reg_tclApplication trAppReg(reg);

                 scd_vTraceMsg(TR_LEVEL_SYSTEM_MIN,"SCD application path found AppID 0x%04X = %s", u16AppID, szTempName);

                 // store it for next query
                 poApp = rfoGetTheAppRegMngr().poAddClone(u16AppID, trAppReg);

                 if (poApp == NULL)
                 {
                    scd_vTraceMsg(TR_LEVEL_FATAL,"scd_poGetApp could not Add clone");
                 }
               }
               else
               {
                 NORMAL_M_ASSERT_ALWAYS();
                 poApp = NULL;
               }
            }
            else
            {
               scd_vTraceMsg(TR_LEVEL_ERRORS,"SCD application not found AppID %d errorcode 0x%04x", u16AppID, OSAL_u32ErrorCode());
            }
         }
         else
         {
            scd_vTraceMsg(TR_LEVEL_FATAL,"scd_poGetApp could not perform OSAL_s32IOControl(OSAL_C_S32_IOCTRL_REGLOOKUPAPPPATH) due to invalid registry handle");
         }
      }
      #endif
   }

   if (poApp == NULL)
   {
      scd_vTraceMsg(TR_LEVEL_ERRORS,"SCD application not found AppID %d", u16AppID);
   }
   
   //-- return application entry point
   return poApp;
}

/*************************************************************************
| function implementation (scope: global)
|*************************************************************************/
extern "C" 
{
tVoid dummy_scd_dll(tVoid) {};

#ifdef SIMPLE_SCD
tBool scd_bTestLookupIoControl()
{
   tBool bRetVal = FALSE;

   if (fd != OSAL_ERROR)
   {
      OSAL_trIOCtrlRegistryValue srReg;
      srReg.s32Type = 0; // SPM

      tS32 bRet = OSAL_s32IOControl(fd, OSAL_C_S32_IOCTRL_REGLOOKUPAPPPATH, (intptr_t)&srReg);
      if ((bRet != OSAL_ERROR) && ((OSAL_u32ErrorCode() != OSAL_E_WRONGFUNC) && (OSAL_u32ErrorCode() != OSAL_E_NOTSUPPORTED)))
      {
         srReg.s32Type = 256; // SPM
         bRet = OSAL_s32IOControl(fd, OSAL_C_S32_IOCTRL_REGLOOKUPSRVPATH, (intptr_t)&srReg);
         if ((bRet != OSAL_ERROR) && ((OSAL_u32ErrorCode() != OSAL_E_WRONGFUNC) && (OSAL_u32ErrorCode() != OSAL_E_NOTSUPPORTED)))
         {
            bRetVal = TRUE;
         }
      }
   }
   else
   {
     scd_vTraceMsg(TR_LEVEL_FATAL,"scd_bTestLookupIoControl could not perform OSAL_s32IOControl(OSAL_C_S32_IOCTRL_REGLOOKUPAPPPATH / ..SRVPATH) due to invalid registry handle");
   }

   if (TRUE == bRetVal)
   {
     scd_vTraceMsg(TR_LEVEL_COMPONENT,"Platform can handle SCD Lookup");
   }
   else
   {
     scd_vTraceMsg(TR_LEVEL_ERRORS,"Platform can NOT handle SCD Lookup");
   }

   return (bRetVal);
}
#endif

/******************************************************************************
 *FUNCTION:    scd_init
 *DESCRIPTION: create message-pool
 *             this function must be called once at startrup of the system
 *
 *PARAMETER:   tVoid
 *
 *RETURNVALUE: TRUE, FALSE
 *
 *HISTORY:
 *30.11.2000 Rev.1.0 K7/EFT21-Kuhn
 *           initial revision
 *06.03.2003 Rev.1.0 CM-DI/ESA1-Fischer
 *           scd_init only creates message-pool, create message-queue
 *           with scd_bCreateQueue
 *****************************************************************************/

tBool scd_init( tVoid )
{
   reg_tclRegKey oReg;

   tU32  u32MaxPoolSize = 0;
   tBool bRet;

   // open trace-device
   scd_vInitTrace();

   scd_vTraceMsg(TR_LEVEL_COMPONENT,"scd_init called");

   #ifdef SIMPLE_SCD
   if (fd == OSAL_ERROR)
   {
      fd = OSAL_IOOpen(OSAL_C_STRING_DEVICE_REGISTRY "/" REGSTRING_LOCAL_MACHINE "/" REGKEY_BLAUPUNKT, OSAL_EN_READWRITE);

      if (fd == OSAL_ERROR)
      {
         scd_vTraceMsg(TR_LEVEL_FATAL,"scd_init: could not open registry device for path '%s'",
                       OSAL_C_STRING_DEVICE_REGISTRY "/" REGSTRING_LOCAL_MACHINE "/" REGKEY_BLAUPUNKT);
      }
   }

   bLookupIoControlAvailable = scd_bTestLookupIoControl();
   #endif

   bRet = rfoGetTheAppRegMngr().bInit();
   if(bRet == TRUE)
   {
      bRet = scd_tclServiceAppMap::rfoGetServiceAppMap().bInit();
      if(bRet == FALSE)
      {
         scd_vTraceMsg(TR_LEVEL_FATAL,"scd_init: init ServiceAppMap failed");
      }
   }
   else
   {
      scd_vTraceMsg(TR_LEVEL_FATAL,"scd_init: init AppRegMngr failed");
   }

   if(bRet == TRUE)
   {
      // try to read message-pool-size from registry, if failed use default
      if (   (oReg.bOpen(OSAL_C_STRING_DEVICE_REGISTRY "/" REGSTRING_LOCAL_MACHINE "/" REGKEY_BLAUPUNKT)==FALSE)
         ||  (oReg.bQueryU32(SCD_MAX_POOL_SIZE_VALUE_NAME,&u32MaxPoolSize)==FALSE) )
      {
         u32MaxPoolSize = SCD_MESSAGE_POOL_SIZE;
      }

      // create message pool, if case of failure, try to open it. if this fails, too --> ERROR!!!
      if ( OSAL_s32MessagePoolCreate(u32MaxPoolSize)!= OSAL_OK )
      {
         if (OSAL_s32MessagePoolOpen() != OSAL_OK)
         {
            scd_vTraceMsg(TR_LEVEL_ERRORS,"scd_init: can't create or open message-pool");
            bRet = FALSE;
         }
      }
      else
      {
         bMsgPoolCreatedByMe = TRUE;
      }

      oReg.vClose();
   }

   return ( bRet );
}/*scd_init*/

/******************************************************************************
 *FUNCTION:    scd_exit
 *DESCRIPTION: delete message-pool
 *
 *PARAMETER:   tVoid
 *
 *RETURNVALUE:
 *
 *HISTORY:
 *30.11.2000 Rev.1.0 K7/EFT21-Kuhn
 *           initial revision
 *06.03.2003 Rev.1.1 CM-DI/ESA1-Fischer
 *           scd_exit only deletes message-pool, delete message-queue
 *           with scd_s32DeleteQueue
 *****************************************************************************/

tVoid scd_exit( tVoid )
{
   scd_vTraceMsg(TR_LEVEL_COMPONENT,"scd_exit called");

   #ifdef SIMPLE_SCD
   if (fd != OSAL_ERROR)
   {
      if (OSAL_s32IOClose(fd) == OSAL_ERROR)
      {
         scd_vTraceMsg(TR_LEVEL_FATAL,"scd_exit : could not close registry device");
      }
      else
      {
         fd = OSAL_ERROR;
      }
   }
   #endif

   //close the message-pool
   (tVoid)OSAL_s32MessagePoolClose();

   if ( bMsgPoolCreatedByMe )
   {
      // delete message-pool
      (tVoid)OSAL_s32MessagePoolDelete();
   }

   rfoGetTheAppRegMngr().vFree();

   scd_tclServiceAppMap::rfoGetServiceAppMap().vFree();

   // close trace-device
   scd_vCloseTrace();

}/*scd_exit*/


/******************************************************************************
 *FUNCTION:    scd_u32GetNbrOfAppIds
 *DESCRIPTION: gives number of applications offering this service
 *             how it works : the regitry is structured as follows:
 *                HKEY_LOCAL_MACHINE\SOFTWARE\BLAUPUNKT\PROCESS
 *             under this key there are several subkeys, which define the
 *             processes runing. in this function alle processes are queried
 *             by enumerating all keys. each (process) key has several subkeys
 *             defining the application implemented by this process. to querry
 *             all applications all keys are enumerated. each aplication has a
 *             subkey 'services'. below this subkey there are several key definig the
 *             services. querying the services is done by enumerating teh subkeys.
 *             every service subkey has a value, called serviceid. this function
 *             compares every serviceid it findes with the given serviceid. if they
 *             match a counter is incremented.
 *             at the end this counter is copied in the output variable.
 *PARAMETER:   (I)  ulServiceId
 *             (->O) pulNbrOfAppIds
 *
 *RETURNVALUE:
 *
 *HISTORY:
 *12.06.2001 Rev.1.0 K7/EFT21-Kuhn
 *           initial revision
 *****************************************************************************/
tU32 scd_u32GetNbrOfAppIds (tU16 u16ServiceId)
{
  tU32  u32Count=0;
  scd_vTraceMsg(TR_LEVEL_USER_4,"scd_u32GetNbrOfAppIds for service %d called", u16ServiceId);

  // iterate processes
  reg_tclProcessIterator oProcIter;
  while(oProcIter.bIsDone() == FALSE)
  {
     reg_tclProcess oProc;
     oProc = oProcIter.oItem();

     if(oProc.bIsValid())
     {
        reg_tclApplicationIterator oAppIter(oProc);
        while(oAppIter.bIsDone() == FALSE)
        {
           reg_tclApplication oApp;
           oApp = oAppIter.oItem();
           if(oApp.bIsValid())
           {
              reg_tclServiceIterator oSvcIter(oApp);
              while(oSvcIter.bIsDone() == FALSE)
              {
                 reg_tclService oSvc;
                 oSvc = oSvcIter.oItem();
                 if(oSvc.bIsValid())
                 {
                    tU32 u32SvcID;
                    if(oSvc.bGetID(&u32SvcID) == TRUE)
                    {
                       if (u32SvcID==u16ServiceId)
                       {
                          //it is, so increment counter
                          u32Count ++;
                       }//if
                    }/*if*/
                    else
                    {
                       scd_vTraceMsg(TR_LEVEL_ERRORS,"scd_u32GetNbrOfAppIds: can't retrieve service id");
                    }/*else*/
                 }/*if*/
                 else
                 {
                    scd_vTraceMsg(TR_LEVEL_ERRORS,"scd_u32GetNbrOfAppIds: app key invalid");
                 }/*else*/
                 oSvcIter.vNext();
              }/*while*/
           }/*if*/
           else
           {
              scd_vTraceMsg(TR_LEVEL_ERRORS,"scd_u32GetNbrOfAppIds: app key invalid");
           }/*else*/
           oAppIter.vNext();
        }/*while*/
     }/*if*/
     else
     {
        scd_vTraceMsg(TR_LEVEL_ERRORS,"scd_u32GetNbrOfAppIds: invalid handle");
     }/*else*/
     oProcIter.vNext();
  }/*while*/

  scd_vTraceMsg(TR_LEVEL_SYSTEM_MIN,"scd_u32GetNbrOfAppIds: Service %d cnt %d", u16ServiceId, u32Count);

  return(u32Count);
}/*scd_u32GetNbrOfAppIds*/


/******************************************************************************
*FUNCTION:    scd_bGetAppIdFromServiceId
*DESCRIPTION: gives list of ids of applications offering the given service
*             how it works : the regitry is structured as follows:
*                HKEY_LOCAL_MACHINE\SOFTWARE\BLAUPUNKT\PROCESS
*             under this key there are several subkeys, which define the
*             processes runing. in this function alle processes are queried
*             by enumerating all keys. each (process) key has several subkeys
*             defining the application implemented by this process. to querry
*             all applications all keys are enumerated. for each application the
*             application information (applicationid and version) is stored
*             temporarily in a set of variables. each aplication has a
*             subkey 'services'. below this subkey there are several key definig the
*             services. querying the services is done by enumerating teh subkeys.
*             every service subkey has a value, called serviceid. this function
*             compares every serviceid it findes with the given serviceid. if they
*             match the previously saved application information (version and
*             applicationid) is copied in the client provided memory.
*             this will be done for all applications which provide the
*             service the client is looking for. it will be checked that
*             only as much data is writen as the user provided memory can hold.
*             (this is done by comparing the internal counter with the
*             client provided size of the memory)
*
*  bReadRegistry  pu32NbrOfAppInfos  Description
*   TRUE           >  1              read registry/update the map, search in the map
*   TRUE           == 1              search in the map, if not in the map: read registry/update the map and search again in the map
*   FALSE          all values        search in the map
*
*PARAMETER:   u16ServiceId      : [in] serviceId the client is looking for
*             pu32NbrOfAppInfos : [in] max number of ScdAppInfo to be stored in prScdAppInfo, [out] real number of ScdAppInfo stored in prScdAppInfo
*             prScdAppInfo      : [out] pointer where to store ScdAppInfo
*             bReadRegistry     : [in]  boolean to indicates to read the registry/update the map or not
* 
* 
*RETURNVALUE:
* FALSE       if parameters were incorrect, no out-parameters were modified
* TRUE        search was executed and out-parameters have been updated - If the number of elements found is less than the mex requested, only the found number part of prScdAppInfo have been updated
*
*
*HISTORY:
*30.11.2000 Rev.1.0 K7/EFT21-Kuhn
*           initial revision
*02.03.2011 CM-AI/PJ-GM54 Creux
*           use the new scd_tclServiceAppMap::bGetAppIdFromServiceId()
*09.03.2011 CM-AI/PJ-GM54 Creux
*           add new parameter bReadRegistry
*****************************************************************************/
tBool scd_bGetAppIdFromServiceId (tU16 u16ServiceId,
    tU32 *pu32NbrOfAppInfos,
    trScdAppInfo *prScdAppInfo,
    tBool bReadRegistry)
{
  tBool bResult = FALSE;

  //first of all, check if the pointers pu32NbrOfAppInfos and prScdAppInfo are legal
  if ((prScdAppInfo == NULL)||(pu32NbrOfAppInfos == NULL))
  {
     //one or both pointer(s) are not legal, so return with error
     scd_vTraceMsg(TR_LEVEL_FATAL,"scd_bGetAppIdFromServiceId %d invalid parameter (pointer null)", u16ServiceId);
     return (FALSE);
  }

  #ifdef SCD_TRACE_TIMING
  OSAL_trThreadControlBlock tcb;
  OSAL_tThreadID tid = OSAL_ThreadWhoAmI();
  OSAL_s32ThreadControlBlock(tid, &tcb);
  OSAL_tMSecond StartTime = tcb.runningTime;
  #endif

  if (!bLookupIoControlAvailable)
  {
     // get the app id from the service map
     bResult = scd_tclServiceAppMap::rfoGetServiceAppMap().bGetAppIdFromServiceId(u16ServiceId, pu32NbrOfAppInfos, prScdAppInfo, bReadRegistry);
     if (*pu32NbrOfAppInfos == 0)
     {
        scd_vTraceMsg(TR_LEVEL_ERRORS,"scd_bGetAppIdFromServiceId for service %d called, no AppId found", u16ServiceId);
     }
     if (* pu32NbrOfAppInfos > 1)
     {
        scd_vTraceMsg(TR_LEVEL_ERRORS,"scd_bGetAppIdFromServiceId for service %d called, multiple AppId found", u16ServiceId);
     }
  }
  #ifdef SIMPLE_SCD
  else
  {
     if (fd != OSAL_ERROR)
     {
        OSAL_trIOCtrlRegistryValue srReg;
        srReg.s32Type = u16ServiceId;
   
        if (OSAL_s32IOControl(fd, OSAL_C_S32_IOCTRL_REGLOOKUPSRVPATH, (intptr_t)&srReg) != OSAL_ERROR)
        {
           *pu32NbrOfAppInfos = 1;
           prScdAppInfo->u16AppId = (tU16)srReg.s32Type;
   
           scd_vTraceMsg(TR_LEVEL_COMPONENT,"scd_bGetAppIdFromServiceId result %d", srReg.s32Type);
   
           bResult = TRUE;
        }
        else
        {
            scd_vTraceMsg(TR_LEVEL_FATAL,"SCD application not found u16ServiceId %d errorcode 0x%04x", u16ServiceId, OSAL_u32ErrorCode());
        }
     }
     else
     {
        scd_vTraceMsg(TR_LEVEL_FATAL,"SCD application could not open registry for u16ServiceId %d errorcode 0x%04x", u16ServiceId, OSAL_u32ErrorCode());
     }
  }
  #endif

  #ifdef SCD_TRACE_TIMING
  OSAL_s32ThreadControlBlock(tid, &tcb); 
  scd_vTraceMsg(TR_LEVEL_ERRORS,"scd_bGetAppIdFromServiceId: mapsearch %d ms", tcb.runningTime - StartTime);
  #endif 

  scd_vTraceMsg(TR_LEVEL_SYSTEM_MIN,"scd_bGetAppIdFromServiceId for service %d called, result %d number %d App %d", u16ServiceId, bResult, *pu32NbrOfAppInfos, prScdAppInfo->u16AppId);

  return bResult;
}/*scd_bGetAppIdFromServiceId*/

/******************************************************************************
*FUNCTION:    scd_bGetAppInterfaceConfiguration
*DESCRIPTION:
*
*PARAMETER:   (I)    u16AppId      : ApplicationId the client is looking for
*
*RETURNVALUE: appThreadPrio
*
*HISTORY:
*     unknown
*****************************************************************************/

tBool scd_bGetAppInterfaceConfiguration( tU16 u16AppID, scd_trAppInterfaceConfiguration * prAIFConf)
{
  tBool bFound = FALSE;

  scd_vTraceMsg(TR_LEVEL_USER_4,"scd_bGetAppInterfaceConfiguration for AppId 0x%04X called", u16AppID);

  //-- get application entry point to the registry
  reg_tclApplication* poApp = scd_poGetApp(u16AppID);

  if(poApp != NULL)
  {
     reg_tclAppConfig oAppCfg(*poApp);
     bFound = TRUE;

     if (oAppCfg.bGetMsgCountIMbx (&(prAIFConf->u32MsgCountIntermediateMbx))==FALSE)
     {
        bFound = FALSE;
     }/*if*/
     else if (oAppCfg.bGetWatchdogInterval (&(prAIFConf->u32WatchdogInterval))==FALSE)
     {
        /* downwards compatibility: ignore read error -> watchdog off */
        prAIFConf->u32WatchdogInterval = 0;
     }/*if*/

     scd_vTraceMsg(TR_LEVEL_SYSTEM_MIN,"scd_bGetAppInterfaceConfiguration for AppId 0x%04X = Msg %d Wdt %d ", u16AppID, prAIFConf->u32MsgCountIntermediateMbx, prAIFConf->u32WatchdogInterval);

  }

  if (bFound == FALSE)
  {
     scd_vTraceMsg(TR_LEVEL_FATAL,"scd_bGetAppInterfaceConfiguration for AppId 0x%04X returned false", u16AppID);
  }

  return ( bFound );
}/*scd_bGetAppInterfaceConfiguration*/


/******************************************************************************
*FUNCTION:    scd_bGetAppConfigurationValue
*DESCRIPTION:
*
*PARAMETER:   (I)    u16AppId                   : ApplicationId the client is looking for
*PARAMETER:   (I)    coszKeyName                : Key-Name
*PARAMETER:   (I)    coszValueName              : Value-Name
*
*RETURNVALUE: Value
*
*HISTORY:
*     unknown
*****************************************************************************/

tBool scd_bGetAppConfigurationValue( tU16 u16AppID, tCString coszKeyName, tCString coszValueName, tPU32 pu32Value)
{
  tBool bFound = FALSE;

  //-- get application entry point to the registry
  reg_tclApplication* poApp = scd_poGetApp(u16AppID);

  if(poApp != NULL)
  {
      reg_tclRegKey oKey;
      if (oKey.bOpen(*poApp, (const tChar*) coszKeyName))
      {
         if (oKey.bQueryU32((const tChar*) coszValueName, pu32Value) == TRUE)
         {
            scd_vTraceMsg(TR_LEVEL_SYSTEM_MIN,"scd_bGetAppConfigurationString: App 0x%04X Key %s value %s result %d", u16AppID, coszKeyName, coszValueName, *pu32Value);
            bFound = TRUE;
         }
      }
      else
      {
         scd_vTraceMsg(TR_LEVEL_ERRORS,"scd_bGetAppConfigurationValue: could not open key %s value %s", coszKeyName, coszValueName);
      }
  }

  return ( bFound );
}/*scd_bGetAppConfigurationValue*/



/******************************************************************************
*FUNCTION:    scd_bGetAppConfigurationString
*DESCRIPTION:
*
*PARAMETER:   (I)    u16AppId                : ApplicationId the client is looking for
*PARAMETER:   (I)    coszKeyName             : Key-Name
*PARAMETER:   (I)    coszValueName              : Value-Name
*PARAMETER:   (O)    szOutputString                                     : initialized pointer for to be read string
*PARAMETER:   (I)    u32MaxLength                                       : maximum string length (number of allcoated bytes)
*
*RETURNVALUE: Value
*
*HISTORY:
*     13.12.2005 CM-DI/ESP3-Pirklbauer      initial Version
*****************************************************************************/

tBool scd_bGetAppConfigurationString( tU16 u16AppID, tCString coszKeyName, tCString coszValueName, tString szOutputString, tU32 u32MaxLength)
{
  tBool bFound = FALSE;

  //-- get application entry point to the registry
  reg_tclApplication* poApp = scd_poGetApp(u16AppID);

  if(poApp != NULL)
  {
      reg_tclRegKey oKey;
      if (oKey.bOpen(*poApp, (const tChar*) coszKeyName))
      {
         if (oKey.bQueryString((const tChar*) coszValueName, szOutputString,u32MaxLength) == TRUE)
         {
            bFound = TRUE;
            scd_vTraceMsg(TR_LEVEL_SYSTEM_MIN,"scd_bGetAppConfigurationString: App 0x%04X Key %s value %s result %s", u16AppID, coszKeyName, coszValueName, szOutputString);
         }
      }
      else
      {
         scd_vTraceMsg(TR_LEVEL_ERRORS,"scd_bGetAppConfigurationString: could not open key %s value %s", coszKeyName, coszValueName);
      }

  }
  return ( bFound );
}/*scd_bGetAppConfigurationValue*/

/******************************************************************************
*FUNCTION:    scd_bGetProcConfigurationValue
*DESCRIPTION: get registry key of process the app belongs to 
*
*PARAMETER:   (I)    u16AppId                   : ApplicationId the client is looking for
*PARAMETER:   (I)    coszKeyName                : Key-Name
*PARAMETER:   (I)    coszValueName              : Value-Name
*
*RETURNVALUE: Value

*****************************************************************************/

tBool scd_bGetProcConfigurationValue( tU16 u16AppID, tCString coszKeyName, tCString coszValueName, tPU32 pu32Value)
{
  tBool bFound = FALSE;

  //-- get application entry point to the registry
  reg_tclApplication* poApp = scd_poGetApp(u16AppID);

  if(poApp != NULL)
  {
     reg_tclRegKey oKey(*poApp);
     if(oKey.bNavigateKeyNameUp())
     {
        if (
           (coszKeyName == 0)
           || (OSAL_u32StringLength(coszKeyName) == 0)
           || oKey.bRelOpen((const tChar*) coszKeyName)
           )
        {
           if (oKey.bQueryU32((const tChar*) coszValueName, pu32Value) == TRUE)
              bFound = TRUE;
        }
     }
  }

  return ( bFound );
}/*scd_bGetProcConfigurationValue*/



/******************************************************************************
*FUNCTION:    scd_bGetProcConfigurationString
*DESCRIPTION: get registry key of process the app belongs to
*
*PARAMETER:   (I)    u16AppId                : ApplicationId the client is looking for
*PARAMETER:   (I)    coszKeyName             : Key-Name
*PARAMETER:   (I)    coszValueName              : Value-Name
*PARAMETER:   (O)    szOutputString                                     : initialized pointer for to be read string
*PARAMETER:   (I)    u32MaxLength                                       : maximum string length (number of allcoated bytes)
*
*RETURNVALUE: Value
*****************************************************************************/

tBool scd_bGetProcConfigurationString( tU16 u16AppID, tCString coszKeyName, tCString coszValueName, tString szOutputString, tU32 u32MaxLength)
{
  tBool bFound = FALSE;

  //-- get application entry point to the registry
  reg_tclApplication* poApp = scd_poGetApp(u16AppID);

  if(poApp != NULL)
  {
     reg_tclRegKey oKey(*poApp);
     if(oKey.bNavigateKeyNameUp())
     {
        if (
           (coszKeyName == 0)
           || (OSAL_u32StringLength(coszKeyName) == 0)
           || oKey.bRelOpen((const tChar*) coszKeyName)
           )
        {
           if (oKey.bQueryString((const tChar*) coszValueName, szOutputString,u32MaxLength) == TRUE)
              bFound = TRUE;
        }
     }
  }
  return ( bFound );
}/*scd_bGetProcConfigurationValue*/




/******************************************************************************
*FUNCTION:    scd_OpenQueue
*DESCRIPTION: opens the existing inputqueue of the given application
*PARAMETER:   (I) u16AppId
*
*RETURNVALUE: handle to the queue
*
*HISTORY:
*30.11.2000 Rev.1.0 K7/EFT21-Kuhn
*           initial revision
*****************************************************************************/
OSAL_tMQueueHandle scd_OpenQueue (tU16 u16AppId)
{
   tChar szName[SCD_MAX_STRING_LENGTH] = "";

   OSAL_tMQueueHandle hMessageQueueHandle;
   tS32 s32Ret;

   if (OSALUTIL_s32SaveNPrintFormat((tString)szName,
                                    sizeof(szName),
                                    "mbx_%i",
                                    u16AppId) != OSAL_ERROR)
   {
     s32Ret = OSAL_s32MessageQueueOpen(szName,OSAL_EN_READWRITE,&hMessageQueueHandle);

     if ( s32Ret != OSAL_OK )
     {
		tU32 u32OsalErrorCode = OSAL_u32ErrorCode();
        scd_vTraceMsg(TR_LEVEL_FATAL,"scd_OpenQueue failed for AppId = 0x%04X", u16AppId);
		scd_vTraceMsg(TR_LEVEL_FATAL,"scd_OpenQueue:OSAL_s32MessageQueueOpen failed for AppId 0x%04X: Handle: %d, OsalErrorCode 0x%08X (%s)", u16AppId, hMessageQueueHandle, u32OsalErrorCode, OSAL_coszErrorText(u32OsalErrorCode) );

        hMessageQueueHandle=OSAL_C_INVALID_HANDLE;
        // NORMAL_M_ASSERT_ALWAYS();
     }
     else
     {
        scd_vTraceMsg(TR_LEVEL_USER_3,"scd_OpenQueue success for AppId = 0x%04X", u16AppId);
     }
   }
   else
   {
     NORMAL_M_ASSERT_ALWAYS();
     hMessageQueueHandle=OSAL_C_INVALID_HANDLE;
   }

   return(hMessageQueueHandle);
}/*scd_OpenQueue*/

/******************************************************************************
*FUNCTION:    scd_s32CloseQueue
*DESCRIPTION: closes the inputqueue (only the handel is affected)
*PARAMETER:   (I) queueHdl
*
*RETURNVALUE:
*
*HISTORY:
*30.11.2000 Rev.1.0 K7/EFT21-Kuhn
*           initial revision
*****************************************************************************/
tS32    scd_s32CloseQueue       (OSAL_tMQueueHandle queueHdl)
{
  return(OSAL_s32MessageQueueClose(queueHdl));
}/*scd_s32CloseQueue*/

/******************************************************************************
*FUNCTION:    scd_bCreateQueue
*DESCRIPTION: creates inputqueue of the given application
*PARAMETER:   (I) u16AppId
*
*RETURNVALUE: tBool: TRUE : message-queue created
*                    FALSE: message-queue create failed
*
*HISTORY:
*06.03.2003 Rev.1.0 CM-DI/ESA1-Fischer
*           initial revision
*****************************************************************************/
tBool scd_bCreateQueue (tU16 u16AppId)
{
   OSAL_tMQueueHandle hTempMessageQueueHdl = OSAL_C_INVALID_HANDLE;
   tChar szName[SCD_MAX_STRING_LENGTH] = "";
   tU32  u32MaxMessagesInQueue = 0;
   tBool bRetVal = FALSE;

   //-- get application entry point to the registry
   reg_tclApplication* poApp = scd_poGetApp(u16AppId);

   if(poApp != NULL)
   {
      // get max num of messages and message-size
      if (  poApp->bGetMaxMCount(&u32MaxMessagesInQueue) )
      {
         // reg-entry found, create mailboxname
         if (OSALUTIL_s32SaveNPrintFormat((tString)szName,
                                          sizeof(szName),
                                          "mbx_%i",
                                          u16AppId) != OSAL_ERROR)
         {
           //create mailbox
           if (OSAL_s32MessageQueueCreate(szName,
                 u32MaxMessagesInQueue,
                 SCD_MAILBOX_MAX_MESSAGE_LENGTH,
                 OSAL_EN_READWRITE, &hTempMessageQueueHdl)==OSAL_ERROR)
           {
				/* impossible to deliver the message */
				tU32 u32OsalErrorCode = OSAL_u32ErrorCode();
   
				scd_vTraceMsgErrmem(TR_LEVEL_FATAL,"scd_bCreateQueue:OSAL_s32MessageQueueCreate failed for AppId 0x%04X, Name= %s, Size=%d,%d", u16AppId, szName, u32MaxMessagesInQueue, SCD_MAILBOX_MAX_MESSAGE_LENGTH);
				scd_vTraceMsgErrmem(TR_LEVEL_FATAL,"scd_bCreateQueue:OSAL_s32MessageQueueCreate failed for AppId 0x%04X: Handle: %d, OsalErrorCode 0x%08X (%s)", u16AppId, hTempMessageQueueHdl, u32OsalErrorCode, OSAL_coszErrorText(u32OsalErrorCode) );
           }
           else
           {
              // --Close the message queue in order to avoid handle leak.
              (tVoid)OSAL_s32MessageQueueClose(hTempMessageQueueHdl);
              scd_vTraceMsg(TR_LEVEL_USER_1,"scd_bCreateQueue: OSAL_s32MessageQueueCreate success for AppId 0x%04X", u16AppId);
              bRetVal = TRUE;
           }
         }
         else
         {
           NORMAL_M_ASSERT_ALWAYS();
         }
      } // end if message-size and message-queue-size successfully read
      else
      {
         scd_vTraceMsgErrmem(TR_LEVEL_FATAL,"scd_bCreateQueue: failed to find bGetMaxMCount for AppId 0x%04X", u16AppId);
         NORMAL_M_ASSERT_ALWAYS();
      }
   } // end if application is valid
   else
   {
      scd_vTraceMsg(TR_LEVEL_FATAL,"scd_bCreateQueue: could not find AppId 0x%04X in Registry", u16AppId);
   }

   return(bRetVal);
} // end function scd_bCreateQueue

/******************************************************************************
*FUNCTION:    scd_s32DeleteQueue
*DESCRIPTION: deletes the inputqueue of the given application
*PARAMETER:   (I) tU16 u16AppId
*
*RETURNVALUE: tS32: OSAL_OK
*                   OSAL_ERROR
*
*HISTORY:
*06.03.2003 Rev.1.0 CM-DI/ESA1-Fischer
*           initial revision
*****************************************************************************/
tS32 scd_s32DeleteQueue (tU16 u16AppId)
{
   tChar szName[SCD_MAX_STRING_LENGTH] = "";
   tS32  s32RetVal;

   // create mailboxname
   if ((s32RetVal = OSALUTIL_s32SaveNPrintFormat((tString)szName,
                                                 sizeof(szName),
                                                 "mbx_%i",
                                                 u16AppId)) != OSAL_ERROR)
   {
     //delete mailbox
     if ( (s32RetVal = OSAL_s32MessageQueueDelete(szName)) == OSAL_ERROR )
     {
        scd_vTraceMsg(TR_LEVEL_ERRORS,"scd_s32DeleteQueue: OSAL_s32MessageQueueDelete failed for AppId 0x%04X", u16AppId);
     }/*if*/
     else
     {
        scd_vTraceMsg(TR_LEVEL_USER_1,"scd_s32DeleteQueue: OSAL_s32MessageQueueDelete success for AppId 0x%04X", u16AppId);
     }/*else*/
   }
   else
   {
     NORMAL_M_ASSERT_ALWAYS();
   }

   return(s32RetVal);
}/*scd_s32DeleteQueue*/

} // extern "C"

/******************************************************************************
*FUNCTION:    bGetValue
*DESCRIPTION: gets configuration-value from registry
*PARAMETER:   (I)   u16AppId:          unique application-identifier
*             (->I) rfoRegKey          reg-object
*             (->O) pu32Value          put data here
*             (->I) coszValue          first key

*
*RETURNVALUE: tBool: TRUE : OK
*                    FALSE: ERROR
*
*HISTORY:
*12.09.2003 Rev.1.0 CM-DI/ESA1-Fischer
*           initial revision
*****************************************************************************/
static tBool bGetValue( tU16 u16AppId, reg_tclRegKey &rfoRegKey, tPU32 pu32Value, const tChar *coszValue )
{
   if ( !rfoRegKey.bQueryU32 (coszValue, pu32Value) )
   {
      scd_vTraceMsg(TR_LEVEL_ERRORS,"scd_s32GetThreadConfiguration: failed to find key %s in app 0x%04X", coszValue, u16AppId);
      return (FALSE);
   }

   return (TRUE);
}

/******************************************************************************
*FUNCTION:    bGetValue
*DESCRIPTION: gets configuration-value from registry
*PARAMETER:   (I)   u16AppId:          unique application-identifier
*             (->I) rfoRegKey          reg-object
*             (->O) pu32Value          put data here
*             (->I) coszValue          first key
*             (->I) coszSecondValue    additional second key

*
*RETURNVALUE: tBool: TRUE : OK
*                    FALSE: ERROR
*
*HISTORY:
*12.09.2003 Rev.1.0 CM-DI/ESA1-Fischer
*           initial revision
*****************************************************************************/
static tBool bGetValue( tU16 u16AppId, reg_tclRegKey &rfoRegKey, tPU32 pu32Value, tCString coszFirstValue, tCString coszSecondValue )
{
   tChar szTempName[REG_C_U32_KEYNAME_MAXLEN];

   if (OSALUTIL_s32SaveNPrintFormat((tString)szTempName,
                                    sizeof(szTempName),
                                    "%s_%s", 
                                    coszFirstValue, 
                                    coszSecondValue) != OSAL_ERROR)
   {
     if ( !rfoRegKey.bQueryU32 (szTempName, pu32Value) )
     {
        scd_vTraceMsg(TR_LEVEL_ERRORS,"scd_s32GetThreadConfiguration: failed to find key %s in app 0x%04X", szTempName, u16AppId);
        return (FALSE);
     }
   }
   else
   {
     NORMAL_M_ASSERT_ALWAYS();
     return (FALSE);
   }

   return (TRUE);
}

extern "C"
{

/******************************************************************************
*FUNCTION:    scd_s32GetThreadConfiguration
*DESCRIPTION: returns the configuration for the given threadname
*PARAMETER:   (I)   u16AppId:       unique application-identifier
*             (->I) coszName:       thread-name (same as the parameter
*                                   given to OSAL_ThreadCreate)
*             (->O) pu32Priority:   thread-priority
*             (->O) ps32Stacksize:  stack-size

*
*RETURNVALUE: tS32: OSAL_OK
*                   OSAL_ERROR
*
*HISTORY:
*28.03.2003 Rev.1.0 CM-DI/ESA1-Fischer
*           initial revision
*****************************************************************************/
tS32 scd_s32GetThreadConfiguration(
   tU16 u16AppId,
   tCString coszName,
   tPU32 pu32Priority,
   tPS32 ps32Stacksize,
   OSAL_tMSecond *pWdgInterval,
   OSAL_tMSecond *pMaxWdgInterval
)
{
   tS32 s32RetVal = OSAL_ERROR;
   tBool bAppFound = FALSE;
   scd_vTraceMsg(TR_LEVEL_USER_4,"scd_s32GetThreadConfiguration for app 0x%04X and thread %s called",
      u16AppId, coszName);

  //-- get application entry point to the registry
  reg_tclApplication* poApp = scd_poGetApp(u16AppId);

  if(poApp != NULL)
  {
      // reg-entry for application found
      bAppFound = TRUE;
      reg_tclRegKey oRegKey;
      reg_tclApplication oApp;

      if ( oRegKey.bOpen(*poApp, REGKEY_APP_THREAD) ) // open key APP_THREAD in application
      {
         if (  bGetValue( u16AppId, oRegKey, pu32Priority, coszName, REGSTRING_PRIO )
            && bGetValue( u16AppId, oRegKey, (tPU32)ps32Stacksize, coszName, REGSTRING_STCK ) )
         {
            if (  pWdgInterval
               && pMaxWdgInterval
               && oApp.bOpen(*poApp, REGKEY_APP_CONFIG) ) // open key APP_CONFIG
            {
               tU32 u32MSecsToTics=0, u32WdgIntervall=0, u32MaxWdgIntervall=0;
               if (  bGetValue( u16AppId, oApp, &u32MSecsToTics, REGVALUE_WDG_MSPERTICK )
                  && bGetValue( u16AppId, oRegKey, &u32WdgIntervall, coszName, REGVALUE_WTICKS )
                  && bGetValue( u16AppId, oRegKey, &u32MaxWdgIntervall, coszName, REGVALUE_MAXWTICKS ) )
               {
                  *pWdgInterval = (OSAL_tMSecond)(u32MSecsToTics*u32WdgIntervall);
                  *pMaxWdgInterval = (OSAL_tMSecond)(u32MSecsToTics*u32MaxWdgIntervall);
                  s32RetVal = OSAL_OK;
                  scd_vTraceMsg(TR_LEVEL_SYSTEM_MIN,"scd_s32GetThreadConfiguration: app 0x%04X name %s prio %d stc %d wdg %d Int %d", u16AppId, coszName, *pu32Priority, *ps32Stacksize, *pWdgInterval, *pMaxWdgInterval);
               }
               else
               {
                   scd_vTraceMsg(TR_LEVEL_ERRORS,"scd_s32GetThreadConfiguration: app 0x%04X name %s prio %d stc %d does not found Watchdog Informations in Registry", u16AppId, coszName, *pu32Priority, *ps32Stacksize);
               }
            }
            else
            {
               scd_vTraceMsg(TR_LEVEL_SYSTEM_MIN,"scd_s32GetThreadConfiguration: app 0x%04X name %s prio %d stc %d wdg NULL Int NULL", u16AppId, coszName, *pu32Priority, *ps32Stacksize);
               s32RetVal = OSAL_OK;
            }
         }
         else
         {
           scd_vTraceMsg(TR_LEVEL_ERRORS,"scd_s32GetThreadConfiguration: the configuration for thread %s in app 0x%04X could not be found in Registry", coszName, u16AppId);
         }
      } // end if open REGKEY_APP_THREAD
      else
      {
         scd_vTraceMsg(TR_LEVEL_ERRORS,"scd_s32GetThreadConfiguration: failed to find REGKEY_APP_THREAD for app 0x%04X", u16AppId);
      }
  }

   if ( !bAppFound )
   {
      scd_vTraceMsg(TR_LEVEL_ERRORS,"scd_s32GetThreadConfiguration: no entry for app 0x%04X found", u16AppId);
   }

   return(s32RetVal);
}


/**
 * check if an application's system registry is available
 * (will check for existing AppID in registry only)
 *
 * @param   u16AppID Application ID to be searched

 * @return  true, if key "APPID" was found in corresponding application folder, false otherwise
 *
 * @author CM-DI/ESP3-Friedrichs
 * @date   2005-19-13
 */

tBool scd_bAppRegistryAvailable (tU16 u16AppID)
{
   tU32 u32DummyValue;
   tBool bKeyFound = scd_bGetAppConfigurationValue(u16AppID, "", REGVALUE_APPID, &u32DummyValue);
   return bKeyFound;
}


/**
 * reads the SWBlockID for a given AppID
 *
 * @param   u16AppID        Application ID to be searched
 * @param   pu32SWBlockID   Return-Value for SWBlockID
 *
 * @return  TRUE    if SWBlockID for given AppID was found, otherwise FALSE
 *
 * @author CM-DI/PJ-CF11-Wiedemann
 * @date   04.09.2007
 */
 
tBool scd_bGetSWBlockID(tU16 u16AppID, tU32* pu32SWBlockID)
{
   tBool bResult = FALSE;

   scd_vTraceMsg(TR_LEVEL_USER_4,"scd_bGetSWBlockID for app 0x%04X called", u16AppID);

   if(pu32SWBlockID != NULL)
   {
      // get Application entry
      const reg_tclApplication* poApp = scd_poGetApp(u16AppID);

      if(poApp != NULL)
      {
         // open new regkey
         reg_tclRegKey oKey(*poApp);
         if(oKey.bNavigateKeyNameUp())
         {
            if (oKey.bQueryU32((const tChar*) REGVALUE_SWBLOCKID, pu32SWBlockID) == TRUE)
            {
               scd_vTraceMsg(TR_LEVEL_SYSTEM_MIN,"scd_bGetSWBlockID: find REGVALUE_SWBLOCKID for app 0x%04X = %d", u16AppID, *pu32SWBlockID);
               bResult = TRUE;
            }
            else
            {
               scd_vTraceMsg(TR_LEVEL_ERRORS,"scd_bGetSWBlockID: failed to find REGVALUE_SWBLOCKID for app 0x%04X", u16AppID);
            }
         }
         else
         {
            scd_vTraceMsg(TR_LEVEL_ERRORS,"scd_bGetSWBlockID: failed navigate SWBLOCK");
         }
      }
      else
      {
         scd_vTraceMsg(TR_LEVEL_ERRORS,"scd_bGetSWBlockID: app 0x%04X not found", u16AppID);
      }
   }
   else
   {
      scd_vTraceMsg(TR_LEVEL_ERRORS,"scd_bGetSWBlockID: invalid pointer pu32SWBlockID");
   }

   return bResult;
}

}
