/*****************************************************************************
 * FILE:         reg_BR211Iterator.h
 *
 * PROJECT:      VASCO
 *
 * SW-COMPONENT: REG (completely OSAL based module)
 *
 *----------------------------------------------------------------------------
 * DESCRIPTION:  Encapsulation of Blaupunkt's Registry
 *
 *               The registry is needed for initialisation of
 *               message queues at system setup (see scd-library).
 *
 * REMARK:       
 *
 *----------------------------------------------------------------------------
 * COPYRIGHT:    (c) 2000 Robert Bosch GmbH, Hildesheim
 * HISTORY:      
 * Date      |   Modification                         | Author
 * 03.04.01  |   Initial revision                     | Bruns,EFG32
 *****************************************************************************/

#ifndef REG_ITERATOR_H
   #define REG_ITERATOR_H

//-----------------------------------------------------------------------------
// class "reg_tclSubKeyIterator"
//
// iterates subkeys of a given registry key
//-----------------------------------------------------------------------------
template <class T>
class reg_tclSubKeyIterator
{
   protected:

      T oCurrentKey;

      // descriptor
      OSAL_trIOCtrlDir *pDir;

      // "path" to "pDir"; This is the base key.
      tChar szCurrentKeyName[REG_C_U32_KEYNAME_MAXLEN]; // current keyname

      // actual relative entry (== current key)
      OSAL_trIOCtrlDirent *pDirent;


      reg_tclSubKeyIterator() {};

   public:

      // constructor
      reg_tclSubKeyIterator(const reg_tclRegKey& poBaseKey);
      // constructor
      reg_tclSubKeyIterator(const tChar *coszKeyName);
      // destrcutor
      virtual ~reg_tclSubKeyIterator();

      reg_tclSubKeyIterator& operator++();

      T& operator*();

      tBool bSetAt  (const tChar *coszOffset);
      tVoid vFirst  (tVoid);
      tVoid vNext   (tVoid);
      tBool bIsDone (tVoid) const;
      T&    oItem   (tVoid);

   private:
      // avoid copy
      reg_tclSubKeyIterator(const reg_tclSubKeyIterator &corfoSubKeyIterator);
      // avoid assignment
      reg_tclSubKeyIterator& operator=(const reg_tclSubKeyIterator&);

};


typedef reg_tclSubKeyIterator<reg_tclRegKey>     reg_tclStdKeyIterator;



template <class T>
reg_tclSubKeyIterator<T>::reg_tclSubKeyIterator(const reg_tclRegKey& poBaseKey)
{
   pDir = OSALUTIL_prOpenDir(poBaseKey.cpGetClassKeyName());
   if (pDir)
   {
      tU32 u32StringLength = (tU32)OSAL_u32StringLength(poBaseKey.cpGetClassKeyName());
      u32StringLength = u32StringLength>(REG_C_U32_KEYNAME_MAXLEN-1)?(REG_C_U32_KEYNAME_MAXLEN-1):u32StringLength;
      (void)OSAL_szStringNCopy(szCurrentKeyName, poBaseKey.cpGetClassKeyName(), u32StringLength); //lint !e522: Expected void type, assignment, increment or decrement
      szCurrentKeyName[u32StringLength]='\0';
      pDirent = OSALUTIL_prReadDir(pDir);
   }
   else
   {
      szCurrentKeyName[0] = '\0';
      pDirent = OSAL_NULL;
   }
}

template <class T>
reg_tclSubKeyIterator<T>::reg_tclSubKeyIterator(const tChar *coszKeyName)
{
   pDir = OSALUTIL_prOpenDir(coszKeyName);
   if (pDir)
   {
      tU32 u32StringLength = (tU32)OSAL_u32StringLength(coszKeyName);
      u32StringLength = u32StringLength>(REG_C_U32_KEYNAME_MAXLEN-1)?(REG_C_U32_KEYNAME_MAXLEN-1):u32StringLength;
      (void)OSAL_szStringNCopy(szCurrentKeyName, coszKeyName, u32StringLength); //lint !e522: Expected void type, assignment, increment or decrement
      szCurrentKeyName[u32StringLength]='\0';
      pDirent = OSALUTIL_prReadDir(pDir);
   }
   else
   {
      szCurrentKeyName[0] = '\0';
      pDirent = OSAL_NULL;
   }
}

template <class T>
reg_tclSubKeyIterator<T>::~reg_tclSubKeyIterator()
{
   if (pDir)
   {
      OSALUTIL_s32CloseDir(pDir); //lint !e522: Expected void type, assignment, increment or decrement
      pDir = OSAL_NULL;
   }
   pDirent = OSAL_NULL; // pDirent is a pointer to a structure in pDir, reset it to OSAL_NULL
   szCurrentKeyName[0] = '\0';
}

template <class T>
reg_tclSubKeyIterator<T>& reg_tclSubKeyIterator<T>::operator++()
{
   this->vNext();
   return *this;
}


template <class T>
T& reg_tclSubKeyIterator<T>::operator*()
{
   return this->oItem();
}

template <class T>
tBool reg_tclSubKeyIterator<T>::bSetAt(const tChar *coszOffset)
{
   if (pDir)
   {
      OSALUTIL_s32CloseDir(pDir); //lint !e522: Expected void type, assignment, increment or decrement
      pDir = OSAL_NULL;
   }
   pDir = OSALUTIL_prOpenDir(coszOffset);
   if (pDir)
   {
      tU32 u32StringLength = (tU32)OSAL_u32StringLength(coszOffset);
      u32StringLength = u32StringLength>(REG_C_U32_KEYNAME_MAXLEN-1)?(REG_C_U32_KEYNAME_MAXLEN-1):u32StringLength;
      (void)OSAL_szStringNCopy(szCurrentKeyName, coszOffset, u32StringLength); //lint !e522: Expected void type, assignment, increment or decrement
      szCurrentKeyName[u32StringLength]='\0';
      pDirent = OSALUTIL_prReadDir(pDir);
      return TRUE;
   }
   else
   {
      szCurrentKeyName[0] = '\0';
      pDirent = OSAL_NULL;
      return FALSE;
   }
}


template <class T>
tVoid  reg_tclSubKeyIterator<T>::vFirst(tVoid)
{
   this->bSetAt(szCurrentKeyName);
}

template <class T>
tVoid reg_tclSubKeyIterator<T>::vNext()
{
   if (pDir)
   {
      pDirent = OSALUTIL_prReadDir(pDir);
   }
}


template <class T>
tBool reg_tclSubKeyIterator<T>::bIsDone(tVoid) const
{
   if (pDir == OSAL_NULL || pDirent == OSAL_NULL)
   {
      return TRUE;
   }
   else
      return FALSE;
}


template <class T>
T& reg_tclSubKeyIterator<T>::oItem(tVoid)
{
   if (pDir && pDirent)
   {
      tChar  szTempName[REG_C_U32_KEYNAME_MAXLEN + 1];         
      OSALUTIL_s32NPrintFormat((tPS8)szTempName, REG_C_U32_KEYNAME_MAXLEN, "%s/%s", szCurrentKeyName, (tPChar)(pDirent->s8Name));
      szTempName[REG_C_U32_KEYNAME_MAXLEN] = 0;
      oCurrentKey.bSetCurrentKeyName(szTempName);
   }
   return oCurrentKey; //lint !e1536: Exposing low access member (need access in operator*)
}



//-----------------------------------------------------------------------------
// class "reg_tclValueIterator"
//
// iterates values of a given registry key
//-----------------------------------------------------------------------------
class reg_tclValueIterator
{
   protected:

      // descriptor
      OSAL_trIOCtrlDir *pDir;

      // "path" to "pDir"; This is the base key.
      tChar szCurrentKeyName[REG_C_U32_KEYNAME_MAXLEN];

      // actual relative entry (== current key)
      OSAL_trIOCtrlDirent *pDirent;


      reg_tclValueIterator() :pDir(OSAL_NULL),pDirent(OSAL_NULL) {*szCurrentKeyName=0;};

   public:

      // constructor
      reg_tclValueIterator(const reg_tclRegKey& poBaseKey);
      reg_tclValueIterator(const tChar *coszKeyName);

      // destrcutor
      ~reg_tclValueIterator();

      reg_tclValueIterator& operator++();

      tChar* operator*();

      tVoid vFirst(tVoid);
      tVoid vNext(tVoid);
      tBool bIsDone(tVoid) const;

      tChar* szValueName(tVoid);

      // queries for value/string 
      tBool bGetU32    (tU32 *rdwResult) const;
      tBool bGetString (tChar *rdwResult, tU32 u32MaxBufLen) const;

   private:
      // avoid copy
      reg_tclValueIterator(const reg_tclValueIterator &corfoValueIterator);
      // avoid assignment
      reg_tclValueIterator& operator=(const reg_tclValueIterator&);
};

class reg_tclProcessIterator : public reg_tclSubKeyIterator<reg_tclProcess>
{
public:
   reg_tclProcessIterator();
};

class reg_tclAppConfigIterator : public reg_tclSubKeyIterator<reg_tclAppConfig>
{
public:
   reg_tclAppConfigIterator();
};


class reg_tclApplicationIterator : public reg_tclSubKeyIterator<reg_tclApplication>
{
public:
   reg_tclApplicationIterator(const reg_tclProcess &rfcooOrigin);
private:
   // declare private default-ctor to avoid uninitialized objects
   reg_tclApplicationIterator(); //lint !e1704: Constructor has private access specification
};


class reg_tclServiceIterator : public reg_tclSubKeyIterator<reg_tclService>
{
public:
   reg_tclServiceIterator(const reg_tclApplication &rfcooOrigin);
private:
   // declare private default-ctor to avoid uninitialized objects
   reg_tclServiceIterator(); //lint !e1704: Constructor has private access specification
};

#endif //REG_ITERATOR_H
