///////////////////////////////////////////////////////////////////////////////
// FILE:         fwl_list.h
// PROJECT:      VASCO
//-----------------------------------------------------------------------------
// DESCRIPTION:  framework library implementation of list container
//-----------------------------------------------------------------------------
// COPYRIGHT:    (c) 2000 Robert Bosch GmbH, Hildesheim
// HISTORY:      
// Date      |   Modification                         | Author
// xx.11.00  |   Initial revision                     | M.Gibron
// 08.01.00  |   Introduced proper module prefix and  | C.Bruns, EFG32
//               first check-in
//               
///////////////////////////////////////////////////////////////////////////////

#ifndef __LIST_CONTAINER__
#define __LIST_CONTAINER__

template <class Type> class fwl_ListIterator;

///////////////////////////////////////////////////////////////////////////////
//
// class: fwl_ListNode<Type>
//
// This structure represents the nodes contained in a dynamically linked list.
//
///////////////////////////////////////////////////////////////////////////////

template <class Type>
class fwl_ListNode
{
 public:
    fwl_ListNode (const Type& item, fwl_ListNode* prev, fwl_ListNode* next);
    
    fwl_ListNode* _prev;
    fwl_ListNode* _next;
    Type          _item;
 private:
    // declare private default-ctor to avoid uninitialized objects
    fwl_ListNode(); //lint !e1704 Constructor has private access specification
};

///////////////////////////////////////////////////////////////////////////////
//
// Class: fwl_List<Type>
//
// The class models a dynamically linked list, that allocates its items on the
// heap. This class provides only basic functionality, that will be refined in
// subclasses like Set<Type> or Bag<Type>.
//
///////////////////////////////////////////////////////////////////////////////

template <class Type>
class fwl_List
{
   friend class fwl_ListIterator<Type>;
   
public:
   fwl_List (tVoid);
   virtual ~fwl_List (tVoid);
   
   tBool bRemoveFirst(tVoid);
   tVoid vClear(tVoid);
   
   virtual tVoid vAdd (const Type& item);
   // definition moved to class-declaration 'cause of compile-errors in navif_lib with nucleus-compiler
   virtual tVoid nRemove (const Type& item)
   {
      // point to first element in list
      fwl_ListNode<Type>* help = _head;
      // search element in list
      while
         (
         (help != OSAL_NULL)
         && (help->_item != item)
         )
      {
         help = help->_next;
      }
      if ( help != OSAL_NULL )
      {
         // element found, update linkage between successor and predecessor
         vRemove( help );
      }
   };
   virtual tS32  nSize (tVoid) const;
   
private:
   virtual tVoid vRemove( fwl_ListNode<Type>*& rfpListNode )
   {
      if ( rfpListNode != OSAL_NULL )
      {
         // element found, update linkage between successor and predecessor
         fwl_ListNode<Type>* last = rfpListNode->_prev;
         fwl_ListNode<Type>* next = rfpListNode->_next;
         
         if (last == OSAL_NULL )
         {
            // element to delete is head, set head to successor of head
            _head = next;
            // no access to iterator class "_current = _head" possible!
            // -> don't forget iterator becomes invaild immediately - udo -
         }
         else
         {
            last->_next = next; 
            // no access to iterator class "_current = last" possible!
            // -> don't forget iterator becomes invaild immediately - udo -
         }
         if ( next == OSAL_NULL )
         {
            // element to delete is tail, set tail to predecessor of tail
            _tail = last;
         }
         else
         {
            next->_prev = last;
         }
         _size--;
         OSAL_DELETE rfpListNode; // added by - udo -
         rfpListNode = OSAL_NULL;
      }
   }

   fwl_ListNode<Type>* _head;
   fwl_ListNode<Type>* _tail;
   tS32 _size;
};


///////////////////////////////////////////////////////////////////////////////
//
// Class: Iterator
//
// Abstract base class, that defines the interface for all Iterator subclasses.
//
///////////////////////////////////////////////////////////////////////////////

template <class Type>
class fwl_Iterator
{
public:
   virtual tVoid vFirst (tVoid) = 0;
   virtual tVoid vNext  (tVoid) = 0;
   virtual tBool bIsDone(tVoid) = 0;
   virtual Type& oItem  (tVoid) = 0;
   
protected:
   fwl_Iterator();
   virtual ~fwl_Iterator() {};
};

///////////////////////////////////////////////////////////////////////////////
//
// Class: fwl_ListIterator
//
// This is a special kind of iterator, that knows how to traverse List objects.
//
///////////////////////////////////////////////////////////////////////////////

template <class Type>
class fwl_ListIterator : public fwl_Iterator<Type>
{
public:
   fwl_ListIterator (fwl_List<Type>* list);
   
   virtual tVoid  vFirst  (tVoid);
   virtual tVoid  vNext   (tVoid);
   virtual tBool  bIsDone (tVoid);
   virtual Type&  oItem   (tVoid); // oItem returns a reference to the current element of the list, which MUST be non-empty
   
   /////////////////////////////////////////////////////////////////////////////
   //
   // Die folgenden Funktionen gewaehren den von dieser Klasse abgeleiteten 
   // Klassen den Zugriff auf die Interna der List-Klasse, ohne dass sie
   // explizit in der Klasse Stack als 'friend' deklariert werden muessen.
   //
   /////////////////////////////////////////////////////////////////////////////
protected:
   
   tS32   Size (tVoid) const;
   tVoid  Forward (tVoid);
   tVoid  Backward (tVoid);
   tVoid  Head (tVoid);
   tVoid  Tail (tVoid);
   Type&  GetItem (tVoid);
   
private:
   // declare private default-ctor to avoid uninitialized objects
   fwl_ListIterator (); //lint !e1704 Constructor has private access specification
   
   fwl_List<Type>*     _container; // Reference to the container
   fwl_ListNode<Type>* _current;   // Current position of iterator
};

///////////////////////////////////////////////////////////////////////////////
//
// Class: ReverseListIterator
//
// Spezielle Ausprgung eines Listeniterators, der die Liste rckwrts durch-
// schreitet.
//
// Author: Michael Gibron
// Date  : 12.05.2000
//
///////////////////////////////////////////////////////////////////////////////

template <class Type>
class ReverseListIterator : public fwl_ListIterator<Type>
{
public:
   ReverseListIterator (fwl_List<Type>* list);
   
   tVoid  vFirst  (tVoid);
   tVoid  vNext   (tVoid);
private:
   // declare private default-ctor to avoid uninitialized objects
   ReverseListIterator (); //lint !e1704 Constructor has private access specification
};

///////////////////////////////////////////////////////////////////////////////
//
// Vorbergehende Einbindung der Implementierungsdatei
//
///////////////////////////////////////////////////////////////////////////////

#include <string.h>
#include <stdlib.h>

///////////////////////////////////////////////////////////////////////////////
//
// Konstruktor fr ListNode-Objekte
//
///////////////////////////////////////////////////////////////////////////////

template <class Type>
fwl_ListNode<Type>::fwl_ListNode
(
   const Type&   item,
   fwl_ListNode* prev,
   fwl_ListNode* next
) 
 : _prev(prev)
 , _next(next)
 , _item(item)
{}

///////////////////////////////////////////////////////////////////////////////
//
// Listen-Konstruktor
//
///////////////////////////////////////////////////////////////////////////////

template <class Type>
fwl_List<Type>::fwl_List (tVoid) 
 : _head(OSAL_NULL)
 , _tail(OSAL_NULL)
 , _size(0) 
{
   //... empty ...
}

template <class Type>
fwl_List<Type>::~fwl_List (tVoid)
{
   vClear(); //lint !e1551
   //... now empty ...
   _tail = OSAL_NULL;
   _head = OSAL_NULL;
   
} 

template <class Type>
tBool fwl_List<Type>::bRemoveFirst (tVoid)
{
   tBool bRemove = FALSE;
   // helper on current head
   fwl_ListNode<Type>* help = _head;
   
   if (OSAL_NULL != help)
   {
      // head points to successor
      _head = help->_next;
      // successor valid?
      if(OSAL_NULL != _head)
      {
         // reset predecessor of 'new' head 
         help->_next->_prev = OSAL_NULL;
      }
      else
      {
         // no successor --> list empty
         _tail = OSAL_NULL;
         _head = OSAL_NULL;
      }
      OSAL_DELETE help;
      help = OSAL_NULL;
      bRemove = TRUE;
   }
   return bRemove;
}

template <class Type>
tVoid fwl_List<Type>::vClear (tVoid)
{
   while( bRemoveFirst() ){};
}

template <class Type>
tVoid fwl_List<Type>::vAdd (const Type& item)
{
   if (_size == 0)
   {
      fwl_ListNode<Type>* help = OSAL_NEW fwl_ListNode<Type> (
         item,
         OSAL_NULL,
         OSAL_NULL
         );
      if ( help != OSAL_NULL )
      {
         _tail = _head = help;
         ++_size;
      }
   } 
   else
   {
      fwl_ListNode<Type>* help = _tail;
      
      if ( OSAL_NULL != help )
      {
         _tail = OSAL_NEW fwl_ListNode<Type>(item,_tail,OSAL_NULL);
         
         if ( OSAL_NULL != _tail )
         {
            help->_next = _tail;
            ++_size;
         }
         else
         {
            _tail = help;
         }
      }
   }
}

template <class Type>
tS32 fwl_List<Type>::nSize (tVoid) const
{
   return _size;
}


template <class Type>
fwl_Iterator<Type>::fwl_Iterator (tVoid)
{}

///////////////////////////////////////////////////////////////////////////////
//
// Iterator-Klasse zur Iteration durch Stack-Container-Klassen
//
///////////////////////////////////////////////////////////////////////////////

template <class Type>
fwl_ListIterator<Type>::fwl_ListIterator (fwl_List<Type>* liste)
 : _container(liste)
 , _current(OSAL_NULL)
{
   if ( OSAL_NULL != _container )
   {
      _current = _container->_head;
   }
}

template <class Type>
tVoid fwl_ListIterator<Type>::vFirst (tVoid)
{
   if ( OSAL_NULL != _container )
   {
      _current = _container->_head;
   }
   else
   {
      _current = OSAL_NULL;
   }
}

template <class Type>
tVoid fwl_ListIterator<Type>::vNext (tVoid)
{
   if ( OSAL_NULL != _current )
   {
      _current = _current->_next;
   }
}

template <class Type>
tBool fwl_ListIterator<Type>::bIsDone (tVoid)
{
   //... Zugriff auf den privaten Bereich des Containers ...
   return (_current == OSAL_NULL) ? TRUE : FALSE;
}

// oItem returns a reference to the current element of the list, which MUST be non-empty
// check on 'bIsDone==FALSE' before call of oItem
template <class Type>
Type& fwl_ListIterator<Type>::oItem (tVoid)
{
   OSAL_vAssert(OSAL_NULL != _current);
   return (_current->_item); //lint !e613 Possible use of null pointer, CM-DI/ESP2-Fischer, 2006-09-06: bIsDone==FALSE is checked before call of oItem in whole components-folder (proved on 2006-09-06)
}

template <class Type>
tS32 fwl_ListIterator<Type>::Size (tVoid) const
{
   tS32 s32Size = 0;
   
   if ( OSAL_NULL != _container )
   {
      s32Size = _container->_size;
   }
   
   return s32Size;
}

template <class Type>
tVoid fwl_ListIterator<Type>::Head (tVoid)
{
   if ( OSAL_NULL != _container )
   {
      _current = _container->_head;
   }
}

template <class Type>
tVoid fwl_ListIterator<Type>::Tail (tVoid)
{
   if ( OSAL_NULL != _container )
   {
      _current = _container->_tail;
   }
}

template <class Type>
tVoid fwl_ListIterator<Type>::Forward (tVoid)
{
   if ( OSAL_NULL != _current )
   {
      _current = _current->_next;
   }
}

template <class Type>
tVoid fwl_ListIterator<Type>::Backward (tVoid)
{
   if ( OSAL_NULL != _current )
   {
      _current = _current->_prev;
   }
}

///////////////////////////////////////////////////////////////////////////////
//
// ReverseListIterator
//
///////////////////////////////////////////////////////////////////////////////

template <class Type>
ReverseListIterator<Type>::ReverseListIterator (fwl_List<Type>* liste)
 : fwl_ListIterator<Type>(liste)
{
   fwl_ListIterator<Type>::Tail();
}

template <class Type>
tVoid ReverseListIterator<Type>::vFirst (tVoid)
{
   fwl_ListIterator<Type>::Tail();
}

template <class Type>
tVoid ReverseListIterator<Type>::vNext (tVoid)
{
   fwl_ListIterator<Type>::Backward();
}

#endif //__LIST_CONTAINER__
