/******************************************************************************
 * FILE:          ail_basics.c
 * SW-COMPONENT:  AIL
 * DESCRIPTION:  
 * AUTHOR:        
 * COPYRIGHT:     (c) 2001 Robert Bosch GmbH, Hildesheim
 * HISTORY:      
 ******************************************************************************/

/******************************************************************************
| includes                                                                    
|   1)system- and project- includes
|   2)needed interfaces from external components
|   3)internal and external interfaces from this component 
|-----------------------------------------------------------------------------*/

// regular includes
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#ifndef TRACE_S_ALREADY_INCLUDE_TYPES
#define TRACE_S_IMPORT_INTERFACE_TYPES
#include "trace_if.h"
#endif

#define SCD_S_IMPORT_INTERFACE_GENERIC
#include "scd_if.h"
#define AIL_S_IMPORT_INTERFACE_GENERIC
#define AIL_S_IMPORT_INTERFACE_TRACE
#include "ail_if.h"

/******************************************************************************
| defines and macros (scope: modul-local)
|-----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: global)
|-----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: modul-local)
|-----------------------------------------------------------------------------*/

/******************************************************************************
| function prototype (scope: modul-local)
|-----------------------------------------------------------------------------*/
#ifdef AIL_MESSAGE_COUNTER_ENABLED
tVoid AddCCAInfo(const amt_tclBaseMessage& oMsgObject, tBool bSend); 
#endif
/******************************************************************************
| function implementation (scope: modul-local)
|-----------------------------------------------------------------------------*/

/******************************************************************************
| function implementation (scope: global)
|-----------------------------------------------------------------------------*/


/*******************************************************************************
 *FUNCTION:    ail_bIpcMessagePost
 *DESCRIPTION: 
 *   This function posts a message with a given priority into a specified 
 *   message queue.
 *   ATTENTION: Message will not be delete if posting it has failed!!
 *PARAMETER:   
 *   tU16 u16MyAppId (->I)
 *   identifier of calling application  
 *
 *   OSAL_tMQueueHandle hMQ (->I)
 *   handle of queue to post in
 *
 *   amt_tclBaseMessage* poMsgObject (->I)
 *   message to be sent
 * 
 *   tU32 u32Prio (I)
 *   priority with which message has to be sent
 *                
 *RETURNVALUE:
 *   tBool
 *       TRUE     message was sent successfully
 *       FALSE    message could not be sent
 *
 *HISTORY:
 ******************************************************************************/
tU32 ail_bIpcMessagePost(tU16 u16MyAppId, OSAL_tMQueueHandle hMQ, amt_tclBaseMessage* poMsgObject, tU32 u32Prio)
{
   tU32 u32OsalErrorCode = OSAL_E_NOERROR;

   if (ail_bIsTraceActive(TR_LEVEL_ERRORS))
   {
      // get current used poolspace
      TR_tenTraceLevel enTraceLevel = TR_LEVEL_USER_4;
      tS32 s32AbsolutePoolSpace = OSAL_s32MessagePoolGetAbsoluteSize();
      tS32 s32CurrentPoolSpace = OSAL_s32MessagePoolGetCurrentSize();
      tU8 u8MsgBufferInUse = 0;
   
      if ( s32AbsolutePoolSpace > 0 )
      {
         u8MsgBufferInUse = (tU8)(((tU8)100) - ((tU8)((s32CurrentPoolSpace * 100) / s32AbsolutePoolSpace)));
   
         if ( u8MsgBufferInUse > AIL_C_U8_TRACE_THRESHOLD_MSG_PRINT )
         /* more than xx% poolspace actually used */
         {
            enTraceLevel = TR_LEVEL_ERRORS;
         }
      }
   
      if (  u8MsgBufferInUse > AIL_C_U8_TRACE_THRESHOLD_MSG_PRINT
            #ifdef AIL_MESSAGE_TRACE_EACH_MESSAGE
                     && !ail_bIsTraceActive( TR_LEVEL_USER_4 )
            #endif
            )
      /* print out SendInfo without raw message data, if too much poolspace is used && TR_LEVEL_USER_4 isn't enabled */
      {
         ail_vTraceMsg( enTraceLevel,
            "Application 0x%04x,%d: Posting %d bytes to Queue 0x%08X (App 0x%04x, SubId%d), %d percent of MsgPool used, poolspace %d\n",
            u16MyAppId,
            poMsgObject->u16GetSourceSubID(),
            poMsgObject->u32GetSize(),
            hMQ,
            poMsgObject->u16GetTargetAppID(),
            poMsgObject->u16GetTargetSubID(),
            u8MsgBufferInUse,
            s32CurrentPoolSpace );
      }
      else
      /* print out with raw message data, if SendInfo isn't printed out by force and the requested TraceLevel is active */
      {
        #ifdef AIL_MESSAGE_TRACE_EACH_MESSAGE
        if (ail_bIsTraceActive(enTraceLevel))
        {
               ail_vTraceHexMsg( TR_CLASS_AIL, enTraceLevel, FALSE,
                  poMsgObject->u32GetSize(), poMsgObject->pu8GetSharedMemBase(),
                  "Application 0x%04x,%d: Posting %d bytes to Queue 0x%08X (AppId 0x%04x, SubId %d), %d percent of MsgPool used, poolspace %d, Raw Data:",
                  u16MyAppId,
                  poMsgObject->u16GetSourceSubID(),
                  poMsgObject->u32GetSize(),
                  hMQ,
                  poMsgObject->u16GetTargetAppID(),
                  poMsgObject->u16GetTargetSubID(),
                  u8MsgBufferInUse,
                  s32CurrentPoolSpace );
         }
        #endif
      }
   }

   if ( hMQ != OSAL_C_INVALID_HANDLE )
   {
      #ifdef AIL_MESSAGE_COUNTER_ENABLED
      AddCCAInfo(*poMsgObject, TRUE);
      #endif

      OSAL_trMessage* pMsg = poMsgObject->prGetOSALMsgHandle();

      if ((pMsg != NULL) && (pMsg->enLocation != OSAL_EN_MEMORY_INVALID))
      {
         // note: caller must use OSAL prios like OSAL_C_U32_MQUEUE_PRIORITY_HIGHEST
         if (OSAL_s32MessageQueuePost(hMQ, 
                                      (tCU8*)poMsgObject->prGetOSALMsgHandle(), 
                                      sizeof(OSAL_trMessage), 
                                      u32Prio) == OSAL_ERROR)
         {
            /* impossible to deliver the message */
            u32OsalErrorCode = OSAL_u32ErrorCode();
   
            ail_vTraceMsg( TR_LEVEL_ERRORS,
               "Application 0x%04x: Post to Queue 0x%08x has failed: error 0x%08X (%s)",
               u16MyAppId, hMQ, u32OsalErrorCode, ail_coszOSALError(u32OsalErrorCode) );
         }
      }
      else
      {
         u32OsalErrorCode = OSAL_EN_MEMORY_INVALID;
         ail_vTraceMsg( TR_LEVEL_FATAL, "Application 0x%04x: Application has try to send invalid message", u16MyAppId);
      }
   }
   else
   {
      /* it would be an osal fatal error condition */
      u32OsalErrorCode = OSAL_C_INVALID_HANDLE;
      ail_vTraceMsg( TR_LEVEL_ERRORS, "Application 0x%04x: Post to Queue 0x%08x has failed: invalid QueueHandle", u16MyAppId, hMQ );
   }

   return (u32OsalErrorCode);
}

/*******************************************************************************
 *FUNCTION:   ail_bIpcMessageWait
 *DESCRIPTION:
 *   This function waits for a message in a specified message queue.
 *
 *PARAMETER:   
 *   tU16 u16MyAppId (->I)
 *   identifier of calling application  
 *
 *   OSAL_tMQueueHandle hMQ (->I)
 *   handle of queue to wait for
 *
 *   amt_tclBaseMessage* poMsgObject (->I)
 *   message to be received
 * 
 *   tU32 *pu32Prio (->O)
 *   returns priority of the received message
 *
 *   OSAL_tMSecond rTimeout (I)
 *   timeout for message-queue-wait
 *                
 *RETURNVALUE:
 *   tBool
 *       TRUE     message received
 *       FALSE    no message received or message invalid
 *
 *HISTORY:
 ******************************************************************************/
tBool ail_bIpcMessageWait(tU16 u16MyAppId, OSAL_tMQueueHandle hMQ, amt_tclBaseMessage *poMsgObject, tU32 *pu32Prio, OSAL_tMSecond rTimeout)
{
   tS32 s32MsgCount, s32OSRet;

   s32MsgCount = OSAL_s32MessageQueueWait(hMQ, (tU8*)poMsgObject->prGetOSALMsgHandle(), sizeof(OSAL_trMessage), pu32Prio, rTimeout);

   if( s32MsgCount > 0 )
   /* valid size of received data */
   {
      poMsgObject->pu8SharedMemBase = OSAL_pu8MessageContentGet(*poMsgObject->prGetOSALMsgHandle(), OSAL_EN_READWRITE);

      if (poMsgObject->pu8SharedMemBase == OSAL_NULL)
      {
         s32OSRet = OSAL_ERROR;
      }
      else
      {
         // Copy real message size from shared-memory CCA byte-stream to the 'u32DynMsgSize' 
         // attribute of object amt_tclBaseMessage::amt_tclMappableMessage.
         poMsgObject->vSetDynMsgSize(poMsgObject->u32GetSize());

         s32OSRet = OSAL_OK;
         #ifdef AIL_MESSAGE_COUNTER_ENABLED
         AddCCAInfo(*poMsgObject, FALSE);
         #endif
      }
   }
   else
   {
      /* debug line to visual the error reason */
      tU32 u32ErrorReason = OSAL_u32ErrorCode();

      if (u32ErrorReason != OSAL_E_TIMEOUT)
      {
         ail_vTraceMsg( TR_LEVEL_ERRORS, "Application 0x%04x: Wait on Queue failed: error 0x%08X (%s)",
            u16MyAppId, u32ErrorReason, ail_coszOSALError(u32ErrorReason) );
   
         // provide error code for the application (could be modified by the trace)
         OSAL_vSetErrorCode(u32ErrorReason);
      }
      s32OSRet = OSAL_ERROR;
   }

   return ((s32OSRet == OSAL_ERROR) ? FALSE : TRUE);

}

/*******************************************************************************
 *FUNCTION:   
 *DESCRIPTION:
 *PARAMETER:   
 *                
 *RETURNVALUE:
 *
 *HISTORY:
 ******************************************************************************/
tBool ail_bConnectApplicationQueue(tU16 u16MyAppId, OSAL_tMQueueHandle* phOutQueue, tU16 u16DestinationAppId)
{
   tBool                   bOueueOpened = TRUE;
   OSAL_tMQueueHandle      hQueue;
   
   hQueue = scd_OpenQueue ( u16DestinationAppId );

   if ( hQueue == OSAL_C_INVALID_HANDLE )
   {
      bOueueOpened = FALSE;
   }
   else
   {
      *phOutQueue = hQueue;
   }

   ail_vTraceMsg( TR_LEVEL_SYSTEM, "Application 0x%04x: Queue %s to application 0x%04x",
      u16MyAppId, ( (bOueueOpened) ? "succesfully opened" : "opening failed"  ), u16DestinationAppId );

   return ( bOueueOpened );
}

/*******************************************************************************
 *FUNCTION:   ail_vIpcClearMessageQueue
 *DESCRIPTION:
 *   This function reads messages from a specified message queue
 *   and deletes the messages from the message-pool
 *
 *PARAMETER:   
 *   tU16 u16MyAppId (->I)
 *   identifier of calling application  
 *
 *   OSAL_tMQueueHandle hMQ (->I)
 *   handle of queue to be cleared
 *
 *RETURNVALUE:
 *   tVoid
 *
 *HISTORY:
 * 21.03.2003 CM-DI/ESA1-Fischer
 * Initial
 *
 ******************************************************************************/
tVoid ail_vIpcClearMessageQueue(tU16 u16MyAppId, OSAL_tMQueueHandle hMQ)
{
   tS32                 s32MsgSize = 1; // start while-loop
   amt_tclBaseMessage   oMsgObject;

   while( s32MsgSize > 0 )
   {
      // get message from queue
      s32MsgSize = OSAL_s32MessageQueueWait(hMQ, (tU8*)oMsgObject.prGetOSALMsgHandle(), sizeof(OSAL_trMessage), NULL, 0);

      if( s32MsgSize > 0 )
      {
         // valid message?
         oMsgObject.pu8SharedMemBase = OSAL_pu8MessageContentGet(*(oMsgObject.prGetOSALMsgHandle()), OSAL_EN_READWRITE);

         if ( oMsgObject.pu8SharedMemBase != OSAL_NULL )
         {
            ail_vTraceMsg( TR_LEVEL_COMPONENT,
               "Application 0x%04x deletes unhandled msg from 0x%04x,%d with msg-type %d",
               u16MyAppId, oMsgObject.u16GetSourceAppID(), oMsgObject.u16GetSourceSubID(),
               oMsgObject.u8GetType() );

            // delete message from message-pool
            oMsgObject.bDelete();
         }
      }
   }
}

tCString ail_coszServiceStatus( tU8 u8ServiceStatus )
{
   switch ( u8ServiceStatus )
   {
      case AMT_C_U8_SVCSTATE_NOT_AVAILABLE:
         return("not available");
      case AMT_C_U8_SVCSTATE_AVAILABLE:
         return("available");
      case AMT_C_U8_SVCSTATE_REG_DENIED:
         return("reg denied");
      default:
         return("unknown");
   }
}

tCString ail_coszServiceRegisterConf( tU8 u8ServiceRegisterConf )
{
   switch ( u8ServiceRegisterConf )
   {
      case AMT_C_U8_REGCONF_NO_SUCCESS:
         return("no success");
      case AMT_C_U8_REGCONF_SUCCESS:
         return("success");
      case AMT_C_U8_REGCONF_SERVICE_ALREADY_REGISTERED:
         return("no success : service already registered");
      case AMT_C_U8_REGCONF_SERVICE_VERSION_NOT_SUPPORTED:
         return("no success : service version not supported");
      case AMT_C_U8_REGCONF_SERVICE_REGISTRATION_PENDING:
         return("no success : service registration pending");
      case AMT_C_U8_REGCONF_SERVER_DOES_NOT_RESPOND_TIMEOUT:
         return("no success : server does not respond timeout");
      case AMT_C_U8_REGCONF_SERVICE_DOES_NOT_EXIST:
         return("no success : service does not exist");
      default:
         return("unknown");
   }
}

tCString ail_coszAppStatus( tU32 u32AppStatus )
{
   switch ( u32AppStatus )
   {
      case AMT_C_U32_STATE_INVALID:
         return("INVALID");
      case AMT_C_U32_STATE_UNINITALIZED:
         return("UNINITIALIZED");
      case AMT_C_U32_STATE_INITIALIZED:
         return("INITIALIZED");
      case AMT_C_U32_STATE_NORMAL:
         return("NORMAL");
      case AMT_C_U32_STATE_DIAGNOSIS:
         return("DIAGNOSIS");
      case AMT_C_U32_STATE_PAUSE:
         return("PAUSE");
      case AMT_C_U32_STATE_PREPARE_DOWNLOAD:
         return("DOWNLOAD");
      case AMT_C_U32_STATE_RECEIVE_READY:
         return("RECEIVE_READY");
      case AMT_C_U32_STATE_OFF:
         return("OFF");
      default:
         return("unknown");
   }
}

tCString ail_coszCVMStatus( tU32 u32CVMStatus )
{
   switch ( u32CVMStatus )
   {
      case AMT_C_U32_NORMAL_VOLTAGE:
         return ("NORMAL(9V..16V)");
      case AMT_C_U32_LOW_VOLTAGE:
         return ("LOW(<9V)");
      case AMT_C_U32_HIGH_VOLTAGE:
         return ("HIGH(>16V)");
      default:
         return ("unknown");
   }
}

tCString ail_coszCommunicationError( const ail_tenCommunicationError &corfenCommunicationError )
{
   switch ( corfenCommunicationError )
   {
      case AIL_EN_N_NO_ERROR:
         return ("NO_ERROR");
      case AIL_EN_N_POST_MESSAGE_FAILED:
         return ("POST_MESSAGE_FAILED");
      case AIL_EN_N_SERVICE_AVAILABILITY_LOCK:
         return ("SERVICE_AVAILABILITY_LOCK");
      case AIL_EN_N_REGISTRATION_DENIED:
         return ("REGISTRATION_DENIED");
      case AIL_EN_N_ANSWER_TIMEOUT:
         return ("ANSWER_TIMEOUT");
      case AIL_EN_N_UNDEFINED_ABORT:
         return ("UNDEFINED_ABORT");
      case AIL_EN_N_NOT_ALL_INFORMED:
         return ("NOT_ALL_INFORMED");
      case AIL_EN_N_NOT_SEND:
         return ("NOT_SEND");
      case AIL_EN_N_INVALID_THREAD_CONTEXT:
         return ("INVALID_THREAD_CONTEXT");
      case AIL_EN_N_INVALID_PARAMETER:
         return ("INVALID_PARAMETER");
      case AIL_EN_N_INVALID_STATE:
         return ("INVALID_STATE");
      case AIL_EN_N_INVALID_MESSAGE_OBJECT:
         return ("INVALID MESSAGE OBJECT");
      case AIL_EN_N_REGISTRATION_DENIED_INVALID_REGID:
         return ("REGISTRATION_DENIED => BECAUSE INVALID REGISTER-ID USED");
      case AIL_EN_N_REGISTRATION_DENIED_NOT_REGISTERED:
         return ("REGISTRATION_DENIED => BECAUSE NOT REGISTERED (ANYMORE) FOR THIS SERVICE");
      case AIL_EN_N_REGISTRATION_DENIED_SERVICE_MISMATCH:
         return ("REGISTRATION_DENIED => BECAUSE PASSED SERVICE-ID DOESN'T MATCH WITH EXISTING REGISTRATION FOR THIS REGISTER-ID");
      case AIL_EN_N_REGISTRATION_DENIED_APPLICATION_MISMATCH:
         return ("REGISTRATION_DENIED => BECAUSE PASSED APP-ID DOESN'T MATCH WITH EXISTING REGISTRATION FOR THIS REGISTER-ID");
      case AIL_EN_N_REGISTRATION_DENIED_SERVICE_OR_SUBID_MISMATCH:
         return ("REGISTRATION_DENIED => BECAUSE THERE IS NO REGISTRATION FOUND WITH THIS COMBINATION OF SERVICE-ID AND APP-SUB-ID");
      case AIL_EN_N_REGISTRATION_DENIED_SERVICE_NOT_FOUND:
         return ("REGISTRATION_DENIED => BECAUSE THE SERVICE IS NOT OFFERED ANYMORE (SERVER SHUTTED DOWN)");
      default:
         return ("unknown");
   }
}

tCString ail_coszOSALError( tU32 u32OsalError )
{
   switch ( u32OsalError )
   {
      case OSAL_E_NOERROR:
         return ("not error");
      case OSAL_E_NOPERMISSION:
         return ("function call not allowed");
      case OSAL_E_INVALIDVALUE:
         return ("wrong parameter");
      case OSAL_E_NOSPACE:
         return ("no more ressources(memory)");
      case OSAL_E_BUSY:
         return ("ressource is in use");
      case OSAL_E_INTERRUPT:
         return ("interrupt during wait");
      case OSAL_E_BADFILEDESCRIPTOR:
         return ("not existing filedescriptor");
      case OSAL_E_MSGTOOLONG:
         return ("message is to long");
      case OSAL_E_QUEUEFULL:
         return ("message queue is full");
      case OSAL_E_TIMEOUT:
         return ("function abort because of timeout");
      case OSAL_E_UNKNOWN:
         return ("unknown error");
      default:
         return ("unknown error, see osal-include");
   }
}


