/******************************************************************************
 * FILE:          ail_ListHelp.cpp
 * SW-COMPONENT:  AIL
 * DESCRIPTION:  
 * AUTHOR:        
 * COPYRIGHT:     (c) 2001 Robert Bosch GmbH, Hildesheim
 * HISTORY:      
 ******************************************************************************/

/******************************************************************************
| includes                                                                    
|   1)system- and project- includes
|   2)needed interfaces from external components
|   3)internal and external interfaces from this component 
|-----------------------------------------------------------------------------*/

// regular includes
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define SCD_S_IMPORT_INTERFACE_GENERIC
#include "scd_if.h"

#define AIL_S_IMPORT_INTERFACE_GENERIC
#define AIL_S_IMPORT_INTERFACE_TRACE
#include "ail_if.h"

/******************************************************************************
| defines and macros (scope: modul-local)
|-----------------------------------------------------------------------------*/

const tChar ail_coszPoolSemaphoreName[] = { "AS" };  //AIL-Semaphorepool
#define AIL_C_U16_GENERATION_NAME_MAX  (tU16)20

/******************************************************************************
| typedefs (scope: modul-local)
|-----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: global)
|-----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: modul-local)
|-----------------------------------------------------------------------------*/

/******************************************************************************
| function prototype (scope: modul-local)
|-----------------------------------------------------------------------------*/

/******************************************************************************
| function implementation (scope: modul-local)
|-----------------------------------------------------------------------------*/

/******************************************************************************
| function implementation (scope: global)
|-----------------------------------------------------------------------------*/

 
/*******************************************************************************
 *FUNCTION:   
 *DESCRIPTION:
 *PARAMETER:   
 *                
 *RETURNVALUE:
 *
 *HISTORY:
 ******************************************************************************/
tBool 
ail_bTakeListAccess
(
   OSAL_tSemHandle hSemId
)
{
   tBool bSemTake = FALSE;

   tS32 s32Ret = OSAL_s32SemaphoreWait( hSemId, OSAL_C_U32_INFINITE );
   
   if ( s32Ret == OSAL_OK )
   {
      bSemTake = TRUE;
   }
   else
   {
     NORMAL_M_ASSERT_ALWAYS();
   }

   return ( bSemTake );
}

/*******************************************************************************
 *FUNCTION:   
 *DESCRIPTION:
 *PARAMETER:   
 *                
 *RETURNVALUE:
 *
 *HISTORY:
 ******************************************************************************/
tBool 
ail_bReleaseListAccess
(
   OSAL_tSemHandle hSemId
)
{
   tBool bSemPost = FALSE;

   tS32 s32Ret = OSAL_s32SemaphorePost( hSemId );
   
   if ( s32Ret == OSAL_OK )
   {
      bSemPost = TRUE;
   }
   else
   {
     NORMAL_M_ASSERT_ALWAYS();
   }

   return ( bSemPost );
}

/*******************************************************************************
 *FUNCTION:   
 *DESCRIPTION:
 *PARAMETER:   
 *                
 *RETURNVALUE:
 *
 *HISTORY:
 ******************************************************************************/
OSAL_tMQueueHandle 
ail_hGetReceiverQueue
(
   ail_tclOpenedQueueList*    poOpenedQueueList,
   OSAL_tSemHandle            hSemId,
   tU16                       u16AppId
)
{
   OSAL_tMQueueHandle   hMQ = OSAL_C_INVALID_HANDLE;
   tBool                bListAccessSheltered = FALSE;

   if ( poOpenedQueueList != OSAL_NULL )
   {
      if ( hSemId == OSAL_C_INVALID_HANDLE )
      {
         /* access already protected */
         bListAccessSheltered = TRUE;
      }
      else if ( ail_bTakeListAccess( hSemId ) )
      {
         /* access protected now */
         bListAccessSheltered = TRUE;
      }


      if ( bListAccessSheltered )
      /* access allowed */
      {
         tBool bFound = FALSE;
         ail_tclOpenedQueueIterator iter(poOpenedQueueList);

         for (; !iter.bIsDone() && !bFound; iter.vNext())
         {
            if ( iter.oItem().bIsOpenedForApp(u16AppId, &hMQ) )
            {
               bFound = TRUE;
            }
         }    
      
         if ( !bFound )
         /* the queue is used first time */ 
         {
            hMQ = scd_OpenQueue(u16AppId);
            if ( hMQ != OSAL_C_INVALID_HANDLE )
            {
               /* add the opened queue to the corresponding list */
               ail_tclOpenedQueue  oOpenedQueue(hMQ, u16AppId);
               poOpenedQueueList->vAdd(oOpenedQueue);
            }
         }
         if ( hSemId != OSAL_C_INVALID_HANDLE )
         /* release the list access */
         {
            ail_bReleaseListAccess( hSemId ); //lint !e522: Expected void type, assignment, increment or decrement
         }
      }
   }

   return( hMQ );
}

/*******************************************************************************
 *FUNCTION:    ail_bDeleteFromOpenedQueueList
 *DESCRIPTION: deletes handle to messagequeue mbx_'u16AppId' from queue-list
 *PARAMETER:   poOpenedQueueList list with handles to opened queues
               hSemId            semaphore to protect list-access
               u16AppId          delete handle to queue of this application
                                 if u16AppId==AMT_C_U16_APPID_INVALID, all
                                 queues will be closed
 *                
 *RETURNVALUE: tBool:   delete successful
 *
 *HISTORY:
 * 07.03.03 CM-DI/ESA1-Fischer  Initial revision
 ******************************************************************************/
tBool
ail_bDeleteFromOpenedQueueList
(
   ail_tclOpenedQueueList* poOpenedQueueList,
   OSAL_tSemHandle         hSemId,
   tU16                    u16AppId
)
{
   tBool bListAccessSheltered = FALSE;
   tBool bSuccess = FALSE;

   if ( hSemId == OSAL_C_INVALID_HANDLE )
   {
      /* access already protected */
      bListAccessSheltered = TRUE;
   }
   else if ( ail_bTakeListAccess( hSemId ) )
   {
      /* access protected now */
      bListAccessSheltered = TRUE;
   }

   if ( bListAccessSheltered )
   /* access allowed */
   {
      if ( poOpenedQueueList )
      {
         tBool bElementsInList = TRUE;

         while ( bElementsInList )
         {
            tBool bEntryDeleted = FALSE;
            ail_tclOpenedQueueIterator iter(poOpenedQueueList);

            // find opened queue to server in list
            while ( !bEntryDeleted && !iter.bIsDone() )
            {
               if (  u16AppId == AMT_C_U16_APPID_INVALID
                  || iter.oItem().u16GetAppId() == u16AppId )
               {
                  // found an open queue to server,
                  //  close handle and remove entry from list
                  scd_s32CloseQueue((iter.oItem()).hGetQueue());
                  // every nRemove invalidates iterator!
                  poOpenedQueueList->nRemove(iter.oItem());
                  bEntryDeleted = TRUE;
                  bSuccess = TRUE;
               }
               else
               {
                  // get next entry
                  iter.vNext();
               }
            } // end while list-end not found and no entry is deleted

            if ( !bEntryDeleted )
            {
               bElementsInList = FALSE;
            }
         } // end while elements in list
      } // else poOpenedQueueList doesn't exist

      if ( hSemId != OSAL_C_INVALID_HANDLE )
      /* release the list access */
      {
         ail_bReleaseListAccess( hSemId ); //lint !e522: Expected void type, assignment, increment or decrement
      }
   } // else no access allowed

   return( bSuccess );
}

/*******************************************************************************
 *FUNCTION:    ail_bCheckRegistrationsToApp
 *DESCRIPTION: checks for existing registrations to given application
 *PARAMETER:   poServiceReferenceList  registrations to other services
               poServiceRegistryList   registrations for own service
               hSemId                  semaphore to protect list-access
               u16AppId                delete handle to queue of this application
 *                
 *RETURNVALUE: tBool: TRUE: registration to this app exist
 *
 *HISTORY:
 * 19.03.03 CM-DI/ESA1-Fischer  Initial revision
 ******************************************************************************/
tBool
ail_bCheckRegistrationsToApp
(
   ail_tclServiceReferenceList *poServiceReferenceList,
   ail_tclServiceRegistryList  *poServiceRegistryList,
   OSAL_tSemHandle              hSemId,
   tU16                         u16AppId
)
{
   tBool bListAccessSheltered = FALSE;
   tBool bFound = FALSE;

   if ( hSemId == OSAL_C_INVALID_HANDLE )
   {
      /* access already protected */
      bListAccessSheltered = TRUE;
   }
   else if ( ail_bTakeListAccess( hSemId ) )
   {
      /* access protected now */
      bListAccessSheltered = TRUE;
   }

   if ( bListAccessSheltered )
   /* access allowed */
   {
      // search in service-reference-list
      if ( poServiceReferenceList )
      {
         ail_tclServiceReferenceIterator  iter(poServiceReferenceList);

         for ( ; !iter.bIsDone() && !bFound; iter.vNext())
         {
            // check for other registrations to this server
            if ( iter.oItem().u16GetServerAppId() == u16AppId )
            {
               bFound = TRUE;
            }
         }
      } // else poServiceReferenceList doesn't exist

      // if not found, search in service-registry-list
      if ( !bFound && poServiceRegistryList )
      {
         ail_tclServiceRegistryIterator  iter(poServiceRegistryList);

         for ( ; !iter.bIsDone() && !bFound; iter.vNext())
         {
            // check for other registrations of this client
            if ( iter.oItem().u16GetClientAppId() == u16AppId )
            {
               bFound = TRUE;
            }
         }
      } // else found item or poServiceRegistryList doesn't exist

      if ( hSemId != OSAL_C_INVALID_HANDLE )
      /* release the list access */
      {
         ail_bReleaseListAccess( hSemId ); //lint !e522: Expected void type, assignment, increment or decrement
      }
   } // else no access allowed

   return( bFound );
}

/*******************************************************************************
 *FUNCTION:    ail_enCheckServerMailValidity()
 *
 *DESCRIPTION: refers to ServiceRegistryList, adds send parameters and tests  
 *             validity, if parameters are valid and service is registered
 * 
 *PARAMETER:   tU16                          u16AppId ( -> I )
 *             OSAL_tSemHandle               hSemId ( -> I )
 *             amt_tclServiceData*           poMessage ( -> I )
 *             ail_tclServiceRegistryList*   poServiceRegistryList ( -> I )
 *                
 *RETURNVALUE: ail_tenCommunicationError enComRet
 *
 *HISTORY:
 * 09.08.01 CM-DI/ECC32-Brunke   Initial revision
 * 12.02.02 CM-DI/PDP3-Andres    bugfix: only valid semaphore handle will be
 *                               released
 * 08.07.02 CM-DI/ESN1-Bruns     bugfix: TargetSubId is taken from stored item
 *                               from registration now.
 ******************************************************************************/

ail_tenCommunicationError
ail_enCheckServerMailValidity
(
   tU16                             u16AppId,
   OSAL_tSemHandle                  hSemId, 
   amt_tclServiceData*              poMessage,
   ail_tclServiceRegistryList*      poServiceRegistryList
)
{
   ail_tenCommunicationError  enComRet = AIL_EN_N_NO_ERROR;
   tBool                      bListAccessSheltered = FALSE;

   tU16  u16TargetAppId = poMessage->u16GetTargetAppID();
   tU16  u16ServiceId = poMessage->u16GetServiceID();
   tU16  u16RegisterId = poMessage->u16GetRegisterID();
   tU16  u16SourceAppId = poMessage->u16GetSourceAppID();

   if ( u16RegisterId != AMT_C_U16_REGID_INVALID )
   /* it always requires a valid RegisterId */
   {
      if ( hSemId == OSAL_C_INVALID_HANDLE )
      {
         /* access already protected */
         bListAccessSheltered = TRUE;
      }
      else if ( ail_bTakeListAccess( hSemId ) )
      {
         /* access protected now */
         bListAccessSheltered = TRUE;
      }

      if ( bListAccessSheltered )
      /* access allowed */
      {
         tBool bFound = FALSE;

         ail_tclServiceRegistryIterator  iter(poServiceRegistryList);

         for ( ; !iter.bIsDone() && !bFound; iter.vNext())
         {
            /* check correctness */
            if ( iter.oItem().bIsDataSet( u16RegisterId ) )
            {
               /* for a server ServiceId and ClientId can be supplemented  */
               bFound = TRUE;   
            
               poMessage->vSetTargetSubID( iter.oItem().u16GetClientSubId());
               if ( u16SourceAppId == AMT_C_U16_APPID_INVALID )
               /* not undefined source is always set to own Id */
               {
                  u16SourceAppId = u16AppId;
                  poMessage->vSetSourceAppID( u16SourceAppId );
               }

               if ( u16TargetAppId == AMT_C_U16_APPID_INVALID )
               /* destination not set */
               {
                  /* add the destination to the message */
                  u16TargetAppId = iter.oItem().u16GetClientAppId();
                  poMessage->vSetTargetAppID( u16TargetAppId );
               }
               else if ( u16TargetAppId != iter.oItem().u16GetClientAppId() )
               /* check the destination */
               {
                  /* any conflict in application and AIL */
                  enComRet = AIL_EN_N_REGISTRATION_DENIED_APPLICATION_MISMATCH;
               }
               if ( u16ServiceId == AMT_C_U16_SERVICE_UNDEF )
               /* ServiceId not set */
               {
                  /* add the service to the message */
                  u16ServiceId = iter.oItem().u16GetServiceId();
                  poMessage->vSetServiceID( u16ServiceId );
               }
               else if ( u16ServiceId != iter.oItem().u16GetServiceId() )
               /* check the service */
               {
                  /* any conflict in application and AIL */
                  enComRet = AIL_EN_N_REGISTRATION_DENIED_SERVICE_MISMATCH;
               }
            }
         }
         if ( !bFound )
         {
            /* in the meantime registration number has become invalid */
            enComRet = AIL_EN_N_REGISTRATION_DENIED_NOT_REGISTERED;
         }
         /* release the list access */
         if ( hSemId != OSAL_C_INVALID_HANDLE )
         {
            ail_bReleaseListAccess( hSemId ); //lint !e522: Expected void type, assignment, increment or decrement
         }
      }
      else
      {
         enComRet = AIL_EN_N_UNDEFINED_ABORT;
      }
   }
   else
   {
      enComRet = AIL_EN_N_REGISTRATION_DENIED_INVALID_REGID;
   }      
   return ( enComRet );
}


/*******************************************************************************
 *FUNCTION:    ail_enCheckClientMailValidity()
 *
 *DESCRIPTION: refers to ServiceRefrenceList, adds send parameters and tests  
 *             validity, if parameter are valid and service is available
 * 
 *PARAMETER:   tU16                          u16AppId ( -> I )
 *             OSAL_tSemHandle               hSemId ( -> I )
 *             amt_tclServiceData*           poMessage ( -> I )
 *             ail_tclServiceReferenceList*  poServiceReferenceList ( -> I )
 *                
 *RETURNVALUE: ail_tenCommunicationError enComRet
 *
 *HISTORY:
 * 09.08.01 CM-DI/ECC32-Brunke   Initial revision
 * 12.02.02 CM-DI/PDP3-Andres    bugfix: only valid semaphore handle will be
 *                               released
 ******************************************************************************/

ail_tenCommunicationError
ail_enCheckClientMailValidity
(
   tU16                             u16AppId,
   OSAL_tSemHandle                  hSemId, 
   amt_tclServiceData*              poMessage,
   ail_tclServiceReferenceList*     poServiceReferenceList
)
{
   ail_tenCommunicationError  enComRet = AIL_EN_N_NO_ERROR;
   tBool                      bListAccessSheltered = FALSE;

   tU16  u16TargetAppId = poMessage->u16GetTargetAppID();
   tU16  u16ServiceId = poMessage->u16GetServiceID();
   tU16  u16RegisterId = poMessage->u16GetRegisterID();
   tU16  u16SourceAppId = poMessage->u16GetSourceAppID();
   tU16  u16SourceSubId = poMessage->u16GetSourceSubID();
   tU8   u8MsgOpCode = poMessage->u8GetOpCode();

   if ( hSemId == OSAL_C_INVALID_HANDLE )
   {
      /* access already protected */
      bListAccessSheltered = TRUE;
   }
   else if ( ail_bTakeListAccess( hSemId ) )
   {
      /* access protected now */
      bListAccessSheltered = TRUE;
   }

   if ( bListAccessSheltered )
   /* access allowed */
   {
      tBool    bFound = FALSE;
      tU8      u8ServiceState = AMT_C_U8_SVCSTATE_NOT_AVAILABLE;   

      ail_tclServiceReferenceIterator  iter( poServiceReferenceList );

      for ( ; !iter.bIsDone() && !bFound; iter.vNext())
      {
         /* check correctness */
         if ( iter.oItem().bIsDataSet( u16ServiceId, u16SourceSubId ) )
         {
            /* for a client only ServiceId is required
               and RegisterId and ServerId can be added */
            bFound = TRUE;   

            if ( u16SourceAppId == AMT_C_U16_APPID_INVALID )
            /* not undefined source is always set to own Id */
            {
               u16SourceAppId = u16AppId;
               poMessage->vSetSourceAppID( u16SourceAppId );
            }

            if ( u16TargetAppId == AMT_C_U16_APPID_INVALID )
            /* destination not set */
            {
               /* add the destination to the message */
               u16TargetAppId = iter.oItem().u16GetServerAppId();
               poMessage->vSetTargetAppID( u16TargetAppId );
            }
            else if ( u16TargetAppId != iter.oItem().u16GetServerAppId() )
            /* check the destination */
            {
               /* only possible, if using same service from different servers */
               bFound = FALSE;
            }
            if ( bFound )
            {
               if ( u16RegisterId == AMT_C_U16_REGID_INVALID )
               /* RegisterId not set */
               {
                  /* add the service to the message */
                  u16RegisterId = iter.oItem().u16GetRegisterId();
                  poMessage->vSetRegisterID( u16RegisterId );
               }
               else if ( u16RegisterId != iter.oItem().u16GetRegisterId() )
               /* check the registration */
               {
                  /* only possible, if registered twice at the same application */
                  bFound = FALSE;
               }
            }
            if ( bFound )
            {
               u8ServiceState = iter.oItem().u8GetServiceState();
            }
         }
      }
      if ( !bFound )
      /* specified communication channel is not authorized - why ever - */
      {
         /* typical deadlock, if application is waiting for 
            ::vOnServiceState(REGID_INVAID), but no message was generated
            -> necessary to send message to myself? */
         enComRet = AIL_EN_N_REGISTRATION_DENIED_SERVICE_OR_SUBID_MISMATCH;
      }
      /* allow ReleaseUpreg and PortAcknowledge even if service is not available */
      else if (   u8ServiceState == AMT_C_U8_SVCSTATE_NOT_AVAILABLE
               && u8MsgOpCode != AMT_C_U8_CCAMSG_OPCODE_RELUPREG
               && u8MsgOpCode != AMT_C_U8_CCAMSG_OPCODE_CLIENT_PORT_ACK )
      {
         enComRet = AIL_EN_N_SERVICE_AVAILABILITY_LOCK;
      }
      else if ( u8ServiceState == AMT_C_U8_SVCSTATE_REG_DENIED )
      {
         enComRet = AIL_EN_N_REGISTRATION_DENIED_SERVICE_NOT_FOUND;
      }
      /* else: u8ServiceState == AMT_C_U8_SVCSTATE_AVAILABLE
            || (  u8ServiceState == AMT_C_U8_SVCSTATE_NOT_AVAILABLE
                 && u8MsgOpCode == AMT_C_U8_CCAMSG_OPCODE_RELUPREG )
            || ( u8ServiceState == AMT_C_U8_SVCSTATE_NOT_AVAILABLE
                 && u8MsgOpCode == AMT_C_U8_CCAMSG_OPCODE_CLIENT_PORT_ACK ) */

      /* release the list access */
      if ( hSemId != OSAL_C_INVALID_HANDLE )
      {
         ail_bReleaseListAccess( hSemId ); //lint !e522: Expected void type, assignment, increment or decrement
      }
   }
   else
   {
      enComRet = AIL_EN_N_UNDEFINED_ABORT;
   }
   return ( enComRet );
}


/*******************************************************************************
 *FUNCTION:    ail_hGetSemaphoreFromList()
 *
 *DESCRIPTION: refers to PoolSemaphoreList, searchs an unused semaphore in list,
 *             opens it and returns handle.
 *             if no unused semaphore exist, it creates a new one and
 *             returns handle.
 * 
 *PARAMETER:   tU16                      u16AppId (I)
 *             OSAL_tSemHandle           hGLA (->I)
 *             ail_tclPoolSemaphoreList *poPoolSemaphoreList (->I)
 *                
 *RETURNVALUE: OSAL_tSemHandle
 *             handle to semaphore in list or OSAL_C_INVALID_HANDLE if fails
 *
 *HISTORY:
 * 28.04.03 CM-DI/ESA1-Fischer   Initial revision
 ******************************************************************************/

OSAL_tSemHandle
ail_hGetSemaphoreFromList
(
   tU16                      u16AppId,             // application-id
   OSAL_tSemHandle           hGLA,                 // general list-access
   ail_tclPoolSemaphoreList *poPoolSemaphoreList   // list with created semaphores
)
{
   // locals
   OSAL_tSemHandle   hSemId = OSAL_C_INVALID_HANDLE;
   tU16              u16SemId = 0;
   tBool             bListAccessSheltered = FALSE;
   tBool             bSemFound = FALSE;

   if ( hGLA == OSAL_C_INVALID_HANDLE )
   {
      // access already protected
      bListAccessSheltered = TRUE;
   }
   else if ( ail_bTakeListAccess( hGLA ) )
   {
      // access protected now
      bListAccessSheltered = TRUE;
   }

   if ( bListAccessSheltered )
   {
      // access allowed

      if (  u16AppId != AMT_C_U16_APPID_INVALID
         && poPoolSemaphoreList != OSAL_NULL )
      {
         // parameters are valid

         // search unused entry in pool-semaphore-list
         ail_tclPoolSemaphoreIterator  iter( poPoolSemaphoreList );

         for ( ; !iter.bIsDone() && !bSemFound; iter.vNext())
         {
            // if we have to create a new semaphore, we need a unique id
            ++u16SemId;
            if ( iter.oItem().u16GetId() > u16SemId )
               u16SemId = iter.oItem().u16GetId();

            // is semaphore created and free?
            if ( iter.oItem().enGetStatus() == AIL_EN_POOLSEM_CREATED )
            {
               // get semaphore-name, don't forget to delete string
               tString szSemName = iter.oItem().szGetName();

               // open semaphore
               if (  szSemName != OSAL_NULL
                  && OSAL_s32SemaphoreOpen( (tCString)szSemName, &hSemId) != OSAL_ERROR )
               {
                  // semaphore found, change state in list and break loop
                  iter.oItem().vSetStatus(AIL_EN_POOLSEM_USED);
                  iter.oItem().vSetHandle(hSemId);

                  tS32  s32SemValue = 0;
                  tBool bCantResetSemaphore = FALSE;
                  if ( OSAL_s32SemaphoreGetValue(hSemId, &s32SemValue) != OSAL_ERROR )
                  {
                     if ( s32SemValue > 0 )
                     {
                        // semaphore must be initialized by zero!
                        while ( s32SemValue > 0 )
                        {
                           if ( OSAL_s32SemaphoreWait(hSemId, 0) == OSAL_ERROR )
                           {
                              ail_vTraceMsg( TR_LEVEL_COMPONENT,
                                 "ail_hGetSemaphoreFromList: SemaphoreWait failed" );
                              bCantResetSemaphore = TRUE;
                           }
                           --s32SemValue;
                        }
                        if (  OSAL_s32SemaphoreGetValue(hSemId, &s32SemValue) == OSAL_ERROR
                           || s32SemValue > 0 )
                        {
                           bCantResetSemaphore = TRUE;
                        }
                     }
                  }
                  else
                  {
                     bCantResetSemaphore = TRUE;
                  }

                  if ( bCantResetSemaphore )
                  {
                     ail_vTraceMsg( TR_LEVEL_COMPONENT,
                        "ail_hGetSemaphoreFromList: got semaphore from pool" \
                        "can't reset sem-value to 0, SemId: %d", hSemId );

                     // delete semaphore and remove it from list
                     tString szName = iter.oItem().szGetName();

                     if ( szName != OSAL_NULL )
                     {
                        if ( OSAL_s32SemaphoreDelete((tCString)szName) != OSAL_ERROR )
                        {
                           // not really needed, but if nRemove fails
                           iter.oItem().vSetHandle(OSAL_C_INVALID_HANDLE);
                           iter.oItem().vSetStatus(AIL_EN_POOLSEM_DELETED);
                        }
                        else
                        {
                           ail_vTraceMsg( TR_LEVEL_COMPONENT,
                              "ail_hGetSemaphoreFromList: semaphore-delete " \
                              "failed, SemId: %d, SemName: %s", hSemId, szName );
                        }
                     }
                     else
                     {
                        ail_vTraceMsg( TR_LEVEL_COMPONENT,
                           "ail_hGetSemaphoreFromList: can't get name from " \
                           "semaphore in pool, SemId: %d", hSemId );
                     }

                     OSAL_vMemoryFree (szName);
                     // remove entry from list, "nRemove"-operation influences the validity of the iterator
                     poPoolSemaphoreList->nRemove( iter.oItem() );

                     // now try to create a new semaphore (break)
                  } // end if bCantResetSemaphore
                  else
                  {
                     // we found a semaphore
                     bSemFound = TRUE;
                  }

                  // free string-memory
                  OSAL_vMemoryFree (szSemName);

                  // stop search
                  break;

               } // end if semaphore-name == NULL or SemaphoreOpen failed
               else
               {
                  // free string-memory
                  OSAL_vMemoryFree (szSemName);

                  // couldn't open semaphore, return error
                  hSemId = OSAL_C_INVALID_HANDLE;
               }
            }
         } // end for

         // if no semaphore found, create a new one
         if ( !bSemFound )
         {
            // create a unique semaphore name
            ++u16SemId;
            tChar szCreationName[AIL_C_U16_GENERATION_NAME_MAX] = "\0";

            if (OSALUTIL_s32SaveNPrintFormat((tString)szCreationName,
                                             sizeof(szCreationName),
                                             "%s%x_%x", 
                                             ail_coszPoolSemaphoreName, 
                                             u16AppId, 
                                             u16SemId) != OSAL_ERROR)
            {
              // create semaphore, must be initialized by zero!
              if ( OSAL_ERROR != OSAL_s32SemaphoreCreate((tCString)szCreationName, &hSemId, 0) )
              {
                 // store semaphore in list
                 ail_tclPoolSemaphore oPoolSemaphore(
                    u16SemId, (tCString)szCreationName, AIL_EN_POOLSEM_USED, hSemId );
                 poPoolSemaphoreList->vAdd(oPoolSemaphore);
              }
              else
              {
                 // couldn't create semaphore, return error
                 hSemId = OSAL_C_INVALID_HANDLE;
              }
            }
            else
            {
              NORMAL_M_ASSERT_ALWAYS();
              hSemId = OSAL_C_INVALID_HANDLE;
            }
         } // end if no semaphore found (create a new one)

      } // end if parameters are valid, hSemId stays OSAL_C_INVALID_HANDLE

      // release list-access
      if ( hGLA != OSAL_C_INVALID_HANDLE )
      {
         ail_bReleaseListAccess( hGLA );
      }

   } // else hSemId stays OSAL_C_INVALID_HANDLE

   return ( hSemId );
}

/*******************************************************************************
 *FUNCTION:    ail_bReleaseSemaphoreInList()
 *
 *DESCRIPTION: refers to PoolSemaphoreList, searchs given semaphore in list
 *             and closes it.
 * 
 *PARAMETER:   OSAL_tSemHandle           hGLA (->I)
 *             OSAL_tSemHandle           hSemId (->I)
 *             ail_tclPoolSemaphoreList *poPoolSemaphoreList (->I)
 *                
 *RETURNVALUE: tBool
 *             TRUE: handle found and closed
 *
 *HISTORY:
 * 28.04.03 CM-DI/ESA1-Fischer   Initial revision
 ******************************************************************************/
tBool
ail_bReleaseSemaphoreInList
(
   OSAL_tSemHandle           hGLA,                 // general list-access
   OSAL_tSemHandle           hSemId,               // release this semaphore from list
   ail_tclPoolSemaphoreList *poPoolSemaphoreList   // list with created semaphores
)
{
   // locals
   tBool bListAccessSheltered = FALSE;
   tBool bSuccess = FALSE;

   if ( hGLA == OSAL_C_INVALID_HANDLE )
   {
      // access already protected
      bListAccessSheltered = TRUE;
   }
   else if ( ail_bTakeListAccess( hGLA ) )
   {
      // access protected now
      bListAccessSheltered = TRUE;
   }

   if ( bListAccessSheltered )
   {
      // access allowed

      if (  poPoolSemaphoreList != OSAL_NULL
         && hSemId != OSAL_C_INVALID_HANDLE )
      {
         // parameters are valid

         // search given semaphore in pool-semaphore-list
         ail_tclPoolSemaphoreIterator  iter( poPoolSemaphoreList );

         for ( ; !iter.bIsDone(); iter.vNext())
         {
            // semaphore-handle equal and entry used?
            if (  iter.oItem().hGetHandle()  == hSemId
               && iter.oItem().enGetStatus() == AIL_EN_POOLSEM_USED )
            {
               // entry found, sem-value must be zero
               tS32  s32SemValue = 0;
               tBool bCantResetSemaphore = FALSE;
               if ( OSAL_s32SemaphoreGetValue(hSemId, &s32SemValue) != OSAL_ERROR )
               {
                  if ( s32SemValue > 0 )
                  {
                     // semaphore must be initialized by zero!
                     while ( s32SemValue > 0 )
                     {
                        if ( OSAL_s32SemaphoreWait(hSemId, 0) == OSAL_ERROR )
                        {
                           ail_vTraceMsg( TR_LEVEL_COMPONENT,
                              "ail_hGetSemaphoreFromList: SemaphoreWait failed" );
                           bCantResetSemaphore = TRUE;
                        }
                        --s32SemValue;
                     }
                     if (  OSAL_s32SemaphoreGetValue(hSemId, &s32SemValue) == OSAL_ERROR
                        || s32SemValue > 0 )
                     {
                        bCantResetSemaphore = TRUE;
                     }
                  }
               }
               else
               {
                  bCantResetSemaphore = TRUE;
               }

               if ( bCantResetSemaphore )
               {
                  ail_vTraceMsg( TR_LEVEL_COMPONENT,
                     "ail_bReleaseSemaphoreInList: " \
                     "can't reset sem-value to 0, SemId: %d", hSemId );

                  // close semaphore
                  if ( OSAL_s32SemaphoreClose( hSemId) != OSAL_ERROR )
                  {
                     // semaphore closed
                     bSuccess = TRUE;
                  } // else return error

                  // delete semaphore and remove it from list
                  tString szName = iter.oItem().szGetName();

                  iter.oItem().vSetHandle(OSAL_C_INVALID_HANDLE);

                  if (  szName != OSAL_NULL
                     && OSAL_s32SemaphoreDelete((tCString)szName) != OSAL_ERROR )
                  {
                     // not really needed, but if nRemove fails
                     iter.oItem().vSetStatus(AIL_EN_POOLSEM_DELETED);
                  }
                  else
                  {
                     // try to delete later
                     iter.oItem().vSetStatus(AIL_EN_POOLSEM_USED);
                  }

                  OSAL_vMemoryFree (szName);
                  // remove entry from list, "nRemove"-operation influences the validity of the iterator
                  poPoolSemaphoreList->nRemove( iter.oItem() );
               }
               else
               {
                  //close entry and change state in list
                  iter.oItem().vSetHandle(OSAL_C_INVALID_HANDLE);
                  iter.oItem().vSetStatus(AIL_EN_POOLSEM_CREATED);

                  // close semaphore
                  if ( OSAL_s32SemaphoreClose( hSemId) != OSAL_ERROR )
                  {
                     // semaphore closed
                     bSuccess = TRUE;
                  } // else return error
               }
               
               break;
            }
         } // end for
      } // end if parameters are valid, else return error

      // release list-access
      if ( hGLA != OSAL_C_INVALID_HANDLE )
      {
         ail_bReleaseListAccess( hGLA );
      }
   } // end if list-access sheltered, else return error

   return ( bSuccess );
}//lint !e429 Custodial pointer 'poPoolSemaphoreList' has not been freed or returned -> there couldn't be any memory leak 

/*******************************************************************************
 *FUNCTION:    ail_bDeleteSemaphoresFromList()
 *
 *DESCRIPTION: refers to PoolSemaphoreList, deletes all semaphores in list
 * 
 *PARAMETER:   OSAL_tSemHandle           hGLA (->I)
 *             ail_tclPoolSemaphoreList *poPoolSemaphoreList (->I)
 *                
 *RETURNVALUE: tBool
 *             TRUE: all semaphores successfully closed
 *
 *HISTORY:
 * 28.04.03 CM-DI/ESA1-Fischer   Initial revision
 ******************************************************************************/
tBool
ail_bDeleteSemaphoresFromList
(
   OSAL_tSemHandle           hGLA,                 // general list-access
   ail_tclPoolSemaphoreList *poPoolSemaphoreList   // list with created semaphores
)
{
   // locals
   tBool bListAccessSheltered = FALSE;
   tBool bSuccess = FALSE;

   if ( hGLA == OSAL_C_INVALID_HANDLE )
   {
      // access already protected
      bListAccessSheltered = TRUE;
   }
   else if ( ail_bTakeListAccess( hGLA ) )
   {
      // access protected now
      bListAccessSheltered = TRUE;
   }

   if ( bListAccessSheltered )
   {
      // access allowed

      if ( poPoolSemaphoreList != OSAL_NULL )
      {
         // parameters are valid
         bSuccess = TRUE;

         // get all entrys from pool-semaphore-list
         tBool bElementsInList = TRUE;

         while ( bElementsInList )
         {
            ail_tclPoolSemaphoreIterator iter(poPoolSemaphoreList);

            if ( !iter.bIsDone() )
            {
               // does semaphore exist?
               if (  iter.oItem().enGetStatus() != AIL_EN_POOLSEM_DELETED
                  && iter.oItem().enGetStatus() != AIL_EN_POOLSEM_UNINITIALIZED )
               {
                  // delete semaphore
                  tString szName = iter.oItem().szGetName();

                  if (  szName != OSAL_NULL
                     && OSAL_s32SemaphoreDelete((tCString)szName) != OSAL_ERROR )
                  {
                     iter.oItem().vSetHandle(OSAL_C_INVALID_HANDLE);
                     iter.oItem().vSetStatus(AIL_EN_POOLSEM_DELETED);
                  }
                  else
                  {
                     bSuccess = FALSE;
                  }

                  OSAL_vMemoryFree (szName);
               }

               // remove entry from list, "nRemove"-operation influences the validity of the iterator
               poPoolSemaphoreList->nRemove( iter.oItem() );
            }
            else
            {
               // if no following entry, leave loop
               bElementsInList = FALSE;
            }
         } // end for
      } // end if parameters are valid, else return error

      // release list-access
      if ( hGLA != OSAL_C_INVALID_HANDLE )
      {
         ail_bReleaseListAccess( hGLA );
      }
   } // end if list-access sheltered, else return error

   return ( bSuccess );
}

