/*****************************************************************************
 * FILE:          FIExtractor.cpp
 * SW-COMPONENT:  
 * DESCRIPTION:   
 *                
 * AUTHOR:        CM-DI/ESA2-Bruns
 * COPYRIGHT:     (c) 2004 Blaupunkt Werke GmbH
 * HISTORY:       
 *                19.04.04 Rev. 1.0 CM-DI/ESA2-Bruns
 *                         Initial Revision;
 *****************************************************************************/




//-----------------------------------------------------------------------------
// General includes
//-----------------------------------------------------------------------------
#include "FIExtractor.h"
#include "FIUtil.h"
#include "assert.h"              // temporary!





/*****************************************************************************
| function implementation
|-----------------------------------------------------------------------------*/
tclFIExtractor::tclFIExtractor(DOMDocument *poDOMDocPtr, tclFIServiceTree *poServiceTreePtr)
{
   poDOMDoc      = poDOMDocPtr;
   poServiceTree = poServiceTreePtr;
}




tclFIExtractor::~tclFIExtractor()
{
}




bool tclFIExtractor::bIsValid()
{
   return ((0 != poDOMDoc) && (0 != poServiceTree));
}




/*****************************************************************************
 * bExtract
 * Extract 
 *****************************************************************************/
bool tclFIExtractor::bExtract()
{
   bool bSuccess = false;

   if(0 != poServiceTree)
   {
      tclFITypeNodePool *poGlobalTypePool = poServiceTree->poGetUsedGlobalTypePool();
      tclFITypeNodePool *poGlobalErrorPool= poServiceTree->poGetUsedGlobalErrorPool();
      tclFITypeNodePool *poMessagePool    = poServiceTree->poGetMessagePool();

      vPrintMsg("** Extract from file: %s\n", poServiceTree->oGetFilename().c_str());

      bSuccess = bExtractService();
      vPrintMsg("   Service %s\n", poServiceTree->oGetShortname().c_str());
      // all types
      vPrintMsg("   Extracting types...");
      bSuccess &= bExtractTypes(poGlobalTypePool, FI_SCHEMA_PRIMARY_TYPE, 
                                 FI_SCHEMA_IDSTRING_NAME, FI_SCHEMA_IDSTRING_DEFAULTVALUE,
                                 true);
      vPrintMsg(" ok.\n");
      // all error codes
      vPrintMsg("   Extracting error codes...");
      bSuccess &= bExtractTypes(poGlobalErrorPool, FI_SCHEMA_IDSTRING_ERRORDEF, 
                                 FI_SCHEMA_IDSTRING_NAME, FI_SCHEMA_IDSTRING_SERVICE_ID, 
                                 false, EN_OBJTYPE_ERRORDEF);
      vPrintMsg(" ok.\n");
      // all methods
      vPrintMsg("   Extracting methods...");
      bSuccess &= bExtractMessages(poMessagePool, FI_SCHEMA_PRIMARY_METHOD, 
                                 FI_SCHEMA_IDSTRING_NAME, FI_SCHEMA_IDSTRING_SERVICE_ID, 
                                 false, EN_OBJTYPE_MESSAGE_METHOD,
                                 FI_SCHEMA_METHOD_OPCODE);
      vPrintMsg(" ok.\n");
      // all properties
      vPrintMsg("   Extracting properties...");
      bSuccess &= bExtractMessages(poMessagePool, FI_SCHEMA_PRIMARY_PROPERTY, 
                                 FI_SCHEMA_IDSTRING_NAME, FI_SCHEMA_IDSTRING_SERVICE_ID, 
                                 false, EN_OBJTYPE_MESSAGE_PROPERTY,
                                 FI_SCHEMA_PROPERTY_OPCODE);
      vPrintMsg(" ok.\n");
      // dependency calculation (type <-> type,(error))
      vPrintMsg("   Calculate type dependencies...");
      bSuccess &= bExtractDependencies(poGlobalTypePool, poGlobalErrorPool, poGlobalTypePool);
      vPrintMsg(" ok.\n");
      // dependency calculation (message <-> type,error)
      vPrintMsg("   Calculate message dependencies...");
      bSuccess &= bExtractDependencies(poGlobalTypePool, poGlobalErrorPool, poMessagePool);
      vPrintMsg(" ok.\n");

      vPrintMsg("   Calculate type size...");
      bSuccess &= bCalculateTypeSize(poGlobalTypePool);
      vPrintMsg(" ok.\n");

      vPrintMsg("   Calculate error size...");
      bSuccess &= bCalculateTypeSize(poGlobalErrorPool);
      vPrintMsg(" ok.\n");

      vPrintMsg("   Calculate message size...");
      bSuccess &= bCalculateTypeSize(poMessagePool);
      vPrintMsg(" ok.\n");

/*    cout << "------------------------------------------" << endl;
      vector<tclFINode*> *poNodeList = poMessagePool->poNodeList;
      for (int i=0; i< poNodeList->size(); i++)
      {
         tclFINode *poNode = (*poNodeList)[i];
         cout << "Name: " << poNode->oGetName() << endl;
      }
      cout << "------------------------------------------" << endl; */
   }

   return bSuccess;
}




bool tclFIExtractor::bExtractService()
{
   bool bResult = false;
   DOMNodeList *poServiceRootNodeList = poGetNodeListFromDoc(poDOMDoc, FI_SCHEMA_IDSTRING_SERVICE);

   if(0 != poServiceRootNodeList)
   {
      for(int i=0; i < poServiceRootNodeList->getLength(); i++)
      {
         DOMNode *poServiceRootNode = poServiceRootNodeList->item(i);
         if(poServiceRootNode && (poServiceRootNode->getNodeType() == DOMNode::ELEMENT_NODE))
         {
            bResult = true;
            DOMElement *poServiceRootElement = static_cast<DOMElement*>(poServiceRootNode);
            // Read catalogue Fullname, Shortname, Service-ID
            string oTempName;
            if (_bGetElementFirstChildValue(oTempName, poServiceRootElement, FI_SCHEMA_IDSTRING_SERVICE_FULLNAME, 0))
               poServiceTree->vSetFullname(oTempName);
            else
               vWarning("Missing full name of service!");

            if (_bGetElementFirstChildValue(oTempName, poServiceRootElement, FI_SCHEMA_IDSTRING_SERVICE_SHORTNAME, 0))
               poServiceTree->vSetShortname(oTempName);
            else
            {
               vWarning("Missing short name of service!");
               bResult = false;
            }

            string oServiceString;
            if (_bGetElementFirstChildValue(oServiceString, poServiceRootElement, FI_SCHEMA_IDSTRING_SERVICE_ID, 0))
               poServiceTree->vSetServiceID(atoi(oServiceString.c_str()));
            else
            {
               vWarning("Missing id number of service!");
               bResult = false;
            }

            // Read catalogue Version-Ranges
            DOMNodeList *poVersionNodeList= poGetNodeListFromElement(poServiceRootElement, FI_SCHEMA_IDSTRING_SERVICE_VERSION);
            int iNumServiceVersions = poVersionNodeList->getLength();
            if (iNumServiceVersions < 1)
            {
               vWarning("Missing version info!");
               bResult = false;
            }               

            for(int i=0; i < iNumServiceVersions; i++)
            {
               DOMNode *poVersionNode = poVersionNodeList->item(i);
               string oTempStringMajor, oTempStringMinor, oTempStringPatch;
               if (!bGetElementAttribute(oTempStringMajor, poVersionNode, FI_SCHEMA_IDSTRING_SERVICE_VERSIONMAJOR))
                  vWarning("Missing major version id!");
               if (!bGetElementAttribute(oTempStringMinor, poVersionNode, FI_SCHEMA_IDSTRING_SERVICE_VERSIONMINOR))
                  vWarning("Missing minor version id!");
               if (!bGetElementAttribute(oTempStringPatch, poVersionNode, FI_SCHEMA_IDSTRING_SERVICE_VERSIONPATCH))
                  vWarning("Missing patch version id!");
               string oTempVersionName(oTempStringMajor + "." + oTempStringMinor+"."+oTempStringPatch);
               tclFIVersion oTempVersion(oTempVersionName);
               poServiceTree->bAddCatalogueVersion(oTempVersion);
            }
         }
         else
         {
            vWarning("Missing or invalid %s node!",FI_SCHEMA_IDSTRING_SERVICE);
         }
      }
   }
   return bResult;
}




bool tclFIExtractor::bExtractVersion(
               tclFIVersionRange &oResultVersionRange, 
               DOMElement *pDOMElement)
{
   bool bSuccess = true;
   string oVersionMinString, oVersionMaxString;

   if(!bGetElementAttribute(oVersionMinString, pDOMElement, FI_SCHEMA_IDSTRING_VERSIONMIN))
      oVersionMinString = FI_VERSION_LIMIT_LOW;

   if(!bGetElementAttribute(oVersionMaxString, pDOMElement, FI_SCHEMA_IDSTRING_VERSIONMAX))
      oVersionMaxString = FI_VERSION_LIMIT_HIGH;

   if(bGetElementAttribute(oVersionMinString, pDOMElement, FI_SCHEMA_IDSTRING_VERSIONSINGLE))
      oVersionMaxString = oVersionMinString;


   // Set version range of object
   oResultVersionRange.vSetVersionRange(oVersionMinString, oVersionMaxString);
   return true;
}




bool tclFIExtractor::bExtractTypeType(tenFINodeType &enResultType, DOMNode *pDOMNode)
{
   bool bSuccess = true;

   if(0 != poGetSubNode1(pDOMNode, FI_SCHEMA_TYPE_STRUCTURE))
      enResultType = EN_OBJTYPE_STRUCTURE;
   else if(0 != poGetSubNode1(pDOMNode, FI_SCHEMA_TYPE_ENUMERATION))
      enResultType = EN_OBJTYPE_ENUMERATION;
   else if(0 != poGetSubNode1(pDOMNode, FI_SCHEMA_TYPE_BITFIELD))
      enResultType = EN_OBJTYPE_BITFIELD;
   else if(0 != poGetSubNode1(pDOMNode, FI_SCHEMA_TYPE_ARRAY))  // MOST-compatible extensions
      enResultType = EN_OBJTYPE_ARRAY;                          // (mkoch, 2010-03-03)
   else if(0 != poGetSubNode1(pDOMNode, FI_SCHEMA_TYPE_FRAME))  // 
      enResultType = EN_OBJTYPE_FRAME;                          //
   else if(0 != poGetSubNode1(pDOMNode, FI_SCHEMA_TYPE_UNION))  //
      enResultType = EN_OBJTYPE_UNION;                          // --------------------------
   else
   {
      bSuccess = false;
      DOMElement *poDOMElement = poGetSubNode1(pDOMNode, FI_SCHEMA_TYPE_TYPEDEF);
      if(0 != poDOMElement)
      {
         if(bHasAttribute(poDOMElement, FI_SCHEMA_TYPE_ATTR_XSTYPE))
         {
            enResultType = EN_OBJTYPE_ATOM;
            bSuccess = true;
         }
         else
         {
            string oTypeString;
            if(bGetElementAttribute(oTypeString, poDOMElement, FI_SCHEMA_TYPE_ATTR_BASETYPE))
            {
               if(oTypeString[0] == 't')
               {
                  enResultType = EN_OBJTYPE_TYPE;
                  bSuccess = true;
               }
               else
               {
                  enResultType = EN_OBJTYPE_TYPE;
                  bSuccess = true;
               }
            }
         }
      }
   }
   return bSuccess;
}




bool tclFIExtractor::bExtractDescription(
               tclFIDocumentation &oDescription, DOMElement *pDOMElement)
{
   DOMElement *pDOMDesc = poGetSubNode1(pDOMElement, FI_SCHEMA_IDSTRING_DESCRIPTION);
   if(0 != pDOMDesc)
   {
      // vPrintNode(0, pDOMElement);
      DOMNode *pDOMComment = pGetFirstCommentSubNode(pDOMDesc);
      if(0 != pDOMComment)
      {
         if (pDOMComment->getNodeValue() != 0)
         {
            string oDescString = X(pDOMComment->getNodeValue());
            oDescription.vSetDescription(oDescString);
         }
      }
   }
   return true;
}




/*****************************************************************************
 * poGetFINodeFromDOMNode
 * ----------------------
 * Read values for a new FI-Node from a DOM-Node.
 *****************************************************************************/
tclFINode *
tclFIExtractor::poGetFINodeFromDOMNode(
               tclFINodePool *poTargetNodePool,    // Pool to add new node to
               DOMNode *poDOMStartNode,            // Start location in DOM tree
               char *pcNameTag,                    // Tag for node "name"
               char* pcValueTag,                   // Tag for node "default value"
               tenFINodeType enType,               // Type of node to set
               char *pcNamePrefixForCreation)      // Prefix for node's name, has effect
{                                                  // on pool search (exist) of node
   tclFINode  *poNewFINode = 0;

   if(poDOMStartNode->getNodeType() == DOMNode::ELEMENT_NODE)
   {
      // DOMNode
      DOMElement* poDOMElement = static_cast<DOMElement*>(poDOMStartNode);
      // Name
      string oNodeNameString;
      bGetElementAttribute(oNodeNameString, poDOMElement, pcNameTag);
      if(0 != pcNamePrefixForCreation)
         oNodeNameString = string(pcNamePrefixForCreation) + oNodeNameString;
      // ID -> oDefaultValue
      // Christian: So funktioniert das nicht grundstzlich! Default Values hngen am ersten Subknoten
      string oDefaultValue;
      switch (enType)
      {
      case EN_OBJTYPE_TYPE:
         bGetSubNode1Attribute(oDefaultValue, poDOMElement, FI_SCHEMA_TYPE_TYPEDEF, pcValueTag);
         break;
      case EN_OBJTYPE_ENUMERATION:
         bGetSubNode1Attribute(oDefaultValue, poDOMElement, FI_SCHEMA_TYPE_ENUMERATION, pcValueTag);
         break;
      case EN_OBJTYPE_BITFIELD:
         bGetSubNode1Attribute(oDefaultValue, poDOMElement, FI_SCHEMA_TYPE_BITFIELD, pcValueTag);
         break;
      default:
         bGetElementAttribute(oDefaultValue, poDOMElement, pcValueTag);
      }

      // Versionrange
      tclFIVersionRange oVersionRange;
      bExtractVersion(oVersionRange, poDOMElement);
      // Description
      tclFIDocumentation oDescription;
      bExtractDescription(oDescription, poDOMElement);

      // If node exist -> return from pool, else add new node to pool
      if(poTargetNodePool->bExist(oNodeNameString))
      {
         // node could exist because > 1 catalogues are read in -> types are 
         // read more than once, but should not be multiple added 
         poNewFINode = poTargetNodePool->poGetNode(oNodeNameString.c_str());
      }
      else
      {
         poNewFINode = new tclFINode (poDOMElement, oNodeNameString, enType,
                                          oDefaultValue, oVersionRange, oDescription);
         if(0 != poNewFINode)
            poTargetNodePool->bAdd(poNewFINode);
      }
   }
   return poNewFINode;
}




bool
tclFIExtractor::bExtractDependencyLink(
               tclFINode *poDepStartNode,
               tclFITypeNodePool *poSourceTypePool,
               DOMNode *poDOMNode, 
               char *pcNameTag, 
               char *pcBaseTypeTag, 
               char* pcValueTag,
               char* pcFallbackTag,
               tenFIDependencyClass enClass,
               bool bBaseTypeGlobal,
               bool bBaseTypeFromLength)
{
   bool bSuccess = false;
   
   if(poDOMNode->getNodeType() == DOMNode::ELEMENT_NODE)
   {
      // DOMElement
      DOMElement* poDOMElement = static_cast<DOMElement*>(poDOMNode);

      // Basetype
      string oBaseTypeNameString;
      if(bBaseTypeGlobal)
         oBaseTypeNameString = string(pcBaseTypeTag);
      else
         bGetElementAttribute(oBaseTypeNameString, poDOMElement, pcBaseTypeTag);
         if (bBaseTypeFromLength)
         {
            int iLength = atoi(oBaseTypeNameString.c_str());
            if ((iLength > 0) && (iLength <= 8))
               oBaseTypeNameString = "tU8";
            else if (iLength <= 16)
               oBaseTypeNameString = "tU16";
            else if (iLength <= 32)
               oBaseTypeNameString = "tU32";
            else if (iLength <= 64)
               oBaseTypeNameString = "tU64";
            else
            {
               string oElementNameString;
               bGetElementAttribute(oElementNameString, poDOMElement, pcNameTag);
               vError("Invalid length '%s' specified for element '%s' in '%s'!",
                  oBaseTypeNameString.c_str(),oElementNameString.c_str(),poDepStartNode->oGetName().c_str());
            }
         }


      // Name
      string oNodeNameString;
      bGetElementAttribute(oNodeNameString, poDOMElement, pcNameTag);

      // use types from "poSourceTypePool"
      tclFINode *poTargetTypeNode = poSourceTypePool->poGetNode(oBaseTypeNameString.c_str());
      if (poTargetTypeNode == NULL)
      {
         vError("Unknown type '%s' referenced from element '%s' in '%s'!",
            oBaseTypeNameString.c_str(),oNodeNameString.c_str(),poDepStartNode->oGetName().c_str());
      }

      // oDefaultValue
      string oDefaultValue;
      bGetElementAttribute(oDefaultValue, poDOMElement, pcValueTag);

      // oFallbacktValue
      string oFallbackValue;
      bGetElementAttribute(oFallbackValue, poDOMElement, pcFallbackTag);

      // Versionrange
      tclFIVersionRange oVersionRange;
      bExtractVersion(oVersionRange, poDOMElement);
      // description
      tclFIDocumentation oDescription;
      bExtractDescription(oDescription, poDOMElement);

      if ((0 != poTargetTypeNode) && (0 != poDepStartNode))
      {

         // Christian: What is the reason for the following line? poDownlink should be always 0!
         tclFITypeDependency  *poDownLink = poDepStartNode->poFindDependency(oNodeNameString);
         if(0  == poDownLink)
         {
            // install child link
            poDownLink = new tclFITypeDependency (oNodeNameString,enClass,
                                                  oDefaultValue, oFallbackValue,
                                                  oVersionRange,poTargetTypeNode, oDescription);
            if(0 != poDownLink)
               bSuccess = poDepStartNode->bAddDependency(poDownLink);

            /* In case of unions dependencies become asymetric. An UpLink from data type to 
               type specifier is establish but nor vice versa, because of difficulties in size
               calculation.
            if (poElementTargetTypeNode)
            {
               tclFITypeDependency *poElementDownLink = new tclFITypeDependency (oNodeNameString,enClass,
                                                            oDefaultValue, oFallbackValue,
                                                            oVersionRange,poElementTargetTypeNode, oDescription);
               if(0 != poElementDownLink)
                  bSuccess = poDepStartNode->bAddDependency(poElementDownLink);
            }
            */
         }
         else
         {
            string oErrMsg = "Downlink already exists! Node '";
            oErrMsg += poDepStartNode->oGetName() + "' already has child with name '";
            oErrMsg += oNodeNameString + "'.";
            vWarning((char*)oErrMsg.c_str());
         }

         if((bSuccess) && (0 != poTargetTypeNode))
         {
            bSuccess = false;
            // install back link to parent
            tclFITypeDependency *poUpLink = new tclFITypeDependency (oNodeNameString,enClass,
                                                          oDefaultValue, oFallbackValue,
                                                          oVersionRange,poDepStartNode, oDescription);
            if(0 != poUpLink)
               bSuccess = poTargetTypeNode->bAddParentLink(poUpLink);

            // check for type dependencies in enumeration or bitfield elements
            // these may be used in unions.
            if ((enClass == EN_DEPENDENCY_CLASS_UNION) &&
                ((poTargetTypeNode->enGetType() == EN_OBJTYPE_ENUMERATION) ||
                 (poTargetTypeNode->enGetType() == EN_OBJTYPE_BITFIELD)))
            {
               DOMNodeList *poSelectNodeList2;
               if (poTargetTypeNode->enGetType() == EN_OBJTYPE_ENUMERATION)
                  poSelectNodeList2 = poGetNodeListFromElement(poTargetTypeNode->poGetDOMElement(), FI_SCHEMA_TYPE_ENUM_ELEM);
               else
                  poSelectNodeList2 = poGetNodeListFromElement(poTargetTypeNode->poGetDOMElement(), FI_SCHEMA_TYPE_BITFIELD_BIT);

               if(0 != poSelectNodeList2)
               {
                  int iMaxSelected2 = poSelectNodeList2->getLength();
                  for(int iSelectedIndex2=0; iSelectedIndex2<iMaxSelected2; iSelectedIndex2++)
                  {
                     DOMNode *poDOMSubElemNode = poSelectNodeList2->item(iSelectedIndex2);
                     if(0 != poDOMSubElemNode)
                     {
                        // Name
                        string oElementNameString;
                        bGetElementAttribute(oElementNameString, poDOMSubElemNode, pcNameTag);
                        tclFINode *poElementTargetTypeNode = 
                           poElementTargetTypeNode = poSourceTypePool->poGetNode(oElementNameString.c_str());
                        if (poElementTargetTypeNode)
                        {
                           tclFITypeDependency *poElementUpLink = 
                              new tclFITypeDependency ("",
                                                       EN_DEPENDENCY_CLASS_ENUMERATION,
                                                       oDefaultValue, oFallbackValue,
                                                       oVersionRange,poTargetTypeNode, oDescription);
                           if(0 != poElementUpLink)
                              bSuccess = poElementTargetTypeNode->bAddParentLink(poElementUpLink);
                        }
                     }
                  }
               }
            }
         }
      }
   }
   return bSuccess;
}



               
bool
tclFIExtractor::bExtractDependencyData(
               tclFINode   *poTargetNode,
               tclFITypeNodePool *poSourceTypePool,
               DOMElement *poDOMStartElementOrigin, 
               char *pcSelector1Tag, 
               char *pcSelector2Tag, 
               char *pcNameTag,
               char *pcBaseTypeTag,
               char* pcValueTag,
               char* pcFallbackTag,
               tenFIDependencyClass enClass,
               bool bBaseTypeGlobal,
               bool bBaseTypeFromLength)
{
   bool bSuccess = false;
   if(0 != poDOMStartElementOrigin)
   {
      bSuccess = true;
      tclFITypeDependencyList *poTargetDepdendencyList = poTargetNode->poGetDependencyList();

      DOMElement *poDOMStartElement = poDOMStartElementOrigin;
      if(0 != pcSelector1Tag)
         poDOMStartElement = poGetSubNode1(poDOMStartElementOrigin, pcSelector1Tag);

      if(0 == poDOMStartElement)
      {
         vPrintNode(0, poDOMStartElementOrigin);
         assert(0);
      }

      string oBaseTypeString;
      if(bBaseTypeGlobal)
         bGetElementAttribute(oBaseTypeString, poDOMStartElement, pcBaseTypeTag);
      else
         oBaseTypeString = string(pcBaseTypeTag);

      DOMNodeList *poSelectNodeList2 = poGetNodeListFromElement(poDOMStartElement, pcSelector2Tag);
      if(0 != poSelectNodeList2)
      {
         int iMaxSelected2 = poSelectNodeList2->getLength();
         for(int iSelectedIndex2=0; iSelectedIndex2<iMaxSelected2; iSelectedIndex2++)
         {
            DOMNode *poDOMSubElemNode = poSelectNodeList2->item(iSelectedIndex2);
            if(0 != poDOMSubElemNode)
            {
               pcBaseTypeTag = const_cast<char*>(oBaseTypeString.c_str());
               bSuccess &= bExtractDependencyLink(poTargetNode,
                                                  poSourceTypePool, 
                                                  poDOMSubElemNode, 
                                                  pcNameTag,
                                                  pcBaseTypeTag,
                                                  pcValueTag,
                                                  pcFallbackTag,
                                                  enClass,
                                                  bBaseTypeGlobal,
                                                  bBaseTypeFromLength);
            }
         }
      }
   }
   return bSuccess;
}

bool
tclFIExtractor::bExtractDependencyData2(
               tclFINode   *poTargetNode,
               tclFITypeNodePool *poSourceTypePool,
               DOMElement *poDOMStartElementOrigin, 
               char *pcSelector1Tag, 
               char *pcNameTag,
               char *pcBaseTypeTag,
               char* pcValueTag,
               char* pcFallbackTag,
               bool bBaseTypeGlobal)
{
   bool bSuccess = false;
   if(0 != poDOMStartElementOrigin)
   {
      bSuccess = true;
      tclFITypeDependencyList *poTargetDepdendencyList = poTargetNode->poGetDependencyList();

      DOMElement *poDOMStartElement = poDOMStartElementOrigin;
      if(0 != pcSelector1Tag)
         poDOMStartElement = poGetSubNode1(poDOMStartElementOrigin, pcSelector1Tag);

      if(0 == poDOMStartElement)
      {
         vPrintNode(0, poDOMStartElementOrigin);
         assert(0);
      }

      string oBaseTypeString;
      if(bBaseTypeGlobal)
         bGetElementAttribute(oBaseTypeString, poDOMStartElement, pcBaseTypeTag);
      else
         oBaseTypeString = string(pcBaseTypeTag);


      DOMNodeList *poSelectNodeList2 = poDOMStartElement->getChildNodes();
      if(0 != poSelectNodeList2)
      {
         int iMaxSelected2 = poSelectNodeList2->getLength();
         for(int iSelectedIndex2=0; iSelectedIndex2<iMaxSelected2; iSelectedIndex2++)
         {
            DOMNode *poDOMSubElemNode = poSelectNodeList2->item(iSelectedIndex2);
            if(0 != poDOMSubElemNode)
            {
               pcBaseTypeTag = const_cast<char*>(oBaseTypeString.c_str());
               string oDOMNodeName = X(poDOMSubElemNode->getNodeName());
               tenFIDependencyClass enClass = EN_DEPENDENCY_CLASS_UNDEFINED;
               if (oDOMNodeName == FI_SCHEMA_TYPE_STRUCTURE_FIELD)
                  enClass = EN_DEPENDENCY_CLASS_FIELD;
               if (oDOMNodeName == FI_SCHEMA_TAG_MSG_PARAM)
               {
                  enClass = EN_DEPENDENCY_CLASS_PARAM;
                  pcBaseTypeTag = FI_SCHEMA_TAG_MSG_PARAM_TYPE;
               }
               if (oDOMNodeName == FI_SCHEMA_TYPE_STRUCTURE_LIST)
                  enClass = EN_DEPENDENCY_CLASS_LIST;
               if (oDOMNodeName == FI_SCHEMA_TYPE_STRUCTURE_UNION)
                  enClass = EN_DEPENDENCY_CLASS_UNION;
               if (enClass != EN_DEPENDENCY_CLASS_UNDEFINED)
               {
                  bSuccess &= bExtractDependencyLink(poTargetNode,
                                                     poSourceTypePool, 
                                                     poDOMSubElemNode, 
                                                     pcNameTag,
                                                     pcBaseTypeTag,
                                                     pcValueTag,
                                                     pcFallbackTag,
                                                     enClass,
                                                     bBaseTypeGlobal);
               }
            }
         }
      }
   }
   return bSuccess;
}


               
bool tclFIExtractor::bExtractTypes(tclFITypeNodePool *poTargetPool,
                                   char *pcTypeTag, char *pcNameTag, char *pcValueTag,
                                   bool bCalculateType, tenFINodeType enType)
{
   bool bSuccess = true;
   
   // Collect all "type" elements 
   DOMNodeList* poNodeList = poGetNodeListFromDoc(poDOMDoc, pcTypeTag);
   int iMaxIndex  = poNodeList->getLength();
   for(int iIndex=0; (iIndex < iMaxIndex) && (true == bSuccess); iIndex++)
   {
      DOMNode *poDOMNode = poNodeList->item(iIndex);

      if(bCalculateType)
         bSuccess = bExtractTypeType(enType, poDOMNode);
      
      if(bSuccess)
      {
         tclFINode  *poNewFINode = poGetFINodeFromDOMNode(poTargetPool, poDOMNode, pcNameTag, pcValueTag, enType, "");
         if(0 == poNewFINode)
         {
            bSuccess = false;
         }
      }
   }
   return bSuccess;
}





// Extra methode, da Zwischenknoten fr Aufnahme des OpCodes erstellt werden mu. 
bool tclFIExtractor::bExtractMessages(
               tclFITypeNodePool *poTargetPool,
               char *pcTypeTag, 
               char *pcNameTag, 
               char *pcValueTag,
               bool bCalculateType, 
               tenFINodeType enType,
               char *pcOpCodeTypeTag)
{
   bool bSuccess = false;

   DOMNodeList* poDOMMsgMainList = poGetNodeListFromDoc(poDOMDoc, pcTypeTag);
   if(0 != poDOMMsgMainList )
   {
      // iterate all messages or properties
      bSuccess = true;
      int iMaxMainIndex = poDOMMsgMainList->getLength();
      for(int iMainIndex=0; iMainIndex<iMaxMainIndex; iMainIndex++)
      {
         DOMNode *poMainNode = poDOMMsgMainList->item(iMainIndex);

         // Create main message node
         tclFINode  *poFIMsgMainNode = poGetFINodeFromDOMNode(poTargetPool, poMainNode, 
                                                              pcNameTag, pcValueTag, 
                                                              enType, "");
         if(0 != poFIMsgMainNode)
         {
            // Create intermediate node(s)
            bSuccess &= bExtractMessageOpCodes(poTargetPool, poFIMsgMainNode, pcOpCodeTypeTag);
         }
         else
         {
            bSuccess = false;
         }
      }
   }
   return bSuccess;
}




bool tclFIExtractor::bExtractMessageOpCodes(
               tclFITypeNodePool *poTargetPool,
               tclFINode *poMsgMainNode,
               char *pcOpCodeTypeTag)
{
   bool bSuccess = false;
   DOMElement *pDOMMainElement = poMsgMainNode->poGetDOMElement();
   if(0 != pDOMMainElement)
   {
      // Iterate pmessage or mmessage elements
      DOMNodeList *poDOMSubMsgList = poGetNodeListFromElement(pDOMMainElement, pcOpCodeTypeTag);
      if(0 != poDOMSubMsgList)
      {
         bSuccess = true;
         int iSubMainIndex = poDOMSubMsgList->getLength();
         for(int iSubIndex=0; iSubIndex<iSubMainIndex; iSubIndex++)
         {
            // create intermediate node for each opcode
            DOMNode *poDOMSubNode = poDOMSubMsgList->item(iSubIndex);
            
            string oOpCodeNamePrefix = poMsgMainNode->oGetName()+string(".");
            tclFINode  *poNewOpCodeNode = poGetFINodeFromDOMNode(poTargetPool, 
                                                                 poDOMSubNode, 
                                                                 FI_SCHEMA_OPCODE_TAG, 
                                                                 "", 
                                                                 EN_OBJTYPE_OPCODE, 
                                                                 const_cast<char*>(oOpCodeNamePrefix.c_str()));
            // connect mainnode + subnode
            string pMainSubLinkType = poNewOpCodeNode->oGetName();
            bSuccess &= bExtractDependencyLink(poMsgMainNode,
                                               poTargetPool,
                                               poDOMSubNode,
                                               "opcode",
                                               const_cast<char*>(pMainSubLinkType.c_str()),
                                               "", // value
                                               "", // fallback value
                                               EN_DEPENDENCY_CLASS_OPCODE,
                                               true);
         }
      }
   }
   return bSuccess;
}




bool tclFIExtractor::bExtractDependencies(
               tclFINodePool *poSourceTypePool, 
               tclFINodePool *poSourceErrorPool,
               tclFINodePool *poIteratePool)
{
   bool bSuccess = true;

   int iMaxTypes = poIteratePool->iGetNumNodes();
   for(int i=0; i < iMaxTypes; i++)
   {
      tclFINode *poType = poIteratePool->poGetNode(i, EN_NONE);
      if((0 != poType) && (FI_NODE_SIZE_NOTCALC == poType->iGetDataSize()))

      {
         DOMElement *poDOMElement = poType->poGetDOMElement();
         //tclFITypeDependencyList *poTargetDepList = poType->poGetDependencyList();
         switch(poType->enGetType())
         {
            case EN_OBJTYPE_ATOM:
               break;

// MOST-compatibility (mkoch, 2010-03-09)
            case EN_OBJTYPE_ARRAY:
               bSuccess &= bExtractDependencyData(poType, poSourceTypePool,  
                                          poDOMElement, FI_SCHEMA_TYPE_ARRAY, "field",
                                          FI_SCHEMA_IDSTRING_NAME, FI_SCHEMA_TYPE_ATTR_BASETYPE, 
                                          FI_SCHEMA_IDSTRING_DEFAULTVALUE, // value
                                          "", // fallback-value
                                          EN_DEPENDENCY_CLASS_DERIVED, false);
               break;

            case EN_OBJTYPE_UNION:
               bSuccess &= bExtractDependencyData(poType, poSourceTypePool,  
                                          poDOMElement, FI_SCHEMA_TYPE_UNION, "field",
                                          FI_SCHEMA_IDSTRING_NAME, FI_SCHEMA_TYPE_ATTR_BASETYPE, 
                                          FI_SCHEMA_IDSTRING_DEFAULTVALUE, // value
                                          "", // fallback-value
                                          EN_DEPENDENCY_CLASS_DERIVED, false);
               break;

            case EN_OBJTYPE_FRAME:
               bSuccess &= bExtractDependencyData(poType, poSourceTypePool,  
                                          poDOMElement, FI_SCHEMA_TYPE_FRAME,
                                          "signal", FI_SCHEMA_IDSTRING_NAME, 
                                          "length", "", // value
                                          "", // fallback-value
                                          EN_DEPENDENCY_CLASS_BITFIELD, false, true);
               break;
// End of MOST-compatibility (mkoch, 2010-03-09)

            case EN_OBJTYPE_TYPE:
               bSuccess &= bExtractDependencyData(poType, poSourceTypePool,  
                                          poDOMElement, 0, FI_SCHEMA_TYPE_TYPEDEF,
                                          FI_SCHEMA_IDSTRING_NAME, FI_SCHEMA_TYPE_ATTR_BASETYPE, 
                                          FI_SCHEMA_IDSTRING_DEFAULTVALUE, // value
                                          "", // fallback-value
                                          EN_DEPENDENCY_CLASS_DERIVED, false);
               break;
            case EN_OBJTYPE_BITFIELD:
               bSuccess &= bExtractDependencyData(poType, poSourceTypePool, 
                                          poDOMElement, FI_SCHEMA_TYPE_BITFIELD, 
                                          FI_SCHEMA_TYPE_BITFIELD_BIT, FI_SCHEMA_IDSTRING_NAME,
                                          FI_SCHEMA_TYPE_ATTR_BASETYPE, 
                                          FI_SCHEMA_IDSTRING_VALUE,     // value
                                          "",                           // fallback
                                          EN_DEPENDENCY_CLASS_BITFIELD, true);
               break;
            case EN_OBJTYPE_ENUMERATION:
               bSuccess &= bExtractDependencyData(poType, poSourceTypePool, 
                                          poDOMElement, FI_SCHEMA_TYPE_ENUMERATION,
                                          FI_SCHEMA_TYPE_ENUM_ELEM, FI_SCHEMA_IDSTRING_NAME,
                                          FI_SCHEMA_TYPE_ATTR_BASETYPE, 
                                          FI_SCHEMA_IDSTRING_VALUE,    // value
                                          FI_SCHEMA_TAG_FALLBACKVALUE, // fallback
                                          EN_DEPENDENCY_CLASS_ENUMERATION, true);
               break;
            case EN_OBJTYPE_STRUCTURE:
               bSuccess &= bExtractDependencyData2(poType,poSourceTypePool,
                                          poDOMElement, FI_SCHEMA_TYPE_STRUCTURE,
                                          FI_SCHEMA_IDSTRING_NAME,
                                          FI_SCHEMA_TYPE_ATTR_BASETYPE, 
                                          FI_SCHEMA_IDSTRING_DEFAULTVALUE, // value
                                          "",                              // fallback
                                          false);
               /* An Christian: so funktioniert das nicht! ndert die Reihenfolge der Paramter!!
               // All fields
               bSuccess &= bExtractDependencyData(poType,poSourceTypePool,
                                          poDOMElement, FI_SCHEMA_TYPE_STRUCTURE,
                                          FI_SCHEMA_TYPE_STRUCTURE_FIELD, FI_SCHEMA_IDSTRING_NAME,
                                          FI_SCHEMA_TYPE_ATTR_BASETYPE, 
                                          FI_SCHEMA_IDSTRING_DEFAULTVALUE, // value
                                          "",                              // fallback
                                          EN_DEPENDENCY_CLASS_FIELD, false);
               // All lists
               bSuccess &= bExtractDependencyData(poType,poSourceTypePool, 
                                          poDOMElement, FI_SCHEMA_TYPE_STRUCTURE,
                                          FI_SCHEMA_TYPE_STRUCTURE_LIST, FI_SCHEMA_IDSTRING_NAME,
                                          FI_SCHEMA_TYPE_ATTR_BASETYPE,  
                                          FI_SCHEMA_IDSTRING_DEFAULTVALUE, // value
                                          "",                              // fallback
                                          EN_DEPENDENCY_CLASS_LIST, false);
               // All unions
               bSuccess &= bExtractDependencyData(poType, poSourceTypePool,
                                          poDOMElement, FI_SCHEMA_TYPE_STRUCTURE,
                                          FI_SCHEMA_TYPE_STRUCTURE_UNION, FI_SCHEMA_IDSTRING_NAME,
                                          FI_SCHEMA_TYPE_ATTR_BASETYPE, 
                                          FI_SCHEMA_IDSTRING_DEFAULTVALUE, // value
                                          "",                              // fallback
                                          EN_DEPENDENCY_CLASS_UNION, false);
                                          */
                  break;
            case EN_OBJTYPE_MESSAGE_METHOD:
            case EN_OBJTYPE_MESSAGE_PROPERTY:
               {
                  // iterate sub nodes
                  tclFITypeDependencyList *pSubDepList = poType->poGetDependencyList();
                  if(0 != pSubDepList)
                  {
                     tclFITypeDependencyListIterator iter = pSubDepList->begin();
                     while(iter != pSubDepList->end())
                     {
                        tclFITypeDependency *poTargetDep = *iter;
                        tclFINode *poTargetNode = poTargetDep->poGetTarget();
                        DOMElement *poDOMTargetNode = poTargetNode->poGetDOMElement();
                        // vPrintNode(0, poDOMTargetNode);
                        // extract dependencies
                        if(poTargetDep->oGetName().compare("Error") == 0)
                        {
                           bSuccess &= bExtractDependencyData(poTargetNode,
                                                      poSourceErrorPool, 
                                                      poDOMTargetNode, 
                                                      0,
                                                      FI_SCHEMA_TAG_MSG_ERROR, // Sel 2
                                                      FI_SCHEMA_TAG_MSG_ERRORCODE,
                                                      FI_SCHEMA_TAG_MSG_ERRORCODE, 
                                                      0, // value
                                                      0, // fallback
                                                      EN_DEPENDENCY_CLASS_ERRORDEF,
                                                      false);
                        }
                        //else   /* create dependencies either way  */
                        {
                           bSuccess &= bExtractDependencyData2(poTargetNode,
                                                      poSourceTypePool, 
                                                      poDOMTargetNode, 
                                                      0,
                                                      FI_SCHEMA_IDSTRING_NAME,
                                                      FI_SCHEMA_TAG_MSG_LIST_TYPE, 
                                                      FI_SCHEMA_IDSTRING_DEFAULTVALUE, // value
                                                      "", // fallback
                                                      false);
                           /* An Christian: so funktioniert das nicht! ndert die Reihenfolge der Paramter!!
                           bSuccess &= bExtractDependencyData(poTargetNode,
                                                      poSourceTypePool, 
                                                      poDOMTargetNode, 
                                                      0,
                                                      FI_SCHEMA_TAG_MSG_PARAM, // Sel 2
                                                      FI_SCHEMA_IDSTRING_NAME,
                                                      FI_SCHEMA_TAG_MSG_PARAM_TYPE, 
                                                      FI_SCHEMA_IDSTRING_DEFAULTVALUE, // value
                                                      "", // fallback
                                                      EN_DEPENDENCY_CLASS_PARAM, 
                                                      false);
                           bSuccess &= bExtractDependencyData(poTargetNode,
                                                      poSourceTypePool, 
                                                      poDOMTargetNode, 
                                                      0,
                                                      FI_SCHEMA_TAG_MSG_LIST, // Sel 2
                                                      FI_SCHEMA_IDSTRING_NAME,
                                                      FI_SCHEMA_TAG_MSG_LIST_TYPE, 
                                                      0, // value
                                                      0, // fallback
                                                      EN_DEPENDENCY_CLASS_LIST, 
                                                      false);
                                                      */
                        }
                        iter++;
                     }
                  }
               }
               break;
            case EN_OBJTYPE_ERRORDEF:
               break;
            default: ;
               // EN_OBJTYPE_UNDEFINED,
         }
      }
   }
   
   return bSuccess;
}




int  tclFIExtractor::iCalculateTypeSize(tclFINode *poNode)
{
   // toDO: check this!
   if(0 == poNode)
      assert(0);

   // Already calculated?
   int iDataSize = poNode->iGetDataSize();
   if(FI_NODE_SIZE_NOTCALC == iDataSize)
   {
      /*
      // Has Version information --> ..._DYNAMIC
      if(poNode->bHasVersionInformation())
      {
         iDataSize = FI_NODE_SIZE_DYNAMIC;
      }
      else
      */
      {
         iDataSize = 0;
         switch(poNode->enGetType())
         {
            case EN_OBJTYPE_ERRORDEF:
               iDataSize = SIZEOF_ERROR_NODE;
               break;
            case EN_OBJTYPE_ATOM:
               iDataSize = poNode->iGetDataSize();
               break;
            case EN_OBJTYPE_MESSAGE_METHOD:
            case EN_OBJTYPE_MESSAGE_PROPERTY:
               iDataSize = 0;
               break;
            case EN_OBJTYPE_OPCODE:
            case EN_OBJTYPE_TYPE:
            case EN_OBJTYPE_STRUCTURE:
               {
                  iDataSize = 0;
                  // Iterate dependencies
                  tclFITypeDependencyList *poDepList = poNode->poGetDependencyList();
                  if(0 != poDepList)
                  {
                     tclFITypeDependencyListIterator oDepIter = poDepList->begin();
                     while((oDepIter != poDepList->end()))
                     {
                        if(((*oDepIter)->bHasVersionInformation())
                         || (EN_DEPENDENCY_CLASS_UNION == (*oDepIter)->enGetClass() )
                         || (EN_DEPENDENCY_CLASS_LIST == (*oDepIter)->enGetClass() ))
                        {
                           iDataSize = FI_NODE_SIZE_DYNAMIC;
                           break;
                        }
                        else if(EN_DEPENDENCY_CLASS_ERRORDEF == (*oDepIter)->enGetClass())
                        {
                           iDataSize = SIZEOF_ERROR_NODE;
                           break;
                        }
                        else
                        {
                           int iTempSize = iCalculateTypeSize((*oDepIter)->poGetTarget());
                           if(iTempSize == FI_NODE_SIZE_DYNAMIC)
                           {
                              iDataSize = FI_NODE_SIZE_DYNAMIC;
                              break;
                           }
                           iDataSize += iTempSize;
                        }
                        oDepIter++;
                     }
                  }
               }
               break;
            case EN_OBJTYPE_ENUMERATION:
            case EN_OBJTYPE_BITFIELD:
               // base type of node
                  iDataSize = 0;
                  // first dependency is representant for basetype!
                  tclFITypeDependencyList *poDepList = poNode->poGetDependencyList();
                  if ((0 != poDepList) && (poDepList->size() >  0))
                  {
                     tclFITypeDependency *poFirstDep = (*poDepList)[0];
                     if(0 != poFirstDep)
                     {
                        tclFINode *poFirstNode = poFirstDep->poGetTarget();
                        if(0 != poFirstNode)
                           iDataSize = poFirstNode->iGetDataSize();
                        else
                           assert(0);
                     }
                     else
                        assert(0);
                  }
                  else
                     assert((0 != poDepList) && (poDepList->size() >  0));
               break;
         }
      }
   }
   return iDataSize;
}




bool tclFIExtractor::bCalculateTypeSize(
               tclFINodePool *poIteratePool)
{
   int iMaxTypes = poIteratePool->iGetNumNodes();
   for(int i=0; i < iMaxTypes; i++)
   {
      tclFINode *poType = poIteratePool->poGetNode(i, EN_BYDEPENDENCY);
      if(0 != poType)
      {
         int iDataSize = iCalculateTypeSize(poType);
         if (FI_NODE_SIZE_NOTCALC == iDataSize)
            assert(0);

         poType->vSetDataSize(iDataSize);
      }
   }
   return true;
}
