/******************************************************************************
* $Revision:   1.8  $
* $Date:   Nov 19 2003 14:12:24  $
* A changelog can be found at the end of this file
*******************************************************************************
* FILE:         et_Trace.cpp
* PROJECT:      ELeNa
*------------------------------------------------------------------------------
* DESCRIPTION:  elena trace functions
*------------------------------------------------------------------------------
* COPYRIGHT:    (c) 2000 Robert Bosch GmbH, Hildesheim
* HISTORY:
* Date      |   Modification       | Author
* 23.05.01  |   K7/EPE     Perick  | initial version
* 13.12.01  |   CM-CR/EES4 Perick  | format string extension
* 15.08.03  |   CM-CR/EES4 Perick  | add errmem handling
*
******************************************************************************/


#include "et_Trace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"
// if etrace is built with this switch this functions support the ETG backend
// the function pointer are set with calling the ET_TRACE_OPEN macro
#endif

#define C_TRACE_MAX 239

OSAL_tIODescriptor et_gfdTrace=0;

static tVoid vDoTrace(tU16 u16CompId, tU16 u16LevelId, const tS8* pcos8Buffer, tU32 u32Length)
{
   if ((et_gfdTrace != 0) && (et_gfdTrace != OSAL_ERROR))
   {
      OSAL_trIOWriteTrace trace;

      trace.u32Class = (tU32)u16CompId;
      trace.u32Level = (tU32)u16LevelId;
      trace.pcos8Buffer = pcos8Buffer;
      trace.u32Length = u32Length;
      (tVoid) OSAL_s32IOWrite(et_gfdTrace, (const tS8*)&trace, sizeof(trace));
   }
}

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
static etg_tVoid et_vDoTraceEtg(etg_tU16 u16TraceClass, etg_tU16 enTraceLevel, const etg_tS8* pcos8Buffer, etg_tU32 u32Length)
{
   if ((et_gfdTrace != 0) && (et_gfdTrace != OSAL_ERROR))
   {
      OSAL_trIOWriteTrace trace;

      trace.u32Class = (tU32)u16TraceClass;
      trace.u32Level = (tU32)enTraceLevel;
      trace.pcos8Buffer = pcos8Buffer;
      trace.u32Length = u32Length;
      (tVoid) OSAL_s32IOWrite(et_gfdTrace, (const tS8*)&trace, sizeof(trace));
   }
}

static etg_tBool et_bIsTraceActiveEtg(etg_tU16 u16TraceClass, etg_tU16 enTraceLevel)
{
   etg_tBool bActive = FALSE;

   if ((et_gfdTrace != 0) && (et_gfdTrace != OSAL_ERROR))
   {
      OSAL_trIOCtrlActivTrace rActive;

      rActive.enTraceClass = (TR_tenTraceClass)u16TraceClass;
      rActive.enTraceLevel = (TR_tenTraceLevel)enTraceLevel;
      rActive.bIsActive = FALSE;

      if (OSAL_OK == OSAL_s32IOControl(et_gfdTrace,
                                       OSAL_C_S32_IOCTRL_ISACTIVE,
                                       (intptr_t)&rActive))
      {
         bActive = (etg_tBool)rActive.bIsActive;
      }
   }

   return bActive;
}

static etg_tS32 et_s32RegTraceCallbackEtg(etg_tU16 eChanId, ETG_tpfCallback pfvCb) {
   tS32 s32Error = OSAL_ERROR;
   OSAL_trIOCtrlLaunchChannel  oTraceChannel;
   oTraceChannel.enTraceChannel = (TR_tenTraceChan)eChanId;
   oTraceChannel.pCallback = (OSAL_tpfCallback)pfvCb;
   OSAL_tIODescriptor fdTrace = OSAL_IOOpen( OSAL_C_STRING_DEVICE_TRACE, OSAL_EN_READWRITE );

   if(fdTrace != OSAL_ERROR) {
      s32Error = OSAL_s32IOControl( fdTrace, OSAL_C_S32_IOCTRL_CALLBACK_REG, (intptr_t)&oTraceChannel );
   }
   tS32 s32Error2 = OSAL_s32IOClose( fdTrace );
   return (etg_tS32)(s32Error==OSAL_ERROR)?s32Error:s32Error2;
}

static etg_tS32 et_s32UnregTraceCallbackEtg(etg_tU16 eChanId, ETG_tpfCallback pfvCb) {
   tS32 s32Error = OSAL_ERROR;
   OSAL_trIOCtrlLaunchChannel  oTraceChannel;
   oTraceChannel.enTraceChannel = (TR_tenTraceChan)eChanId;
   oTraceChannel.pCallback = (OSAL_tpfCallback)pfvCb;
   OSAL_tIODescriptor fdTrace = OSAL_IOOpen( OSAL_C_STRING_DEVICE_TRACE, OSAL_EN_READWRITE );

   if(fdTrace != OSAL_ERROR) {
      s32Error = OSAL_s32IOControl( fdTrace, OSAL_C_S32_IOCTRL_CALLBACK_UNREG, (intptr_t)&oTraceChannel );
   }
   tS32 s32Error2 = OSAL_s32IOClose( fdTrace );
   return (etg_tS32)(s32Error==OSAL_ERROR)?s32Error:s32Error2;
}

#ifdef VARIANT_S_FTR_ENABLE_ETRACE_ERRMEM
static etg_tVoid et_vDoTraceErrmemEtg(etg_tU16 u16Entry, etg_tInt eEntryType, etg_tU16 u16EntryLength, const etg_tU8* pcou8Buffer)
{
   OSAL_tIODescriptor fd_errmem = 0;
   trErrmemEntry  rErrmemEntry = {0};

   fd_errmem = OSAL_IOOpen(OSAL_C_STRING_DEVICE_ERRMEM, OSAL_EN_WRITEONLY);

   if(fd_errmem != OSAL_ERROR)
   {
      rErrmemEntry.u16Entry = u16Entry;
      rErrmemEntry.eEntryType = (tenErrmemEntryType)eEntryType;
      rErrmemEntry.u16EntryCount = 1;
      rErrmemEntry.u16EntryLength = u16EntryLength;

      if(OSAL_s32ClockGetTime(&rErrmemEntry.rEntryTime)!=OSAL_OK ){
         (tVoid) OSAL_pvMemorySet(&rErrmemEntry.rEntryTime, 0, sizeof(rErrmemEntry.rEntryTime));
      }
      if(rErrmemEntry.u16EntryLength > ERRMEM_MAX_ENTRY_LENGTH) {
         // ensure no memory writer
         rErrmemEntry.u16EntryLength = ERRMEM_MAX_ENTRY_LENGTH;
      }
      (tVoid) OSAL_pvMemoryCopy(rErrmemEntry.au8EntryData, pcou8Buffer, rErrmemEntry.u16EntryLength);

      (tVoid) OSAL_s32IOWrite( fd_errmem, (tPCS8)&rErrmemEntry, sizeof(rErrmemEntry));

      (tVoid) OSAL_s32IOClose(fd_errmem);
   }
}
#endif // VARIANT_S_FTR_ENABLE_ETRACE_ERRMEM
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN

// Funktionen zum Tracing
static tU32 u32PushWord(tU8* pu8Buf, tU32 u32Index, tU16 u16Word)
{
   pu8Buf[u32Index++] = (tU8)(((u16Word)>>8) & 0xFF);
   pu8Buf[u32Index++] = (tU8) ((u16Word)     & 0xFF);

   return u32Index;
}

static tU32 u32PushLong(tU8* pu8Buf, tU32 u32Index, tU32 u32Long)
{
   pu8Buf[u32Index++] = (tU8)(((u32Long)>>24) & 0xFF);
   pu8Buf[u32Index++] = (tU8)(((u32Long)>>16) & 0xFF);
   pu8Buf[u32Index++] = (tU8)(((u32Long)>> 8) & 0xFF);
   pu8Buf[u32Index++] = (tU8) ((u32Long)      & 0xFF);

   return u32Index;
}

tBool et_vTraceOpen(tVoid)
{
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   if(ETG_pfvDoTrace == NULL) {
      ETG_pfvDoTrace = et_vDoTraceEtg;
   }
   if(ETG_pfbIsTraceActive == NULL) {
      ETG_pfbIsTraceActive = et_bIsTraceActiveEtg;
   }
   if(ETG_pfs32RegTraceCallback == NULL) {
      ETG_pfs32RegTraceCallback = et_s32RegTraceCallbackEtg;
   }
   if(ETG_pfs32UnregTraceCallback == NULL) {
      ETG_pfs32UnregTraceCallback = et_s32UnregTraceCallbackEtg;
   }
#ifdef VARIANT_S_FTR_ENABLE_ETRACE_ERRMEM
   if(ETG_pfvDoErrMemTrace == NULL) {
      ETG_pfvDoErrMemTrace = et_vDoTraceErrmemEtg;
   }
#endif // VARIANT_S_FTR_ENABLE_ETRACE_ERRMEM
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN

   if(et_gfdTrace == 0) {
       et_gfdTrace = OSAL_IOOpen(OSAL_C_STRING_DEVICE_TRACE,OSAL_EN_WRITEONLY);
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
       etg_vSetTraceDevice(et_gfdTrace);
#endif
   }

   if (et_gfdTrace == OSAL_ERROR)
   {
      et_gfdTrace = 0;
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      etg_vSetTraceDevice(et_gfdTrace);
#endif
      return FALSE;
   }

   return TRUE;
}

tBool et_bIsTraceActive(tU16 u16CompID, tU16 u16LevelId)
{
   tBool bActive = FALSE;

   if ((et_gfdTrace != 0) && (et_gfdTrace != OSAL_ERROR))
   {
      OSAL_trIOCtrlActivTrace rActive;

      rActive.enTraceClass = (TR_tenTraceClass)u16CompID;
      rActive.enTraceLevel = (TR_tenTraceLevel)u16LevelId;
      rActive.bIsActive = FALSE;

      if (OSAL_OK == OSAL_s32IOControl(et_gfdTrace,
                                       OSAL_C_S32_IOCTRL_ISACTIVE,
                                       (intptr_t)&rActive))
      {
         bActive = rActive.bIsActive;
      }
   }

   return bActive;
}

tVoid et_vTrace(tU16 u16CompId, tU16 u16LevelId, const tChar *pcFormat, ...)
{
   if (et_bIsTraceActive(u16CompId, u16LevelId))
   {
      tChar pcBuffer[C_TRACE_MAX];
      OSAL_tVarArgList argList;

      OSAL_VarArgStart(argList, pcFormat); //lint !e1055 !e64 !e516 !e530 !e746 !e534 !e416 !e1773 /*Review happened by Grosse-Pawig*/
      (tVoid)OSAL_s32VarNPrintFormat(pcBuffer, C_TRACE_MAX, pcFormat, argList); //lint !e530
      pcBuffer[C_TRACE_MAX-1] = '\0';

      vDoTrace(u16CompId, u16LevelId, (const tS8*)pcBuffer, OSAL_u32StringLength(pcBuffer)+1);
   }
}

tVoid et_vTraceBuffer(tU16 u16CompId, tU16 u16LevelId, tU32 u32Length, const tChar *pcBuffer)
{
   if (et_bIsTraceActive(u16CompId, u16LevelId))
   {
      tChar buffer[C_TRACE_MAX];

      if(u32Length > C_TRACE_MAX)
         u32Length = C_TRACE_MAX;

      (tVoid) OSAL_pvMemoryCopy(buffer, pcBuffer, u32Length);

      vDoTrace(u16CompId, u16LevelId, (const tS8*)buffer, u32Length);
   }
}

tVoid
et_vTraceBinary(tU16 u16CompId, tU16 u16LevelId, ...) //lint !e579 /*Review happened by Grosse-Pawig*/
{
   if (et_bIsTraceActive(u16CompId, u16LevelId))
   {
      tU8 au8Buffer[C_TRACE_MAX];
      tU32 u32Length = 0;
      OSAL_tVarArgList argList;
      tBool bDone = FALSE;
      au8Buffer[0] = 0;
      OSAL_VarArgStart(argList, u16LevelId); //lint !e1055 !e64 !e516 !e530 !e534 !e416 !e662 !e1773 /**Review happened by Grosse-Pawig*/

      while (bDone == FALSE)
      {
         et_enTraceType enType = (et_enTraceType) OSAL_VarArg(argList,tInt/*et_enTraceType*/); //lint !e64 !e78 !e1055 !e516 !e530 !e662 !e746 !e826 /*Review happened by Grosse-Pawig*/
         switch (enType)
         {
            case ET_EN_T8:
            {
               tU8 u8Arg = (tU8) OSAL_VarArg(argList, tInt/*tU8*/); //lint !e64 !e78 !e1055 !e516 !e530 !e661 !e662 /*Review happened by Grosse-Pawig*/
               if (u32Length + sizeof(tU8) < C_TRACE_MAX)
               {
                  au8Buffer[u32Length++] = u8Arg;
               }
               else
               {
                  bDone = TRUE;
               }
               break;
            }

            case ET_EN_T8LIST:
            {
               tUInt unListSize = OSAL_VarArg(argList, tUInt); //lint !e64 !e78 !e1055 !e516 !e530 !e661 !e662 !e732 /*Review happened by Grosse-Pawig*/
               const tU8* pu8Arg = OSAL_VarArg(argList, const tU8*); //lint !e64 !e78 !e1055 !e516 !e530 !e26 !e10 !e661 !e662 /*Review happened by Grosse-Pawig*/

               for (tUInt nIdx = 0;
                    (nIdx < unListSize) && (bDone == FALSE);
                    ++nIdx)
               {
                  if ((u32Length + sizeof(tU8)) < C_TRACE_MAX)
                  {
                     au8Buffer[u32Length++] = pu8Arg[nIdx];
                  }
                  else
                  {
                     bDone = TRUE;
                  }
               }
               break;
            }

            case ET_EN_T16:
            {
               tU16 u16Arg = (tU16) OSAL_VarArg(argList, tInt/*tU16*/); //lint !e64 !e78 !e1055 !e516 !e530 !e661 !e662 /*Review happened by Grosse-Pawig*/
               if (u32Length + sizeof(tU16) < C_TRACE_MAX)
               {
                  u32Length = u32PushWord(au8Buffer, u32Length, u16Arg);
               }
               else
               {
                  bDone = TRUE;
               }
               break;
            }

            case ET_EN_T16LIST:
            {
               tUInt unListSize = OSAL_VarArg(argList, tUInt); //lint !e64 !e78 !e1055 !e516 !e530 !e661 !e662 !e732 /*Review happened by Grosse-Pawig*/
               const tU16* pu16Arg = OSAL_VarArg(argList, const tU16*); //lint !e64 !e78 !e1055 !e516 !e530 !e26 !e10 !e661 !e662 /*Review happened by Grosse-Pawig*/

               for (tUInt nIdx = 0;
                    (nIdx < unListSize) && (bDone == FALSE);
                    ++nIdx)
               {
                  if ((u32Length + sizeof(tU16)) < C_TRACE_MAX)
                  {
                     u32Length = u32PushWord(au8Buffer, u32Length, pu16Arg[nIdx]);
                  }
                  else
                  {
                     bDone = TRUE;
                  }
               }
               break;
            }

            case ET_EN_T32:
            {
               tU32 u32Arg = OSAL_VarArg(argList, tU32); //lint !e64 !e78 !e1055 !e516 !e530 !e661 !e662 !e732 /*Review happened by Grosse-Pawig*/
               if (u32Length + sizeof(tU32) < C_TRACE_MAX)
               {
                  u32Length = u32PushLong(au8Buffer, u32Length, u32Arg);
               }
               else
               {
                  bDone = TRUE;
               }
               break;
            }

            case ET_EN_T32LIST:
            {
               tUInt unListSize = OSAL_VarArg(argList, tUInt); //lint !e64 !e78 !e1055 !e516 !e530 !e661 !e662 !e732 /*Review happened by Grosse-Pawig*/
               const tU32* pu32Arg = OSAL_VarArg(argList, const tU32*); //lint !e64 !e78 !e1055 !e516 !e530 !e26 !e10 !e661 !e662 /*Review happened by Grosse-Pawig*/

               for (tUInt nIdx = 0;
                    (nIdx < unListSize) && (bDone == FALSE);
                    ++nIdx)
               {
                  if ((u32Length + sizeof(tU32)) < C_TRACE_MAX)
                  {
                     u32Length = u32PushLong(au8Buffer, u32Length, pu32Arg[nIdx]);
                  }
                  else
                  {
                     bDone = TRUE;
                  }
               }
               break;
            }

            case ET_EN_STRING:
            {
               tCString coszArg = OSAL_VarArg(argList, tCString); //lint !e64 !e78 !e1055 !e516 !e530 !e661 !e662 /*Review happened by Grosse-Pawig*/
               tU32 u32SLen = OSAL_u32StringLength(coszArg)+1;

               if ((u32Length + u32SLen) > C_TRACE_MAX)
               {
                  u32SLen = C_TRACE_MAX - u32Length;
               }
               (tVoid)OSAL_szStringNCopy((tString)&au8Buffer[u32Length], coszArg, u32SLen);
               au8Buffer[C_TRACE_MAX-1] = '\0';
               u32Length += u32SLen;
               break;
            }

            case ET_EN_DONE:
            default:
            {
               bDone = TRUE;
               break;
            }
         }
      }

      OSAL_VarArgEnd(argList);

      vDoTrace(u16CompId, u16LevelId, (const tS8*)au8Buffer, u32Length);
   }
}

tVoid et_vTraceClose(tVoid)
{
   if(et_gfdTrace != 0)
   {
       (tVoid) OSAL_s32IOClose(et_gfdTrace);
       et_gfdTrace = 0;
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
       etg_vSetTraceDevice(et_gfdTrace);
#endif
   }
}


tBool et_bIsTraceActiveCWrapper(tU16 u16CompID, tU16 u16LevelId)
{
   return et_bIsTraceActive(u16CompID, u16LevelId);
}

tVoid et_vTraceBinaryCWrapper(tU16 u16CompId, tU16 u16LevelId, ...) //lint !e579  /*Review happened by Grosse-Pawig*/
{
   if (et_bIsTraceActive(u16CompId, u16LevelId))
   {

      tU8 au8Buffer[C_TRACE_MAX];
      tU32 u32Length = 0;
      OSAL_tVarArgList argList;
      tBool bDone = FALSE;
      au8Buffer[0] = 0;

      OSAL_VarArgStart(argList, u16LevelId); //lint !e1055 !e64 !e516 !e530 !e534 !e416 !e662 !e1773  /*Review happened by Grosse-Pawig*/

      while (bDone == FALSE)
      {
         et_enTraceType enType = (et_enTraceType) OSAL_VarArg(argList, tInt /*et_enTraceType*/); //lint !e64 !e78 !e1055 !e516 !e530 !e661 !e662 !e826 /*Review happened by Grosse-Pawig*/
         switch (enType)
         {
            case ET_EN_T8:
            {
               tU8 u8Arg = (tU8) OSAL_VarArg(argList, tInt /*tU8*/); //lint !e64 !e78 !e1055 !e516 !e530 !e661 !e662 /*Review happened by Grosse-Pawig*/
               if (u32Length + sizeof(tU8) < C_TRACE_MAX)
               {
                  au8Buffer[u32Length++] = u8Arg;
               }
               else
               {
                  bDone = TRUE;
               }
               break;
            }

            case ET_EN_T8LIST:
            {
               tUInt unListSize = OSAL_VarArg(argList, tUInt); //lint !e64 !e78 !e1055 !e516 !e530 !e661 !e662 !e732 /*Review happened by Grosse-Pawig*/
               const tU8* pu8Arg = OSAL_VarArg(argList, const tU8*); //lint !e64 !e78 !e1055 !e516 !e530 !e26 !e10 !e661 !e662 /*Review happened by Grosse-Pawig*/

               for (tUInt nIdx = 0;
                    (nIdx < unListSize) && (bDone == FALSE);
                    ++nIdx)
               {
                  if ((u32Length + sizeof(tU8)) < C_TRACE_MAX)
                  {
                     au8Buffer[u32Length++] = pu8Arg[nIdx];
                  }
                  else
                  {
                     bDone = TRUE;
                  }
               }
               break;
            }

            case ET_EN_T16:
            {
               tU16 u16Arg = (tU16) OSAL_VarArg(argList, tInt /*tU16*/); //lint !e64 !e78 !e1055 !e516 !e530 !e661 !e662 /*Review happened by Grosse-Pawig*/
               if (u32Length + sizeof(tU16) < C_TRACE_MAX)
               {
                  u32Length = u32PushWord(au8Buffer, u32Length, u16Arg);
               }
               else
               {
                  bDone = TRUE;
               }
               break;
            }

            case ET_EN_T16LIST:
            {
               tUInt unListSize = (tUInt) OSAL_VarArg(argList, tUInt); //lint !e64 !e78 !e1055 !e516 !e530 !e661 !e662 /*Review happened by Grosse-Pawig*/
               const tU16* pu16Arg = OSAL_VarArg(argList, const tU16*); //lint !e64 !e78 !e1055 !e516 !e530 !e26 !e10 !e661 !e662 /*Review happened by Grosse-Pawig*/

               for (tUInt nIdx = 0;
                    (nIdx < unListSize) && (bDone == FALSE);
                    ++nIdx)
               {
                  if ((u32Length + sizeof(tU16)) < C_TRACE_MAX)
                  {
                     u32Length = u32PushWord(au8Buffer, u32Length, pu16Arg[nIdx]);
                  }
                  else
                  {
                     bDone = TRUE;
                  }
               }
               break;
            }

            case ET_EN_T32:
            {
               tU32 u32Arg = (tU32) OSAL_VarArg(argList, tU32); //lint !e64 !e78 !e1055 !e516 !e530 !e661 !e662 /*Review happened by Grosse-Pawig*/
               if (u32Length + sizeof(tU32) < C_TRACE_MAX)
               {
                  u32Length = u32PushLong(au8Buffer, u32Length, u32Arg);
               }
               else
               {
                  bDone = TRUE;
               }
               break;
            }

            case ET_EN_T32LIST:
            {
               tUInt unListSize = (tUInt) OSAL_VarArg(argList, tUInt); //lint !e64 !e78 !e1055 !e516 !e530 !e661 !e662 /*Review happened by Grosse-Pawig*/
               const tU32* pu32Arg = OSAL_VarArg(argList, const tU32*); //lint !e64 !e78 !e1055 !e516 !e530 !e26 !e10 !e661 !e662 /*Review happened by Grosse-Pawig*/

               for (tUInt nIdx = 0;
                    (nIdx < unListSize) && (bDone == FALSE);
                    ++nIdx)
               {
                  if ((u32Length + sizeof(tU32)) < C_TRACE_MAX)
                  {
                     u32Length = u32PushLong(au8Buffer, u32Length, pu32Arg[nIdx]);
                  }
                  else
                  {
                     bDone = TRUE;
                  }
               }
               break;
            }

            case ET_EN_STRING:
            {
               tCString coszArg = OSAL_VarArg(argList, tCString); //lint !e64 !e78 !e1055 !e516 !e530 !e661 !e662 /*Review happened by Grosse-Pawig*/
               tU32 u32SLen = OSAL_u32StringLength(coszArg)+1;

               if ((u32Length + u32SLen) > C_TRACE_MAX)
               {
                  u32SLen = C_TRACE_MAX - u32Length;
               }
               (tVoid)OSAL_szStringNCopy((tString)&au8Buffer[u32Length], coszArg, u32SLen);
               au8Buffer[C_TRACE_MAX-1] = '\0';
               u32Length += u32SLen;
               break;
            }

            case ET_EN_DONE:
            default:
            {
               bDone = TRUE;
               break;
            }
         }
      }

      OSAL_VarArgEnd(argList);

      vDoTrace(u16CompId, u16LevelId, (const tS8*)au8Buffer, u32Length);
   }
}

tVoid et_vErrmemStringNormal(tU16 u16Entry, const tChar* pcFormat, ...)
{
   OSAL_tIODescriptor fd_errmem;
   trErrmemEntry  rErrmemEntry = {0};
   tChar pcBuffer[C_TRACE_MAX];
   OSAL_tVarArgList argList;

   fd_errmem = OSAL_IOOpen(OSAL_C_STRING_DEVICE_ERRMEM, OSAL_EN_WRITEONLY);

   if(fd_errmem != OSAL_ERROR)
   {
      OSAL_VarArgStart(argList, pcFormat);  //lint !e1055 !e64 !e516 !e530 !e534 !e416 !e662 !e1773  /*Review happened by Grosse-Pawig*/
      (tVoid) OSAL_s32VarNPrintFormat(pcBuffer, C_TRACE_MAX, pcFormat, argList); //lint !e530
      pcBuffer[C_TRACE_MAX-1] = '\0';

      rErrmemEntry.u16Entry = u16Entry;
      if(OSAL_s32ClockGetTime(&rErrmemEntry.rEntryTime)!=OSAL_OK ){
         (tVoid) OSAL_pvMemorySet(&rErrmemEntry.rEntryTime, 0, sizeof(rErrmemEntry.rEntryTime));
      }
      rErrmemEntry.eEntryType = eErrmemEntryNormal;
      rErrmemEntry.u16EntryLength = (tU16)(OSAL_u32StringLength(pcBuffer)+1);
      #if ((OSAL_OS == OSAL_TENGINE) || (OSAL_OS == OSAL_LINUX))
      if (rErrmemEntry.u16EntryLength > (ERRMEM_MAX_ENTRY_LENGTH - 2))
      {
         rErrmemEntry.u16EntryLength = ERRMEM_MAX_ENTRY_LENGTH - 2;
      }
      rErrmemEntry.au8EntryData[0] = (tU8) (((tU16)u16Entry) & 0xFF);
      rErrmemEntry.au8EntryData[1] = (tU8) (((tU16)u16Entry) >> 8) & 0xFF;
      (tVoid) OSAL_pvMemoryCopy(&rErrmemEntry.au8EntryData[2], pcBuffer, rErrmemEntry.u16EntryLength);
     #else
      if (rErrmemEntry.u16EntryLength > ERRMEM_MAX_ENTRY_LENGTH)
      {
         rErrmemEntry.u16EntryLength = ERRMEM_MAX_ENTRY_LENGTH;
      }
      (tVoid) OSAL_pvMemoryCopy(rErrmemEntry.au8EntryData, pcBuffer, rErrmemEntry.u16EntryLength);
     #endif
      (tVoid) OSAL_s32IOWrite( fd_errmem, (tPCS8)&rErrmemEntry, sizeof(rErrmemEntry));

      (tVoid) OSAL_s32IOClose(fd_errmem);
   }
}


tVoid et_vErrmemString(tU16 u16Entry, tenErrmemEntryType eType, const tChar* pcFormat, ...)
{
   OSAL_tIODescriptor fd_errmem;
   trErrmemEntry  rErrmemEntry = {0};
   tChar pcBuffer[C_TRACE_MAX];
   OSAL_tVarArgList argList;

   fd_errmem = OSAL_IOOpen(OSAL_C_STRING_DEVICE_ERRMEM, OSAL_EN_WRITEONLY);

   if(fd_errmem != OSAL_ERROR)
   {
      OSAL_VarArgStart(argList, pcFormat);  //lint !e1055 !e64 !e516 !e530 !e534 !e416 !e662 !e1773  /*Review happened by Grosse-Pawig*/
      (tVoid) OSAL_s32VarNPrintFormat(pcBuffer, C_TRACE_MAX, pcFormat, argList); //lint !e530
      pcBuffer[C_TRACE_MAX-1] = '\0';

      rErrmemEntry.u16Entry = u16Entry;
      if(OSAL_s32ClockGetTime(&rErrmemEntry.rEntryTime)!=OSAL_OK ){
         (tVoid) OSAL_pvMemorySet(&rErrmemEntry.rEntryTime, 0, sizeof(rErrmemEntry.rEntryTime));
      }
      rErrmemEntry.eEntryType = eType;
      rErrmemEntry.u16EntryLength = (tU16)(OSAL_u32StringLength(pcBuffer)+1);
      #if ((OSAL_OS == OSAL_TENGINE) || (OSAL_OS == OSAL_LINUX))
      if (rErrmemEntry.u16EntryLength > (ERRMEM_MAX_ENTRY_LENGTH - 2))
      {
         rErrmemEntry.u16EntryLength = ERRMEM_MAX_ENTRY_LENGTH - 2;
      }
      rErrmemEntry.au8EntryData[0] = (tU8) (((tU16)u16Entry) & 0xFF);
      rErrmemEntry.au8EntryData[1] = (tU8) (((tU16)u16Entry) >> 8) & 0xFF;
      (tVoid) OSAL_pvMemoryCopy(&rErrmemEntry.au8EntryData[2], pcBuffer, rErrmemEntry.u16EntryLength);
      #else
      if (rErrmemEntry.u16EntryLength > ERRMEM_MAX_ENTRY_LENGTH)
      {
         rErrmemEntry.u16EntryLength = ERRMEM_MAX_ENTRY_LENGTH;
      }
      (tVoid) OSAL_pvMemoryCopy(rErrmemEntry.au8EntryData, pcBuffer, rErrmemEntry.u16EntryLength);
      #endif

      (tVoid) OSAL_s32IOWrite( fd_errmem, (tPCS8)&rErrmemEntry, sizeof(rErrmemEntry));

      (tVoid) OSAL_s32IOClose(fd_errmem);
   }
}

/*
 * $Log:   //hi230124/projects/Vasco/swnavi/archives/components/etrace/et_Trace.cpp-arc  $
 *
 *    Rev 1.8   Nov 19 2003 14:12:24   PEC2HI
 * add et_vErrmemString(..)
 *
 *    Rev 1.7   Aug 15 2003 16:24:36   PEC2HI
 * add errmem handling
 *
 *    Rev 1.6   Jul 29 2003 12:55:10   PEC2HI
 * trace buffer size
 *
 *    Rev 1.5   14 May 2003 09:58:50   IR62HI
 * return of value in et_bIsTraceActiveCWrapper()
 *
 *    Rev 1.4   May 12 2003 17:11:34   bef2hi
 * Changes for new GPS library
 *
 *    Rev 1.3   Feb 18 2003 09:52:10   GGS2HI
 * Added functionality:
 * - et_bIsTraceActive() to check whether a given trace level is active
 * - et_vTraceBinary() for binary traces interpreted by TTFis
 * - new macros for binary traces
 */

