//-----------------------------------------------------------------------------
//
//  sha1.c
//
//-----------------------------------------------------------------------------
//
//  Implementation of SHA-1 secure hashing acc. to FIPS-180-1 and
//  Hashed Message Authentication Code HMAC acc. to RFC 2202
//
//  Copyright (c) 2006 Blaupunkt GmbH, Hildesheim
//
//  Author: Holger Listle, CM-DI/ESN
//	(based on code from A.Weimerskirch, M.Wolf, Ruhr University Bochum)
//
//-----------------------------------------------------------------------------


#include "bpcl.h"
#include "bpcl_int.h"

tU8 global_key_mask4[] = { 0x2b, 0xa0, 0x68, 0x4b, 0xe1, 0x8c, 0xb0, 0x86, 0x31, 0xdc, };

// FIPS-180-1 padding sequence
static tU8 sha1_padding[64] = {
 (tU8) 0x80, (tU8) 0, (tU8) 0, (tU8) 0, (tU8) 0, (tU8) 0, (tU8) 0, (tU8) 0,
	(tU8) 0, (tU8) 0, (tU8) 0, (tU8) 0, (tU8) 0, (tU8) 0, (tU8) 0, (tU8) 0,
	(tU8) 0, (tU8) 0, (tU8) 0, (tU8) 0, (tU8) 0, (tU8) 0, (tU8) 0, (tU8) 0, 
	(tU8) 0, (tU8) 0, (tU8) 0, (tU8) 0, (tU8) 0, (tU8) 0, (tU8) 0, (tU8) 0,
	(tU8) 0, (tU8) 0, (tU8) 0, (tU8) 0, (tU8) 0, (tU8) 0, (tU8) 0, (tU8) 0,
	(tU8) 0, (tU8) 0, (tU8) 0, (tU8) 0, (tU8) 0, (tU8) 0, (tU8) 0, (tU8) 0,
	(tU8) 0, (tU8) 0, (tU8) 0, (tU8) 0, (tU8) 0, (tU8) 0, (tU8) 0, (tU8) 0,
	(tU8) 0, (tU8) 0, (tU8) 0, (tU8) 0, (tU8) 0, (tU8) 0, (tU8) 0, (tU8) 0
};

static void sha1_process(tsSHA1Context *ctx, tU8 *byte_64_block);

//-----------------------------------------------------------------------------
// BPCL_SHA1_Init()
//-----------------------------------------------------------------------------

tErrCode BPCL_SHA1_Init(tsSHA1Context *ctx) {

	// byte length = 0
	ctx->total_bytes_Lo = 0;
	ctx->total_bytes_Hi = 0;

	// FIPS 180-1 init values
	ctx->vector[0] = 0x67452301;
	ctx->vector[1] = 0xEFCDAB89;
	ctx->vector[2] = 0x98BADCFE;
	ctx->vector[3] = 0x10325476;
	ctx->vector[4] = 0xC3D2E1F0;

	// successful
	return BPCL_OK;
} // BPCL_SHA1_Init()


//-----------------------------------------------------------------------------
// BPCL_SHA1_Update()
//-----------------------------------------------------------------------------

tErrCode BPCL_SHA1_Update(
	tsSHA1Context		*ctx,			// Initialized Context
    tU8					*p_data,		// data for sha1 update
    tU32				data_len		// length of data in bytes
) {

	tU32	left, fill;
	tU32	i;


	if(data_len == 0) {
		return BPCL_ERR_BAD_PARAMETER;
	}

	// chunk_length = n * 64 byte + left
	left = ctx->total_bytes_Lo & 0x3F;

	// fill bytes remain to 64 byte block
	fill = 64 - left;

	// total = total + chunk_length
	ctx->total_bytes_Lo += data_len;

	// mask 32 bit
	ctx->total_bytes_Lo &= 0xFFFFFFFF;

	if(ctx->total_bytes_Lo < data_len)	{
		ctx->total_bytes_Hi++;
	}

	// if we have something in the buffer (left > 0) and 
	// the chunk has enougth data to fill a 64 byte block (chunk_length >= fill)
	if( (left > 0) && (data_len >= fill) )	{
		// fill buffer with data from new chunk
		for( i = 0; i < fill; i++ ) {
			ctx->buffer[left + i] = p_data[i];
		}

		// process 64 byte buffer block
		sha1_process( ctx, ctx->buffer );

		data_len -= fill;
		p_data   += fill;
		left     = 0;
	}

	// process all remaining 64 byte chunks
	while(data_len >= 64) {
		sha1_process( ctx, p_data );
		data_len -= 64;
		p_data  += 64;
	}

	// if final chunk_length between 1..63 byte
	if(data_len > 0) {
		// append remainder to 64 byte into buffer resp. fill the empty buffer
		for( i = 0; i < data_len; i++ ) {
			ctx->buffer[left + i] = p_data[i];
		}
	}

	// successfull
	return 0;
} // BPCL_SHA1_Update()

//-----------------------------------------------------------------------------
// BPCL_SHA1_Finish()
//-----------------------------------------------------------------------------

tErrCode BPCL_SHA1_Finish(
	tsSHA1Context		*ctx,			// Initialized Context
	tU32				*p_sha1_hash	// Buffer to carry hash value
) {

	tU32			last, padn;
	tU32			high, low;
	tU8				msglen[8];
	tErrCode		ret;


	// build msglen array[8  8-bit] from total[2  32-bit] = n  64 byte
	high = ( ctx->total_bytes_Lo >> 29 ) | ( ctx->total_bytes_Hi <<  3 );
	low  = ( ctx->total_bytes_Lo <<  3 );
	M_U32_TO_BUF( high, msglen, 0 );
	M_U32_TO_BUF( low,  msglen, 4 ); 

	// total = n  64 bytes + last
	last = ctx->total_bytes_Lo & 0x3F;

	// number of padding zeros 
	padn = ( last < 56 ) ? ( 56 - last ) : ( 120 - last );

	// update SHA-1 context with remaining buffer and padding to 64 bytes with bit sequence (1,0,...,0)
	ret = BPCL_SHA1_Update( ctx, sha1_padding, padn );

	// update SHA-1 context with total length
	ret = BPCL_SHA1_Update( ctx, msglen, 8 );

	// assign final hash words
	p_sha1_hash[0] = ctx->vector[0];
	p_sha1_hash[1] = ctx->vector[1];
	p_sha1_hash[2] = ctx->vector[2];
	p_sha1_hash[3] = ctx->vector[3];
	p_sha1_hash[4] = ctx->vector[4];
	(tVoid)ret;
	// successful
	return 0;
} // BPCL_SHA1_Finish()


//-----------------------------------------------------------------------------
// BPCL_HMAC_SHA1()
//-----------------------------------------------------------------------------

tErrCode BPCL_HMAC_SHA1(
	tU8					*p_key,			// key array
	tU32				key_len,		// length of key in bytes
	tU8					*p_data,		// data to be hmac'ed
	tU32				data_len,		// length of data in bytes
	tU32				*p_hmac_hash	// Buffer to carry HMAC
) {

	// declarations
	tsSHA1Context		inner_sha1_ctx;
	tsSHA1Context		outer_sha1_ctx;
	tsSHA1Context		key_sha1_ctx;
	tU32				inner_sha[5]; 
	tU32				key_sha[5];
	tU8				buf[64];
	tU32				i;
	tErrCode			ret;



	// if key_length > 64 byte block, then hash the key
	if( key_len > 64 ) {
		ret = BPCL_SHA1_Init( &key_sha1_ctx );
		ret = BPCL_SHA1_Update( &key_sha1_ctx, p_key, key_len );
		ret = BPCL_SHA1_Finish( &key_sha1_ctx, key_sha );

		M_U32_TO_BUF( key_sha[0], p_key, 0 );
		M_U32_TO_BUF( key_sha[1], p_key, 4 );
		M_U32_TO_BUF( key_sha[2], p_key, 8 );
		M_U32_TO_BUF( key_sha[3], p_key, 12 );
		M_U32_TO_BUF( key_sha[4], p_key, 16 );

		key_len = 20;
	}
  

	// --- inner_sha1_ctx -------------------------

	// init inner_sha1_ctx
	ret = BPCL_SHA1_Init( &inner_sha1_ctx );

	// init buf with 0x36 padding
	for ( i = 0; i < 64; i++ ) {
		buf[i] = (tU8) 0x36;
	}

	// build buffer with key and padding
	for ( i = 0; i < key_len; i++ ) {
		buf[i] = (tU8) ( p_key[i]  ^  0x36 );
	}

	// update with inner_sha1_ctx key and data
	ret = BPCL_SHA1_Update( &inner_sha1_ctx, buf, 64 );
	ret = BPCL_SHA1_Update( &inner_sha1_ctx, p_data, data_len );
	ret = BPCL_SHA1_Finish( &inner_sha1_ctx, inner_sha );

	// --- outer_sha1_ctx -------------------------

	// init outer_sha1_ctx
	ret = BPCL_SHA1_Init( &outer_sha1_ctx ) ;

	// build buffer with key and padding
	for ( i = 0; i < key_len; i++ ) {
		buf[i] =  (tU8) ( p_key[i] ^ 0x5C );
	}

	for ( i = key_len; i < 64; i++ )  { 
		buf[i] = (tU8) 0x5C;
	}

	// update outer_sha1_ctx with key
	ret = BPCL_SHA1_Update( &outer_sha1_ctx, buf, 64);

	// fill buffer with inner_sha
	M_U32_TO_BUF( inner_sha[0], buf, 0 );
	M_U32_TO_BUF( inner_sha[1], buf, 4 );
	M_U32_TO_BUF( inner_sha[2], buf, 8 );
	M_U32_TO_BUF( inner_sha[3], buf, 12 );
	M_U32_TO_BUF( inner_sha[4], buf, 16 );

	// update outer_sha1_ctx with inner_sha
	ret = BPCL_SHA1_Update( &outer_sha1_ctx, buf, 20 ) ;

	// finalize outer_sha1_ctx and write to hmac_hash output
	ret = BPCL_SHA1_Finish( &outer_sha1_ctx, p_hmac_hash );

	(tVoid)ret;

	// successful
	return BPCL_OK;

} // BPCL_HMAC_SHA1()



static void sha1_process( tsSHA1Context *ctx, 
                          tU8 *byte_64_block ) {

	tU32	temp, W[16];
	tU32	A, B, C, D, E;

	// concatenate 64 bytes to 16  32-bit words
	M_BUF_TO_U32( W[0],  byte_64_block,  0 );
	M_BUF_TO_U32( W[1],  byte_64_block,  4 );
	M_BUF_TO_U32( W[2],  byte_64_block,  8 );
	M_BUF_TO_U32( W[3],  byte_64_block, 12 );
	M_BUF_TO_U32( W[4],  byte_64_block, 16 );
	M_BUF_TO_U32( W[5],  byte_64_block, 20 );
	M_BUF_TO_U32( W[6],  byte_64_block, 24 );
	M_BUF_TO_U32( W[7],  byte_64_block, 28 );
	M_BUF_TO_U32( W[8],  byte_64_block, 32 );
	M_BUF_TO_U32( W[9],  byte_64_block, 36 );
	M_BUF_TO_U32( W[10], byte_64_block, 40 );
	M_BUF_TO_U32( W[11], byte_64_block, 44 );
	M_BUF_TO_U32( W[12], byte_64_block, 48 );
	M_BUF_TO_U32( W[13], byte_64_block, 52 );
	M_BUF_TO_U32( W[14], byte_64_block, 56 );
	M_BUF_TO_U32( W[15], byte_64_block, 60 );

// extends 16  32-bit words to 80  32-bit words
#define EXTENDED_W(t) (								\
  temp = W[(t -  3) & 0x0F] ^ W[(t - 8) & 0x0F] ^	\
         W[(t - 14) & 0x0F] ^ W[ t      & 0x0F],	\
  ( W[t & 0x0F] = M_ROTL(temp,1) )					\
)

// main formula
#define P(a,b,c,d,e,Wi) {							\
  e += M_ROTL(a,5) + F(b,c,d) + K + Wi;				\
  b  = M_ROTL(b,30);								\
}

  // init A..E
  A = ctx->vector[0];
  B = ctx->vector[1];
  C = ctx->vector[2];
  D = ctx->vector[3];
  E = ctx->vector[4];

// round I (0..19)
#define F(x,y,z) (z ^ (x & (y ^ z)))
#define K 0x5A827999

  P( A, B, C, D, E, W[0]  );
  P( E, A, B, C, D, W[1]  );
  P( D, E, A, B, C, W[2]  );
  P( C, D, E, A, B, W[3]  );
  P( B, C, D, E, A, W[4]  );
  P( A, B, C, D, E, W[5]  );
  P( E, A, B, C, D, W[6]  );
  P( D, E, A, B, C, W[7]  );
  P( C, D, E, A, B, W[8]  );
  P( B, C, D, E, A, W[9]  );
  P( A, B, C, D, E, W[10] );
  P( E, A, B, C, D, W[11] );
  P( D, E, A, B, C, W[12] );
  P( C, D, E, A, B, W[13] );
  P( B, C, D, E, A, W[14] );
  P( A, B, C, D, E, W[15] );
  P( E, A, B, C, D, EXTENDED_W(16) );
  P( D, E, A, B, C, EXTENDED_W(17) );
  P( C, D, E, A, B, EXTENDED_W(18) );
  P( B, C, D, E, A, EXTENDED_W(19) );

#undef K
#undef F

// round II (20..39)
#define F(x,y,z) (x ^ y ^ z)
#define K 0x6ED9EBA1

  P( A, B, C, D, E, EXTENDED_W(20) );
  P( E, A, B, C, D, EXTENDED_W(21) );
  P( D, E, A, B, C, EXTENDED_W(22) );
  P( C, D, E, A, B, EXTENDED_W(23) );
  P( B, C, D, E, A, EXTENDED_W(24) );
  P( A, B, C, D, E, EXTENDED_W(25) );
  P( E, A, B, C, D, EXTENDED_W(26) );
  P( D, E, A, B, C, EXTENDED_W(27) );
  P( C, D, E, A, B, EXTENDED_W(28) );
  P( B, C, D, E, A, EXTENDED_W(29) );
  P( A, B, C, D, E, EXTENDED_W(30) );
  P( E, A, B, C, D, EXTENDED_W(31) );
  P( D, E, A, B, C, EXTENDED_W(32) );
  P( C, D, E, A, B, EXTENDED_W(33) );
  P( B, C, D, E, A, EXTENDED_W(34) );
  P( A, B, C, D, E, EXTENDED_W(35) );
  P( E, A, B, C, D, EXTENDED_W(36) );
  P( D, E, A, B, C, EXTENDED_W(37) );
  P( C, D, E, A, B, EXTENDED_W(38) );
  P( B, C, D, E, A, EXTENDED_W(39) );

#undef K
#undef F

// round III (40..59)
#define F(x,y,z) ((x & y) | (z & (x | y)))
#define K 0x8F1BBCDC

  P( A, B, C, D, E, EXTENDED_W(40) );
  P( E, A, B, C, D, EXTENDED_W(41) );
  P( D, E, A, B, C, EXTENDED_W(42) );
  P( C, D, E, A, B, EXTENDED_W(43) );
  P( B, C, D, E, A, EXTENDED_W(44) );
  P( A, B, C, D, E, EXTENDED_W(45) );
  P( E, A, B, C, D, EXTENDED_W(46) );
  P( D, E, A, B, C, EXTENDED_W(47) );
  P( C, D, E, A, B, EXTENDED_W(48) );
  P( B, C, D, E, A, EXTENDED_W(49) );
  P( A, B, C, D, E, EXTENDED_W(50) );
  P( E, A, B, C, D, EXTENDED_W(51) );
  P( D, E, A, B, C, EXTENDED_W(52) );
  P( C, D, E, A, B, EXTENDED_W(53) );
  P( B, C, D, E, A, EXTENDED_W(54) );
  P( A, B, C, D, E, EXTENDED_W(55) );
  P( E, A, B, C, D, EXTENDED_W(56) );
  P( D, E, A, B, C, EXTENDED_W(57) );
  P( C, D, E, A, B, EXTENDED_W(58) );
  P( B, C, D, E, A, EXTENDED_W(59) );

#undef K
#undef F

// round IV (60..79)
#define F(x,y,z) (x ^ y ^ z)
#define K 0xCA62C1D6

  P( A, B, C, D, E, EXTENDED_W(60) );
  P( E, A, B, C, D, EXTENDED_W(61) );
  P( D, E, A, B, C, EXTENDED_W(62) );
  P( C, D, E, A, B, EXTENDED_W(63) );
  P( B, C, D, E, A, EXTENDED_W(64) );
  P( A, B, C, D, E, EXTENDED_W(65) );
  P( E, A, B, C, D, EXTENDED_W(66) );
  P( D, E, A, B, C, EXTENDED_W(67) );
  P( C, D, E, A, B, EXTENDED_W(68) );
  P( B, C, D, E, A, EXTENDED_W(69) );
  P( A, B, C, D, E, EXTENDED_W(70) );
  P( E, A, B, C, D, EXTENDED_W(71) );
  P( D, E, A, B, C, EXTENDED_W(72) );
  P( C, D, E, A, B, EXTENDED_W(73) );
  P( B, C, D, E, A, EXTENDED_W(74) );
  P( A, B, C, D, E, EXTENDED_W(75) );
  P( E, A, B, C, D, EXTENDED_W(76) );
  P( D, E, A, B, C, EXTENDED_W(77) );
  P( C, D, E, A, B, EXTENDED_W(78) );
  P( B, C, D, E, A, EXTENDED_W(79) );

#undef K
#undef F

  // assign vectors
  ctx->vector[0] += A;
  ctx->vector[1] += B;
  ctx->vector[2] += C;
  ctx->vector[3] += D;
  ctx->vector[4] += E;
} // sha1_process()


#ifdef _BPCL_TEST
#include <string.h> // strlen()

//-----------------------------------------------------------------------------
//	Test implementation
//-----------------------------------------------------------------------------

tErrCode BPCL_SHA1_Test(tU8 test_type) {

	tU32			i, j;
	tU8				buffer[16384] = { 0 };	/* 16kB for performance tests */
	tU32			hash_buf[5];
	tsSHA1Context	ctx;
	tErrCode		rc = BPCL_OK;

	struct {
		tU8		*data;
		tU32	ref_hash[5];
	} test_data[3] = {
		// FIPS-180-1 Test Vectors
		{
			"abc",
			{ (tU32)0xA9993E36, (tU32)0x4706816A, (tU32)0xBA3E2571,
			  (tU32)0x7850C26C, (tU32)0x9CD0D89D }
		}, {
			"abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq",
			{ (tU32)0x84983E44, (tU32)0x1C3BD26E, (tU32)0xBAAE4AA1,
			  (tU32)0xF95129E5, (tU32)0xE54670F1 }
		}, {
			// 1 mio a's -> 50 a's loop 20000 times
			"aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa",
			{ (tU32)0x34AA973C, (tU32)0xD4C4DAA4, (tU32)0xF61EEB2B,
			  (tU32)0xDBAD2731, (tU32)0x6534016F }
		}
	};

	printf("\n----------------------------------------------------------\n");
	printf("             SHA-1 Test\n");
	printf("----------------------------------------------------------\n");


	if(test_type & BPCL_TEST_COMPLIANCE) {
		printf("\nSHA-1 Compliance Test\n");
		for (i = 0; i < 3; i++) {
			BPCL_SHA1_Init(&ctx);
			if(i < 2) {
				BPCL_SHA1_Update(&ctx,test_data[i].data,
					             strlen(test_data[i].data));
			} else {
				for(j = 0; j < 20000; j++) {
					BPCL_SHA1_Update(&ctx,test_data[i].data,
								     strlen(test_data[i].data));
				}
			}
			BPCL_SHA1_Finish(&ctx,hash_buf);

			for(j=0; j<5; j++) {
				if(hash_buf[j] != test_data[i].ref_hash[j]) {
					rc = BPCL_ERR_TEST_FAILED;
					break;
				}
			}

			if(rc == BPCL_OK) {
				printf("SHA-1 Compliance Test #%d okay\n", i);
			} else {
				printf("SHA-1 Compliance Test #%d FAILED\n", i);
				rc = BPCL_OK;
			}
		}
		printf("\nSHA-1 Compliance Test DONE\n");
	}
	
	if(test_type & BPCL_TEST_PERFORMANCE) {

		printf("\nSHA-1 Performance Test\n");

		BPCL_TimerStart();
		BPCL_SHA1_Init(&ctx);
		for(i = 1; i < 640; ++i) {
			/* Loop over 10MB of data (640 * 16kB) */
			BPCL_SHA1_Update(&ctx, buffer, 16384);
		}
		BPCL_SHA1_Finish(&ctx,hash_buf);
		BPCL_TimerReport("SHA-1 Hashing@10MB");

		printf("\nSHA-1 Performance Test DONE\n");
	}
	return rc;

}

#endif
