/***************************************************************************/
/*! 
    \file        random_gen.c

    \brief       Basic random generation functions

    \author      Marko Wolf (mwolf@crypto.rub.de),
                 Andr Weimerskirch (weika@crypto.rub.de)
				 Adopted to BPCL CM-DI/ESN-Listle

    \version     1.1
    
    \date        July 19, 2004
    
    \warning     None

    \par Reviews:
    \li          19.07.2004 Splint 3.1.1 (no warnings)

*/
/***************************************************************************/

/***************************************************************************
* 1. INCLUDES                                                              *	
****************************************************************************/

#include "bpcl.h"
#include "bpcl_int.h"

#include <string.h> // memcpy()

/***************************************************************************
 * 2. DEFINES                                                              *
 ***************************************************************************/

/***************************************************************************
 * 3. DEFINITIONS                                                          *
 ***************************************************************************/

// random state (poor initial values, just for the case rnd_init() wasn't called)
static tU32 seed_x = 521288629;
static tU32 seed_y = 362436069;

/***************************************************************************
 * 4. IMPLEMENTATION OF FUNCTIONS                                          *
 ***************************************************************************/

/***************************************************************************/
/*!
   \par       FUNCTION F-RND/10: rnd_init

   \par       DESCRIPTION

              This function initializes the PRNG

   \par       EQUATION / REFERENCE

              None

   \par       INPUT ARGUMENTS

   \param     const (t_U32)  seed1       - 32-bit seed1 for rand generator

   \param     const (t_U32)  seed2       - 32-bit seed2 for rand generator

   \par       OUTPUT ARGUMENTS

   \param     None

   \par       RETURN VALUE

   \param     (int)  0                   - successful execution

   \param     (int) -1                   - parameter error

   \par       NOTES / WARNINGS / TODO's

   \note      None
*/
/***************************************************************************/
extern int rnd_seed( tU32 seed1,
                     tU32 seed2 ) {
  
  // check seed
  if ( (seed1 == 0) || (seed2 == 0) ) return -1;  

  // init seed
  seed_x = seed1;   
  seed_y = seed2;

  // successful
  return 0;

}



/***************************************************************************/
/*!
   \par       FUNCTION F-RND/20: rnd_word

   \par       DESCRIPTION

              This function returns a pseudo-random number 0 through max

   \par       EQUATION / REFERENCE

              Z = uniformly distributed over [0..max]

   \par       INPUT ARGUMENTS

   \param     const (t_U32)  *max          - upper bound of random value

   \par       OUTPUT ARGUMENTS

   \param     None

   \par       RETURN VALUE

   \param     (t_U32)                      - random value

   \par       NOTES / WARNINGS / TODO's

   \note      all last bits of max value have to be 1!
*/
/***************************************************************************/
extern tU32 rnd_word( tU32 max )
{

#if (C_SKIP_PARAMETER_CHECK == 0)

  // parameter check
  if (!(( max == 0x1UL )          | ( max == 0x3UL )          | ( max == 0x7UL )        | ( max == 0xfUL )        | 
        ( max == 0x1fUL )         | ( max == 0x3fUL )         | ( max == 0x7fUL )       | ( max == 0xffUL )       | 
        ( max == 0x1ffUL )        | ( max == 0x3ffUL )        | ( max == 0x7ffUL )      | ( max == 0xfffUL )      | 
        ( max == 0x1fffUL )       | ( max == 0x3fffUL )       | ( max == 0x7fffUL )     | ( max == 0xffffUL )     | 
        ( max == 0x1ffffUL )      | ( max == 0x3ffffUL )      | ( max == 0x7ffffUL )    | ( max == 0xfffffUL )    | 
        ( max == 0x1fffffUL )     | ( max == 0x3fffffUL )     | ( max == 0x7fffffUL )   | ( max == 0xffffffUL )   | 
        ( max == 0x1ffffffUL )    | ( max == 0x3ffffffUL )    | ( max == 0x7ffffffUL )  | ( max == 0xfffffffUL )  | 
        ( max == 0x1fffffffUL )   | ( max == 0x3fffffffUL )   | ( max == 0x7fffffffUL ) | ( max == 0xffffffffUL ) 
      ))     
  {
    // break
    assert( 1 == 0 );
  }

#endif
  
   // compute 2 new 16-bit random states
   seed_x = 18000 * ( seed_x & 0xFFFF) + ( seed_x >> 16 );
   seed_y = 30903 * ( seed_y & 0xFFFF) + ( seed_y >> 16 );

   // return random bits
   return ( ( (seed_x << 16) + (seed_y & 65535) ) & max );
}



/***************************************************************************/
/*!
   \par       FUNCTION F-RND/30: rnd_byte_array

   \par       DESCRIPTION

              This function returns a random r of r_length bytes

   \par       EQUATION / REFERENCE

              r = uniformly distributed over [0..max]

              K[i] = h( K[i-1] )
              
              r[i] = E ( r[i-1], h(K[i-1]) = IV )

              E = AES-ECB
              h = SHA-1

   \par       INPUT ARGUMENTS

   \param     const (t_U8) *K0_r0      - at least 36-bytes random seed consiting of K0 & r0

   \param     (t_U32) r_length         - length of request random byte array

   \par       OUTPUT ARGUMENTS

   \param     (t_U8) *r                - random byte array

   \par       RETURN VALUE

   \param     +0                       - successful execution

   \param     -1                       - parameter error

   \param     -2                       - internal SHA-1 error

   \param     -3                       - internal AES error

   \par       NOTES / WARNINGS / TODO's

   \note      Always use a new unique seed K0_r0 of at least 36 bytes from a physical TRNG!
*/
/***************************************************************************/
extern int rnd_byte_array( const tU8 *K0_r0,
                                 tU8 *r,
                                 tU32 r_length  )
{

	// declarations
	tU8				K_a[20], K_b[20]; // 160-bit K
	tU8				r_i[16];          // 128-bit r
	tU32				K_a_Hash[5], K_b_Hash[5];
	tS32				i, ret;
	tsSHA1Context		sha_ctx;

	//--- check parameters ----------------------------------------------------

	// check pointers
	if ( (K0_r0 == NULL) || (r == NULL) ) return -1;

	// check if is key set
	if ( r_length < 1 ) return -1;

	//--- build random array --------------------------------------------------

	// init K_a with K0
	for(i = 0; i<20; ++i) {
		K_a[i] = K0_r0[i];
	}
	for(i = 0; i<16; ++i) {
		r_i[i] = K0_r0[20+i];
	}

	do {
		// K_a  = h(K) with h => SHA-1
		ret = BPCL_SHA1_Init( &sha_ctx );
		if ( ret != BPCL_OK ) return -2;

		ret = BPCL_SHA1_Update( &sha_ctx, K_a, 20 );
		if ( ret != BPCL_OK ) return -2;

		ret = BPCL_SHA1_Finish( &sha_ctx, K_a_Hash );
		if ( ret != BPCL_OK ) return -2;

		M_U32_TO_BUF( K_a_Hash[0] , K_a, 0 );
		M_U32_TO_BUF( K_a_Hash[1], K_a, 4 );
		M_U32_TO_BUF( K_a_Hash[2], K_a, 8 );
		M_U32_TO_BUF( K_a_Hash[3], K_a, 12 );
		M_U32_TO_BUF( K_a_Hash[4], K_a, 16 );

		// K_b  = h(K_a) with h => SHA-1
		ret = BPCL_SHA1_Init( &sha_ctx );
		if ( ret != BPCL_OK ) return -2;

		ret = BPCL_SHA1_Update( &sha_ctx, K_a, 20 );
		if ( ret != BPCL_OK ) return -2;

		ret = BPCL_SHA1_Finish( &sha_ctx, K_b_Hash );
		if ( ret != BPCL_OK ) return -2;

		M_U32_TO_BUF( K_b_Hash[0], K_b, 0 );
		M_U32_TO_BUF( K_b_Hash[1], K_b, 4 );
		M_U32_TO_BUF( K_b_Hash[2], K_b, 8 );
		M_U32_TO_BUF( K_b_Hash[3], K_b, 12 );
		M_U32_TO_BUF( K_b_Hash[4], K_b, 16 );

		// r_i = E( r_0, K_b )
		ret = BPCL_AES_Encrypt(BPCL_AES_MODE_ECB, BPCL_AES_OP_ARGKEY, BPCL_AES_KEYSIZE_128,
								 K_b, 0, r_i, r_i, r_i, 16);

		if ( ret != BPCL_OK ) return -3;

		i = 0;
		while ( (i < 16) && (r_length > 0) ) {
			*r = r_i[i];
			i++;
			r++;
			r_length--;
		}

	} while ( r_length > 0 );

	return 0;
}


