/***************************************************************************/
/*! 
    \file        word_math.h

    \brief       Basic word arithmetic definitions

    \author      Marko Wolf (mwolf@crypto.rub.de),
                 Andr Weimerskirch (weika@crypto.rub.de)
    
    \version     3.3
    
    \date        August 20, 2004
    
    \warning     None

    \par Reviews:
    \li          28.01.2004 Reviewd by Andr Weimerskirch
    \li          19.04.2004 Reviewd by Andr Weimerskirch
    \li          25.08.2004 Splint 3.1.1 (no warnings)

*/
/***************************************************************************/

/***************************************************************************/

#if (C_PLATFORM_ARM == 1) && (C_NUM_BITS == 32) && (C_USE_INLINE_ASSEMBLER == 1)

// to save CPSR state when using ARM assembler
static tU32  u32_temp_macro;

#else

// 64-bit temp for ANSI-C word operations
static tU64  u64_temp_macro;

#endif

// counter var for add/subtract loops
static tU32  cnt_temp_macro;


#if (C_BIG_ENDIAN == 1)

// big endian
#define lo64(a) (tU32) ((a) & 0xFFFFFFFF);
#define hi64(a) (tU32) ( ((a) >> 32) & 0xFFFFFFFF );

#else

// little endian
#define lo64(a) (tU32) ((a) & 0xFFFFFFFF);
#define hi64(a) (tU32) ( ((a) >> 32) & 0xFFFFFFFF );

#endif


// word_is_even
#define word_is_even( a ) \
(int)( (*(a) & C_WORD_ONE) == C_WORD_ZERO )

// word_is_even
#define word_is_odd( a ) \
(int)( (*(a) & C_WORD_ONE) )

// word_add_with_carry
#if (C_PLATFORM_ARM == 1) && (C_NUM_BITS == 32) && (C_USE_INLINE_ASSEMBLER == 1)

#define word_add_with_carry( c, carry, a, b )          \
__asm                                                  \
{                                                      \
  MRS  u32_temp_macro, CPSR;                           \
  AND  u32_temp_macro, 0x20000000, *(carry), LSL 29;   \
  MSR  CPSR_f, u32_temp_macro;                         \
  ADCS *(c), *(a), *(b);                               \
  MRS  u32_temp_macro, CPSR;                           \
  AND   *(carry), 1, u32_temp_macro, LSR 29;           \
}

#else

#define word_add_with_carry( c, carry, a, b )          \
{                                                      \
  u64_temp_macro = (tU64) *(a) + *(b) + *(carry);     \
  *(carry) = hi64(u64_temp_macro);                     \
  *(c)     = lo64(u64_temp_macro);                     \
}

#endif


// word_subtract_with_borrow
#if (C_PLATFORM_ARM == 1) && (C_NUM_BITS == 32) && (C_USE_INLINE_ASSEMBLER == 1)

#define word_subtract_with_borrow( c, borrow, a, b )   \
__asm                                                  \
{                                                      \
  MRS  u32_temp_macro, CPSR;                           \
  AND  u32_temp_macro, 0x20000000, *(borrow), LSL 29;  \
  MSR  CPSR_f, u32_temp_macro;                         \
  SBCS *(c), *(a), *(b);                               \
  MRS  u32_temp_macro, CPSR;                           \
  AND   *(borrow), 1, u32_temp_macro, LSR 29;          \
}

#else

#define word_subtract_with_borrow( c, borrow, a, b )   \
{                                                      \
  u64_temp_macro = (tU64) *(a) + ~*(b) + *(borrow);   \
  *(borrow) = hi64(u64_temp_macro);                    \
  *(c)      = lo64(u64_temp_macro);                    \
}

#endif


// word_add_with_carry_loop
#define word_add_with_carry_loop( c, carry_out, a, b, from_word, to_word )                           \
{                                                                                                    \
  *(carry_out) = C_WORD_ZERO;                                                                        \
  for ( cnt_temp_macro = from_word; cnt_temp_macro < to_word; cnt_temp_macro++ )      \
  {                                                                                                  \
    word_add_with_carry( c[cnt_temp_macro], (carry_out), a[cnt_temp_macro], b[cnt_temp_macro] );     \
  }                                                                                                  \
}

// word_subtract_with_borrow_loop
#define word_subtract_with_borrow_loop( c, borrow_out, a, b, from_word, to_word )                    \
{                                                                                                    \
  *(borrow_out) = C_WORD_ONE;                                                                        \
  for ( cnt_temp_macro = from_word; cnt_temp_macro < to_word; cnt_temp_macro++ )      \
  {                                                                                                  \
    word_subtract_with_borrow( c[cnt_temp_macro], (borrow_out), a[cnt_temp_macro], b[cnt_temp_macro] ); \
  }                                                                                                  \
}



//--- word_multiply------------------------------------------------------------

#if (C_PLATFORM_ARM == 1) && (C_NUM_BITS == 32) && (C_USE_INLINE_ASSEMBLER == 1)

// ARM Assembler
#define word_multiply(c_high, c_low, a, b) \
__asm{                                     \
        UMULL (c_low), (c_high), (a), (b)  \
     }                                      


#else

// ANSI C
#define word_multiply(c_high, c_low, a, b) \
{                                          \
  u64_temp_macro = (tU64) (a) * (b);      \
  (c_high) = hi64(u64_temp_macro);         \
  (c_low)  = lo64(u64_temp_macro);         \
}

#endif
