//-----------------------------------------------------------------------------
//
//  ECC_TEST.C
//
//-----------------------------------------------------------------------------
//
//  Test Implementation for ECC Asymmetric Cryptography interface
//
//  Copyright (c) 2006 Blaupunkt GmbH, Hildesheim
//
//  Author: Holger Listle, CM-DI/ESN (based on CCL which was originally
//			created by M. Wolf, A. Weimerskirch, Ruhr University Bochum)
//
//-----------------------------------------------------------------------------

#ifdef _BPCL_TEST

#include "bpcl.h"
#include "bpcl_int.h"

#include <memory.h>	// memcmp()


tU8 ref_priv_key[BPCL_ECC_PRIVATE_KEY_SIZE] = {
	0x9E, 0x3A, 0xDB, 0x43, 0x17, 0xAB, 0x42, 0x01,
	0x62, 0x6F, 0xD8, 0x61, 0xF8, 0x12, 0xF3, 0x88,
	0x4F, 0x5A, 0xA0, 0x74
};

tU8 ref_pub_key[BPCL_ECC_PUBLIC_KEY_SIZE] = {
	0x51, 0xD3, 0xD5, 0xD4, 0xD9, 0x87, 0x66, 0x1F,
	0xC3, 0x3F, 0xE5, 0xEF, 0x28, 0xAF, 0x37, 0xEB,
	0x2E, 0x9D, 0x87, 0x02, 0x80, 0xA7, 0xA2, 0x84,
	0xD2, 0xAA, 0xC0, 0x86, 0x07, 0x25, 0x95, 0xDC,
	0xDB, 0xEE, 0xB5, 0xD4, 0x1C, 0x68, 0xC2, 0x24
};

tU8 ref_sig[BPCL_ECC_SIGNATURE_SIZE] = {
	0x04, 0x6C, 0x4B, 0x4D, 0x19, 0xAE, 0x05, 0x7E,
	0xD5, 0xF9, 0x1D, 0xE9, 0x6E, 0x6D, 0x55, 0xCC,
	0x33, 0x6A, 0x57, 0x5D, 0x00, 0x00, 0x00, 0x00,
	0xC9, 0x67, 0xDC, 0x29, 0xC2, 0x76, 0xA4, 0x31,
	0x4D, 0x00, 0x58, 0x99, 0x77, 0xE0, 0xAD, 0x54,
	0xAE, 0x38, 0x64, 0xBD, 0x00, 0x00, 0x00, 0x00,
};

tU8 ref_dh_key[20] = {
	0x21, 0x83, 0x5A, 0xC5, 0xEB, 0xC0, 0x51, 0x8A,
	0x38, 0x43, 0x92, 0x09, 0xF9, 0x3A, 0xE5, 0xDF,
	0x9D, 0xDE, 0x8F, 0xC8
};

tU8 ref_hash[20] = { 0 };

tU8 ref_seed[64] = {
	41,27,19,73,49,75,64,73,45,51,13,32,87,89,54,15,37,28,67,40,
	49,88,49,33,78,65,51,65,75,59,51,53,84,42,56,18,89,19,73,37,
	42,72,74,58,51,52,12,10,92,50,67,60,45,70,98,68,61,33,75,35,
	12,55,12,45
};

tErrCode
BPCL_ECC_Test(tU8 test_type) {

	tU8			priv_key1[BPCL_ECC_PRIVATE_KEY_SIZE];
	tU8			priv_key2[BPCL_ECC_PRIVATE_KEY_SIZE];
	tU8			pub_key1 [BPCL_ECC_PUBLIC_KEY_SIZE];
	tU8			pub_key2 [BPCL_ECC_PUBLIC_KEY_SIZE];

	tU8			sig[BPCL_ECC_SIGNATURE_SIZE];
	tU8			dh_key1[20];
	tU8			dh_key2[20];

	tU32		sec, msec;
	tErrCode	ret;

	printf("\n----------------------------------------------------------\n");
	printf("             ECC Test\n");
	printf("----------------------------------------------------------\n");

	printf("--- #1 Key pair creation\n\n");

	// Create key pair
	BPCL_TimerStart();
	ret = BPCL_EC_Create_Key_Pair(&(ref_seed[0]), 36, priv_key1, pub_key1);
	BPCL_TimerGet(&sec, &msec);
	if(ret == BPCL_OK) {
		M_MASK_KEY(priv_key1, BPCL_ECC_PRIVATE_KEY_SIZE)
		M_MASK_KEY(pub_key1, BPCL_ECC_PUBLIC_KEY_SIZE)
		if(	memcmp(ref_priv_key, priv_key1, BPCL_ECC_PRIVATE_KEY_SIZE) == 0 &&
			memcmp(ref_pub_key, pub_key1, BPCL_ECC_PUBLIC_KEY_SIZE) == 0) {
			printf("Key-pair creation successful and reference-compliant\n");
		} else {
			printf("Key-pair created successfully - BUT DON'T MATCH REFERENCE!\n");
			BPCL_PrintBuffer("Private Key 1", priv_key1, BPCL_ECC_PRIVATE_KEY_SIZE);
			BPCL_PrintBuffer("Private Key Reference", ref_priv_key, BPCL_ECC_PRIVATE_KEY_SIZE);
			BPCL_PrintBuffer("Public Key 1", pub_key1, BPCL_ECC_PUBLIC_KEY_SIZE);
			BPCL_PrintBuffer("Public Key Reference", ref_pub_key, BPCL_ECC_PUBLIC_KEY_SIZE);
		}
		printf("Execution time: %d.%03d sec\n\n",sec,msec);
	} else {
		printf("Key-pair creation FAILED - Error Code 0x%02x\n\n", ret);
	}


	printf("--- #2 ECDSA Signature Generation\n\n");
	BPCL_TimerStart();
	M_MASK_KEY(priv_key1, BPCL_ECC_PRIVATE_KEY_SIZE)
	ret = BPCL_EC_DSA_Create(ref_hash,priv_key1,&(ref_seed[2]),36,sig);
	BPCL_TimerGet(&sec, &msec);
	if(ret == BPCL_OK) {
		if(	memcmp(ref_sig, sig, BPCL_ECC_SIGNATURE_SIZE) == 0 ) {
			printf("Signature Generation successful and reference-compliant\n");
		} else {
			printf("Signature generated successfully - BUT DON'T MATCH REFERENCE!\n");
			BPCL_PrintBuffer("Signature Data", sig, BPCL_ECC_SIGNATURE_SIZE);
			BPCL_PrintBuffer("Reference Signature", ref_sig, BPCL_ECC_SIGNATURE_SIZE);
		}
		printf("Execution time: %d.%03d sec\n\n",sec,msec);
	} else {
		printf("Signature Generation FAILED - Error Code 0x%02x\n\n", ret);
	}

	printf("--- #3 ECDSA Signature Verification\n\n");
	BPCL_TimerStart();
	M_MASK_KEY(pub_key1, BPCL_ECC_PUBLIC_KEY_SIZE)
	ret = BPCL_EC_DSA_Verify(ref_hash,pub_key1,sig);
	BPCL_TimerGet(&sec, &msec);
	if(ret == BPCL_OK) {
		printf("Signature Verification (unmodifed data) successful\n");
		printf("Execution time: %d.%03d sec\n\n",sec,msec);
	} else {
		printf("Signature Generation FAILED - Error Code 0x%02x\n\n", ret);
	}

	// invert a single bit and try again
	sig[0] ^= 0x01;
	BPCL_TimerStart();
	ret = BPCL_EC_DSA_Verify(ref_hash,pub_key1,sig);
	BPCL_TimerGet(&sec, &msec);
	if(ret == BPCL_ERR_ECC_SIGNATURE_MISMATCH) {
		printf("Signature Verification (modifed data) successful - Mismatch recognized\n");
		printf("Execution time: %d.%03d sec\n\n",sec,msec);
	} else if(ret == BPCL_OK) {
		printf("Signature Verification FAILED - Signature recognized as valid\n\n", ret);
	} else {
		printf("Signature Verification FAILED - Error Code 0x%02x\n\n", ret);
	}


	printf("--- #4 ECDH Shared Secret Generation\n\n");
	// Create second key pair
	ret = BPCL_EC_Create_Key_Pair(&(ref_seed[1]), 36, priv_key2, pub_key2);

	BPCL_TimerStart();
	ret = BPCL_EC_DH(priv_key1, pub_key2, dh_key1, 20);
	BPCL_TimerGet(&sec, &msec);
	if(ret == BPCL_OK) {
		printf("DH key 1 computation successful\n");
		printf("Execution time: %d.%03d sec\n\n",sec,msec);
		ret = BPCL_EC_DH(priv_key2, pub_key1, dh_key2, 20);
		M_MASK_KEY(dh_key1, 20)
		M_MASK_KEY(dh_key2, 20)
		if(ret == BPCL_OK) {
			printf("DH key 2 computation successful\n");
			if(memcmp(dh_key1, dh_key2, 20) == 0) {
				if(memcmp(dh_key1, ref_dh_key, 20) == 0) {
					printf("DH keys match and are reference-compliant - Success.\n");
				} else {
					printf("DH keys 1+2 match - BUT DON'T MATCH REFERENCE!\n");
					BPCL_PrintBuffer("DH Key 1", dh_key1, 20);
					BPCL_PrintBuffer("Reference DH Key", ref_dh_key, 20);
				}
			} else {
				printf("DH keys DON'T match - FAILED.\n");
				BPCL_PrintBuffer("DH Key 1", dh_key1, 20);
				BPCL_PrintBuffer("DH Key 2", dh_key2, 20);
			}
		} else {
			printf("DH key 2 computation FAILED - Error Code 0x%02x\n\n", ret);
		}
	} else {
		printf("DH key 1 computation FAILED - Error Code 0x%02x\n\n", ret);
	}

	return BPCL_OK;
}

#endif // _BPCL_TEST

