//-----------------------------------------------------------------------------
//
//  ECC_IF.C
//
//-----------------------------------------------------------------------------
//
//  Implementation of ECC Asymmetric Cryptography interface
//
//  Copyright (c) 2006 Blaupunkt GmbH, Hildesheim
//
//  Author: Holger Listle, CM-DI/ESN (based on CCL which was originally
//			created by M. Wolf, A. Weimerskirch, Ruhr University Bochum)
//
//  History: 20-01-10 - RBEI/ECF1 - jev1kor - Resolved lint prio2 warnings
//-----------------------------------------------------------------------------

#include "bpcl.h"
#include "bpcl_int.h"

#include "ecc.h"
#include "ecc_secp160r1.h"

//-----------------------------------------------------------------------------
// ECC curve setup
//-----------------------------------------------------------------------------

// choose special algorithms and params
static tsECField field_sec160r1_basis_p = {
	&field_element_multiply_w5_fast,                   /* fptr_fe_multiply */
	&field_element_square_w5_fast,                     /* fptr_fe_square   */
	&field_element_reduce_secp160r1,                   /* fptr_fe_reduce   */
	{ C_POSITIVE, curve32_words_secp160r1_my_p, 6 },   /* precalc_my_p     */ 
	{ C_POSITIVE, curve32_words_secp160r1_prime_p, 5 } /* prime_p          */
};

// SEC p160r1 from ecc_secp160r1.h
static const tsECCurve curve_ccl = {
	{ C_POSITIVE, curve32_words_secp160r1_prime_p, 5 },             /* prime_p             */
	{ C_POSITIVE, curve32_words_secp160r1_coefficient_a, 5 },       /* coefficient_a       */
	{ C_POSITIVE, curve32_words_secp160r1_coefficient_b, 5 },       /* coefficient_b       */
	{   C_PT_AFFINE,                                                /* base point G type   */
		{ C_POSITIVE, curve32_words_secp160r1_base_point_G_x, 5 },    /* base point G->x     */   
		{ C_POSITIVE, curve32_words_secp160r1_base_point_G_y, 5 },    /* base point G->y     */   
		{ C_POSITIVE, curve32_words_secp160r1_base_point_G_z, 5 }, }, /* base point G->z     */   
	{ C_POSITIVE, curve32_words_secp160r1_base_point_order_n, 6 },  /* base_point_order_n  */
	1,                                                              /* cofactor_h          */
	{ C_POSITIVE, curve32_words_secp160r1_my_p, 6 },                /* precalculated my_p  */
	{ C_POSITIVE, curve32_words_secp160r1_my_n, 7 },                /* precalculated my_n  */
	&field_sec160r1_basis_p,                                        /* ecc field parameter */
};

// SEC p160r1 45 precomputed base points with window size of 5
static const tsECPrecomputedP curve_ccl_precomputed_G = {
	curve32_point_secp160r1_precomputed_G_w5,  /* array of precomputed points       */
	45,                                        /* number of precomputed points      */       
	(tU8) 5                                   /* window size of precomputed points */
};

//-----------------------------------------------------------------------------
// Local function prototypes
//-----------------------------------------------------------------------------

static int copy_key_to_field_element(
	tsECFieldElement	*field_element,
	const tU8           *key,
	tU32				key_len,
	tU32				key_first_byte,
	tU32				key_last_byte
);

static int copy_key_to_ecc_point(
	const tU8	*key,
	tU32		key_len,
	tsECPoint	*ecc_point
);


//-----------------------------------------------------------------------------
// BPCL_EC_Create_Key_Pair()
//-----------------------------------------------------------------------------
extern tErrCode
BPCL_EC_Create_Key_Pair(
	tU8					*p_seed,		// RNG seed value
	tU32				seed_len,		// seed value length
	tU8					*p_priv_key,	// Buffer to hold private key
	tU8					*p_pub_key		// Buffer to hold public key
) {
	tErrCode ret;
	
	ret = BPCL_Create_Symmetric_Key(p_seed, seed_len, p_priv_key,
			BPCL_ECC_PRIVATE_KEY_SIZE);

	if(ret == BPCL_OK) {
		ret = BPCL_EC_Compute_Public_Key(p_priv_key, p_pub_key);
	}

	return ret;

} // BPCL_EC_Create_Key_Pair()

//-----------------------------------------------------------------------------
// BPCL_EC_Compute_Public_Key()
//
// History
// 20-10-10 | Jeryn Mathew (RBEI/ECF1) | Prio2 lint resolved. Removed use of goto
//-----------------------------------------------------------------------------
extern tErrCode
BPCL_EC_Compute_Public_Key(
	tU8					*p_priv_key,	// Input private key
	tU8					*p_pub_key		// Buffer to hold public key
) {

	tS32					ret;
	tU32					i;  
	tsECPoint				pt_public_key;
	tsECFieldElement		fe_private_key;
	tErrCode				return_code = BPCL_OK;
	tU8						tmpSK[BPCL_ECC_PRIVATE_KEY_SIZE];


	field_element_allocate( &fe_private_key, curve_ccl.prime_p.length );

	// set fe_private_key ( private_key < group order n )
	fe_private_key.sign = C_POSITIVE;

	for(i = 0; i < BPCL_ECC_PRIVATE_KEY_SIZE; ++i) {
		tmpSK[i] = p_priv_key[i];
	}
	M_MASK_KEY(tmpSK, BPCL_ECC_PRIVATE_KEY_SIZE)
	for ( i = 0; i < fe_private_key.length; i++ ) {
		M_BUF_TO_U32( fe_private_key.word[i], tmpSK, 4 * i );
	}

	//--- build corresponding pt_public_key -----------------------------------

	// alloc pt_public_key
	ret = point_allocate( &pt_public_key, curve_ccl.prime_p.length, C_PT_AFFINE );
	if ( ret != 0 ) {
		return_code = BPCL_ERR_MALLOC;
		// free fe_private_key
      field_element_free( &fe_private_key );

      return return_code;
	}

	i = 0;
	do
	{
		// calculate pt_public_key
		ret = ecc_public_key_generation( &pt_public_key, &fe_private_key, &curve_ccl );
		if ( ret != 0 ) {
			return_code = BPCL_ERR_ECC_KEY_GENERATION;
			
         // free pt_public_key
         ret = point_free ( &pt_public_key );
         if ( ret != 0 ) return_code = BPCL_ERR_UNSPECIFIED;
         // free fe_private_key
         field_element_free( &fe_private_key );

         return return_code;
		}

		// pt_public_key validation
		ret = ecc_public_key_validation( &pt_public_key, &curve_ccl );

	}  while ( (ret != 0) && (i++ < 20) );

	if ( ret != 0 ) {
		return_code = BPCL_ERR_ECC_KEY_GENERATION;
		
      // free pt_public_key
      ret = point_free ( &pt_public_key );
      if ( ret != 0 ) return_code = BPCL_ERR_UNSPECIFIED;
      // free fe_private_key
      field_element_free( &fe_private_key );

      return return_code;
	}

	//--- write keys ----------------------------------------------------------


	// private key field element (padding with 0)
	for ( i = 0; i < fe_private_key.length; i ++ ) {
		M_U32_TO_BUF( fe_private_key.word[i], p_priv_key, 4 * i );
	}
	for ( i = ( 4 * fe_private_key.length ); i < BPCL_ECC_PRIVATE_KEY_SIZE; i ++ ) {
		p_priv_key[i] = (tU8) 0;
	}
	M_MASK_KEY(p_priv_key, BPCL_ECC_PRIVATE_KEY_SIZE)

	// public key point (padding with 0)
	for ( i = 0; i < pt_public_key.x.length; i ++ ) {
		M_U32_TO_BUF( pt_public_key.x.word[i], p_pub_key, 4 * i );
	}
	for ( i = 0; i < pt_public_key.y.length; i ++ ) {
		M_U32_TO_BUF( pt_public_key.y.word[i], p_pub_key, 4 * (i + pt_public_key.x.length) );
	}
	for ( i = ( 4 * (pt_public_key.x.length + pt_public_key.y.length) ); i < BPCL_ECC_PUBLIC_KEY_SIZE; i ++ ) {
		p_pub_key[i] = (tU8) 0;
	}
	M_MASK_KEY(p_pub_key, BPCL_ECC_PUBLIC_KEY_SIZE)

	//--- finish --------------------------------------------------------------

	// free pt_public_key
	ret = point_free ( &pt_public_key );
	if ( ret != 0 ) return_code = BPCL_ERR_UNSPECIFIED;

	// free fe_private_key
	field_element_free( &fe_private_key );

	return return_code;
} // BPCL_EC_Compute_Public_Key()

//-----------------------------------------------------------------------------
// BPCL_EC_DH()
//
// History
// 20-10-10 | Jeryn Mathew (RBEI/ECF1) | Prio2 lint resolved. Removed use of goto
//-----------------------------------------------------------------------------
extern tErrCode
BPCL_EC_DH(
	tU8					*p_my_priv_key,	// My private key
	tU8					*p_oth_pub_key,	// Partner's public key
	tU8					*p_shared_key,	// Buffer to hold resulting shared key
	tU32				shared_key_len	// required key length (max 20 bytes)
) {

	tsECFieldElement	DH_Key;
    tsECFieldElement	SK;
    tsECPoint			PK;
	tU8					tmp_pub_key[BPCL_ECC_PUBLIC_KEY_SIZE];
	tU8					tmp_priv_key[BPCL_ECC_PRIVATE_KEY_SIZE];
    tU8					buffer[20];
	tErrCode			returnCode = BPCL_OK;
	tU32				i;
    

	if(shared_key_len > 20 || (shared_key_len % 4)) {
		return BPCL_ERR_BAD_PARAMETER;
	}
    
	
	//--- compute DH_Key  ---------------------------------------------------

    if (point_allocate( &PK, curve_ccl.prime_p.length , C_PT_AFFINE ))
        return BPCL_ERR_MALLOC;
  
	for(i = 0; i < BPCL_ECC_PUBLIC_KEY_SIZE; ++i) {
		tmp_pub_key[i] = p_oth_pub_key[i];
	}
	M_MASK_KEY(tmp_pub_key, BPCL_ECC_PUBLIC_KEY_SIZE)

	for(i = 0; i < BPCL_ECC_PRIVATE_KEY_SIZE; ++i) {
		tmp_priv_key[i] = p_my_priv_key[i];
	}
	M_MASK_KEY(tmp_priv_key, BPCL_ECC_PRIVATE_KEY_SIZE)

    if (copy_key_to_ecc_point( tmp_pub_key, BPCL_ECC_PUBLIC_KEY_SIZE, &PK )) {
        returnCode = BPCL_ERR_ECC_KEY_ERROR;
        point_free( &PK );
    
        return returnCode;
    }

    field_element_allocate( &SK, curve_ccl.prime_p.length );

    if (copy_key_to_field_element( &SK, tmp_priv_key, BPCL_ECC_PRIVATE_KEY_SIZE,
									0, curve_ccl.prime_p.length * 4 - 1 )) {
        returnCode = BPCL_ERR_ECC_KEY_ERROR;
        field_element_free( &SK );
        point_free( &PK );
    
        return returnCode;
    }

    // build DH_Key
    field_element_allocate( &DH_Key, curve_ccl.prime_p.length );

    if (ecdh_compute_shared_secret( &DH_Key, &SK, &PK, &curve_ccl )) {
        returnCode = BPCL_ERR_UNSPECIFIED;
        
        field_element_free( &DH_Key );
        field_element_free( &SK );
        point_free( &PK );
    
        return returnCode;
    }

    // convert DH_Key into 20 bytes
    M_U32_TO_BUF( DH_Key.word[0], buffer,  0 );
    M_U32_TO_BUF( DH_Key.word[1], buffer,  4 );
    M_U32_TO_BUF( DH_Key.word[2], buffer,  8 );
    M_U32_TO_BUF( DH_Key.word[3], buffer, 12 );
    M_U32_TO_BUF( DH_Key.word[4], buffer, 16 );


	// copy & mask
	for(i=0; ((i < shared_key_len) && (i < 20)); ++i) {
		p_shared_key[i] = buffer[i];
	}
	M_MASK_KEY(p_shared_key, shared_key_len);

 
   field_element_free( &DH_Key );
 
   field_element_free( &SK );
 
   point_free( &PK );
    
   return returnCode;
} // BPCL_EC_DH()

//-----------------------------------------------------------------------------
// BPCL_EC_DSA_Create()
//
// History
// 20-10-10 | Jeryn Mathew (RBEI/ECF1) | Prio2 lint resolved. Removed use of goto
//-----------------------------------------------------------------------------
extern tErrCode
BPCL_EC_DSA_Create(
	tU8					*p_msg_hash,	// Hash of msg (20 bytes)
	tU8					*p_priv_key,	// Signer's private key
	tU8					*p_seed,		// Signature seed value
	tU32				seed_len,		// length of seed value
	tU8					*p_signature	// Buffer to hold signature
) {

	tsECSignature		sig_S;
	tsECFieldElement	fe_C_hash, fe_rnd_k, fe_SKz;
	tU8					*buf;
	tU8					tmp_priv_key[BPCL_ECC_PRIVATE_KEY_SIZE];
	tU32				i;
	int					ret;
	tErrCode			return_code = BPCL_OK;

	(tVoid)seed_len;
	// alloc C_hash (although SHA-1 returns only a 160-bit hash, we need a
	// field_element ~ n = 192-bit for the signature algorithm)
	field_element_allocate( &fe_C_hash, curve_ccl.base_point_order_n.length );
	for( i = 0; i < 5; ++i ) {
		fe_C_hash.word[i] = ((tU32*)((tVoid*)p_msg_hash))[i];
	}

	// build fe_C_hash (higher words of 160-bit hash to zero)
	fe_C_hash.sign = C_POSITIVE;
	for ( i = 5; i < curve_ccl.base_point_order_n.length; i++ ) {
		fe_C_hash.word[i] = C_WORD_ZERO;
	}

	//--- signature generation ------------------------------------------------

	// alloc signature & k & private server key
	field_element_allocate( &sig_S.r,  curve_ccl.base_point_order_n.length );
	field_element_allocate( &sig_S.s,  curve_ccl.base_point_order_n.length );
	field_element_allocate( &fe_rnd_k, curve_ccl.base_point_order_n.length );
	field_element_allocate( &fe_SKz,   curve_ccl.prime_p.length );

	// allocate temporary buffer
	buf = (tU8*) M_MEM_ALLOC( fe_rnd_k.length * 4 );
	if ( buf == NULL ) {
		return_code = BPCL_ERR_MALLOC;
		// free field elements for signature
      field_element_free( &sig_S.r );
      field_element_free( &sig_S.s );
      field_element_free( &fe_rnd_k );
      field_element_free( &fe_SKz );
      M_MEM_FREE( buf );

      // free field element for hash
      field_element_free( &fe_C_hash );

      // return
      return return_code;
	}

	// convert private server key SKz
	for(i = 0; i < BPCL_ECC_PRIVATE_KEY_SIZE; ++i) {
		tmp_priv_key[i] = p_priv_key[i];
	}
	M_MASK_KEY(tmp_priv_key, BPCL_ECC_PRIVATE_KEY_SIZE)
	
	ret = copy_key_to_field_element( &fe_SKz, tmp_priv_key, 20, 0,
									 curve_ccl.prime_p.length * 4 - 1 ); 
	if ( ret != 0 ) {
		return_code = BPCL_ERR_ECC_KEY_ERROR;
		// free field elements for signature
      field_element_free( &sig_S.r );
      field_element_free( &sig_S.s );
      field_element_free( &fe_rnd_k );
      field_element_free( &fe_SKz );
      M_MEM_FREE( buf );

      // free field element for hash
      field_element_free( &fe_C_hash );

      // return
      return return_code;
	}

	// build random_k < n
	fe_rnd_k.sign = C_POSITIVE;
	ret = rnd_byte_array( p_seed, buf, fe_rnd_k.length * 4 );
	if ( ret != 0 ) {
		return_code = BPCL_ERR_RANDOM_GENERATION;
		// free field elements for signature
      field_element_free( &sig_S.r );
      field_element_free( &sig_S.s );
      field_element_free( &fe_rnd_k );
      field_element_free( &fe_SKz );
      M_MEM_FREE( buf );

      // free field element for hash
      field_element_free( &fe_C_hash );

      // return
      return return_code;
	}
	for ( i = 0; i < (fe_rnd_k.length - 1); i++ ) {
		M_BUF_TO_U32( fe_rnd_k.word[i], buf, 4 * i );
	}
	fe_rnd_k.word[fe_rnd_k.length - 1] = C_WORD_ZERO; // k < n

	// S   = Sig( C', SKz ) with Sig => ECDSA
	ret = ecdsa_signature_generation( 
			&sig_S,                   /* result: ECC signature   */
			&fe_C_hash,               /* hash of signing message */
			&fe_SKz,                  /* app. private key        */
			&fe_rnd_k,                /* unpredictable integer   */
			&curve_ccl_precomputed_G, /* precomputed base points */
			&curve_ccl );             /* ECC curve over GP       */

	if( ret != 0 ) {
		return_code = BPCL_ERR_ECC_SIG_GENERATION;
		// free field elements for signature
      field_element_free( &sig_S.r );
      field_element_free( &sig_S.s );
      field_element_free( &fe_rnd_k );
      field_element_free( &fe_SKz );
      M_MEM_FREE( buf );

      // free field element for hash
      field_element_free( &fe_C_hash );

      // return
      return return_code;
	}

	//--- write signature -----------------------------------------------------

	// write signature ( r | s )
	for ( i = 0; i < sig_S.r.length; i++ ) {
		M_U32_TO_BUF( sig_S.r.word[i], p_signature, 4 * i );
	}
	for ( i = 0; i < sig_S.s.length; i++ ) {
		M_U32_TO_BUF( sig_S.s.word[i], p_signature, 4 * (i + sig_S.r.length) );
	}

	//--- finish --------------------------------------------------------------

	// free field elements for signature
	field_element_free( &sig_S.r );
	field_element_free( &sig_S.s );
	field_element_free( &fe_rnd_k );
	field_element_free( &fe_SKz );
	M_MEM_FREE( buf );

	// free field element for hash
	field_element_free( &fe_C_hash );

	// return
	return return_code;
} // BPCL_EC_DSA_Create()

//-----------------------------------------------------------------------------
// BPCL_EC_DSA_Verify()
//
// History
// 20-10-10 | Jeryn Mathew (RBEI/ECF1) | Prio2 lint resolved. Removed use of goto
//-----------------------------------------------------------------------------

extern tErrCode
BPCL_EC_DSA_Verify(
	tU8					*p_msg_hash,	// Hash of msg
	tU8					*p_public_key,	// Signer's public key
	tU8					*p_signature	// Input reference signature
) {

	tS32				ret;
	tErrCode			return_code = BPCL_OK;
	tU32				i;
	tsECSignature		sig_S;
	tsECFieldElement	fe_C_hash;
	tsECPoint			pt_PKz;
	tU8					tmp_pub_key[BPCL_ECC_PUBLIC_KEY_SIZE];

	//--- check parameters  ---------------------------------------------------

	// alloc C_hash (although SHA-1 returns only a 160-bit hash, we need a
	// field_element ~ n = 192-bit for the signature algorithm)
	field_element_allocate( &fe_C_hash, curve_ccl.base_point_order_n.length );
	for( i = 0; i < 5; ++i ) {
		fe_C_hash.word[i] = ((tU32*)((tVoid*)p_msg_hash))[i];
	}

	// build fe_C_hash (higher words of 160-bit hash to zero)
	fe_C_hash.sign = C_POSITIVE;
	for ( i = 5; i < curve_ccl.base_point_order_n.length; i++ ) {
		fe_C_hash.word[i] = C_WORD_ZERO;
	}

	//--- signature verification ----------------------------------------------

	// allocate signature values
	field_element_allocate( &sig_S.r,  curve_ccl.base_point_order_n.length );
	field_element_allocate( &sig_S.s,  curve_ccl.base_point_order_n.length );
	ret = point_allocate( &pt_PKz, curve_ccl.prime_p.length, C_PT_AFFINE );
	if ( ret != 0 ) {
		return_code = BPCL_ERR_MALLOC;
		
      // free signature values
      ret = point_free( &pt_PKz );
      if ( ret != 0 ) {
         return_code = BPCL_ERR_UNSPECIFIED;
      }

      field_element_free( &sig_S.r );
      field_element_free( &sig_S.s );
      field_element_free( &fe_C_hash );
      
      // return
      return return_code;
	}

	for(i = 0; i < BPCL_ECC_PUBLIC_KEY_SIZE; ++i) {
		tmp_pub_key[i] = p_public_key[i];
	}
	M_MASK_KEY(tmp_pub_key, BPCL_ECC_PUBLIC_KEY_SIZE)

		// get public key point
	ret = copy_key_to_ecc_point( tmp_pub_key, BPCL_ECC_PUBLIC_KEY_SIZE, &pt_PKz );
	if ( ret != 0 ) {
		return_code = BPCL_ERR_ECC_KEY_ERROR;
		
      // free signature values
      ret = point_free( &pt_PKz );
      if ( ret != 0 ) {
         return_code = BPCL_ERR_UNSPECIFIED;
      }

      field_element_free( &sig_S.r );
      field_element_free( &sig_S.s );
      field_element_free( &fe_C_hash );

      // return
      return return_code;
	}

	// build signature
	sig_S.r.sign = C_POSITIVE;
	sig_S.s.sign = C_POSITIVE;

	for ( i = 0; i < sig_S.r.length; i++ ) {
		M_BUF_TO_U32( sig_S.r.word[i], p_signature, 4 * i );
	}
	for ( i = 0; i < sig_S.s.length; i++ ) {
		M_BUF_TO_U32( sig_S.s.word[i], p_signature, 4 * (i + sig_S.r.length) );
	}

	// C'' =? Ver( S, PKz )
	ret = ecdsa_signature_verification( &fe_C_hash, &sig_S, &pt_PKz, &curve_ccl_precomputed_G, &curve_ccl );
	if ( ret != 0 ) {
		return_code = BPCL_ERR_ECC_SIGNATURE_MISMATCH;
	}

	//--- finish --------------------------------------------------------------

	// free signature values
	ret = point_free( &pt_PKz );
	if ( ret != 0 ) {
		return_code = BPCL_ERR_UNSPECIFIED;
	}

	field_element_free( &sig_S.r );
	field_element_free( &sig_S.s );
	field_element_free( &fe_C_hash );

	// return
	return return_code;
} // BPCL_EC_DSA_Verify()



//-----------------------------------------------------------------------------
// Local functions
//-----------------------------------------------------------------------------

static int copy_key_to_field_element(
	tsECFieldElement	*field_element,
	const tU8           *key,
	tU32				key_len,
	tU32				key_first_byte,
	tU32				key_last_byte
) {

	int i;

	if(key_len != 20) {
		return -1;
	}

	// assign field element
	field_element->sign   = C_POSITIVE;
	field_element->length = (key_last_byte - key_first_byte + 1) >> 2;
	for ( i = 0; i < (int)(field_element->length); i++ ) {
		M_BUF_TO_U32( field_element->word[i], key, key_first_byte + i * 4 );
	}

	return 0;
}

//-----------------------------------------------------------------------------
// copy_key_to_ecc_point()
//
// History
// 20-10-10 | Jeryn Mathew (RBEI/ECF1) | Prio2 lint resolved. Removed use of goto
//-----------------------------------------------------------------------------
static int copy_key_to_ecc_point(
	const tU8	*key,
	tU32		key_len,
	tsECPoint	*ecc_point
) {

	tU32					key_length_half;
	tS32					ret;
	tsECFieldElement		fe_first_half, fe_second_half;
	int						return_code = 0;


	//--- check parameters ----------------------------------------------------

	// check pointers
	if ( (key == NULL) || (ecc_point == NULL) ) return -1;

	// check if is key set
	if (key_len < 1 ) return -1;

	// check key_length
	if ( key_len % 4 != 0 ) return -1;

	// check ecc_point ( key = X | Y in bytes )
	if ( ecc_point->x.length < (key_len >> 3) ) return -1;
	if ( ecc_point->y.length < (key_len >> 3) ) return -1;
	if ( ecc_point->z.length < (key_len >> 3) ) return -1;

	//--- copy key to point ---------------------------------------------------

	key_length_half = key_len/2;
	field_element_allocate( &fe_first_half,  (tU32) ( key_length_half >> 2)  );
	field_element_allocate( &fe_second_half, (tU32) ( key_length_half >> 2) );

	// link to fe_first_half
	ret = copy_key_to_field_element( &fe_first_half, key, key_length_half,
									0, key_length_half - 1 );
	if ( ret != 0 ) { 
		return_code = -2;
		
      field_element_free( &fe_first_half );
      field_element_free( &fe_second_half );

      // successful
      return return_code;
	}

	// link to fe_second_half
	ret = copy_key_to_field_element( &fe_second_half, key, key_length_half,
									key_length_half, key_len - 1 );
	if ( ret != 0 ) { 
		return_code = -2;
		
      field_element_free( &fe_first_half );
      field_element_free( &fe_second_half );

      // successful
      return return_code;
	}

	// only affine keys allowed now
	ecc_point->type = C_PT_AFFINE;

	// x = fe_first_half
	ret = field_element_assign( &ecc_point->x, &fe_first_half,  fe_first_half.length );
	if ( ret != 0 ) { 
		return_code = -2;
		
      field_element_free( &fe_first_half );
      field_element_free( &fe_second_half );

      // successful
      return return_code;
	}

	// y = fe_second_half
	ret = field_element_assign( &ecc_point->y, &fe_second_half, fe_second_half.length );
	if ( ret != 0 ) { 
		return_code = -2;
		
      field_element_free( &fe_first_half );
      field_element_free( &fe_second_half );

      // successful
      return return_code;
	}

	// z = 0
	ret = field_element_set_zero( &ecc_point->z );
	if ( ret != 0 ) { 
		return_code = -2;
	}

	//--- finish --------------------------------------------------------------

	field_element_free( &fe_first_half );
	field_element_free( &fe_second_half );

	// successful
	return return_code;
}
