/***************************************************************************/
/*! 
    \file        gfp_field_math.c

    \brief       GF(p) multiprecision integer arithmetic implementations

    \author      Marko Wolf (mwolf@crypto.rub.de),
                 Andr Weimerskirch (weika@crypto.rub.de)
				 Adopted to BPCL CM-DI/ESN-Listle
    
    \version     2.8
    
    \date        September 6, 2004

    \warning     None
    
    \par Reviews:
    \li          19.04.2004 Reviewd by Andr Weimerskirch
    \li          06.09.2004 Splint 3.1.1 (-weak no warnings)

    \history
                 05.01.2009 Modified by Divya.H (RBEI/ECM1) to make assert 
                 calls return error

*/
/***************************************************************************/

/***************************************************************************
* 1. INCLUDES                                                              *    
****************************************************************************/

#include "bpcl.h"
#include "bpcl_int.h"
#include "ecc.h"
#include "ecc_wordmath.h"


/***************************************************************************
 * 2. DEFINES                                                              *
 ***************************************************************************/

// uses only static auxiliary vars and avoid most allocations/deallocations
// (faster, but requires a special setup for every curve )
#define C_USE_STATIC_VARS 1 // 1 = ON, 0 = OFF

/***************************************************************************
 * 3. DEFINITIONS                                                          *
 ***************************************************************************/

#if (C_USE_STATIC_VARS == 1)

// malloc substitution for fe_subtract/fe_divide_2 (160-bit)
#define C_FE_TMP_MAX_SUB_WORDS 7

// malloc substitution fe_mod_mul/fe_mod_sqr (160-bit)
#define C_FE_TMP_MAX_MUL_WORDS 12

// malloc substitution fe_mod_inv (160-bit)
#define C_FE_TMP_MAX_INV_WORDS 7

// static auxiliary computing arrays
static tU32 words_temp_sub[C_FE_TMP_MAX_SUB_WORDS];
static tU32 words_temp_mul[C_FE_TMP_MAX_MUL_WORDS];
static tU32 words_temp_inv1[C_FE_TMP_MAX_INV_WORDS];
static tU32 words_temp_inv2[C_FE_TMP_MAX_INV_WORDS];
static tU32 words_temp_inv3[C_FE_TMP_MAX_INV_WORDS];
static tU32 words_temp_inv4[C_FE_TMP_MAX_INV_WORDS];

#endif

/***************************************************************************
 * 4. IMPLEMENTATION OF FUNCTIONS                                          *
 ***************************************************************************/

/***************************************************************************/
/*!
   \par       FUNCTION F20/10: field_element_set_zero

   \par       DESCRIPTION

              This function sets a field element to zero

   \par       EQUATION / REFERENCE

              a = +0

   \par       INPUT ARGUMENTS

   \param     (tsECFieldElement) *a       - pointer to field element, that will be set to zero

   \par       OUTPUT ARGUMENTS

   \param     (tsECFieldElement) *a       - pointer to field element, that was set to zero

   \par       RETURN VALUE

   \param     (int)  0                   - successful execution

   \param     (int) -1                   - parameter error

   \par       NOTES / WARNINGS / TODO's

   \note      None
*/
/***************************************************************************/
extern int field_element_set_zero( tsECFieldElement *a )
{
  // declarations
  tU32 i;
  
#if (C_SKIP_PARAMETER_CHECK == 0)

  // check parameters
  if ( (a == NULL) || (a->length < 1) )
  {
    return -1;
  }

#endif
  
  // for i from 0 to length - 1 do a[i] = 0
  for ( i = 0; i <  a->length; i++ )
  {
    a->word[i] = C_WORD_ZERO;
  }

  // + sign
  a->sign = C_POSITIVE;
  
  // successful
  return 0;
}



/***************************************************************************/
/*!
   \par       FUNCTION F20/20: field_element_set_one

   \par       DESCRIPTION

              This function sets a field element to one

   \par       EQUATION / REFERENCE

              a = +1

   \par       INPUT ARGUMENTS

   \param     (tsECFieldElement) *a       - pointer to field element, that will be set to one

   \par       OUTPUT ARGUMENTS

   \param     (tsECFieldElement) *a       - pointer to field element, that was set to one

   \par       RETURN VALUE

   \param     (int)  0                   - successful execution

   \param     (int) -1                   - parameter error

   \par       NOTES / WARNINGS / TODO's

   \note      None
*/
/***************************************************************************/
extern int field_element_set_one( tsECFieldElement *a )
{
  // declarations
  tU32 i;
  
#if (C_SKIP_PARAMETER_CHECK == 0)

  // check parameters
  if ( (a == NULL) || (a->length < 1) )
  {
    return -1;
  }

#endif

  // a[0] = 1
  a->word[0] = C_WORD_ONE;
  
  // for i from 1 to length - 1 do a[i] = 0
  for ( i = 1; i <  a->length; i++ )
  {
    a->word[i] = C_WORD_ZERO;
  }
  
  // + sign
  a->sign = C_POSITIVE;
  
  // successful
  return 0;
}



/***************************************************************************/
/*!
   \par       FUNCTION F20/30: field_element_is_zero

   \par       DESCRIPTION

              This function checks if a field element is zero

   \par       EQUATION / REFERENCE

              a =? +/-0

   \par       INPUT ARGUMENTS

   \param     const (tsECFieldElement) *a - pointer to field element, that will be checked

   \par       OUTPUT ARGUMENTS

   \param     None

   \par       RETURN VALUE

   \param     (int)  0                   - field element is NOT zero

   \param     (int)  1                   - field element is zero
   
   \param     (int) -1                   - parameter error

   \par       NOTES / WARNINGS / TODO's

   \note      None
*/
/***************************************************************************/
extern int field_element_is_zero( const tsECFieldElement *a )
{
  // declarations
  tU32 i;

#if (C_SKIP_PARAMETER_CHECK == 0)

  // check parameters
  if ( (a == NULL) || (a->length < 1) )
  {
    return -1;
  }

#endif

  // for i from 0 to t do a[i] =? 0
  for ( i = 0; i < a->length; i++ )
  {
    if ( a->word[i] != C_WORD_ZERO )
    { 
      // not zero
      return 0;
    }  
  }
  
  // sign irrelevant
  
  // is zero
  return 1;
}



/***************************************************************************/
/*!
   \par       FUNCTION F20/40: field_element_is_one

   \par       DESCRIPTION

              This function checks if a field element is one

   \par       EQUATION / REFERENCE

              a =? +1

   \par       INPUT ARGUMENTS

   \param     const (tsECFieldElement) *a - pointer to field element, that will be checked

   \par       OUTPUT ARGUMENTS

   \param     None

   \par       RETURN VALUE

   \param     (int)  0                   - field element is NOT one

   \param     (int)  1                   - field element is one

   \param     (int) -1                   - parameter error

   \par       NOTES / WARNINGS / TODO's

   \note      None
*/
/***************************************************************************/
extern int field_element_is_one( const tsECFieldElement *a )
{
  // declarations
  tU32 i;

#if (C_SKIP_PARAMETER_CHECK == 0)

  // check parameters
  if ( (a == NULL) || (a->length < 1) )
  {
    return -1;
  }

#endif

  // a[0] =? 1
  if ( a->word[0] != C_WORD_ONE )
  { 
    // not one
    return 0;
  }  

  // for i from 1 to t-1 do a[i] =? 0
  for ( i = 1; i < a->length; i++ )
  {
    if ( a->word[i] != C_WORD_ZERO )
    { 
      // not one
      return 0;
    }  
  }
  
 // + sign?
 if ( a->sign != C_POSITIVE )
 { 
   // not one
   return 0;
 }  

  // is one
  return 1;
}



/***************************************************************************/
/*!
   \par       FUNCTION F20/50: field_element_is_negative

   \par       DESCRIPTION

              This function checks if a field element is less than zero

   \par       EQUATION / REFERENCE

              a <? -0

   \par       INPUT ARGUMENTS

   \param     const (tsECFieldElement) *a - pointer to field element, that will be checked

   \par       OUTPUT ARGUMENTS

   \param     None

   \par       RETURN VALUE

   \param     (int)  0                   - field element a >= +/-0

   \param     (int) +1                   - field element a < -0

   \param     (int) -1                   - parameter error

   \par       NOTES / WARNINGS / TODO's

   \note      None

   \history
              05.01.2009 Modified by Divya.H (RBEI/ECM1) to make assert 
              calls return error
*/
/***************************************************************************/
extern int field_element_is_negative( const tsECFieldElement *a )
{
  // declarations
  int ret;
  
#if (C_SKIP_PARAMETER_CHECK == 0)

  // check parameters
  if ( (a == NULL) || (a->length < 1) )
  {
    return -1;
  }

#endif

  // check sign
  if ( a->sign != C_POSITIVE )
  { 
    // - sign
    
    // check for a = - 0
    ret = field_element_is_zero( a ); 
    BPCL_CHECK_ERROR((ret >= 0));    
    if ( ret == 1 )
    {
      // a = - 0
      return 0;
    }
    else
    {    
      // a < 0
      return 1;
    }
  }  
  else
  {
    // + sign
    
    // a >= +0
    return 0;
  }
}



/***************************************************************************/
/*!
   \par       FUNCTION F20/60: field_element_absolute_compare

   \par       DESCRIPTION

              This function compares two field elements

   \par       EQUATION / REFERENCE

              |a| =? |b|

   \par       INPUT ARGUMENTS

   \param     const (tsECFieldElement) *a - pointer to field element a, checked with b

   \param     const (tsECFieldElement) *b - pointer to field element b, checked with a

   \par       OUTPUT ARGUMENTS

   \param     None

   \par       RETURN VALUE

   \param     (int) -2                   - parameter error

   \param     (int) -1                   - a < b

   \param     (int)  0                   - a = b

   \param     (int) +1                   - a > b

   \par       NOTES / WARNINGS / TODO's

   \note      Compares only the absolute values!
*/
/***************************************************************************/
extern int field_element_absolute_compare( const tsECFieldElement *a,
                                           const tsECFieldElement *b )
{
  // declarations
  tU32 i;

#if (C_SKIP_PARAMETER_CHECK == 0)
    
  // check parameters
  if ( (a == NULL) || (b == NULL)  || (a->length < 1)  || (b->length < 1) )
  {
    return -2;
  }

#endif

  // check lengths (most likely case first)
  if ( a->length == b->length)
  {
    
    // start mutal compare at their common highest word
    i = a->length;

  }
  else if ( a->length > b->length )
  {
    
    // check if there are upper non zero words in a
    for ( i = ( a->length - 1 ); i > ( b->length - 1 ); i-- )
    {
      if ( a->word[i] != C_WORD_ZERO )
      {
         // a > b
         return 1;
      }
    }

    // start mutal compare at highest b word
    i = b->length;

  }
  else // ( a->length < b->length )
  {
    
    // check if there are upper non zero words in b
    for ( i = ( b->length - 1 ); i > ( a->length - 1 ); i-- )
    {
      if ( b->word[i] != C_WORD_ZERO )
      {
         // a < b 
         return -1;
      }
    }

    // start mutal compare at highest b word
    i = a->length;

  }

   
  // compare the corresponding words until a different word pair is found
  while ( i > 0 ) 
  {
    // next lower pair
    i--;

    if ( a->word[i] > b->word[i] )
    {
      // a > b
      return 1;
    }
    
    if ( a->word[i] < b->word[i] )
    {
      // a < b
      return -1;
    }
        
  }

  // a = b
  return 0;
}



/***************************************************************************/
/*!
   \par       FUNCTION F20/70: field_element_compare

   \par       DESCRIPTION

              This function compares two field elements

   \par       EQUATION / REFERENCE

              a =? b

              decision table

              a     b   |a|>|b|   |a|<|b|  |a|=|b|!=0  |a|=|b|=0

              +     +     +1        -1          0           0
              +     -     +1        +1         +1           0
              -     +     -1        -1         -1           0
              -     -     -1        +1          0           0

   \par       INPUT ARGUMENTS

   \param     const (tsECFieldElement) *a - pointer to field element a, checked with b

   \param     const (tsECFieldElement) *b - pointer to field element b, checked with a

   \par       OUTPUT ARGUMENTS

   \param     None

   \par       RETURN VALUE

   \param     (int) -2                   - parameter error

   \param     (int) -1                   - a < b

   \param     (int)  0                   - a = b

   \param     (int) +1                   - a > b

   \par       NOTES / WARNINGS / TODO's

   \note      None

   \history
              05.01.2009 Modified by Divya.H (RBEI/ECM1) to make assert 
              calls return error
*/
/***************************************************************************/
extern int field_element_compare( const tsECFieldElement *a,
                                  const tsECFieldElement *b ) 
{
  // declarations
  tU32 i;
  tS32 a_is_neg,b_is_neg;
    
#if (C_SKIP_PARAMETER_CHECK == 0)

  // check parameters
  if ( (a == NULL) || (b == NULL)  || (a->length < 1)  || (b->length < 1) )
  {
    return -2;
  }

#endif

  // get signs
  a_is_neg = field_element_is_negative( a );
  BPCL_CHECK_ERROR((a_is_neg >= 0));

  b_is_neg = field_element_is_negative( b );
  BPCL_CHECK_ERROR((b_is_neg >= 0));

  // check signs
  if ( (a_is_neg == 1 ) && (b_is_neg == 0) )
  {
    // a < b
    return -1;
  }
  else if ( (a_is_neg == 0) && (b_is_neg == 1) )
  {
    // a > b
    return 1;
  }


  // check lengths (most likely case first)
  if ( a->length == b->length)
  {
    
    // start mutal compare at their common highest word
    i = a->length;

  }
  else if ( a->length > b->length )
  {
    
    // check if there are upper non zero words in a
    for ( i = ( a->length - 1 ); i > ( b->length - 1 ); i-- )
    {
      if ( a->word[i] != C_WORD_ZERO )
      {
         // a > b
         return 1;
      }
    }

    // start mutal compare at highest b word
    i = b->length;

  }
  else // ( a->length < b->length )
  {
    
    // check if there are upper non zero words in b
    for ( i = ( b->length - 1 ); i > ( a->length - 1 ); i-- )
    {
      if ( b->word[i] != C_WORD_ZERO )
      {
         // a < b 
         return -1;
      }
    }

    // start mutal compare at highest b word
    i = a->length;

  }

 
  
  // compare the corresponding words until a different word pair is found
  while ( i > 0 ) 
  {

    // next lower pair
    i--;

     // a[i] > b[i]
    if ( a->word[i] > b->word[i] )
    {
    
      if ( a_is_neg == 0 )
      {
        // a > 0 -> a > b
        return 1;
      }
      else
      {
        // a < 0 -> a < b
        return -1;
      }

    }
    
    // a[i] < b[i]
    if ( a->word[i] < b->word[i] )
    {

      if ( b_is_neg == 0 )
      {
        // b > 0 -> a < b
        return -1;
      }
      else
      {
        // b < 0 -> a > b
        return 1;
      }

    }
       
  }

  // a = b
  return 0;
}



/***************************************************************************/
/*!
   \par       FUNCTION F20/80: field_element_assign

   \par       DESCRIPTION

              This function assigns a field element to another field element

   \par       EQUATION / REFERENCE

              c = a 

   \par       INPUT ARGUMENTS

   \param     (tsECFieldElement) *a                     - pointer to field element, whose value will be assigned

   \param     const (tU32) word_length_of_a           - number of assigned words of the field element a

   \par       OUTPUT ARGUMENTS

   \param     (tsECFieldElement) *c                     - pointer to field element, that will be reassigned

   \par       RETURN VALUE

   \param     (int)  0                   - successful execution

   \param     (int) -1                   - parameter error

   \par       NOTES / WARNINGS / TODO's

   \note      None
*/
/***************************************************************************/
extern int field_element_assign( tsECFieldElement *c,
                          const  tsECFieldElement *a,
                          const  tU32            word_length_of_a )
{
  // declarations
  tU32 i;

#if (C_SKIP_PARAMETER_CHECK == 0)

  // check parameters
  if ( (a == NULL) || (c == NULL) || (a->length < 1)  || (c->length < 1) || 
       (word_length_of_a < 1) || (word_length_of_a > a->length) || (word_length_of_a > c->length) )
  {
    return -1;
  }

#endif
  
  // for i from 0 to a->length do c[i] = a[i]
  for ( i = 0; i < word_length_of_a; i++ )
  {
    c->word[i] = a->word[i];
  }

  // for i from word_length_of_a to c->length do c[i] = 0
  for ( i = word_length_of_a; i < c->length; i++ )
  {
    c->word[i] = C_WORD_ZERO;
  }
  
  // sign
  c->sign = a->sign;

  // successful
  return 0;
}




/***************************************************************************/
/*!
   \par       FUNCTION F20/90: field_element_shift_n_right

   \par       DESCRIPTION

              This function shifts a field element right by n bits

   \par       EQUATION / REFERENCE

              c = a >> n = a / 2^(n)

   \par       INPUT ARGUMENTS

   \param     (tsECFieldElement) *a       - pointer to field element, that should be shifted right

   \param     const (tU32) n            - number of bits shifted
   
   \par       OUTPUT ARGUMENTS

   \param     (tsECFieldElement) *c       - pointer to field element, that has been shifted right n times

   \par       RETURN VALUE

   \param     (int)  0                   - successful execution

   \param     (int) -1                   - parameter error

   \par       NOTES / WARNINGS / TODO's

   \note      None

   \history
              05.01.2009 Modified by Divya.H (RBEI/ECM1) to make assert 
              calls return error
*/
/***************************************************************************/
extern int field_element_shift_n_right( tsECFieldElement *c,
                                  const tsECFieldElement *a,
                                  const tU32            n )
{
  // declarations
  int ret;
  tU32 i;
  tU32 word_shift,bit_shift;
  tU32 from_pred;

#if (C_SKIP_PARAMETER_CHECK == 0)

  // check parameters
  if ( (a == NULL) || (c == NULL) || (a->length < 1)  || (c->length < 1) ||
       (c->length != a->length) )
  {
    return -1;
  }

#endif

  // translation into words & bits < C_NUM_BITS
  word_shift = (tU32) ( n / C_NUM_BITS );
  bit_shift  = (tU32) ( n % C_NUM_BITS );

  // shift >= length of a
  if ( word_shift >=  a->length )
  {
    // c = 0
    ret = field_element_set_zero( c );
    BPCL_CHECK_ERROR((ret==0));

    // succesful stop
    return 0;
  }

  // bit shift of a[i] with words from word_shift
  for ( i = 0; i < ( a->length - word_shift - 1 ); i++ )
  { 
     // to avoid << 32 bug
     if (bit_shift == 0)
     {
       from_pred = C_WORD_ZERO;
     }  
     else
     {
       from_pred = a->word[i + word_shift + 1] << (C_NUM_BITS - bit_shift);
     }

     c->word[i] = ( a->word[i + word_shift] >> bit_shift ) | ( from_pred );
  }

  // last a[i] has no predecessor to inherits from the last word
  c->word[a->length - word_shift - 1] = a->word[a->length - 1] >> bit_shift;

  // set leading words zero
  for (i = ( a->length - word_shift ); i < a->length; i++ ) 
  {
    c->word[i] = C_WORD_ZERO;
  }

  // sign 
  c->sign = a->sign;

  // successful
  return 0;
}



/***************************************************************************/
/*!
   \par       FUNCTION F20/100: field_element_shift_n_left

   \par       DESCRIPTION

              This function shifts a field element left by n bits

   \par       EQUATION / REFERENCE

              c = a << n = a * 2^(n)

   \par       INPUT ARGUMENTS

   \param     (tsECFieldElement) *a       - pointer to field element, that should be shifted left

   \param     const (tU32) n            - number of bits shifted left
   
   \par       OUTPUT ARGUMENTS

   \param     (tsECFieldElement) *c       - pointer to field element, that has been shifted left n times

   \par       RETURN VALUE

   \param     (int)  0                   - successful execution

   \param     (int) -1                   - parameter error

   \par       NOTES / WARNINGS / TODO's

   \note      None

   \history
              05.01.2009 Modified by Divya.H (RBEI/ECM1) to make assert 
              calls return error
*/
/***************************************************************************/
extern int field_element_shift_n_left( tsECFieldElement *c,
                                 const tsECFieldElement *a,  
                                 const tU32            n )
{
  // declarations
  int ret;
  tU32 i;
  tU32 word_shift,bit_shift;
  tU32 from_succ;

#if (C_SKIP_PARAMETER_CHECK == 0)

  // check parameters
  if ( (a == NULL) || (c == NULL) || (a->length < 1)  || (c->length < 1) ||
       (c->length != a->length) )
  {
    return -1;
  }

#endif

  // translation into words & bits < C_NUM_BITS
  word_shift = (tU32) ( n / C_NUM_BITS );
  bit_shift  = (tU32) ( n % C_NUM_BITS );

  // shift >= length of a
  if ( word_shift >= a->length )
  {
    // a = 0
    ret = field_element_set_zero( c );
    BPCL_CHECK_ERROR((ret==0));

    // succesful stop
    return 0;
  }

  // bit shift of a[i] with words from word_shift
  for (i = ( a->length - 1 ); i > word_shift; i--)
  { 
     // to avoid >> 32 bug
     if (bit_shift == 0)
     {
       from_succ = C_WORD_ZERO;
     }  
     else
     {
       from_succ = a->word[i - word_shift - 1] >> (C_NUM_BITS - bit_shift);
     }

     c->word[i] = ( a->word[i - word_shift] << bit_shift ) | ( from_succ );
  }

  // last a[i] has no successor to inherits from
  c->word[word_shift] = a->word[0] << bit_shift;

  // set last words zero
  for (i = 0; i <= (word_shift - 1); i++) 
  {
    c->word[i] = C_WORD_ZERO;
  }

  // sign 
  c->sign = a->sign;

  // successful
  return 0;
}



/***************************************************************************/
/*!
   \par       FUNCTION F20/110: field_element_add

   \par       DESCRIPTION

              This function adds two field elements

   \par       EQUATION / REFERENCE

              c = a + b
              
              decision table ( S=|a|+|b|, D=|a-b| )
              
              a     b   |a|>|b|   |a|<|b|  |a|=|b|!=0  |a|=|b|=0

              +     +     +S        +S         +S      +/-S,D = 0 
              +     -     +D        -D       +/-D = 0  +/-S,D = 0
              -     +     -D        +D       +/-D = 0  +/-S,D = 0
              -     -     -S        -S         -S      +/-S,D = 0

              ==> No extra handling for |a|=|b| necessary

   \par       INPUT ARGUMENTS

   \param     const (tsECFieldElement) *a - pointer to field element addend a

   \param     const (tsECFieldElement) *b - pointer to field element addend b

   \par       OUTPUT ARGUMENTS

   \param     (tsECFieldElement) *c       - pointer to field element summand c

   \par       RETURN VALUE

   \param     (int)  0                   - successful execution

   \param     (int) -1                   - parameter error

   \param     (int) -2                   - overflow

   \par       NOTES / WARNINGS / TODO's

   \note      None

   \history
              05.01.2009 Modified by Divya.H (RBEI/ECM1) to make assert 
              calls return error
*/
/***************************************************************************/
extern int field_element_add( tsECFieldElement *c,
                        const tsECFieldElement *a,
                        const tsECFieldElement *b )
{
  // declarations 
  int ret;
  tU32 i;
  tU32 carry, borrow, tmp;
  tU32 min_words, max_words, a_is_shorter;

#if (C_SKIP_PARAMETER_CHECK == 0)

  // check parameters
  if ( (a == NULL) || (b == NULL) || (c == NULL) || (a->length < 1) || (b->length < 1) || (c->length < 1)  ||
       (c->length < a->length) || (c->length < b->length) )
  {
    return -1;
  }  

#endif
  
  // get length infos of a & b (most liklely case first)
  if ( a->length == b->length )
  {
     min_words = a->length;
     max_words = min_words;
     a_is_shorter = 2;
  }
  else if ( a->length > b->length )
  {
    min_words = b->length;
    max_words = a->length;
    a_is_shorter = 0;
  }
  else
  {
    min_words = a->length;
    max_words = b->length;
    a_is_shorter = 1;
  }

 
  // sign a =? sign b
  if ( a->sign == b->sign )
  {
     // |c| = |a| + |b|
     
     // -> assembler optimized add loop
     word_add_with_carry_loop( &c->word, &carry, &a->word, &b->word, 0, min_words );
  
     // overlapping a
     if ( a_is_shorter == 0 )
     {
       // zero 
       tmp = C_WORD_ZERO;

       // pad c with a + carry + 0 
       for ( i = min_words; i < a->length; i++ )
       {
          word_add_with_carry( &c->word[i], &carry, &a->word[i], &tmp );
       }
     }
     else if ( a_is_shorter == 1) // overlapping b
     {
	   // zero 
       tmp = C_WORD_ZERO;
       
       // pad c with b + carry + 0 
       for ( i = min_words; i < b->length; i++ )
       {
          word_add_with_carry( &c->word[i], &carry, &b->word[i], &tmp );
       }
     }

     // pad c with zeros
     if ( max_words < c->length )
     {
       // take carry
       c->word[max_words] = carry;
       carry = C_WORD_ZERO;
          
       // pad zeros
       for ( i = ( max_words + 1 ); i < c->length; i++ )
       {
         c->word[i] = C_WORD_ZERO;
       }
     }  

     // overflow?
     if ( carry == C_WORD_ONE )
     {
       return -2;
     }

     // sign c = sign a = sign b
     c->sign = a->sign;
  }
  else
  {
    
    // |a| >=? |b|
    ret = field_element_absolute_compare( a, b );
	 BPCL_CHECK_ERROR((ret >= -1));

    if (  ret >= 0 )
    {
       // |c| = |a| - |b| 
       
       // -> assembler optimized subtract loop
       word_subtract_with_borrow_loop( &c->word, &borrow, &a->word, &b->word, 0, min_words );
       
       // a is longer than b (b has no higher non-zero words than a, because |a| >= |b|)
       if ( a_is_shorter == 0 )
       {
         // zero 
         tmp = C_WORD_ZERO;

         // pad c with a + borrow - 0 
         // no underflow possible, because a has at least one non-zero word to handle borrow, due to |a| >= |b|
         for ( i = min_words; i < a->length; i++ )
         {
           word_subtract_with_borrow( &c->word[i], &borrow, &a->word[i], &tmp );
         }
       }

       // pad zeros (ignore borrow, due to |a| > |b|)
       for ( i = a->length; i < c->length; i++ )
       {
         c->word[i] = C_WORD_ZERO;
       }

       // sign c = sign a
       c->sign = a->sign;

    }
    else
    {
         
       // |c| = |b| - |a|

       // -> assembler optimized subtract loop
       word_subtract_with_borrow_loop( &c->word, &borrow, &b->word, &a->word, 0, min_words );
  
       // b is longer than a (a has no higher non-zero words than b, because |b| >= |a|)
       if ( a_is_shorter == 1 )
       {
         // zero 
         tmp = C_WORD_ZERO;

         // pad c with b + borrow - 0 
         // no underflow possible, because b has at least one non-zero word to handle borrow, due to |b| >= |a|
         for ( i = min_words; i < b->length; i++ )
         {
           word_subtract_with_borrow( &c->word[i], &borrow, &b->word[i], &tmp );
         }
       }

       // pad zeros
       for ( i = b->length; i < c->length; i++ )
       {
         c->word[i] = C_WORD_ZERO;
       }

       // sign c = sign b
       c->sign = b->sign;
    }
      
  }

  // successful
  return 0;
}



/***************************************************************************/
/*!
   \par       FUNCTION F20/120: field_element_subtract

   \par       DESCRIPTION

              This function subtracts two field elements
                   
   \par       EQUATION / REFERENCE

              c = a - b

   \par       INPUT ARGUMENTS

   \param     const (tsECFieldElement) *a - pointer to field element minuend a

   \param     const (tsECFieldElement) *b - pointer to field element subtrahend b

   \par       OUTPUT ARGUMENTS

   \param     (tsECFieldElement) *c       - pointer to field element difference c

   \par       RETURN VALUE

   \param     (int)  0                   - successful execution

   \param     (int) -1                   - parameter error

   \param     (int) -2                   - overflow (error in execution of field_element_add() )
    
   \par       NOTES / WARNINGS / TODO's

   \note      Uses memory allocation!

   \history
              05.01.2009 Modified by Divya.H (RBEI/ECM1) to make assert 
              calls return error
*/
/***************************************************************************/
extern int field_element_subtract( tsECFieldElement *c,
                             const tsECFieldElement *a,
                             const tsECFieldElement *b )
{
  // declarations
  int ret;
  
  // temporary field element
  tsECFieldElement fe_tmp;
  
#if (C_SKIP_PARAMETER_CHECK == 0)

  // check parameters
  if ( (a == NULL) || (b == NULL) || (c == NULL) || (a->length < 1) || (b->length < 1) || (c->length < 1)  ||
       (c->length < a->length) || (c->length < b->length) )
  {
    return -1;
  }  

#endif
  
  // allocate memory for fe_tmp to store b
#if (C_USE_STATIC_VARS == 0)
  field_element_allocate( &fe_tmp, b->length );
#else
  fe_tmp.sign = C_POSITIVE;
  fe_tmp.word = &words_temp_sub[0];
  fe_tmp.length = b->length;
  BPCL_CHECK_ERROR((b->length <= C_FE_TMP_MAX_SUB_WORDS));
#endif

  // fe_tmp = b
  ret = field_element_assign( &fe_tmp, b, b->length );
  #if (C_USE_STATIC_VARS == 0)
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f", &fe_tmp);  
  #else
  BPCL_CHECK_ERROR((ret==0));
  #endif
    
  // sign tmp = opposite of sign b
  if ( b->sign == C_POSITIVE )
  {
    fe_tmp.sign = C_NEGATIVE;
  }
  else
  {
    fe_tmp.sign = C_POSITIVE;
  }

  // c = a - b = a + (-b) = a + tmp
  ret = field_element_add( c , a , &fe_tmp );

  // free fe_tmp
#if (C_USE_STATIC_VARS == 0)
  field_element_free( &fe_tmp );
#endif

  // return ret from add
  return ret;
}



/***************************************************************************/
/*!
   \par       FUNCTION F20/130: field_element_multiply_classical

   \par       DESCRIPTION

              This function multiplies two field elements

   \par       EQUATION / REFERENCE

              c = a  b

   \par       INPUT ARGUMENTS

   \param     const (tsECFieldElement) *a - pointer to field element factor a

   \param     const (tsECFieldElement) *b - pointer to field element factor b

   \par       OUTPUT ARGUMENTS

   \param     (tsECFieldElement) *c       - pointer to field element double product c

   \par       RETURN VALUE

   \param     (int)  0                   - successful execution

   \param     (int) -1                   - parameter error

   \par       NOTES / WARNINGS / TODO's

   \note      Result is not modular reduced and a double field element!

   \history
              05.01.2009 Modified by Divya.H (RBEI/ECM1) to make assert 
              calls return error
*/
/***************************************************************************/
extern int field_element_multiply_classical( tsECFieldElement *c,
                                       const tsECFieldElement *a,
                                       const tsECFieldElement *b )
{
  // declarations
  tU32 u,v;
  tU32 carry,r, carry_count;
  tU32 i,j;

#if (C_SKIP_PARAMETER_CHECK == 0)

  // check parameters
  if ( (a == NULL) || (b == NULL) || (c == NULL) || (a->length < 1) || (b->length < 1) || (c->length < 1)  ||
       (c->length < (a->length + b->length)) )
  {
    return -1;
  }  

#endif

  // 1.) for i = 0 to (a->length + b->length - 1) c[i] = 0
  for ( i = 0; i < c->length; i++ )
  {
    c->word[i] = C_WORD_ZERO;
  }

  // 2.) for i = 0 to b->length - 1
  for ( i = 0; i < b->length; i++ )
  {
    // 2.1) r = 0
    r = C_WORD_ZERO;
    
    // 2.2) for j = 0 to a->length - 1 do
    for ( j = 0; j < a->length; j++ )
    {
      // (uv) = a[j] * b[i]
      word_multiply( u, v, a->word[j], b->word[i] );

      // v = v + c[i+j]
      carry = C_WORD_ZERO;
      word_add_with_carry( &v, &carry, &v, &c->word[i+j] );

      // carry_count = carry
      carry_count = carry;

      // v = v + r
      carry = C_WORD_ZERO;
      word_add_with_carry( &v, &carry, &v, &r );

      // carry_count = carry_count + carry
      carry_count += carry;

      // c[i+j] = v
      c->word[i+j] = v;

      // r = u + carry_count
      carry = C_WORD_ZERO;
      word_add_with_carry( &r, &carry, &u, &carry_count );
	   BPCL_CHECK_ERROR((carry == C_WORD_ZERO));

    } // for j

    // 2.3) c[i + a->length)
    c->word[i + a->length] = r;

  } // for i


  // sign ( a->sign XOR b->sign )
  c->sign = (tU8) ( (a->sign == C_POSITIVE) ^ (b->sign == C_POSITIVE) );

  // successful
  return 0;
}



/***************************************************************************/
/*!
   \par       FUNCTION F20/131: field_element_multiply_w5_fast

   \par       DESCRIPTION

              This function multiplies two field elements

   \par       EQUATION / REFERENCE

              c = a  b

   \par       INPUT ARGUMENTS

   \param     const (tsECFieldElement) *a - pointer to field element factor a

   \param     const (tsECFieldElement) *b - pointer to field element factor b

   \par       OUTPUT ARGUMENTS

   \param     (tsECFieldElement) *c       - pointer to field element double product c

   \par       RETURN VALUE

   \param     (int)  0                   - successful execution

   \param     (int) -1                   - parameter error

   \par       NOTES / WARNINGS / TODO's

   \note      Result is not modular reduced and a double field element!

   \note      Faster due to unlooping!
*/
/***************************************************************************/
extern int field_element_multiply_w5_fast( tsECFieldElement *c,
                                     const tsECFieldElement *a,
                                     const tsECFieldElement *b )
{
  // declarations
  tU32 r0,r1,r2,u,v;
  tU32 carry;
    
  
#if (C_SKIP_PARAMETER_CHECK == 0)

  // check parameters
  if ( (a == NULL) || (b == NULL) || (c == NULL) ||
       (c->length != 10) || (a->length != 5) || (b->length != 5) )
  {
    return -1;
  }  

#endif

  // 1.) r0 = 0, r1 = 0, r2 = 0 (w32_null = 0)
  r0 = C_WORD_ZERO;
  r1 = C_WORD_ZERO;
  r2 = C_WORD_ZERO;

  // 2.) for k = 0 to 2*(t-1) do
  // 2.1) for each element of { (i,j) | i + j = k , 0 <= i,j < t)
           
  // (uv) = a[0] * b[0]
  word_multiply( u, v, a->word[0], b->word[0] );
          
  // r0 = add_with_carry(r0,v) -> carry = 0
  carry = C_WORD_ZERO;
  word_add_with_carry( &r0, &carry, &r0, &v );
         
  // r1 = add_with_carry(r1,u)
  word_add_with_carry( &r1, &carry, &r1, &u );

  // r2 = add_with_carry(r2,0)
  r2 += carry;
     

  // 2.2) c->word[k] = r0, r0 = r1, r1 = r2, r2 = 0
  c->word[0] = r0;
  r0   = r1;
  r1   = r2;
  r2   = C_WORD_ZERO;

  
  // (uv) = a[0] * b[1]
  word_multiply( u, v, a->word[0], b->word[1] );
          
  // r0 = add_with_carry(r0,v) -> carry = 0
  carry = C_WORD_ZERO;
  word_add_with_carry( &r0, &carry, &r0, &v );
         
  // r1 = add_with_carry(r1,u)
  word_add_with_carry( &r1, &carry, &r1, &u );

  // r2 = add_with_carry(r2,0)
  r2 += carry;
     
  
  // (uv) = a[1] * b[0]
  word_multiply( u, v, a->word[1], b->word[0] );
          
  // r0 = add_with_carry(r0,v) -> carry = 0
  carry = C_WORD_ZERO;
  word_add_with_carry( &r0, &carry, &r0, &v );
         
  // r1 = add_with_carry(r1,u)
  word_add_with_carry( &r1, &carry, &r1, &u );

  // r2 = add_with_carry(r2,0)
  r2 += carry;
     

  // 2.2) c->word[k] = r0, r0 = r1, r1 = r2, r2 = 0
  c->word[1] = r0;
  r0   = r1;
  r1   = r2;
  r2   = C_WORD_ZERO;


  // (uv) = a[0] * b[2]
  word_multiply( u, v, a->word[0], b->word[2] );
          
  // r0 = add_with_carry(r0,v) -> carry = 0
  carry = C_WORD_ZERO;
  word_add_with_carry( &r0, &carry, &r0, &v );
         
  // r1 = add_with_carry(r1,u)
  word_add_with_carry( &r1, &carry, &r1, &u );

  // r2 = add_with_carry(r2,0)
  r2 += carry;
     

  // (uv) = a[1] * b[1]
  word_multiply( u, v, a->word[1], b->word[1] );
          
  // r0 = add_with_carry(r0,v) -> carry = 0
  carry = C_WORD_ZERO;
  word_add_with_carry( &r0, &carry, &r0, &v );
         
  // r1 = add_with_carry(r1,u)
  word_add_with_carry( &r1, &carry, &r1, &u );

  // r2 = add_with_carry(r2,0)
  r2 += carry;
     
  
  // (uv) = a[2] * b[0]
  word_multiply( u, v, a->word[2], b->word[0] );
          
  // r0 = add_with_carry(r0,v) -> carry = 0
  carry = C_WORD_ZERO;
  word_add_with_carry( &r0, &carry, &r0, &v );
         
  // r1 = add_with_carry(r1,u)
  word_add_with_carry( &r1, &carry, &r1, &u );

  // r2 = add_with_carry(r2,0)
  r2 += carry;
     

  // 2.2) c->word[k] = r0, r0 = r1, r1 = r2, r2 = 0
  c->word[2] = r0;
  r0   = r1;
  r1   = r2;
  r2   = C_WORD_ZERO;


  // (uv) = a[0] * b[3]
  word_multiply( u, v, a->word[0], b->word[3] );
          
  // r0 = add_with_carry(r0,v) -> carry = 0
  carry = C_WORD_ZERO;
  word_add_with_carry( &r0, &carry, &r0, &v );
         
  // r1 = add_with_carry(r1,u)
  word_add_with_carry( &r1, &carry, &r1, &u );

  // r2 = add_with_carry(r2,0)
  r2 += carry;
     

  // (uv) = a[1] * b[2]
  word_multiply( u, v, a->word[1], b->word[2] );
          
  // r0 = add_with_carry(r0,v) -> carry = 0
  carry = C_WORD_ZERO;
  word_add_with_carry( &r0, &carry, &r0, &v );
         
  // r1 = add_with_carry(r1,u)
  word_add_with_carry( &r1, &carry, &r1, &u );

  // r2 = add_with_carry(r2,0)
  r2 += carry;
     
  
  // (uv) = a[2] * b[1]
  word_multiply( u, v, a->word[2], b->word[1] );
          
  // r0 = add_with_carry(r0,v) -> carry = 0
  carry = C_WORD_ZERO;
  word_add_with_carry( &r0, &carry, &r0, &v );
      
  // r1 = add_with_carry(r1,u)
  word_add_with_carry( &r1, &carry, &r1, &u );

  // r2 = add_with_carry(r2,0)
  r2 += carry;
     
  
  // (uv) = a[3] * b[0]
  word_multiply( u, v, a->word[3], b->word[0] );
        
  // r0 = add_with_carry(r0,v) -> carry = 0
  carry = C_WORD_ZERO;
  word_add_with_carry( &r0, &carry, &r0, &v );
      
  // r1 = add_with_carry(r1,u)
  word_add_with_carry( &r1, &carry, &r1, &u );

  // r2 = add_with_carry(r2,0)
  r2 += carry;
     
  // 2.2) c->word[k] = r0, r0 = r1, r1 = r2, r2 = 0
  c->word[3] = r0;
  r0   = r1;
  r1   = r2;
  r2   = C_WORD_ZERO;

  
  // (uv) = a[0] * b[4]
  word_multiply( u, v, a->word[0], b->word[4] );
         
  // r0 = add_with_carry(r0,v) -> carry = 0
  carry = C_WORD_ZERO;
  word_add_with_carry( &r0, &carry, &r0, &v );
      
  // r1 = add_with_carry(r1,u)
  word_add_with_carry( &r1, &carry, &r1, &u );

  // r2 = add_with_carry(r2,0)
  r2 += carry;
     
  
  // (uv) = a[1] * b[3]
  word_multiply( u, v, a->word[1], b->word[3] );
         
  // r0 = add_with_carry(r0,v) -> carry = 0
  carry = C_WORD_ZERO;
  word_add_with_carry( &r0, &carry, &r0, &v );
       
  // r1 = add_with_carry(r1,u)
  word_add_with_carry( &r1, &carry, &r1, &u );

  // r2 = add_with_carry(r2,0)
  r2 += carry;
     
  
  // (uv) = a[2] * b[2]
  word_multiply( u, v, a->word[2], b->word[2] );
        
  // r0 = add_with_carry(r0,v) -> carry = 0
  carry = C_WORD_ZERO;
  word_add_with_carry( &r0, &carry, &r0, &v );
       
  // r1 = add_with_carry(r1,u)
  word_add_with_carry( &r1, &carry, &r1, &u );

  // r2 = add_with_carry(r2,0)
  r2 += carry;
     
  
  // (uv) = a[3] * b[1]
  word_multiply( u, v, a->word[3], b->word[1] );
          
  // r0 = add_with_carry(r0,v) -> carry = 0
  carry = C_WORD_ZERO;
  word_add_with_carry( &r0, &carry, &r0, &v );
       
  // r1 = add_with_carry(r1,u)
  word_add_with_carry( &r1, &carry, &r1, &u );

  // r2 = add_with_carry(r2,0)
  r2 += carry;
     
  
  // (uv) = a[4] * b[0]
  word_multiply( u, v, a->word[4], b->word[0] );

  // r0 = add_with_carry(r0,v) -> carry = 0
  carry = C_WORD_ZERO;
  word_add_with_carry( &r0, &carry, &r0, &v );
         
  // r1 = add_with_carry(r1,u)
  word_add_with_carry( &r1, &carry, &r1, &u );

  // r2 = add_with_carry(r2,0)
  r2 += carry;
     

  // 2.2) c->word[k] = r0, r0 = r1, r1 = r2, r2 = 0
  c->word[4] = r0;
  r0   = r1;
  r1   = r2;
  r2   = C_WORD_ZERO;

  // (uv) = a[1] * b[4]
  word_multiply( u, v, a->word[1], b->word[4] );

  // r0 = add_with_carry(r0,v) -> carry = 0
  carry = C_WORD_ZERO;
  word_add_with_carry( &r0, &carry, &r0, &v ); 
         
  // r1 = add_with_carry(r1,u)
  word_add_with_carry( &r1, &carry, &r1, &u );
  
  // r2 = add_with_carry(r2,0)
  r2 += carry;
     

  // (uv) = a[2] * b[3]
  word_multiply( u, v, a->word[2], b->word[3] );
           
  // r0 = add_with_carry(r0,v) -> carry = 0
  carry = C_WORD_ZERO;
  word_add_with_carry( &r0, &carry, &r0, &v );
           
  // r1 = add_with_carry(r1,u)
  word_add_with_carry( &r1, &carry, &r1, &u );
  
  // r2 = add_with_carry(r2,0)
  r2 += carry;
     

  // (uv) = a[3] * b[2]
  word_multiply( u, v, a->word[3], b->word[2] );
            
  // r0 = add_with_carry(r0,v) -> carry = 0
  carry = C_WORD_ZERO;
  word_add_with_carry( &r0, &carry, &r0, &v );  
         
  // r1 = add_with_carry(r1,u)
  word_add_with_carry( &r1, &carry, &r1, &u );
  
  // r2 = add_with_carry(r2,0)
  r2 += carry;
     
  
  // (uv) = a[4] * b[1]
  word_multiply( u, v, a->word[4], b->word[1] );
          
  // r0 = add_with_carry(r0,v) -> carry = 0
  carry = C_WORD_ZERO;
  word_add_with_carry( &r0, &carry, &r0, &v ); 
         
  // r1 = add_with_carry(r1,u)
  word_add_with_carry( &r1, &carry, &r1, &u );

  // r2 = add_with_carry(r2,0)
  r2 += carry;
     
  
  // 2.2) c->word[k] = r0, r0 = r1, r1 = r2, r2 = 0
  c->word[5] = r0;
  r0   = r1;
  r1   = r2;
  r2   = C_WORD_ZERO;

  // (uv) = a[2] * b[4]
  word_multiply( u, v, a->word[2], b->word[4] );
          
  // r0 = add_with_carry(r0,v) -> carry = 0
  carry = C_WORD_ZERO;
  word_add_with_carry( &r0, &carry, &r0, &v );
       
  // r1 = add_with_carry(r1,u)
  word_add_with_carry( &r1, &carry, &r1, &u );

  // r2 = add_with_carry(r2,0)
  r2 += carry;
     

  // (uv) = a[3] * b[3]
  word_multiply( u, v, a->word[3], b->word[3] );
         
  // r0 = add_with_carry(r0,v) -> carry = 0
  carry = C_WORD_ZERO;
  word_add_with_carry( &r0, &carry, &r0, &v );
         
  // r1 = add_with_carry(r1,u)
  word_add_with_carry( &r1, &carry, &r1, &u );

  // r2 = add_with_carry(r2,0)
  r2 += carry;
     

  // (uv) = a[4] * b[2]
  word_multiply( u, v, a->word[4], b->word[2] );
      
  // r0 = add_with_carry(r0,v) -> carry = 0
  carry = C_WORD_ZERO;
  word_add_with_carry( &r0, &carry, &r0, &v );
         
  // r1 = add_with_carry(r1,u)
  word_add_with_carry( &r1, &carry, &r1, &u );

  // r2 = add_with_carry(r2,0)
  r2 += carry;
     

  // 2.2) c->word[k] = r0, r0 = r1, r1 = r2, r2 = 0
  c->word[6] = r0;
  r0   = r1;
  r1   = r2;
  r2   = C_WORD_ZERO;

  // (uv) = a[3] * b[4]
  word_multiply( u, v, a->word[3], b->word[4] );
        
  // r0 = add_with_carry(r0,v) -> carry = 0
  carry = C_WORD_ZERO;
  word_add_with_carry( &r0, &carry, &r0, &v );
       
  // r1 = add_with_carry(r1,u)
  word_add_with_carry( &r1, &carry, &r1, &u );

  // r2 = add_with_carry(r2,0)
  r2 += carry;
     

  // (uv) = a[4] * b[3]
  word_multiply( u, v, a->word[4], b->word[3] );
        
  // r0 = add_with_carry(r0,v) -> carry = 0
  carry = C_WORD_ZERO;
  word_add_with_carry( &r0, &carry, &r0, &v );
     
  // r1 = add_with_carry(r1,u)
  word_add_with_carry( &r1, &carry, &r1, &u );

  // r2 = add_with_carry(r2,0)
  r2 += carry;
     

  // 2.2) c->word[k] = r0, r0 = r1, r1 = r2, r2 = 0
  c->word[7] = r0;
  r0   = r1;
  r1   = r2;
  r2   = C_WORD_ZERO;

  // (uv) = a[4] * b[4]
  word_multiply( u, v, a->word[4], b->word[4] );
       
  // r0 = add_with_carry(r0,v) -> carry = 0
  carry = C_WORD_ZERO;
  word_add_with_carry( &r0, &carry, &r0, &v );
       
  // r1 = add_with_carry(r1,u)
  word_add_with_carry( &r1, &carry, &r1, &u );

  // r2 = add_with_carry(r2,0)
  r2 += carry;

  // 2.2) c->word[k] = r0, r0 = r1, r1 = r2, r2 = 0
  c->word[8] = r0;
  r0   = r1;
  r1   = r2;
  r2   = C_WORD_ZERO;

  // 3.) c->word[2t-1] = r0
  c->word[9] = r0;


  // sign ( a->sign XOR b->sign )
  c->sign = (tU8) ( (a->sign == C_POSITIVE) ^ (b->sign == C_POSITIVE) );

  // successful
  return 0;
}



/***************************************************************************/
/*!
   \par       FUNCTION F20/140: field_element_square_classical

   \par       DESCRIPTION

              This function multiplies two field elements

   \par       EQUATION / REFERENCE

              c = |a|

   \par       INPUT ARGUMENTS

   \param     const (tsECFieldElement) *a - pointer to field element square factor a

   \par       OUTPUT ARGUMENTS

   \param     (tsECFieldElement) *c       - pointer to field element square product c

   \par       RETURN VALUE

   \param     (int)  0                   - successful execution

   \param     (int) -1                   - parameter error

   \par       NOTES / WARNINGS / TODO's

   \note      Result is NOT modular reduced and a double field element!
*/
/***************************************************************************/
extern int field_element_square_classical( tsECFieldElement *c,
                                     const tsECFieldElement *a )
{
  // declarations
  tU32 r0,r1,r2,u,v;
  tU32 carry;
  tU32 i,j,k;
  
#if (C_SKIP_PARAMETER_CHECK == 0)

  // check parameters
  if ( (a == NULL) || (c == NULL) || (a->length < 1) || (c->length < 1) || 
       (c->length < 2 * a->length) )
  {
    return -1;
  }  

#endif

  // 1.) r0 = 0, r1 = 0, r2 = 0 (w32_null = 0, carry = 0)
  r0    = C_WORD_ZERO;
  r1    = C_WORD_ZERO;
  r2    = C_WORD_ZERO;
  carry = C_WORD_ZERO;
  
  // 2.) for k = 0 to 2*(t-1) do
  for ( k = 0; k <= 2 * (a->length - 1); k++ )
  {

    // 2.1) for each element of { (i,j) | i + j = k , 0 <= i,j < t)
    for ( i = 0; i < a->length; i++ )
    {
      for ( j = 0; j < a->length; j++ )
      {
        if ( ((i+j) == k) && (i <= j) )
        {
        
          // (uv) = a[i] * a[j]
          word_multiply( u, v, a->word[i], a->word[j] );

          if ( i < j )
          {
            // (uv) << 1  <=> (uv) = 2 * (uv) <=> 2*u && 2*v with carry
            carry = C_WORD_ZERO;
            word_add_with_carry( &v, &carry, &v, &v );
            word_add_with_carry( &u, &carry, &u, &u );

            // r2 = add_with_carry(r2,0) <=> r2 gets carry from 2*u
            r2 += carry;
          }
          
          // r0 = add(r0,v) -> carry = 0
          carry = C_WORD_ZERO;
          word_add_with_carry( &r0, &carry, &r0, &v );
         
          // r1 = add_with_carry(r1,u)
          word_add_with_carry( &r1, &carry, &r1, &u );

          // r2 = add_with_carry(r2,0)
          r2 += carry;

        } // if (i+j <= k)
      } // for j
    } // for i

    // 2.2) c[k] = r0, r0 = r1, r1 = r2, r2 = 0
    c->word[k] = r0;
    r0         = r1;
    r1         = r2;
    r2         = C_WORD_ZERO;
    
  } // for k
  
  // 3.) c[2t-1] = r0
  c->word[2 * a->length - 1] = r0;

  
  // set upper c words zero
  for ( i = (2 * a->length); i < c->length; i++ )
  {
    c->word[i] = C_WORD_ZERO;
  }

  // sign (always +)
  c->sign = C_POSITIVE;

  // successful
  return 0;
}



/***************************************************************************/
/*!
   \par       FUNCTION F20/141: field_element_square_w5_fast

   \par       DESCRIPTION

              This function multiplies two field elements

   \par       EQUATION / REFERENCE

              c = |a|

   \par       INPUT ARGUMENTS

   \param     const (tsECFieldElement) *a - pointer to field element square factor a

   \par       OUTPUT ARGUMENTS

   \param     (tsECFieldElement) *c       - pointer to field element square product c

   \par       RETURN VALUE

   \param     (int)  0                   - successful execution

   \param     (int) -1                   - parameter error

   \par       NOTES / WARNINGS / TODO's

   \note      Result is NOT modular reduced and a double field element!

   \note      Faster due to unlooping!
*/
/***************************************************************************/
extern int field_element_square_w5_fast( tsECFieldElement *c,
                                   const tsECFieldElement *a )
{
  // declarations
  tU32 r0,r1,r2,u,v;
  tU32 carry;
    
  
#if (C_SKIP_PARAMETER_CHECK == 0)

  // check parameters
  if ( (a == NULL) || (c == NULL) || (c->length != 10) || (a->length != 5) )
  {
    return -1;
  }  

#endif

  // 1.) r0 = 0, r1 = 0, r2 = 0 (w32_null = 0, carry = 0)
  r0    = C_WORD_ZERO;
  r1    = C_WORD_ZERO;
  r2    = C_WORD_ZERO;
  carry = C_WORD_ZERO;
  
  // 2.) for k = 0 to 2*(t-1) do
  // 2.1) for each element of { (i,j) | i + j = k , 0 <= i,j < t)
       
  // (uv) = a[0] * a[0]
  word_multiply( u, v, a->word[0], a->word[0] );
  
  // r0 = add(r0,v) -> carry = 0
  carry = C_WORD_ZERO;
  word_add_with_carry( &r0, &carry, &r0, &v );
       
  // r1 = add_with_carry(r1,u)
  word_add_with_carry( &r1, &carry, &r1, &u );

  // r2 = add_with_carry(r2,0)
  r2 += carry;
      
  // 2.2) c->word[k] = r0, r0 = r1, r1 = r2, r2 = 0
  c->word[0] = r0;
  r0   = r1;
  r1   = r2;
  r2   = C_WORD_ZERO;


  // (uv) = a[0] * a[1]
  word_multiply( u, v, a->word[0], a->word[1] );

  // (uv) << 1  <=> (uv) = 2 * (uv) <=> 2*u && 2*v with carry
  carry = C_WORD_ZERO;
  word_add_with_carry( &v, &carry, &v, &v );
  
  word_add_with_carry( &u, &carry, &u, &u );

  // r2 = add_with_carry(r2,0) <=> r2 gets carry from 2*u
  r2 += carry;
 
  // r0 = add(r0,v) -> carry = 0
  carry = C_WORD_ZERO;
  word_add_with_carry( &r0, &carry, &r0, &v );
  
         
  // r1 = add_with_carry(r1,u)
  word_add_with_carry( &r1, &carry, &r1, &u );
  

  // r2 = add_with_carry(r2,0)
  r2 += carry;
      
  // 2.2) c->word[k] = r0, r0 = r1, r1 = r2, r2 = 0
  c->word[1] = r0;
  r0   = r1;
  r1   = r2;
  r2   = C_WORD_ZERO;


  // (uv) = a[0] * a[2]
  word_multiply( u, v, a->word[0], a->word[2] );
  

  // (uv) << 1  <=> (uv) = 2 * (uv) <=> 2*u && 2*v with carry
  carry = C_WORD_ZERO;
  word_add_with_carry( &v, &carry, &v, &v );
  
  word_add_with_carry( &u, &carry, &u, &u );
  

  // r2 = add_with_carry(r2,0) <=> r2 gets carry from 2*u
  r2 += carry;

  // r0 = add(r0,v) -> carry = 0
  carry = C_WORD_ZERO;
  word_add_with_carry( &r0, &carry, &r0, &v );
  
         
  // r1 = add_with_carry(r1,u)
  word_add_with_carry( &r1, &carry, &r1, &u );
  

  // r2 = add_with_carry(r2,0)
  r2 += carry;
      

  // (uv) = a[1] * a[1]
  word_multiply( u, v, a->word[1], a->word[1] );
  

  // r0 = add(r0,v) -> carry = 0
  carry = C_WORD_ZERO;
  word_add_with_carry( &r0, &carry, &r0, &v );
  
         
  // r1 = add_with_carry(r1,u)
  word_add_with_carry( &r1, &carry, &r1, &u );
  

  // r2 = add_with_carry(r2,0)
  r2 += carry;
      

  // 2.2) c->word[k] = r0, r0 = r1, r1 = r2, r2 = 0
  c->word[2] = r0;
  r0   = r1;
  r1   = r2;
  r2   = C_WORD_ZERO;


  // (uv) = a[0] * a[3]
  word_multiply( u, v, a->word[0], a->word[3] );
  

  // (uv) << 1  <=> (uv) = 2 * (uv) <=> 2*u && 2*v with carry
  carry = C_WORD_ZERO;
  word_add_with_carry( &v, &carry, &v, &v );
  
  word_add_with_carry( &u, &carry, &u, &u );
  

  // r2 = add_with_carry(r2,0) <=> r2 gets carry from 2*u
  r2 += carry;

  // r0 = add(r0,v) -> carry = 0
  carry = C_WORD_ZERO;
  word_add_with_carry( &r0, &carry, &r0, &v );
  
         
  // r1 = add_with_carry(r1,u)
  word_add_with_carry( &r1, &carry, &r1, &u );
  

  // r2 = add_with_carry(r2,0)
  r2 += carry;
     

  // (uv) = a[1] * a[2]
  word_multiply( u, v, a->word[1], a->word[2] );
  

  // (uv) << 1  <=> (uv) = 2 * (uv) <=> 2*u && 2*v with carry
  carry = C_WORD_ZERO;
  word_add_with_carry( &v, &carry, &v, &v );
  
  word_add_with_carry( &u, &carry, &u, &u );
  

  // r2 = add_with_carry(r2,0) <=> r2 gets carry from 2*u
  r2 += carry;

  // r0 = add(r0,v) -> carry = 0
  carry = C_WORD_ZERO;
  word_add_with_carry( &r0, &carry, &r0, &v );
  
         
  // r1 = add_with_carry(r1,u)
  word_add_with_carry( &r1, &carry, &r1, &u );
  

  // r2 = add_with_carry(r2,0)
  r2 += carry;
      
  // 2.2) c->word[k] = r0, r0 = r1, r1 = r2, r2 = 0
  c->word[3] = r0;
  r0   = r1;
  r1   = r2;
  r2   = C_WORD_ZERO;


  // (uv) = a[0] * a[4]
  word_multiply( u, v, a->word[0], a->word[4] );
  

  // (uv) << 1  <=> (uv) = 2 * (uv) <=> 2*u && 2*v with carry
  carry = C_WORD_ZERO;
  word_add_with_carry( &v, &carry, &v, &v );
  
  word_add_with_carry( &u, &carry, &u, &u );
  

  // r2 = add_with_carry(r2,0) <=> r2 gets carry from 2*u
  r2 += carry;

  // r0 = add(r0,v) -> carry = 0
  carry = C_WORD_ZERO;
  word_add_with_carry( &r0, &carry, &r0, &v );
  
         
  // r1 = add_with_carry(r1,u)
  word_add_with_carry( &r1, &carry, &r1, &u );
  

  // r2 = add_with_carry(r2,0)
  r2 += carry;
      

  // (uv) = a[1] * a[3]
  word_multiply( u, v, a->word[1], a->word[3] );
  

  // (uv) << 1  <=> (uv) = 2 * (uv) <=> 2*u && 2*v with carry
  carry = C_WORD_ZERO;
  word_add_with_carry( &v, &carry, &v, &v );
  
  word_add_with_carry( &u, &carry, &u, &u );
  

  // r2 = add_with_carry(r2,0) <=> r2 gets carry from 2*u
  r2 += carry;

  // r0 = add(r0,v) -> carry = 0
  carry = C_WORD_ZERO;
  word_add_with_carry( &r0, &carry, &r0, &v );
  
         
  // r1 = add_with_carry(r1,u)
  word_add_with_carry( &r1, &carry, &r1, &u );
  

  // r2 = add_with_carry(r2,0)
  r2 += carry;
      

  // (uv) = a[2] * a[2]
  word_multiply( u, v, a->word[2], a->word[2] );
  

  // r0 = add(r0,v) -> carry = 0
  carry = C_WORD_ZERO;
  word_add_with_carry( &r0, &carry, &r0, &v );
  
         
  // r1 = add_with_carry(r1,u)
  word_add_with_carry( &r1, &carry, &r1, &u );
  

  // r2 = add_with_carry(r2,0)
  r2 += carry;
      
  // 2.2) c->word[k] = r0, r0 = r1, r1 = r2, r2 = 0
  c->word[4] = r0;
  r0   = r1;
  r1   = r2;
  r2   = C_WORD_ZERO;


  // (uv) = a[1] * a[4]
  word_multiply( u, v, a->word[1], a->word[4] );
  

  // (uv) << 1  <=> (uv) = 2 * (uv) <=> 2*u && 2*v with carry
  carry = C_WORD_ZERO;
  word_add_with_carry( &v, &carry, &v, &v );
  
  word_add_with_carry( &u, &carry, &u, &u );
  

  // r2 = add_with_carry(r2,0) <=> r2 gets carry from 2*u
  r2 += carry;

  // r0 = add(r0,v) -> carry = 0
  carry = C_WORD_ZERO;
  word_add_with_carry( &r0, &carry, &r0, &v );
  
         
  // r1 = add_with_carry(r1,u)
  word_add_with_carry( &r1, &carry, &r1, &u );
  

  // r2 = add_with_carry(r2,0)
  r2 += carry;


  // (uv) = a[2] * a[3]
  word_multiply( u, v, a->word[2], a->word[3] );
  

  // (uv) << 1  <=> (uv) = 2 * (uv) <=> 2*u && 2*v with carry
  carry = C_WORD_ZERO;
  word_add_with_carry( &v, &carry, &v, &v );
  
  word_add_with_carry( &u, &carry, &u, &u );
  

  // r2 = add_with_carry(r2,0) <=> r2 gets carry from 2*u
  r2 += carry;

  // r0 = add(r0,v) -> carry = 0
  carry = C_WORD_ZERO;
  word_add_with_carry( &r0, &carry, &r0, &v );
  
         
  // r1 = add_with_carry(r1,u)
  word_add_with_carry( &r1, &carry, &r1, &u );
  

  // r2 = add_with_carry(r2,0)
  r2 += carry;

  // 2.2) c->word[k] = r0, r0 = r1, r1 = r2, r2 = 0
  c->word[5] = r0;
  r0   = r1;
  r1   = r2;
  r2   = C_WORD_ZERO;
      

  // (uv) = a[2] * a[4]
  word_multiply( u, v, a->word[2], a->word[4] );
  

  // (uv) << 1  <=> (uv) = 2 * (uv) <=> 2*u && 2*v with carry
  carry = C_WORD_ZERO;
  word_add_with_carry( &v, &carry, &v, &v );
  
  word_add_with_carry( &u, &carry, &u, &u );
  

  // r2 = add_with_carry(r2,0) <=> r2 gets carry from 2*u
  r2 += carry;

  // r0 = add(r0,v) -> carry = 0
  carry = C_WORD_ZERO;
  word_add_with_carry( &r0, &carry, &r0, &v );
  
         
  // r1 = add_with_carry(r1,u)
  word_add_with_carry( &r1, &carry, &r1, &u );
  

  // r2 = add_with_carry(r2,0)
  r2 += carry;
      

  // (uv) = a[3] * a[3]
  word_multiply( u, v, a->word[3], a->word[3] );
  

  // r0 = add(r0,v) -> carry = 0
  carry = C_WORD_ZERO;
  word_add_with_carry( &r0, &carry, &r0, &v );
  
         
  // r1 = add_with_carry(r1,u)
  word_add_with_carry( &r1, &carry, &r1, &u );
  

  // r2 = add_with_carry(r2,0)
  r2 += carry;
      
  // 2.2) c->word[k] = r0, r0 = r1, r1 = r2, r2 = 0
  c->word[6] = r0;
  r0   = r1;
  r1   = r2;
  r2   = C_WORD_ZERO;


  // (uv) = a[3] * a[4]
  word_multiply( u, v, a->word[3], a->word[4] );
  

  // (uv) << 1  <=> (uv) = 2 * (uv) <=> 2*u && 2*v with carry
  carry = C_WORD_ZERO;
  word_add_with_carry( &v, &carry, &v, &v );
  
  word_add_with_carry( &u, &carry, &u, &u );
  

  // r2 = add_with_carry(r2,0) <=> r2 gets carry from 2*u
  r2 += carry;

  // r0 = add(r0,v) -> carry = 0
  carry = C_WORD_ZERO;
  word_add_with_carry( &r0, &carry, &r0, &v );
  
         
  // r1 = add_with_carry(r1,u)
  word_add_with_carry( &r1, &carry, &r1, &u );
  

  // r2 = add_with_carry(r2,0)
  r2 += carry;
      

  // 2.2) c->word[k] = r0, r0 = r1, r1 = r2, r2 = 0
  c->word[7] = r0;
  r0   = r1;
  r1   = r2;
  r2   = C_WORD_ZERO;


  // (uv) = a[4] * a[4]
  word_multiply( u, v, a->word[4], a->word[4] );
  

  // r0 = add(r0,v) -> carry = 0
  carry = C_WORD_ZERO;
  word_add_with_carry( &r0, &carry, &r0, &v );
  
         
  // r1 = add_with_carry(r1,u)
  word_add_with_carry( &r1, &carry, &r1, &u );
  

  // r2 = add_with_carry(r2,0)
  r2 += carry;
      
  // 2.2) c->word[k] = r0, r0 = r1, r1 = r2, r2 = 0
  c->word[8] = r0;
  r0   = r1;
  r1   = r2;
  r2   = C_WORD_ZERO;
    
  
  // 3.) c->word[9] = r0
  c->word[9] = r0;


  // sign (always +)
  c->sign = C_POSITIVE;

  // successful
  return 0;
}



/***************************************************************************/
/*!
   \par       FUNCTION F20/150: field_element_reduce_barrett

   \par       DESCRIPTION

              This function reduces a field elements with modulus p using Barrett

   \par       EQUATION / REFERENCE

              c = |a| mod p

   \par       INPUT ARGUMENTS
                      
   \param     const (tsECFieldElement) *a    - pointer to field element double

   \param     const (tsECFieldElement) *p    - pointer to field element modulus p

   \param     const (tsECFieldElement) *my_p - pointer to precomputed field element my = b^(2k)/p

   \par       OUTPUT ARGUMENTS

   \param     (tsECFieldElement) *c          - pointer to field element result c

   \par       RETURN VALUE

   \param     (int)  0                      - successful execution

   \param     (int) -1                      - parameter error

   \par       NOTES / WARNINGS / TODO's

   \warning   None
	
   \history
              05.01.2009 Modified by Divya.H (RBEI/ECM1) to make assert 
              calls return error
*/
/***************************************************************************/
extern int field_element_reduce_barrett( tsECFieldElement *c,
                                   const tsECFieldElement *a,
                                   const tsECFieldElement *my_p,
                                   const tsECFieldElement *p )
{

  // declarations
  int ret;
  tsECFieldElement q,q3;
  tsECFieldElement r1,r2,r;

  
#if (C_SKIP_PARAMETER_CHECK == 0)

  // check parameters
  if ( (c == NULL) || (a == NULL) || (p == NULL) || (my_p == NULL) ||
       (c->length < p->length) || (a->length < 1) || (p->length < 1) ||
       (a->sign != C_POSITIVE) )
  {
    return -1;
  }  

#endif

  // reduction
  if ( a->length >= p->length )
  {
  
   // 1.) ------------------------------------------
  
    // q = a [p->length - 1, .. , a->length]  ( = Q1 )
    q.sign     = a->sign;
    q.word     = &a->word[p->length - 1];
    q.length   = a->length - (p->length - 1);

    // allocate q3
    field_element_allocate( &q3, my_p->length + q.length );

    // q3 = q * my ( = Q2 )
    ret = field_element_multiply_classical( &q3, &q, my_p );
	 BPCL_CHECKERROR_FREEMEM((ret==0), "%f", &q3);

    // q = q3[p->length + 1 .. q3->length]  ( = Q3 )
    q.sign     = q3.sign;
    q.word     = &q3.word[p->length + 1];
    q.length   = q3.length - (p->length + 1);


    // 2.) ------------------------------------------

    // allocate r (for possible 3.) r needs at least same length as q3 (else p->length + 2 would be enougth)
    field_element_allocate( &r, q3.length);
 
    // if Q3 != 0 then
    ret = field_element_is_zero( &q );
    BPCL_CHECKERROR_FREEMEM((ret > -1), "%f%f", &r,&q3);
  
    if ( ret == 0 )
    {
    
      // r1 = a [0 .. p->length + 1]
      r1.sign   = a->sign;
      r1.word   = &a->word[0];
      r1.length = MIN( p->length + 1, a->length );

      // allocate r2
      field_element_allocate( &r2, q.length + p->length );
    
      // r2 = q * p
      ret = field_element_multiply_classical( &r2, &q, p );
	   BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f", &r2, &r, &q3);

      // r2 = r2 [0.. p->length + 1]
      r2.length = p->length + 1;

      // r = r1 - r2
      ret = field_element_subtract( &r, &r1, &r2 );
	   BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f", &r2, &r, &q3);

      // free r2
      field_element_free( &r2 );
    }
    else // Q3 == 0
    {
      // r = r1 ( because r2 = 0 )
      ret = field_element_assign( &r, a, MIN( p->length + 1, a->length ) );
      BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f", &r,&q3);
    }
  }
  else // ( a->length < p->length )
  {
    // allocate r
    field_element_allocate( &r, p->length + 2 );

    // allocate q3
    field_element_allocate( &q3, p->length + 2 );
    
    // r = r1 ( because r2 = 0 )
    ret = field_element_assign( &r, a, MIN( p->length + 1, a->length ) );
    BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f", &r,&q3);
  }

  // 3.) ------------------------------------------

  // do not return negative numbers!
  if ( field_element_is_negative( &r ) == 1 )
  {
      // q3 = b^(k+1) = (2^32)^(p->length + 1) => (max p->length + 2 words  1,0,0,0,0)
      ret = field_element_set_one( &q3 );
      BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f", &r,&q3);
      ret = field_element_shift_n_left( &q3, &q3, C_NUM_BITS * (p->length + 1) );
      BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f", &r,&q3);

      // r = r + b^(k+1)
      ret = field_element_add( &r, &r, &q3 );
      BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f", &r,&q3);
  }

 
  // 4.) ------------------------------------------

  // while r >= p, do r = r - p
  while ( field_element_absolute_compare( &r, p ) > -1 )
  {
    // r = r - p
    ret = field_element_subtract( &r, &r, p );
    BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f", &r,&q3);
  }

 // 5.) ------------------------------------------

 // assign c
 ret = field_element_assign( c, &r, p->length );
 BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f", &r,&q3);

 // free
 field_element_free( &q3 );
 field_element_free( &r );

  // successful
  return 0;


}



/***************************************************************************/
/*!
   \par       FUNCTION F20/151: field_element_reduce_secp160r1

   \par       DESCRIPTION

              This function reduces a field elements with modulus p
              
   \par       EQUATION / REFERENCE

              c = |a| mod p (secp160r1 only)

   \par       INPUT ARGUMENTS
                      
   \param     const (tsECFieldElement) *a    - pointer to field element double

   \param     const (tsECFieldElement) *my_p - unused pointer for function pointer compatibility

   \param     const (tsECFieldElement) *p    - pointer to field element modulus p = 2^160-2^31-1

   \par       OUTPUT ARGUMENTS

   \param     (tsECFieldElement) *c          - pointer to field element result c

   \par       RETURN VALUE

   \param     (int)  0                      - successful execution

   \param     (int) -1                      - parameter error

   \par       NOTES / WARNINGS / TODO's

   \note      Reduces only field elements of curve secp160r1!

   \note      No self assignement c = f(c) permitted, because field element (c) != double field element(a)

   \note      No check for correct p = 2^160 - 2^31 - 1 (too costly)

   \note      Ignore warning for unused my_p, its necessary for compatibility with the function pointer

   \history
              05.01.2009 Modified by Divya.H (RBEI/ECM1) to make assert 
              calls return error
*/
/***************************************************************************/
extern int field_element_reduce_secp160r1( tsECFieldElement *c,
                                     const tsECFieldElement *a,
                                     const tsECFieldElement *my_p,
                                     const tsECFieldElement *p )
{
  // declarations
  tU32 fe_tmp_words[6];

  tsECFieldElement fe_tmp;

  tU32 w32_tmp;
  tU32 carry, carry_cum;
  
  int ret;
  
#if (C_SKIP_PARAMETER_CHECK == 0)

  // check parameters
  if ( (c == NULL) || (a == NULL) || (p == NULL) || (a->sign != C_POSITIVE ) ||
       (c->length != 5) || (a->length != 10)  || (p->length != 5) )
  {
    return -1;
  }  

#endif
  
  (tVoid)my_p;
  // init fe_tmp
  fe_tmp.sign   = C_POSITIVE;
  fe_tmp.word   = fe_tmp_words;
  fe_tmp.length = 6;

  // c0 = a->word[0] ..
  carry_cum = C_WORD_ZERO;
  fe_tmp.word[0] = a->word[0];
  
  // .. + a->word[5] ..
  w32_tmp = a->word[5];
  carry = C_WORD_ZERO;
  word_add_with_carry( &fe_tmp.word[0], &carry, &fe_tmp.word[0], &w32_tmp );
  carry_cum += carry;
  
  // .. + a->word[9]>>1 ..
  w32_tmp = a->word[9] >> 1;
  carry = C_WORD_ZERO;
  word_add_with_carry( &fe_tmp.word[0], &carry, &fe_tmp.word[0], &w32_tmp );
  carry_cum += carry;
  
  // .. + a->word[5]<<31 ..
  w32_tmp = a->word[5] << 31;
  carry = C_WORD_ZERO;
  word_add_with_carry( &fe_tmp.word[0], &carry, &fe_tmp.word[0], &w32_tmp );
  carry_cum += carry;

  
  // .. + ((a->word[9]>>1)<<31)..
  w32_tmp = ((a->word[9]>>1)<<31);
  carry = C_WORD_ZERO;
  word_add_with_carry( &fe_tmp.word[0], &carry, &fe_tmp.word[0], &w32_tmp );
  carry_cum += carry;

  // c1 = carry_cum + ..
  fe_tmp.word[1] = carry_cum;
  carry_cum = C_WORD_ZERO;
  
  // .. + a->word[1] ..
  w32_tmp = a->word[1];
  carry = C_WORD_ZERO;
  word_add_with_carry( &fe_tmp.word[1], &carry, &fe_tmp.word[1], &w32_tmp );
  carry_cum += carry;
  
  // .. + a->word[5]>>1 ..
  w32_tmp = a->word[5] >> 1;
  carry = C_WORD_ZERO;
  word_add_with_carry( &fe_tmp.word[1], &carry, &fe_tmp.word[1], &w32_tmp );
  carry_cum += carry;
  
  // .. + a->word[6] ..
  w32_tmp = a->word[6];
  carry = C_WORD_ZERO;
  word_add_with_carry( &fe_tmp.word[1], &carry, &fe_tmp.word[1], &w32_tmp );
  carry_cum += carry;
  
  // .. + a->word[9]>>2 ..
  w32_tmp = a->word[9] >> 2;
  carry = C_WORD_ZERO;
  word_add_with_carry( &fe_tmp.word[1], &carry, &fe_tmp.word[1], &w32_tmp );
  carry_cum += carry;
  
  // .. + a->word[6]<<31 ..
  w32_tmp = a->word[6] << 31;
  carry = C_WORD_ZERO;
  word_add_with_carry( &fe_tmp.word[1], &carry, &fe_tmp.word[1], &w32_tmp );
  carry_cum += carry;
  
  // c2 = carry_cum + ..
  fe_tmp.word[2] = carry_cum;
  carry_cum = C_WORD_ZERO;
  
  // .. + a->word[2] ..
  w32_tmp = a->word[2];
  carry = C_WORD_ZERO;
  word_add_with_carry( &fe_tmp.word[2], &carry, &fe_tmp.word[2], &w32_tmp );
  carry_cum += carry;

  // .. + a->word[6]>>1 ..
  w32_tmp = a->word[6] >> 1;
  carry = C_WORD_ZERO;
  word_add_with_carry( &fe_tmp.word[2], &carry, &fe_tmp.word[2], &w32_tmp );
  carry_cum += carry;
  
  // .. + a->word[7] ..
  w32_tmp = a->word[7];
  carry = C_WORD_ZERO;
  word_add_with_carry( &fe_tmp.word[2], &carry, &fe_tmp.word[2], &w32_tmp );
  carry_cum += carry;
  
  // .. + a->word[7]<<31 ..
  w32_tmp = a->word[7] << 31;
  carry = C_WORD_ZERO;
  word_add_with_carry( &fe_tmp.word[2], &carry, &fe_tmp.word[2], &w32_tmp );
  carry_cum += carry;
  
  // c3 = carry_cum + ..
  fe_tmp.word[3] = carry_cum;
  carry_cum = C_WORD_ZERO;
  
  // .. + a->word[3] ..
  w32_tmp = a->word[3];
  carry = C_WORD_ZERO;
  word_add_with_carry( &fe_tmp.word[3], &carry, &fe_tmp.word[3], &w32_tmp );
  carry_cum += carry;
  
  // .. + a->word[7]>>1 ..
  w32_tmp = a->word[7] >> 1;
  carry = C_WORD_ZERO;
  word_add_with_carry( &fe_tmp.word[3], &carry, &fe_tmp.word[3], &w32_tmp );
  carry_cum += carry;

  // .. + a->word[8] ..
  w32_tmp = a->word[8];
  carry = C_WORD_ZERO;
  word_add_with_carry( &fe_tmp.word[3], &carry, &fe_tmp.word[3], &w32_tmp );
  carry_cum += carry;
  
  // .. + a->word[8]<<31 ..
  w32_tmp = a->word[8] << 31;
  carry = C_WORD_ZERO;
  word_add_with_carry( &fe_tmp.word[3], &carry, &fe_tmp.word[3], &w32_tmp );
  carry_cum += carry;
  
  // c4 = carry_cum + ..
  fe_tmp.word[4] = carry_cum;
  carry_cum = C_WORD_ZERO;
  
  // .. + a->word[4] ..
  w32_tmp = a->word[4];
  carry = C_WORD_ZERO;
  word_add_with_carry( &fe_tmp.word[4], &carry, &fe_tmp.word[4], &w32_tmp );
  carry_cum += carry;
  
  // .. + a->word[9] ..
  w32_tmp = a->word[9];
  carry = C_WORD_ZERO;
  word_add_with_carry( &fe_tmp.word[4], &carry, &fe_tmp.word[4], &w32_tmp );
  carry_cum += carry;
  
  // .. + a->word[9]<<31 ..
  w32_tmp = a->word[9]<<31;
  carry = C_WORD_ZERO;
  word_add_with_carry( &fe_tmp.word[4], &carry, &fe_tmp.word[4], &w32_tmp );
  carry_cum += carry;
  
  // .. + a->word[8]>>1 ..
  w32_tmp = a->word[8]>>1;
  carry = C_WORD_ZERO;
  word_add_with_carry( &fe_tmp.word[4], &carry, &fe_tmp.word[4], &w32_tmp );
  carry_cum += carry;

  // c5 = carry_cum
  fe_tmp.word[5] = carry_cum;

  // tmp > p ?
  ret = field_element_compare( &fe_tmp, p );
  BPCL_CHECK_ERROR((ret > -2));

  // while tmp > p => tmp = tmp - p
  while (  ret >= 0 )
  {
    ret = field_element_subtract( &fe_tmp, &fe_tmp, p );
    BPCL_CHECK_ERROR((ret==0));

    ret = field_element_compare( &fe_tmp, p );
    BPCL_CHECK_ERROR((ret > -2));
  }
  
  // c = fe_tmp
  ret = field_element_assign( c, &fe_tmp, 5 );
  BPCL_CHECK_ERROR((ret==0));

  // succesful
  return 0;
}



/***************************************************************************/
/*!
   \par       FUNCTION F20/200: field_element_modular_add

   \par       DESCRIPTION

              This function adds two field elements under modulus p

   \par       EQUATION / REFERENCE

              c = (|a| + |b|) mod p

   \par       INPUT ARGUMENTS

   \param     const (tsECFieldElement) *a - pointer to field element addend a

   \param     const (tsECFieldElement) *b - pointer to field element addend b

   \param     const (tsECField) *fGP   - pointer to field parameter

   \par       OUTPUT ARGUMENTS

   \param     (tsECFieldElement) *c       - pointer to field element summand c

   \par       RETURN VALUE

   \param     (int)  0                   - successful execution

   \param     (int) -1                   - parameter error

   \par       NOTES / WARNINGS / TODO's

   \note      a->length = b->length = c->length (otherwise function becomes too costly) 

   \note      a and b have to be already modular reduced!

   \history
              05.01.2009 Modified by Divya.H (RBEI/ECM1) to make assert 
              calls return error
*/
/***************************************************************************/
extern int field_element_modular_add( tsECFieldElement *c,
                                const tsECFieldElement *a,
                                const tsECFieldElement *b,
                                const tsECField     *fGP )
{
  // declarations 
  tU32 carry, borrow;
  int ret;
  tU32 i;

#if (C_SKIP_PARAMETER_CHECK == 0)

  // check parameters
  if ( (c == NULL) || (a == NULL) || (b == NULL) || (fGP == NULL) || (&fGP->prime_p == NULL) || 
       (c->length < 1) || (a->length < 1) || (b->length < 1) || (fGP->prime_p.length < 1) ||  
       (a->sign != C_POSITIVE) || (b->sign != C_POSITIVE) ||
       (c->length != a->length) || (c->length != b->length) || (c->length != fGP->prime_p.length) )/*lint !e774 */ /*The algorithm wants this check to be done*/ 
  {
    return -1;
  }  

#endif

  // 1.) & 2.)
  // for i from 0 to t-1 do c[i] = add_with_carry(a[i], b[i])
  // -> assembler optimized add loop
  word_add_with_carry_loop( &c->word, &carry, &a->word, &b->word, 0, a->length );

  // 3.) if carry bit is set, then subtract p from c = {c[t-1],..,c[1],c[0]}
  if ( carry == C_WORD_ONE )
  {
    // for i from 0 to t-1 do c[i] = subtract_with_borrow(c[i], p[i]) 
    // -> assembler optimized subtract loop
    word_subtract_with_borrow_loop( &c->word, &borrow, &c->word, &fGP->prime_p.word, 0, a->length );
  }
  
  // 4.) if c >= p, then c = c - p
  i = c->length;
  ret = 0;
  while ( i > 0 )
  {
    // next lower pair
    i--;

    if ( c->word[i] < fGP->prime_p.word[i] )
    {
      // c < p
      ret = -1;
      break;
    }

    if ( c->word[i] > fGP->prime_p.word[i] )
    {
      // c > p
      ret = 1;
      break;
    }
  }

  if (  ret == 1 )
  {
    // for i from 0 to t-1 do c[i] = subtract_with_borrow(c[i], p[i]) 
    // -> assembler optimized subtract loop
    word_subtract_with_borrow_loop( &c->word, &borrow, &c->word, &fGP->prime_p.word, 0, a->length );
  }
  else if ( ret == 0 )
  {
    ret = field_element_set_zero( c );
    BPCL_CHECK_ERROR((ret==0));
  }
  
  // sign (always +)
  c->sign = C_POSITIVE;

  // successful
  return 0;
}



/***************************************************************************/
/*!
   \par       FUNCTION F20/210: field_element_modular_subtract

   \par       DESCRIPTION

              This function subtracts two field elements under modulus p

   \par       EQUATION / REFERENCE

              c = (|a| - |b|) mod p

   \par       INPUT ARGUMENTS

   \param     const (tsECFieldElement) *a - pointer to field element minuend a

   \param     const (tsECFieldElement) *b - pointer to field element subtrahend b

   \param     const (tsECField) *fGP   - pointer to field parameter

   \par       OUTPUT ARGUMENTS

   \param     (tsECFieldElement) *c       - pointer to field element difference c

   \par       RETURN VALUE

   \param     (int)  0                   - successful execution

   \param     (int) -1                   - parameter error

   \par       NOTES / WARNINGS / TODO's

   \note      a->length = b->length = c->length (otherwise function becomes too costly) 

   \note      a and b have to be already modular reduced!
*/
/***************************************************************************/
extern int field_element_modular_subtract( tsECFieldElement *c,
                                     const tsECFieldElement *a,
                                     const tsECFieldElement *b,
                                     const tsECField     *fGP )
{
  // declarations 
  tU32 carry, borrow;

#if (C_SKIP_PARAMETER_CHECK == 0)

  // check parameters
  if ( (c == NULL) || (a == NULL) || (b == NULL) || (fGP == NULL) || (&fGP->prime_p == NULL) ||
       (c->length < 1) || (a->length < 1) || (b->length < 1) || (fGP->prime_p.length < 1) ||  
       (a->sign != C_POSITIVE) || (b->sign != C_POSITIVE) ||
       (c->length != a->length) || (c->length != b->length) || (c->length != fGP->prime_p.length) )/*lint !e774 */ /*The algorithm wants this check to be done*/
  {
    return -1;
  }  

#endif
  
  // 1.) & 2.)
  // for i from 0 to t-1 do c[i] = subtract_with_borrow(a[i], b[i]) 
  // -> assembler optimized subtract loop (borrow_in = 1)
  word_subtract_with_borrow_loop( &c->word, &borrow, &a->word, &b->word, 0, a->length );
  
  // 3.) if carry bit is set (i.e. borrow == 0), then add p to c = {c[t-1],..,c[1],c[0]}
  if ( borrow == C_WORD_ZERO )
  {
    // for i from 0 to t-1 do c[i] = add_with_carry(c[i], p[i])
    // -> assembler optimized add loop (carry_in = 0)
    word_add_with_carry_loop( &c->word, &carry, &c->word, &fGP->prime_p.word, 0, a->length );
  }
  
  // sign (always +)
  c->sign = C_POSITIVE;

  // successful
  return 0;
}



/***************************************************************************/
/*!
   \par       FUNCTION F20/220: field_element_modular_multiply

   \par       DESCRIPTION

              This function multiplies two field elements under modulus p

   \par       EQUATION / REFERENCE

              c = |a|  |b| mod p

   \par       INPUT ARGUMENTS

   \param     const (tsECFieldElement) *a    - pointer to field element factor a

   \param     const (tsECFieldElement) *b    - pointer to field element factor b

   \param     const (tsECField) *fGP      - pointer to field parameter

   \par       OUTPUT ARGUMENTS

   \param     (tsECFieldElement) *c          - pointer to field element product c mod p

   \par       RETURN VALUE

   \param     (int)  0                      - successful execution

   \param     (int) -1                      - parameter error

   \param     (int) -2                      - function pointer not set

   \par       NOTES / WARNINGS / TODO's

   \note      General function using the indicated function pointers!

   \note      With memory allocation!

   \history
              05.01.2009 Modified by Divya.H (RBEI/ECM1) to make assert 
              calls return error
*/
/***************************************************************************/
extern int field_element_modular_multiply( tsECFieldElement *c,
                                     const tsECFieldElement *a,
                                     const tsECFieldElement *b,
                                     const tsECField     *fGP )
{
  // declarations
  int ret;
  tsECFieldElement fe_tmp;

  
#if (C_SKIP_PARAMETER_CHECK == 0)

  // check parameters
  if ( (c == NULL) || (a == NULL) || (b == NULL) || (fGP == NULL) || (&fGP->precalc_my_p == NULL) || (&fGP->prime_p == NULL) ||
       (c->length < 1) || (a->length < 1) || (b->length < 1) || (fGP->prime_p.length < 1) ||
       (a->sign != C_POSITIVE) || (b->sign != C_POSITIVE) ||  
       (c->length != fGP->prime_p.length) )/*lint !e774 */ /*The algorithm wants this check to be done*/
  {
    return -1;
  }  
  
  // check function pointers
  if ( ( fGP->fptr_fe_multiply == NULL) || ( fGP->fptr_fe_reduce == NULL) )
  {
    return -2;
  }

#endif

  // allocate fe_tmp (doubled length of factors)
#if (C_USE_STATIC_VARS == 0)
  field_element_allocate( &fe_tmp, a->length + b->length );
#else
  fe_tmp.word = &words_temp_mul[0];
  fe_tmp.length = a->length + b->length;
  BPCL_CHECK_ERROR(((a->length + b->length) <= C_FE_TMP_MAX_MUL_WORDS));

#endif

  // multiply
  ret = (*fGP->fptr_fe_multiply) ( &fe_tmp, a, b );
  #if (C_USE_STATIC_VARS == 0)
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f", &fe_tmp);
  #else
  BPCL_CHECK_ERROR((ret==0));
  #endif

  // reduce
  ret = (*fGP->fptr_fe_reduce) ( c, &fe_tmp, &fGP->precalc_my_p, &fGP->prime_p );
  #if (C_USE_STATIC_VARS == 0)
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f", &fe_tmp);
  #else
  BPCL_CHECK_ERROR((ret==0));
  #endif

  // free fe_tmp
#if (C_USE_STATIC_VARS == 0)
  field_element_free( &fe_tmp );
#endif

  // succesful
  return 0;
}



/***************************************************************************/
/*!
   \par       FUNCTION F20/230: field_element_modular_square

   \par       DESCRIPTION

              This function squares a field elements under modulus p

   \par       EQUATION / REFERENCE

              c = |a| mod p

   \par       INPUT ARGUMENTS

   \param     const (tsECFieldElement) *a    - pointer to field element square factor a

   \param     const (tsECField) *fGP      - pointer to field parameter

   \par       OUTPUT ARGUMENTS

   \param     (tsECFieldElement) *c          - pointer to field element square product c mod p

   \par       RETURN VALUE

   \param     (int)  0                      - successful execution

   \param     (int) -1                      - parameter error

   \param     (int) -2                      - function pointer not set

   \par       NOTES / WARNINGS / TODO's

   \note      General function using the indicated function pointers!

   \note      With memory allocation!

   \history
              05.01.2009 Modified by Divya.H (RBEI/ECM1) to make assert 
              calls return error
*/
/***************************************************************************/
extern int field_element_modular_square( tsECFieldElement *c,
                                   const tsECFieldElement *a,
                                   const tsECField     *fGP )
{
  // declarations
  int ret;
  tsECFieldElement fe_tmp;

  
#if (C_SKIP_PARAMETER_CHECK == 0)

  // check parameters
  if ( (c == NULL) || (a == NULL) || (fGP == NULL) || (&fGP->precalc_my_p == NULL) || (&fGP->prime_p == NULL) ||
       (c->length < 1) || (a->length < 1) || (fGP->prime_p.length < 1) ||
       (a->sign != C_POSITIVE) || (c->length < fGP->prime_p.length) )/*lint !e774 */ /*The algorithm wants this check to be done*/
  {
    return -1;
  }  
  
  // check function pointers
  if ( (fGP->fptr_fe_square == NULL) || (fGP->fptr_fe_reduce == NULL) )
  {
    return -2;
  }

#endif
  
  // allocate memory for fe_tmp (doubled length of factors)
#if (C_USE_STATIC_VARS == 0)
  field_element_allocate( &fe_tmp, 2 * a->length ); 
#else
  fe_tmp.word   = &words_temp_mul[0];
  fe_tmp.length = 2 * a->length;
  BPCL_CHECK_ERROR(((2 * a->length) <= C_FE_TMP_MAX_MUL_WORDS));
#endif

  // multiply
  ret = (*fGP->fptr_fe_square) ( &fe_tmp, a );
  #if (C_USE_STATIC_VARS == 0)
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f", &fe_tmp);
  #else
  BPCL_CHECK_ERROR((ret==0));
  #endif
  
  // reduce
  ret = (*fGP->fptr_fe_reduce) ( c, &fe_tmp, &fGP->precalc_my_p, &fGP->prime_p );
  #if (C_USE_STATIC_VARS == 0)
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f", &fe_tmp);
  #else
  BPCL_CHECK_ERROR((ret==0));
  #endif

  // free fe_tmp
#if (C_USE_STATIC_VARS == 0)
  field_element_free( &fe_tmp );
#endif

  // succesful
  return 0;
}



/***************************************************************************/
/*!
   \par       FUNCTION F20/240: field_element_modular_invert

   \par       DESCRIPTION

              This function reverses a field elements with modulus p (inversion)
              applying the Extended Euclidean algorithm

   \par       EQUATION / REFERENCE

              c = |a|^(-1) mod p

   \par       INPUT ARGUMENTS

   \param     const (tsECFieldElement) *a - pointer to field element a

   \param     const (tsECField) *fGP   - pointer to field parameter

   \par       OUTPUT ARGUMENTS

   \param     (tsECFieldElement) *c       - pointer to field element result c mod p

   \par       RETURN VALUE

   \param     (int)  0                   - successful execution

   \param     (int) -1                   - parameter error

   \param     (int) -2                   - a is not invertable

   \par       NOTES / WARNINGS / TODO's

   \note      Uses memory allocation!

   \history
              05.01.2009 Modified by Divya.H (RBEI/ECM1) to make assert 
              calls return error
*/
/***************************************************************************/
extern int field_element_modular_invert( tsECFieldElement *c,
                                   const tsECFieldElement *a,
                                   const tsECField     *fGP )
{
  // declarations
  tsECFieldElement  u;
  tsECFieldElement  v;
  tsECFieldElement lA;
  tsECFieldElement lC;

  int ret;
  
#if (C_SKIP_PARAMETER_CHECK == 0)

   // check parameters
  if ( (c == NULL) || (a == NULL) || (fGP == NULL) || (&fGP->prime_p == NULL) || 
       (c->length < 1) || (a->length < 1) || (fGP->prime_p.length < 1) ||
       (a->sign != C_POSITIVE) ||  
       (c->length != a->length) || (c->length != fGP->prime_p.length) )/*lint !e774 */ /*The algorithm wants this check to be done*/
  {
    return -1;
  }  

#endif

  // allocate memory for auxiliary variables => length + overflow
#if (C_USE_STATIC_VARS == 0)
  field_element_allocate( &u, a->length + 1 ); 
  field_element_allocate( &v, a->length + 1 ); 
  field_element_allocate( &lA, a->length + 1 ); 
  field_element_allocate( &lC, a->length + 1 ); 
#else
  u.word    = &words_temp_inv1[0];
  u.length  = a->length + 1;
  v.word    = &words_temp_inv2[0];
  v.length  = a->length + 1;
  lA.word   = &words_temp_inv3[0];
  lA.length = a->length + 1;
  lC.word   = &words_temp_inv4[0];
  lC.length = a->length + 1;
  BPCL_CHECK_ERROR(((a->length + 1) <= C_FE_TMP_MAX_INV_WORDS));
#endif
  
  // 1.) u = a, v = p, A = 1, C = 0
  ret = field_element_assign( &u, a, a->length );
  #if (C_USE_STATIC_VARS == 0)
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f", &u,&v,&lA,&lC);
  #else
  BPCL_CHECK_ERROR((ret==0));
  #endif
  ret = field_element_assign( &v, &fGP->prime_p, fGP->prime_p.length );
  #if (C_USE_STATIC_VARS == 0)
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f", &u,&v,&lA,&lC);
  #else
  BPCL_CHECK_ERROR((ret==0));
  #endif
  ret = field_element_set_one( &lA );
  #if (C_USE_STATIC_VARS == 0)
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f", &u,&v,&lA,&lC);
  #else
  BPCL_CHECK_ERROR((ret==0));
  #endif
  ret = field_element_set_zero( &lC );
  #if (C_USE_STATIC_VARS == 0)
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f", &u,&v,&lA,&lC);
  #else
  BPCL_CHECK_ERROR((ret==0));
  #endif

  // 2.) while ( u != 0 ) do
  while ( field_element_is_zero( &u ) == 0 )
  {
    
    // 2.1) while u is even do
    while ( word_is_even( &u.word[0] ) > 0 )
    {
      // u = u / 2
      ret = field_element_shift_n_right( &u, &u, 1 );
      #if (C_USE_STATIC_VARS == 0)
      BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f", &u,&v,&lA,&lC);
      #else
      BPCL_CHECK_ERROR((ret==0));
      #endif
      
      // if A is even 
      if ( word_is_even( &lA.word[0] ) > 0 )
      {
        // then A =  A / 2
        ret = field_element_shift_n_right( &lA, &lA, 1 );
		#if (C_USE_STATIC_VARS == 0)
        BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f", &u,&v,&lA,&lC);
        #else
        BPCL_CHECK_ERROR((ret==0));
        #endif
      }
      else
      {
        // else A = ( A + p ) / 2
        ret = field_element_add( &lA , &lA , &fGP->prime_p );
		#if (C_USE_STATIC_VARS == 0)
        BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f", &u,&v,&lA,&lC);
        #else
        BPCL_CHECK_ERROR((ret==0));
        #endif

        ret = field_element_shift_n_right( &lA, &lA, 1 );
		#if (C_USE_STATIC_VARS == 0)
        BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f", &u,&v,&lA,&lC);
        #else
        BPCL_CHECK_ERROR((ret==0));
        #endif
      }

    } // end while u is even
    
    
    // 2.2) while v is even do
    while ( word_is_even( &v.word[0] ) > 0 )
    { 
      // v = v / 2
      ret = field_element_shift_n_right( &v, &v, 1 );
      #if (C_USE_STATIC_VARS == 0)
      BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f", &u,&v,&lA,&lC);
      #else
      BPCL_CHECK_ERROR((ret==0));
      #endif
       
      // if C is even
      if ( word_is_even( &lC.word[0] ) > 0 )
      {
        // then C = C / 2
        ret = field_element_shift_n_right( &lC, &lC, 1 );
		#if (C_USE_STATIC_VARS == 0)
        BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f", &u,&v,&lA,&lC);
        #else
        BPCL_CHECK_ERROR((ret==0));
        #endif
      }
      else
      {
        // else C = ( C + p ) / 2
        ret = field_element_add( &lC , &lC , &fGP->prime_p );
		#if (C_USE_STATIC_VARS == 0)
        BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f", &u,&v,&lA,&lC);
        #else
        BPCL_CHECK_ERROR((ret==0));
        #endif
        
        ret = field_element_shift_n_right( &lC, &lC, 1 );
		#if (C_USE_STATIC_VARS == 0)
        BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f", &u,&v,&lA,&lC);
        #else
        BPCL_CHECK_ERROR((ret==0));
        #endif
      }

    } // end while v is even
    
    
    // 2.3) if u >= v
    if ( field_element_compare( &u, &v ) >= 0 )
    {
      // then u = u - v, A = A - C     
      ret = field_element_subtract( &u, &u, &v );
      #if (C_USE_STATIC_VARS == 0)
      BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f", &u,&v,&lA,&lC);
      #else
      BPCL_CHECK_ERROR((ret==0));
      #endif
      ret = field_element_subtract( &lA, &lA, &lC );
      #if (C_USE_STATIC_VARS == 0)
      BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f", &u,&v,&lA,&lC);
      #else
      BPCL_CHECK_ERROR((ret==0));
      #endif
    }
    else
    {
      // else v = v - u, C = C - A
      ret = field_element_subtract( &v , &v , &u );
      #if (C_USE_STATIC_VARS == 0)
      BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f", &u,&v,&lA,&lC);
      #else
      BPCL_CHECK_ERROR((ret==0));
      #endif
      ret = field_element_subtract( &lC , &lC , &lA );
	   #if (C_USE_STATIC_VARS == 0)
      BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f", &u,&v,&lA,&lC);
      #else
      BPCL_CHECK_ERROR((ret==0));
      #endif
    }
  }

  
  // 3.) return C mod p
  
  // if v != 1, gcd(a, p) != 1 and therefore a is not invertable
  if ( v.word[0] != C_WORD_ONE )
  { 
      return -2;
  }
  
  // while C < 0 then C = C + p
  while ( field_element_is_negative( &lC ) == 1 )
  {
    ret = field_element_add( &lC, &lC, &fGP->prime_p );
	 #if (C_USE_STATIC_VARS == 0)
    BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f", &u,&v,&lA,&lC);
    #else
    BPCL_CHECK_ERROR((ret==0));
    #endif
  }
  
  // while C > p then C = C - p 
  while ( field_element_compare( &lC, &fGP->prime_p ) > 0 )
  {
    ret = field_element_subtract( &lC, &lC, &fGP->prime_p );
	 #if (C_USE_STATIC_VARS == 0)
    BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f", &u,&v,&lA,&lC);
    #else
    BPCL_CHECK_ERROR((ret==0));
    #endif
  }

  // assign field element c (without leading word)
  ret = field_element_assign( c, &lC, c->length );
  #if (C_USE_STATIC_VARS == 0)
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f%f%f%f", &u,&v,&lA,&lC);
  #else
  BPCL_CHECK_ERROR((ret==0));
  #endif

  // free auxiliary variables
#if (C_USE_STATIC_VARS == 0)
  field_element_free( &u );
  field_element_free( &v );
  field_element_free( &lA );
  field_element_free( &lC );
#endif

  // successful
  return 0;
}




/***************************************************************************/
/*!
   \par       FUNCTION F20/250: field_element_modular_divide_2

   \par       DESCRIPTION

              This function divides a field elements by 2 under modulus p

   \par       EQUATION / REFERENCE

              c = |a|/2 mod p

   \par       INPUT ARGUMENTS

   \param     const (tsECFieldElement) *a - pointer to field element that will be divided by 2

   \param     const (tsECField) *fGP   - pointer to field parameter

   \par       OUTPUT ARGUMENTS

   \param     (tsECFieldElement) *c       - pointer to resulting field element

   \par       RETURN VALUE

   \param     (int)  0                   - successful execution

   \param     (int) -1                   - parameter error

   \param     (int) -2                   - function pointer not set

   \par       NOTES / WARNINGS / TODO's

   \note      General function using the indicated function pointers!

   \note      With memory allocation!

   \history
              05.01.2009 Modified by Divya.H (RBEI/ECM1) to make assert 
              calls return error
*/
/***************************************************************************/
extern int field_element_modular_divide_2( tsECFieldElement *c,
                                     const tsECFieldElement *a,
                                     const tsECField     *fGP )
{
  // declarations
  int ret;
  tsECFieldElement fe_tmp;
  tU32 carry;

#if (C_SKIP_PARAMETER_CHECK == 0)

  // check parameters
  if ( (c == NULL) || (a == NULL) || (fGP == NULL) || (&fGP->prime_p == NULL) || 
       (c->length < 1) || (a->length < 1) || (fGP->prime_p.length < 1) ||  
       (c->length != a->length) || (c->length != fGP->prime_p.length) )/*lint !e774 */ /*The algorithm wants this check to be done*/
  {
    return -1;
  }  

#endif
  
  // a =? even
  if ( word_is_even(&a->word[0]) > 0 )
  {
    // simple 1-bit right shift
    ret = field_element_shift_n_right( c, a, 1 );
    BPCL_CHECK_ERROR((ret==0));
  }
  else
  {
    // allocate memory for fe_tmp => word_length of a + overflow word
#if (C_USE_STATIC_VARS == 0)
  field_element_allocate( &fe_tmp, a->length + 1 );
#else
  fe_tmp.word = &words_temp_sub[0];
  fe_tmp.length = a->length + 1;
  BPCL_CHECK_ERROR((a->length + 1 <= C_FE_TMP_MAX_SUB_WORDS));
#endif

    // fe_tmp->sign
    fe_tmp.sign = C_POSITIVE;

    // fe_tmp = a + p
    // -> assembler optimized add loop
    word_add_with_carry_loop( &fe_tmp.word, &carry, &a->word, &fGP->prime_p.word, 0, fGP->prime_p.length );
    
    // last word takes carry
    fe_tmp.word [a->length] = carry;

    // fe_tmp = fe_tmp / 2 (fe_tmp is even)
    ret = field_element_shift_n_right( &fe_tmp, &fe_tmp, 1 );
	 #if (C_USE_STATIC_VARS == 0)
    BPCL_CHECKERROR_FREEMEM((ret==0), "%f", &fe_tmp);
	 #else
	 BPCL_CHECK_ERROR((ret==0));
    #endif

    // assign c = fe_tmp (but ignore the highest word!)
    ret = field_element_assign( c, &fe_tmp, fe_tmp.length - 1 );
	 #if (C_USE_STATIC_VARS == 0)
    BPCL_CHECKERROR_FREEMEM((ret==0), "%f", &fe_tmp);
	 #else
	 BPCL_CHECK_ERROR((ret==0));
    #endif

    // free fe_tmp
#if (C_USE_STATIC_VARS == 0)
    field_element_free( &fe_tmp );
#endif
  }
  
  // successful
  return 0;
}



/***************************************************************************/
/*!
   \par       FUNCTION F20/300: field_element_degree

   \par       DESCRIPTION

              This function returns the degree of a field element a
                   
   \par       EQUATION / REFERENCE

              a < 2^(degree + 1) <=> highest bit set is 2^(degree)

   \par       INPUT ARGUMENTS

   \param     const (tsECFieldElement) *a - pointer to field element, that will be analyzed

   \par       OUTPUT ARGUMENTS

   \param     None

   \par       RETURN VALUE

   \param     (int) >= 0                 - degree of field element

   \param     (int)   -1                 - parameter error

   \param     (int)   -2                 - field element is zero or less

   \par       NOTES / WARNINGS / TODO's

   \note      Inefficient in 16-bit or 8-bit mode (better rewrite then)!
*/
/***************************************************************************/
extern int field_element_degree( const tsECFieldElement *a ) 
{ 
  // declarations   
  int i;
  int highest_bit;

#if (C_SKIP_PARAMETER_CHECK == 0)

  // check parameters
  if ( (a == NULL) || (a->length < 1) )
  {
    return -1;
  }

#endif

  // find highest nonzero word
  i = (int) ( a->length - 1 );
  while ( (a->word[i] == C_WORD_ZERO) && (i > 0) ) 
  {
    i = i - 1;
  }

  // find highest nonzero bit    
  if      (a->word[i] >= C_WORD_BIT_32)
    highest_bit = 32;
  else if (a->word[i] >= C_WORD_BIT_31)
    highest_bit = 31;
  else if (a->word[i] >= C_WORD_BIT_30)
    highest_bit = 30;
  else if (a->word[i] >= C_WORD_BIT_29)
    highest_bit = 29;
  else if (a->word[i] >= C_WORD_BIT_28)
    highest_bit = 28;        
  else if (a->word[i] >= C_WORD_BIT_27)
    highest_bit = 27;
  else if (a->word[i] >= C_WORD_BIT_26)
    highest_bit = 26;
  else if (a->word[i] >= C_WORD_BIT_25)
    highest_bit = 25;
  else if (a->word[i] >= C_WORD_BIT_24)
    highest_bit = 24;            
  else if (a->word[i] >= C_WORD_BIT_23)
    highest_bit = 23;
  else if (a->word[i] >= C_WORD_BIT_22)
    highest_bit = 22;
  else if (a->word[i] >= C_WORD_BIT_21)
    highest_bit = 21;
  else if (a->word[i] >= C_WORD_BIT_20)
    highest_bit = 20;
  else if (a->word[i] >= C_WORD_BIT_19)
    highest_bit = 19;    
  else if (a->word[i] >= C_WORD_BIT_18)
    highest_bit = 18;
  else if (a->word[i] >= C_WORD_BIT_17)
    highest_bit = 17;
  else if (a->word[i] >= C_WORD_BIT_16) 
    highest_bit = 16; 
  else if (a->word[i] >= C_WORD_BIT_15) 
    highest_bit = 15;
  else if (a->word[i] >= C_WORD_BIT_14) 
    highest_bit = 14;
  else if (a->word[i] >= C_WORD_BIT_13) 
    highest_bit = 13;
  else if (a->word[i] >= C_WORD_BIT_12) 
    highest_bit = 12;
  else if (a->word[i] >= C_WORD_BIT_11) 
    highest_bit = 11;
  else if (a->word[i] >= C_WORD_BIT_10) 
    highest_bit = 10;
  else if (a->word[i] >= C_WORD_BIT_09) 
    highest_bit = 9;
  else if (a->word[i] >= C_WORD_BIT_08) 
    highest_bit = 8;
  else if (a->word[i] >= C_WORD_BIT_07) 
    highest_bit = 7;
  else if (a->word[i] >= C_WORD_BIT_06) 
    highest_bit = 6;
  else if (a->word[i] >= C_WORD_BIT_05) 
    highest_bit = 5;
  else if (a->word[i] >= C_WORD_BIT_04) 
    highest_bit = 4;
  else if (a->word[i] >= C_WORD_BIT_03) 
    highest_bit = 3;
  else if (a->word[i] >= C_WORD_BIT_02) 
    highest_bit = 2;
  else if (a->word[i] >= C_WORD_BIT_01) 
    highest_bit = 1;
  else // if (a->word[i] == C_WORD_ZERO or less) 
    return -2;
        
  // return degree
  return ( (i * C_NUM_BITS) + highest_bit - 1 );
}



/***************************************************************************/
/*!
   \par       FUNCTION F20/310: field_element_wNAF

   \par       DESCRIPTION

              This function returns the non-adjacent form (NAF) of k
                   
   \par       EQUATION / REFERENCE

              k_NAF = wNAF(k)

   \par       INPUT ARGUMENTS

   \param     (tS8) *k_NAF              - pointer to tS8 array k_NAF[]

   \param     const (tU32) k_NAF_length - length of k_NAF[] array

   \param     const (tsECFieldElement) *k - pointer to field element k

   \param     const (int) window_size    - NAF window size

   \par       OUTPUT ARGUMENTS

   \param     None

   \par       RETURN VALUE

   \param     (int) >= 0                 - length of k_NAF

   \param     (int)   -1                 - parameter error

   \param     (int)   -2                 - k_NAF[] array too short

   \par       NOTES / WARNINGS / TODO's

   \note      k_NAF needs to be allocated at least tS8[32 * word_length of a + 1]!!

   \note      For execution efficiency we work only with the last word of k!

   \note      Among w consecutive coefficients, at most one is nonzero (i.e. nonzero density = 1/(w+1) )

   \history
              05.01.2009 Modified by Divya.H (RBEI/ECM1) to make assert 
              calls return error
*/
/***************************************************************************/
extern int field_element_wNAF( tS8            *k_NAF,
                         const tU32            k_NAF_length,
                         const tsECFieldElement *k,
                         const tU8             window_size )
{
  // declarations
  int ret;
  int window_full, window_half;
  int u;
  int pos_k;
  tsECFieldElement local_k;
  tU32 i;

#if (C_SKIP_PARAMETER_CHECK == 0)

  // check parameters ( 2[binary] <= windowsize < 8 [limit of tS8] )
  if ( (k_NAF == NULL) || (k == NULL) || (k->length < 1) || (k_NAF_length < 1) || (window_size < (tU8)2) || (window_size > (tU8)7)  )
  {
    return -1;
  }      

#endif

  // allocate auxiliary variable variable local_k
  field_element_allocate( &local_k, k->length );

  // local_k = k
  ret = field_element_assign( &local_k, k, k->length );
  BPCL_CHECKERROR_FREEMEM((ret==0), "%f", &local_k);

  // assign l the degree of k (= highest bit set)
  pos_k =  field_element_degree( &local_k );

  // calculate windows
  window_half = 2;
  for ( i = 0; i < (tU32)(window_size - (tU8)2); i++ )
  {
    window_half = window_half * 2;
  }
  window_full = window_half * 2;
  
  // 1.) i = 0
  i = 0;
        
  // 2.) while k >= 1 do
  while( (pos_k >= 32) || (local_k.word[0] != C_WORD_ZERO) )
  {     
    
    // check k_NAF[] array length
    if ( i == k_NAF_length )
    {
      // free local_k
      field_element_free( &local_k );

      // return error
      return -2;
    }

    // 2.1) if k is odd then
    if ( word_is_odd(&local_k.word[0]) == 1 )
    { 
      
      // 2.1) u = k[i] mod 2^(w) with -2^(w-1) <= u < 2^(w)
      // NAF(w=2): k_NAF[i] = (tS8) (2 - (local_k[0] % 4));
      u = (int) ( 2 - ((int)local_k.word[0] % window_full) );
      if ( u >= window_half )
      {
        u = u - window_full;
      }
      k_NAF[i] = (tS8) u;

      // 2.1) k = k - k[i]
      local_k.word[0] = local_k.word[0] - k_NAF[i];

    }
    else
    {
      // 2.2) k[i] = 0
      k_NAF[i] = (tS8) 0;
    }
    
    // 2.3) k = k/2
    ret = field_element_shift_n_right( &local_k, &local_k, 1 );
    BPCL_CHECKERROR_FREEMEM((ret==0), "%f", &local_k);

    // 2.3) i = i + 1
    i = i + 1;

    // for execution efficiency we work only with the last word of k
    pos_k = pos_k - 1;

  }

  // free local_k
  field_element_free( &local_k );

  // return length of k_NAF
  return (int) i;   
}

