//-----------------------------------------------------------------------------
//
//  BPCL_INT.H
//
//-----------------------------------------------------------------------------
//
//  Internal Definitions and Interfaces for Blaupunkt Cryptography Library
//
//  Copyright (c) 2006 Blaupunkt GmbH, Hildesheim
//
//  Author: Holger Listle, CM-DI/ESN
//
//  \history
//              05.01.2009 Modified by Divya.H (RBEI/ECM1) to make assert 
//              calls return error
//-----------------------------------------------------------------------------

// #define NDEBUG

/* Fix for MMS 219437. Assert calls in BPCL should be made to return error.
   Hence, if condition evaluates to 0, BPCL should return error.
   Any memory allocated should be freed before returning error. */
#define BPCL_CHECKERROR_FREEMEM(condition,fmt,...) \
if((condition)==0) \
{ \
return (BPCL_Free_Memory(fmt, __VA_ARGS__)); \
}

/* Fix for MMS 219437. Assert calls in BPCL should be made to return error.
   Hence, if condition evaluates to 0, BPCL should return error. */
#define BPCL_CHECK_ERROR(condition) \
if((condition)==0) \
{ \
return (-1); \
}

extern tU8 _tmpU8; // RC4.c

/*****************************************************************************
 *
 * FUNCTION:	     BPCL_Free_Memory 
 *
 * DESCRIPTION:	Free memory before returning error.
 *                Error is returned because this function is called 
 *                to free memory only when an error occurs in BPCL.
 *
 * PARAMETERS:
 *  fmt    - format specifier for variable number of arguments
 *
 *  (...)  - variable number of arguments specifying addresses 
 *           of memory to be freed
 *
 * RETURNVALUE:
 *	   -1   - return error
 *			Error is returned because this function is called 
 *           to free memory only when an error occurs in BPCL.
 *
 * NOTE:
 * Format specifiers and their respective arguments should be used as follows:
 *
 * For freeing field_element:
 * BPCL_Free_Memory( "%f", <address of field_element> );
 *
 * For freeing point:
 * BPCL_Free_Memory( "%p", <address of point> );
 *
 * For freeing memory allocated to M_MEM_ALLOC:
 * BPCL_Free_Memory( "%m", <address to be freed> );
 *
 * For freeing memory allocated to array points:
 * BPCL_Free_Memory( "%<n>a", 
 *                        <address of the last index of array to be freed> );
 * <n> -Number of indexes in the array to be freed
 *
 * HISTORY:
 *   DATE      |  AUTHOR  |         MODIFICATION
 *	 05/01/09   |  Divya H |	Initial version
 *
 ****************************************************************************/
extern tS8 BPCL_Free_Memory(tString pcFmt, ...);
//-----------------------------------------------------------------------------
// Macros
//-----------------------------------------------------------------------------

#define M_BUF_TO_U32(d, s, i) {			\
  (d) = ( (tU32) (s)[(i)    ] << 24 ) |	\
        ( (tU32) (s)[(i) + 1] << 16 ) |	\
        ( (tU32) (s)[(i) + 2] <<  8 ) |	\
        ( (tU32) (s)[(i) + 3]       );	\
}

#define M_U32_TO_BUF(s, d, i) {			\
  (d)[(i)    ] = (tU8)((s) >> 24);		\
  (d)[(i) + 1] = (tU8)((s) >> 16);		\
  (d)[(i) + 2] = (tU8)((s) >>  8);		\
  (d)[(i) + 3] = (tU8)((s)      );		\
}

#define M_BUF_TO_U64(d, s, i) {			\
  (d) = ( (tU64) (s)[(i)    ] << 56 ) |	\
        ( (tU64) (s)[(i) + 1] << 48 ) |	\
        ( (tU64) (s)[(i) + 2] << 40 ) |	\
        ( (tU64) (s)[(i) + 3] << 32 ) |	\
        ( (tU64) (s)[(i) + 4] << 24 ) |	\
        ( (tU64) (s)[(i) + 5] << 16 ) |	\
        ( (tU64) (s)[(i) + 6] <<  8 ) |	\
        ( (tU64) (s)[(i) + 7]       );	\
}

#define M_U64_TO_BUF(s, d, i) {			\
  (d)[(i)    ] = (tU8)((s) >> 56);		\
  (d)[(i) + 1] = (tU8)((s) >> 48);		\
  (d)[(i) + 2] = (tU8)((s) >> 40);		\
  (d)[(i) + 3] = (tU8)((s) >> 32);		\
  (d)[(i) + 4] = (tU8)((s) >> 24);		\
  (d)[(i) + 5] = (tU8)((s) >> 16);		\
  (d)[(i) + 6] = (tU8)((s) >>  8);		\
  (d)[(i) + 7] = (tU8)((s)      );		\
}

#define MAX(x,y) (((x)<(y))?(y):(x))
#define MIN(x,y) (((x)<(y))?(x):(y))

#define M_ROTL(x,n)	(((x) << (n)) | (((x) & 0xFFFFFFFF) >> (32 - (n))))
#define M_ROTR(x,n)	(((x) >> (n)) | (((x) & 0xFFFFFFFF) << (32 - (n))))

#define M_SWAPBYTE(x,y)		{_tmpU8 = *(x); *(x) = *(y); *(y) = _tmpU8;}

#include <stdio.h>
#define	M_DEBUG_PRINT(m)	printf("DEBUG: %s\n", m);


//-----------------------------------------------------------------------------
// Constants
//-----------------------------------------------------------------------------

// size of a word in bit
#define C_NUM_BITS 32

// special words
#define C_WORD_ALL    0xffffffffUL
#define C_WORD_MSB    0x80000000UL
#define C_WORD_ZERO   0x00000000UL
#define C_WORD_ONE    0x00000001UL

// define in 8/16-bit mode words > 8/16-bit as 0xffff/0xffffffff
// or better rewrite gfp_field_math.c/field_element_degree()
#define C_WORD_BIT_01 0x00000001UL
#define C_WORD_BIT_02 0x00000002UL
#define C_WORD_BIT_03 0x00000004UL
#define C_WORD_BIT_04 0x00000008UL
#define C_WORD_BIT_05 0x00000010UL
#define C_WORD_BIT_06 0x00000020UL
#define C_WORD_BIT_07 0x00000040UL
#define C_WORD_BIT_08 0x00000080UL
#define C_WORD_BIT_09 0x00000100UL 
#define C_WORD_BIT_10 0x00000200UL
#define C_WORD_BIT_11 0x00000400UL
#define C_WORD_BIT_12 0x00000800UL
#define C_WORD_BIT_13 0x00001000UL
#define C_WORD_BIT_14 0x00002000UL
#define C_WORD_BIT_15 0x00004000UL
#define C_WORD_BIT_16 0x00008000UL
#define C_WORD_BIT_17 0x00010000UL
#define C_WORD_BIT_18 0x00020000UL
#define C_WORD_BIT_19 0x00040000UL
#define C_WORD_BIT_20 0x00080000UL
#define C_WORD_BIT_21 0x00100000UL
#define C_WORD_BIT_22 0x00200000UL
#define C_WORD_BIT_23 0x00400000UL
#define C_WORD_BIT_24 0x00800000UL
#define C_WORD_BIT_25 0x01000000UL
#define C_WORD_BIT_26 0x02000000UL
#define C_WORD_BIT_27 0x04000000UL
#define C_WORD_BIT_28 0x08000000UL
#define C_WORD_BIT_29 0x10000000UL
#define C_WORD_BIT_30 0x20000000UL
#define C_WORD_BIT_31 0x40000000UL
#define C_WORD_BIT_32 0x80000000UL

// sign of a field element
#define C_POSITIVE (tU8) 0
#define C_NEGATIVE (tU8) 1

// ECC point type
#define C_PT_NONE     (tU8) 0
#define C_PT_AFFINE   (tU8) 1
#define C_PT_JACOBIAN (tU8) 2

//added to remove lint warnings
#define C_SKIP_PARAMETER_CHECK 0 
#define C_USE_INLINE_ASSEMBLER 0

//-----------------------------------------------------------------------------
// ECC types
//-----------------------------------------------------------------------------

// field element
typedef struct {
  tU8			sign;
  tU32			*word;
  tU32			length; 
} tsECFieldElement;


// function pointer multiply ( result, factor a, factor b )
typedef int (*tfEC_FE_Multiply) (
	tsECFieldElement*, 
	const tsECFieldElement*, 
	const tsECFieldElement*
);

// function pointer square ( result, factor a )
typedef int (*tfEC_FE_Square) (
	tsECFieldElement*, 
	const tsECFieldElement*
);

// function pointer reduce ( result, reducer, my_prime, prime )
typedef int (*tfEC_FE_Reduce) (
	tsECFieldElement*,
	const tsECFieldElement*,
	const tsECFieldElement*,
	const tsECFieldElement*
);

// field parameters
typedef struct {
  tfEC_FE_Multiply		fptr_fe_multiply;
  tfEC_FE_Square		fptr_fe_square;
  tfEC_FE_Reduce		fptr_fe_reduce;
  tsECFieldElement		precalc_my_p;
  tsECFieldElement		prime_p;
} tsECField;


// elliptic curve point
typedef struct {
  tU8					type;
  tsECFieldElement		x;
  tsECFieldElement		y;
  tsECFieldElement		z;
} tsECPoint;

// elliptic curve
typedef struct {
  tsECFieldElement		prime_p;
  tsECFieldElement		coefficient_a;
  tsECFieldElement		coefficient_b;
  tsECPoint				base_point_G;
  tsECFieldElement		base_point_order_n;
  tU32					cofactor_h;
  tsECFieldElement		precalc_my_p;
  tsECFieldElement		precalc_my_n;
  tsECField				*ecc_field_params;  
} tsECCurve;

// elliptic curve signature
typedef struct {
  tsECFieldElement		r;
  tsECFieldElement		s;
} tsECSignature;

// precomputed point P
typedef struct {
  tsECPoint				*precomputed_P;
  tU16					length_precomputed_P;
  tU8					window_size;
} tsECPrecomputedP;


//-----------------------------------------------------------------------------
// Key masking
//-----------------------------------------------------------------------------

#define  global_key_mask0	_gkmka	// hide 'global_key_mask' from symbol table
#define  global_key_mask1	_gkmkb
#define  global_key_mask2	_gkmkc
#define  global_key_mask3	_gkmkd
#define  global_key_mask4	_gkmke

extern tU8  global_key_mask0[];
extern tU8  global_key_mask1[];
extern tU8  global_key_mask2[];
extern tU8  global_key_mask3[];
extern tU8  global_key_mask4[];
extern int  _mi;

#define  M_MASK_KEY(k,l) { \
	for(_mi = 0; _mi < (int)(l); ++_mi) {		\
		if(_mi < 10) {								\
			(k)[_mi] ^= global_key_mask0[_mi];		\
		} else if(_mi < 20) {							\
			(k)[_mi] ^= global_key_mask1[_mi-10];	\
		} else if(_mi < 30) {							\
			(k)[_mi] ^= global_key_mask2[_mi-20];	\
		} else if(_mi < 40) {							\
			(k)[_mi] ^= global_key_mask3[_mi-30];	\
		} else  {									\
			(k)[_mi] ^= global_key_mask4[_mi%10];	\
		}											\
	}												\
}

//-----------------------------------------------------------------------------
// Test definitions
//-----------------------------------------------------------------------------

#ifdef _BPCL_TEST

#define  BPCL_ERR_TEST_FAILED	(0xF0)

#define  BPCL_TEST_COMPLIANCE	(0x01)
#define  BPCL_TEST_PERFORMANCE	(0x02)
#define  BPCL_TEST_ALL			(BPCL_TEST_COMPLIANCE|BPCL_TEST_PERFORMANCE)

extern void BPCL_TimerStart();
extern void BPCL_TimerGet(tU32 *p_sec, tU32 *p_msec);
extern void BPCL_TimerReport(const char *s);
extern void BPCL_PrintBuffer(tU8 *s, tU8* buf, tU32 buflen);

#endif

