/******************************************************************************
 *
 * FILE:
 *     bpcl_error_handling.c
 *
 * REVISION:
 *     1.0
 *
 * AUTHOR:
 *     (c) 2009, Robert Bosch Engineering and Business Solutions Limited, 
 *               ECM/ECM-1, Divya H, divya.h@in.bosch.com
 *
 * CREATED:
 *     05/01/2009 - Divya H
 *
 * DESCRIPTION:
 *       This file contains BPCL error handling code.
 *
 * NOTES:
 *
 * MODIFIED:
 *   DATE     |  AUTHOR  |         MODIFICATION
 *   30/8/12  |  kgr1kor | Lint Fix
 *
 *****************************************************************************/

/****************************************************************************
  | includes of component-internal interfaces
  | (scope: component-local)
  |------------------------------------------------------------------------*/
/* Interface for variable number of arguments */
#include <stdarg.h>

/* Interface for BPCL typedef data types */
#include "bpcl.h"

/* Interface for BPCL functions */
#include "bpcl_int.h"

/* Interface for field_element_free() function */
#include "ecc.h"

/*****************************************************************
| defines and macros (scope: module-local)
|----------------------------------------------------------------*/

/*****************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------*/

/*****************************************************************
| variable definition (scope: global)
|----------------------------------------------------------------*/
   
/*****************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------*/

/*****************************************************************
| function prototype (scope: module-local)
|----------------------------------------------------------------*/

/*****************************************************************
| function implementation (scope: module-local)
|----------------------------------------------------------------*/

/*****************************************************************************
 *
 * FUNCTION:	     BPCL_Free_Memory 
 *
 * DESCRIPTION:	Free memory before returning error.
 *                Error is returned because this function is called 
 *                to free memory only when an error occurs in BPCL.
 *
 * PARAMETERS:
 *  fmt    - format specifier for variable number of arguments
 *
 *  (...)  - variable number of arguments specifying addresses 
 *           of memory to be freed
 *
 * RETURNVALUE:
 *	   -1   - return error
 *			Error is returned because this function is called 
 *           to free memory only when an error occurs in BPCL.
 *
 * NOTE:
 * Format specifiers and their respective arguments should be used as follows:
 *
 * For freeing field_element:
 * BPCL_Free_Memory( "%f", <address of field_element> );
 *
 * For freeing point:
 * BPCL_Free_Memory( "%p", <address of point> );
 *
 * For freeing memory allocated to M_MEM_ALLOC:
 * BPCL_Free_Memory( "%m", <address to be freed> );
 *
 * For freeing memory allocated to array points:
 * BPCL_Free_Memory( "%<n>a", 
 *                        <address of the last index of array to be freed> );
 * <n> -Number of indexes in the array to be freed
 *
 * HISTORY:
 *   DATE      |  AUTHOR  |         MODIFICATION
 *	 05/01/09   |  Divya H |	Initial version
 *
 ****************************************************************************/
tS8 BPCL_Free_Memory(tString pcFmt, ...)
{
	/* points to each unnamed argument in turn */
	va_list paArgPtr;
	/* points to the format string */
	tString pcFmtPtr = NULL;
	/* address passed as argument for processing freeing of memory */
	uintptr_t uiAddress=0;
	tU8 u8Loop=0;
	tU32 u32Index=0;
	tChar cBuf[4]={'\0'};

	/* No memory to free. Just return error. Error is returned because this 
	function is called to free memory only when an error occurs in BPCL */ 
	if(pcFmt==NULL)
	{
	     return -1;
	}

	/* make paArgPtr point to 1st unnamed arg */
	va_start(paArgPtr, pcFmt); 

	/* process format string pcFmt and act accordingly */
	for (pcFmtPtr = pcFmt; *pcFmtPtr; pcFmtPtr++) 
	{
		switch (*++pcFmtPtr) 
		{
		    /* field element type to be freed */
			case 'f':
			uiAddress = va_arg(paArgPtr, tU32);
			field_element_free((tsECFieldElement*)uiAddress);
			break;

			/* point type to be freed */
			case 'p':
			uiAddress = va_arg(paArgPtr, tU32);
			point_free((tsECPoint*)uiAddress);
			break;

			/* memory to be freed */
			case 'm':
			uiAddress = va_arg(paArgPtr, tU32);
			M_MEM_FREE((void*)uiAddress);
			break;
			
			/* array points to be freed */
			/* fall through intended */
             case '0': case '1': case '2': case '3': case '4': 
             case '5': case '6': case '7': case '8': case '9': 
             for(u8Loop=0; ((u8Loop < 4) && ((*pcFmtPtr)!='a'));
                          u8Loop++,pcFmtPtr++)
			{
			 cBuf[u8Loop]=*pcFmtPtr;
			}    
			u32Index=atoi(cBuf);
			memset(cBuf,0,sizeof(cBuf));
	   		uiAddress = va_arg(paArgPtr, tU32);
             for (u8Loop=0; u8Loop<u32Index; u8Loop++)
             {
              point_free((tsECPoint*) (uiAddress-(u8Loop*sizeof(tsECPoint))));
			}
			break;

			default:			
			break;
		}
	}

   /* clean up when done */
   va_end(paArgPtr);
    
	/* return error after freeing memory. Error is returned because this 
	function is called to free memory only when an error occurs in BPCL */
	return -1;
}

