//-----------------------------------------------------------------------------
//
//  BPCL.H
//
//-----------------------------------------------------------------------------
//
//  Definitions and Interfaces for Blaupunkt Cryptography Library
//
//  Copyright (c) 2006 Blaupunkt GmbH, Hildesheim
//
//  Author: Holger Listle, CM-DI/ESN
//
//-----------------------------------------------------------------------------

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#ifdef __cplusplus
extern "C" {
#endif

#define C_PLATFORM_ARM 1
#define C_PLATFORM_WIN 0
#define C_PLATFORM_SUN 0
#define C_PLATFORM_T_ENGINE 0

// target endian (little = 0, big = 1)
#define C_BIG_ENDIAN 0

#if (C_PLATFORM_WIN == 1)

  typedef unsigned __int64   tU64;
  typedef unsigned int       tU32;
  typedef unsigned short     tU16;
  typedef unsigned char      tU8;

  typedef signed __int64     tS64;
  typedef signed long        tS32;
  typedef signed short       tS16;
  typedef signed char        tS8;

  typedef char*              tString;

# include <stdlib.h>
# define  M_MEM_ALLOC(n)	malloc(n)
# define  M_MEM_FREE(p)		free(p)
# define  M_MEM_SET(d,c,n)  memset(d,c,n)
# define  M_MEM_COPY(d,s,n) memcpy(d,s,n)
/*
#elif (C_PLATFORM_ARM == 1)

  typedef unsigned long long tU64;
  typedef unsigned int       tU32;
  typedef unsigned short     tU16;
  typedef unsigned char      tU8;

  typedef signed long long   tS64;
  typedef signed long        tS32;
  typedef signed short       tS16;
  typedef signed char        tS8;

  typedef char*              tString;
*/
#elif (C_PLATFORM_SUN == 1)

  typedef unsigned long long tU64;
  typedef unsigned int       tU32;
  typedef unsigned short     tU16;
  typedef unsigned char      tU8;

  typedef signed long long   tS64;
  typedef signed long        tS32;
  typedef signed short       tS16;
  typedef signed char        tS8;

  typedef char*              tString;

# include <stdlib.h>
# define  M_MEM_ALLOC(n)	malloc(n)
# define  M_MEM_FREE(p)		free(p)

#elif (C_PLATFORM_T_ENGINE==1)

  typedef unsigned long long tU64;
  typedef unsigned int       tU32;
  typedef unsigned short     tU16;
  typedef unsigned char      tU8;

  typedef signed long long   tS64;
  typedef signed long        tS32;
  typedef signed short       tS16;
  typedef signed char        tS8;

  typedef char*              tString;

# include <stdlib.h>
# define  M_MEM_ALLOC(n)	Kmalloc(n)
# define  M_MEM_FREE(p)		Kfree(p)

#else

//#error "Platform either not set or not supported!"

#endif

# define  M_MEM_ALLOC(n)	malloc(n)
# define  M_MEM_FREE(p)		free(p)
# define  M_MEM_SET(d,c,n)  memset(d,c,n)
# define  M_MEM_COPY(d,s,n) memcpy(d,s,n)
typedef  tS32  tErrCode;



//-----------------------------------------------------------------------------
// STATUS/ERROR CODES
//-----------------------------------------------------------------------------

#define  BPCL_OK									(0x00)
#define  BPCL_ERR_BAD_PARAMETER						(0x01)
#define  BPCL_ERR_MALLOC							(0x02)
#define  BPCL_ERR_RANDOM_GENERATION					(0x03)

#define  BPCL_ERR_AES_INTERNAL_KEY_NOT_SUPPORTED	(0x10)
#define  BPCL_ERR_AES_INTERNAL_KEY_OUT_OF_RANGE		(0x11)
#define  BPCL_ERR_AES_SECURE_MODE_NOT_SUPPORTED		(0x12)

#define  BPCL_ERR_ECC_KEY_ERROR						(0x40)
#define  BPCL_ERR_ECC_SIG_GENERATION				(0x41)
#define  BPCL_ERR_ECC_SIGNATURE_MISMATCH			(0x42)
#define  BPCL_ERR_ECC_KEY_GENERATION				(0x43)

#define  BPCL_ERR_LIC_BAD_INPUT                     (0x81)

#define  BPCL_ERR_UNSPECIFIED						(0xFF)

//-----------------------------------------------------------------------------
// Key handling (key_util.c)
//-----------------------------------------------------------------------------

tErrCode BPCL_Create_Symmetric_Key(
	const tU8 *p_seed,
	tU32 seed_len,
	tU8 *p_key_buffer,
	tU32 key_len
);

//-----------------------------------------------------------------------------
// AES Encryption/Decryption (aes.c)
//
// Notes:
// - HW AES Acceleration is available on Dragon/Tiger only and only under
//   following conditions: Mode must be ECB, CFB or CTR, key size must be
//   256 bits. HW Acceleration is automatically used if supported.
// - Internal Keys are only provided for HW Accelerated AES
// - Secure Operation is only supported on Dragon HW Accelerated AES
// - Providing data not padded to 128 bit blocks results in additional
//   buffer copy time overhead in modes requiring block data
//-----------------------------------------------------------------------------

#define  BPCL_AES_MODE_ECB		(0x01)	// Electronic Code Book
#define  BPCL_AES_MODE_CBC		(0x02)	// Cipher Block Chaining
#define  BPCL_AES_MODE_CFB		(0x03)	// Cipher FeedBack
#define  BPCL_AES_MODE_OFB		(0x04)	// Output FeedBack
#define  BPCL_AES_MODE_CTR		(0x05)	// CounTeR
#define  BPCL_AES_MODE_CTS		(0x06)	// CounTer-Shift (CANDy-specific)

#define  BPCL_AES_KEYSIZE_128	(0x01)	// 128-bit keys
#define  BPCL_AES_KEYSIZE_192	(0x02)	// 192-bit keys
#define  BPCL_AES_KEYSIZE_256	(0x03)	// 256-bit keys

// Operation control for HW AES
#define  BPCL_AES_OP_ARGKEY		(0x01)	// Key is provided by caller
#define  BPCL_AES_OP_INDKEY		(0x02)	// Use internal indexed key
#define  BPCL_AES_OP_SECKEY		(0x03)	// Encrypted Key is provided by caller

#define  BPCL_AES_FEAT_FIXED_T_BOXES	1	// 1 = ON, 0 = OFF
#define  BPCL_AES_FEAT_DO_MASKING		0   // 1 = ON, 0 = OFF
#define  BPCL_AES_FEAT_NEW_MASK_FREQ	16  // apply every n 16_bytes_block
											// en-/decryption a new mask (1.255)

extern tErrCode
BPCL_AES_Encrypt(
	tU8				mode,		// BPCL_AES_MODE_*
	tU8				key_mode,	// BPCL_AES_OP_*
	tU8				key_size,	// BPCL_AES_KEYSIZE_*
	tU8				*p_key,		// Buffer holding key
	tU8				key_idx,	// Index of internal key to be used
	tU8				*p_iv,		// Buffer holding 128-bit initialization vector
	tU8				*p_plain,	// Buffer holding plain data
	tU8				*p_cipher,	// Buffer to carry encrypted data (providing
								// NULL or p_plain replaces plain data)
	tU32			data_len	// size of p_plain in bytes
);

extern tErrCode
BPCL_AES_Decrypt(
	tU8				mode,		// BPCL_AES_MODE_*
	tU8				key_mode,	// BPCL_AES_OP_*
	tU8				key_size,	// BPCL_AES_KEYSIZE_*
	tU8				*p_key,		// Buffer holding key
	tU8				key_idx,	// Index of internal key to be used
	tU8				*p_iv,		// Buffer holding 128-bit initialization vector
	tU8				*p_cipher,	// Buffer holding encrypted data
	tU8				*p_plain,	// Buffer to carry plain data (providing
								// NULL or p_plain replaces encrypted data)
	tU32			data_len	// size of p_plain in bytes
);

#ifdef _BPCL_TEST
extern tErrCode BPCL_AES_Test(tU8 test_type);
#endif

//-----------------------------------------------------------------------------
// RC4 Stream cipher (rc4.c)
//-----------------------------------------------------------------------------

typedef struct  {      
	tU8		state[256];       
	tU8		x;        
	tU8		y;
} tsRC4Context;

extern void
BPCL_RC4_PrepareKey(
	tU8				*p_key,
	tU32			key_len,
	tsRC4Context	*p_ctx
);

extern void
BPCL_RC4_Crypt(
	tU8				*p_data,
	tU32			data_len,
	tsRC4Context	*p_ctx
);

#ifdef _BPCL_TEST
extern tErrCode BPCL_RC4_Test(tU8 test_type);
#endif

//-----------------------------------------------------------------------------
// RC5 Block cipher (rc5.c)
//-----------------------------------------------------------------------------

extern void
BPCL_RC5_Setup(
	tU8				*p_key		// Buffer holding key (128-bits)
);

extern tErrCode
BPCL_RC5_Encrypt(
	tU32			*p_plain,	// Buffer holding plain data
	tU32			*p_cipher,	// Buffer to carry encrypted data (providing
								// NULL or p_plain replaces plain data)
	tU32			data_len	// size of p_plain in bytes
);

extern tErrCode
BPCL_RC5_Decrypt(
	tU32			*p_cipher,	// Buffer holding encrypted data
	tU32			*p_plain,	// Buffer to carry plain data (providing
								// NULL or p_plain replaces encrypted data)
	tU32			data_len	// size of p_plain in bytes
);

#ifdef _BPCL_TEST
extern tErrCode BPCL_RC5_Test(tU8 test_type);
#endif

//-----------------------------------------------------------------------------
// CRC32 implementation acc. RFC 1662, [Page 20] (CRC32.c)
// Usage:
//	1.			crc = BPCL_CRC32_Update(BPCL_CRC32_INITVALUE, data, len);
//	2..n		crc = BPCL_CRC32_Update(crc, data, len);
//	final		crc ^= BPCL_CRC32_DONEVALUE;
//-----------------------------------------------------------------------------

#define BPCL_CRC32_INITVALUE  0xffffffff
#define BPCL_CRC32_DONEVALUE  0xffffffff

tU32 BPCL_CRC32_Update(
	tU32	old_crc, 
	tU8		*p_data, 
	tU32	data_len
);

#ifdef _BPCL_TEST
extern tErrCode BPCL_CRC32_Test(tU8 test_type);
#endif

//-----------------------------------------------------------------------------
// SHA-1 Secure Hashing (sha1.c)
//-----------------------------------------------------------------------------

typedef struct {
  tU32	total_bytes_Hi; // high word of 64-bit value for bytes count
  tU32	total_bytes_Lo; // low word of 64-bit value for bytes count
  tU32	vector[5];      // 5  32-bit hash words
  tU8	buffer[64];     // 64 byte buffer
} tsSHA1Context;

extern tErrCode
BPCL_SHA1_Init(
	tsSHA1Context		*ctx			// Context to be initialized
);

extern tErrCode
BPCL_SHA1_Update(
	tsSHA1Context		*ctx,			// Initialized Context
    tU8					*p_data,		// data for sha1 update
    tU32				data_len		// length of data in bytes
);

extern tErrCode
BPCL_SHA1_Finish(
	tsSHA1Context		*ctx,			// Initialized Context
	tU32				*p_sha1_hash	// Buffer to carry hash value
);

//-----------------------------------------------------------------------------
// HMAC Hashed Message Authentication Code (sha1.c)
//-----------------------------------------------------------------------------

extern tErrCode
BPCL_HMAC_SHA1(
	tU8					*p_key,			// key array
	tU32				key_len,		// length of key in bytes
	tU8					*p_data,		// data to be hmac'ed
	tU32				data_len,		// length of data in bytes
	tU32				*p_hmac_hash	// Buffer to carry HMAC
);

#ifdef _BPCL_TEST
extern tErrCode BPCL_SHA1_Test(tU8 test_type);
#endif

//-----------------------------------------------------------------------------
// ECC key utilities
//-----------------------------------------------------------------------------

// multiplication does dummy operations to prevent SPA attacks
// (protection against SPA attacks, but longer execution times)
#define  BPCL_ECC_FEAT_SPA_PROTECTION	1 // 1 = ON, 0 = OFF

// all multiplication operations are masked with base point order n to
// prevent DPA attacks => k  P = ( k + x  n )  P
// (protection against DPA attacks, but longer execution times)
#define  BPCL_ECC_FEAT_DPA_PROTECTION	1 // 1 = ON, 0 = OFF

#define  BPCL_ECC_PRIVATE_KEY_SIZE		(20)
#define  BPCL_ECC_PUBLIC_KEY_SIZE		(40)
#define  BPCL_ECC_SIGNATURE_SIZE		(48)

extern tErrCode
BPCL_EC_Create_Key_Pair(
	tU8					*p_seed,		// RNG seed value
	tU32				seed_len,		// seed value length
	tU8					*p_priv_key,	// Buffer to hold private key
	tU8					*p_pub_key		// Buffer to hold public key
);

extern tErrCode
BPCL_EC_Compute_Public_Key(
	tU8					*p_priv_key,	// Input private key
	tU8					*p_pub_key		// Buffer to hold public key
);

//-----------------------------------------------------------------------------
// ECC based Diffie-Hellman (ECDH)
//-----------------------------------------------------------------------------

extern tErrCode
BPCL_EC_DH(
	tU8					*p_my_priv_key,	// My private key
	tU8					*p_oth_pub_key,	// Partner's public key
	tU8					*p_shared_key,	// Buffer to hold resulting shared key
	tU32				shared_key_len	// required key length (max 20 bytes)
);

//-----------------------------------------------------------------------------
// ECC based Signatures (ECDSA)
//-----------------------------------------------------------------------------

extern tErrCode
BPCL_EC_DSA_Create(
	tU8					*p_msg_hash,	// Hash of msg
	tU8					*p_priv_key,	// Signer's private key
	tU8					*p_seed,		// Signature seed value
	tU32				seed_len,		// length of seed value
	tU8					*p_signature	// Buffer to hold signature
);

extern tErrCode
BPCL_EC_DSA_Verify(
	tU8					*p_msg_hash,	// Hash of msg
	tU8					*p_public_key,	// Signer's public key
	tU8					*p_signature	// Input reference signature
);

#ifdef _BPCL_TEST
extern tErrCode BPCL_ECC_Test(tU8 test_type);
#endif

//-----------------------------------------------------------------------------
// Random generation functions (random_gen.c)
//-----------------------------------------------------------------------------

extern int	rnd_seed( tU32 seed1, tU32 seed2 );

extern tU32	rnd_word( tU32 max );

extern int	rnd_byte_array(
	const tU8 *K0_r0,
	tU8 *r,
	tU32 r_length
);

#ifdef __cplusplus
}
#endif
