//-----------------------------------------------------------------------------
//
//  RC4.C
//
//-----------------------------------------------------------------------------
//
//  Implementation of Streamcipher RC4
//
//  Copyright (c) 2006 Blaupunkt GmbH, Hildesheim
//
//  Author: Holger Listle, CM-DI/ESN
//
//-----------------------------------------------------------------------------

#include "bpcl.h"
#include "bpcl_int.h"

tU8 global_key_mask2[] = { 0x71, 0xec, 0xe2, 0x7c, 0x29, 0xe5, 0xda, 0x32, 0x73, 0x04, };
tU8 _tmpU8;
int _mi;

void BPCL_RC4_PrepareKey(
	tU8				*p_key,
	tU32			key_len,
	tsRC4Context	*p_ctx
) {
	tU8		idx1, idx2;
	tU8		*p_state;
	tU32	i;

	p_state = &p_ctx->state[0];
	for(i = 0; i < 256; i++) {
		p_state[i] = (tU8)i;
	}
	p_ctx->x = p_ctx->y = 0;
	idx1     = idx2     = 0;

	for(i = 0; i < 256; i++) {
		idx2 = (p_key[idx1] + p_state[i] + idx2) % 256;
		M_SWAPBYTE(&p_state[i], &p_state[idx2]);
		idx1 = (idx1 + 1) % (tU8)key_len;
	}
} /* BPCL_RC4_PrepareKey() */



void BPCL_RC4_Crypt(
	tU8				*p_data,
	tU32			data_len,
	tsRC4Context	*p_ctx
) {
	tU8		x, y, xorIndex;
	tU8		*p_state;
	tU32	i;

	x = p_ctx->x;
	y = p_ctx->y;
	p_state = &p_ctx->state[0];

	for(i = 0; i < data_len; i++)
	{
		x = (x + 1) % 256;
		y = (p_state[x] + y) % 256;
		M_SWAPBYTE(&p_state[x], &p_state[y]);
		xorIndex = (p_state[x] + p_state[y]) % 256;
		p_data[i] ^= p_state[xorIndex];
	}

	p_ctx->x = x;
	p_ctx->y = y;
} /* BPCL_RC4_Crypt() */

#ifdef _BPCL_TEST

#include <memory.h>	// memcpy(), memcmp()

//-----------------------------------------------------------------------------
//	Test implementation
//-----------------------------------------------------------------------------

tS32 BPCL_RC4_Test(tU8 test_type) {
	
	tU32			i;
	tU8				src[8];
	tU8				buffer[16384] = { 0L };	/* 16kB for performance tests */
	tsRC4Context	ctx;
	tErrCode		rc = BPCL_OK;


	struct {
		tU8  key[8];
		tU8  plain[8];
		tU8  cipher[8];
	} test_data[3] = {
		{
			{ 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
			{ 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
			{ 0x75, 0xb7, 0x87, 0x80, 0x99, 0xe0, 0xc5, 0x96 },
		}, {
			{ 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
			{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
			{ 0x74, 0x94, 0xc2, 0xe7, 0x10, 0x4b, 0x08, 0x79 },
		}, {
			{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
			{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
			{ 0xde, 0x18, 0x89, 0x41, 0xa3, 0x37, 0x5d, 0x3a },
		}
	};
	
	printf("\n----------------------------------------------------------\n");
	printf("             RC4 Test\n");
	printf("----------------------------------------------------------\n");

	if(test_type & BPCL_TEST_COMPLIANCE) {
		printf("\nRC4 Compliance Test\n");
		for(i=0; i<3; ++i) {
			memcpy(src,test_data[i].plain,8);
			BPCL_RC4_PrepareKey(test_data[i].key, 8, &ctx);
			BPCL_RC4_Crypt(src, 8, &ctx);
			if( memcmp(src, test_data[i].cipher, 8) != 0 ) {
				printf("RC4 Compliance Test #%d failed:\n", i);
				rc = BPCL_ERR_TEST_FAILED;
			} else {
				printf("RC4 Compliance Test #%d okay\n", i);
			}
		}
		printf("\nRC4 Compliance Test DONE\n");
	}
	
	if(test_type & BPCL_TEST_PERFORMANCE) {

		printf("\nRC4 Performance Test\n");

		BPCL_RC4_PrepareKey(test_data[i].key, 8, &ctx);

		BPCL_TimerStart();
		for(i = 0; i < 640; ++i) {
			/* Loop over 10MB of data (640 * 16kB) */
			BPCL_RC4_Crypt(buffer, 16384, &ctx);
		}
		BPCL_TimerReport("RC4 Encryption@10MB");

		printf("\nRC4 Performance Test DONE\n");
	}
	return rc;
} /* BPCL_RC4_Test() */

#endif /* _BPCL_TEST */
