/******************************************************************************
 * FILE: profilecontrolhelper.h         
 * SW-COMPONENT: ProfileControlHelper
 * DESCRIPTION:  Interface Definition for ProfileControlHelper
 * AUTHOR:       TMS GmbH Roland Jentsch
 * COPYRIGHT:     (c) 2008 Robert Bosch GmbH, Hildesheim
 * HISTORY:      
 ******************************************************************************/

 /******************************************************************************
| includes of component-internal interface, if necessary
| (scope: component-local)
|-----------------------------------------------------------------------------*/

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

// include private definitions (internal settings and so on)
#include "profilecontrolhelper_internal.h"

    // include interface file 
#include "profilecontrolhelper.h"

// include trace functionality
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"


/******************************************************************************
| defines and macros (scope: global)
|-----------------------------------------------------------------------------*/
// see defines in the profilecontrolhelper_internal.h file

/******************************************************************************
| typedefs (scope: global)
|-----------------------------------------------------------------------------*/
// no typedefs

/*******************************************************************************
| variable declaration (scope: global)
|-----------------------------------------------------------------------------*/
// no global variables

/*******************************************************************************
| function prototypes (scope: local)
|-----------------------------------------------------------------------------*/

/* 
    
        all of the internal functions got the prefix pchl -> ProfileControlHelperLibrary

*/


/*******************************************************************************
 *FUNCTION:   pchl_szGetPath
 *DESCRIPTION:
 *   This local function returns the path of a given filename
 *
 *PARAMETER:   
 *  tString szFilePath -> output: pointer to output variable
 * tU32 nFilePathMaxLength -> input: length of output variable
 * tCString szFilePathName -> input:path and filename 
 * 
 *RETURNVALUE:
 * pointer to return path, same as szFilePath
 *
 *HISTORY:
 ******************************************************************************/

static tCString pchl_szGetPath(tString szFilePath, tU32 nFilePathMaxLength, tCString szFilePathName)
{
   tString szFileName = (tString) OSAL_ps8StringRSearchChar(szFilePathName, '/');
   if (szFileName == OSAL_NULL)
      szFileName = (tString) OSAL_ps8StringRSearchChar(szFilePathName, '\\');
   if (szFileName != OSAL_NULL)
   {
        tU32 nPathLength = (tU32)(szFileName - szFilePathName + 1); // + 1 because of including the "/" character
        (tVoid) OSAL_szStringNCopy(szFilePath, szFilePathName, nFilePathMaxLength);
        
        if (nPathLength >= nFilePathMaxLength)
            szFilePath[nFilePathMaxLength - 1] = 0;
        else
            szFilePath[nPathLength] = 0;
   }
   else
   {
        szFilePath[0] = 0;
   }
   return (tCString)(szFilePath);
}

/*******************************************************************************
 *FUNCTION:   pchl_szGetFileName
 *DESCRIPTION:
 *   This local function returns a pointer to the name of a given path/filename
 *
 *PARAMETER:   
 * tCString szFilePathName -> input:path and filename 
 * 
 *RETURNVALUE:
 * pointer to return filename including extension(inside input variable)
 *
 *HISTORY:
 ******************************************************************************/

static tCString pchl_szGetFileName(tCString szFilePathName)
{
   tString szFileName; 

   if (szFilePathName == OSAL_NULL)
   {
       ET_TRACE_ERROR(TR_COMP_SYSTEM, "pchl_szGetFileName: Parameter NULL ");
       return "";
   }
   
   szFileName = (tString) OSAL_ps8StringRSearchChar(szFilePathName, '/');
   if (szFileName == OSAL_NULL)
      szFileName = (tString) OSAL_ps8StringRSearchChar(szFilePathName, '\\');
   return (tCString)((szFileName != OSAL_NULL) ? (szFileName + 1) : (szFilePathName));
}

/*******************************************************************************
 *FUNCTION:   pchl_szGetFileExt
 *DESCRIPTION:
 *   This local function returns a pointer to the extension of a given path/filename
 *
 *PARAMETER:   
 * tCString szFilePathName -> input:path and filename 
 * 
 *RETURNVALUE:
 * pointer to return Extension (inside input variable)
 *
 *HISTORY:
 ******************************************************************************/

static tCString pchl_szGetFileExt(tCString szFilePathName)
{
   tCString szFileName;
   tString szFileExt;
   
   if (szFilePathName == OSAL_NULL)
   {    
       ET_TRACE_ERROR(TR_COMP_SYSTEM, "pchl_szGetFileExt: Parameter NULL ");
       return "";
   }
   
   szFileName = pchl_szGetFileName(szFilePathName);
   szFileExt = (tString) OSAL_ps8StringRSearchChar(szFileName, '.');
   return (tCString)((szFileExt != OSAL_NULL) ? (szFileExt + 1) : "");
}

/*******************************************************************************
 *FUNCTION:   pchl_szGetFileNameOnly
 *DESCRIPTION:
 *   This local function returns the name (without extension)  of a given file/pathname
 *
 *PARAMETER:   
 *  tString szFilePath -> output: pointer to output variable
 * tU32 nFilePathMaxLength -> input: length of output variable
 * tCString szFilePathName -> input:path and filename 
 * 
 *RETURNVALUE:
 * pointer to return path, same as szFileName
 *
 *HISTORY:
 ******************************************************************************/
 
static tCString pchl_szGetFileNameOnly(tString szFileName, tU32 nFileNameMaxLength, tCString szFilePathName)
{
    tCString szFileExt;
    tU32 nExtLength;
    
    if (szFilePathName == OSAL_NULL)
    {
       ET_TRACE_ERROR(TR_COMP_SYSTEM, "pchl_szGetFileNameOnly: Parameter szFilePathName NULL ");
       return szFileName;
    }

    if (szFileName == OSAL_NULL)
    {
       ET_TRACE_ERROR(TR_COMP_SYSTEM, "pchl_szGetFileNameOnly: Parameter szFileName NULL ");
       return szFileName;
    }
    
    szFileExt  = pchl_szGetFileExt(szFilePathName);
    nExtLength = OSAL_u32StringLength(szFileExt);
    if (nExtLength == 0)
    {
        tU32 nNameLength;
        tCString szFileNameWithExt  = pchl_szGetFileName(szFilePathName);
        (tVoid)OSAL_szStringNCopy(szFileName, szFileNameWithExt, nFileNameMaxLength);
        szFileName[nFileNameMaxLength - 1] = 0;

        // test for name ends with "."
        nNameLength = OSAL_u32StringLength(szFileName);
        if (nNameLength > 0)
        {
            if (szFileName[nNameLength - 1] == '.')
            {
                szFileName[nNameLength - 1] = 0;
            }
        }
    }
    else
    {
        tU32 nLength;
        tCString szFileNameWithExt  = pchl_szGetFileName(szFilePathName);
        (tVoid)OSAL_szStringNCopy(szFileName, szFileNameWithExt, nFileNameMaxLength);
        nLength = OSAL_u32StringLength(szFileNameWithExt) - nExtLength - 1; // -1 because of '.'
        if (nLength >= nFileNameMaxLength)
        {
            nLength = nFileNameMaxLength - 1;
        }
        szFileName[nLength] = 0;
    }
    return szFileName;
}

/*******************************************************************************
 *FUNCTION:   pchl_nGetRegistryValue
 *DESCRIPTION:
 *   This local function returns the name (without extension)  of a given file/pathname
 *
 *PARAMETER:   
 *  tString szFilePath -> output: pointer to output variable
 * tU32 nFilePathMaxLength -> input: length of output variable
 * tCString szFilePathName -> input:path and filename 
 * 
 *RETURNVALUE:
 * pointer to return path, same as szFileName
 *
 *HISTORY:
 ******************************************************************************/

static tS32 pchl_nGetRegistryValue(tCString szRegPath, tCString szRegKey)
{
    OSAL_tIODescriptor RegFd; 
    tChar szRegDirName[OSAL_C_U32_MAX_PATHLENGTH];
    
    if (szRegPath == OSAL_NULL)
    {
       ET_TRACE_ERROR(TR_COMP_SYSTEM, "pchl_nGetRegistryValue: Parameter szRegPath NULL ");
       return OSAL_ERROR;
    }

    if (szRegKey == OSAL_NULL)
    {
       ET_TRACE_ERROR(TR_COMP_SYSTEM, "pchl_nGetRegistryValue: Parameter szRegKey NULL ");
       return OSAL_ERROR;
    }
    
    OSAL_s32PrintFormat((tString)szRegDirName, 
                          "%s/%s", 
                          OSAL_C_STRING_DEVICE_REGISTRY, 
                          szRegPath);
      
    RegFd = OSAL_IOOpen (szRegDirName, (OSAL_tenAccess)OSAL_EN_READONLY );
    if( RegFd != OSAL_ERROR )
    {     
        OSAL_trIOCtrlRegistryValue regValue;
        (tVoid)OSAL_szStringNCopy(regValue.s8Name, szRegKey, OSAL_C_U32_MAX_PATHLENGTH);
        if (OSAL_OK == OSAL_s32IOControl(RegFd, OSAL_C_S32_IOCTRL_REGREADVALUE, (intptr_t) &regValue))
        {
            if (regValue.s32Type == OSAL_C_S32_VALUE_S32)
            {
                  tS32 *s32RegValAdr = (tS32*) (&regValue.s8Value);
                  return *s32RegValAdr;
            }
            else
            {
                ET_TRACE_ERROR(TR_COMP_SYSTEM, "pchl_nGetRegistryValue: wrong registry value type ");
            }
        }
        else
        {
            ET_TRACE_ERROR(TR_COMP_SYSTEM, "pchl_nGetRegistryValue: registry value does not exist ");
        }
    }
    else
    {
        ET_TRACE_ERROR(TR_COMP_SYSTEM, "pchl_nGetRegistryValue: registry key could not be opened ");
    }

    return OSAL_ERROR;
}

/*******************************************************************************
| function prototypes (scope: global)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
 *FUNCTION:   pchl_fGetProfilePathName
 *DESCRIPTION:
 *   This global function returns the profile variant of the file or pathname of a given file/pathname
 *
 *PARAMETER:   
 *  tString szProfileFilePathName -> output: pointer to output variable
 * tU32 nProfileFilePathNameMaxLength -> input: length of output variable
 * tCString szFilePathName -> input:path and filename 
 * tU32 nProfileNumber -> input: the number if the profile
 * 
 *RETURNVALUE:
 * tBool boolean value if the translation to the profile variant is successful
 *
 *HISTORY:
 ******************************************************************************/

tBool pchl_fGetProfilePathName(tString szProfileFilePathName, tU32 nProfileFilePathNameMaxLength, tCString szFilePathName, tU32 nProfileNumber)
{   
    tChar szPathName[OSAL_C_U32_MAX_PATHLENGTH] = "";
    tChar szFileName[OSAL_C_U32_MAX_PATHLENGTH] = "";
    
    tCString szExt = pchl_szGetFileExt(szFilePathName);
    
    pchl_szGetPath(szPathName, nProfileFilePathNameMaxLength, szFilePathName);
    pchl_szGetFileNameOnly(szFileName, nProfileFilePathNameMaxLength, szFilePathName);
    
    ET_TRACE_INFO(TR_COMP_SYSTEM, (tChar*)"\n");
    ET_TRACE_INFO(TR_COMP_SYSTEM, (tChar*)szExt);
    ET_TRACE_INFO(TR_COMP_SYSTEM, szPathName);
    ET_TRACE_INFO(TR_COMP_SYSTEM, szFileName);
    
    if (OSALUTIL_s32SaveNPrintFormat(szProfileFilePathName, nProfileFilePathNameMaxLength, "%s%s_d%d.%s", 
                                        szPathName, szFileName, nProfileNumber, szExt) != OSAL_ERROR)
    {
        ET_TRACE_INFO(TR_COMP_SYSTEM, szProfileFilePathName);
        ET_TRACE_INFO(TR_COMP_SYSTEM, "\n");
        return TRUE;
    }
    else
    {
        ET_TRACE_ERROR(TR_COMP_SYSTEM, "pchl_szGetProfilePathName: could not create pathname ");
        return FALSE;
    }
}

/*******************************************************************************
 *FUNCTION:   pchl_nGetProfileNumber
 *DESCRIPTION:
 *   This global function returns the number of the current profile of the given type
 *
 *PARAMETER:   
 * tenProfileType eProfile -> input: typ of the profile (could be eProfileWriteOld or eProfileReadNew)
 * 
 *RETURNVALUE:
 * tS32 returns the number of the current profile of the given type
 *
 *HISTORY:
 ******************************************************************************/

tS32 pchl_nGetProfileNumber(tenProfileType eProfile)
{
    switch (eProfile)
    {
        case eProfileWriteOld: return pchl_nGetRegistryValue(PROFILE_CONTROL_HELPER_REGISTRY_PATH, PROFILE_CONTROL_HELPER_OLD_REGISTRY_KEY);
        case eProfileReadNew: return pchl_nGetRegistryValue(PROFILE_CONTROL_HELPER_REGISTRY_PATH, PROFILE_CONTROL_HELPER_NEW_REGISTRY_KEY);
        default: return OSAL_ERROR;
    }
}

/*******************************************************************************
 *FUNCTION:   pchl_fGetProfileFilePathName
 *DESCRIPTION:
 *   This global function returns the profile filepathname of a given file/pathname for the given profile
 *
 *PARAMETER:   
 *  tString szProfileFilePathName -> output: pointer to output variable
 * tU32 nProfileFilePathNameMaxLength -> input: length of output variable
 * tCString szFilePathName -> input:path and filename 
 * tenProfileType eProfile -> input: typ of the profile (could be eProfileWriteOld or eProfileReadNew)
 * 
 *RETURNVALUE:
 * tBool boolean value if the translation to the profile filepathname variant is successful
 *
 *HISTORY:
 ******************************************************************************/

tBool pchl_fGetProfileFilePathName(tString szFilePath, tU32 nFilePathMaxLength, tCString szFilePathName, tenProfileType eProfile)
{
    tS32 nProfileId = pchl_nGetProfileNumber(eProfile);
    if (nProfileId != OSAL_ERROR)
    {
        return pchl_fGetProfilePathName(szFilePath, nFilePathMaxLength, szFilePathName, nProfileId);
    }
    else
    {
        ET_TRACE_ERROR(TR_COMP_SYSTEM, "pchl_szGetProfileFilePathName: using default key ");
        return FALSE;
    }
}

/*******************************************************************************
 *FUNCTION:   OSAL_IOOpenProfileFileEx
 *DESCRIPTION:
 *   This global function returns the osal file handle to a given file/pathname for the given profile
 *    This function is a replavement of the normal OSAL_IOOpen for such files 
 *
 *PARAMETER:   
 * tCString coszName -> input: origin path and filename (without converted to profile)
 * OSAL_tenAccess enAccess -> input: Acces options to the file
 * tenProfileType eProfile -> input: typ of the profile (could be eProfileWriteOld or eProfileReadNew)
 * tBool fUseDefaultFileIfNotExist -> input: determine the handling of not existing files (automatically use the non converted /inputfilename if existing)
 * 
 *RETURNVALUE:
 * the osal file handle to a given file/pathname for the given profile  or OSAL_ERROR in case of error
 *
 *HISTORY:
 ******************************************************************************/

OSAL_tIODescriptor OSAL_IOOpenProfileFileEx(tCString coszName, tenProfileType eProfile, tBool fUseDefaultFileIfNotExist)
{
    OSAL_tIODescriptor fd = OSAL_ERROR;
    OSAL_tenAccess enAccess;
    tBool ret;
    tChar szFileName[OSAL_C_U32_MAX_PATHLENGTH] = "";
    
    #ifdef PROFILE_CONTROL_HELPER_USED_METHOD_GM
    /* now at first a special handling in GM for the "other" id */
    tS32 nOldProfileId = pchl_nGetProfileNumber(eProfileWriteOld);
    tS32 nNewProfileId = pchl_nGetProfileNumber(eProfileReadNew);
    if ((nNewProfileId == PROFILECTRL_PROFILE_OTHER) && (nOldProfileId != nNewProfileId) && (nNewProfileId != OSAL_ERROR))
    {
        /* now special handling for read data in case switch from any normal id to other id, we read the the priveous one.  
                          The calling software can handle this also in that way, that the software doenst read the settings and hold the old one in memory  */
        // test on plausilility
        if (eProfile == eProfileReadNew)
        {
            ET_TRACE_ERROR(TR_COMP_SYSTEM, "OSAL_IOOpenProfileFileEx: Why should the old other profile readed ? ");
        }
        eProfile = eProfileWriteOld;
    }
    #endif
    
    // automatic detect of access mode
    if (eProfileReadNew == eProfile)
    {
        enAccess = OSAL_EN_READONLY;
    }
    else
    {
        enAccess = OSAL_EN_READWRITE;
    }

    // get file name    
    ret = pchl_fGetProfileFilePathName(szFileName, OSAL_C_U32_MAX_PATHLENGTH, coszName, eProfile);
    if (ret)
    {
        fd = OSAL_IOOpen(szFileName, enAccess);
        ret = (OSAL_ERROR != fd);
        
    }
    
    // use alternative filename, if needed 
    if (!ret && fUseDefaultFileIfNotExist)
    {
        if (eProfileReadNew == eProfile) 
        {
            fd = OSAL_IOOpen(coszName, enAccess);
            ret = (OSAL_ERROR != fd);
        }
    }

    return fd;
}


/*******************************************************************************
 *FUNCTION:   OSAL_IOOpenProfileFile
 *DESCRIPTION:
 *   This global function returns the osal file handle to a given file/pathname for the given profile
 *    This function is a replavement of the normal OSAL_IOOpen for such files 
 *
 *PARAMETER:   
 * tCString coszName -> input: origin path and filename (without converted to profile)
 * OSAL_tenAccess enAccess -> input: Acces options to the file
 * tenProfileType eProfile -> input: typ of the profile (could be eProfileWriteOld or eProfileReadNew)
 * 
 *RETURNVALUE:
 * the osal file handle to a given file/pathname for the given profile  or OSAL_ERROR in case of error
 *
 *HISTORY:
 ******************************************************************************/

OSAL_tIODescriptor OSAL_IOOpenProfileFile(tCString coszName, tenProfileType eProfile)
{
   return OSAL_IOOpenProfileFileEx(coszName, eProfile, PROFILE_CONTROL_HELPER_DEFAULT_HANDLING);
}

/*******************************************************************************
 *FUNCTION:   OSAL_IOCreateProfileFile
 *DESCRIPTION:
 *   This global function creates a new file and returns the osal file handle to the file/pathname for the given profile
 *    This function is a replavement of the normal OSAL_IOCreate for such files 
 *
 *PARAMETER:   
 * tCString coszName -> input: origin path and filename (without converted to profile)
 * OSAL_tenAccess enAccess -> input: Acces options to the file
 * tenProfileType eProfile -> input: typ of the profile (could be eProfileWriteOld or eProfileReadNew)
 * 
 *RETURNVALUE:
 * the osal file handle to a given file/pathname for the given profile or OSAL_ERROR in case of error
 *
 *HISTORY:
 ******************************************************************************/

OSAL_tIODescriptor OSAL_IOCreateProfileFile(tCString coszName, tenProfileType eProfile)
{
    tChar szFileName[OSAL_C_U32_MAX_PATHLENGTH] = "";
    tBool ret = pchl_fGetProfileFilePathName(szFileName, OSAL_C_U32_MAX_PATHLENGTH, coszName, eProfile);
    if (ret)
    {
        OSAL_tenAccess enAccess;
        
        /* automatic detect of access mode */
        if (eProfileReadNew == eProfile)
        {
            enAccess = OSAL_EN_READONLY;
        }
        else
        {
            enAccess = OSAL_EN_READWRITE;
        }
    
        return OSAL_IOCreate(szFileName, enAccess);
    }
    else
    {
        return OSAL_ERROR;
    }
}

/*******************************************************************************
 *FUNCTION:   OSAL_s32IORemoveProfileFile
 *DESCRIPTION:
 *   This global function removes the file/pathname for the given profile
 *   This function is a replavement of the normal OSAL_IORemove for such files 
 *
 *PARAMETER:   
 * tCString coszName -> input: origin path and filename (without converted to profile)
 * tenProfileType eProfile -> input: typ of the profile (could be eProfileWriteOld or eProfileReadNew)
 * 
 *RETURNVALUE:
 * OSAL_OK or OSAL_ERROR in case of error
 *
 *HISTORY:
 ******************************************************************************/

tS32 OSAL_s32IORemoveProfileFile(tCString coszName, tenProfileType eProfile)
{
    tChar szFileName[OSAL_C_U32_MAX_PATHLENGTH] = "";
    tBool ret = pchl_fGetProfileFilePathName(szFileName, OSAL_C_U32_MAX_PATHLENGTH, coszName, eProfile);
    if (ret)
    {
        return OSAL_s32IORemove(szFileName);
    }
    else
    {
        return OSAL_ERROR;
    }
}
 
/*  Middle Level functions */
 
OSAL_tIODescriptor OSAL_IOCreateProfileFileById(tCString coszName, OSAL_tenAccess enAccess, tU32 nProfileId)
{
    tChar szFileName[OSAL_C_U32_MAX_PATHLENGTH] = "";
    if (pchl_fGetProfilePathName(szFileName, OSAL_C_U32_MAX_PATHLENGTH, coszName, nProfileId) == TRUE)
    {
        return OSAL_IOCreate(szFileName, enAccess);
    }
    else
    {
        return OSAL_ERROR;
    }
}

OSAL_tIODescriptor OSAL_IOOpenProfileFileById(tCString coszName, OSAL_tenAccess enAccess, tU32 nProfileId)
{
    tChar szFileName[OSAL_C_U32_MAX_PATHLENGTH] = "";
    if (pchl_fGetProfilePathName(szFileName, OSAL_C_U32_MAX_PATHLENGTH, coszName, nProfileId) == TRUE)
    {
        return OSAL_IOOpen(szFileName, enAccess);
    }
    else
    {
        return OSAL_ERROR;
    }
}

tS32 OSAL_s32IORemoveProfileFileById(tCString coszName, tU32 nProfileId)
{
    tChar szFileName[OSAL_C_U32_MAX_PATHLENGTH] = "";
    if (pchl_fGetProfilePathName(szFileName, OSAL_C_U32_MAX_PATHLENGTH, coszName, nProfileId) == TRUE)
    {
        return OSAL_s32IORemove(szFileName);
    }
    else
    {
        return OSAL_ERROR;
    }
}

