/************************************************************************
* FILE:         ahl_tclMessageThread.h
* PROJECT:      VWLLNF
* SW-COMPONENT: (framework)
*----------------------------------------------------------------------
*
* DESCRIPTION:  base workthread's with Message communication
*              
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2009 Robert Bosch GmbH, Hildesheim
* HISTORY:      
* Date      | Author   | Modification
* 30.07.08  | Hessling | added Thread with Message comunication to AHL                     
* 16.02.09  | Hessling | suspend and resume added, more documentation done                     
* 01.03.09  | Hessling | seperation in MessageThread and EventThread
*************************************************************************/
#ifndef AHL_TCLMESSAGETHREAD_H
#define AHL_TCLMESSAGETHREAD_H


#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#include <queue>


#include "ahl_tclBaseThread.h"

/**
* class ahl_tclMessageThread is an minimal task having an Message inbox to communicate
* with the thread function.
* By inheriting the ahl_tclMessageThread the implementer defines:
* - Messages on which the thread shall react
* - parameters to be passed into the thread as membervariables of the thread
*   (don't forget semaphores if the varables are also used in other instances)
* - the bThreadFunction() which is called when an Message occurs and just reacts 
*   on Messages coded in the message queue. The function will be called
*   whenever an Message has been posted by vPostMessage().
* As typical for AHL the implementation is good practice and most commonly 
* used functionality. (about 64 times in di_middleware_server).
* It is the more simple and ressource saving as the ahl_BaseWorkThread becasue
* it does not use CCA message box.
* Design:
* tclThread care about thread handling functions and makes sure that the developper 
* has as little as possible to do with thread functions of OSAL. The thread is by
* default reached with Message handling, wherein the developper can self define the 
* complete range of Messages. In parallel tclThread cares about controlling the thread 
* so that starting and stopping does avoid typical implementation errors like shutting
* down during still open handles, or freeing memory which is still in use by the thread.
* 
* @usage: "How to use this framework"
* 1. inherit the class and write an enum list of envents you want to handle in
*    bThreadFunction(). Put all data the thread will need as protected (or private)
*    Membervariables in your inheriting class. (e.g. a queue of values received from 
*    an ioPort being observed etc.)
* 2. According to typical class implementation put in AIL::bOnInit() the bStart() 
*    function and in AIL::vOnApplicationClose() bStop(). If you additionally want to 
*    save resources during OFF the entry may vSuspend() the thread and leaving may
*    vResume().
* 3. If you have to handle more than one thread you may want to shutdown them in 
*    parallel. For this case shutdown can be nonblocking initiated with vPostStopRequest()
*    you should afterwards wait for finla closure with blocking bStop() to make sure all
*    threads have really stopped.
*/
template<class MT, tU32 AHL_MAX_QUEUE_SIZE=OSAL_C_U32_MAX>
class ahl_tclMessageThread : public ahl_tclBaseThread 
{
public:
   /**
   * a message will be copied into the MessageQueue and further this copy will be 
   * provided to the thread function for processing. The message copy will be 
   * destroyed, after the thread function has returned.
   */
   typedef MT tclMessage;
   typedef std::queue<tclMessage> tMsgQueue;

   /** CTOR creates the thread attributes. normally all threads should be defined via
   * the internal registry. For this case AppId and threadName should identify the 
   * thread. In cases where this data is not available you can also control the stack
   * size and priority with this constuctor. In case that no thread name is given, the 
   * this pointer address will be used to generate a unique string.
   * @param u16AppId your application id helping to find the entries in registry
   * @param tCString a string of max 31 chars defining the name also in registry
   *        if name is not given (0 or "") the constructor will create an individual   
   *        from the this pointer using usign the default parameters only.
   * @param u32DefaultStackSize if value cannot be retrieved from reg
   * @param u32DefaultPrio if value cannot be retrieved from reg
   */
   ahl_tclMessageThread(tU16 u16AppId=((tU16)-1), 
      tCString szThreadName=0, 
      tS32 s32DefaultStackSize=1024, 
      tU32 u32DefaultPrio=OSAL_C_U32_THREAD_PRIORITY_LOWEST) 
   : ahl_tclBaseThread(u16AppId, szThreadName, s32DefaultStackSize, u32DefaultPrio),
     _hMsgBoxSem(OSAL_C_INVALID_HANDLE),
     _hMsgQueueSem(OSAL_C_INVALID_HANDLE),
     _bMsgQueueOverflowed(FALSE)
   {
      tChar as8Name[OSAL_C_U32_MAX_NAMELENGTH];
      (tVoid)OSAL_szStringNCopy(as8Name,_as8ThreadName, OSAL_C_U32_MAX_NAMELENGTH);
      tU32 pos = OSAL_u32StringLength(as8Name)-1;

      // no priority inheritance for the worker thread function
      as8Name[pos] ='2';
      (tVoid)OSAL_s32SemaphoreCreate(as8Name, &_hMsgBoxSem, 0);

      // with priority inheritance for the lock on message queue
      as8Name[pos] ='3';
      (tVoid)OSAL_s32SemaphoreCreate(as8Name, &_hMsgQueueSem, 1);
   }

   /** DTOR destroys the thread attributes. Before that bStop() is called to make sure 
   * that thread is shut down and will not further access any member variables.
   */
   virtual ~ahl_tclMessageThread(){
      // make sure we have stopped the thread at first.
      (tVoid)bStop(); //lint !e1551 Function may throw exception
      tChar as8Name[OSAL_C_U32_MAX_NAMELENGTH];
      (tVoid)OSAL_szStringNCopy(as8Name,_as8ThreadName, OSAL_C_U32_MAX_NAMELENGTH);
      tU32 pos = OSAL_u32StringLength(as8Name)-1;
      as8Name[pos] ='2';
      (tVoid)OSAL_s32SemaphoreClose(_hMsgBoxSem);
      (tVoid)OSAL_s32SemaphoreDelete(as8Name);
      as8Name[pos] ='3';
      (tVoid)OSAL_s32SemaphoreClose(_hMsgQueueSem);
      (tVoid)OSAL_s32SemaphoreDelete(as8Name);
   }

   /** bThreadFunction() is called every time when an Message occurs.
   * if the function returns false the while loop in the thread is left. 
   */
   virtual tBool bThreadFunction(const tclMessage& rMsg, tBool bQueueOverflowed)=0;

   /** vPostMessage() is called to send an message to the thread for
   * further action. The function can be called from any thread.
   */
   tVoid vPostMessage(const tclMessage& msg){
      //post the message
      (tVoid)OSAL_s32SemaphoreWait(_hMsgQueueSem, OSAL_C_TIMEOUT_FOREVER);
      if(_MsgQueue.size() < AHL_MAX_QUEUE_SIZE) {
         _MsgQueue.push(msg);
      } else {
         _bMsgQueueOverflowed = TRUE;
      }
      (tVoid)OSAL_s32SemaphorePost(_hMsgQueueSem);
      (tVoid)OSAL_s32SemaphorePost(_hMsgBoxSem);
   }

   /** Clears the message queue */
   tVoid vClearMessages(){
      vClearTriggers();
   }

   /* virtual function to add the possibility to remove the content of a message at shutdown, 
      if the message content is not given free in the destructor (like in CCA messages)
      and a special free function must be called instead
   */
   virtual tVoid vDeleteMessageContent(tclMessage& /* rMsg */) {}

protected:
   /** Clears the triggers (events / message queues)*/
   virtual tVoid vClearTriggers(){ 
      (tVoid)OSAL_s32SemaphoreWait(_hMsgQueueSem, OSAL_C_TIMEOUT_FOREVER);
      // reset the semaphore to 0
      tS32 v=0;
      (tVoid)OSAL_s32SemaphoreGetValue(_hMsgBoxSem, &v);
      while( v > 0) {
         (tVoid)OSAL_s32SemaphoreWait(_hMsgBoxSem, OSAL_C_TIMEOUT_FOREVER);
         (tVoid)OSAL_s32SemaphoreGetValue(_hMsgBoxSem, &v);
      }
      // remove all messages
      while(!_MsgQueue.empty()) {
         vDeleteMessageContent(_MsgQueue.front());
         _MsgQueue.pop();
      }
      _bMsgQueueOverflowed = FALSE;
      (tVoid)OSAL_s32SemaphorePost(_hMsgQueueSem);
   }

   /**
   * Adds a trigger (event / message) which shall just brings the thread out of wait position. This function is called when base thread function wants to control thread condition. The trigger information will be removed and not reaching the bThreadFunction() 
   */
   virtual tVoid vTrigger(){
      (tVoid)OSAL_s32SemaphoreWait(_hMsgQueueSem, OSAL_C_TIMEOUT_FOREVER);
      if(_MsgQueue.size() < AHL_MAX_QUEUE_SIZE) {
         _MsgQueue.push(tclMessage());
      } else {
         _bMsgQueueOverflowed = TRUE;
      }
      (tVoid)OSAL_s32SemaphorePost(_hMsgQueueSem);
      (tVoid)OSAL_s32SemaphorePost(_hMsgBoxSem);
   }

   /**
   * This function shall infinitively block at the trigger inbox (eventWait / messageWait etc.)
   * @return tBool shall be false in case that the wait function cannot be processed. 
   */
   virtual tBool bWaitForTrigger(){
      // wait for next action to do
      if(OSAL_ERROR == OSAL_s32SemaphoreWait(_hMsgBoxSem, OSAL_C_TIMEOUT_FOREVER)) {
         return FALSE;
      } else {
         return TRUE;
      }
   }

   /**
   * This function implements the thread work function the parameter is the pointer to this of the given instance. The function enables inheriting classes to define a thread function with the implementation specific parameters.
   * @return tBool shall be false if thread shall leave the loop. 
   */
   virtual tBool bBaseThreadFunction(tPVoid pvInst){
      if(pvInst==0) return FALSE;
      ahl_tclMessageThread* poThis = (ahl_tclMessageThread*)pvInst;
      // get the next message
      (tVoid)OSAL_s32SemaphoreWait(poThis->_hMsgQueueSem, OSAL_C_TIMEOUT_FOREVER);
      // make sure we really have a message
      tclMessage* pMsg=0;
      tBool bOverflowed=FALSE;
      if(poThis->_MsgQueue.size()>0) {
         pMsg = OSAL_NEW tclMessage(poThis->_MsgQueue.front());
         bOverflowed = poThis->_bMsgQueueOverflowed;
         poThis->_MsgQueue.pop();
         poThis->_bMsgQueueOverflowed = FALSE;
         (tVoid)OSAL_s32SemaphorePost(poThis->_hMsgQueueSem);
      } else {
         // ERROR no message avail but semaphore posted...
         // ignore and go back to wait
         (tVoid)OSAL_s32SemaphorePost(poThis->_hMsgQueueSem);
         return TRUE;
      }
      // call to worker function ...
      tBool bRes = bThreadFunction(*pMsg, bOverflowed);
      if(pMsg) 
      {
         OSAL_DELETE pMsg;
      }
      return bRes;
   }

private:
   /** message inbox signal */
   OSAL_tSemHandle   _hMsgBoxSem;
   /** semaphore to protect the queue */
   OSAL_tSemHandle   _hMsgQueueSem;
   /** messages from user thread */
   tMsgQueue _MsgQueue;
   /** true if one message was lost will be reset as soon as one value is taken out */
   tBool _bMsgQueueOverflowed; //
};

#endif // AHL_TCLMESSAGETHREAD_H
//EOF

