/************************************************************************
* FILE:         ahl_tclEventThread.h
* PROJECT:      VWLLNF
* SW-COMPONENT: (framework)
*----------------------------------------------------------------------
*
* DESCRIPTION:  base workthread's with event communication
*              
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2009 Robert Bosch GmbH, Hildesheim
* HISTORY:      
* Date      | Author   | Modification
* 30.07.08  | Hessling | added Thread with event comunication to AHL                     
* 16.02.09  | Hessling | suspend and resume added, more documentation done                     
* 01.03.09  | Hessling | seperation in MessageThread and EventThread
*************************************************************************/
#ifndef AHL_EVENTTHREAD_H
#define AHL_EVENTTHREAD_H


#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#include "ahl_tclBaseThread.h"

/**
* class ahl_tclEventThread is an minimal task having an event inbox to communicate
* with the thread function.
* By inheriting the ahl_tclEventThread the implementer defines:
* - events on which the thread shall react
* - parameters to be passed into the thread as membervariables of the thread
*   (don't forget semaphores if the varables are also used in other instances)
* - the bThreadFunction() which is called when an event occurs and just reacts 
*   on events coded in the event field. The function will be called
*   whenever an event has been posted by vPostEvent().
* As typical for AHL the implementation is good practice and most commonly 
* used functionality. (about 64 times in di_middleware_server).
* It is the more simple and ressource saving as the ahl_BaseWorkThread becasue
* it does not use CCA message box.
* Design:
* tclThread care about thread handling functions and makes sure that the developper 
* has as little as possible to do with thread functions of OSAL. The thread is by
* default reached with event handling, wherein the developper can self define the 
* complete range of events. In parallel tclThread cares about controlling the thread 
* so that starting and stopping does avoid typical implementation errors like shutting
* down during still open handles, or freeing memory which is still in use by the thread.
* 
* @usage: "How to use this framework"
* 1. inherit the class and write an enum list of envents you want to handle in
*    bThreadFunction(). Put all data the thread will need as protected (or private)
*    Membervariables in your inheriting class. (e.g. a queue of values received from 
*    an ioPort being observed etc.)
* 2. According to typical class implementation put in AIL::bOnInit() the bStart() 
*    function and in AIL::vOnApplicationClose() bStop(). If you additionally want to 
*    save resources during OFF the entry may vSuspend() the thread and leaving may
*    vResume().
* 3. If you have to handle more than one thread you may want to shutdown them in 
*    parallel. For this case shutdown can be nonblocking initiated with vPostStopRequest()
*    you should afterwards wait for finla closure with blocking bStop() to make sure all
*    threads have really stopped.
*/
class ahl_tclEventThread : public ahl_tclBaseThread
{
public:
   /** CTOR creates the thread attributes. normally all threads should be defined via
   * the internal registry. For this case AppId and threadName should identify the 
   * thread. In cases where this data is not available you can also control the stack
   * size and priority with this constuctor. In case that no thread name is given, the 
   * this pointer address will be used to generate a unique string.
   * @param u16AppId your application id helping to find the entries in registry
   * @param tCString a string of max 31 chars defining the name also in registry
   *        if name is not given (0 or "") the constructor will create an individual   
   *        from the this pointer using usign the default parameters only.
   * @param u32DefaultStackSize if value cannot be retrieved from reg
   * @param u32DefaultPrio if value cannot be retrieved from reg
   */
   ahl_tclEventThread(tU16 u16AppId=((tU16)-1), 
                 tCString szThreadName=0, 
                 tS32 s32DefaultStackSize=1024, 
                 tU32 u32DefaultPrio=OSAL_C_U32_THREAD_PRIORITY_LOWEST);

   /** DTOR destroys the thread attributes. Before that bStop() is called to make sure 
   * that thread is shut down and will not further access any member variables.
   */
   virtual ~ahl_tclEventThread();

   /** bThreadFunction() is called every time when an event occurs.
   * if the function returns false the while loop in the thread is left
   * and the thread ends.
   */
   virtual tBool bThreadFunction(OSAL_tEventMask u32EventMask)=0;

   /** vPostEvent() is called to send an event to the thread for
   * further action. The function can be called from any thread.
   */
   tVoid vPostEvent(OSAL_tEventMask u32Event);

   /** Clears the events */
   tVoid vClearEvents();

protected:

   /** Clears the triggers (events / message queues)*/
   virtual tVoid vClearTriggers();

   /**
    * Adds a trigger (event / message) which shall just brings the thread out
    * of wait position. This function is called when base thread function wants
    * to control thread condition. The trigger information will be removed and
    * not reaching the bThreadFunction()
    */
   virtual tVoid vTrigger() ;

   /**
    * This function shall infinitively block at the trigger inbox
    * (eventWait / messageWait etc.)
    * @return false if the wait function cannot be processed because of errors.
    */
   virtual tBool bWaitForTrigger();

   /**
    * This function implements the thread work function the parameter is the
    * pointer to this of the given instance. The function enables inheriting
    * classes to define a thread function with the implementation specific
    * parameters.
    * @return tBool shall be false if thread shall leave the loop. 
    */
   virtual tBool bBaseThreadFunction(tPVoid pvInst);

private:
   /** actual event to be processed */
   OSAL_tEventMask _oResultMask;
   /** events from user thread */
   OSAL_tEventHandle _hEvent;
};

#endif // AHL_THREAD_H
//EOF

