///////////////////////////////////////////////////////////
//  ahl_tclBaseConfigStore.cpp
//  Implementation of the Class ahl_tclBaseConfigStore
//  Created on:      20-Sep-2007 
//  Original author: Matthias Hessling
///////////////////////////////////////////////////////////

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

// linking registry access functions
#define REG_S_IMPORT_INTERFACE_GENERIC
#include "reg_if.h"

// defines for this file
#define AHL_CFG_USER_PROFILE_NUM_KEY_DIR "/dev/registry/LOCAL_MACHINE/SOFTWARE/BLAUPUNKT/PROCESS/SYSTEM/"
#define AHL_CFG_USER_PROFILE_NUM_KEY "USER_PROFILE_NUM"
#define AHL_CFG_USER_PROFILE_NUM_DEFAULT 1
#define AHL_CFG_USER_PROFILE_NUM_MAX 4

#include "ahl_tclBaseConfigStore.h"
#include "ahl_trace.h"

// -- trace identifier ------- define this value for every file new 
#ifdef FILE_NUMBER
#undef FILE_NUMBER
#endif
#define FILE_NUMBER TRC::ahl_tclBaseConfigStore_cpp
// -- trace identifier ------- end

#define AHL_CFG_USE_REGISTRY // enables using of registry entry as input for current 
#define AHL_MAX_FILE_NAME_LENGTH 255
#define AHL_CFG_FILE_NAME "--cfg.dat" // m-cfg.dat, u1cfg.dat, u2cfg.dat ...


ahl_tclBaseConfigStore::ahl_tclBaseConfigStore(tCString szConfigFileDir, 
                                               tU16 u16Major, 
                                               tU8 u8Minor)
   :mCfgHeader(), mu8UserId(0), mbUserIdOk(FALSE), mbUserIdFromReg(TRUE), 
   mu8UserIdPos(0), mszMachineConfigFile(0), mszUserConfigFile(0),
   mu16FileVerMaj(0), mu8FileVerMin(0), mbFileVerOk(FALSE)
{
   TRACE_FLOW_DEF(TRC::FnConstructor);
   mCfgHeader.u16VersionMaj = u16Major;
   mCfgHeader.u8VersionMin = u8Minor;
   if(szConfigFileDir && *szConfigFileDir) {
      tU32 dirLen = (tU32)OSAL_u32StringLength(szConfigFileDir); 
      tU32 cfgLen = (tU32)OSAL_u32StringLength(AHL_CFG_FILE_NAME); 
      if ((dirLen>0) && ((dirLen+cfgLen+1)<= AHL_MAX_FILE_NAME_LENGTH)) {
         vCreateDir(szConfigFileDir);
         mszMachineConfigFile = OSAL_NEW tChar[dirLen+cfgLen+1]; 
         TRACE_ON_ERROR_NULL(TRC::FnConstructor, mszMachineConfigFile);
         if(mszMachineConfigFile==0) {
            TRACE_ERROR(TRC::FnConstructor, TRC::ErrNullPointerCantReserveMemOnHeap);
         } else {
            (tVoid)OSAL_pvMemoryCopy(&mszMachineConfigFile[0], szConfigFileDir, dirLen);
            (tVoid)OSAL_pvMemoryCopy(&mszMachineConfigFile[dirLen], AHL_CFG_FILE_NAME, cfgLen); // TODO check!
            mszMachineConfigFile[dirLen] = 'm'; 
            mszMachineConfigFile[dirLen+cfgLen] = 0; // end of string
         }
         TRACE_DBG_LS(TRC::FnConstructor, TR_LEVEL_USER_3, mszMachineConfigFile);
         mszUserConfigFile = OSAL_NEW tChar[dirLen+cfgLen+1]; 
         TRACE_ON_ERROR_NULL(TRC::FnConstructor, mszUserConfigFile);
         if(mszUserConfigFile==0) {
            TRACE_ERROR(TRC::FnConstructor, TRC::ErrNullPointerCantReserveMemOnHeap);
         } else {
            (tVoid)OSAL_pvMemoryCopy(&mszUserConfigFile[0], szConfigFileDir, dirLen);
            (tVoid)OSAL_pvMemoryCopy(&mszUserConfigFile[dirLen], AHL_CFG_FILE_NAME, cfgLen); 
            mszUserConfigFile[dirLen] = 'u'; 
            mszUserConfigFile[dirLen+cfgLen] = 0; // end of string
         }
         /* we don't set the id in the name now, because we dont know the id jet
          * function vSetUserIdInFileName() will care of
          * mszUserConfigFile[mu8UserIdPos] = (char) (0x30 | (u8UserId & 0x07));
          */
         mu8UserIdPos = (tU8)(dirLen+1);    // pos of number of user profile in file name
         TRACE_DBG_LS(TRC::FnConstructor, TR_LEVEL_USER_3, mszUserConfigFile);
      } else {
         TRACE_ERROR(TRC::FnConstructor, TRC::ErrValueOutOfRange);  
      }
   }
}

ahl_tclBaseConfigStore::~ahl_tclBaseConfigStore() 
{
   if(mszMachineConfigFile) {
      OSAL_DELETE[] mszMachineConfigFile;
      mszMachineConfigFile =0 ;
   }
   if(mszUserConfigFile) {
      OSAL_DELETE[] mszUserConfigFile;
      mszUserConfigFile =0 ;
   }
}

tBool ahl_tclBaseConfigStore::bGetConfigFileVersion(tU16 &ru16Major, tU8 &ru8Minor) const
{
   ru16Major = mu16FileVerMaj;
   ru8Minor = mu8FileVerMin ;
   return mbFileVerOk;
}

tVoid ahl_tclBaseConfigStore::vCreateDir(tCString szDir) const
{
   TRACE_FLOW_DEF(TRC::FnCreateDir);
   tChar BasePathName[AHL_MAX_FILE_NAME_LENGTH];
   tChar CurrPathName[AHL_MAX_FILE_NAME_LENGTH];
   // pointer to parameter string at place with next dir which needs to be created
   const tChar * pLeftPathName = szDir;
   tChar * pNextSlash = NULL; // end pointer
   OSAL_trIOCtrlDir * prIOCtrlDir = NULL;
   tU32 uDevStrLen = (tU32)OSAL_u32StringLength("/dev/");

   (tVoid)OSAL_pvMemorySet(BasePathName, 0x00, AHL_MAX_FILE_NAME_LENGTH);
   (tVoid)OSAL_pvMemorySet(CurrPathName, 0x00, AHL_MAX_FILE_NAME_LENGTH);
   do
   {
      /* determine the parent current (base) path of current path */
      if ((0   == OSAL_u32StringLength(BasePathName)) 
          || ('/' != BasePathName[OSAL_u32StringLength(BasePathName) - 1]) )
      {
         (tVoid)OSAL_szStringConcat(BasePathName, "/");
      }
      (tVoid)OSAL_szStringConcat(BasePathName, CurrPathName);
      
      /* determine the current pathname */
      (tVoid)OSAL_szStringCopy(CurrPathName, pLeftPathName + 1);
      pNextSlash = (tChar*)OSAL_ps8StringSearchChar(CurrPathName, '/');
      if (pNextSlash)
      {
         * pNextSlash = 0; // terminate string at place of slash
         // don't create dir if no more slash designating last part shallbe a dir 
         if (uDevStrLen < OSAL_u32StringLength(BasePathName))
         {
            prIOCtrlDir = OSALUTIL_prOpenDir(BasePathName);
            if (NULL != prIOCtrlDir)
            { 
               (tVoid)OSALUTIL_s32CreateDir(prIOCtrlDir->fd, CurrPathName);
               (tVoid)OSALUTIL_s32CloseDir(prIOCtrlDir);
            }
         }
      } 
      pLeftPathName = OSAL_ps8StringSearchChar(pLeftPathName + 1, '/');
   }
   while (pLeftPathName);
}

tVoid ahl_tclBaseConfigStore::vLoadMachineData()
{
   TRACE_FLOW_DEF(TRC::FnLoadMachineData);
   tPCU8 pcDefaultData=0;
   tU32 uDefaultLen=0;
   vGetMachineDefaultData(pcDefaultData, uDefaultLen);
   if((pcDefaultData==0) ||(uDefaultLen==0)) {
      // ok nothing to do
      return;
   }
   // open file
   tPU8 pu8Buf=0;
   tU32 uBufLen=0;
   if(bLoadData(mszMachineConfigFile, pu8Buf, uBufLen, pcDefaultData, uDefaultLen)) {
      vSetMachineData(pu8Buf, uBufLen);
      if(pu8Buf) {
         OSAL_DELETE[] pu8Buf;
      }
   } else {
      vSetMachineData(pcDefaultData, uDefaultLen);
   }
}

tVoid ahl_tclBaseConfigStore::vCreateConfigFile(tCString szFileName, tPCU8 pcBuf, tU32 u32Len)
{
   TRACE_FLOW_DEF(TRC::FnCreateConfigFile);
   OSAL_tIODescriptor fd = OSAL_IOCreate(szFileName , OSAL_EN_READWRITE);
   TRACE_ON_ERROR_OSAL(TRC::FnCreateConfigFile, fd);
   if(OSAL_ERROR != fd) 
   {
      // write the default data
      tS32 s32Return;
      mCfgHeader.u32BufLen = u32Len;
      s32Return = OSAL_s32IOWrite(fd, (tPS8)&mCfgHeader, sizeof(mCfgHeader));
      TRACE_ON_ERROR_OSAL(TRC::FnCreateConfigFile, s32Return);

      // put default data in
      s32Return = OSAL_s32IOWrite(fd, (tPCS8)pcBuf, u32Len);
      TRACE_ON_ERROR_OSAL(TRC::FnCreateConfigFile, s32Return);
      // close file because only then data is written
      s32Return = OSAL_s32IOClose(fd);
      TRACE_ON_ERROR_OSAL(TRC::FnCreateConfigFile, s32Return);
   }
}

tVoid ahl_tclBaseConfigStore::vStoreMachineData()
{
   TRACE_FLOW_DEF(TRC::FnStoreMachineData);
   tPCU8 pcu8Data=0;
   tU32 u32Len=0;
   vGetMachineData(pcu8Data, u32Len);
   if((pcu8Data==0)||(u32Len==0)) {
      // no machine config data set so nothing to do
      return;
   }
   // store in file
   vStoreData(mszMachineConfigFile, pcu8Data, u32Len);
}

tVoid ahl_tclBaseConfigStore::vStoreData(tCString szConfigFile,  
                    tPCU8 pcu8Buf, 
                    tU32 u32BufLen)
{
   TRACE_FLOW_DEF(TRC::FnStoreData);
   // check if config file is set
   if (szConfigFile == 0) {
      TRACE_ERROR(TRC::FnStoreData, TRC::ErrNoFileNameAvailable);
      return;
   }
   if (pcu8Buf == 0) {
      TRACE_ERROR(TRC::FnStoreData, TRC::ErrNullPointerInstanceMissing);
      return;
   }
   // open file
   OSAL_tIODescriptor fd;
   fd = OSAL_IOOpen(szConfigFile, OSAL_EN_READWRITE);
   if(fd == OSAL_ERROR){
      // first to create the file, because it may not exist
      TRACE_DBG_LS(TRC::FnStoreData,TR_LEVEL_COMPONENT, 
         "open error; try to create the file...");
      vCreateConfigFile(szConfigFile, pcu8Buf, u32BufLen);
      fd = OSAL_IOOpen(szConfigFile, OSAL_EN_READWRITE);
      if(fd == OSAL_ERROR){
         TRACE_ERROR_OSAL(TRC::FnStoreData);
      } else {
         (tVoid)OSAL_s32IOClose(fd);
      }
   } else {
      // check if data is same as file if so, we don't write again
      struct stCfgHeader oHeader; // version Maj, Min, u16 reserved, 32bit length
      mCfgHeader.u32BufLen = u32BufLen; // version was already set in constructor
      tS32 rlen = OSAL_s32IORead(fd, (tPS8)&oHeader, sizeof(oHeader));
      TRACE_ON_ERROR_OSAL(TRC::FnLoadData, rlen);
      if((rlen != OSAL_ERROR) && (rlen == (tS32)sizeof(oHeader))) {
         if(0==OSAL_s32MemoryCompare(&mCfgHeader, &oHeader, (tU32)rlen)){
            tPS8 ps8Buf = OSAL_NEW tS8[oHeader.u32BufLen];
            TRACE_ON_ERROR_NULL(TRC::FnLoadData, ps8Buf);
            rlen = OSAL_s32IORead(fd, ps8Buf, oHeader.u32BufLen);
            TRACE_ON_ERROR_OSAL(TRC::FnLoadData, rlen);
            if(ps8Buf && rlen>0) { // rlen is sizeof(oHeader) but lint didn't know...
               if(0==OSAL_s32MemoryCompare(ps8Buf, (tPCS8)pcu8Buf, (tU32)rlen)){
                  // everything fine. nothing to store
                  OSAL_DELETE[] ps8Buf;
                  rlen = OSAL_s32IOClose(fd);
                  TRACE_ON_ERROR_OSAL(TRC::FnLoadData, rlen);
                  return;
               }
               OSAL_DELETE[] ps8Buf;
            }
         } //lint !e593 prio3 reviewed
      }
      tS32 ret = OSALUTIL_s32FSeek(fd, 0, SEEK_SET);
      TRACE_ON_ERROR_OSAL(TRC::FnLoadUserData,ret);
      // write data to file from given buffer
      tS32 s32Return;
      s32Return = OSAL_s32IOWrite(fd, (tPS8)&mCfgHeader, sizeof(mCfgHeader));
      TRACE_ON_ERROR_OSAL(TRC::FnStoreData, s32Return);
      TRACE_INFO_RES(TRC::FnStoreData, s32Return);
      // put data in
      s32Return = OSAL_s32IOWrite(fd, (tPCS8)pcu8Buf, u32BufLen);
      TRACE_ON_ERROR_OSAL(TRC::FnStoreData, s32Return);
      TRACE_INFO_RES(TRC::FnStoreData, s32Return);
      // close file because only then data is written
      s32Return = OSAL_s32IOClose(fd);
      TRACE_ON_ERROR_OSAL(TRC::FnStoreData, s32Return);
   }
}


tVoid ahl_tclBaseConfigStore::vSetUserId(tU8 u8UserId)
{
   if(u8UserId <= 0x07) {
      if(mszUserConfigFile) {
         vSetUserIdInFileName();
      }
      mu8UserId = u8UserId;
      mbUserIdOk = TRUE;
      mbUserIdFromReg = FALSE;
   } else {
      TRACE_ERROR(TRC::FnSetUserId, TRC::ErrValueOutOfRange);
   }
}

tBool ahl_tclBaseConfigStore::bGetUserId(tU8& ru8UserId)const
{
   if(mbUserIdOk) {
      ru8UserId = mu8UserId;
   }
	return mbUserIdOk;
}

tVoid ahl_tclBaseConfigStore::vSetUserIdFromReg(tBool bOn){
   mbUserIdFromReg = bOn;
   if(mbUserIdFromReg) {
      mu8UserId = u8GetUserIdFromRegistry();
      mbUserIdOk = TRUE;
   }
}

tVoid ahl_tclBaseConfigStore::vSetUserIdInFileName() 
{
   // checks have been done before
   if( mszUserConfigFile ) {
      mszUserConfigFile[mu8UserIdPos] = (tChar) (0x30 | (mu8UserId & 0x07)); 
   }
}

tU8 ahl_tclBaseConfigStore::u8GetUserIdFromRegistry() const 
{
   TRACE_FLOW_DEF(TRC::FnGetUserIdFromRegistry);
#ifdef AHL_CFG_USE_REGISTRY
   reg_tclRegKey reg(AHL_CFG_USER_PROFILE_NUM_KEY_DIR);  
   tU32 u32UserProfId=0;
   if(reg.bQueryU32(AHL_CFG_USER_PROFILE_NUM_KEY,&u32UserProfId)) {
      if (u32UserProfId == 0) {
         u32UserProfId = AHL_CFG_USER_PROFILE_NUM_DEFAULT;
      } else if (u32UserProfId > AHL_CFG_USER_PROFILE_NUM_MAX) {
         u32UserProfId = AHL_CFG_USER_PROFILE_NUM_MAX;
      } else {
         // do nothing value was fine
      }
   } else {
      // something wrong with registry so take internal default
      u32UserProfId = AHL_CFG_USER_PROFILE_NUM_DEFAULT;
   }
   TRACE_INFO_RES(TRC::FnGetUserIdFromRegistry, u32UserProfId);
   return (tU8)u32UserProfId;
#else
   return AHL_CFG_USER_PROFILE_NUM_DEFAULT;
#endif
}

tBool ahl_tclBaseConfigStore::bLoadData(tCString szConfigFile, 
                                        tPU8 &rpu8Buf, 
                                        tU32 &ru32Length, 
                                        tPCU8 pcu8DefBuf, 
                                        tU32 u32DefLen) 
{
   TRACE_FLOW_DEF(TRC::FnLoadData);
   // check if config file is set
   if (szConfigFile == 0) {
      TRACE_ERROR(TRC::FnLoadData, TRC::ErrNoFileNameAvailable);
      return FALSE;
   }
   // open file
   OSAL_tIODescriptor fd;
   tBool bRet = FALSE;
   fd = OSAL_IOOpen(szConfigFile, OSAL_EN_READWRITE);
   if(fd == OSAL_ERROR){
      // first to create the file, because it may not exist
      tU32 err = OSAL_u32ErrorCode();
      // inform developer about that case
      TRACE_DBG_LS(TRC::FnLoadData,TR_LEVEL_COMPONENT,OSAL_coszErrorText(err));
      TRACE_DBG_LS(TRC::FnLoadData,TR_LEVEL_COMPONENT,"Try creating a new file...");
      vCreateConfigFile(szConfigFile, pcu8DefBuf, u32DefLen);
      mu16FileVerMaj = mCfgHeader.u16VersionMaj;
      mu8FileVerMin = mCfgHeader.u8VersionMin;
      mbFileVerOk = TRUE;
      // because file wasn't there use default data for config value
      bRet = FALSE;
   } else {
      // read the header of the file to see versioning
      // now check application version
      struct stCfgHeader oHeader; // version Maj, Min, u16 reserved, 32bit length
      tS32 rlen = OSAL_s32IORead(fd, (tPS8)&oHeader, sizeof(oHeader));
      TRACE_ON_ERROR_OSAL(TRC::FnLoadData, rlen);
      if((rlen == OSAL_ERROR) || ((tU32)rlen < sizeof(oHeader))) {
         // wrong file length, try to override with correct data
         TRACE_DBG_LS(TRC::FnLoadData,TR_LEVEL_COMPONENT, 
            "corrupted header trying creating a new file...");
         rlen = OSAL_s32IOClose(fd);
         TRACE_ON_ERROR_OSAL(TRC::FnLoadData, rlen);
         // because file wasn't ok use default data for config value
         vCreateConfigFile(szConfigFile, pcu8DefBuf, u32DefLen);
         mu16FileVerMaj = mCfgHeader.u16VersionMaj;
         mu8FileVerMin = mCfgHeader.u8VersionMin;
         mbFileVerOk = TRUE;
         bRet = FALSE;
      } else if(mCfgHeader.u16VersionMaj != oHeader.u16VersionMaj) {
         // wrong file length, try to override with correct data
         TRACE_DBG_LS(TRC::FnLoadData,TR_LEVEL_COMPONENT, 
            "wrong major version trying creating a new file...");
         // because file wasn't right version use default data for config value
         tS32 ret = OSALUTIL_s32FSeek(fd, 0, SEEK_SET);
         TRACE_ON_ERROR_OSAL(TRC::FnLoadUserData,ret);
         mCfgHeader.u32BufLen = u32DefLen; // version was already set in constructor
         ret = OSAL_s32IOWrite(fd, (tPS8)&mCfgHeader, sizeof(mCfgHeader));
         TRACE_ON_ERROR_OSAL(TRC::FnStoreData, ret);
         ret = OSAL_s32IOWrite(fd, (tPCS8)pcu8DefBuf, u32DefLen);
         TRACE_ON_ERROR_OSAL(TRC::FnLoadUserData,ret);
         mu16FileVerMaj = mCfgHeader.u16VersionMaj;
         mu8FileVerMin = mCfgHeader.u8VersionMin;
         mbFileVerOk = TRUE;
         bRet = FALSE;
      } else {
         if(mCfgHeader.u8VersionMin != oHeader.u8VersionMin) {
            TRACE_DBG_LSN(TRC::FnLoadData, TR_LEVEL_COMPONENT, 
                     "Minor version differs. It high word app; low word file:", 
                     (((tU32)mCfgHeader.u16VersionMaj<<16)+oHeader.u8VersionMin));
            // this place is for later if minor versions do need exception handling
         }
         // now we know value should fit. Because of differences in minor version, 
         mu16FileVerMaj = oHeader.u16VersionMaj;
         mu8FileVerMin = oHeader.u8VersionMin;
         mbFileVerOk = TRUE;
         // read first defaults in the buffer
         tU32 uFileLen = oHeader.u32BufLen; 
         // use buffer with maximum length
         ru32Length = (u32DefLen > uFileLen)? u32DefLen : uFileLen;
         TRACE_INFO_RES(TRC::FnLoadData, ru32Length);
         rpu8Buf = OSAL_NEW tU8[ru32Length];
         TRACE_ON_ERROR_NULL(TRC::FnLoadData, rpu8Buf);
         if(rpu8Buf==0) {
            bRet = FALSE;
            ru32Length = 0;
            TRACE_ERROR(TRC::FnLoadData, TRC::ErrNullPointerCantReserveMemOnHeap);
         } else {
            (tVoid)OSAL_pvMemoryCopy(rpu8Buf, pcu8DefBuf, u32DefLen);
            // than override with content of file
            tS32 rlen2 = OSAL_s32IORead(fd, (tPS8)rpu8Buf, uFileLen);
            TRACE_ON_ERROR_OSAL(TRC::FnLoadData, rlen2);
            TRACE_INFO_RES(TRC::FnLoadData, rlen2);
            if((tU32)rlen2 != uFileLen) {
               TRACE_DBG_LSN(TRC::FnLoadData, TR_LEVEL_COMPONENT, 
                        "File length differs from expected in header. It was:", 
                        rlen2);
            }
            // let application read the config data
            bRet = TRUE; // caller must clear buffer!
         }
      }
      // close file because OSAL says: "Do not keep open files long time
      tS32 ret = OSAL_s32IOClose(fd);
      TRACE_ON_ERROR_OSAL(TRC::FnLoadData, ret);
   }
   return bRet;
}

tVoid ahl_tclBaseConfigStore::vLoadUserData()
{
   TRACE_FLOW_DEF(TRC::FnLoadUserData);
   tPCU8 pcDefaultData=0;
   tU32 uDefaultLen=0;
   vGetUserDefaultData(pcDefaultData, uDefaultLen);
   if((pcDefaultData==0) || (uDefaultLen==0)) {
      // no default triggers that no data shall be loaded
      return;
   }
   if(mbUserIdFromReg) {
      // look into registry for the ID
      mu8UserId = u8GetUserIdFromRegistry();
      mbUserIdOk = TRUE;
   } else if (!mbUserIdOk) {
      TRACE_ERROR(TRC::FnLoadUserData, TRC::ErrNoUserIdDefined);
      return;
   } else {
      // everything fine
   }
   // update user Id in filename
   vSetUserIdInFileName();
   // open file
   tPU8 pu8Buf=0;
   tU32 uBufLen=0;
   if(bLoadData(mszUserConfigFile, pu8Buf, uBufLen, pcDefaultData, uDefaultLen)) {
      TRACE_INFO_RES(TRC::FnLoadUserData, 1);
      vSetUserData(pu8Buf, uBufLen);
      if(pu8Buf) {
         OSAL_DELETE[] pu8Buf;
      }
   } else {
      TRACE_INFO_RES(TRC::FnLoadUserData, 2);
      vSetUserData(pcDefaultData, uDefaultLen);
   }
}

tVoid ahl_tclBaseConfigStore::vStoreUserData()
{
   TRACE_FLOW_DEF(TRC::FnStoreUserData);
   tPCU8 pcu8Data=0;
   tU32 u32Len=0;
   vGetUserData(pcu8Data, u32Len);
   if((pcu8Data==0)||(u32Len==0)) {
      // no user config data set so nothing to do
      return;
   }
   if (!mbUserIdOk) {
      TRACE_WARN(TRC::FnStoreUserData, TRC::WarnNoUserIdSetWontStoreConfig);
      return;
   }
   TRACE_INFO_RES(TRC::FnStoreUserData, mu8UserId);
   // make sure right ID is in filename
   vSetUserIdInFileName();
   // store in file
   vStoreData(mszUserConfigFile, pcu8Data, u32Len);
}

tVoid ahl_tclBaseConfigStore::vRemoveMachineData(tVoid){
   TRACE_FLOW_DEF(TRC::FnRemoveMachineData);
   tS32 ret = OSAL_s32IORemove(mszMachineConfigFile);
   if(ret == OSAL_ERROR) {  
      tU32 u32ErrorCode = OSAL_u32ErrorCode();
      if (u32ErrorCode != OSAL_E_DOESNOTEXIST) {
         // in case that the file does not exist the file successfully removed
         // all other are errors
         TRACE_ERROR_OSAL_CODE(TRC::FnRemoveMachineData,u32ErrorCode);
      } else {
         TRACE_INFO_RES(TRC::FnRemoveMachineData, OSAL_OK);
      }
   } else {
      TRACE_INFO_RES(TRC::FnRemoveMachineData, ret);
   }
} //lint !e1762 prio3 reviewed: this is not made const because of having same prototype as vRemoveUserData()

tVoid ahl_tclBaseConfigStore::vRemoveUserData(tVoid){
   TRACE_FLOW_DEF(TRC::FnRemoveUserData);
   // make sure right ID is in filename
   vSetUserIdInFileName();
   tS32 ret = OSAL_s32IORemove(mszUserConfigFile);
   if(ret == OSAL_ERROR) {  
      tU32 u32ErrorCode = OSAL_u32ErrorCode();
      if (u32ErrorCode != OSAL_E_DOESNOTEXIST) {
         // in case that the file does not exist the file successfully removed
         // all other are errors
         TRACE_ERROR_OSAL_CODE(TRC::FnRemoveMachineData,u32ErrorCode);
      } else {
         TRACE_INFO_RES(TRC::FnRemoveUserData, OSAL_OK);
      }
   } else {
      TRACE_INFO_RES(TRC::FnRemoveUserData, ret);
   }
}

tVoid ahl_tclBaseConfigStore::vRemoveAllUserData(tVoid){
   TRACE_FLOW_DEF(TRC::FnRemoveAllUserData);
   tBool b = mbUserIdFromReg; // value gets changed by setUserId() we restore it afterwards
   for(tU8 u=AHL_CFG_USER_PROFILE_NUM_DEFAULT; u <= AHL_CFG_USER_PROFILE_NUM_MAX; u++) {
      vSetUserId(u);
      vRemoveUserData();
   }
   mbUserIdFromReg = b;
}
