/******************************************************************************
 * $Revision:   1.3  $
 * $Date:   14 Jun 2012 15:32:56  $
 * A changelog can be found at the end of this file.
 ******************************************************************************
 * FILE:         ahl_CCAMethodTable.cpp
 * SW-COMPONENT: AHL
 * DESCRIPTION:  A specialised notification table for methods
 * AUTHOR:       BSOT Jentsch
 * COPYRIGHT:    (c) 2012 Bosch
 * HISTORY:
 * 10.06.12      Rev 1.0
 *               Initial revision
 *****************************************************************************/
#define ET_TRACE_WARNING_ON

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define AMT_S_IMPORT_INTERFACE_GENERIC
#include "amt_if.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

#include "ahl_ccaExtensionTrace.h"
#include "ahl_ccaMethodTable.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_AHL_CCA_EXTENSION
#include "trcGenProj/Header/ahl_ccaMethodTable.cpp.trc.h"
#endif


ahl_tclCCAMethodTable::ahl_tclCCAMethodTable(ail_tclOneThreadAppInterface* pMainApp)
 : _pMainApp(pMainApp)
 , _oMethodTable()
 , _oMethodHandlingTypeMap()
 , _hLock(OSAL_C_INVALID_HANDLE)
 , _abTimerCreated(FALSE)
 , _abTimerStarted(FALSE)
{
   (tVoid)OSALUTIL_s32NPrintFormat((tS8*)_szLockName, sizeof(_szLockName), "ahl_ML%p", (void *)this);
   _szLockName[sizeof(_szLockName) - 1] = '\0';
}

tBool ahl_tclCCAMethodTable::bInit()
{
   if (OSAL_s32SemaphoreCreate(_szLockName, &_hLock, 1) == OSAL_OK)
   {
      return TRUE;
   }
   else
   {
      et_vErrmemStringNormal(TR_CLASS_AHL_CCA_EXTENSION, "ahl_tclCCAMethodTable: Create of semaphore %s returns error: %s", _szLockName, OSAL_coszErrorText(OSAL_u32ErrorCode()));
      NORMAL_M_ASSERT_ALWAYS();
   }
   return FALSE;
}

tBool ahl_tclCCAMethodTable::bDestroy()
{
   if (OSAL_s32SemaphoreClose(_hLock) == OSAL_OK)
   {
      if (OSAL_s32SemaphoreDelete(_szLockName) == OSAL_OK)
      {
         _hLock = OSAL_C_INVALID_HANDLE;
         return TRUE;
      }
   }
   return FALSE;
}

tBool ahl_tclCCAMethodTable::bLock() const
{
   if ((_hLock != OSAL_C_INVALID_HANDLE) &&
       (OSAL_s32SemaphoreWait(_hLock, OSAL_C_TIMEOUT_FOREVER) == OSAL_OK))
   {
      return TRUE;
   }
   return FALSE;
}

tVoid ahl_tclCCAMethodTable::vUnlock() const
{
   NORMAL_M_ASSERT(OSAL_s32SemaphorePost(_hLock) == OSAL_OK);
}

tBool ahl_tclCCAMethodTable::bIsMethodRunning(tU16 u16FktID) const
{
   ahl_tRunningMethodMap::const_iterator it = _oMethodTable.find(u16FktID);
   if (it != _oMethodTable.end())
   {
      return TRUE;
   }

   return FALSE;
}

tBool ahl_tclCCAMethodTable::bSetRunningMethod(const amt_tclServiceData& corfoMsg)
{
   tBool bSuccess = FALSE;

   tU16 nFID = corfoMsg.u16GetFunctionID();

   if (bIsMethodRunning(nFID) == FALSE)
   {
      bSuccess = TRUE;
      ahl_tRunningMethod sMethod;

      sMethod.u16AppID = corfoMsg.u16GetSourceAppID();
      sMethod.u16AppSubID = corfoMsg.u16GetSourceSubID();
      sMethod.u16SrvID = corfoMsg.u16GetServiceID();
      sMethod.u16RegisterID = corfoMsg.u16GetRegisterID();
      sMethod.u16CmdCounter = corfoMsg.u16GetCmdCounter();
      sMethod.u16FunctionID = nFID;
      sMethod.u32EndTime = OSAL_ClockGetElapsedTime() + nGetMethodHandlingTimeout(nFID);
      sMethod.bEntryIsFree = FALSE;
      _oMethodTable[nFID] = sMethod;

      // let Timer for Method duration running
      vStartTimer();
   }
   else
   {
      // should never called if the Method is still running (this is an internal check)
      NORMAL_M_ASSERT_ALWAYS();
   }
   return bSuccess;
}

tBool ahl_tclCCAMethodTable::bClearRunningMethod(tU16 u16FktID)
{
   tBool bSuccess = FALSE;

   ahl_tRunningMethodMap::iterator it = _oMethodTable.find(u16FktID);
   if (it != _oMethodTable.end())
   {
      bSuccess = TRUE;
      _oMethodTable.erase(it);
   }

   if (_oMethodTable.size() == 0)
   {
      // no more running methods -> stop timer
      vStopTimer();
   }

   return bSuccess;
}

// called in case of double MethodStart
tBool ahl_tclCCAMethodTable::bStoreMethodstartMessage(tU16 u16FunctionId, amt_tclServiceData* poMsg)
{
   tBool bSuccess = FALSE;
   amt_tclServiceData* pStoredMessage = OSAL_NEW amt_tclServiceData();
   if (pStoredMessage != NULL)
   {
      tU32 size = poMsg->u32GetSize();
      ETG_TRACE_COMP(("bStoreMethodstartMessage FID %d GetSize %d DynSize %d" , u16FunctionId , size , (poMsg->u32GetDynMsgSize()) ));
      // copy all content of message while cloning
      poMsg->vSetDynMsgSize(poMsg->u32GetSize());
      ETG_TRACE_COMP(("bStoreMethodstartMessage FID %d GetSize %d DynSize %d" , u16FunctionId , size , (poMsg->u32GetDynMsgSize()) ));
      
      if ((pStoredMessage->bCloneMessageContent(poMsg) == TRUE) && (pStoredMessage->bIsValid() == TRUE))
      {
         bSuccess = TRUE;
         ahl_tStoredMessagesMap::iterator it = _oStoredMessagesMap.find(u16FunctionId);
         if (it != _oStoredMessagesMap.end())
         {
            // create an copy of the message and let the framework delete the origin one !!!
               it->second.push_back(pStoredMessage);
         }
         else
         {
            ahl_tclServiceDataMessageArray newMessageArray;
            _oStoredMessagesMap[u16FunctionId] = newMessageArray;
            _oStoredMessagesMap[u16FunctionId].push_back(pStoredMessage);
         }
      }
      else
      {
         NORMAL_M_ASSERT_ALWAYS();
      }
   }
   else
   {
      FATAL_M_ASSERT_ALWAYS();
   }
   return bSuccess;//lint !e429
}

// called in case of double MethodStart
amt_tclServiceData* ahl_tclCCAMethodTable::pGetStoredMethodstartMessage(tU16 u16FunctionId)
{
   amt_tclServiceData* poStoredMessage = NULL;

   ahl_tStoredMessagesMap::iterator it = _oStoredMessagesMap.find(u16FunctionId);
   if (it != _oStoredMessagesMap.end())
   {
      if (it->second.size() > 0)
      {
         poStoredMessage = it->second.front();
         // remove from queue
         it->second.pop_front();
      }
   }

   return poStoredMessage;
}



tBool ahl_tclCCAMethodTable::bSetMethodHandlingType(tU16 u16FktID, ahl_tenMethodMultipleRequestHandlingType eHandlingType, tU32 nTimeout)
{
   tBool bSuccess = FALSE;

   if ((eHandlingType != AHL_EN_METHOD_MRHT_NORMAL) && (_hLock == OSAL_C_INVALID_HANDLE))
   {
      // avoid creation of semaphore if no 
      bInit();
   }

   ahl_tMethodHandlingTypeMap::iterator it = _oMethodHandlingTypeMap.find(u16FktID);
   if (it == _oMethodHandlingTypeMap.end())
   {
      ahl_tMethodHandlingType sMethodHandlingType;
      sMethodHandlingType.eHandlingType = eHandlingType;
      sMethodHandlingType.u32Timeout = nTimeout;

      _oMethodHandlingTypeMap[u16FktID] = sMethodHandlingType;
      NORMAL_M_ASSERT(sGetMethodHandlingType(u16FktID) == eHandlingType);

      bSuccess = TRUE;
   }
   else
   {
      NORMAL_M_ASSERT_ALWAYS();
   }

   return bSuccess;
}

ahl_tenMethodMultipleRequestHandlingType ahl_tclCCAMethodTable::sGetMethodHandlingType(tU16 u16FktID) const
{
   ahl_tMethodHandlingTypeMap::const_iterator it = _oMethodHandlingTypeMap.find(u16FktID);
   if (it != _oMethodHandlingTypeMap.end())
   {
      if (it->second.eHandlingType > AHL_EN_METHOD_MRHT_QUEUEING)
      {
         ETG_TRACE_ERRMEM(("FID: %d, HandlingType: %d, Timeout: %d" , u16FktID, (*it).second.eHandlingType, (*it).second.u32Timeout )); 
         NORMAL_M_ASSERT_ALWAYS();
      }
      return it->second.eHandlingType;
   }

   return AHL_EN_METHOD_MRHT_NORMAL;
}

tU32 ahl_tclCCAMethodTable::nGetMethodHandlingTimeout(tU16 u16FktID)
{
   ahl_tMethodHandlingTypeMap::iterator it = _oMethodHandlingTypeMap.find(u16FktID);
   if (it != _oMethodHandlingTypeMap.end())
   {
      return it->second.u32Timeout;
   }
   else
   {
      NORMAL_M_ASSERT_ALWAYS();
      return AHL_METHOD_TIMER_DEFAULT_TIMEOUT;
   }
}

tVoid ahl_tclCCAMethodTable::vTraceTable(tU16 u16ClassID) const
{
      ET_TRACE_ERROR(u16ClassID, "        Current method handlings ...");

      ET_TRACE_ERROR(u16ClassID, "          MethodHandlingTypeMap: Size %d" 
         _ _oMethodHandlingTypeMap.size()); //lint !e1776 prio3 reviewed: param3 is the trick to put the variable parameter list into a macro 

      ahl_tMethodHandlingTypeMap::const_iterator it;
      for ( it=_oMethodHandlingTypeMap.begin() ; it != _oMethodHandlingTypeMap.end(); it++ )
      {
         ET_TRACE_ERROR(u16ClassID,
            "            FID: %d, HandlingType: %d, Timeout: %d" 
            _ (*it).first
            _ (*it).second.eHandlingType
            _ (*it).second.u32Timeout); //lint !e1776 prio3 reviewed: param3 is the trick to put the variable parameter list into a macro 
      }

      ET_TRACE_ERROR(u16ClassID, "          Current pending methods = %d" _ _oMethodTable.size()); //lint !e1776 prio3 reviewed: param3 is the trick to put the variable parameter list into a macro 

      ahl_tRunningMethodMap::const_iterator it2;
      for ( it2=_oMethodTable.begin() ; it2 != _oMethodTable.end(); it2++ )
      {
         ET_TRACE_ERROR(u16ClassID,
            "            FID: %d/%d, AppID: 0x%04X, SubID: 0x%04X, RegId: %d, CmdCnt:%d, Srv: 0x%04X" 
            _ (*it2).first
            _ (*it2).second.u16FunctionID
            _ (*it2).second.u16AppID
            _ (*it2).second.u16AppSubID
            _ (*it2).second.u16RegisterID
            _ (*it2).second.u16CmdCounter
            _ (*it2).second.u16SrvID); //lint !e1776 prio3 reviewed: param3 is the trick to put the variable parameter list into a macro 
      }

      ET_TRACE_ERROR(u16ClassID, "          Timer created: %d, Timer started: %d" _ _abTimerCreated _ _abTimerStarted );
}

tVoid ahl_tclCCAMethodTable::vStartTimer()
{
   if (_abTimerCreated == FALSE)
   {
      if (_atclTimer.bInit(this, &ahl_tclCCAMethodTable::vOnTimerCallback, NULL) == TRUE)
      {
         _abTimerCreated = TRUE;
      }
      else
      {
         NORMAL_M_ASSERT_ALWAYS();
      }
   }
   if (_abTimerCreated == TRUE && _abTimerStarted == FALSE)
   {
      if (_atclTimer.bStart(AHL_METHOD_TIMER_CYCLE, AHL_METHOD_TIMER_CYCLE) == TRUE)
      {
         _abTimerStarted = TRUE;
      }
      else
      {
         NORMAL_M_ASSERT_ALWAYS();
      }
   }
}

tVoid ahl_tclCCAMethodTable::vStopTimer()
{
   if (_abTimerCreated == TRUE && _abTimerStarted == TRUE)
   {
      if (_atclTimer.bStop())
      {
         _abTimerStarted = FALSE;
      }
      else
      {
         NORMAL_M_ASSERT_ALWAYS();
      }
   }
   else
   {
      NORMAL_M_ASSERT_ALWAYS();
   }
}


tVoid ahl_tclCCAMethodTable::vOnTimerCallback(tVoid*)
{
   if (bLock() == TRUE)
   {
      ahl_tRunningMethodMap::const_iterator it;
      for ( it = _oMethodTable.begin() ; it != _oMethodTable.end(); it++ )
      {
         OSAL_tMSecond nCurrentTime = OSAL_ClockGetElapsedTime();
         if (nCurrentTime  > (*it).second.u32EndTime)
         {  
            ET_TRACE_ERROR(TR_CLASS_AIL, "vOnTimerCallback FID %d is timed out (method run longer then the maximum defined runtime %d %d). Remove method from list and send error message to client." _ (*it).second.u16FunctionID _ nCurrentTime _ (*it).second.u32EndTime);
            et_vErrmemStringNormal(TR_CLASS_AIL, "vOnTimerCallback FID %d is timed out (method run longer then the maximum defined runtime %d %d). Remove method from list and send error message to client." _ (*it).second.u16FunctionID _ nCurrentTime _ (*it).second.u32EndTime);

            amt_tclServiceDataError oErrorMsg;
            oErrorMsg.vInitServiceData(
                          _pMainApp->u16GetAppId(),            // Source AppID
                          (*it).second.u16AppID,                // Target AppID
                          AMT_C_U8_CCAMSG_STREAMTYPE_NODATA,    // StreamType
                          0,                                    // StreamCounter
                          (*it).second.u16RegisterID,           // RegisterID
                          (*it).second.u16CmdCounter,           // CmdCounter,
                          (*it).second.u16SrvID,                // ServiceID,
                          (*it).second.u16FunctionID,           // Function ID
                          AMT_C_U8_CCAMSG_OPCODE_ERROR          // Opcode Error
                            );

            oErrorMsg.vSetErrorData(AMT_C_U16_ERROR_INTERNAL_FAILURE);
            NORMAL_M_ASSERT(_pMainApp->enPostMessage(&oErrorMsg, TRUE) == AIL_EN_N_NO_ERROR);
//            NORMAL_M_ASSERT( bClearRunningMethod((*it).second.u16FunctionID) == TRUE);
            it = _oMethodTable.begin();
            if (it == _oMethodTable.end())
            {
               break;
            }
         }
      }
      vUnlock();
   }
   else
   {
      NORMAL_M_ASSERT_ALWAYS();
   }
}

