/********************************************************************//**
 * @file         ahl_StreamMessageTarget.h
 * @ingroup      StreamMsgMap
 *
 * @component    Application Help Library
 *
 * @description  Framework for message map abstraction.
 *
 *               This class represents a message target. This Framework 
 *               is similar to MFC's CmdTarget.
 *
 *               This code is copied from mm_MessageTarget.h and modified
 *               for ahl_tclStreamer instead of amt_tclServiceData.  It
 *               also subsumes the functionality of ahl_tclBaseWork.
 *
 * @author       CM-AM/ENG-DI-Gieseke
 * @copyright    2002,2006 Blaupunkt GmbH
 ************************************************************************/


#ifndef AHL_STREAM_MESSAGE_TARGET_H
#define AHL_STREAM_MESSAGE_TARGET_H


#define OSAL_S_IMPORT_INTERFACE_GENERIC   /*!< Import all OSAL features */
#include "osal_if.h"

#include "ahl_StreamFuncMsgMap.h"         // use message map

class ahl_tclStreamer;

/********************************************************************//**
 * @class       ahl_tclStreamMessageTarget
 * @ingroup     StreamMsgMap
 *
 * @description Message target base class.
 *
 *              This is the abstraction of a simple Stream-message target.
 *
 * @author      CM-AM/ENG-DI-Gieseke
 ************************************************************************/
class ahl_tclStreamMessageTarget
{
public:
   ahl_tclStreamMessageTarget();

   /********************************************************************//**
    * @method      virtual ~ahl_tclStreamMessageTarget()    
    *
    * @description The object destructor.
    *
    *              Because the object contains a virtual method
    *              the object should also contain a virtual destructor !
    *              This is only important for derived classes 
    *              and prevents compiler and lint warnings
    * @return      -
    *
    * @author      CM-DI/TMS Roland Jentsch
    ************************************************************************/
   virtual ~ahl_tclStreamMessageTarget();

protected:
   /********************************************************************//**
    * @method      bDefaultStreamMsgHandler
    *
    * @description Handler for incoming Stream messages.
    *
    *              This method performs the actual message dispatching.
    *
    *              It searches a handler for the message @a poMessage in
    *              the message map supplied by the message target @a pThis.
    *              After the first handler is found, this handler is
    *              executed, and the method returns.
    *
    * @note        This method expects the next data in the stream
    *              @a poMessage to be the function ID.  It extracts the
    *              ID from the stream and performs the dispatching based
    *              on this information.
    *
    * @question    Why is this method static?
    *
    * @param       pThis     (->I) \n
    *              The supplier of the message handler methods.
    *
    * @param       poMessage (->I) \n
    *              The message that is to be handled.
    *
    * @return      @c TRUE if a handler has been found and executed,
    *              @c FALSE otherwise.
    *
    * @author      CM-AM/ENG-DI-Gieseke
    ************************************************************************/
   static tBool bDefaultStreamMsgHandler(ahl_tclStreamMessageTarget *pThis,
                                         ahl_tclStreamer *poMessage);

   /********************************************************************//**
    * @method      pFindMsgEntry
    *
    * @description Find message handler for a given message.
    *
    *              This method examines a list of message map entries
    *              for an entry that matches the function ID of the
    *              supplied message.
    *
    * @param       lpEntry  (->I) \n
    *              Pointer to the list of message map entries that is to
    *              be examined.
    *
    * @param       pMessage (->I) \n
    *              The message for which a matching entry is to be found.
    *
    * @param       u16FuncId  (I) \n
    *              The function ID for which a matching entry is to be
    *              found.
    *
    * @return      A pointer to the matching entry, or @c NULL if no entry
    *              matched the function ID of @a pMessage.
    *
    * @author      CM-AM/ENG-DI-Gieseke
    ************************************************************************/
   const ahl_tStreamMsgMapEntry *pFindMsgEntry(const ahl_tStreamMsgMapEntry* lpEntry,
                                               ahl_tclStreamer *pMessage,
                                               tU16 u16FuncId) const;

   /**
    * @name The default message map
    *
    * These data members define the (empty) default message map of every
    * message target.
    *@{*/

   /** The default message map. */
   static const ahl_tStreamMsgMap _oStreamMsgMap;

   /** The entries of the default message map. */
   static const ahl_tStreamMsgMapEntry _oStreamMsgEntries[];
   /*@}*/

   /********************************************************************//**
    * @method      poGetStreamMsgMap
    *
    * @description Retrieve pointer to actual message map.
    *
    *              This method is called to retrieve the most derived
    *              message map of this object, i.e. the one where the
    *              search for a message handler is actually started.
    *
    *              To achieve this goal, this function has to be
    *              overridden by every class that defines a message map.
    *
    *              If the message maps are declared and defined with
    *              the #DECLARE_STREAMMSG_MAP and #BEGIN_STREAMMSG_MAP
    *              macros, the user does not have to worry about this
    *              fact, since these macros also take care to declare
    *              and define an appropriate poGetStreamMsgMap() override.
    *
    * @return      A pointer to the message map of this object.
    *
    * @author      CM-AM/ENG-DI-Gieseke
    *
    * @see         DECLARE_STREAMMSG_MAP, BEGIN_STREAMMSG_MAP
    ************************************************************************/
   virtual const ahl_tStreamMsgMap* poGetStreamMsgMap() const;

public:
   /********************************************************************//**
    * @method      vDispatchMessage
    *
    * @description Dispatch a stream message.
    *
    *              This method performs the actual message dispatching.
    *
    *              It searches a handler for the message @a poMessage in
    *              the message map supplied of this message target.
    *
    *              After the first handler is found, this handler is
    *              executed, and the method returns.  Otherwise, if no
    *              handler is found, it calls vOnUnknownMessage().
    *
    * @param       poMessage (->I) \n
    *              The message that is to be dispatched.
    *
    * @return      -
    *
    * @author      CM-AM/ENG-DI-Gieseke
    *
    * @see         vOnUnknownMessage()
    ************************************************************************/
   tVoid vDispatchMessage(ahl_tclStreamer *poMessage);

   /********************************************************************//**
    * @method      vOnUnknownMessage
    *
    * @description Handle unknown messages.
    *
    *              This method is called whenever a message is dispatched
    *              for which no message handler has been registered.
    *
    *              Classes derived from ahl_tclStreamMessageTarget should
    *              override this method to do any necessary handling.
    *
    * @param       poMessage (->I) \n
    *              The message in question.
    *
    * @return      -
    *
    * @author      CM-AM/ENG-DI-Gieseke
    *
    * @see         vDispatchMessage()
    ************************************************************************/
   virtual tVoid vOnUnknownMessage(ahl_tclStreamer *poMessage);
};

#endif
