/********************************************************************//**
 * @file         ahl_ListStreamer.h
 * @ingroup      Streamer
 *
 * @component    Application Help Library
 *
 * @description  This file defines a specialised data stream class.
 *
 * @author       CM-AM/ENG-DI-Streitfeld
 * @copyright    2002,2006 Blaupunkt GmbH
 ************************************************************************/

#ifndef ahl_ListStreamer_h
#define ahl_ListStreamer_h

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define  AHL_S_IMPORT_INTERFACE_STREAMER
#include "ahl_if.h"

class ahl_tb8;
class ahl_te8;
class ahl_tclStreamableType;
class ahl_tclStreamableList;


//##ModelId=3D9023CF03D1

/********************************************************************//**
 * @class       ahl_tclListStreamer
 * @ingroup     Streamer
 *
 * @description A specialised stream class.
 *
 *              This class extends the basic ahl_tclStreamer to work
 *              not only with basic data types, but also with the
 *              more complex streamable types.
 *
 * @author      CM-AM/ENG-DI-Streitfeld
 ************************************************************************/
class ahl_tclListStreamer : public ahl_tclStreamer
{
public:
   /********************************************************************//**
    * @method      ahl_tclListStreamer
    *
    * @description Default constructor.
    *
    *              The default constructor constructs an empty stream.
    *              Before the instance can be used for data exchange,
    *              it has to be assigned an external stream buffer
    *              with vSetStreamBuffer().
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   ahl_tclListStreamer();

   /********************************************************************//**
    * @method      ahl_tclListStreamer
    *
    * @description Constructor.
    *
    *              This constructor constructs a stream with its own
    *              memory buffer.  The stream is initially empty.
    *
    * @param       u32StreamSize (I) \n
    *                 The maximum size of the stream buffer.
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   //##ModelId=3D9025F60102
   ahl_tclListStreamer (tU32 u32StreamSize);

   /********************************************************************//**
    * @method      ~ahl_tclListStreamer
    *
    * @description Destructor.
    *
    *              If the streamer instance maintains its own buffer,
    *              the destructor releases the buffers memory.
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
      virtual ~ahl_tclListStreamer(); //ABR Compiler Warning 


   /** @name Writing data into a stream. */
   /*@{*/

   ahl_tclListStreamer& operator << (bool& bValue)
   {//lint !e1511 PQM_authorized_multi_177. Reason: impact of overloading of operators proofed and not critical
      ahl_tclStreamer::operator << (bValue);
      return *this;
   }

   ahl_tclListStreamer& operator << (tU8& u8Value)
   {//lint !e1511 PQM_authorized_multi_177. Reason: impact of overloading of operators proofed and not critical
      ahl_tclStreamer::operator << (u8Value);
      return *this;
   }

   //##ModelId=3DAEBD2E0358
   ahl_tclListStreamer& operator << (tS8& s8Value)
      {//lint !e1511 PQM_authorized_multi_177. Reason: impact of overloading of operators proofed and not critical
         ahl_tclStreamer::operator << (s8Value);
         return *this;
      }

   //##ModelId=3DAEBD2E038A
   ahl_tclListStreamer& operator << (tU16& u16Value)
      {//lint !e1511 PQM_authorized_multi_177. Reason: impact of overloading of operators proofed and not critical
         ahl_tclStreamer::operator << (u16Value);
         return *this;
      }

   //##ModelId=3DB93F6A0346
   ahl_tclListStreamer& operator << (tS16& s16Value)
      {//lint !e1511 PQM_authorized_multi_177. Reason: impact of overloading of operators proofed and not critical
         ahl_tclStreamer::operator << (s16Value);
         return *this;
      }

   //##ModelId=3DAEBD2E03BC
   ahl_tclListStreamer& operator << (tU32& u32Value)
      {//lint !e1511 PQM_authorized_multi_177. Reason: impact of overloading of operators proofed and not critical
         ahl_tclStreamer::operator << (u32Value);
         return *this;
      }

   //##ModelId=3DB93F7900E5
   ahl_tclListStreamer& operator << (tS32& s32Value)
      {//lint !e1511 PQM_authorized_multi_177. Reason: impact of overloading of operators proofed and not critical
         ahl_tclStreamer::operator << (s32Value);
         return *this;
      }

   ahl_tclListStreamer& operator << (tU64& u64Value)
      {//lint !e1511 PQM_authorized_multi_177. Reason: impact of overloading of operators proofed and not critical
         ahl_tclStreamer::operator << (u64Value);
         return *this;
      }

   ahl_tclListStreamer& operator << (tS64& s64Value)
      {//lint !e1511 PQM_authorized_multi_177. Reason: impact of overloading of operators proofed and not critical
         ahl_tclStreamer::operator << (s64Value);
         return *this;
      }
   /********************************************************************//**
    * @method      operator <<
    *
    * @description Append a @link ahl_tclStreamableList streamable
    *              list@endlink.
    *
    *              This method appends the contents of a streamable list
    *              to the stream.
    *
    * @question    Why this explicit overload for ahl_tclStreamableList?
    *              The implementation is the same as for the generic
    *              ahl_tclStreamableType case...
    *
    * @pre         There has to be enough space left in the stream buffer
    *              to receive the data.
    *
    * @post        The data is stored in the stream, and the write
    *              position is adjusted accordingly.
    *
    * @param       rfoValue (I) \n
    *                 The list whose contents are to be inserted.
    *
    * @return      This stream object; this allows chaining of stream
    *              operations.
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   //##ModelId=3D92E66600EB
   ahl_tclListStreamer& operator << (ahl_tclStreamableList& rfoValue);//lint !e1511 PQM_authorized_multi_177. Reason: impact of overloading of operators proofed and not critical

   /********************************************************************//**
    * @method      operator <<
    *
    * @description Append a @link ahl_tclStreamableType streamable
    *              type@endlink.
    *
    *              This method appends the contents of any streamable type
    *              to the stream.
    *
    * @pre         There has to be enough space left in the stream buffer
    *              to receive the data.
    *
    * @post        The data is stored in the stream, and the write
    *              position is adjusted accordingly.
    *
    * @param       rfoValue (I) \n
    *                 The object whose contents are to be inserted.
    *
    * @return      This stream object; this allows chaining of stream
    *              operations.
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   //##ModelId=3D92E6B00227
   ahl_tclListStreamer& operator << (ahl_tclStreamableType& rfoValue);//lint !e1511 PQM_authorized_multi_177. Reason: impact of overloading of operators proofed and not critical

   /********************************************************************//**
    * @method      operator <<
    *
    * @description Append an ahl_te8 value.
    *
    *              This method appends a single ahl_te8 value to the stream.
    *
    * @pre         There has to be enough space left in the stream buffer
    *              to receive the data.
    *
    * @post        The data is stored in the stream, and the write
    *              position is adjusted accordingly.
    *
    * @param       e8Value (I) \n
    *                 The value to insert.
    *
    * @return      This stream object; this allows chaining of stream
    *              operations.
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   //##ModelId=3DE4DBD50386
   ahl_tclListStreamer& operator << (const ahl_te8& e8Value);//lint !e1511 PQM_authorized_multi_177. Reason: impact of overloading of operators proofed and not critical

   /********************************************************************//**
    * @method      operator <<
    *
    * @description Append an ahl_tb8 value.
    *
    *              This method appends a single ahl_tb8 value to the stream.
    *
    * @pre         There has to be enough space left in the stream buffer
    *              to receive the data.
    *
    * @post        The data is stored in the stream, and the write
    *              position is adjusted accordingly.
    *
    * @param       b8Value (I) \n
    *                 The value to insert.
    *
    * @return      This stream object; this allows chaining of stream
    *              operations.
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   //##ModelId=3DF4B9BC00DA
   ahl_tclListStreamer& operator << (const ahl_tb8& b8Value);//lint !e1511 PQM_authorized_multi_177. Reason: impact of overloading of operators proofed and not critical

   /********************************************************************//**
    * @method      operator <<
    *
    * @description Append a tF32 value.
    *
    *              This method appends a single tF32 value to the stream.
    *
    * @pre         There has to be enough space left in the stream buffer
    *              to receive the data.
    *
    * @post        The data is stored in the stream, and the write
    *              position is adjusted accordingly.
    *
    * @param       f32Value (I) \n
    *                 The value to insert.
    *
    * @return      This stream object; this allows chaining of stream
    *              operations.
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   ahl_tclListStreamer& operator << (tF32& f32Value)
      {//lint !e1511 PQM_authorized_multi_177. Reason: impact of overloading of operators proofed and not critical
         tU32 *pu32Temp =  (tU32*)&f32Value;
         ahl_tclStreamer::operator << (*pu32Temp);
         return *this;
      }
   /*@}*/

   /** @name Reading data from a stream. */
   /*@{*/
   //##ModelId=3DB93FF60062

   //##ModelId=3DAEBD2F0039
   ahl_tclListStreamer& operator >> (tU8& u8Value)
      {//lint !e1511 PQM_authorized_multi_177. Reason: impact of overloading of operators proofed and not critical
         ahl_tclStreamer::operator >> (u8Value);
         return *this;
      }

   //##ModelId=3DB9403B00A8
   ahl_tclListStreamer& operator >> (tS8& s8Value)
      {//lint !e1511 PQM_authorized_multi_177. Reason: impact of overloading of operators proofed and not critical
         ahl_tclStreamer::operator >> (s8Value);
         return *this;
      }

   //##ModelId=3DAEBD2F0061
   ahl_tclListStreamer& operator >> (tU16& u16Value)
      {//lint !e1511 PQM_authorized_multi_177. Reason: impact of overloading of operators proofed and not critical
         ahl_tclStreamer::operator >> (u16Value);
         return *this;
      }

   //##ModelId=3DB940540270
   ahl_tclListStreamer& operator >> (tS16& s16Value)
      {//lint !e1511 PQM_authorized_multi_177. Reason: impact of overloading of operators proofed and not critical
         ahl_tclStreamer::operator >> (s16Value);
         return *this;
      }

   //##ModelId=3DAEBD2F0093
   ahl_tclListStreamer& operator >> (tU32& u32Value)
      {//lint !e1511 PQM_authorized_multi_177. Reason: impact of overloading of operators proofed and not critical
         ahl_tclStreamer::operator >> (u32Value);
         return *this;
      }

   //##ModelId=3DB9405B00EA
   ahl_tclListStreamer& operator >> (tS32& s32Value)
      {//lint !e1511 PQM_authorized_multi_177. Reason: impact of overloading of operators proofed and not critical
         ahl_tclStreamer::operator >> (s32Value);
         return *this;
      }

   ahl_tclListStreamer& operator >> (tU64& u64Value)
      {//lint !e1511 PQM_authorized_multi_177. Reason: impact of overloading of operators proofed and not critical
         ahl_tclStreamer::operator >> (u64Value);
         return *this;
      }

   //##ModelId=3DB9405B00EA
   ahl_tclListStreamer& operator >> (tS64& s64Value)
      {//lint !e1511 PQM_authorized_multi_177. Reason: impact of overloading of operators proofed and not critical
         ahl_tclStreamer::operator >> (s64Value);
         return *this;
      }
   /********************************************************************//**
    * @method      operator >>
    *
    * @description Read a @link ahl_tclStreamableList streamable
    *              list@endlink.
    *
    *              This method extracts the contents of a streamable
    *              list from the stream.
    *
    * @question    Why this explicit overload for ahl_tclStreamableList?
    *              The implementation is the same as for the generic
    *              ahl_tclStreamableType case...
    *
    * @pre         The stream contains enough unread bytes to fulfil
    *              the read request.
    *
    * @post        The read position is adjusted accordingly.
    *
    * @param       rfoValue (I) \n
    *                 The place to store the read value.
    *
    * @return      This stream object; this allows chaining of stream
    *              operations.
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   //##ModelId=3D92E66A037B
   ahl_tclListStreamer& operator >> (ahl_tclStreamableList& rfoValue);//lint !e1511 PQM_authorized_multi_177. Reason: impact of overloading of operators proofed and not critical

   /********************************************************************//**
    * @method      operator >>
    *
    * @description Read a @link ahl_tclStreamableList streamable
    *              list@endlink.
    *
    *              This method extracts the contents of any streamable
    *              type from the stream.
    *
    * @pre         The stream contains enough unread bytes to fulfil
    *              the read request.
    *
    * @post        The read position is adjusted accordingly.
    *
    * @param       rfoValue (I) \n
    *                 The place to store the read value.
    *
    * @return      This stream object; this allows chaining of stream
    *              operations.
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   //##ModelId=3D92E6B00245
   ahl_tclListStreamer& operator >> (ahl_tclStreamableType& rfoValue);//lint !e1511 PQM_authorized_multi_177. Reason: impact of overloading of operators proofed and not critical

   /********************************************************************//**
    * @method      operator >>
    *
    * @description Read an ahl_te8 value.
    *
    *              This method reads a single ahl_te8 value from the stream.
    *
    * @pre         The stream contains enough unread bytes to fulfil
    *              the read request.
    *
    * @post        The read position is adjusted accordingly.
    *
    * @param       e8Value (I) \n
    *                 The place to store the read value.
    *
    * @return      This stream object; this allows chaining of stream
    *              operations.
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   //##ModelId=3DE4DBD503D6
   ahl_tclListStreamer& operator >> (ahl_te8& e8Value);//lint !e1511 PQM_authorized_multi_177. Reason: impact of overloading of operators proofed and not critical

   /********************************************************************//**
    * @method      operator >>
    *
    * @description Read an ahl_tb8 value.
    *
    *              This method reads a single ahl_tb8 value from the stream.
    *
    * @pre         The stream contains enough unread bytes to fulfil
    *              the read request.
    *
    * @post        The read position is adjusted accordingly.
    *
    * @param       b8Value (I) \n
    *                 The place to store the read value.
    *
    * @return      This stream object; this allows chaining of stream
    *              operations.
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   //##ModelId=3DF4B9BC0152
   ahl_tclListStreamer& operator >> (ahl_tb8& b8Value);//lint !e1511 PQM_authorized_multi_177. Reason: impact of overloading of operators proofed and not critical

   /********************************************************************//**
    * @method      operator >>
    *
    * @description Read a tF32 value.
    *
    *              This method reads a single tF32 value from the stream.
    *
    * @pre         The stream contains enough unread bytes to fulfil
    *              the read request.
    *
    * @post        The read position is adjusted accordingly.
    *
    * @param       f32Value (I) \n
    *                 The place to store the read value.
    *
    * @return      This stream object; this allows chaining of stream
    *              operations.
    *
    * @author      CM-AM/ENG-DI-Streitfeld
    ************************************************************************/
   ahl_tclListStreamer& operator >> (tF32& f32Value)
      {//lint !e1511 PQM_authorized_multi_177. Reason: impact of overloading of operators proofed and not critical
         tU32 *pu32Temp = (tU32*)&f32Value;
         ahl_tclStreamer::operator >> (*pu32Temp);
         return *this;
      }
   /*@}*/
};


#endif
