/************************************************************************
 * FILE:         ahl_BaseWorkThread.h
 * PROJECT:      ELeNa
 * SW-COMPONENT: (framework)
 *----------------------------------------------------------------------
 *
 * DESCRIPTION:  Framework for base workthread's
 *               (it's a modification of the workthread
 *                implementation of C.Bruns/EFG32)
 *              
 *----------------------------------------------------------------------
 * COPYRIGHT:    (c) 2000 Robert Bosch GmbH, Hildesheim
 * HISTORY:      
 * Date      | Author   | Modification
 * 20.06.01  | Perick   | initial version
 * 17.07.01  | Perick   | added Work-Threads to Application Library 
 *                     
 *************************************************************************/


#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define AMT_S_IMPORT_INTERFACE_GENERIC
#include "amt_if.h"                        // use CCA messages
//#define AIL_S_IMPORT_INTERFACE_GENERIC
//#include "ail_if.h"

#include "ahl_BaseWorkThread.h"           // use workthread's message map



//-----------------------------------------------------------------------------
// Constructor 
//-----------------------------------------------------------------------------
ahl_tclBaseWorkThread::ahl_tclBaseWorkThread(const tChar *pcNewThreadName, 
                                             tU32 u32Prio, 
                                             tU32 u32StackSize, 
                                             tU32 u32nMaxQueueLen)
{
    //tBool bSuccess = TRUE;    ABR Compiler warning 
   _bTerminateFlag   = FALSE;
   _bIsValidFlag = FALSE;

   // setup thread's name and attributes
   (tVoid) OSAL_szStringCopy(_szThreadName, pcNewThreadName);

   OSAL_trThreadAttribute rThAttr;
   rThAttr.szName = _szThreadName;
   rThAttr.u32Priority = u32Prio;
   rThAttr.s32StackSize = u32StackSize;
   rThAttr.pfEntry = (OSAL_tpfThreadEntry)dwThreadProc; 
   rThAttr.pvArg = (tPVoid)this;

   // create thread suspended
   _oThreadID = OSAL_ThreadCreate(&rThAttr);

   // open pool, if failure -> do nothing any more...
   if(_oThreadID != OSAL_ERROR)
   {
      (tVoid) OSAL_s32MessagePoolOpen();
      // That was wrong: if(OSAL_ERROR != OSAL_s32MessagePoolOpen())
      {
         // setup thread's IN-Queue
         if(TRUE == bCreateInQueue(_szThreadName, u32nMaxQueueLen, 
                           sizeof(OSAL_trMessage)))
         {
            _bIsValidFlag = TRUE;
         }
         else
         {
            //ET_TRACE_ERROR(100,"ahl_tclWorkThread::Constr - failure <bCreateInQueue>\n");
         }
      }
   }
   else
   {
      //ET_TRACE_ERROR(100,"ahl_tclWorkThread::Constr - failure <OSAL_ThreadCreate>\n");
   }
}



//-----------------------------------------------------------------------------
// Destructor "~ahl_tclWorkThread"
//
// XXX: Dont' know if MessagePool must be closed... seems that the pool must
// be closed elsewhere...
//-----------------------------------------------------------------------------
ahl_tclBaseWorkThread::~ahl_tclBaseWorkThread()
{
        
}


//-----------------------------------------------------------------------------
// bIsValid
//
// Returns TRUE if thread object is successfully created
//-----------------------------------------------------------------------------
tBool ahl_tclBaseWorkThread::bIsValid() const
{
   return _bIsValidFlag;
}



//-----------------------------------------------------------------------------
// bCreateInQueue
//
// Creates workthread's IN-Queue
//-----------------------------------------------------------------------------
tBool ahl_tclBaseWorkThread::bCreateInQueue(const tChar *pcInQueueName,
                                            tU32 nMaxMessages, 
                                            tU32 nSizeOfOneMessage)
{
   tBool bSuccess = FALSE;
   (tVoid) nSizeOfOneMessage;  //ABR INFO 715 bgu2hi: param not used here

   if (OSAL_OK == OSAL_s32MessageQueueCreate(pcInQueueName, nMaxMessages, 
                                             sizeof(OSAL_trMessage), 
                                             OSAL_EN_READWRITE, 
                                             &_oInQueueHandle))
   {
      bSuccess = TRUE;
   }
   return bSuccess;
}



//-----------------------------------------------------------------------------
// poGetInQueue
//
// Returns the workthread's IN-Queue
//-----------------------------------------------------------------------------
OSAL_tMQueueHandle* ahl_tclBaseWorkThread::poGetInQueue()
{
   return &_oInQueueHandle; /*lint !e1536*/  //ABR Review Perick
}



//-----------------------------------------------------------------------------
// dwThreadProc (private, static)
//
// The main entry procedure. Is called when thread is activated and calls 
// the virtual thread-function.
//-----------------------------------------------------------------------------
//tVoid __stdcall wil_tclWorkThread::dwThreadProc(tVoid *pvArg)
OSAL_tpfThreadEntry ahl_tclBaseWorkThread::dwThreadProc(tVoid *pvArg)
{

   // cast to its own
   ahl_tclBaseWorkThread* pToMe = (ahl_tclBaseWorkThread*)pvArg;
   if(pToMe->bIsValid() == TRUE)
   {
      // call Thread functionality
      pToMe->vThreadFunction();
   }
   else
   {
      //ET_TRACE_ERROR(0,"ahl_tclWorkThread::dwThreadProc - Thread is invalid!!!\n");
   }
   return 0;
}



//-----------------------------------------------------------------------------
// vThreadFunction (virtual)
//
// returns thread name
//-----------------------------------------------------------------------------
tChar* ahl_tclBaseWorkThread::pcGetThreadName()
{
   return _szThreadName; /*lint !e1536*/  //ABR Review Perick
}



//-----------------------------------------------------------------------------
// vThreadFunction 
//
// The main thread function. Is called when thread is activated.
//-----------------------------------------------------------------------------
tVoid ahl_tclBaseWorkThread::vThreadFunction()
{
   amt_tclBaseMessage oBaseMessage;
   while(_bTerminateFlag == FALSE)
   {
      if (bMessageWait(&oBaseMessage, OSAL_C_U32_INFINITE) == TRUE)
      {
         if(_bTerminateFlag == false)
            vDispatchMessage(&oBaseMessage);
         else
         {
            oBaseMessage.bDelete(); //delete kill dummy
         }
      }
   }
   
   // kill queue
   //ET_TRACE_INFO1(100, "ahl_tclBaseWorkThread: start to kill %s\n", _szThreadName);
   (tVoid) OSAL_s32MessageQueueClose(_oInQueueHandle);
   (tVoid) OSAL_s32MessageQueueDelete(_szThreadName);
   (tVoid) OSAL_s32MessagePoolClose();
   //ET_TRACE_INFO1(100, "ahl_tclBaseWorkThread: %s killed\n", _szThreadName);  
}

//-----------------------------------------------------------------------------
// the way to terminate this thread... (provisory)
//-----------------------------------------------------------------------------
tBool ahl_tclBaseWorkThread::bTerminate()
{
   _bTerminateFlag = TRUE;

   //amt_tclPowerMessage oDummyMsg(0, 0, 0, 0, 0); // message to wakeup thread
   //ail_bIpcMessagePost(0, _oInQueueHandle, &oDummyMsg, 
   //                    OSAL_C_U32_MQUEUE_PRIORITY_HIGHEST);
   
   return TRUE;
}


//-----------------------------------------------------------------------------
// bActivate
//
// Activates the thread
//-----------------------------------------------------------------------------
tBool ahl_tclBaseWorkThread::bActivate()
{
   tBool bSuccess = FALSE;
   if(OSAL_OK == OSAL_s32ThreadActivate(_oThreadID))
   {
      bSuccess = TRUE;
   }

   return bSuccess;
}


//-----------------------------------------------------------------------------
// bSuspend
//
// Suspends the thread
//-----------------------------------------------------------------------------
tBool ahl_tclBaseWorkThread::bSuspend()
{
   tBool bSuccess = FALSE;
   if(OSAL_OK == OSAL_s32ThreadSuspend(_oThreadID))
   {
      bSuccess = TRUE;
   }
   return bSuccess;
}


//-----------------------------------------------------------------------------
// vDispatchMessage
//
// This is the default dispatcher for incomming ServiceData messages. This 
// method can be called from user's own thread if a message is received.
// 
// Calls specific handler (here ServiceDataHandler ONLY)
//-----------------------------------------------------------------------------
tVoid ahl_tclBaseWorkThread::vDispatchMessage(amt_tclBaseMessage *poBaseMessage)
{
   tU8 u8MsgType = poBaseMessage->u8GetType();

   switch(u8MsgType)
   {
      case AMT_C_U8_CCAMSGTYPE_SVCDATA:
         {
            amt_tclServiceData oServiceData(poBaseMessage);
            //ServiceData Messages are dispatched by the MSG_MAP mechanism
            if(!(ahl_tclBaseWorkThread::bDefaultSvcDataHandler(this, &oServiceData)))
               vOnUnknownMessage(poBaseMessage);  
         }
         break;
      default:
         {
            vOnUnknownMessage(poBaseMessage);
         }
   }
}

tVoid ahl_tclBaseWorkThread::vOnUnknownMessage(amt_tclBaseMessage* poBaseMessage)
{
      //ET_TRACE_WARNING( 0,"BaseWorkThread vOnNewMessage: unknown message\n");
      poBaseMessage->bDelete();
}

tBool ahl_tclBaseWorkThread::bMessageWait(amt_tclBaseMessage *pMsgObject, OSAL_tMSecond nTimeout)
{
   tBool         bSuccess        = FALSE;
   tU32          u32Prio         = OSAL_C_U32_MQUEUE_PRIORITY_HIGHEST;
   OSAL_tMQueueHandle *oInQueue  = poGetInQueue();
   tS32          s32SizeInBytes;


   s32SizeInBytes = OSAL_s32MessageQueueWait(*oInQueue,
                                             (tU8*)pMsgObject->prGetOSALMsgHandle(), 
                                             sizeof(OSAL_trMessage),
                                             &u32Prio, 
                                             nTimeout);
   if(s32SizeInBytes > 0)
   {
      if(pMsgObject->bEnableAccess())
         bSuccess = TRUE;
   }

   return bSuccess;
}

