/************************************************************************
* FILE:          ahl_BaseClient.cpp
* PROJECT:       All
* SW-COMPONENT:  AHL
*----------------------------------------------------------------------
*
* DESCRIPTION: 
*              
*----------------------------------------------------------------------
*
* AUTHOR:  ESE-Neumann  (jn82hi)
*          TMS-Kempen   (kmr2hi)
*          TMS-Jentsch  (jhr2hi)           
*
* COPYRIGHT:    (c) 2008 Robert Bosch GmbH, Hildesheim
*
* HISTORY:      
*************************************************************************/

#define GENERICMSGS_S_IMPORT_INTERFACE_GENERIC
#include "generic_msgs_if.h"

/* Inclusion of OSAL Interface */
#ifndef OSAL_S_IMPORT_INTERFACE_GENERIC
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"
#endif

#define AMT_S_IMPORT_INTERFACE_GENERIC
#include "amt_if.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ETRACE_S_IMPORT_INTERFACE_ERRMEM
#include "etrace_if.h"

#define SCD_S_IMPORT_INTERFACE_GENERIC
#include "scd_if.h"

#include "ahl_BaseOneThreadApp.h"
#include "ahl_BaseOneThreadService.h"
#include "ahl_ccaExtensionTrace.h"

/************************************************
* Helper class for administration of a property 
*************************************************/
#define AHL_MAX_FUNCTION_TIME 500

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_AHL_CCA_EXTENSION
#include "trcGenProj/Header/ahl_BaseOneThreadService.cpp.trc.h"
#endif

/************************************************
* Base class of a client handler
*************************************************/

/* constructor */
ahl_tclBaseOneThreadService::ahl_tclBaseOneThreadService( 
      ahl_tclBaseOneThreadApp*   poMainAppl,
      tU16            u16ServiceID,
      tU16            u16ApplID,
      tU16            u16ServiceMajorVersion,
      tU16            u16ServiceMinorVersion,
      tU16            u16ServicePatchVersion
   )
   : ahl_tclBaseOneThreadObject(poMainAppl, u16ServiceID)
   , _tclRegisterPair(AMT_C_U32_STATE_INVALID, AMT_C_U32_STATE_OFF)
   , _tclUnregisterPair(AMT_C_U32_STATE_INVALID, AMT_C_U32_STATE_OFF)
   , _u16ServiceMajorVersion(u16ServiceMajorVersion)
   , _u16ServiceMinorVersion(u16ServiceMinorVersion)
   , _u16ServicePatchVersion(u16ServicePatchVersion)
   , _u16AppID(u16ApplID)
   , _bServiceAvailableAllowedOnAppstate(FALSE)
   , _bServiceAvailableAllowedOnPrivateReason(TRUE)
   , _bServiceAvailable(FALSE)
   , clProperties()
   , clMethodTable(poMainAppl)
{
   _bAutoHandleGet = TRUE;
   _bAutoHandleSet = TRUE;
   _bAutoHandleIncDec = TRUE;
   _bEnableHighPrioFIDs = TRUE;

   /* register this client to the Application */
   if (_poMainAppl != NULL)
   {
       _poMainAppl->vAddObject(this);
       if (_u16AppID == AMT_C_U16_APPID_INVALID)
       {
          _u16AppID = _poMainAppl->u16GetAppId();
          if (_u16AppID == AMT_C_U16_APPID_INVALID)
          {
             ETG_TRACE_ERRMEM(("%x: App Id of main app is invalid, check your derived class", 
                              ETG_ENUM(ail_u16AppId, (tU16)_u16AppID)  ));
          }
       }
   }
   else
   {
       /* base class should never be NULL */
       ETG_TRACE_ERRMEM(("%x: mainapp is NULL, check your derived class", ETG_ENUM(ail_u16AppId, (tU16)_u16AppID)  ));
       FATAL_M_ASSERT_ALWAYS();
   }

   _tclRegisterPair.bSetDefault(AHL_PAIRLIST_DEFAULT_SERVICE_AVAILABLE);
   _tclUnregisterPair.bSetDefault(AHL_PAIRLIST_DEFAULT_SERVICE_UNAVAILABLE);
   
   clProperties._oNotificationTable.bInit(_u16AppID, 
                                           u16GetServiceID(),
                                           u16ServiceMajorVersion,
                                           u16ServiceMinorVersion,
                                           u16ServicePatchVersion);
}

ahl_tclBaseOneThreadService::ahl_tclBaseOneThreadService( 
      ahl_tclBaseOneThreadApp*   poMainAppl,
      tU16            u16ServiceID,
      tU16            u16ServiceMajorVersion,
      tU16            u16ServiceMinorVersion,
      tU16            u16ServicePatchVersion
   )
   : ahl_tclBaseOneThreadObject(poMainAppl, u16ServiceID)
   , _tclRegisterPair(AMT_C_U32_STATE_INVALID, AMT_C_U32_STATE_OFF)
   , _tclUnregisterPair(AMT_C_U32_STATE_INVALID, AMT_C_U32_STATE_OFF)
   , _u16ServiceMajorVersion(u16ServiceMajorVersion)
   , _u16ServiceMinorVersion(u16ServiceMinorVersion)
   , _u16ServicePatchVersion(u16ServicePatchVersion)
   , _u16AppID(AMT_C_U16_APPID_INVALID)
   , _bServiceAvailableAllowedOnAppstate(FALSE)
   , _bServiceAvailableAllowedOnPrivateReason(TRUE)
   , _bServiceAvailable(FALSE)
   , clProperties()
   , clMethodTable(poMainAppl)
{
   _bAutoHandleGet = TRUE;
   _bAutoHandleSet = TRUE;
   _bAutoHandleIncDec = TRUE;
   _bEnableHighPrioFIDs = TRUE;

   /* register this client to the Application */
   if (_poMainAppl != NULL)
   {
       _poMainAppl->vAddObject(this);
       _u16AppID = _poMainAppl->u16GetAppId();
       if (_u16AppID == AMT_C_U16_APPID_INVALID)
       {
          ETG_TRACE_ERRMEM(("%x: App Id of main app is invalid, check your derived class", 
                           ETG_ENUM(ail_u16AppId, (tU16)_u16AppID)  ));
       }
   }
   else
   {
       /* base class should never be NULL */
       ETG_TRACE_ERRMEM(("%x: mainapp is NULL, check your derived class", 
                        ETG_ENUM(ail_u16AppId, (tU16)_u16AppID)  ));
       FATAL_M_ASSERT_ALWAYS();
   }

   _tclRegisterPair.bSetDefault(AHL_PAIRLIST_DEFAULT_SERVICE_AVAILABLE);
   _tclUnregisterPair.bSetDefault(AHL_PAIRLIST_DEFAULT_SERVICE_UNAVAILABLE);
   
   clProperties._oNotificationTable.bInit(_u16AppID, 
                                           u16GetServiceID(),
                                           u16ServiceMajorVersion,
                                           u16ServiceMinorVersion,
                                           u16ServicePatchVersion);
}

/* destructor */
ahl_tclBaseOneThreadService::~ahl_tclBaseOneThreadService(tVoid)
{
   clMethodTable.bDestroy();
   clProperties._oNotificationTable.bDestroy();

   /* base class should never be NULL */
   try
   {
       /* register this client to the Application */
       if (_poMainAppl != NULL)
       {
           _poMainAppl->vRemoveObject(this);
           _poMainAppl = NULL; 
       }
       else
       {
           ETG_TRACE_ERRMEM(("%x ~: mainapp is NULL, check your derived class", 
                            ETG_ENUM(ail_u16AppId, (tU16)_u16AppID)  ));
       }
   }
   catch (...)    /*ABR lint warning*/
   {
      NORMAL_M_ASSERT_ALWAYS();  
   }
   
}

/*************************************************************************
*
* FUNCTION: tVoid vOnNewAppState( tU32 u32OldAppState, tU32 u32AppState )
* 
* DESCRIPTION: handles state change messages form the spm
*
* PARAMETER: old state, new state
*
* RETURNVALUE: void
*
*************************************************************************/

tVoid ahl_tclBaseOneThreadService::vOnNewAppState(tU32 u32OldAppState,  tU32 u32AppState)
{
   ETG_TRACE_USR1(("%x Service->vOnNewAppState %d -> %d ", 
                   ETG_ENUM(ail_u16AppId, (tU16)_u16AppID), 
                   ETG_ENUM(ail_u32CCAState, (tU16)u32OldAppState), 
                   ETG_ENUM(ail_u32CCAState, (tU16)u32AppState)  )); 
   
   if( u32OldAppState == u32AppState ) 
   { 
      return; // switching to the same state -> do nothing
   } 

   if ( _tclRegisterPair.bIfPairIncluded(u32OldAppState, u32AppState) == TRUE)
   {
      _bServiceAvailableAllowedOnAppstate = TRUE;

      ETG_TRACE_USR1(("%x Service->vOnNewAppState set ServiceAvailableAllowedOnAppstate of service %x to %d ", 
                      ETG_ENUM(ail_u16AppId, (tU16)_u16AppID), 
                      ETG_ENUM(ail_u16ServiceId, (tU16)u16GetServiceID()), _bServiceAvailableAllowedOnAppstate  )); 

      vSetServiceState();
   }

   if ( _tclUnregisterPair.bIfPairIncluded(u32OldAppState, u32AppState) == TRUE)
   {
      _bServiceAvailableAllowedOnAppstate = FALSE;

      ETG_TRACE_USR1(("%x Service->vOnNewAppState set ServiceAvailableAllowedOnAppstate of service %x to %d ", 
                      ETG_ENUM(ail_u16AppId, (tU16)_u16AppID), 
                      ETG_ENUM(ail_u16ServiceId, (tU16)u16GetServiceID()), _bServiceAvailableAllowedOnAppstate  )); 

      vSetServiceState();
   }

   vOnNewAppStateNotify(u32OldAppState, u32AppState);
}

tVoid ahl_tclBaseOneThreadService::vSetPrivateServiceAvailableAllowed(tBool fServiceAvailableAllowed)
{
   _bServiceAvailableAllowedOnPrivateReason = fServiceAvailableAllowed;
   vSetServiceState();
}

tVoid ahl_tclBaseOneThreadService::vSetServiceState()
{
   if ((_bServiceAvailableAllowedOnAppstate == TRUE) && (_bServiceAvailableAllowedOnPrivateReason == TRUE))
   {
      if (_bServiceAvailable != TRUE)
      {
         /* --- set service available --- */
         _bServiceAvailable = TRUE;
         _poMainAppl->vServiceAvailabilityChanged(u16GetServiceID(), AMT_C_U8_SVCSTATE_AVAILABLE);
         vUpdateAllProperties();

         // Inform the derived user classes about the change
         vOnServiceAvailable();
      }
   }
   else
   {
      if (_bServiceAvailable == TRUE)
      {
         // Inform the derived user classes about the change
         vOnServiceUnavailable();

         /* --- set service unavailable --- */
         _poMainAppl->vServiceAvailabilityChanged(u16GetServiceID(), AMT_C_U8_SVCSTATE_NOT_AVAILABLE);
         _bServiceAvailable = FALSE;
      }
   }
}

tVoid ahl_tclBaseOneThreadService::vMyDispatchMessage(amt_tclServiceData *pServiceDataMessage)
{
    tU16 u16FunctionId = pServiceDataMessage->u16GetFunctionID();
    tU8  u8OpType = pServiceDataMessage->u8GetOpCode();
    tU16 u16ServiceID = pServiceDataMessage->u16GetServiceID();

    if (u16ServiceID != _u16ServiceID)
    {
        ETG_TRACE_ERRMEM(("%x vMyDispatchMessage: ServiceID not matching %x != %x", 
                         ETG_ENUM(ail_u16AppId, (tU16)_u16AppID) , 
                         ETG_ENUM(ail_u16ServiceId, (tU16)u16ServiceID), 
                         ETG_ENUM(ail_u16ServiceId, (tU16)_u16ServiceID) ));
    }

    OSAL_tMSecond msStart, msEnd;
    msStart = OSAL_ClockGetElapsedTime();

      switch (u8OpType)
      {
          case AMT_C_U8_CCAMSG_OPCODE_UPREG:
             vOnUpReg(u16FunctionId, pServiceDataMessage);
          break;
          case AMT_C_U8_CCAMSG_OPCODE_RELUPREG:
             vOnRelUpReg(u16FunctionId, pServiceDataMessage);
          break;
          case AMT_C_U8_CCAMSG_OPCODE_GET:
             vOnGet(u16FunctionId, pServiceDataMessage);
          break;
          case AMT_C_U8_CCAMSG_OPCODE_SET:
             vOnSet(u16FunctionId, pServiceDataMessage);
          break;
          case AMT_C_U8_CCAMSG_OPCODE_METHODSTART:
             vOnMethodStart(u16FunctionId, pServiceDataMessage);
          break;
          case AMT_C_U8_CCAMSG_OPCODE_METHODABORT:
             vOnMethodAbort(u16FunctionId, pServiceDataMessage);
          break;
          case AMT_C_U8_CCAMSG_OPCODE_INCREMENT:
             vOnIncrement(u16FunctionId, pServiceDataMessage);
          break;
          case AMT_C_U8_CCAMSG_OPCODE_DECREMENT:
             vOnDecrement(u16FunctionId, pServiceDataMessage);
          break;
          case AMT_C_U8_CCAMSG_OPCODE_PURESET:
             vOnPureSet(u16FunctionId, pServiceDataMessage);
          break;
          default:
             vOnNewMessage(pServiceDataMessage);
      }

      msEnd = OSAL_ClockGetElapsedTime();

      if ((msEnd - msStart) > AHL_MAX_FUNCTION_TIME)
      {
         ETG_TRACE_ERR(("%x vDispatchMessage: derived server class needs a long time (%d ms) in process message for SRV %x of type %d", 
                        ETG_ENUM(ail_u16AppId, (tU16)_u16AppID) , 
                        msEnd - msStart,
                        ETG_ENUM(ail_u16ServiceId, (tU16)u16ServiceID), 
                        ETG_ENUM(ail_u8OperationCode, (tU8)u8OpType) ));
      }
}

tVoid ahl_tclBaseOneThreadService::vOnServiceState( tU16 /*u16ServiceId*/, tU16 /*u16ServerId*/, tU16 /*u16RegisterId*/, tU8 /*u8ServiceState*/, tU16 /*u16SubId*/)
{
   // do nothing, should never be called
   ETG_TRACE_ERRMEM(("%x vOnServiceState: This should never called in a server !?", ETG_ENUM(ail_u16AppId, (tU16)_u16AppID)  ));
}

ail_tenCommunicationError ahl_tclBaseOneThreadService::eUpdateClients(tU16 u16FunctionId, amt_tclServiceData* poMessage)
{
   ail_tenCommunicationError enCommunicationError = AIL_EN_N_NO_ERROR;

   if (clProperties._oNotificationTable.bLock() == TRUE)
   {
      ETG_TRACE_USR1(("%x eUpdateClients called for service %x and FID %d", 
                       ETG_ENUM(ail_u16AppId, (tU16)_u16AppID), ETG_ENUM(ail_u16ServiceId, (tU16)_u16ServiceID), u16FunctionId  ));

      ahl_tNotification* poNextNotify = clProperties._oNotificationTable.poGetNextNotification(u16FunctionId);
      if (poNextNotify != NULL)
      {
         // new type of factory
         amt_tclServiceData oFactoryMessage;
         amt_tclServiceData* poFactoryMessage;
         tBool bHandover;

         bHandover = bStatusMessageFactory(u16FunctionId, oFactoryMessage, poMessage);
         if (bHandover == TRUE)
         {
            poFactoryMessage = &oFactoryMessage;
         }
         else
         {
            // old type of factory, only for backwards compability
            poFactoryMessage = pStatusMessageFactory(poNextNotify->u16AppID, u16FunctionId, poNextNotify->u16RegisterID, poNextNotify->u16CmdCounter);
         }

         if (poFactoryMessage != NULL)
         {
            ahl_tNotification* poCurrentNotify;
            while (poNextNotify != NULL)
            {
               {
                  amt_tclServiceData* poCurrentMessage;
                  amt_tclServiceData oCurrentMessage;
                  poCurrentNotify = poNextNotify;
                  poNextNotify = clProperties._oNotificationTable.poGetNextNotification(u16FunctionId, poNextNotify);

                  // only clone if this is not the last message
                  if (poNextNotify != NULL)
                  {
                     if (oCurrentMessage.bCloneMessageContent(poFactoryMessage) == TRUE)
                     {
                        poCurrentMessage = &oCurrentMessage;
                     }
                     else
                     {
                        ETG_TRACE_ERRMEM(("%x eUpdateClients: bCloneMessageContent returns false", ETG_ENUM(ail_u16AppId, (tU16)_u16AppID)  ));
                        poCurrentMessage = NULL;
                     }
                  }  
                  else
                  {
                     // it is the last message, use origin factory message
                     poCurrentMessage = poFactoryMessage;
                  }

                  if (poCurrentMessage != NULL)
                  {
                      vInitServiceData(*poCurrentMessage, poCurrentNotify->u16AppID, poCurrentNotify->u16RegisterID, poCurrentNotify->u16CmdCounter, u16FunctionId, AMT_C_U8_CCAMSG_OPCODE_STATUS, poCurrentNotify->u16SubID);
                      ail_tenCommunicationError enPostCommunicationError = ePostMessage(poCurrentMessage);

                      // preserve error code 
                      if (enCommunicationError == AIL_EN_N_NO_ERROR)
                      {
                         enCommunicationError = enPostCommunicationError;
                      }
                   }
                   else
                   {
                      enCommunicationError = AIL_EN_N_NOT_ALL_INFORMED;
                   }
                }
            }

            if (bHandover != TRUE)
            {
               vDestroyFactoryStatusMessage(poFactoryMessage);
            }
         }
         else
         {
            ETG_TRACE_FATAL(("%x eUpdateClients: the StatusMessageFactory of service %x does not return a valid message for FID %d", 
                             ETG_ENUM(ail_u16AppId, (tU16)_u16AppID), ETG_ENUM(ail_u16ServiceId, (tU16)_u16ServiceID), u16FunctionId  ));
         }
      }
      clProperties._oNotificationTable.vUnlock();
   }
   else
   {
      ETG_TRACE_ERRMEM(("%x eUpdateClients: could not lock notification table", ETG_ENUM(ail_u16AppId, (tU16)_u16AppID)  ));
      enCommunicationError = AIL_EN_N_UNDEFINED_ABORT;
   }

   return enCommunicationError;
}

ail_tenCommunicationError ahl_tclBaseOneThreadService::eUpdateRequestingClient(amt_tclServiceData* poMessage)
{
   tBool bHandover;
   tU16 u16FunctionId = poMessage->u16GetFunctionID();
   tU16 u16SourceAppID = poMessage->u16GetSourceAppID(); // AppID of client (we return the new status data to the sender)
   tU16 u16SourceSubID = poMessage->u16GetSourceSubID();
   tU16 u16RegisterID = poMessage->u16GetRegisterID();
   tU16 u16CmdCounter = poMessage->u16GetCmdCounter();
   ail_tenCommunicationError enCommunicationError;

   amt_tclServiceData oFactoryMessage;
   amt_tclServiceData* pMsg;

   // Create a status message by using our own 'Status Message Factory' 
   // The factory will create a 'related' CCA status message with the 'relevant' member variables.
   
   bHandover = bStatusMessageFactory(u16FunctionId, oFactoryMessage, poMessage);
   if (bHandover == TRUE)
   {
      vInitServiceData(oFactoryMessage, u16SourceAppID, u16RegisterID, u16CmdCounter, u16FunctionId, AMT_C_U8_CCAMSG_OPCODE_STATUS, u16SourceSubID);
      pMsg = &oFactoryMessage;
   }
   else
   {
      // only for backward compability
      pMsg = pStatusMessageFactory(u16SourceAppID, u16FunctionId, u16RegisterID, u16CmdCounter);
   }

   // Send message
   enCommunicationError = ePostMessage(pMsg);
   if (enCommunicationError != AIL_EN_N_NO_ERROR)
   {
      if (enCommunicationError != AIL_EN_N_INVALID_PARAMETER)
      {  
         ETG_TRACE_ERRMEM(("%x eUpdateRequestingClient: ePostMessage() to app %x for service %x failed with error %x.", 
                      ETG_ENUM(ail_u16AppId, (tU16)_u16AppID), ETG_ENUM(ail_u16AppId, (tU16)u16SourceAppID), ETG_ENUM(ail_u16ServiceId, (tU16)_u16ServiceID), enCommunicationError  ));
      }
      else
      {  
           enCommunicationError = AIL_EN_N_NOT_SEND;
           ETG_TRACE_FATAL(("%x eUpdateRequestingClient: the StatusMessageFactory of service %x does not return a valid message for FID %d", 
                            ETG_ENUM(ail_u16AppId, (tU16)_u16AppID), ETG_ENUM(ail_u16ServiceId, (tU16)_u16ServiceID), u16FunctionId  ));
      }
   }

   // Note: The message sent was created via "OSAL_NEW" (see creation of status message in the 
   //       StatusMessageFactory) and therefore has to be deleted here explictly. Otherwise you will 
   //       create a memory leak.          
   if (bHandover != TRUE)
   {
      vDestroyFactoryStatusMessage(pMsg);
   }

   return enCommunicationError;
}

tBool ahl_tclBaseOneThreadService::bOnAcceptNewRegister(tU16 /*u16ServiceId*/, tU16 /*u16ClientAppId*/, tU16 /*u16ClientSubId*/, tU16* /*pu16RegisterId*/)
{
   return TRUE;
}

tVoid ahl_tclBaseOneThreadService::vOnUnregister(tU16 /*u16ServiceId*/, tU16 u16RegisterId)
{
    clProperties._oNotificationTable.bRemoveAllEntriesWithRegID(u16RegisterId);
}

ail_tenCommunicationError ahl_tclBaseOneThreadService::ePostMessage(amt_tclServiceData* poMessage)
{
   if ((poMessage != NULL) && (poMessage->bIsValid() == TRUE))
   {
      tU8 u8OpCode = poMessage->u8GetOpCode();

      // check opcode if this signals a method end
      if ((u8OpCode == AMT_C_U8_CCAMSG_OPCODE_METHODRESULT) || 
          (u8OpCode == AMT_C_U8_CCAMSG_OPCODE_METHODRESULTLAST) || 
          (u8OpCode == AMT_C_U8_CCAMSG_OPCODE_ERROR)) 
      {
         tU16 u16FID = poMessage->u16GetFunctionID();
         if (clMethodTable.sGetMethodHandlingType(u16FID) != AHL_EN_METHOD_MRHT_NORMAL)
         {
            ETG_TRACE_USR1(("%x ePostMessage: Service 0x%04X posting of MessageResult for FID %d", 
                                ETG_ENUM(ail_u16AppId, (tU16)_u16AppID), ETG_ENUM(ail_u16ServiceId, (tU16)_u16ServiceID), u16FID  ));
            if (clMethodTable.bLock())
            {
               if (clMethodTable.bIsMethodRunning(u16FID) == TRUE)
               {
                  ETG_TRACE_COMP(("%x ePostMessage: Service 0x%04X clear flag for running method for FID %d", 
                                   ETG_ENUM(ail_u16AppId, (tU16)_u16AppID), ETG_ENUM(ail_u16ServiceId, (tU16)_u16ServiceID), u16FID  ));
   
                  NORMAL_M_ASSERT(clMethodTable.bClearRunningMethod(u16FID) == TRUE);
               }
   
               amt_tclServiceData* poStoredMessage = clMethodTable.pGetStoredMethodstartMessage(u16FID);
               // now we have the message out of the storing queue and we can unlock
               clMethodTable.vUnlock();
               if (poStoredMessage != NULL)
               {
                  if (poStoredMessage->bIsValid())
                  {
                     // send the queued message again to the own queue and main thread 
                     if (_poMainAppl->bPostIpcMessage(poStoredMessage->u16GetSourceAppID(),_poMainAppl->hGetMyInQueue(),poStoredMessage,OSAL_C_U32_MQUEUE_PRIORITY_HIGHEST))
                     {
                        ETG_TRACE_SYS_MIN(("%x ePostMessage: Service 0x%04X end of method FID %d detected and pending MethodStart in queue -> reinsert pending MethodStart now", 
                                         ETG_ENUM(ail_u16AppId, (tU16)_u16AppID), ETG_ENUM(ail_u16ServiceId, (tU16)_u16ServiceID), u16FID  ));
                     }
                     else
                     {
                        NORMAL_M_ASSERT_ALWAYS();
                        poStoredMessage->bDelete();
                     }
                  }
                  else
                  {
                     // Should never happen, because of invalid message in vector
                     NORMAL_M_ASSERT_ALWAYS();
                  }
                  // delete container
                  OSAL_DELETE poStoredMessage;
               }
               else
               {
                  ETG_TRACE_USR1(("%x ePostMessage: Service 0x%04X no pending methodstart for FID %d", 
                                   ETG_ENUM(ail_u16AppId, (tU16)_u16AppID), ETG_ENUM(ail_u16ServiceId, (tU16)_u16ServiceID), u16FID  ));
               }
            }
            else
            {
               NORMAL_M_ASSERT_ALWAYS();
            }
         }
      }
      if ((_bEnableHighPrioFIDs == TRUE) && (sHighPrioFIDs.find(poMessage->u16GetFunctionID()) != sHighPrioFIDs.end()))
      {
         return _poMainAppl->enPostMessage(poMessage, TRUE, AIL_C_U32_CCA_PRIO_MSG_SERVICE_DATA_SERVER_HIGH);
      }
      else
      {
         return _poMainAppl->enPostMessage(poMessage, TRUE);
      }
   }

   return AIL_EN_N_INVALID_PARAMETER;
}

tVoid ahl_tclBaseOneThreadService::vOnNewMessage(amt_tclServiceData* /*poMessage*/)
{
   ETG_TRACE_ERRMEM(("%x vOnNewMessage: please overwrite this creation function in your derived class", ETG_ENUM(ail_u16AppId, (tU16)_u16AppID)  ));
}

tVoid ahl_tclBaseOneThreadService::vOnMethodStart(tU16 u16FunctionId, amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR1(("%x vOnMethodStart for function id %d", ETG_ENUM(ail_u16AppId, (tU16)_u16AppID) , u16FunctionId ));

   ahl_tenMethodMultipleRequestHandlingType eMRHT = clMethodTable.sGetMethodHandlingType(u16FunctionId);

   if (bIfServiceAvailable())
   {
      switch (eMRHT)
      {
         case AHL_EN_METHOD_MRHT_NORMAL:
            if(!(ahl_tclBaseOneThreadService::bDefaultSvcDataHandler(this, poMessage)))
            {
                vOnUnknownMessage(poMessage);
            } 
         break;
         case AHL_EN_METHOD_MRHT_REJECT:
                NORMAL_M_ASSERT(clMethodTable.bLock() == TRUE);
                if (clMethodTable.bIsMethodRunning(u16FunctionId) == FALSE)
                {
                   // Method is not running
                   NORMAL_M_ASSERT(clMethodTable.bSetRunningMethod(poMessage) == TRUE);
                   clMethodTable.vUnlock();
                   if(!(ahl_tclBaseOneThreadService::bDefaultSvcDataHandler(this, poMessage)))
                   {
                       vOnUnknownMessage(poMessage);
                   }
                }
                else
                {
                   // Method is still running -> reject second request now
                   clMethodTable.vUnlock();
                   ETG_TRACE_ERR(("%x vOnMethodStart Service %x : Method with function ID %d is still running, send error message (reject second call)", 
                                      ETG_ENUM(ail_u16AppId, (tU16)_u16AppID) , ETG_ENUM(ail_u16ServiceId, (tU16)u16GetServiceID()), u16FunctionId ));
      
                   amt_tclServiceDataError oErrorMsg(*poMessage, AMT_C_U16_ERROR_FUNCTION_BUSY);
                   ail_tenCommunicationError enPostResult = _poMainAppl->enPostMessage(&oErrorMsg);
                   if (enPostResult != AIL_EN_N_NO_ERROR)
                   {
                      ETG_TRACE_ERRMEM(("%x vOnMethodStart Service %x : Method with function ID %d is still running, send error message failed with error %d", 
                                         ETG_ENUM(ail_u16AppId, (tU16)_u16AppID) , ETG_ENUM(ail_u16ServiceId, (tU16)u16GetServiceID()), u16FunctionId, enPostResult ));
                   }
                }
         break;
         case AHL_EN_METHOD_MRHT_QUEUEING:
            NORMAL_M_ASSERT(clMethodTable.bLock() == TRUE);
            if (clMethodTable.bIsMethodRunning(u16FunctionId) == FALSE)
            {
               // Method is not running
               NORMAL_M_ASSERT(clMethodTable.bSetRunningMethod(poMessage) == TRUE);
               clMethodTable.vUnlock();

               // all fine -> start method know 
               if(!(ahl_tclBaseOneThreadService::bDefaultSvcDataHandler(this, poMessage)))
               {
                   vOnUnknownMessage(poMessage);
               }
            }
            else
            {
               // method still running -> store message until the first method call is ended (checked in the methodresult posts)
               if (clMethodTable.bStoreMethodstartMessage(u16FunctionId, poMessage))
               {
                  clMethodTable.vUnlock();
                  ETG_TRACE_SYS_MIN(("%x vOnMethodStart Service %x : Method with function ID %d is still running, store second request", 
                                     ETG_ENUM(ail_u16AppId, (tU16)_u16AppID) , ETG_ENUM(ail_u16ServiceId, (tU16)u16GetServiceID()), u16FunctionId ));
               }
               else
               {
                  clMethodTable.vUnlock();
                  ETG_TRACE_ERRMEM(("%x vOnMethodStart Service %x : Method with function ID %d is still running. Internal error Method could not be stored", 
                                     ETG_ENUM(ail_u16AppId, (tU16)_u16AppID) , ETG_ENUM(ail_u16ServiceId, (tU16)u16GetServiceID()), u16FunctionId ));
               }
            }
         break;
         default:
            ETG_TRACE_ERRMEM(("%x vOnMethodStart Service %x : unknown MethodHandlingType %d for function ID %d", 
                               ETG_ENUM(ail_u16AppId, (tU16)_u16AppID) , ETG_ENUM(ail_u16ServiceId, (tU16)u16GetServiceID()), eMRHT, u16FunctionId ));
      }
   }
   else
   {
      ETG_TRACE_SYS_MIN(("%x vOnMethodStart Service %x not available, send error message", 
                         ETG_ENUM(ail_u16AppId, (tU16)_u16AppID) , ETG_ENUM(ail_u16ServiceId, (tU16)u16GetServiceID()) ));

      amt_tclServiceDataError oErrorMsg(*poMessage, AMT_C_U16_ERROR_TEMPORARY_NOT_AVAILABLE);
      ail_tenCommunicationError enPostResult = ePostMessage(&oErrorMsg);
      if (enPostResult != AIL_EN_N_NO_ERROR)
      {
         ETG_TRACE_FATAL(("%x vOnMethodStart Service %x : Method with function ID %d is not accepted because service not available, post of error message failed with %d", 
                            ETG_ENUM(ail_u16AppId, (tU16)_u16AppID) , ETG_ENUM(ail_u16ServiceId, (tU16)u16GetServiceID()), u16FunctionId, enPostResult));
      }
   }
}

tVoid ahl_tclBaseOneThreadService::vOnMethodAbort(tU16 u16FunctionId, amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR1(("%x vOnMethodAbort for function id %d", ETG_ENUM(ail_u16AppId, (tU16)_u16AppID) , u16FunctionId ));

   if(!(ahl_tclBaseOneThreadService::bDefaultSvcDataHandler(this, poMessage)))
   {
       vOnUnknownMessage(poMessage);
   }
}

tVoid ahl_tclBaseOneThreadService::vOnSet(tU16 u16FunctionId, amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR1(("%x vOnSet for function id %d", ETG_ENUM(ail_u16AppId, (tU16)_u16AppID) , u16FunctionId ));

   if (bIfServiceAvailable())
   {
      if(!(ahl_tclBaseOneThreadService::bDefaultSvcDataHandler(this, poMessage)))
      {
         if (_bAutoHandleSet)
         {
            vGenericSetHandler(u16FunctionId, poMessage, AMT_C_U8_CCAMSG_OPCODE_SET);
         }
         else
         {
            vOnUnknownMessage(poMessage);
         }
      }
   }
   else
   {
      ETG_TRACE_SYS_MIN(("%x vOnSet %x Service not available, send error message", 
                         ETG_ENUM(ail_u16AppId, (tU16)_u16AppID) , ETG_ENUM(ail_u16ServiceId, (tU16)u16GetServiceID()) ));

      amt_tclServiceDataError oErrorMsg(*poMessage, AMT_C_U16_ERROR_TEMPORARY_NOT_AVAILABLE);
      ePostMessage(&oErrorMsg);
   }
}

tVoid ahl_tclBaseOneThreadService::vOnIncrement(tU16 u16FunctionId, amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR1(("%x vOnIncrement for function id %d", ETG_ENUM(ail_u16AppId, (tU16)_u16AppID) , u16FunctionId ));

   if (bIfServiceAvailable())
   {
      if(!(ahl_tclBaseOneThreadService::bDefaultSvcDataHandler(this, poMessage)))
      {
         if (_bAutoHandleIncDec)
         {
            vGenericSetHandler(u16FunctionId, poMessage, AMT_C_U8_CCAMSG_OPCODE_INCREMENT);
         }
         else
         {
            vOnUnknownMessage(poMessage);
         }
      }
   }
   else
   {
      ETG_TRACE_SYS_MIN(("%x vOnIncrement %x Service not available, send error message", 
                         ETG_ENUM(ail_u16AppId, (tU16)_u16AppID) , ETG_ENUM(ail_u16ServiceId, (tU16)u16GetServiceID()) ));

      amt_tclServiceDataError oErrorMsg(*poMessage, AMT_C_U16_ERROR_TEMPORARY_NOT_AVAILABLE);
      ePostMessage(&oErrorMsg);
   }
}

tVoid ahl_tclBaseOneThreadService::vOnDecrement(tU16 u16FunctionId, amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR1(("%x vOnDecrement for function id %d", ETG_ENUM(ail_u16AppId, (tU16)_u16AppID) , u16FunctionId ));

   if (bIfServiceAvailable())
   {
      if(!(ahl_tclBaseOneThreadService::bDefaultSvcDataHandler(this, poMessage)))
      {
         if (_bAutoHandleIncDec)
         {
            vGenericSetHandler(u16FunctionId, poMessage, AMT_C_U8_CCAMSG_OPCODE_DECREMENT);
         }
         else
         {
            vOnUnknownMessage(poMessage);
         }
      }
   }
   else
   {
      ETG_TRACE_SYS_MIN(("%x vOnDecrement %x Service not available, send error message", 
                         ETG_ENUM(ail_u16AppId, (tU16)_u16AppID) , ETG_ENUM(ail_u16ServiceId, (tU16)u16GetServiceID()) ));

      amt_tclServiceDataError oErrorMsg(*poMessage, AMT_C_U16_ERROR_TEMPORARY_NOT_AVAILABLE);
      ePostMessage(&oErrorMsg);
   }
}

tVoid ahl_tclBaseOneThreadService::vOnPureSet(tU16 u16FunctionId, amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR1(("%x vOnPureSet for function id %d", ETG_ENUM(ail_u16AppId, (tU16)_u16AppID) , u16FunctionId ));

   if (bIfServiceAvailable())
   {
      if(!(ahl_tclBaseOneThreadService::bDefaultSvcDataHandler(this, poMessage)))
      {
         if (_bAutoHandleSet)
         {
            vGenericSetHandler(u16FunctionId, poMessage, AMT_C_U8_CCAMSG_OPCODE_PURESET);
         }
         else
         {
            vOnUnknownMessage(poMessage);
         }
      }
   }
   else
   {
      ETG_TRACE_SYS_MIN(("%x vOnPureSet %x Service not available, send error message", 
                         ETG_ENUM(ail_u16AppId, (tU16)_u16AppID) , ETG_ENUM(ail_u16ServiceId, (tU16)u16GetServiceID()) ));

      amt_tclServiceDataError oErrorMsg(*poMessage, AMT_C_U16_ERROR_TEMPORARY_NOT_AVAILABLE);
      ePostMessage(&oErrorMsg);
   }
}

tVoid ahl_tclBaseOneThreadService::vOnGet(tU16 u16FunctionId, amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR1(("%x vOnGet for function id %x", ETG_ENUM(ail_u16AppId, (tU16)_u16AppID) , u16FunctionId ));

   // vOnGet is used by vOnRelUpReg und RelUpReg messages are allowed also if the service is not available
   if (bIfServiceAvailable() || (poMessage->u8GetOpCode() == AMT_C_U8_CCAMSG_OPCODE_RELUPREG))
   {
      if(!(ahl_tclBaseOneThreadService::bDefaultSvcDataHandler(this, poMessage)))
      {
         if (_bAutoHandleGet == TRUE)
         {
            ail_tenCommunicationError enResult = eUpdateRequestingClient(poMessage);

            if (enResult != AIL_EN_N_NO_ERROR)
            {  
               ETG_TRACE_FATAL(("%x vOnGet: eUpdateRequestingClient returns error %x function ID %x", 
                                ETG_ENUM(ail_u16AppId, (tU16)_u16AppID) , enResult, u16FunctionId ));
            }
         }
         else
         {
            vOnUnknownMessage(poMessage);
         }
      }
   }
   else
   {
      ETG_TRACE_SYS_MIN(("%x vOnGet Service %x not available, send error message", 
                         ETG_ENUM(ail_u16AppId, (tU16)_u16AppID) , ETG_ENUM(ail_u16ServiceId, (tU16)u16GetServiceID()) ));

      amt_tclServiceDataError oErrorMsg(*poMessage, AMT_C_U16_ERROR_TEMPORARY_NOT_AVAILABLE);
      ePostMessage(&oErrorMsg);
   }
}

tVoid ahl_tclBaseOneThreadService::vOnUpReg(tU16 u16FunctionId, amt_tclServiceData* poMessage)
{
   if (clProperties._oNotificationTable.bAddNotification(poMessage) == TRUE)
   {
      // create a status message answer
      vOnGet(u16FunctionId, poMessage);
   }
   else
   {
      amt_tclServiceDataError oError(*poMessage, AMT_C_U16_ERROR_UPREG_FAILURE);

      ETG_TRACE_ERRMEM(("%x vMyDispatchMessage: Notification for service %x function ID %x could not be added", 
                       ETG_ENUM(ail_u16AppId, (tU16)_u16AppID) , ETG_ENUM(ail_u16ServiceId, (tU16)u16GetServiceID()), u16FunctionId ));
      ePostMessage(&oError);
   }
}

tVoid ahl_tclBaseOneThreadService::vOnRelUpReg(tU16 u16FunctionId, amt_tclServiceData* poMessage)
{
   tU16 u16RegisterID = poMessage->u16GetRegisterID();

   if (clProperties._oNotificationTable.bRemoveNotification(u16FunctionId, u16RegisterID) == TRUE)
   {
      // create an status message answer
      vOnGet(u16FunctionId, poMessage);
   }
   else
   {
      amt_tclServiceDataError oError(*poMessage, CCA_C_U16_ERROR_RELUPREG_FAILURE);

      ETG_TRACE_ERRMEM(("%x vMyDispatchMessage: Notification for service %x function ID %x could not be removed  ", 
                       ETG_ENUM(ail_u16AppId, (tU16)_u16AppID) , ETG_ENUM(ail_u16ServiceId, (tU16)u16GetServiceID()), u16FunctionId ));
      ePostMessage(&oError);
   }
}

tBool ahl_tclBaseOneThreadService::bGetServiceVersion (tU16 /*u16ServiceID*/, tU16 &rfu16MajorVersion, tU16 &rfu16MinorVersion, tU16 &rfu16PatchVersion)
{
    rfu16MajorVersion = _u16ServiceMajorVersion;
    rfu16MinorVersion = _u16ServiceMinorVersion;
    rfu16PatchVersion = _u16ServicePatchVersion;

    return TRUE;
}

amt_tclServiceData* ahl_tclBaseOneThreadService::pStatusMessageFactory(tU16 /*u16Target*/, tU16 /*u16FunctionId*/, tU16 /*u16RegisterID*/, tU16 /*u16CmdCounter*/)
{
   return NULL;
}

tBool ahl_tclBaseOneThreadService::bStatusMessageFactory(tU16 /*u16FunctionId*/, amt_tclServiceData& /*roOutMsg*/, amt_tclServiceData* /*poIncomingMessage*/)
{
   return FALSE;
}

tVoid ahl_tclBaseOneThreadService::vDestroyFactoryStatusMessage(amt_tclServiceData* poMessage)
{
   delete poMessage;
}

tVoid ahl_tclBaseOneThreadService::vUpdateAllProperties()
{
   if (clProperties._oNotificationTable.bLock() == TRUE)
   {
      tU32 i;
      std::vector<tU16> tclIDArray;

      // store current table entries to avoid locking over the complete updateclients function
      for(tFunctionMap::const_iterator it = clProperties._oNotificationTable.m_FktIdMap.begin(); it != clProperties._oNotificationTable.m_FktIdMap.end(); ++it)
      {
         if (it->second != 0)
            tclIDArray.push_back(it->first);
      }

      clProperties._oNotificationTable.vUnlock();

      for(i=0;i<tclIDArray.size();++i)
      {
            eUpdateClients(tclIDArray[i]);
      }
   }
   else
   {
      ETG_TRACE_ERRMEM(("%x vUpdateAllProperties: could not lock notification table", ETG_ENUM(ail_u16AppId, (tU16)_u16AppID) ));
   }
}

/* this function is a helper to filter trace from the specific app, clienthandler, service */
tBool ahl_tclBaseOneThreadService::bTraceClassFilter(tU16 u16Level) const
{
   return _poMainAppl->bTraceClassFilter(u16Level);
}

tVoid ahl_tclBaseOneThreadService::vOnServiceUnavailable()
{
   // only to inform derived classes 
}

tVoid ahl_tclBaseOneThreadService::vOnServiceAvailable()
{
   // only to inform derived classes 
}

tVoid ahl_tclBaseOneThreadService::vTraceInternalState(TR_tenTraceLevel enLevel) const
{
   // tracing use ail trace class because of using callback from ail framework
   ail_vTraceMsg(enLevel, "     Service %04x ", u16GetServiceID());
   ail_vTraceMsg(enLevel, "       provides Version %d.%d.%d ", _u16ServiceMajorVersion, _u16ServiceMinorVersion, _u16ServicePatchVersion);
   ail_vTraceMsg(enLevel, "       Service is available %d (%d,%d)", _bServiceAvailable, _bServiceAvailableAllowedOnAppstate, _bServiceAvailableAllowedOnPrivateReason);
   ail_vTraceMsg(enLevel, "       Tracing notification table for properties of service now ... ", _bServiceAvailable, _bServiceAvailableAllowedOnAppstate, _bServiceAvailableAllowedOnPrivateReason);
   ail_vTraceMsg(enLevel, "       Automatic handling status for SET: %d, GET: %d, IncDec: %d, HighPrioFIDs: %d ", _bAutoHandleGet, _bAutoHandleSet, _bAutoHandleIncDec, _bEnableHighPrioFIDs);
   clProperties._oNotificationTable.vTraceTable(TR_CLASS_AIL);
   clMethodTable.vTraceTable(TR_CLASS_AIL);
}

tVoid ahl_tclBaseOneThreadService::vInitServiceData(amt_tclServiceData& rServiceDataMessage, tU16 u16TargetAppID, tU16 u16RegisterID, tU16 u16CmdCounter, tU16 u16FunctionId, tU8 u8Opcode, tU16 u16TargetSubID) const
{
   rServiceDataMessage.vInitServiceData(
                 _u16AppID,                            // Source AppID
                 u16TargetAppID,                       // Target AppID
                 AMT_C_U8_CCAMSG_STREAMTYPE_NODATA,    // StreamType
                 0,                                    // StreamCounter
                 u16RegisterID,                        // RegisterID
                 u16CmdCounter,                        // CmdCounter,
                 _u16ServiceID,                        // ServiceID,
                 u16FunctionId,                        // Function ID
                 u8Opcode                              // Opcode STATUS 
                   );

   rServiceDataMessage.vSetTargetSubID(u16TargetSubID);
}

tVoid ahl_tclBaseOneThreadService::vGenericSetHandler(tU16 u16FunctionId, amt_tclServiceData* poMessage, tU8 u8OpType)
{
   // Create a local variable, which will reflect, if the current property 
   // really has been changed or not.
   tBool bPropertyChanged = FALSE;

   // Create a local status, which is set to FALSE, when the received message
   // could not be dispatched correctly or the FID is unknown. The detailed
   // error code will be stored to u16Error.
   tBool bSuccess;
   tU16 u16Error  = 0xFFFF;

   switch (u8OpType)
   {
      case AMT_C_U8_CCAMSG_OPCODE_SET:
            bSuccess = bProcessSet(poMessage, bPropertyChanged, u16Error);
      break;
      case AMT_C_U8_CCAMSG_OPCODE_INCREMENT:
            bSuccess = bProcessIncDec(poMessage, bPropertyChanged, u16Error);
      break;
      case AMT_C_U8_CCAMSG_OPCODE_DECREMENT:
            bSuccess = bProcessIncDec(poMessage, bPropertyChanged, u16Error);
      break;
      case AMT_C_U8_CCAMSG_OPCODE_PURESET:
            bSuccess = bProcessSet(poMessage, bPropertyChanged, u16Error);
      break;
      default:
         ETG_TRACE_ERRMEM(("%x vGenericSetHandler: unknown u8OpType %0d for function ID %x", 
                          ETG_ENUM(ail_u16AppId, (tU16)_u16AppID) , u8OpType, u16FunctionId ));
         bSuccess = FALSE;
      break;
   }

   // Inform the client which has requested a 'SET' of the property about the
   // result via a dedicated status message (or send an error message). 
   if( bSuccess == TRUE)
   {
      ail_tenCommunicationError enResult;

      if (u8OpType != AMT_C_U8_CCAMSG_OPCODE_PURESET)
      {
         // Post Status message to the requesting client   
         enResult = eUpdateRequestingClient(poMessage);
         if (enResult != AIL_EN_N_NO_ERROR)
         {  
             ETG_TRACE_FATAL(("%x vGenericSetHandler: eUpdateRequestingClient returns error %x function ID %x", 
                              ETG_ENUM(ail_u16AppId, (tU16)_u16AppID) , enResult, u16FunctionId ));
         }
      }

      // Notify all registered clients about a change of the property (if it has changed)
      if(bPropertyChanged == TRUE)
      {
         // Update 'status' for all registered clients
         enResult = eUpdateClients(u16FunctionId, poMessage);
         if (enResult != AIL_EN_N_NO_ERROR)
         {
            // We never expect to come here.
            ETG_TRACE_FATAL(("%x vGenericSetHandler: eUpdateClients returns error %x function ID %x", 
                             ETG_ENUM(ail_u16AppId, (tU16)_u16AppID) , enResult, u16FunctionId ));
         }
      }
   }
   else
   {
      // Create AMT error message and set ErrorCode 
      amt_tclServiceDataError oErrorMsg(*poMessage, u16Error);
      // NOTE: The constructor of amt_tclServiceDataError is re-using the 
      // received message 'poMessage' It will collect all necessary Service 
      // Information (Source-, Target AppID, RegID, ..) and exchange the 
      // Source- and Target AppID and even set the Opcode to ERROR. 
      // Additionally the ErrorCode value is set to the second provided 
      // parameter. For the new message new memory is allocated, i.e. the
      // received message 'poMessage' is not re-used in terms of memory and the
      // implicite call of 'poMessage->bDelete();' by the framework after 
      // leaving this handler function does not harm.

      // Post message
      ail_tenCommunicationError enResult = ePostMessage(&oErrorMsg);
         
      if (enResult != AIL_EN_N_NO_ERROR)
      {
         // We can come here e.g. if the client has changed his application 
         // state to OFF in the meantime. Therefore we don't throw an assert.
         // NORMAL_M_ASSERT_ALWAYS();
         ETG_TRACE_SYS_MIN(("%x vGenericSetHandler: send error message returns error %x function ID %x", 
                            ETG_ENUM(ail_u16AppId, (tU16)_u16AppID) , enResult, u16FunctionId ));
      }
   }

   // NOTE: There is no need to call 'poMessage->bDelete();' here, as the 
   // incoming message will be deleted by the CCX framework automatically when 
   // leaving this function.
}

tBool ahl_tclBaseOneThreadService::bProcessSet(amt_tclServiceData* /*poMessage*/, tBool& /*bPropertyChanged*/, tU16& /*u16Error*/)
{
   return FALSE;
}

tBool ahl_tclBaseOneThreadService::bProcessIncDec(amt_tclServiceData* /*poMessage*/, tBool& /*bPropertyChanged*/, tU16& /*u16Error*/)
{
   return FALSE;
}

tVoid ahl_tclBaseOneThreadService::vSetConfiguration(const ahl_sServiceConfig& oServiceConfig)
{
   _bAutoHandleGet =  oServiceConfig.bAutoHandleGet;
   _bAutoHandleSet =  oServiceConfig.bAutoHandleSet;
   _bAutoHandleIncDec = oServiceConfig.bAutoHandleIncDec;
   _bEnableHighPrioFIDs = oServiceConfig.bEnableHighPrioFIDs;
}

/*************************************************************************
*
* FUNCTION:tVoid vOnUnknownMessage(amt_tclBaseMessage* poMessage)
* 
* DESCRIPTION: handle unkown message
*
* PARAMETER:  unkown message
*
* RETURNVALUE: tVoid
*
*************************************************************************/
tVoid ahl_tclBaseOneThreadService::vOnUnknownMessage( amt_tclBaseMessage* poMessage )
{
  /* +++
   the frame work calls this function when it receives a message with a unknown 
   FID or a message which is not handled in a client. You can use it for error handling.
  +++ */

   if(poMessage->u8GetType() == AMT_C_U8_CCAMSGTYPE_SVCDATA)
   {
      amt_tclServiceData oServiceData(poMessage);
      ETG_TRACE_ERR(("%x vOnUnknownMessage: SVDData from App=%x OpCode=%d SRV=%x FID=%d", 
                 ETG_ENUM(ail_u16AppId, (tU16)_u16AppID), 
                 ETG_ENUM(ail_u16AppId, (tU16)oServiceData.u16GetSourceAppID()), 
                 ETG_ENUM(ail_u8OperationCode, (tU8)oServiceData.u8GetOpCode()), 
                 ETG_ENUM(ail_u16ServiceId, (tU16)oServiceData.u16GetServiceID()), 
                 oServiceData.u16GetFunctionID() ));
   }
   else
   {
      ETG_TRACE_ERR(("%x vOnUnknownMessage: SVDData from App=%x Type=%d", 
                 ETG_ENUM(ail_u16AppId, (tU16)_u16AppID), 
                 ETG_ENUM(ail_u16AppId, (tU16)poMessage->u16GetSourceAppID()), 
                 poMessage->u8GetType() ));
   }

  if (!poMessage->bDelete())
  {
     /* +++ delete failed enter error handling here +++ */  
     ETG_TRACE_ERRMEM(("%x vOnUnknownMessage: deleted of Message failed", ETG_ENUM(ail_u16AppId, (tU16)_u16AppID)  ));
  }
}

tBool ahl_tclBaseOneThreadService::bIsTargetFor(tU16 u16ServiceId, tU16 /* u16AppId */, tU16 /* u16RegisterId */, tU16 /* u16AppSubId */) const
{
   if (u16ServiceId == _u16ServiceID)
   {
      return TRUE;
   }

   return FALSE;
}

tVoid ahl_tclBaseOneThreadService::vRemoveRegistration()
{
}

tBool ahl_tclBaseOneThreadService::bSetFIDasHighPrio(tU16 nFID)
{
  if (sHighPrioFIDs.insert(nFID).second)
  {
     return TRUE;
  }
  else
  {
     // means element is still in list (double insert)
     return FALSE;
  }
}

tBool ahl_tclBaseOneThreadService::bSetMethodHandlingType(tU16 u16FktID, ahl_tenMethodMultipleRequestHandlingType eHandlingType, tU32 nTimeout)
{
   return clMethodTable.bSetMethodHandlingType(u16FktID, eHandlingType, nTimeout);
}

