
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#include "utf8_StringUtil.h"

/**
 * This function cuts off a faulty UTF8 character.
 * It is possible that bytes of an UTF8 string are cut off when the string is
 * written to a buffer if the buffer is too small. If the last character is
 * coded as more than one byte, there might be some bytes missing at the end.
 * This function cuts off the tailing bytes until the last character is well
 * coded.\n
 * It starts at the last byte of a string and checks the type of byte. There
 * are three cases possible:\n
 * 1st: \nIt is part of a multi byte character (10xxxxxx):
 *      Walk in head direction and count the number of bytes until the first
 *      byte of the multi bytes character and check if its coded size matches.
 *      If not, the last character is cut off (the first byte of the multi byte
 *      character is set to NULL).\n
 * 2nd: \nThe last byte is the first character of a multi byte character (11xxxxxx):
 *      Cut off this byte, because following bytes are missing.\n
 * 3rd: \nThe last byte is a single byte coded character:
 *      --> do nothing
 *
 * @param szString        string to be checked
 * @param u32BufferLength length of the buffer
 * @pre   szString is not NULL
 * @pre   szString was cut off at the end
 * @attention This function only works if the string was cut off at the end!
 *
 * @author CM-DI/ESP2-Brandes
 * @date   21.03.2005
 */
tVoid UTF8_vCutOffFaultyCharacter( tString szString, tU32 u32BufferLength )
{
   tString szCurrByte = &szString[u32BufferLength-1];

   if( 1 >= u32BufferLength )
      return;

   // set terminating null if it is missing
   if( 0 != *szCurrByte )
      *szCurrByte = 0;

   --szCurrByte;

   // check if byte is not byte of a single character
   if( 0 == UTF8_s32IsSingleByteCodedChar( szCurrByte ) )
   {
      if( 0 == UTF8_s32IsPartOfMultiByteChar( szCurrByte ) )
      {
         // Th current byte is begin of a multi byte character, but following
         // bytes were cut off --> cut off this byte too.
         *szCurrByte = 0;
      }
      else
      {
         tU32 u32CntBytesOfChar = 2;

         // go back one more byte
         --szCurrByte;

         // go to the first byte of the multi byte character
         while( 0 != UTF8_s32IsPartOfMultiByteChar( szCurrByte ) )
         {
            ++u32CntBytesOfChar;
            --szCurrByte;
         }

         // Cut off last bytes if the coded number of bytes does not
         // match with the determined one.
         if( u32CntBytesOfChar != UTF8_u32GetCharacterByteLength(szCurrByte) )
         {
            *szCurrByte = 0;
         }
      }
   }
}

/**
 * This function determines the next character of given character.
 *
 * @param szCurrentCharacter the current character
 * @pre   szCurrentCharacter is not NULL
 * @return pointer to next character or NULL if end of string is reached
 * @author CM-DI/ESP2-Brandes
 * @date   21.03.2005
 */
tString UTF8_szGetNextCharacter( tString szCurrentCharacter )
{
   tU8* pu8CurrByte = (tU8*) szCurrentCharacter;
   tU32 u32CharByteLength = UTF8_u32GetCharacterByteLength(szCurrentCharacter);

   if( 0 == u32CharByteLength )
   {
      // pointer is not the first byte of a multi byte character
      while( NULL != pu8CurrByte && 0 != UTF8_s32IsPartOfMultiByteChar( (tCString)pu8CurrByte ) )
      {
         ++pu8CurrByte;
      }
   }
   else
   {
      pu8CurrByte += u32CharByteLength;
   }

   // set pointer to NULL if end of string is reached
   if( 0 == *pu8CurrByte ) //lint !e794
      pu8CurrByte = NULL;

   return (tString)pu8CurrByte;
}

/**
 * This function determines the next character of given character.
 *
 * @param szCurrentCharacter the current character
 * @pre   szCurrentCharacter is not NULL
 * @return pointer to next character or NULL if end of string is reached
 * @author CM-DI/ESP2-Brandes
 * @date   21.03.2005
 */
tCString UTF8_coszGetNextCharacter( tCString szCurrentCharacter )
{
   const tU8* pcu8CurrByte = (const tU8*) szCurrentCharacter;
   tU32 u32CharByteLength = UTF8_u32GetCharacterByteLength(szCurrentCharacter);

   if( 0 == u32CharByteLength )
   {
      // pointer is not the first byte of a multi byte character
      while( NULL != pcu8CurrByte && 0 != UTF8_s32IsPartOfMultiByteChar( (tCString)pcu8CurrByte ) )
      {
         ++pcu8CurrByte;
      }
   }
   else
   {
      pcu8CurrByte += u32CharByteLength;
   }

   // set pointer to NULL if end of string is reached
   if( 0 == *pcu8CurrByte ) //lint !e794
      pcu8CurrByte = NULL;

   return (tCString)pcu8CurrByte;
}


/**
 * This function is the equivalent to the function OSALUTIL_s32SaveNPrintFormat.
 * It writes formatted data into the given buffer and takes care for a null
 * termination (by using the function UTF8_vCutOffFaultyCharacter).
 *
 * @param szDest          buffer the string is written to
 * @param u32BufferLength length of given buffer
 * @param coszFormat      string which contains the format behaviour
 * @param ...             list of parameter
 * @pre   szDest is not NULL
 * @pre   coszFormat is not NULL
 * @return the number of characters printed, or a negative value if an error occurs
 * @author CM-DI/ESP2-Brandes
 * @date   21.03.2005
 */
tS32 UTF8_s32SaveNPrintFormat( tString szDest, tU32 u32BufferLength, tCString coszFormat, ... )
{
   OSAL_tVarArgList args;
   OSAL_VarArgStart(args,coszFormat);//lint !e530 (this function fills "args")
	
   tS32 result = UTF8_s32SaveVarNPrintFormat(szDest,u32BufferLength,coszFormat,args); //lint !e530 ANSI, no init necessary
   OSAL_VarArgEnd(args);

   return result;
}

/**
 * This function is the equivalent to the function
 * OSALUTIL_s32SaveVarNPrintFormat. It writes formatted data into the given
 * buffer and takes care for a null termination (by using the function
 * UTF8_vCutOffFaultyCharacter).
 *
 * @param szDest          buffer the string is written to
 * @param u32BufferLength length of given buffer
 * @param coszFormat      string which contains the format behaviour
 * @param ...             list of parameter
 * @pre   szDest is not NULL
 * @pre   coszFormat is not NULL
 * @return returns the number of characters stored in buffer, not counting the terminating null character
 * @author CM-DI/ESP2-Brandes
 * @date   21.03.2005
 */
tS32 UTF8_s32SaveVarNPrintFormat( tString szDest, tU32 u32BufferLength, tCString coszFormat, va_list argptr )
{
   tS32 s32Result = OSALUTIL_s32SaveVarNPrintFormat( szDest, u32BufferLength, coszFormat, argptr );
   
   if( 0 > s32Result || (tU32)s32Result >= u32BufferLength )
   {
      // string was cut off --> check last character
      UTF8_vCutOffFaultyCharacter( szDest, u32BufferLength );
   }

   return (tS32)UTF8_u32StringCharCount( szDest );
}

/**
 * This function is the equivalent to the function OSALUTIL_ szSaveStringNCopy.
 * It copies data into the given buffer and takes care for a null termination
 * (by using the function UTF8_vCutOffFaultyCharacter).
 *
 * @param szDest          buffer the string is written to
 * @param coszSource      string be be copied
 * @param u32BufferLength length of given buffer
 * @pre   szDest is not NULL
 * @pre   coszSource is not NULL
 * @return pointer to destination string
 * @author CM-DI/ESP2-Brandes
 * @date   22.03.2005
 */
tString UTF8_szSaveStringCopy( tString szDest, tCString coszSource, tU32 u32BufferLength )
{
   OSALUTIL_szSaveStringNCopy( szDest, coszSource, u32BufferLength );

   tU32 u32CntBytes = OSAL_u32StringLength( szDest ) + 1;
   UTF8_vCutOffFaultyCharacter( szDest, u32CntBytes );

   return szDest;
}

/**
 * This function is the equivalent to the function OSALUTIL_ szSaveStringNCopy.
 * It copies data into the given buffer and takes care for a null termination
 * (by using the function UTF8_vCutOffFaultyCharacter).
 *
 * @param szDest          buffer the string is written to
 * @param coszSource      string be be copied
 * @param u32BufferLength length of given buffer
 * @param u32CntChar      number of characters to be copied
 * @pre   szDest is not NULL
 * @pre   coszSource is not NULL
 * @return pointer to destination string
 * @author CM-DI/ESP2-Brandes
 * @date   21.03.2005
 */
tString UTF8_szSaveStringNCopy( tString szDest, tCString coszSource, tU32 u32BufferLength, tU32 u32CntChar )
{
   tU32 u32CntBytes = UTF8_u32GetNCharacterByteLength( coszSource, u32CntChar )+1;

   if( u32CntBytes > u32BufferLength )
      u32CntBytes = u32BufferLength;
   
   OSALUTIL_szSaveStringNCopy( szDest, coszSource, u32CntBytes );

   UTF8_vCutOffFaultyCharacter( szDest, u32BufferLength );

   return szDest;
}

/**
 * This function is the equivalent to the function
 * OSALUTIL_ szSaveStringNConcat. It appends one string to another one and
 * takes care for a null termination (by using the function
 * UTF8_vCutOffFaultyCharacter).
 *
 * @param szDest          buffer the string is written to
 * @param coszSource      string be be copied
 * @param u32BufferLength length of given buffer (size of whole buffer including the already used bytes) 
 * @pre   szDest is not NULL
 * @pre   coszSource is not NULL
 * @return a pointer to the destination string
 * @author CM-DI/ESP2-Brandes
 * @date   21.03.2005
 */
tString UTF8_szSaveStringConcat( tString szDest, tCString coszSource, tU32 u32BufferLength )
{
   // calculate the number of free bytes
   tU32 u32FreeBufferSize = u32BufferLength;
   if( NULL != szDest && 0 != szDest[0] )
   {
      // sample:
      // buffer size is 8: 12345678
      // string in buffer: Hello
      // --> 3 Bytes left:      xxx
      // In this case 3 bytes are left for the string to be appended
      tU32 u32UsedBytesOfBuffer = OSAL_u32StringLength(szDest);
      if( u32UsedBytesOfBuffer < u32BufferLength )
      {
         u32FreeBufferSize -= u32UsedBytesOfBuffer;
      }
      else
      {
         // no space left to append anything of string 2
         u32FreeBufferSize = 0;
      }
   
      // only append something if more than 1 byte is free
      // (one byte is needed for termination null)
      if( u32FreeBufferSize > 1 )
      {
         // OSALUTIL_szSaveStringNConcat will append up to n bytes:
         //  -n bytes if the given string is too long
         //  -less than n bytes if string can be added completely
         // The termination "0" is added in any case.
         OSALUTIL_szSaveStringNConcat( szDest, coszSource, u32FreeBufferSize );
      }

      UTF8_vCutOffFaultyCharacter( szDest, u32BufferLength );
   }

   return szDest;
}

/**
 * This function is the equivalent to the function
 * OSALUTIL_ szSaveStringNConcat. It appends one string to another one and
 * takes care for a null termination (by using the function
 * UTF8_vCutOffFaultyCharacter).
 *
 * @param szDest          buffer the string is written to
 * @param coszSource      string be be copied
 * @param u32BufferLength length of given buffer
 * @param u32CntChar      number of character to be appended
 * @pre   szDest is not NULL
 * @pre   coszSource is not NULL
 * @return a pointer to the destination string
 * @author CM-DI/ESP2-Brandes
 * @date   21.03.2005
 */
tString UTF8_szSaveStringNConcat( tString szDest, tCString coszSource, tU32 u32BufferLength, tU32 u32CntChar )
{
   tU32 u32CntBytesOfNullTermAppStr = 
      UTF8_u32GetNCharacterByteLength( coszSource, u32CntChar )+1;// count NULL termination too

   // get pointer to the end of first string
   tChar* pszEndOfFirstString = szDest;
   tU32 u32CntBytesStr1 = OSAL_u32StringLength( szDest );
   pszEndOfFirstString += u32CntBytesStr1;

   // string can only be appended if buffer is greater than stored string
   if( u32BufferLength > u32CntBytesStr1+1 )
   {
      tU32 u32CntNecessaryBytes = u32CntBytesStr1 + u32CntBytesOfNullTermAppStr;
      if( u32CntNecessaryBytes > u32BufferLength )
      {
         u32CntBytesOfNullTermAppStr = u32BufferLength-u32CntBytesStr1;
      }

      OSAL_pvMemoryCopy( pszEndOfFirstString, coszSource, u32CntBytesOfNullTermAppStr );

      szDest[u32CntBytesStr1+u32CntBytesOfNullTermAppStr-1] = '\0';
   
      UTF8_vCutOffFaultyCharacter( szDest, u32BufferLength );
   }
   return szDest;
}

/**
 * This function is an equivalent to the function OSAL_s32StringNCompare.
 * It compares up to u32CntChar.
 *
 * @param coszStr1   first string to be compared
 * @param coszStr2   second string to be compared
 * @param u32CntChar number of characters to be compared
 * @pre coszStr1 is not NULL
 * @pre coszStr2 is not NULL
 * @return < 0 string1 less than string2 \n
 *           0 string1 identical to string2 \n
 *         > 0 string1 greater than string2
 * @author CM-DI/ESP2-Brandes
 * @date   21.03.2005
 */
tS32 UTF8_s32StringNCharCompare( tCString coszStr1, tCString coszStr2, tU32 u32CntChar )
{
   tU32 u32CntCharStr1 = UTF8_u32StringCharCount(coszStr1, u32CntChar);
   tU32 u32CntCharStr2 = UTF8_u32StringCharCount(coszStr2, u32CntChar);

   tU32 u32CntCharToCheck = u32CntCharStr1;
   if( u32CntCharStr2 < u32CntCharToCheck)
      u32CntCharToCheck = u32CntCharStr2;
   if( u32CntChar < u32CntCharToCheck)
      u32CntCharToCheck = u32CntChar;

   tU32 u32CntBytesToCheck = UTF8_u32GetNCharacterByteLength(coszStr1,u32CntCharToCheck);

   tS32 s32Result = OSAL_s32MemoryCompare( coszStr1, coszStr2, u32CntBytesToCheck );

   // If the compared characters are equal and the number of characters to
   // compare was modified, the string length has to be analyzed.
   if( 0 == s32Result && u32CntCharToCheck != u32CntChar && u32CntCharStr1 != u32CntCharStr2)
   {
      if( u32CntCharStr1 < u32CntCharStr2 )
      {
         s32Result = -1;
      }
      else
      {
         s32Result = 1;
      }
   }
   return s32Result;
}

/**
 * This function searches the first occurrence of a character within a string.
 *
 * @param coszSource     string in which should be searched
 * @param coszSearchChar character to be searched (it needs not be NULL terminated)
 * @pre   coszSource is not NULL
 * @pre   coszSearchChar is not NULL
 * @return a pointer to the first occurrence of coszSearchChar in string,
 *         or NULL if coszSearchChar is not found
 * @author CM-DI/ESP2-Brandes
 * @date   21.03.2005
 */
tCString UTF8_ps8StringSearchChar( tCString coszSource, tCString coszSearchChar )
{
   tChar caBuffer[7];

   // copy current character into buffer so that the NULL termination can be added
   tU32 u32CharByteLength = UTF8_u32GetCharacterByteLength(coszSearchChar);
   OSAL_pvMemoryCopy( caBuffer, coszSearchChar, u32CharByteLength );
   caBuffer[u32CharByteLength] = '\0';

   return OSAL_ps8StringSubString( coszSource, caBuffer );
}

/**
 * This function scans a string for the last occurrence of a character
 *
 * @param coszSource     string in which should be searched
 * @param coszSearchChar null terminated character to be searched
 * @pre   coszSource is not NULL
 * @pre   coszSearchChar is not NULL
 * @return a pointer to the last occurrence of coszSearchChar in string,
 *         or NULL if coszSearchChar is not found
 * @author CM-DI/ESP2-Brandes
 * @date   21.03.2005
 */
tCString UTF8_ps8StringRSearchChar( tCString coszSource, tCString coszSearchChar )
{
   tCString ps8LastOccurrence = UTF8_ps8StringSearchChar(coszSource,coszSearchChar);

   if( NULL != ps8LastOccurrence )
   {
      tCString ps8FoundPos = UTF8_ps8StringSearchChar( ps8LastOccurrence+1, coszSearchChar );

      while( NULL != ps8FoundPos )
      {
         ps8LastOccurrence = ps8FoundPos;
         ps8FoundPos = UTF8_ps8StringSearchChar( ps8LastOccurrence+1, coszSearchChar );
      }
   }

   return ps8LastOccurrence;
}

/**
 * This function determines a value specifying the length of the substring in
 * coszSource that consists entirely of characters in coszCharSet. If coszSource 
 * begins with a character not in coszCharSet, the function returns 0.
 *
 * @param coszSource  string in which should be searched
 * @param coszCharSet characters to be searched
 * @pre   coszSource is not NULL
 * @pre   coszCharSet is not NULL
 * @return an integer value specifying the length of the substring in string
 *         that consists entirely of characters in coszCharSet. If string
 *         begins with a character not in coszCharSet, the function returns 0
 * @author CM-DI/ESP2-Brandes
 * @date   21.03.2005
 */
tU32 UTF8_u32StringSegment( tCString coszSource, tCString coszCharSet )
{
   tCString coszCurrChar = coszSource;
   tU32 u32CntFoundChar = 0;

   while( NULL != coszCurrChar && 0 != *coszCurrChar)
   {
      // search current character in given character set
      if( NULL == UTF8_ps8StringSearchChar(coszCharSet,coszCurrChar) )
      {
         // leave loop
         coszCurrChar = NULL;
      }
      else
      {
         ++u32CntFoundChar;

         // get next character
         coszCurrChar = UTF8_coszGetNextCharacter(coszCurrChar);
      }
   }
   return u32CntFoundChar;
}

/**
 * This function determines a value value specifying the length of the initial
 * segment of coszSource that consists entirely of characters not in coszCharSet. 
 * If coszSource begins with a character that is in coszCharSet, the function returns 0.
 *
 * @param coszSource  string in which should be searched
 * @param coszCharSet characters to be searched
 * @pre   coszSource is not NULL
 * @pre   coszCharSet is not NULL
 * @return an integer value specifying the length of the substring in string
 *         that consists entirely of characters not in coszCharSet. If string
 *         begins with a character that is in coszCharSet, the function returns 0
 * @author CM-DI/ESP2-Brandes
 * @date   21.03.2005
 */
tU32 UTF8_u32StringNotSegment( tCString coszSource, tCString coszCharSet )
{
   tCString coszCurrChar = coszSource;
   tU32 u32CntNonFoundChar = 0;

   while( NULL != coszCurrChar && 0 != *coszCurrChar)
   {
      // search current character in given character set
      if( NULL == UTF8_ps8StringSearchChar(coszCharSet,coszCurrChar) )
      {
         ++u32CntNonFoundChar;

         // get next character
         coszCurrChar = UTF8_coszGetNextCharacter(coszCurrChar);
      }
      else
      {
         // leaver loop
         coszCurrChar = NULL;
      }
   }
   return u32CntNonFoundChar;
}

/**
 * This function scans strings for characters in specified character sets.
 * It searches for the first occurrence of a character given by character
 * set in the given string (equivalent to ANSI strpbrk).
 *
 * @param coszSource  string in which should be searched
 * @param coszCharSet characters to be searched
 * @pre   coszSource is not NULL
 * @pre   coszCharSet is not NULL
 * @return a pointer to the first occurrence of any character from coszCharSet
 *         in string, or a NULL pointer if the two string arguments have no
 *         characters in common
 * @author CM-DI/ESP2-Brandes
 * @date   21.03.2005
 */
tCString UTF8_ps8StringBreak( tCString coszSource, tCString coszCharSet )
{
   tCString ps8FirstHit = NULL;
   tCString ps8CurrHit  = NULL;
   tCString coszCurrChar = coszCharSet;

   // lopp over all characters in character set
   while( NULL != coszCurrChar && 0 != *coszCurrChar )
   {
      ps8CurrHit = UTF8_ps8StringSearchChar(coszSource,coszCurrChar);

      // get next character
      coszCurrChar = UTF8_coszGetNextCharacter(coszCurrChar);

      // take over found hit
      if( NULL != ps8CurrHit &&
          ( NULL == ps8FirstHit || ps8CurrHit < ps8FirstHit ) )
      {
         ps8FirstHit = ps8CurrHit;
      }
   }

   return ps8FirstHit;
}

/**
 * This function determines the number of character of given string.
 * It starts at the begin of the string and walks up to the end.
 * When counting there are two cases possible:\n
 * 1st: \nCurrent byte is a single coded character (0xxxxxxx) \n -->
 *      increase number of characters and go to the next byte.\n
 * 2nd: \nCurrent byte is the first byte of a multi byte character (11xxxxxx) \n -->
 *      increase number of characters, get count coded number of used bytes and
 *      jump to the first byte of the next character.
 *
 * @param coszStr string of which the characters should be counted
 * @pre   coszStr is not NULL
 * @return the number of characters in string, excluding the terminating NULL
 * @author CM-DI/ESP2-Brandes
 * @date   21.03.2005
 */
tU32 UTF8_u32StringCharCount( tCString coszStr )
{
   return UTF8_u32StringCharCount( coszStr, LIMIT_C_MAX_U32 );
}

tU32 UTF8_u32StringCharCount( tCString coszStr, tU32 u32MaxCntChar )
{
   const tU8* pcu8CurrByte = (const tU8*)coszStr;
   tU32 u32CntChar = 0;
   tU32 u32CharByteLength = 0;

   if (NULL != pcu8CurrByte)
   {
      while( 0 != *pcu8CurrByte && u32CntChar < u32MaxCntChar)
      {
         if( 0 != UTF8_s32IsSingleByteCodedChar( (tCString)pcu8CurrByte ) )
         {
            // single byte coded char
            ++pcu8CurrByte;
         }
         else
         {
            // begin of multi byte character
            u32CharByteLength = UTF8_u32GetCharacterByteLength( (tCString)pcu8CurrByte );

            // check if length is min one byte (may be 0 if it is not the first
            // of the character)
            if( 0 == u32CharByteLength )
               ++u32CharByteLength;
          
            pcu8CurrByte += u32CharByteLength;
         }
         ++u32CntChar;
      }
   }
   return u32CntChar;
}


/**
 * This function checks if given character is alphanumeric
 * ('A'-'Z', 'a'-'z' or '0'-'9')
 *
 * @param coszStr character to be checked
 * @pre coszStr is not NULL
 * @attention This function only works with single byte characters.
 * @return a non-zero value if coszStr is within the ranges
 *         A \96 Z, a \96 z, or 0 \96 9
 * @author CM-DI/ESP2-Brandes
 * @date   21.03.2005
 */
tS32 UTF8_s32IsAlphaNum( tCString coszStr )
{
   if( 0 == UTF8_s32IsSingleByteCodedChar( coszStr ) )
      return 0;

   return OSAL_s32IsAlphaNum( *((tCU8*)coszStr) );
}

/**
 * This function checks if given character is alpha
 * ('A'-'Z' or 'a'-'z')
 *
 * @param coszStr character to be checked
 * @pre coszStr is not NULL
 * @return a non-zero value if coszStr is within the ranges
 *         A \96 Z or a \96 z
 * @author CM-DI/ESP2-Brandes
 * @date   21.03.2005
 */
tS32 UTF8_s32IsAlpha( tCString coszStr )
{
   if( 0 == UTF8_s32IsSingleByteCodedChar( coszStr ) )
      return 0;

   return OSAL_s32IsAlpha( *((tCU8*)coszStr) );
}

/**
 * This function checks if given character is numeric ('0'-'9').
 *
 * @param coszStr character to be checked
 * @pre coszStr is not NULL
 * @return a non-zero value if coszStr is within the ranges
 *         0 \96 9
 * @author CM-DI/ESP2-Brandes
 * @date   21.03.2005
 */
tS32 UTF8_s32IsDigit( tCString coszStr )
{
   if( 0 == UTF8_s32IsSingleByteCodedChar( coszStr ) )
      return 0;

   return OSAL_s32IsDigit( *((tCU8*)coszStr) );
}

/**
 * This function checks if given character is a particular representation
 * of a space character
 *
 * @param coszStr character to be checked
 * @pre coszStr is not NULL
 * @return a non-zero value if coszStr is a white-space character
 *         (0x09 \96 0x0D or 0x20)
 * @author CM-DI/ESP2-Brandes
 * @date   21.03.2005
 */
tS32 UTF8_s32IsSpace( tCString coszStr )
{
   if( 0 == UTF8_s32IsSingleByteCodedChar( coszStr ) )
      return 0;

   return OSAL_s32IsSpace( *((tCU8*)coszStr) );
}

/**
 * This function checks if given character is a particular representation
 * of a uppercase letter (A - Z)
 *
 * @param coszStr character to be checked
 * @pre coszStr is not NULL
 * @return a non-zero value if coszStr is a uppercase letter
 * @author CM-DI/ESP2-Brandes
 * @date   21.03.2005
 */
tS32 UTF8_s32IsUpper( tCString coszStr )
{
   if( 0 == UTF8_s32IsSingleByteCodedChar( coszStr ) )
      return 0;

   return OSAL_s32IsUpper( *((tCU8*)coszStr) );
}

/**
 * This function checks if given character is a particular representation
 * of a lowercase letter (a \96 z)
 *
 * @param coszStr character to be checked
 * @pre coszStr is not NULL
 * @return a non-zero value if coszStr is a lowercase letter
 * @author CM-DI/ESP2-Brandes
 * @date   21.03.2005
 */
tS32 UTF8_s32IsLower( tCString coszStr )
{
   if( 0 == UTF8_s32IsSingleByteCodedChar( coszStr ) )
      return 0;

   return OSAL_s32IsLower( *((tCU8*)coszStr) );
}

/**
 * This function checks if given character is a particular representation
 * of a hexadecimal digit
 *
 * @param coszStr character to be checked
 * @pre coszStr is not NULL
 * @return a non-zero value if coszStr is a hexadecimal digit
 *         (A \96 F, a \96 f, or 0 \96 9).
 * @author CM-DI/ESP2-Brandes
 * @date   21.03.2005
 */
tS32 UTF8_s32IsHexDigit( tCString coszStr )
{
   if( 0 == UTF8_s32IsSingleByteCodedChar( coszStr ) )
      return 0;

   return OSAL_s32IsHexDigit( *((tCU8*)coszStr) );
}

/**
 * This function checks if given character is a backspace.
 *
 * @param coszStr character to be checked
 * @pre coszStr is not NULL
 * @return a non-zero value if coszStr is a backspace (\b)
 *         
 * @author CM-DI/ESP2-Brandes
 * @date   12.08.2005
 */
tS32 UTF8_s32IsBackspace( tCString coszStr )
{
   if( 0 == UTF8_s32IsSingleByteCodedChar( coszStr ) )
      return 0;

   return *((tCU8*)coszStr) == '\b';
}

/**
 * This function converts character to lowercase one.
 *
 * @param coszStr character to be converted
 * @pre coszStr is not NULL
 * @return 0 if given string uses more than one byte,
 *         otherwise the (converted) value
 * @attention This function only converts one byte values in range from
 *            'A' - 'Z'.
 * @author CM-DI/ESP2-Brandes
 * @date   21.03.2005
 */
tS32 UTF8_s32ToLower( tCString coszStr )
{
   if( 0 == UTF8_s32IsSingleByteCodedChar( coszStr ) )
      return 0;

   return OSAL_s32ToLower( *((tCU8*)coszStr) );
}

/**
 * This function converts character to uppercase one.
 *
 * @param coszStr   character to be converted
 * @pre coszStr is not NULL
 * @return 0 if given string uses more than one byte,
 *         otherwise the (converted) value
 * @attention This function only converts one byte values in range from
 *            'a' - 'z'.
 * @author CM-DI/ESP2-Brandes
 * @date   21.03.2005
 */
tS32 UTF8_s32ToUpper( tCString coszStr )
{
   if( 0 == UTF8_s32IsSingleByteCodedChar( coszStr ) )
      return 0;

   return OSAL_s32ToUpper( *((tCU8*)coszStr) );
}

/**
 * This function counts the number of equal characters from begining till a mismatch is found.
 *
 * @param coszStr1 first string to be compared
 * @param coszStr2 second string to be compared
 * @pre coszStr1 is not NULL
 * @pre coszStr2 is not NULL
 * @return the number of equal character
 * @author CM-DI/ESP2-Brandes
 * @date   21.03.2005
 */
tU32 UTF8_u32GetCntEqualChar( tCString coszStr1, tCString coszStr2 )
{
   tCString copszStr1 = coszStr1;
   tCString copszStr2 = coszStr2;

   tU32 u32CntEqualChar = 0;

   while( *copszStr1 != 0 &&
          *copszStr2 != 0 &&
          *copszStr1 == *copszStr2 )
   {
      if( 0 != UTF8_s32IsBeginOfMultiByteChar( copszStr1 ) )
      {
         // get length of bytes the character uses
         tU32 u32CharByteLength = UTF8_u32GetCharacterByteLength( copszStr1 );

         // copy memory
         if( 0 != OSAL_s32MemoryCompare( copszStr1, copszStr2, u32CharByteLength ) )
         {
            // string is different
            break;
         }

         ++u32CntEqualChar;

         // set pointer to begin of the next character
         copszStr1 += u32CharByteLength;
         copszStr2 += u32CharByteLength;

      }
      else
      {
         // is single byte character
         ++u32CntEqualChar;
         ++copszStr1;
         ++copszStr2;
      }
   }

   return u32CntEqualChar;
}

/**
 * This function counts the number of equal characters from begining till one of the string ends.
 *
 * @param coszStr1 first string to be compared
 * @param coszStr2 second string to be compared
 * @pre coszStr1 is not NULL
 * @pre coszStr2 is not NULL
 * @return the number of equal character
 * @author CM-AI-PJ-CF13 - ani1hi
 * @date   04.05.2011
 */
tU32 UTF8_u32GetCntAllEqualChar( tCString coszStr1, tCString coszStr2 )
{
   tCString copszStr1 = coszStr1;
   tCString copszStr2 = coszStr2;

   tU32 u32CntEqualChar = 0;

   while( *copszStr1 != 0 &&
          *copszStr2 != 0 )
   {
      if( 0 != UTF8_s32IsBeginOfMultiByteChar( copszStr1 ) )
      {
         // get length of bytes the character uses
         tU32 u32CharByteLength = UTF8_u32GetCharacterByteLength( copszStr1 );

         // copy memory
         if( 0 == OSAL_s32MemoryCompare( copszStr1, copszStr2, u32CharByteLength ) )
         {
            ++u32CntEqualChar;
         }
         // set pointer to begin of the next character
         copszStr1 += u32CharByteLength;
         copszStr2 += u32CharByteLength;
      }
      else
      {
         // is single byte character
         if (*copszStr1 == *copszStr2)
         {
            ++u32CntEqualChar;
         }
         ++copszStr1;
         ++copszStr2;
      }
   }

   return u32CntEqualChar;
}

/**
 * This function checks if the current character is part of a multibyte
 * character (and not the first character). It checks the high bits if
 * the first two ones are set to 10xxxxxx.
 *
 * @param coszChar byte to be checked
 * @pre coszChar is not NULL
 * @return a non zero value if given character is part of a multibyte
 *         characer, but not the first one
 * @author CM-DI/ESP2-Brandes
 * @date   21.03.2005
 */
tS32 UTF8_s32IsPartOfMultiByteChar( tCString coszChar )
{
   if( 0 != UTF8_s32IsBeginOfMultiByteChar(coszChar) ||
       *((tCU8*) coszChar) < C_MULTI_CHARACTER_BYTE )
   {
      return 0;
   }
   
   return 1;
}

/**
 * This function determines the number of bytes of a multibyte character.
 *
 * @param coszChar byte to be checked
 * @pre coszChar is not NULL
 * @return 0 if given pointer does not point to the first byte of a character,\n
 *         the number of used bytes for this character otherwise
 * @author CM-DI/ESP2-Brandes
 * @date   21.03.2005
 */
tU32 UTF8_u32GetCharacterByteLength( tCString coszChar )
{
   if( 0 == *((tCU8*)coszChar) || 0 != UTF8_s32IsPartOfMultiByteChar(coszChar) )
      return 0;

   tU32 u32CharByteLength = 1;
   tCU8* pu8Byte = (tCU8*) coszChar;

   // check if string is begin a multibyte character
   if( 0 != UTF8_s32IsBeginOfMultiByteChar( coszChar ) )
   {
      // number of bytes is two now, because min first two bits are set
      u32CharByteLength = 2;

      // check if third bit is set
      if( *pu8Byte >= 0xE0 )
      {
         ++u32CharByteLength;
         // check if fourth bit is set
         if( *pu8Byte >= 0xF0 )
         {
            ++u32CharByteLength;
            // check if fifth bit is set
            if( *pu8Byte >= 0xF8 )
            {
               ++u32CharByteLength;
               // check if sixth bit is set
               if( *pu8Byte >= 0xFC )
               {
                  ++u32CharByteLength;
               }
            }
         }
      }
   }

   return u32CharByteLength;
}

/**
 * This function determines the number of bytes of a multibyte character.
 *
 * @param coszString string whose characters should be checked
 * @param u32CntChar number of characters whose bytes should be counted
 * @pre coszString is not NULL
 * @return the number of bytes of the given number of characters
 * @author CM-DI/ESP2-Brandes
 * @date   21.03.2005
 */
tU32 UTF8_u32GetNCharacterByteLength( tCString coszString, tU32 u32CntChar )
{
   tU32 u32CntCheckedChar = 0;
   tU32 u32CntBytes = 0;
   tCString coszCurrChar = coszString;

   while( NULL != coszCurrChar && u32CntCheckedChar < u32CntChar )
   {
      u32CntBytes += UTF8_u32GetCharacterByteLength( coszCurrChar );
      coszCurrChar = UTF8_coszGetNextCharacter(coszCurrChar);
      ++u32CntCheckedChar;
   }

   return u32CntBytes;
}


