/************************************************************************
 * FILE:         cUtf8String
 * SW-COMPONENT: di_common
 *----------------------------------------------------------------------
 * DESCRIPTION:  This source file contains UTF8 string handling
 *----------------------------------------------------------------------
 * COPYRIGHT:    (c) 2000,2011 Robert Bosch GmbH, Hildesheim
 * HISTORY:      
 * Date      |   Modification               | Author
 *************************************************************************/

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include <string>
#include "osal_if.h"

#include "cUtf8String.h"

//Fatal Assert removed to avoid reset - ERR trace to be added
#define DEB_TRACE_ERROR_MSG_TO_ERRMEM(a) 


cUtf8StringBase::cUtf8StringBase()
{
   _pUtf8String    = NULL;
   _wUtf8StringLen = 0;
}

cUtf8StringBase::cUtf8StringBase(const tU8 *sUTF8Text)
{
   cUtf8StringBase::vAssign(sUTF8Text);
}

/****************************************************************************
* Function          : cUtf8StringBase::vAssign
*----------------------------------------------------------------------------
* Author            : H.Frericks
*----------------------------------------------------------------------------
* Description       : maps a given string to this class
*                     If param nBufferSize is zero, internal allocsize of the
*                     given string is strlen()+1
****************************************************************************/
void cUtf8StringBase::vAssign(const tU8 *sUTF8Text, tU32)
{
   _pUtf8String    = const_cast<tU8*>(sUTF8Text);                 // !note constuctor and vAssign const to prevent modifiations in base class
   DEB_TRACE_ERROR_MSG_TO_ERRMEM(_pUtf8String != 0);
   _wUtf8StringLen = OSAL_u32StringLength(sUTF8Text);
}

cUtf8StringBase::~cUtf8StringBase()
{
   _pUtf8String = NULL;
}

/****************************************************************************
* Function          : bIsPartOfMultiByteChar
*----------------------------------------------------------------------------
* Author            : H.Frericks
*----------------------------------------------------------------------------
* Description       : Checks if the given character is part of a multibyte 
*                     character and not the first character
****************************************************************************/
static bool bIsPartOfMultiByteChar( tU8 u8AnyByte )
{
   return ((u8AnyByte >= 0xC0 || u8AnyByte < 0x80) == false);
}

/****************************************************************************
* Function          : u32GetCharacterByteLengthOfUniCodeChar
*----------------------------------------------------------------------------
* Author            : CM-CR/ESD-Janssen
*----------------------------------------------------------------------------
* Description       : Returns the number of bytes this character occupies in
*                     UTF8 format.
****************************************************************************/
static tU32 u32GetCharacterByteLengthOfUniCodeChar (tU32 u32UniCodeChar)
{
   if (u32UniCodeChar < 0x80)
   {
       return 1;
   }
   if (u32UniCodeChar < 0x800)
   {
       return 2;
   }
   if (u32UniCodeChar < 0x10000)
   {
       return 3;
   }
   DEB_TRACE_ERROR_MSG_TO_ERRMEM(u32UniCodeChar < 0x200000);
   return 4;
}

/****************************************************************************
* Function          : u32GetCharacterByteLength
*----------------------------------------------------------------------------
* Author            : CM-CR/ESD-Janssen
*----------------------------------------------------------------------------
* Description       : Returns the number of bytes the character pointed to
*                     by pubUtf8Char consists of. A UTF8 character may 
*                     consists of 1 .. 4 bytes in this implementation.
****************************************************************************/
static tU32 u32GetCharacterByteLength (tU8 u8StartByte)
{
   if( (u8StartByte == 0) || bIsPartOfMultiByteChar(u8StartByte) )
   {
      DEB_TRACE_ERROR_MSG_TO_ERRMEM(0); 
      return 0;
   }
   if (u8StartByte < 0xC0)
   {   // standard ascii
       return 1;
   }
   if (u8StartByte < 0xE0)
   {   // 2  byte MultiByteChar
       return 2;
   }
   if (u8StartByte < 0xF0)
   {   // 3 byte MultiByteChar
       return 3;
   }
   if (u8StartByte < 0xF8)
   {   // 4 byte MultiByteChar
       return 4;
   }
   if (u8StartByte < 0xFC)
   {   // fifth bit is set, not supported unicode range
       DEB_TRACE_ERROR_MSG_TO_ERRMEM(0); 
       return 5;
   }
   // sixth bit is set, not supported unicode range
   DEB_TRACE_ERROR_MSG_TO_ERRMEM(0); 
   return 6;
}


/****************************************************************************
* Function          : cUtf8StringBase::blIsEmpty
*----------------------------------------------------------------------------
* Author            : CR/AEM Dahlhoff
*----------------------------------------------------------------------------
* Description       : Checks if the string is the empty string "".
****************************************************************************/
tBool cUtf8StringBase::bIsEmpty () const
{
   return (_wUtf8StringLen == 0);
}

/****************************************************************************
* Function          : cUtf8StringBase::bIsEqual
*----------------------------------------------------------------------------
* Author            : H.Frericks
*----------------------------------------------------------------------------
* Description       : Compare string doesn't work with UTF8 code strings, 
*                     so it returns true if the strings match.
****************************************************************************/

bool cUtf8StringBase::bIsEqual (const cUtf8StringBase * pString) const
{
   if(_pUtf8String == NULL || pString == NULL || pString->_pUtf8String == NULL)
       return false;

    return (OSAL_s32StringCompare(_pUtf8String, pString->_pUtf8String) == 0);
}

bool cUtf8StringBase::bIsEqual (const tU8 * pubCStr) const
{
   if(_pUtf8String == NULL || pubCStr == NULL)
       return false;
   
   return (OSAL_s32StringCompare(_pUtf8String, pubCStr) == 0);
}


/****************************************************************************
* Function          : cUtf8StringBase::pubGetCStr
*----------------------------------------------------------------------------
* Author            : CM-CR/ESD-Janssen
*----------------------------------------------------------------------------
* Description       : Returns a pointer to the zero-terminated string held
*                     by this string object.
****************************************************************************/
const tU8 * cUtf8StringBase::pubGetCStr () const
{
   return _pUtf8String;
}

/****************************************************************************
* Function          : cUtf8StringBase::pubGetCStr
*----------------------------------------------------------------------------
* Author            : Frericks
*----------------------------------------------------------------------------
* Description       : Returns a pointer to the zero-terminated string held
*                     by this string object.
****************************************************************************/
// most common equivalent
const tU8 * cUtf8StringBase::c_str () const
{
   return _pUtf8String;
}


/****************************************************************************
* Function          : cUtf8StringBase::uwGetLength
*----------------------------------------------------------------------------
* Author            : CM-CR/ESD-Janssen
*----------------------------------------------------------------------------
* Description       : Returns the length of the string in bytes, excluding a 
*                     terminating '\0' character.
****************************************************************************/
tU16 cUtf8StringBase::wGetLength () const
{
   return _wUtf8StringLen;
}
tU32 cUtf8StringBase::u32GetLength () const
{
   return _wUtf8StringLen;
}


/****************************************************************************
* Function          : cUtf8StringBase::itBegin
*----------------------------------------------------------------------------
* Author            : CM-CR/ESD-Janssen
*----------------------------------------------------------------------------
* Description       : Returns an iterator to the first character of the
*                     string.
****************************************************************************/
cUtf8StringIter cUtf8StringBase::itBegin () const
{
   return 0;
}


/****************************************************************************
* Function          : cUtf8StringBase::itEnd
*----------------------------------------------------------------------------
* Author            : CM-CR/ESD-Janssen
*----------------------------------------------------------------------------
* Description       : Returns an iterator to the character position behind
*                     the last character of the string.
*                     Can be used to determine whether an iterator has
*                     reached the end of the string.
****************************************************************************/
cUtf8StringIter cUtf8StringBase::itEnd () const
{
   return _wUtf8StringLen;
}


/****************************************************************************
* Function          : cUtf8StringBase::itNext
*----------------------------------------------------------------------------
* Author            : CM-CR/ESD-Janssen
*----------------------------------------------------------------------------
* Description       : Returns an iterator to the next character position
*                     relative to the given iterator.
*                     The caller does not need to know of how many bytes the
*                     characters consist.
****************************************************************************/
cUtf8StringIter cUtf8StringBase::itNext (const cUtf8StringIter iter) const
{
   if (_pUtf8String == NULL || iter >= _wUtf8StringLen)  
   {
      return itEnd(); 
   }
   if ((_pUtf8String[iter] & 0xC0) == 0x80)
   {  // Oops: not a starting byte */
      // bFixString(); // special case : function is here not const !!! very dirty hack 
      DEB_TRACE_ERROR_MSG_TO_ERRMEM(0); 
      return itEnd(); 
   }
   tU32 nCharByteLength = u32GetCharacterByteLength(_pUtf8String[iter]);
   if (nCharByteLength > 4)
   {  // not supported
      return itEnd(); 
   }
   if (iter >= _wUtf8StringLen)
   {  // Oops, ouside the string 
      DEB_TRACE_ERROR_MSG_TO_ERRMEM(0); 
      return itEnd(); 
   }
   return iter + nCharByteLength;
}


/****************************************************************************
* Function          : cUtf8StringBase::itPrevious
*----------------------------------------------------------------------------
* Author            : CM-CR/ESD-Janssen
*----------------------------------------------------------------------------
* Description       : Returns an iterator to the previous character position
*                     relative to the given iterator.
*                     The caller does not need to know of how many bytes the
*                     characters consists.
****************************************************************************/
cUtf8StringIter cUtf8StringBase::itPrevious (const cUtf8StringIter iter2) const
{
   if (_pUtf8String == NULL)  
   { 
      return itBegin();   // should be never happened, but saftey leave the loop
   }
   cUtf8StringIter iter = iter2;
   do
   {  if (iter == 0) // itBegin
      {
           DEB_TRACE_ERROR_MSG_TO_ERRMEM(0);  /* iterator can't be moved in front of string */
           return itBegin();
      }
      iter--;
   } while ((_pUtf8String[iter] >= 0x80) && ((_pUtf8String[iter] & 0xC0) == 0x80));

   if ((_pUtf8String[iter] >= 0xF8) 
      || ((iter + u32GetCharacterByteLength(_pUtf8String[iter])) != iter2))
   {  // not supported code || buggy string
      DEB_TRACE_ERROR_MSG_TO_ERRMEM(0);  
      return itBegin();
   }
   return iter;
}


/****************************************************************************
* Function          : cUtf8StringBase::GetCharCode
*----------------------------------------------------------------------------
* Author            : CM-DI/PJ-VW34 Dahlhoff
*----------------------------------------------------------------------------
* Description       : Calculates the 32 bit character Unicode for the UTF8
*                     character pointed to by the given iterator.
****************************************************************************/
tU32 cUtf8StringBase::GetCharCode (cUtf8StringIter iter) const
{
   tU16  ulwNumBytes, ulwResult, ulwIdx;
   tU8   ubB;

   if (_pUtf8String == NULL)  
   {
      return 0;   // should be never happened
   }
   // Check out-of-bounds access.
   if (iter >= itEnd()) // itEnd
   {
      if (iter > itEnd())
      {  // "ERROR: Access beyond end in cUtf8String::ulwGetCharCode
         DEB_TRACE_ERROR_MSG_TO_ERRMEM(FALSE);
      }
      return 0;
   }

   ubB = _pUtf8String[iter];
   if (ubB < 0x80)
   {
       // The most common case: an ASCII symbol in the range 0 to 127.
       return ubB;
   }
   // Is a multibyte (or an invalid second-byte). Count size.
   for (ulwNumBytes = 1; ulwNumBytes < 8; ulwNumBytes++)
   {
       if (!(ubB & (0x80 >> ulwNumBytes)))
       {
           break;
       }
   }
   // todo: frericks, optimieren auf 2 bytes, standardfall
   if (ulwNumBytes <= 1 || ulwNumBytes >= 8)
   {
       // Its an invalid second-byte (or or an 0xFF byte).
       // "Bad UTF-8 String or bad iterator. Found byte 0x02X." ,(unsigned int)ubB);
       DEB_TRACE_ERROR_MSG_TO_ERRMEM(0);
       return '!';
   }
   if (ulwNumBytes > (_wUtf8StringLen - iter))
   {
       // The multi-byte character would exceed the remaining bytes in the buffer. This is bad.
       // "Bad UTF-8 String or bad iterator. Not enough buffer content for a %u-byte symbol." ,(unsigned int)ulwNumBytes);
       DEB_TRACE_ERROR_MSG_TO_ERRMEM(0);
       return '!';
   }
   // Assemble the result and return it.
   ulwResult = ubB & (0xFFU >> ulwNumBytes);  
   for (ulwIdx = 1 ; ulwIdx < ulwNumBytes ; ulwIdx++)
   {
       ulwResult = (ulwResult << 6) | (_pUtf8String[iter + ulwIdx] & 0x3F) ;
       // ..... Todo: Do we need an overrun check for symbols with more than 32 bits?
   }
   // Done.
   return ulwResult;
}


/****************************************************************************
* Function          : cUtf8String::itFindChar ( strchr )
*----------------------------------------------------------------------------
* Author            : CM-CR/ESD-Janssen
*----------------------------------------------------------------------------
* Description       : Searches the string "self" for the character "ulwChar"
*                     starting at position "iter".
*                     If the character is found returns an iterator to the
*                     character, otherwise returns an iterator to the end of
*                     the string.
*                     !! works only with ascii < 0x80
****************************************************************************/
cUtf8StringIter cUtf8String::itFindChar (tU32 u32UniCodeChar, cUtf8StringIter iter) const
{
   if (_pUtf8String == NULL)  
   {
       return itEnd();   // should be never happened, but saftey leave the loop
   }
   if (bIsPartOfMultiByteChar(_pUtf8String[iter]) == true)
   {  // Oops
      DEB_TRACE_ERROR_MSG_TO_ERRMEM(FALSE);
      return itEnd();
   }

   if (u32UniCodeChar < 0x80)
   {
       /* ASCII characters can be handled with better performance */
       const tU8  * pubChar = &_pUtf8String[iter];
       while (*pubChar != '\0')
       {
           if (*pubChar == u32UniCodeChar)
           {
               return (cUtf8StringIter) (pubChar - _pUtf8String);
           }
           pubChar++;
       }
       return itEnd();
   }
   cUtf8String temp;
   temp.AllocMem(8);
   temp.vClear();
   temp.vAppendChar(u32UniCodeChar);
   return itStringSearchChar(temp);
}


/****************************************************************************
* Function          : cUtf8StringBase::StringSearchChar
*----------------------------------------------------------------------------
* Author            : Frericks
*----------------------------------------------------------------------------
* Description       : szSearchChar is character to be searched and needs not 
*                     be terminated.
*                     return the pointer or NULL if not found  
****************************************************************************/
const tU8* cUtf8StringBase::szStringSearchChar(const tU8* szSearchChar ) const
{
   if(szSearchChar == NULL || _pUtf8String == NULL)
       return NULL;

   tChar caBuffer[8];

   tU32 u32CharByteLength = u32GetCharacterByteLength(szSearchChar[0]);
   OSAL_pvMemoryCopy( caBuffer, szSearchChar, u32CharByteLength );
   caBuffer[u32CharByteLength] = '\0';
   return (const tU8*)OSAL_ps8StringSubString((const char*)_pUtf8String, (const char*)caBuffer);
}


/****************************************************************************
* Function          : cUtf8StringBase::StringSearchChar
*----------------------------------------------------------------------------
* Author            : Frericks
*----------------------------------------------------------------------------
* Description       : szSearchChar is character to be searched and needs not 
*                     be terminated.
*                     return the iterator or itEnd if not found  
****************************************************************************/
cUtf8StringIter cUtf8StringBase::itStringSearchChar(const tU8* szSearchChar ) const
{
   if(szSearchChar == NULL || _pUtf8String == NULL)
       return itEnd();
   
   const tU8* szResult = szStringSearchChar(szSearchChar);
   
   if (szResult == NULL)
      return itEnd();
   cUtf8StringIter iter = (cUtf8StringIter)(szResult-_pUtf8String);
   return iter;
}


/****************************************************************************
* Function          : cUtf8StringBase::StringSearchChar
*----------------------------------------------------------------------------
* Author            : Frericks
*----------------------------------------------------------------------------
* Description       : szSearchChar is character to be searched and needs not 
*                     be terminated.
*                     return the iterator or itEnd if not found  
****************************************************************************/
cUtf8StringIter cUtf8StringBase::itStringSearchChar(const cUtf8StringBase & oSearch ) const
{
   return itStringSearchChar(oSearch._pUtf8String);
}


/****************************************************************************
* Function          : cUtf8StringBase::uwGetNumberOfCharacters
*----------------------------------------------------------------------------
* Author            : RBIN/ECR - B.S.Pradeep
*----------------------------------------------------------------------------
* Description       : Returns the number of characters in the string
****************************************************************************/
tU16 cUtf8StringBase::wGetNumberOfCharacters () const
{
   tU16              uwCount = 0;
   cUtf8StringIter   itChar = itBegin();

   while (itChar != itEnd())
   {
      itChar = itNext(itChar);
      uwCount++;
   }
   return uwCount;
}

tU32 cUtf8StringBase::u32GetNumberOfCharacters () const
{
   return (tU32)wGetNumberOfCharacters();
}


tChar * cUtf8StringBase::Duplicate (const tChar * pcString, tU16 uwLen) const
{
   tChar * pDupString = (tChar *) OSAL_pvMemoryAllocate(uwLen + 1);
   if (pDupString != NULL)
   {
	   OSAL_pvMemorySet(pDupString, 0, uwLen + 1);
	   OSALUTIL_szSaveStringNCopy(pDupString, pcString, uwLen + 1);
   }
   return pDupString;
}


/****************************************************************************
* bCheckValid - Check for a correct UTF-8 string. 
*----------------------------------------------------------------------------
*  Author  : CM-DI/PJ-VW34 Dahlhoff
*----------------------------------------------------------------------------
*  Returns : true if valid string
****************************************************************************/
tBool cUtf8StringBase::bCheckValid () const
{
// frericks, mit dirks variante abgleichen
   if(_pUtf8String == NULL)
       return FALSE;
   tU8* p = _pUtf8String;
   DEB_TRACE_ERROR_MSG_TO_ERRMEM(_pUtf8String);

   unsigned int cnt,t;
   while (*p)
   {
      if (*p >= 0xC0)
      {
         // Do not allow a 0xFF.
         if (*p == 0xFFu)
         {
            return FALSE;
         }
         // Is a multi-byte symbol. Count bytes needed for this symbol
         for (cnt = 2; cnt < 7; cnt++)
         {
            if (!(*p & (0x80 >> cnt)))
               break;
         }
         // now we need this number of corret continue-bytes.
         for (t = 1; t < cnt; t++)
         {
            if (((p[t]) & 0xC0u) != 0x80)
            {
               return FALSE;
            }
         }
         // skip to next.
         p += cnt;
         continue;
      }
      if (*p >= 0x80)
      {
         return FALSE;
      }   // unexpected continue-byte
      // Otherwise, its a normal character.
      p++;
   }
   return TRUE;
}

/****************************************************************************
* Function          : cUtf8String::encode16bitChars
*----------------------------------------------------------------------------
* Author            : CM-DI/PJ-VW34 Dahlhoff
*----------------------------------------------------------------------------
* Description       : Encode a string of 16 bit symbols to UTF-8.
*                     This function does not use the string class.
*                     If the destination buffer is too small, the
*                     function will shorten the string to fit.
****************************************************************************/
tU16 cUtf8StringBase::encode16bitChars (const tU16 * pUTF16_string, 
   char * pDestBuffer, tU16 uwDestBufferBytes) const
{
// mit dirks avarinte abgleichen !!!
   tU16  ch;
   char  * wr, * pEnd;
   DEB_TRACE_ERROR_MSG_TO_ERRMEM(uwDestBufferBytes > 0);   // make sure a useful value is passed int
   wr = pDestBuffer;
   pEnd = pDestBuffer + uwDestBufferBytes ;
   // loop 16 bit symbols.
   while (wr + 1 < pEnd)
   {
       // Get symbol.
       ch = *(pUTF16_string++);
       // null-terminator?
       if (!ch)
       {
           break;
       }
       // check type.
       if (ch < 0x80u)
       {
           // Encode as one byte.
           *(wr++) = (tU8) ch;
       }
       else if (ch < 0x0800u)
       {
           // Encode as two bytes.
           if (wr + 2 >= pEnd)
           {
               break;
           } // would not fit.
           *(wr++) = (tU8) ((ch >> 6) | 0xC0u);
           *(wr++) = (tU8) ((ch & 0x3Fu) | 0x80u);
       }
       else
       {
           // Encode as three bytes.
           if (wr + 3 >= pEnd)
           {
               break;
           } // would not fit.
           *(wr++) = (tU8) ((ch >> 12) | 0xE0u);
           *(wr++) = (tU8) (((ch >> 6) & 0x3Fu) | 0x80u);
           *(wr++) = (tU8) ((ch & 0x3Fu) | 0x80u);
       }
       // 16 bit input chars can never encode to 4-byte UTF-8.
   }
   // null-terminate output string.
   *wr = 0;
   return (tU16)(wr-pDestBuffer);
};


/****************************************************************************
* DESCRIPTION: default constructor, creates object
****************************************************************************/
cUtf8String::cUtf8String() : cUtf8StringBase()
{
   _wUtf8AllocSize = 0;
   _wDynAllocatedSize = 0;
}

void cUtf8String::vAssign(const tU8 *sUTF8Text, tU32 nBufferSize)
{
   cUtf8StringBase::vAssign(sUTF8Text);
   if (nBufferSize == 0)
      _wUtf8AllocSize = _wUtf8StringLen +1;
   else
      _wUtf8AllocSize = (tU16)nBufferSize;
}

/****************************************************************************
* DESCRIPTION: constructor, creates object and allocates memory "nAllocBufferSize"
****************************************************************************/
tU8* cUtf8String::AllocMem(tU32 nAllocBufferSize)
{
   if (nAllocBufferSize==0)
      nAllocBufferSize=1;
   _pUtf8String = OSAL_NEW tU8[nAllocBufferSize];
   DEB_TRACE_ERROR_MSG_TO_ERRMEM(_pUtf8String);
   _wUtf8StringLen = 0;
   _wDynAllocatedSize = (tU16)nAllocBufferSize;
   _wUtf8AllocSize = _wDynAllocatedSize;
   return _pUtf8String;
}


cUtf8String::~cUtf8String()
{
   if (_wDynAllocatedSize && _pUtf8String)
   {
      OSAL_DELETE [] _pUtf8String;
   }
   _wDynAllocatedSize = 0;
   _wUtf8AllocSize=0;
   _pUtf8String = NULL;
}

/****************************************************************************
* Function          : cUtf8String::vInit
*----------------------------------------------------------------------------
* Author            : CM-CR/ESD-Janssen
*----------------------------------------------------------------------------
* Description       : Assigns a buffer to 
****************************************************************************/
void cUtf8String::vInit (tU8 * pubBuffer, tU16 uwBufferSize)
{
   DEB_TRACE_ERROR_MSG_TO_ERRMEM(uwBufferSize > 0);
   DEB_TRACE_ERROR_MSG_TO_ERRMEM(pubBuffer);
   if (_wDynAllocatedSize && _pUtf8String)
   {
      OSAL_DELETE [] _pUtf8String;
   }
   _wDynAllocatedSize = 0;
   _pUtf8String    = pubBuffer;
   _wUtf8AllocSize = uwBufferSize;
   _pUtf8String[0] = '\0';
   _wUtf8StringLen = 0;
}


/****************************************************************************
* Function          : cUtf8String::vClear
*----------------------------------------------------------------------------
* Author            : CM-CR/ESD-Janssen
*----------------------------------------------------------------------------
* Description       : stringlen=0;
****************************************************************************/
void cUtf8String::vClear ()
{
   _wUtf8StringLen = 0;
   if (_pUtf8String != NULL)
      _pUtf8String[0] = '\0';
}


/****************************************************************************
* Function          : cUtf8String::vSetCStr
*----------------------------------------------------------------------------
* Author            : CM-CR/ESD-Janssen
*----------------------------------------------------------------------------
* Description       : Sets the string content to the zero terminated string
*                     pointed to by 'pubCStr'.
*                     If the string at 'pubCStr' does not fit into the 
*                     UTIL_Utf8String's buffer, it will be truncated.
****************************************************************************/
void cUtf8String::vSetCStr (const tU8 * pubCStr)
{
   vClear();
   vAppendCStr(pubCStr);
}

/****************************************************************************
* Function          : cUtf8String::vSet16bitStr ( add UTF16) 
*----------------------------------------------------------------------------
* Author            : CM-DI/PJ-VW34 Dahlhoff
*----------------------------------------------------------------------------
* Description       : Sets the string content to the zero terminated string
*                     pointed to by 'pubCStr'.
*                     If the string at 'pubCStr' does not fit into the 
*                     UTIL_Utf8String's buffer, it will be truncated.
****************************************************************************/
void cUtf8String::vSet16bitStr (const tU16 * puw16BStr)
{
   // Check parameter
   DEB_TRACE_ERROR_MSG_TO_ERRMEM(_wUtf8AllocSize > 1);
   DEB_TRACE_ERROR_MSG_TO_ERRMEM(_pUtf8String);
   // Conversion done in here. Will cut string if buffer is too short.
   _wUtf8StringLen = encode16bitChars(puw16BStr, (char *) (_pUtf8String), _wUtf8AllocSize);
}

/****************************************************************************
* Function          : cUtf8String::vAppendCStr ( concat, strcat )
*----------------------------------------------------------------------------
* Author            : CM-CR/ESD-Janssen/888888888
*----------------------------------------------------------------------------
* Description       : Appends the zero terminated string pointed to by
*                     'pubCStr' to the UTIL_Utf8String.
*                     If the string at 'pubCStr' does not fit into the 
*                     UTIL_Utf8String's buffer, it will be truncated.
****************************************************************************/
void cUtf8String::vAppendCStr (const tU8 * pubCStr)
{
   tU16  uwNewLength = _wUtf8StringLen + OSAL_u32StringLength(pubCStr);

   if ((uwNewLength < _wUtf8AllocSize) && (_pUtf8String != NULL))
   {
       OSAL_szStringConcat(_pUtf8String, pubCStr);
       _wUtf8StringLen = uwNewLength;
   }
   else
   {
       DEB_TRACE_ERROR_MSG_TO_ERRMEM(0);  /* string buffer too small */
   }
}


/****************************************************************************
* Function          : cUtf8String::vAppendUtf8Sequence ( strcat) 
*----------------------------------------------------------------------------
* Author            : CM-CR/ESD-Janssen
*----------------------------------------------------------------------------
* Description       : Appends a sequence of UTF8 characters to the given UTF8
*                     string. The sequence is ulwNumOfBytes bytes long and is
*                     not(!) zero-terminated.
*                     The function appends as many characters to the string
*                     as will fit into the string.
****************************************************************************/
void cUtf8String::vAppendUtf8Sequence (const tU8 * pubUtf8Seq, tU32 u32NumOfBytes)
{
   tU32  ulwNumOfBytesToCopy = 0;

   if(_pUtf8String == NULL)
   {
      return;
   }
   /* determine number of bytes to copy (copy complete UTF-8 characters only) */
   int iNumOfBytes = (int)u32NumOfBytes;
   while (iNumOfBytes > 0)
   {
      tU32 ulwCharSize = u32GetCharacterByteLength(pubUtf8Seq[ulwNumOfBytesToCopy]);
      if ((tU16) (_wUtf8AllocSize - _wUtf8StringLen) > ulwCharSize + ulwNumOfBytesToCopy)
      {
         ulwNumOfBytesToCopy += ulwCharSize;
      }
      iNumOfBytes -= ulwCharSize;
   }
   DEB_TRACE_ERROR_MSG_TO_ERRMEM(iNumOfBytes==0);
   /* now copy the number of bytes determined */
   OSAL_szStringNConcat(&_pUtf8String[_wUtf8StringLen], pubUtf8Seq, ulwNumOfBytesToCopy);
   _wUtf8StringLen += (tU16) ulwNumOfBytesToCopy;
   _pUtf8String[_wUtf8StringLen] = '\0';
}


/****************************************************************************
* Function          : cUtf8String::vAppendChars
*----------------------------------------------------------------------------
* Author            : RBIN/ECR - B.S.Pradeep
*----------------------------------------------------------------------------
* Description       : Appends the number of characters (segments).
****************************************************************************/
void cUtf8String::vAppendChars (const tU8 * pubUtf8Seq, tU32 ulwNumOfChars)
{
   tU32  ulwByteCounter = 0;

   while (ulwNumOfChars > 0)
   {
       ulwByteCounter += u32GetCharacterByteLength(pubUtf8Seq[ulwByteCounter]);
       ulwNumOfChars--;
   }
   cUtf8String::vAppendUtf8Sequence(pubUtf8Seq, ulwByteCounter);
}


/****************************************************************************
* Function          : cUtf8String::vAppendChar
*----------------------------------------------------------------------------
* Author            : CM-CR/ESD-Janssen
*----------------------------------------------------------------------------
* Description       : Appends the given 'ulwChar' to the UTIL_Utf8String.
*                     If the char does not fit into the UTIL_Utf8String's
*                     buffer, it will be discarded.
****************************************************************************/
void cUtf8String::vAppendChar (tU32 u32UniCodeChar)
{
   const tU32  ulwFreeBytes = _wUtf8AllocSize - _wUtf8StringLen - 1;

   if (_pUtf8String == NULL)
   {
      return;
   }

   if (ulwFreeBytes < u32GetCharacterByteLengthOfUniCodeChar(u32UniCodeChar))
   {
       DEB_TRACE_ERROR_MSG_TO_ERRMEM(0);  /* string buffer too small */
       return;
   }

   if (u32UniCodeChar < 0x80)
   {
       _pUtf8String[_wUtf8StringLen++] = (tU8) u32UniCodeChar;
       _pUtf8String[_wUtf8StringLen] = '\0';
       return;
   }
   if (u32UniCodeChar < 0x800)
   {
       _pUtf8String[_wUtf8StringLen++] = (tU8) (0xC0 | (u32UniCodeChar >> 6));
       _pUtf8String[_wUtf8StringLen++] = (tU8) (0x80 | (u32UniCodeChar & 0x3F));
       _pUtf8String[_wUtf8StringLen] = '\0';
       return;
   }
   if (u32UniCodeChar < 0x10000)
   {
       _pUtf8String[_wUtf8StringLen++] = (tU8) (0xE0 | (u32UniCodeChar >> 12));
       _pUtf8String[_wUtf8StringLen++] = (tU8) (0x80 | ((u32UniCodeChar >> 6) & 0x3F));
       _pUtf8String[_wUtf8StringLen++] = (tU8) (0x80 | (u32UniCodeChar & 0x3F));
       _pUtf8String[_wUtf8StringLen] = '\0';
       return;
   }
   if (u32UniCodeChar < 0x200000)
   {
       _pUtf8String[_wUtf8StringLen++] = (tU8) (0xF0 | (u32UniCodeChar >> 18));
       _pUtf8String[_wUtf8StringLen++] = (tU8) (0x80 | ((u32UniCodeChar >> 12) & 0x3F));
       _pUtf8String[_wUtf8StringLen++] = (tU8) (0x80 | ((u32UniCodeChar >> 6) & 0x3F));
       _pUtf8String[_wUtf8StringLen++] = (tU8) (0x80 | (u32UniCodeChar & 0x3F));
       _pUtf8String[_wUtf8StringLen] = '\0';
       return;
   }
   DEB_TRACE_ERROR_MSG_TO_ERRMEM(0);  /* invalid character */
   return;
}


/****************************************************************************
* Function          : cUtf8String::vDeleteLastCharacter
*----------------------------------------------------------------------------
* Author            : RBIN/ECR - B.S.Pradeep
*----------------------------------------------------------------------------
* Description       : Deletes last character
****************************************************************************/
void cUtf8String::vDeleteLastCharacter ()
{
   if ((_pUtf8String != NULL) && (_wUtf8StringLen > 0))
   {
       _wUtf8StringLen = (tU16) itPrevious(itEnd());
       _pUtf8String[_wUtf8StringLen] = '\0';
   }
}


/****************************************************************************
* Function          : cUtf8String::vDeleteCharacter
*----------------------------------------------------------------------------
* Author            : Schmid
*----------------------------------------------------------------------------
* Description       : Deletes character at iter
****************************************************************************/
void cUtf8String::vDeleteCharacter (cUtf8StringIter itPosition)
{
   if ((_pUtf8String == NULL) || (itPosition >= _wUtf8StringLen))
   {
      return;
   }
   cUtf8StringIter itNextPosition = itNext(itPosition);
   OSAL_pvMemoryCopy(
      &_pUtf8String[itPosition], 
      &_pUtf8String[itNextPosition], 
      itEnd() - itNextPosition + 1 );
   // new length is the old length - length off deleted letter
   _wUtf8StringLen -= (tU16)(itNextPosition - itPosition);
}


/****************************************************************************
* vInsert
*  Insert string at position in self
* Author : Schmid
****************************************************************************/
void cUtf8String::vInsert (const cUtf8StringIter itPosition, const cUtf8String * pString)
{
   if (_pUtf8String == NULL)
   {
      return;
   }

   tU16  uwNewLength = _wUtf8StringLen + pString->_wUtf8StringLen;

   // position is out of string
   DEB_TRACE_ERROR_MSG_TO_ERRMEM(itPosition <= _wUtf8StringLen);

   // enough space left?
   if (uwNewLength < _wUtf8AllocSize)
   {
       // backup required?
       if ((itEnd() - itPosition) > 0)
       {
           // reserve mem for back up the last part of the string
           tU8 * pStoreBackEnd = (tU8 *) OSAL_pvMemoryAllocate(itEnd() - itPosition + 1);
           if (pStoreBackEnd == NULL)
        	   return;
       	   OSAL_pvMemorySet(pStoreBackEnd, 0, (itEnd() - itPosition + 1));
           // back up the last part of the string
           OSAL_pvMemoryCopy((tPVoid) pStoreBackEnd, (tPVoid) & _pUtf8String[itPosition],
            (tU16) (itEnd() - itPosition + 1));

           while (itPosition != itEnd())
           {
               cUtf8String::vDeleteLastCharacter();
           }
           cUtf8String::vAppendCStr(pString->_pUtf8String);
           cUtf8String::vAppendCStr(pStoreBackEnd);

           OSAL_vMemoryFree(pStoreBackEnd);
       }
       else
       {
           cUtf8String::vAppendCStr(pString->_pUtf8String);
       }
   }
   else
   {
       DEB_TRACE_ERROR_MSG_TO_ERRMEM(0);  /* string buffer too small */
   }
}

/****************************************************************************
* bFixString
*----------------------------------------------------------------------------
*  Check for a correct UTF-8 string. If any errors are
*  detected, overwrite bad bytes with '?'.
*----------------------------------------------------------------------------
*  Author  : CM-DI/PJ-VW34 Dahlhoff
*----------------------------------------------------------------------------
*  Returns : true if any byte was changed.
****************************************************************************/
tBool cUtf8String::bFixString (tU8 * pubUtf8Seq) const
{
   tU8* p = pubUtf8Seq;
   unsigned int cnt,t;
   tBool result = FALSE;
   while (*p)
   {
       if (*p >= 0xC0)
       {
           // Is a multi-byte symbol. Count bytes needed for this symbol
           for (cnt = 2; cnt < 7; cnt++)
               if (!(*p & (0x80 >> cnt)))
               {
                   break;
               }
           // now we need this number of corret continue-bytes.
           for (t = 1; t < cnt; t++)
           {
               if (((p[t]) & 0xC0u) != 0x80)
               {
                   // bad follow-up sequence
                   *p = '?';       // overwrite first byte (!) with a question mark. Follow-up bytes will be overwritten later in the loop.
                   cnt = 1;
                   result = TRUE;
                   break;
               }
           }
           // skip to next.
           p += cnt;
           continue;
       }
       else if (*p >= 0x80)
       {
           // unexpected second byte
           *p = '?';
           result = TRUE;
       }
       // Otherwise, its a normal character.
       p++;
   }
   return result;
}


/****************************************************************************
* bFixString
*----------------------------------------------------------------------------
*  Check for a correct UTF-8 string. If any errors are
*  detected, overwrite bad bytes with '?'.
*----------------------------------------------------------------------------
*  Author  : CM-DI/PJ-VW34 Dahlhoff
*----------------------------------------------------------------------------
*  Returns : true if any byte was changed.
****************************************************************************/
tBool cUtf8String::bFixString () 
{
   return bFixString(_pUtf8String);
}


/****************************************************************************
* Function          : s32ToInteger
*----------------------------------------------------------------------------
* Author            : Frericks
*----------------------------------------------------------------------------
* Description       : Converts a string containing an unsigned decimal number
*                     into the corresponding value.
*                     Behaviour for other strings is undefined!!!
****************************************************************************/
tS32 cUtf8StringBase::s32ToInteger() const
{
   if(_pUtf8String != NULL)
       return OSAL_s32AsciiToS32(_pUtf8String);
   else
       return 0;
}


/****************************************************************************
* Function          : s32NPrintFormat, s32VarNPrintFormat
*----------------------------------------------------------------------------
* Author            : Frericks
*----------------------------------------------------------------------------
* This function is the equivalent to the function OSALUTIL_s32SaveNPrintFormat.
* It writes formatted data into the given buffer and takes care for a null
* termination (by using the function UTF8_vCutOffFaultyCharacter).
*
* @param coszFormat      string which contains the format behaviour
* @param ...             list of parameter
****************************************************************************/
tS32 cUtf8String::s32NPrintFormat( tCString coszFormat, ... )
{
	OSAL_tVarArgList args;
	OSAL_VarArgStart(args,coszFormat);//lint !e530 (this function fills "args")
	tS32 result = s32VarNPrintFormat(coszFormat,args); //lint !e530 ANSI, no init necessary
	OSAL_VarArgEnd(args);

	return result;
}


tS32 cUtf8String::s32VarNPrintFormat( tCString coszFormat, va_list argptr )
{
   return OSAL_s32VarNPrintFormat( (tString)_pUtf8String, _wUtf8AllocSize, coszFormat, argptr );
}







