/*******************************************************************************
*
* FILE:          sortlib.c
*
* SW-COMPONENT:  Sorting Lib
*
* PROJECT:       Bosch-CMD
*
* DESCRIPTION:   Sorting Lib for Sqlite
*
* AUTHOR:
*
* COPYRIGHT:     (c) 2011 Robert Bosch GmbH, Hildesheim
*
*******************************************************************************/

/*******************************************************************************
*             INCLUDES
*******************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "sortlib_helper.h"
#include "sortlib.h"


#ifdef SORTLIB_AS_EXTENSION
// we are compiling sortlib as an extension for shared library builds

// normally the sqlite3ext.h shall be included in such an extension.
// This is problematic here because the sortlib shall also be used
// directly and not as an extension. The sqlite3ext.h redefines the normal
// sqlite3 functions to get called via a structure of function pointers.
// This structure is passed by sqlite when loading the sortlib as an
// extension. So, when using sortlib not as an extension the pointer
// to the structure of function pointers is not passed and will remain
// 0, so that a call to an sqlite3 function will result in a NULL pointer
// exception. Thus, the include of sqlite3ext.h is not used and the
// structure of function pointers is ignored.

// sqlite3_api_routines is defined in sqlite3ext.h header which is not used here.
// Therefore, we need to define it here locally to make this name available.
#define sqlite3_api_routines void

#ifdef _WIN32
__declspec(dllexport)
#endif
int sqlite3_sortlibso_init(
    sqlite3 *handle,
    char **pzErrMsg,
    const sqlite3_api_routines *pApi
)
{
  int rc = SQLITE_OK;
  (void) pzErrMsg;
  (void) pApi;
  /* Insert here calls to
  **     sqlite3_create_function_v2(),
  **     sqlite3_create_collation_v2(),
  **     sqlite3_create_module_v2(), and/or
  **     sqlite3_vfs_register()
  ** to register the new features that your extension adds.
  */
  rc = sortlib_register_sort_function( handle );
  return rc;
}
#endif // SORTLIB_AS_EXTENSION



/*******************************************************************************
*             DEFINES
*******************************************************************************/
//#define SORTLIB_VERSION "2.0"
unsigned* sortlib_get_rule_table();

#ifdef _MSC_VER
static unsigned sortlib_next_ucs_char(const char *s, unsigned *index_, unsigned len)
#else
/* prototype needed for specifiying always_inline attribute on gcc */
inline static unsigned sortlib_next_ucs_char(const char *s, unsigned *index_, unsigned len) __attribute__((always_inline));
inline static unsigned sortlib_next_ucs_char(const char *s, unsigned *index_, unsigned len)
#endif
{
  register unsigned tempCh;
  register unsigned size = 0;
  register unsigned idx = *index_;

  tempCh = s[idx++];

   if ((tempCh & 0x80))
   {
      // UTF8: read out char len
      if ((tempCh & 0xe0) == 0xc0) {
         tempCh &= 0x1f;
         size = 1;
      } else if ((tempCh & 0xf0) == 0xe0) {
         tempCh &= 0xf;
         size = 2;
      } else if ((tempCh & 0xf8) == 0xf0){
         tempCh &= 0x07;
         size = 3;
      }

      while(size-- && idx<len) {
         tempCh <<= 6;
         tempCh |= (unsigned char)(s[idx++]) & 0x3f;
      }
   }

  *index_ = idx;
  return tempCh;
}

/*******************************************************************************
** FUNCTION:     sortlib2_compare_digits_in_data
*  DESCRIPTION:  This function compare two strings base on Digits.
*
*
*  INPUT PARAMS:
*                arg1_ - sortlib_dataArray d1: Left String from Digits
*                arg2_ - sortlib_dataArray d2: Right String from Digits
*
*  OUTPUT PARAMS: nextIndex - No of Char Advance
*
*  RETURNVALUE:  Result of Compared Strings
********************************************************************************/
static int sortlib2_compare_digits_in_data( sortlib_inputData *d )
{
  int result = SORTLIB_EQUAL;
  //int isDigit1 = FALSE;
  //int isDigit2 = FALSE;

  //save last indexs
  unsigned index1 = d->lastIndex1; // BOSCHCOR
  unsigned index2 = d->lastIndex2; // BOSCHCOR

  int intLen1 = 0; // BOSCHCORRECTION set initial length to 0
  unsigned ucs1 = d->ucs1;

  //while ( index1 < d->len1 && TRUE == sortlib_isdigit(ucs1) )
  // begin BOSCHCORR
  //while ( index1 < d->len1 && SORTLIB_IS_DIGIT(ucs1) )
  do
  {
      ucs1 = sortlib_next_ucs_char(d->s1,&index1, d->len1);
      if (intLen1==0 && ucs1==SORTLIB_UTF_CODE_0)
      {
         d->lastIndex1 = index1;
      }
      else if (SORTLIB_IS_DIGIT(ucs1))
      {
         ++intLen1;
      }
  } while ( index1 < d->len1 && SORTLIB_IS_DIGIT(ucs1) );
  // end BOSCHCORR

  int intLen2 = 0; // BOSCHCORRECTION set initial length to 0
  unsigned ucs2 = d->ucs2;

  //while ( index2 < d->len2 && TRUE == sortlib_isdigit(ucs2) )
  // begin BOSCHCORR
  //while ( index2 < d->len2 && SORTLIB_IS_DIGIT(ucs2) )
  do
  {
      ucs2 = sortlib_next_ucs_char(d->s2,&index2, d->len2);
      if (intLen2==0 && ucs2==SORTLIB_UTF_CODE_0)
      {
         d->lastIndex2 = index2;
      }
      else if (SORTLIB_IS_DIGIT(ucs2))
      {
         ++intLen2;
      }
  } while ( index2 < d->len2 && SORTLIB_IS_DIGIT(ucs2) );
  // end BOSCHCORR

  if ( intLen1 > intLen2 )
    return SORTLIB_GREATER;
  else if ( intLen1 < intLen2 )
    return SORTLIB_LESS;

   // begin BOSCHCORRECTION set index and ucs to first none-0 character
   d->index1 = d->lastIndex1;
   d->index2 = d->lastIndex2;
   d->ucs1 = sortlib_next_ucs_char(d->s1,&d->index1, d->len1);
   d->ucs2 = sortlib_next_ucs_char(d->s2,&d->index2, d->len2);
   // end BOSCHCORRECTION set index and ucs to first none-0 character

  //d->index1 = d->lastIndex1;
  //d->index2 = d->lastIndex2;
  while ( d->index1 <= d->len1 && d->index2 <= d->len2 )
  {
    //isDigit1 = sortlib_isdigit(d->ucs1);
    //isDigit2 = sortlib_isdigit(d->ucs2);
//    if ( ( TRUE == isDigit1 ) &&
//         ( TRUE == isDigit2 ) )
    if ( ( SORTLIB_IS_DIGIT(d->ucs1) ) &&
         ( SORTLIB_IS_DIGIT(d->ucs2) ) )
    {
      result = SORTLIB_COMPARE_NUMS( d->ucs1, d->ucs2 );

      if ( SORTLIB_EQUAL != result )
      {
        break;
      }
    }
    else
    {
      d->index1 = d->lastIndex1;
      d->index2 = d->lastIndex2;
      break;
    }

    //save last index;
    d->lastIndex1 = d->index1;
    d->lastIndex2 = d->index2;

    if ( d->index1 < d->len1 && d->index2 < d->len2 )
    {
      d->ucs1 = sortlib_next_ucs_char(d->s1,&d->index1, d->len1);
      d->ucs2 = sortlib_next_ucs_char(d->s2,&d->index2, d->len2);
    }
    else
    {
      break;
    }

  }


  return result;
}

static FILE *fpLog = NULL;

/*******************************************************************************
** FUNCTION:     sortlib2_callback_data_compare
*  DESCRIPTION:  This function compare two strings base on GlobalSortingTable.
*                It is callback function,register by sqlite3_create_collation_v2()
*
*  INPUT PARAMS:
*                arg1_ - void *pCtx: NULL Here
*                arg2_ - int len1: The length of source or -1 if
*                                    null-terminated.
*                arg3_ - const void *s1: The source string
*                arg4_ - int len2: The length of target or -1 if
*                                    null-terminated.
*                arg5_ - const void *s2: The target string
*
*  OUTPUT PARAMS:NONE
*
*  RETURNVALUE:  Result of Compared Strings
********************************************************************************/
static int sortlib2_callback_data_compare(
    void* pCtx,
    int len1_,
    const void* s1,
    int len2_,
    const void* s2)
{
   int result;
   unsigned int len1=(unsigned int)len1_;
   unsigned int len2=(unsigned int)len2_;

   unsigned* ruleTable = sortlib_get_rule_table(); //lint !e746 PQM_authorized_260
   //unsigned ruleTable[65535]={0};

   (void)pCtx;//suppress compiler warning of unused variable

   // for test trace messages to a file
   int res=0;
   char *str1=NULL;
   char *str2=NULL;

   // TODO test
   if (!fpLog) {
      //fpLog = fopen("/tmp/sortlib_testdata.txt", "w");
   }

   if (fpLog) {
      str1 = malloc(len1+1);
      str2 = malloc(len2+1);
      if (str1!=NULL)
      {
         memcpy(str1, s1, len1);
         str1[len1] = 0;
      }
      if (str2!=NULL)
      {
         memcpy(str2, s2, len2);
         str2[len2] = 0;
      }
      if (str1!=NULL && str2!=NULL)
      {
         res = strcmp(str1, str2);
      }
      if (res < 0) res = SORTLIB_LESS;
      else if (res > 0) res = SORTLIB_GREATER;
      else res = SORTLIB_EQUAL;
   }

   result = SORTLIB_EQUAL;

   sortlib_inputData d;

   d.s1 = (char*)s1;
   d.len1 = len1;
   d.index1 = 0;
   d.lastIndex1 = 0;

   d.s2 = (char*)s2;
   d.len2 = len2;
   d.index2 = 0;
   d.lastIndex2 = 0;

   d.ucs1 = 0;
   d.ucs2 = 0;

   while ( d.index1 < d.len1 && d.index2 < d.len2 )
   {
      d.lastIndex1 = d.index1;
      d.lastIndex2 = d.index2;
      d.ucs1 = sortlib_next_ucs_char(d.s1,&d.index1, d.len1);
      d.ucs2 = sortlib_next_ucs_char(d.s2,&d.index2, d.len2);

//    if ( ( TRUE == sortlib_isdigit(d.ucs1) ) &&
//         ( TRUE == sortlib_isdigit(d.ucs2) ) )
      if ( ( SORTLIB_IS_DIGIT(d.ucs1) ) &&
           ( SORTLIB_IS_DIGIT(d.ucs2) ))
      {
         result = sortlib2_compare_digits_in_data( &d );

         if ( SORTLIB_EQUAL != result )
         {
            break;
         }

      }
      else
      {
         unsigned int value1 = d.ucs1;
         unsigned int value2 = d.ucs2;
         if (d.ucs1<SORTLIB_ARRAY_SIZE)
         {
            value1 = ruleTable[d.ucs1];
         }
         if (d.ucs2<SORTLIB_ARRAY_SIZE)
         {
            value2 = ruleTable[d.ucs2];
         }
         result = SORTLIB_COMPARE_NUMS( value1, value2 );

         if ( SORTLIB_EQUAL != result )
         {
            break;
         }

      }

   }

   //if the strings are of unequal lengths and the
   //smaller string exactly matches the start of longer string
   //then the longer string should be given GREATER
   if ( SORTLIB_EQUAL == result )
   {
      result = SORTLIB_COMPARE_NUMS( d.len1, d.len2 );
   }

   if (fpLog) {
      if (res != result) {
         fprintf(fpLog, "|%s| ?= |%s|: res=%d, result=%d\n", str1, str2, res, result);
         fflush(fpLog);
      }
   }
   if (str1!=NULL)  free(str1);
   if (str2!=NULL)  free(str2);

   return result;
}

/*******************************************************************************
** FUNCTION:     sortlib_register_sort_function
*  DESCRIPTION:  This function register new Collating Sequences
*
*  INPUT PARAMS:
*                arg1_ - sqlite3 *handle: Handle of Sqlite3 Driver
*
*  OUTPUT PARAMS:NONE
*
*  RETURNVALUE:  Result of New Collating Sequences Function
********************************************************************************/
int sortlib_register_sort_function( sqlite3 *handle )
{

  //Register new Collating Sequences Function
  int result = SQLITE_ERROR;
  if (NULL != handle)
  {
    result = sqlite3_create_collation_v2( handle,
                                    SORTLIB_COLLATE_NAME,
                                    SQLITE_UTF8,
                                    0,
                                    sortlib2_callback_data_compare,
                                    0);
  }
  return result;
}

int sortlib_compare(const void *string1, const void *string2)
{
   int l1=(int)strlen((const char*)string1); // zero terminated strings
   int l2=(int)strlen((const char*)string2);
   return sortlib2_callback_data_compare(NULL, l1, string1, l2, string2);
}

int sortlib_string_compare(const char* string1, int len1, const char* string2, int len2)
{
   return sortlib2_callback_data_compare(NULL, len1, string1, len2, string2);
}


// Function added for Vertical Keyboard Library
unsigned sortlib_get_char_sort_prio(unsigned u32Unicode32BitCodePoint)
{
   // Create Sorting table, if it has not been created so far.
   unsigned* ruleTable = sortlib_get_rule_table(); //lint !e746 PQM_authorized_260

   if( u32Unicode32BitCodePoint < SORTLIB_ARRAY_SIZE )  // Note: SORTLIB_ARRAY_SIZE = 65535 (see sortlib_helper.h)
   {
      return ruleTable[u32Unicode32BitCodePoint];
   }
   else return( u32Unicode32BitCodePoint );
}
