#ifndef MATHGENERIC_HEADER
#define MATHGENERIC_HEADER

/****************************************************************************

 .DATEI                [ mathgeneric.h ]

 .BESCHREIBUNG
  Function declarations for generic math functions.

 .COPYRIGHT            [ (c) 2002 Robert Bosch GmbH, Hildesheim ]
 .AUTOR                [ Arnold J. ]

 * $Log:   //hi230124/projects/Vasco/swnavi/archives/components/math/mathgeneric.h-arc  $
 * 
 *    Rev 1.12   Aug 25 2003 15:15:32   ArnoldJoerg
 * add (default) parameter: crc16f( .., initcrcvalue=0 );
 * remove (unused) function-alias
 * 
 *    Rev 1.11   05 Mar 2003 15:04:06   VOA2HI
 * use of new function fatan2 to capsulate atan2 for arm
 * 
 *    Rev 1.10   29 Aug 2002 14:26:38   ArnoldJoerg
 * protection against using math_abs with unsigned types
 * 
 *    Rev 1.9   08 Aug 2002 15:38:38   ArnoldJoerg
 * add math_abs(), delete math_s32Absolute()
 * 
 *    Rev 1.8   08 Aug 2002 13:08:44   ArnoldJoerg
 * min()/max() -> math_min()/math_max(), add math_s32Absolute()
 * 
 *    Rev 1.7   31 Jul 2002 13:47:30   Fischer
 * changed min, max for CE-compiler
 * 
 *    Rev 1.6   30 Jul 2002 14:41:08   ArnoldJoerg
 * added: min(), max()
 * 
 *    Rev 1.5   19 Jun 2002 14:58:54   ArnoldJoerg
 * prototype for s32Round()
 * 
 *    Rev 1.4   18 Jun 2002 13:40:46   ArnoldJoerg
 * change prototype for PointInLine() and CalcIntersectionTwoLines()
 * 
 *    Rev 1.3   18 Jun 2002 10:28:22   ArnoldJoerg
 * rename floatcmp() to s32Compare()
 * and sgn() to s32Sign()
 * 
 *    Rev 1.2   07 Jun 2002 11:57:52   ArnoldJoerg
 * use OSAL_IF.H instead of OSAL.H
 * 
 *    Rev 1.1   06 Jun 2002 18:22:18   ArnoldJoerg
 * add include-check
 * 
 *    Rev 1.0   06 Jun 2002 14:25:36   ArnoldJoerg
 * Initial revision.
  
 .HISTORIE
  Datum      | Name        Modifikation
 ------------|---------------------------------------------------------------
  03.06.2002 | CM-DI/ESA1-An from mathutil.h: changes / ported for TypE
*****************************************************************************/

#ifndef OSAL_OSTYPES_HEADER
    #pragma message ("-An- MATHGENERIC.H benoetigt OSAL_IF.H mit OSAL_S_IMPORT_INTERFACE_GENERIC")
#endif

#ifndef EPS
#define EPS  (1.0e-6)
#endif

#ifndef EPS_SMALL
#define EPS_SMALL (1.0e-40) //um auch noch Determinaten mit Genauigkeit von 1 cm invertieren zu k�nnen
#endif

//---------------------------------------------------------------------------
//-- generic math functions with integer arithmetic

//!  return CRC16 of buffer
//! \ingroup MOD_GENERIC
tU16 u16CalcCrc( const tU8 * pu8BufferAddress, tU32 u32BufferSize, tU16 u16CrcValue = 0 );

//!  return ratio of U32s
//! \ingroup MOD_GENERIC
tU16 u16Fraction( tU32 dividend, tU32 divisor, tU32 minimum );

//!  return tS16/fraction
//! \ingroup MOD_GENERIC
tS16 s16Div_U16Fraction( tS16 s16Divident, tU16 u16Fraction_Divisor );

//!  32 bit quotient of ...
//! \ingroup MOD_GENERIC
tS32 s32Div_U16Fraction( tS32 s32Dividend, tU16 u16Fraction_Divisor );

//!  tS16*fraction
//! \ingroup MOD_GENERIC
tS16 s16Mul_U16Fraction( tS16 s16Multiplicant, tU16 u16Fraction_Multiplier );

//!  tS32*fraction
//! \ingroup MOD_GENERIC
tS32 s32Mul_U16Fraction( tS32 s32Multiplicant, tU16 u16Fraction_Multiplier );

//!  multiplikation and following division (64Bit-precise provisional result
//! \ingroup MOD_GENERIC
tS32 muldiv32( tS32 slFactorA, tS32 slFactorB, tS32 slDivisor, tBool bRoundUp );

//!  arc tangent in EAU (tan = y/x)
//! \ingroup MOD_GENERIC
tS16 iatan2( tS16 y, tS16 x);

//!  arc tangent for floats: don't use atan2 directly! this is dangerous with arm compiler*/
//! \ingroup MOD_GENERIC
tDouble fatan2( tDouble y, tDouble x);

//!  sinus_fraction (-tS16_MIN ... +tS16_MAX) as cosinus from EAU
//! \ingroup MOD_GENERIC
tS16 icos( tS16 angleEAU );

//!  sinus_fraction (-tS16_MIN ... +tS16_MAX) as sinus from EAU
//! \ingroup MOD_GENERIC
tS16 isin( tS16 angleEAU );

//!  16 bit square root
//! \ingroup MOD_GENERIC
tU16 isqrt( tU32 a );                

//!  square root with check for non negative argument
//! \ingroup MOD_GENERIC
inline tDouble sqrt_withCheck(tDouble a){return (a<=0.0)?0.0:sqrt(a);}
//! \ingroup MOD_GENERIC
inline tFloat  sqrt_withCheck(tFloat  a){return (a<=0.0f)?0.0f:sqrt(a);}

//! acos with check for values out of range
//! \ingroup MOD_GENERIC
inline tDouble acos_withCheck(tDouble a){return (a<-1)?OSAL_C_DOUBLE_PI:((a>1)?0:acos(a));}

//---------------------------------------------------------------------------
//-- generic math functions with float/double arithmetic

//! convert tDouble to tS32.
//! If the double-number is to large or small, the largest/smallest tS32 is returned
//! \ingroup MOD_GENERIC
tS32 tDouble2S32( tDouble number);

//! convert tDouble to tU32.
//! If the double-number is to large or negative, the largest tU32 or 0 is returned
//! \ingroup MOD_GENERIC
tU32 tDouble2U32( tDouble number);

//! convert tDouble to tS16.
//! If the double-number is to large or small, the largest/smallest tS16 is returned
//! \ingroup MOD_GENERIC
tS16 tDouble2S16( tDouble number);

//! convert tDouble to tU16.
//! If the double-number is to large or negative, the largest tU16 or 0 is returned
//! \ingroup MOD_GENERIC
tU16 tDouble2U16( tDouble number);

//! convert tDouble to tU8.
//! If the double-number is to large or negative, the largest tU8 or 0 is returned
//! \ingroup MOD_GENERIC
tU8 tDouble2U8( tDouble number);

//! convert tDouble to tFloat.
//! If the double-number is to large or small, the largest/smallest tFloat is returned
//! \ingroup MOD_GENERIC
tFloat tDouble2Float( tDouble number);


//! compares two floats/doubles.
//! An epsilon is used.
//! Returns  0, if the difference is less epsilon,
//! returns -1, if 1st number is less than the 2nd number,
//! returns +1, if 1st number is more than the 2nd number
//! \ingroup MOD_GENERIC
tS32 s32Compare( tDouble dFirst, tDouble dSecond, tDouble _eps = EPS);

//! compares the given value with zero.
//! An epsilon is used.
//! Returns: -1, when value < 0
//! 0, by value == 0 (+-epsilon)
//! +1, by value > 0;
//! \ingroup MOD_GENERIC
tS32 s32Sign( tDouble dValue, tDouble _eps = EPS);

//!  round double-value to signed integer (tS32)
//! \ingroup MOD_GENERIC
tS32 s32Round( tDouble number );

//---------------------------------------------------------------------------
//-- geometric math functions with float/double arithmetic

//! Ersatz fuer iptdist()
//! Berechnet die Projektion von (pX,pY) auf die Linie durch P1-P2. (ProjX, ProjY) ist
//! der projezierte Punkt. Rueckgabe ist die Entfernung zur Linie.*/
//! \ingroup MOD_GENERIC
tU16 CalcProjectionToLine( const tDouble P1X, 
                           const tDouble P2X, 
                           const tDouble pX, 
                           tDouble *ProjX,
                           const tDouble P1Y,
                           const tDouble P2Y,
                           const tDouble pY,
                           tDouble *ProjY);
//! \ingroup MOD_GENERIC
tU32 u32CalcProjectionToLine( const tDouble P1X, 
                              const tDouble P2X, 
                              const tDouble pX, 
                              tDouble *ProjX,
                              const tDouble P1Y,
                              const tDouble P2Y,
                              const tDouble pY,
                              tDouble *ProjY);
//! \ingroup MOD_GENERIC
tDouble dCalcProjectionToLine( const tDouble P1X, 
                               const tDouble P2X, 
                               const tDouble pX, 
                               tDouble *ProjX,
                               const tDouble P1Y,
                               const tDouble P2Y,
                               const tDouble pY,
                               tDouble *ProjY);
//! Ersatz fuer clostpt_ext. Projeziert P3 auf die Strecke durch P1-P2.
//! Ist das Lot ausserhalb der Strecke, so wird der naechste Endpunkt der Strecke genommen.
//! Es wird die Entfernung zur Strecke returned.*/
//! \ingroup MOD_GENERIC
tU16 CalcNearestPoint( tDouble P1X,    //X-Koord P1 (Anfangspunkt der Strecke)
                       tDouble P2X,    //X-Koord P2 (Endpunkt der Strecke)
                       tDouble P3X,    //X-Koord P3 (Punkt von dem Lot gefaellt wird)
                       tDouble *P4X,   //X-Koord des Ergebnis (naechster Punkt)
                       tDouble P1Y,    //Y-Koord P1
                       tDouble P2Y,    //Y-Koord P2
                       tDouble P3Y,    //Y-Koord P3
                       tDouble *P4Y,   //Y-Koord P4
                       tBool *LotFlag);     //bei TRUE ist P3P4 senkrecht zu P1P2
//! \ingroup MOD_GENERIC
tU32 u32CalcNearestPoint( tDouble P1X,    //X-Koord P1 (Anfangspunkt der Strecke)
                          tDouble P2X,    //X-Koord P2 (Endpunkt der Strecke)
                          tDouble P3X,    //X-Koord P3 (Punkt von dem Lot gefaellt wird)
                          tDouble *P4X,   //X-Koord des Ergebnis (naechster Punkt)
                          tDouble P1Y,    //Y-Koord P1
                          tDouble P2Y,    //Y-Koord P2
                          tDouble P3Y,    //Y-Koord P3
                          tDouble *P4Y,   //Y-Koord P4
                          tBool *LotFlag);     //bei TRUE ist P3P4 senkrecht zu P1P2
//! \ingroup MOD_GENERIC
tDouble dCalcNearestPoint( tDouble P1X,    //X-Koord P1 (Anfangspunkt der Strecke)
                           tDouble P2X,    //X-Koord P2 (Endpunkt der Strecke)
                           tDouble P3X,    //X-Koord P3 (Punkt von dem Lot gefaellt wird)
                           tDouble *P4X,   //X-Koord des Ergebnis (naechster Punkt)
                           tDouble P1Y,    //Y-Koord P1
                           tDouble P2Y,    //Y-Koord P2
                           tDouble P3Y,    //Y-Koord P3
                           tDouble *P4Y,   //Y-Koord P4
                           tBool *LotFlag);     //bei TRUE ist P3P4 senkrecht zu P1P2

//! is the point (x3, y3) inside the linesegment (x1,y1)-(x2,y2)? 
//! the function assumes, that (x3,y3) is on the line defiend by (x1,y1)-(x2,y2)!! 
//! \ingroup MOD_GENERIC
tBool PointInLine(tDouble x1, tDouble y1,
                  tDouble x2, tDouble y2,
                  tDouble x3, tDouble y3, tBool bInclude2ndPoint = FALSE);

//! defines the intersection of two lines given by (x1,y1)-(x2,y2) and line (x3,y3)-(x4,y4)
//! the intersection point is (*IntersectionX, *IntersectionY).
//! Function returns TRUE, if intersection exists.*/
//! \ingroup MOD_GENERIC
tBool CalcIntersectionTwoLines( tDouble x1, tDouble x2, tDouble x3, tDouble x4,
                                tDouble y1, tDouble y2, tDouble y3, tDouble y4,
                                tDouble *IntersectionX, tDouble *IntersectionY);
//! \ingroup MOD_GENERIC
tBool CalcIntersectionTwoSections( tDouble dX1, tDouble dX2, tDouble dX3, tDouble dX4,
                                   tDouble dY1, tDouble dY2, tDouble dY3, tDouble dY4,
                                   tDouble *IntersectionX, tDouble *IntersectionY );
//! \ingroup MOD_GENERIC
template<class T> inline const T& math_max( const T& x, const T& y )
{
   return ((x < y) ? y : x);
}
//! \ingroup MOD_GENERIC
template<class T> inline const T& math_min( const T& x, const T& y )
{
   return ((y < x) ? y : x);
}

//! do not use math_abs() with unsigned types: possible error !!
//! if using with unsigned types, compiler throw "error ...: ... ambiguous call to overloaded function"
//! \ingroup MOD_GENERIC
inline tS64 math_abs( const tS64 & x )
{
   //trick using x == -x will not work on all machines (compiler will remove the code ...)
   //use max values instead
   const tS64 min_negative = (tS64)( ((tU64)1) << 63 );
   const tS64 max_positive = ~min_negative;
   tS64 s64Null = (tS64)0;
   if(x < s64Null)
   {
      if( min_negative == x )
      {
         return max_positive;
      }
      else
      {
         return -x;
      }
   }
   else
   {
      return x;
   }
}
//! \ingroup MOD_GENERIC
inline tS32 math_abs( const tS32 & x )
{
   if( x < 0 )
   {
      if( OSAL_C_S32_MIN == x )
      {
         return (OSAL_C_S32_MAX);
      }
      else
      {
         return (-x);
      }
   }
   else
   {
      return (x);
   }
}
//! \ingroup MOD_GENERIC
inline tS16 math_abs( const tS16 & x )
{
   if( x < 0 )
   {
      if( OSAL_C_S16_MIN == x )
      {
         return (OSAL_C_S16_MAX);
      }
      else
      {
         return (-x);
      }
   }
   else
   {
      return (x);
   }
}
//! \ingroup MOD_GENERIC
inline tS8 math_abs( const tS8 & x )
{
   if( x < 0 )
   {
      if( OSAL_C_S8_MIN == x )
      {
         return (OSAL_C_S8_MAX);
      }
      else
      {
         return (-x);
      }
   }
   else
   {
      return (x);
   }
}
//! \ingroup MOD_GENERIC
inline tFloat math_abs( const tFloat& x )
{
   //riesen cast Aufwand auf double und zurueck, da einer der Compiler kein fabs von tFloat unterstuetzt!!!
   return (tFloat)( fabs(x) );
}
//! \ingroup MOD_GENERIC
inline tDouble math_abs( const tDouble& x)
{
   return fabs(x);
}
//! \ingroup MOD_GENERIC
inline tS32 sgn(tDouble val)
{
   return ((val < 0) ? -1 : ((val > 0) ? +1 : 0));
}

#endif  //* MATHGENERIC_HEADER */
