/*****************************************************************************
* FILE:         veda_appl_main.cpp
* PROJECT:      G3G project
* SW-COMPONENT: VehicleData
*----------------------------------------------------------------------------
*
* DESCRIPTION:  Main Application handler for VehicleData
*              
*----------------------------------------------------------------------------
* COPYRIGHT:    (c) 2015 Robert Bosch GmbH, Hildesheim
*****************************************************************************/


/******************************************************************************/
/* INCLUDES                                                                   */
/******************************************************************************/
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

// include base class
#include "veda_appl_handler.h"
#include "Iveda_ClientVcan.h"
#include "veda_AppMain.h"

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_generic_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_VEHICLEDATA_APPL_HANDLER
#include "trcGenProj/Header/veda_appl_handler.cpp.trc.h"
#endif

//#define HANDLE_DELETE(poClass)   if(poClass) { delete poClass; poClass = NULL;}
#define ONE_SEC_TIME_INTERVAL	1000
#define INVALID_CAN_VALUE 0xFF
#define ISA_CONFIGURATION_BIT 0x08

/**
*  Default Constructor
*/
veda_tclApplHandler::veda_tclApplHandler()
: Iveda_tclApplHandler(NULL)
, _bUnderVoltageOngoing(FALSE)
, _bIsIgnitionOn(FALSE)
, _poCCAServiceHdl(NULL)
, _poCANClientHdl(NULL)
, _GpioId_Reverse(0)
, _hGpioHandle_Reverse(OSAL_ERROR)
{  
   ETG_TRACE_ERR(("veda_tclApplHandler() entered."));

   _tCSMEnggData.clear();

   // Default constructor should not be used
   NORMAL_M_ASSERT_ALWAYS();
};

/**
* User-defined constructor
* @param poMainAppl
*/
veda_tclApplHandler::veda_tclApplHandler(veda_tclAppMain* poMainAppl) 
: Iveda_tclApplHandler(poMainAppl)
, _bUnderVoltageOngoing(FALSE)
, _bIsIgnitionOn(FALSE)
, _poCCAServiceHdl(NULL)
, _poCANClientHdl(NULL)
, _GpioId_Reverse(0)
, _hGpioHandle_Reverse(OSAL_ERROR)
{
   ETG_TRACE_USR4(("veda_tclApplHandler() entered."));

   _tCSMEnggData.clear();
};

/**
* Destructor
*/
veda_tclApplHandler::~veda_tclApplHandler()
{
   _poCCAServiceHdl = NULL;
   _poCANClientHdl = NULL;
};

/**
* vHandleMessage
* @param pMsg
* @return
*/

tVoid veda_tclApplHandler::vHandleMessage(TMsg* pMsg)
{
   ETG_TRACE_USR4(("veda_tclApplHandler::vHandleMessage() entered %u -> data: %d.", ETG_CENUM(vehfunc_tclBaseIf::ECmdTypes, pMsg->eCmd), pMsg->u.u32Data));
   switch(pMsg->eCmd)
   {
   case eCvmEvent:
      {
         if ( pMsg->u.u32Data == 0x00 )
         {
            // Abnormal Voltage Ended
            _vHandleAbnormalVoltageEnd();
         }
         else if ( pMsg->u.u32Data == 0x01 )
         {
            // Abnormal Voltage Started
            _vHandleAbnormalVoltageStart();
         }
      }
      break;
   case eReverseGearStateChanged:
      {
         //! Read the reverse gear state from GPIO and update the clients
         vUpdateReverseGear(_bReadGPIOState_Reverse());
      }
      break;
   case eGetCSMEnggData:
      {
         vGetCSMEngineeringData();
      }
      break;
   default:
      break;
   }
}

/**
* vGetReferences()
* @return
*/
tVoid veda_tclApplHandler::vGetReferences()
{
   ETG_TRACE_USR4(("veda_tclApplHandler::vGetReferences() entered."));

   _poCCAServiceHdl = dynamic_cast<Iveda_tclServiceVehicleIf*>(_cpoMain->getHandler("Iveda_tclServiceVehicleIf"));
   VEDA_NULL_POINTER_CHECK(_poCCAServiceHdl);

   _poCANClientHdl = dynamic_cast<I_veda_tclClientVcan*>(_cpoMain->getHandler("I_veda_tclClientVcan"));
   VEDA_NULL_POINTER_CHECK(_poCANClientHdl);
}

/**
* vHandleTraceMessage
* \param[in]  puchData
* \return     None
*/
tVoid veda_tclApplHandler::vHandleTraceMessage(const tUChar* puchData)
{
   VEDA_NULL_POINTER_CHECK(puchData);

   tU32 u32MsgCode = ((puchData[1]<<8) | puchData[2]);
   ETG_TRACE_USR4(("veda_tclApplHandler::vHandleTraceMessage(): trace command %d", u32MsgCode ));

   switch (u32MsgCode)
   {
   case VEHICLEDATA_VEDA_REVERSEGEAR_INPUT:
      {
         // Input the Reverse Gear CSM Data input
         tU8 param1 = puchData[3];

         vUpdateReverseGear(param1);
      }
      break;

   case VEHICLEDATA_VEDA_NETWORKMSG_INPUT:
      {
         tU32 u32MessageId = ((tU32)puchData[3]<<24) | ((tU32)puchData[4]<<16) | ((tU32)puchData[5] << 8) | (tU32)puchData[6];
         tU32 u32MessageStatus = ((tU32)puchData[7]<<24) | ((tU32)puchData[8]<<16) | ((tU32)puchData[9] << 8) | (tU32)puchData[10];
         tU8 u8Payload[8] = {0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF};

         memcpy(u8Payload, (puchData+11), 8);

         vUpdateCSMEngineeringData(u32MessageId, u32MessageStatus, u8Payload);
      }
      break;

   default:
      break;
   };
}

/**
* vTraceInfo
* \return
*/
tVoid veda_tclApplHandler::vTraceInfo()
{
   ETG_TRACE_USR4(("veda_tclApplHandler::vTraceInfo() entered."));

   ETG_TRACE_USR4(("veda_tclApplHandler::vTraceInfo() _bUnderVoltageOngoing %u", _bUnderVoltageOngoing));
}

/**
* vStartCommunication
* @return
*/
tVoid veda_tclApplHandler::vStartCommunication()
{
   ETG_TRACE_USR4(("veda_tclApplHandler::vStartCommunication() entered."));

   _vConfigureGpio_Reverse();
   
   // to read the current reverse gear state
   tChar strIfName[] = "Iveda_tclApplHandler";
   veda_tclAppMain::theServer()->vPostMsgReverseGearStateChanged(strIfName, TRUE);
}

void veda_tclApplHandler::_vHandleAbnormalVoltageEnd(void)
{
   _bUnderVoltageOngoing = FALSE;
}

void veda_tclApplHandler::_vHandleAbnormalVoltageStart(void)
{
   _bUnderVoltageOngoing = TRUE;
}

void veda_tclApplHandler::vUpdateReverseGear(const tU8 u8ReverseGear) const
{
   ETG_TRACE_USR4(("vUpdateReverseGear() Updated State = %d", ETG_ENUM(VEDA_E8_VEHICLE_REVERSE, u8ReverseGear)));
   VEDA_NULL_POINTER_CHECK(_poCCAServiceHdl);
   _poCCAServiceHdl->sendReverseGearPinStatus(u8ReverseGear);
}

OSAL_tpfGPIOCallback veda_tclApplHandler::_pfCallbackGpioStateChange_Reverse(tVoid* pArg)
{
   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(pArg);

   ETG_TRACE_USR3(("_vCallbackGpioStateChange_Reverse() entered"));

   //post message STATE_CHANGE_REVERSE
   tChar strIfName[] = "Iveda_tclApplHandler";
   veda_tclAppMain::theServer()->vPostMsgReverseGearStateChanged(strIfName, TRUE);

   return (0);
}

void veda_tclApplHandler::_vConfigureGpio_Reverse(void)
{
   ETG_TRACE_USR4(("_vConfigureGpio_Reverse() entered."));

#ifdef OSAL_GEN4 //compiler issue GEN4 --> to be checked
   return;
#endif

   //#if not defined (LSIM)
   tU32  u32OSALError;
   OSAL_trGPIOData  pinData;
   OSAL_trGPIOCallbackData  gpioCallbackData;

   _GpioId_Reverse = (OSAL_tGPIODevID) OSAL_EN_REVERSE_DETECT;
   _hGpioHandle_Reverse = OSAL_IOOpen(OSAL_C_STRING_DEVICE_GPIO, OSAL_EN_READWRITE);
   if (_hGpioHandle_Reverse == OSAL_ERROR)
   {
      u32OSALError = OSAL_u32ErrorCode();
      ETG_TRACE_ERR(("_vConfigureGpio_Reverse() OSAL_IOOpen for dev_GPIO failed. 'Osal Error = %u', %s", u32OSALError, OSAL_coszErrorText(u32OSALError)));
   }
   else
   {
      // Configure pin as input
      if (OSAL_ERROR == OSAL_s32IOControl(_hGpioHandle_Reverse, OSAL_C_32_IOCTRL_GPIO_SET_INPUT, (intptr_t) _GpioId_Reverse))
      {
         u32OSALError = OSAL_u32ErrorCode();
         ETG_TRACE_ERR(("_vConfigureGpio_Reverse() Set GPIO as input failed. 'Osal Error = %u', %s", u32OSALError, OSAL_coszErrorText(u32OSALError)));
      }
      else
      {
         // parameter to pass to the call-back function
         gpioCallbackData.rData.unData.pfCallback = (OSAL_tpfGPIOCallback) _pfCallbackGpioStateChange_Reverse; //my call-back function
         gpioCallbackData.pvArg = (tVoid*) this;
         gpioCallbackData.rData.tId = _GpioId_Reverse;

         //parameter for setting the trigger edge
         pinData.unData.u16Edge = OSAL_GPIO_EDGE_BOTH;
         pinData.tId = _GpioId_Reverse;

         // register call-back
         if (OSAL_ERROR == OSAL_s32IOControl(_hGpioHandle_Reverse, OSAL_C_32_IOCTRL_GPIO_SET_CALLBACK, (intptr_t)(&gpioCallbackData)))
         {
            u32OSALError = OSAL_u32ErrorCode();
            ETG_TRACE_ERR(("_vConfigureGpio_Reverse() callback registration failed. 'Osal Error = %u', %s", u32OSALError, OSAL_coszErrorText(u32OSALError)));
         }
         // Set trigger edge BOTH
         else if (OSAL_ERROR == OSAL_s32IOControl(_hGpioHandle_Reverse, OSAL_C_32_IOCTRL_GPIO_SET_TRIGGER, (intptr_t)(&pinData)))
         {
            u32OSALError = OSAL_u32ErrorCode();
            ETG_TRACE_ERR(("_vConfigureGpio_Reverse() Set Trigger Edge (Both Edges) failed. 'Osal Error = %u', %s", u32OSALError, OSAL_coszErrorText(u32OSALError)));
            // unregister call-back
            gpioCallbackData.rData.unData.pfCallback = NULL;
            OSAL_s32IOControl(_hGpioHandle_Reverse, OSAL_C_32_IOCTRL_GPIO_SET_CALLBACK, (intptr_t)&gpioCallbackData);
            //NORMAL_M_ASSERT_ALWAYS();
         }
         else
         {
            //parameter for setting enabling interrupt
            pinData.unData.bState = TRUE;
            // enable interrupts
            if (OSAL_s32IOControl(_hGpioHandle_Reverse, OSAL_C_32_IOCTRL_GPIO_ENABLE_INT, (intptr_t)(&pinData)) == OSAL_ERROR)
            {
               u32OSALError = OSAL_u32ErrorCode();
               ETG_TRACE_ERR(("_vConfigureGpio_Reverse() Enable Interrupt failed. 'Osal Error = %u', %s", u32OSALError, OSAL_coszErrorText(u32OSALError)));

               // unregister call-back
               gpioCallbackData.rData.unData.pfCallback = NULL;
               OSAL_s32IOControl(_hGpioHandle_Reverse, OSAL_C_32_IOCTRL_GPIO_SET_CALLBACK, (intptr_t)&gpioCallbackData);
            }
         }
      }
   }
}

void veda_tclApplHandler::_vCloseGpio_Reverse(void)
{
   ETG_TRACE_USR4(("_vCloseGpio_Reverse() entered."));

   //#if not defined (LSIM)
   OSAL_trGPIOData  pinData;
   OSAL_trGPIOCallbackData  gpioCallbackData;

   if ((_GpioId_Reverse != 0) && (_hGpioHandle_Reverse != OSAL_ERROR))
   {
      // disable interrupt
      // parameter for setting disabling interrupt
      pinData.tId = _GpioId_Reverse;
      pinData.unData.bState = FALSE;
      (tVoid) OSAL_s32IOControl(_hGpioHandle_Reverse, OSAL_C_32_IOCTRL_GPIO_ENABLE_INT, (intptr_t)(&pinData));

      // unregister call-back
      // parameter to pass to the call-back function
      gpioCallbackData.rData.unData.pfCallback = NULL;
      gpioCallbackData.pvArg = (tVoid*) this;
      gpioCallbackData.rData.tId = _GpioId_Reverse;
      (tVoid) OSAL_s32IOControl(_hGpioHandle_Reverse, OSAL_C_32_IOCTRL_GPIO_SET_CALLBACK, (intptr_t)&gpioCallbackData);

      // close device
      (tVoid) OSAL_s32IOClose(_hGpioHandle_Reverse);
      _hGpioHandle_Reverse = OSAL_ERROR;
   }

   //#endif // #if not defined (LSIM)
}

bool veda_tclApplHandler::_bReadGPIOState_Reverse(void)
{
   ETG_TRACE_USR4(("_vReadGPIOState_Reverse() entered."));

   bool bRet = FALSE;
   OSAL_trGPIOData  gpioData;
   tU32  u32OSALError;

   if ((_GpioId_Reverse != 0) && (_hGpioHandle_Reverse != OSAL_ERROR))
   {
      gpioData.tId = _GpioId_Reverse;

      if (OSAL_ERROR == OSAL_s32IOControl(_hGpioHandle_Reverse, OSAL_C_32_IOCTRL_GPIO_IS_STATE_ACTIVE, (intptr_t)(&gpioData)))
      {
         u32OSALError = OSAL_u32ErrorCode();
         ETG_TRACE_ERR(("_vReadGPIOState_Reverse() Get GPIO State failed. 'Osal Error = %u', %s", u32OSALError, OSAL_coszErrorText(u32OSALError)));
      }
      else
      {
         //state
         bRet = gpioData.unData.bState;
         ETG_TRACE_USR3(("Reverse Gear State = %u", bRet));
      }
   }

   return bRet;
}

void veda_tclApplHandler::vGetCSMEngineeringData(void)
{
   VEDA_NULL_POINTER_CHECK(_poCCAServiceHdl);
   VEDA_NULL_POINTER_CHECK(_poCANClientHdl);

   // This telegram will not be supported anymore planned to be deprecated
   // Get All the Telegrams needed
   //_poCANClientHdl->vGetUSP_Status_Request();

   // Update to client as Method Result
   //_poCCAServiceHdl->sendCSMEngineeringDataMResult(_tCSMEnggData);
}

void veda_tclApplHandler::vUpdateCSMEngineeringData(tU32 u32MessageId, tU32 u32Status, tU8* u8Payload)
{
   tBool bDataUpdated = FALSE;

   vehicle_fi_tcl_CAN_Message oCANMsg;

   oCANMsg.u32MessageId = u32MessageId;
   oCANMsg.u32MessageStatus = u32Status;
   for (tU8 i = 0; i < 8; i++)
   {
      oCANMsg.listData.push_back(*(u8Payload+i));
   }

   bpstl::vector<vehicle_fi_tcl_CAN_Message, bpstl::allocator<vehicle_fi_tcl_CAN_Message> >::iterator it = _tCSMEnggData.begin();

   for (; it != _tCSMEnggData.end(); it++)
   {
      //! If the CAN Telegram already present in the vector (Update the data)
      if ( u32MessageId == it->u32MessageId )
      {
         it->u32MessageStatus = u32Status;
         it->listData = oCANMsg.listData;
         bDataUpdated = TRUE;
      }
   }

   //! If a new Telegram data push it onto the vector
   if (!bDataUpdated)
   {
      _tCSMEnggData.push_back(oCANMsg);
   }
}

void veda_tclApplHandler::vTraceCSMEnggDataInfo() const
{
   bpstl::vector<vehicle_fi_tcl_CAN_Message, bpstl::allocator<vehicle_fi_tcl_CAN_Message> >::const_iterator it = _tCSMEnggData.begin();

   for (; it != _tCSMEnggData.end(); it++)
   {
      ETG_TRACE_USR4(("vTraceCSMEnggDataInfo() MsgId %u MsgStatus %u", it->u32MessageId, it->u32MessageStatus));

      for ( tU8 i = 0; i < 8; i++ )
      {
         ETG_TRACE_USR4(("vTraceCSMEnggDataInfo() Payload Byte[%u]: %u", i, it->listData[i]));
      }
   }
}



// EOF


