/**
 * @copyright    (C) 2012 - 2016 Robert Bosch GmbH.
 *               The reproduction, distribution and utilization of this file as well as the
 *               communication of its contents to others without express authorization is prohibited.
 *               Offenders will be held liable for the payment of damages.
 *               All rights reserved in the event of the grant of a patent, utility model or design.
 * @brief        Function for feature undef element
 * @addtogroup   Datapool undef
 * @{
 */
#include <fcntl.h>	
#include <unistd.h>
#include <sys/mman.h>
#include <sys/stat.h>
#include <errno.h>
#include <grp.h>


#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

#define OSAL_S_IMPORT_INTERFACE_GENERIC		   
#include "osal_if.h"

#define DP_S_IMPORT_INTERFACE_BASE
#define DP_S_IMPORT_INTERFACE_DPPERSACCESS
#define DP_S_IMPORT_INTERFACE_DPSEM
#include "dp_if.h"

#include "ccaservice.h"

#ifdef DP_FEATURE_UNDEF_ELEMENT

#define DP_UNDEF_TRACE_CLASS              (tU16)(TR_COMP_DATAPOOL + 0x05) 

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS DP_UNDEF_TRACE_CLASS
#include "trcGenProj/Header/dpUndef.cpp.trc.h"
#endif

/******************************************************************************* 
|defines and macros 
|------------------------------------------------------------------------------*/

/* name for semaphore*/
#define DP_SEM_NAME_INIT_LOCK             "DP-SemUNInit"
#define DP_SEM_NAME_INIT_LOCK_IN_SHM      "/dev/shm/sem.DP-SemUNInit"
/* name for shared memory*/
#define DP_SHM_GLOBAL_NAME         "DP-Shm" 

/* for hash function */
#define DP_HASH_M                31
#define DP_HASH_FREE              0  //should not set to free;
#define DP_HASH_USE               1  //set in s32InsertHash; 
#define DP_HASH_INVALID           2  //set to s32DeletePoolProperty; could used for new hash => check in

/*offset*/
#define DP_INVALID_OFFSET        0xFFFFFFFF

/*macro for null pointer check*/
#define DP_NULL_POINTER_CHECK_FATAL_ASSERT(pointer) \
if (pointer == NULL) {\
FATAL_M_ASSERT_ALWAYS();\
}

/******************************************************************************/
/* static  variable                                                           */
/******************************************************************************/
dp_tclDpUndef* dp_tclDpUndef::_poInstance=NULL;

// *****************************************************************************
// ********   initialization part 
// *****************************************************************************
dp_tclDpUndef* dp_tclDpUndef::pGetInstance()
{
   if (_poInstance == NULL)
   {
      _poInstance = new dp_tclDpUndef();
   }
   return _poInstance;
}
/******************************************************************************
* dp_tclDpUndef::dp_tclDpUndef() 
*
* DESCRIPTION: constructor
*
*****************************************************************************/
dp_tclDpUndef::dp_tclDpUndef()
{
   _u32OffsetPool=sizeof(tDpShmManage)+DP_SIZE_HASH_TABLE;
   _pvShmProperty=NULL;
   _u32ShmLength=0;
   _poPersMemAccessHdl=NULL;
}
/******************************************************************************
* dp_tclDpUndef::~dp_tclDpUndef() 
*
* DESCRIPTION: destructor
*
*****************************************************************************/
dp_tclDpUndef::~dp_tclDpUndef() 
{
	//todo ??? => delete resources
   ETG_TRACE_USR4(("dp_tclDpUndef(): destructor"));
   _u32OffsetPool=0;
   _pvShmProperty=NULL;
   _u32ShmLength=0;
   _poPersMemAccessHdl=NULL;
}
/******************************************************************************
* void dp_tclDpUndef::vInitInstance(dp_tclDpPersMemAccess* poPersMem)
*
* DESCRIPTION: if shared memory pointer unknown, determine shared memory pointer.
*
* PARAMETERS:  access handle for persistent 
*
* RETURNS: void  
*****************************************************************************/
void dp_tclDpUndef::vInitInstance(dp_tclDpPersMemAccess* poPersMem)
{
    _poPersMemAccessHdl = poPersMem; 
 
    /*if shared memory pointer not valid*/
    if (_pvShmProperty==NULL)
    { 
	    tBool   VbFirstAttach = TRUE;
        sem_t*  VSemLockInit;      
        tS32	   VShmId;  
        dp_tclDpSem  VmyDpSem;
	
	   _u32ShmLength =DP_SIZE_HASH_TABLE+DP_SIZE_ALL_PROPERTY_TOOL;
	   /* protect against initializing race, the second process has to wait until this is finished*/
       /* Linux Programmer's Manual: 
        If O_CREAT is specified in oflag, then the semaphore is created if it does not
        already exist. If both O_CREAT and O_EXCL are
        specified in oflag, then an error is returned if a semaphore with the given
        name already exists.*/
        /* try to create semaphore with name;(if success semaphore's value is set to zeror)*/
        VSemLockInit = sem_open(DP_SEM_NAME_INIT_LOCK, O_EXCL | O_CREAT, VmyDpSem.s32GetAccessRight(), 0);
        if(VSemLockInit != SEM_FAILED)
        { /* no error; semaphore create => first attach */
            /* first attach */
            VbFirstAttach = TRUE;
            /* create shared memory at first process attach */
            VShmId= shm_open(DP_SHM_GLOBAL_NAME, O_EXCL|O_RDWR|O_CREAT|O_TRUNC, VmyDpSem.s32GetAccessRight());
            if(VShmId != -1)
            {
                if(ftruncate(VShmId, _u32ShmLength ) == -1)
                {
                    VShmId=-1;
                }
            }
            else
                ETG_TRACE_ERR(("dp_tclDpUndef::shm_open failed %d ",errno));
        }
        else
        {  /* open semaphore*/
            VSemLockInit= sem_open(DP_SEM_NAME_INIT_LOCK, 0);
            if(VSemLockInit != SEM_FAILED)
            {
                /* lock semaphore*/
                sem_wait(VSemLockInit);
                /* not first attach*/
                VbFirstAttach = FALSE;    
                /* open shared memory*/
                VShmId = shm_open(DP_SHM_GLOBAL_NAME, O_RDWR ,0);  
                if(VShmId == -1)
                    ETG_TRACE_ERR(("dp_tclDpEndUser(): vInitInstance(): shm_open without create failed %d",errno));                  
            }
            else
            {
                VShmId=-1;
                ETG_TRACE_ERR(("dp_tclDpUndef::sem_open without create failed %d ",errno));
            }
        }
        if(VShmId!=-1)
        { /* map global shared memory into address space */
            _pvShmProperty = mmap( NULL,_u32ShmLength,PROT_READ | PROT_WRITE,MAP_SHARED,VShmId,0);       
            /* if no error */
            if(_pvShmProperty != MAP_FAILED)
            { /* if first proccess access */
                if(VbFirstAttach)
                {           
                    /*set all to zero*/            
                    memset(_pvShmProperty,0,_u32ShmLength);    
                    /* create semaphore for access shm property*/          
                    VmyDpSem.vCreateSemShm(&ptGetPointerShmManage()->semShmProperty); 
                    /* create semaphore for access shm elements*/          
                    VmyDpSem.vCreateSemShm(&ptGetPointerShmManage()->semShmElements); 
                    /*set access rights to Group "eco_pdd*/
                    VmyDpSem.s32ChangeGroupAccess(DP_KIND_RES_SHM,VShmId,"VShmId");       
                    VmyDpSem.s32ChangeGroupAccess(DP_KIND_RES_SEM,-1,DP_SEM_NAME_INIT_LOCK_IN_SHM);    
                    /*clear list*/
                    _tListSharedUndefElements.clear();
                    /* 10.11.2014: bta2hi: load properties from file deactivate, for preparing datapool to read early configuration data from NOR without access to FS;
                    Now the function is calles in s32GetPoolProperty; set size to zero for indication if properties load */ 
                    /*s32LoadPoolProperties(); */
                    VmyDpSem.s32LockSemShm(&ptGetPointerShmManage()->semShmProperty);
                    ptGetPointerShmManage()->s32SizeFile=0;
                    VmyDpSem.s32UnLockSemShm(&ptGetPointerShmManage()->semShmProperty);
                }        
            }
            /* init finished, let go! */
            sem_post(VSemLockInit);
        }
    }
}

/******************************************************************************
* tS32 dp_tclDpUndef::s32GetPoolProperty()
*
* DESCRIPTION: get pool property
*
* PARAMETERS:  PpPoolProperty: pointer for pool property
*              PeFunctionCall: function called from function
*              PpDpElement: element information
*
* RETURNS: 
*   DP_S32_ERR_NO_POOL: if pool not exist(error for get; delete or shared)
*   DP_S32_ERR_UNKNOWN:
*   DP_S32_NO_ERR:
*   DP_S32_ERR_ACCESS_RIGHT
*****************************************************************************/
tS32 dp_tclDpUndef::s32GetPoolProperty(tDpUndefPoolProperty* PpPoolProperty,EDpUndefCall PeFunctionCall,dp_tclBaseElement* PpDpElement)
{
  tS32        Vs32Result=DP_S32_ERR_UNKNOWN;
  tS32        Vs32HashValue;
  tU32        Vu32Offset;
  dp_tclDpSem VmyDpSem;
  /* lock semaphore*/
  DP_NULL_POINTER_CHECK_FATAL_ASSERT(_pvShmProperty);
  VmyDpSem.s32LockSemShm(&ptGetPointerShmManage()->semShmProperty);
  /* 10.11.2014: bta2hi: load properties from file call now in function s32GetPoolProperty() , 
  for preparing datapool to read early configuration data from NOR without access to FS;*/
  s32LoadPoolProperties();
  /* get hash from pool name */
  Vs32HashValue=s32GetHash(PpPoolProperty->strPoolName);
  if(Vs32HashValue<0)
  { /* pool doesn't exist*/
    if((PeFunctionCall==eDpUndefGet)||(PeFunctionCall==eDpUndefDelete)||(PeFunctionCall==eDpUndefShared)||(PeFunctionCall==eDpUndefPerformDefSet))
	  {
	    Vs32Result=DP_S32_ERR_NO_POOL;
	    ETG_TRACE_ERR(("dp_tclDpUndef::s32GetPoolProperty() pool not found '%s'",PpPoolProperty->strPoolName));
	  }
	  else
	  { //create new pool properties
	    PpPoolProperty->u32PoolId=(u32GetFreePoolId())<<16;
	    PpPoolProperty->u32PoolVersion=DP_POOL_VERSION_UNDEF;
	    strncpy(PpPoolProperty->strProcessNameCreated,PpPoolProperty->strProcessNameLocated,DP_MAX_LEN_NAME_PROCESS);
        PpPoolProperty->strProcessNameCreated[DP_MAX_LEN_NAME_PROCESS-1] = '\0';
	    PpPoolProperty->u16Access=0;
	    /*for init check if element should be shared over proccesses*/
	    if(PeFunctionCall==eDpUndefInit)
	    {/*set access pool*/
        PpPoolProperty->u16Access=(PpDpElement->u16GetAccessType()>>8);
		    if(PpPoolProperty->u16Access!=0)  
			  ETG_TRACE_USR4(("dp_tclDpUndef::s32GetPoolProperty() shared pool access:%d",PpPoolProperty->u16Access)); 
	    }
      /* insert pool property and get offset*/
	    Vu32Offset=u32PushBackProperty(PpPoolProperty);  
	    if(Vu32Offset!=DP_INVALID_OFFSET)
	    { /* insert hash */
	      Vs32HashValue=s32InsertHash(PpPoolProperty->strPoolName,Vu32Offset);   
	      /* pool insert */
	      if(Vs32HashValue>0)
	      { /*property change save to file*/
		      Vs32Result=s32SavePoolProperties();	
	      }
 	    }
	  }
  }
  else
  { /* pool exist*/
    tDpShmHashTable*       VptHashTable=ptGetPointerShmHashTable()+Vs32HashValue;
	  tDpUndefPoolProperty*  VptPoolProp=ptGetPointerShmPoolProperty(VptHashTable->u32Offset);
	  tBool                  VbAccessPool=FALSE;

	  ETG_TRACE_USR4(("dp_tclDpUndef::s32GetPoolProperty() pool exist %s",PpPoolProperty->strPoolName)); 
    //check if the same process creates the pool property
	  if(strncmp(VptPoolProp->strProcessNameCreated,PpPoolProperty->strProcessNameLocated,DP_MAX_LEN_NAME_PROCESS)==0)
	  { /*copy process name located into shared memory*/
	    VbAccessPool=TRUE;  
	    /*if init: set access pool*/
	    if(PeFunctionCall==eDpUndefInit)
	    {/*set access pool*/
        VptPoolProp->u16Access|=(PpDpElement->u16GetAccessType()>>8);
	      if(VptPoolProp->u16Access!=0)  
		  	  ETG_TRACE_USR4(("dp_tclDpUndef::s32GetPoolProperty() shared pool access:%d",VptPoolProp->u16Access)); 
	    }
	  }
	  else
	  { // not the same process
      // check pools can be shared 
	    switch(PeFunctionCall)
	    {
	      case eDpUndefGet:
	      {//Check if pool read access 
	        if((VptPoolProp->u16Access&0x0004)==0x0004)
		      {//check element shared read
		        VbAccessPool=bCheckElementShouldBeShared(VptPoolProp,PeFunctionCall,PpDpElement);
		      }
	      }break;
	      case eDpUndefSet:
	      case eDpUndefDelete:
	      {//Check if pool write access 
	        if((VptPoolProp->u16Access&0x0002)==0x0002)
		      {//check element shared write
		        VbAccessPool=bCheckElementShouldBeShared(VptPoolProp,PeFunctionCall,PpDpElement);
		      } 
	      }break;
		    case eDpUndefShared: //process should be located in the same process
		    {
		      ETG_TRACE_ERR(("dp_tclDpUndef::s32GetPoolProperty() pool '%s' located the data",PpPoolProperty->strProcessNameLocated));
		    }break;
		    case eDpUndefInit://only the create process can change the access rights
		    case eDpUndefPerformDefSet: //only the create process can set to default
		    default:
		    	break;
	    }//end switch
    }
	  if(VbAccessPool==TRUE)
	  { //check if pool always located in any process
	    if(*VptPoolProp->strProcessNameLocated==0)
	    {
	      strncpy(VptPoolProp->strProcessNameLocated,PpPoolProperty->strProcessNameLocated,DP_MAX_LEN_NAME_PROCESS);
	    }
	    /*get pool property*/
	    memmove((void*)PpPoolProperty,(const void*)VptPoolProp,sizeof(tDpUndefPoolProperty));
	    ETG_TRACE_USR4(("dp_tclDpUndef::s32GetPoolProperty() current process '%s' has the access rights",PpPoolProperty->strProcessNameLocated));
	    ETG_TRACE_USR4(("dp_tclDpUndef::s32GetPoolProperty() pool '%s' located the data",VptPoolProp->strProcessNameLocated));
	    Vs32Result=DP_S32_NO_ERR;
	  }
	  else
	  {
	    Vs32Result=DP_S32_ERR_ACCESS_RIGHT;
	    ETG_TRACE_USR4(("dp_tclDpUndef::s32GetPoolProperty() current process has no access rights"));
	  }
  }
  /*unlock semaphore*/
  VmyDpSem.s32UnLockSemShm(&ptGetPointerShmManage()->semShmProperty);
  return(Vs32Result);
}
/******************************************************************************
* tS32    dp_tclDpUndef::s32DeletePoolProperty(tDpUndefPoolProperty* PpPoolProperty)
*
* DESCRIPTION: delete pool property
*
* PARAMETERS:  PpPoolProperty: pool property
*
* RETURNS: DP_S32_NO_ERR or error code
*****************************************************************************/
tS32    dp_tclDpUndef::s32DeletePoolProperty(tDpUndefPoolProperty* PpPoolProperty)
{
  tS32           Vs32Result=DP_S32_NO_ERR;
  tS32           Vs32HashValue;
  dp_tclDpSem    VmyDpSem;
  /* lock semaphore*/
  DP_NULL_POINTER_CHECK_FATAL_ASSERT(_pvShmProperty);
  VmyDpSem.s32LockSemShm(&ptGetPointerShmManage()->semShmProperty);
  /* get hash from pool name */
  Vs32HashValue=s32GetHash(PpPoolProperty->strPoolName);
  if(Vs32HashValue>=0)
  {/*pool found*/
    tDpShmHashTable*       VptHashTable=ptGetPointerShmHashTable()+Vs32HashValue;
    tDpUndefPoolProperty*  VptPoolProp=ptGetPointerShmPoolProperty(VptHashTable->u32Offset);
	  /*set hash to invalid*/
    VptHashTable->u8State=DP_HASH_INVALID;
	  /*set property to invalid*/
    VptPoolProp->bValid=FALSE;
	  ETG_TRACE_USR4(("dp_tclDpUndef::s32DeletePoolProperty() success"));
	  /*property change save to file*/
    Vs32Result=s32SavePoolProperties();	
  }
  else
  {
    ETG_TRACE_ERR(("dp_tclDpUndef::s32DeletePoolProperty() pool not found"));
  }
  /*unlock semaphore*/
  VmyDpSem.s32UnLockSemShm(&ptGetPointerShmManage()->semShmProperty);
  return(Vs32Result);
}

/******************************************************************************
* void dp_tclDpUndef::bGetPoolNameFromId()
*
* DESCRIPTION: call from function vInitUndef() to load pool properties
*
* PARAMETERS: 
*
* RETURNS: void  
*****************************************************************************/
tBool dp_tclDpUndef::bGetPoolNameFromId(tDpUndefPoolProperty* PpPoolProperty)
{
  tBool          VbFound=FALSE;
  dp_tclDpSem    VmyDpSem;
  /* lock semaphore*/
  DP_NULL_POINTER_CHECK_FATAL_ASSERT(_pvShmProperty);
  VmyDpSem.s32LockSemShm(&ptGetPointerShmManage()->semShmProperty);
 /* 10.11.2014: bta2hi: load properties from file call now in function s32GetPoolProperty() , 
  for preparing datapool to read early configuration data from NOR without access to FS;*/
  s32LoadPoolProperties();

  {/*search property by id */
      tDpUndefPoolProperty*  VptPoolProp=ptGetPointerShmPoolProperty(0);
      tU32                   Vu32Size=_u32OffsetPool;
      if(VptPoolProp!=NULL)
      {
	      tU32  Vu32SizeShm=sizeof(tDpShmManage)+ ((tU32) ptGetPointerShmManage()->s32SizeFile);
        do
	      {   
          if ((VptPoolProp->u32PoolId == PpPoolProperty->u32PoolId)&&(VptPoolProp->bValid==TRUE))
		      {
		        VbFound=TRUE;
			      /*copy name*/
            strncpy(PpPoolProperty->strPoolName,VptPoolProp->strPoolName,DP_MAX_LEN_NAME_UNDEFINED_POOLS);
			      ETG_TRACE_USR4(("bGetPoolNameFromId:: Pool %s found\n",PpPoolProperty->strPoolName));
		      }
          VptPoolProp++;
	        Vu32Size+=sizeof(tDpUndefPoolProperty);
        }while((Vu32Size < Vu32SizeShm)&&(VbFound==FALSE));
      }
  }
  /*unlock semaphore*/
  VmyDpSem.s32UnLockSemShm(&ptGetPointerShmManage()->semShmProperty);
  return(VbFound);
}

/******************************************************************************
* tS32 dp_tclDpUndef::s32SaveElementProperty()
*
* DESCRIPTION: save element properties
*
* PARAMETERS:  Pu32PoolId,PtElementProperty,PeLocation
*
* RETURNS: success oor error code
*****************************************************************************/
tS32 dp_tclDpUndef::s32SaveElementProperty(tU32 Pu32PoolId,tDpUndefElementProperty* PtElementProperty,eDpLocation PeLocation)
{
  tS32           Vs32ReturnCode=DP_S32_NO_ERR;
  tS32           Vs32ReadBytes=0;
  tBool          VbDefaultProperty=FALSE;
  tBool          VbPropertyEqual=FALSE;
  tBool          VbReadProperty=TRUE;
  tChar          VstrFileName[DP_MAX_LEN_FILE];
  tU8*           Vpu8BufferRead = NULL;

  if(_poPersMemAccessHdl==NULL)
  {
    Vs32ReturnCode=DP_S32_ERR_UNKNOWN;
    NORMAL_M_ASSERT_ALWAYS();	
  }
  else
  { /*check if element property default*/
    if(  (PtElementProperty->u8Version==DP_UNDEF_ELEMENT_DEFAULT_PROPERTY_VERSION)
	   &&(PtElementProperty->bVarSize==DP_UNDEF_ELEMENT_DEFAULT_PROPERTY_VAR_SIZE)
	   &&(PtElementProperty->u16AccessType==DP_UNDEF_ELEMENT_DEFAULT_PROPERTY_ACCESSS_TYPE)
	   &&(PtElementProperty->tElemType==DP_UNDEF_ELEMENT_DEFAULT_PROPERTY_ELEMENT_TYPE)
	   &&(PtElementProperty->tStoringType==DP_UNDEF_ELEMENT_DEFAULT_PROPERTY_STORING_TYPE)
	   &&(PtElementProperty->u32DefSetMode==DP_UNDEF_ELEMENT_DEFAULT_PROPERTY_DEF_SET_MODE))
    {
      VbDefaultProperty=TRUE;
	    ETG_TRACE_USR4(("dp_tclDpUndef::s32SaveElementProperty() element %s has default property",PtElementProperty->strElementName));
    }
    /*load properties from file */
    _poPersMemAccessHdl->s32GetDataStreamName((tU16) (Pu32PoolId >> 16), DP_UNDEF_FILE_NAME_ELEMENT_PROPERTIES, VstrFileName);
    Vs32ReadBytes = _poPersMemAccessHdl->s32ImportPersData(VstrFileName,&Vpu8BufferRead,PeLocation,DP_UNDEF_VERSION_FILE_PROPERTIES);
    if ((Vs32ReadBytes < 0)||(Vs32ReadBytes < (tS32)sizeof(tDpUndefElementProperty)))
    {
      Vs32ReadBytes=0;
	    VbReadProperty=FALSE;
	    ETG_TRACE_USR4(("dp_tclDpUndef::s32SaveElementProperty() no properties found in file %s ",VstrFileName));
    }
    //check of element name exist
    tDpUndefElementProperty* VtPosElementPropertiesRead = (tDpUndefElementProperty*) ((void*)Vpu8BufferRead);
    tU32                     Vu32Inc;
    tU32                     Vu32MaxNumber=(((tU32)Vs32ReadBytes)/sizeof(tDpUndefElementProperty));
    for (Vu32Inc=0;Vu32Inc < Vu32MaxNumber;Vu32Inc++)
    { //check name equal
      if(strncmp(VtPosElementPropertiesRead->strElementName,PtElementProperty->strElementName,DP_MAX_LEN_NAME_ELEMENT)==0)
	    {//element found 
	      //if element property new
	      if(memcmp(VtPosElementPropertiesRead,PtElementProperty,sizeof(tDpUndefElementProperty))!=0)
	      { 
	        tS32                     Vs32Copy=Vs32ReadBytes-((Vu32Inc+1)*sizeof(tDpUndefElementProperty));
          tDpUndefElementProperty* VtPosElementPropertiesNext=VtPosElementPropertiesRead;
		      ETG_TRACE_USR4(("dp_tclDpUndef::s32SaveElementProperty() different element property %s has already been stored",PtElementProperty->strElementName));
	        VtPosElementPropertiesNext++;
          //copy => clear actual element property
		      if(Vs32Copy>=0)
		      {
	          memmove(VtPosElementPropertiesRead,VtPosElementPropertiesNext,Vs32Copy);
		      }
	        //decrement size
	        Vs32ReadBytes-=sizeof(tDpUndefElementProperty);
        }
	      else
	      {
	        VbPropertyEqual=TRUE;
		      ETG_TRACE_USR4(("dp_tclDpUndef::s32SaveElementProperty()element property '%s' equal stored property",VtPosElementPropertiesRead->strElementName));
        } 
	      break;
      }
	    VtPosElementPropertiesRead++;
    }
    if(VbPropertyEqual==FALSE)
    {
        tU32 Vu32WriteBytes=Vs32ReadBytes;
		tU8* Vpu8BufferWrite;
        if(VbDefaultProperty==FALSE)
        {//add element property
            Vu32WriteBytes+=sizeof(tDpUndefElementProperty);
        }
         try
        {
            Vpu8BufferWrite = new tU8[Vu32WriteBytes];
        }
        catch(const std::bad_alloc&)
        {
            Vpu8BufferWrite = NULL;
            ETG_TRACE_ERR(("dp_tclDpUndef::s32SaveElementProperty(): error -> Unable to allocate buffer %d '%s'.", Vu32WriteBytes, VstrFileName));
        }
	    if(Vpu8BufferWrite==NULL)
	    {
	      Vs32ReturnCode=DP_S32_ERR_UNKNOWN;
		    NORMAL_M_ASSERT_ALWAYS();
	    }
	    else
	    {//copy read data
        memmove(Vpu8BufferWrite,Vpu8BufferRead,Vs32ReadBytes);
        if(VbDefaultProperty==FALSE)
        {	//add property
	        memmove(Vpu8BufferWrite+Vs32ReadBytes,PtElementProperty,sizeof(tDpUndefElementProperty));
        }
        //if data read and Vu32WriteBytes=0 => make file invalid (TODO clear file=> new interface to PDD)
        if((Vu32WriteBytes==0)&&(VbReadProperty))
        {
          Vu32WriteBytes=1;
        }
        /* should bytes write to file?*/
        if(Vu32WriteBytes!=0)
        {  //save file
          if (_poPersMemAccessHdl->bWriteFfsData(VstrFileName,Vpu8BufferWrite,Vu32WriteBytes,
                                                 PeLocation,DP_UNDEF_VERSION_FILE_PROPERTIES,TRUE)==FALSE)
          {
            Vs32ReturnCode=DP_S32_ERR_UNKNOWN;
            ETG_TRACE_ERR(("dp_tclDpUndef::s32SaveElementProperty(): error -> element properties not saved to file '%s'.", VstrFileName));
          }
        }
	      delete[] Vpu8BufferWrite;	
          Vpu8BufferWrite=NULL;  /*Bug 123144: Safe handling*/         
      }
    }
    delete[] Vpu8BufferRead;
    Vpu8BufferRead=NULL; /*Bug 123144: Safe handling*/
  }
  return(Vs32ReturnCode);
}
/******************************************************************************
* tS32 dp_tclDpUndef::s32DeleteElementProperty()
*
* DESCRIPTION: delete element properties
*
* PARAMETERS:  Pu32PoolId,PeLocation,PpListUndefElementProperty
*
* RETURNS: success oor error code
*****************************************************************************/
tS32 dp_tclDpUndef::s32DeleteElementProperty(tU32 Pu32PoolId,tDpUndefElementProperty* PtElementProperty,eDpLocation PeLocation)
{
  tS32  Vs32ReturnCode=DP_S32_NO_ERR;
  tU8*  Vpu8BufferRead = NULL;
  tBool VbDefaultProperty=FALSE;
  tChar VstrFileName[DP_MAX_LEN_FILE];
  
  if(_poPersMemAccessHdl==NULL)
  {
    Vs32ReturnCode=DP_S32_ERR_UNKNOWN;
    NORMAL_M_ASSERT_ALWAYS();
  }
  else
  { /*check if pool element default*/
    if(  (PtElementProperty->u8Version==DP_UNDEF_ELEMENT_DEFAULT_PROPERTY_VERSION)
	   &&(PtElementProperty->bVarSize==DP_UNDEF_ELEMENT_DEFAULT_PROPERTY_VAR_SIZE)
	   &&(PtElementProperty->u16AccessType==DP_UNDEF_ELEMENT_DEFAULT_PROPERTY_ACCESSS_TYPE)
	   &&(PtElementProperty->tElemType==DP_UNDEF_ELEMENT_DEFAULT_PROPERTY_ELEMENT_TYPE)
	   &&(PtElementProperty->tStoringType==DP_UNDEF_ELEMENT_DEFAULT_PROPERTY_STORING_TYPE)
	   &&(PtElementProperty->u32DefSetMode==DP_UNDEF_ELEMENT_DEFAULT_PROPERTY_DEF_SET_MODE))
    {
      VbDefaultProperty=TRUE;
	  ETG_TRACE_USR4(("dp_tclDpUndef::s32DeleteElementProperty() element %s has default property",PtElementProperty->strElementName));
    }
    if(VbDefaultProperty==FALSE)
    {/* load properties from file */
      _poPersMemAccessHdl->s32GetDataStreamName((tU16) (Pu32PoolId >> 16), DP_UNDEF_FILE_NAME_ELEMENT_PROPERTIES, VstrFileName);
      tS32 Vs32ReadBytes = _poPersMemAccessHdl->s32ImportPersData(VstrFileName,&Vpu8BufferRead,PeLocation,DP_UNDEF_VERSION_FILE_PROPERTIES);
      if ((Vs32ReadBytes < 0)||(Vs32ReadBytes < (tS32)sizeof(tDpUndefElementProperty)))
      {
        Vs32ReturnCode=DP_S32_ERR_UNKNOWN;
	      ETG_TRACE_USR4(("dp_tclDpUndef::s32DeleteElementProperty() no properties found in file %s ",VstrFileName));	
      }
      else
      { //check of element name exist
        tDpUndefElementProperty* VtPosElementPropertiesRead = (tDpUndefElementProperty*) ((void*)Vpu8BufferRead);
        tU32                     Vu32Inc;
        tU32                     Vu32MaxNumber=(((tU32)Vs32ReadBytes)/sizeof(tDpUndefElementProperty));
        for (Vu32Inc=0;Vu32Inc < Vu32MaxNumber;Vu32Inc++)
        { //check name equal
          if(strncmp(VtPosElementPropertiesRead->strElementName,PtElementProperty->strElementName,DP_MAX_LEN_NAME_ELEMENT)==0)
	        {//element found 
	          tS32  Vs32Copy=Vs32ReadBytes-((Vu32Inc+1)*sizeof(tDpUndefElementProperty));
            tDpUndefElementProperty* VtPosElementPropertiesNext=VtPosElementPropertiesRead;
		        ETG_TRACE_USR4(("dp_tclDpUndef::s32DeleteElementProperty() element property %s found",PtElementProperty->strElementName));
	          VtPosElementPropertiesNext++;
            //copy => clear actual element property
		        if(Vs32Copy>=0)
		        {
	            memmove(VtPosElementPropertiesRead,VtPosElementPropertiesNext,Vs32Copy);
		        }
	          //decrement size
	          Vs32ReadBytes-=sizeof(tDpUndefElementProperty);
		        // if datasize ==0 make file invalid (TODO clear file => new interface to PDD)
		        if(Vs32ReadBytes==0)  
		          Vs32ReadBytes=1;
	          //write file back
            if (_poPersMemAccessHdl->bWriteFfsData(VstrFileName,Vpu8BufferRead,Vs32ReadBytes,
                                                   PeLocation,DP_UNDEF_VERSION_FILE_PROPERTIES,TRUE)==FALSE)
            {
              Vs32ReturnCode=DP_S32_ERR_UNKNOWN;
              ETG_TRACE_ERR(("dp_tclDpUndef::s32DeleteElementProperty(): error -> element properties not saved to file '%s'.", VstrFileName));
            }
		        break;
          }
	        VtPosElementPropertiesRead++;
        }
      }
      delete[] Vpu8BufferRead;
      Vpu8BufferRead=NULL; /*Bug 123144: Safe handling*/
    }
  }
  return(Vs32ReturnCode);
}
/******************************************************************************
* tS32 dp_tclDpUndef::s32GetElementProperties(void)
*
* DESCRIPTION: get all element properties
*
* PARAMETERS:  Pu32PoolId,PeLocation,PpListUndefElementProperty
*
* RETURNS: success oor error code
*****************************************************************************/
tS32 dp_tclDpUndef::s32GetElementProperties(tU32 Pu32PoolId,eDpLocation PeLocation,tDpUndefElementProperty **PpListUndefElementProperty)
{
  tS32  Vs32ReadBytes=DP_S32_NO_ERR;
  tU8*  Vpu8BufferRead = NULL;
  tChar VstrFileName[DP_MAX_LEN_FILE];
  
  if(_poPersMemAccessHdl==NULL)
  {
    Vs32ReadBytes=DP_S32_ERR_UNKNOWN;
    NORMAL_M_ASSERT_ALWAYS();	
  }
  else
  { /*load properties from file */
    _poPersMemAccessHdl->s32GetDataStreamName((tU16) (Pu32PoolId >> 16), DP_UNDEF_FILE_NAME_ELEMENT_PROPERTIES, VstrFileName);
    Vs32ReadBytes = _poPersMemAccessHdl->s32ImportPersData(VstrFileName,&Vpu8BufferRead,PeLocation,DP_UNDEF_VERSION_FILE_PROPERTIES);
    if ((Vs32ReadBytes < 0)||(Vs32ReadBytes < (tS32)sizeof(tDpUndefElementProperty)))
    {
      Vs32ReadBytes=DP_S32_ERR_UNKNOWN;
	    ETG_TRACE_USR4(("dp_tclDpUndef::s32GetElementProperties() no properties found in file %s ",VstrFileName));
	    delete[] Vpu8BufferRead;
        Vpu8BufferRead=NULL; /*Bug 123144: Safe handling*/
    }
    else
    {
      *PpListUndefElementProperty=(tDpUndefElementProperty*)((void*)Vpu8BufferRead);
	    ETG_TRACE_USR4(("dp_tclDpUndef::s32GetElementProperties() properties found in file %s ",VstrFileName));
	  //delete[] Vpu8BufferRead; delete in calling function
    }
  }
  return(Vs32ReadBytes);
}
/******************************************************************************
* tS32 dp_tclDpUndef::s32CreateShmElement(void)
*
* DESCRIPTION: create shared element
*
* PARAMETERS: PpstrPoolName: poolname
*             PpstrElementName: element name
*             Pu32ElementSize: element size
*             PpvData: pointer to data
*
* RETURNS: size or 0
*****************************************************************************/
tS32    dp_tclDpUndef::s32CreateShmElement(const tChar* PpstrPoolName,const tChar* PpstrElementName,tU32 Pu32ElementSize,tVoid* PpvData)
{
  tS32          Vs32Size=0;
  tS32          Vs32Hash;
  tDpShmElement VtSharedElement;
  dp_tclDpSem   VmyDpSem;
      
  ETG_TRACE_USR4(("dp_tclDpUndef::s32CreateSharedElement() start"));
  memset(&VtSharedElement,0,sizeof(tDpShmElement));
  snprintf(VtSharedElement.strPoolElementName,DP_UNDEF_ELEMENT_NAME_SHARED_MAX_LEN,"%s_%s",PpstrPoolName,PpstrElementName);
  Vs32Hash=s32CalcHash(VtSharedElement.strPoolElementName); 
  //check list if exist
   tDpUndefSharedElement::iterator VtDpPos = _tListSharedUndefElements.find(Vs32Hash);
   if (VtDpPos == _tListSharedUndefElements.end()) 
   { /*create shared element*/
        VtSharedElement.s32ShmId= shm_open(VtSharedElement.strPoolElementName, O_EXCL|O_RDWR|O_CREAT|O_TRUNC, VmyDpSem.s32GetAccessRight());
        if(VtSharedElement.s32ShmId != -1)
        {
            VtSharedElement.u32SizeShmElement=Pu32ElementSize;
            if(ftruncate(VtSharedElement.s32ShmId, VtSharedElement.u32SizeShmElement ) == -1)
            {
                VtSharedElement.s32ShmId=-1;
                ETG_TRACE_ERR(("dp_tclDpUndef::s32CreateShmElement  ftruncate() fails"));
            }
            else
            { 
                VtSharedElement.vpAdrShmElement = mmap( NULL,VtSharedElement.u32SizeShmElement,PROT_READ | PROT_WRITE,MAP_SHARED,VtSharedElement.s32ShmId,0);       
                /* if no error */
                if(VtSharedElement.vpAdrShmElement != MAP_FAILED)
                {  /*set access rights to Group "eco_pdd*/
                    VmyDpSem.s32ChangeGroupAccess(DP_KIND_RES_SHM,VtSharedElement.s32ShmId,"VtSharedElement.s32ShmId");                 
                    /*copy data*/
                    memmove(VtSharedElement.vpAdrShmElement,PpvData,VtSharedElement.u32SizeShmElement);
                    /*add new element*/
                    _tListSharedUndefElements[Vs32Hash] = VtSharedElement;
                    VtDpPos = _tListSharedUndefElements.find(Vs32Hash);
                }
                else
                {
                    ETG_TRACE_ERR(("dp_tclDpUndef::s32CreateShmElement  mmap() fails"));
                }
            }
        }
        else
        {
            ETG_TRACE_ERR(("dp_tclDpUndef::s32CreateShmElement  shm_open() fails"));
        }
   }
   else
   {
     ETG_TRACE_USR4(("dp_tclDpUndef::s32CreateShmElement  shared element exist"));
   }
   if (VtDpPos != _tListSharedUndefElements.end()) 
   { //increment access
     VtDpPos->second.u32ProcessAccess++;
     ETG_TRACE_USR4(("dp_tclDpUndef::s32CreateSharedElement() counter process read %d ",VtDpPos->second.u32ProcessAccess));
       Vs32Size=(tS32) VtSharedElement.u32SizeShmElement;
   }
  return(Vs32Size);
}
/******************************************************************************
* tS32 dp_tclDpUndef::s32SetShmElement()
*
* DESCRIPTION: set shared element
*
* PARAMETERS: 
*
* RETURNS: size
*****************************************************************************/
tS32    dp_tclDpUndef::s32SetShmElement(const tChar* PpstrPoolName,const tChar* PpstrElementName,tU32 Pu32ElementSize,tVoid* PpvData)
{
  tS32          Vs32Size=0;
  tS32          Vs32ShmId;
  void*         VpAdrShmElement=NULL;
  tChar         VstrPoolElementName[DP_UNDEF_ELEMENT_NAME_SHARED_MAX_LEN];



  ETG_TRACE_USR4(("dp_tclDpUndef::s32SetShmElement() start"));
  memset(&VstrPoolElementName,0,sizeof(VstrPoolElementName));
  snprintf(VstrPoolElementName,DP_UNDEF_ELEMENT_NAME_SHARED_MAX_LEN,"%s_%s",PpstrPoolName,PpstrElementName);
  /*open shm element*/
  Vs32ShmId=shm_open(VstrPoolElementName, O_RDWR ,0);
  if(Vs32ShmId==-1) 
  {
    ETG_TRACE_FATAL(("dp_tclDpUndef::s32SetShmElement() shm_open() fails"));
  }
  else
  {
    VpAdrShmElement = mmap( NULL,Pu32ElementSize,PROT_READ | PROT_WRITE,MAP_SHARED,Vs32ShmId,0);       
    /* if no error */
    if(VpAdrShmElement == MAP_FAILED)
    {
        ETG_TRACE_FATAL(("dp_tclDpUndef::s32SetShmElement() mmap() fails"));
    }
    else
    {/*copy data*/
        Vs32Size=Pu32ElementSize;
        memmove(VpAdrShmElement,PpvData,Vs32Size);
        ETG_TRACE_USR4(("dp_tclDpUndef::s32SetShmElement() success"));
        /*munmap*/
      if(munmap(VpAdrShmElement,Vs32Size) != 0)
      {
        ETG_TRACE_FATAL(("dp_tclDpUndef::s32SetShmElement() munmap() fails"));
      }
    }
  }
  return(Vs32Size);
}
/******************************************************************************
* tS32 dp_tclDpUndef::s32GetShmElement(void)
*
* DESCRIPTION: get shared element
*
* PARAMETERS: 
*
* RETURNS: size
*****************************************************************************/
tS32    dp_tclDpUndef::s32GetShmElement(const tChar* PpstrPoolName,const tChar* PpstrElementName,tU32 Pu32ElementSize,tVoid* PpvData)
{
  tS32          Vs32Size=0;
  tS32          Vs32ShmId;
  void*         VpAdrShmElement=NULL;
  tChar         VstrPoolElementName[DP_UNDEF_ELEMENT_NAME_SHARED_MAX_LEN];

  ETG_TRACE_USR4(("dp_tclDpUndef::s32GetShmElement() start"));
  memset(&VstrPoolElementName,0,sizeof(VstrPoolElementName));
  snprintf(VstrPoolElementName,DP_UNDEF_ELEMENT_NAME_SHARED_MAX_LEN,"%s_%s",PpstrPoolName,PpstrElementName);
  /*open shm element*/
  Vs32ShmId=shm_open(VstrPoolElementName, O_RDWR ,0);
  if(Vs32ShmId==-1) 
  {
    ETG_TRACE_FATAL(("dp_tclDpUndef::s32GetShmElement() shm_open() fails"));
  }
  else
  {
    VpAdrShmElement = mmap( NULL,Pu32ElementSize,PROT_READ | PROT_WRITE,MAP_SHARED,Vs32ShmId,0);       
    /* if no error */
    if(VpAdrShmElement == MAP_FAILED)
	  {
	    ETG_TRACE_FATAL(("dp_tclDpUndef::s32GetShmElement() mmap() fails"));
	  }
	  else
    {/*copy data*/
	    Vs32Size=Pu32ElementSize;
      memmove(PpvData,VpAdrShmElement,Vs32Size);
	    ETG_TRACE_USR4(("dp_tclDpUndef::s32GetShmElement() success"));
	    /*munmap*/
      if(munmap(VpAdrShmElement,Vs32Size) != 0)
      {
        ETG_TRACE_FATAL(("dp_tclDpUndef::s32GetShmElement() munmap() fails"));
      }
	    /*unlink*/
	    if(shm_unlink(VstrPoolElementName)!=0)
	    {
	      int VerrSv = errno;
	      ETG_TRACE_FATAL(("dp_tclDpUndef::s32GetShmElement() shm_unlink() fails %d",VerrSv));
	    }
    }
  }
  return(Vs32Size);
}
/******************************************************************************
* void dp_tclDpUndef::s32DecAccessCountAndGetDataShmElement()
*
* DESCRIPTION: 
*
* PARAMETERS: decrement access counter and get data from shared memory if access counter ==0
*
* RETURNS: size
*****************************************************************************/
tS32   dp_tclDpUndef::s32DecAccessCountAndGetDataShmElement(const tChar* PpstrPoolName,const tChar* PpstrElementName,tU8 **Pu8Data)
{
  tS32          Vs32Size=DP_S32_ERR_UNKNOWN;
  tS32          Vs32Hash;
  tDpShmElement VtSharedElement;
      
  ETG_TRACE_USR4(("dp_tclDpUndef::bDecAccessCountAndGetDataShmElement() start"));
  memset(&VtSharedElement,0,sizeof(tDpShmElement));
  snprintf(VtSharedElement.strPoolElementName,DP_UNDEF_ELEMENT_NAME_SHARED_MAX_LEN,"%s_%s",PpstrPoolName,PpstrElementName);
  Vs32Hash=s32CalcHash(VtSharedElement.strPoolElementName);
  //check list if exist
  tDpUndefSharedElement::iterator VtDpPos = _tListSharedUndefElements.find(Vs32Hash);
  if (VtDpPos == _tListSharedUndefElements.end()) 
  {
     ETG_TRACE_ERR(("dp_tclDpUndef::bDecAccessCountAndGetDataShmElement() element '%s' not exist in list",VtSharedElement.strPoolElementName));
  }
  else
  { /*decrement access*/
    if(VtDpPos->second.u32ProcessAccess>0)
	  {
	    VtDpPos->second.u32ProcessAccess--;
	  }
	  if(VtDpPos->second.u32ProcessAccess==0)
	  {/*copy data to ram */
        tU8* Vpu8Buffer;
        try
        {
            Vpu8Buffer = new tU8[(tU32)VtDpPos->second.u32SizeShmElement];
        }
        catch(const std::bad_alloc&)
        {
            Vpu8Buffer = NULL;
            ETG_TRACE_FATAL(("dp_tclDpUndef::bDecAccessCountAndGetDataShmElement() Unable to allocate %d",(tU32)VtDpPos->second.u32SizeShmElement));
        }
	    if(Vpu8Buffer!=NULL)
        {
	       memmove(Vpu8Buffer,VtDpPos->second.vpAdrShmElement,VtDpPos->second.u32SizeShmElement);
	       Vs32Size=(tS32)VtDpPos->second.u32SizeShmElement;
		   *Pu8Data = Vpu8Buffer;	  
	    }
	    else
	    {
            FATAL_M_ASSERT_ALWAYS();
	    }
    }
	  else
	  {
	    ETG_TRACE_FATAL(("dp_tclDpUndef::bDecAccessCountAndGetDataShmElement() access count %d",VtDpPos->second.u32ProcessAccess));
	  }
  }
  return(Vs32Size);
}

/******************************************************************************
* void dp_tclDpUndef::bDeleteShmElement()
*
* DESCRIPTION: delete shared element
*
* PARAMETERS: 
*
* RETURNS: True: success; False: error
*****************************************************************************/
tBool   dp_tclDpUndef::bDeleteShmElement(const tChar* PpstrPoolName,const tChar* PpstrElementName)
{
  tBool         VbSuccess=FALSE;
  tS32          Vs32Hash;
  tDpShmElement VtSharedElement;

  ETG_TRACE_USR4(("dp_tclDpUndef::bDeleteShmElement() start"));
  memset(&VtSharedElement,0,sizeof(tDpShmElement));
  snprintf(VtSharedElement.strPoolElementName,DP_UNDEF_ELEMENT_NAME_SHARED_MAX_LEN,"%s_%s",PpstrPoolName,PpstrElementName);
  Vs32Hash=s32CalcHash(VtSharedElement.strPoolElementName);
  //check list if exist
   tDpUndefSharedElement::iterator VtDpPos = _tListSharedUndefElements.find(Vs32Hash);
   if (VtDpPos == _tListSharedUndefElements.end()) 
   {
     ETG_TRACE_ERR(("dp_tclDpUndef::bDeleteShmElement() element '%s' not exist in list",VtSharedElement.strPoolElementName));
   }
   else
   { 
	 if(VtDpPos->second.u32ProcessAccess!=0)
	 {
	  ETG_TRACE_ERR(("dp_tclDpUndef::bDeleteShmElement() su32ProcessAccess counter %d !=0",VtDpPos->second.u32ProcessAccess));
	 }
	 else
	 {/*munmap*/
      if(munmap(VtDpPos->second.vpAdrShmElement, VtDpPos->second.u32SizeShmElement) != 0)
      {
        ETG_TRACE_ERR(("dp_tclDpUndef::bDeleteShmElement() munmap() fails"));
      }
	    else
	    { /*unlink*/
	      if(shm_unlink(VtDpPos->second.strPoolElementName)!=0)
	      {
	        ETG_TRACE_ERR(("dp_tclDpUndef::bDeleteShmElement() shm_unlink() fails"));
	      }
	      /*clear from list*/
        _tListSharedUndefElements.erase(VtDpPos);
	      /*check again if erase done */
        VtDpPos = _tListSharedUndefElements.find(Vs32Hash);
        if (VtDpPos == _tListSharedUndefElements.end()) 
        {
            ETG_TRACE_ERR(("dp_tclDpUndef::bDeleteShmElement() erase from list fails"));
        }
	      else
	      {
	        VbSuccess=TRUE;
	      }
      }
    }
  }
  return(VbSuccess);
}
/******************************************************************************
* void dp_tclDpUndef::s32LockSemShmElement()
*
* DESCRIPTION: lock semaphore for shared elements
*
* PARAMETERS: 
*
* RETURNS: void
*****************************************************************************/
void   dp_tclDpUndef::vLockSemShmElement(void)
{ 
  dp_tclDpSem VmyDpSem;

  DP_NULL_POINTER_CHECK_FATAL_ASSERT(_pvShmProperty);
  VmyDpSem.s32LockSemShm(&ptGetPointerShmManage()->semShmElements);
}
/******************************************************************************
* void dp_tclDpUndef::s32UnLockSemShmElement(void)
*
* DESCRIPTION: unlock semaphore for shared elements
*
* PARAMETERS: 
*
* RETURNS: void
*****************************************************************************/
void   dp_tclDpUndef::vUnLockSemShmElement(void)
{
  dp_tclDpSem VmyDpSem;

  DP_NULL_POINTER_CHECK_FATAL_ASSERT(_pvShmProperty);
  VmyDpSem.s32UnLockSemShm(&ptGetPointerShmManage()->semShmElements);
}

/******************************************************************************
* tS32 dp_tclDpUndef::s32LoadPoolProperties(void)
*
* DESCRIPTION: load pool properties to shared memory
*
* PARAMETERS:  Void
*
* RETURNS: success oor error code
*****************************************************************************/
tS32 dp_tclDpUndef::s32LoadPoolProperties() 
{ 
  tS32          Vs32ReturnCode=DP_S32_NO_ERR;
  tU8*          Vpu8Buffer = NULL;
  tChar         VstrName[DP_MAX_LEN_FILE] = { 0 };
  tS32          Vs32ReadBytes;

  if(_poPersMemAccessHdl==NULL)
  {
    Vs32ReturnCode=DP_S32_ERR_UNKNOWN;
    NORMAL_M_ASSERT_ALWAYS();
  }
  else
  { /* 10.11.2014: bta2hi: load properties from file call in function s32GetPoolProperty() , 
    for preparing datapool to read early configuration data from NOR without access to FS;
    s32SizeFile==0 indicates, that the properties has to be load*/ 
    if(ptGetPointerShmManage()->s32SizeFile==0)
    {
      DP_NULL_POINTER_CHECK_FATAL_ASSERT(_pvShmProperty);
      ETG_TRACE_USR4(("dp_tclDpUndef::s32LoadPoolProperties() start"));

      /*import properties from File- Sytem  */	
      OSAL_szStringNCopy( (tString)VstrName, DP_UNDEF_FILE_NAME_POOL_PROPERTIES, DP_MAX_LEN_FILE);
      VstrName[DP_MAX_LEN_FILE-1]='\0';
      Vs32ReadBytes = _poPersMemAccessHdl->s32ImportPersData(VstrName,&Vpu8Buffer,eDpLocation_FILE_SYSTEM_SECURE,DP_UNDEF_VERSION_FILE_PROPERTIES);
      if (Vs32ReadBytes <= (tS32) DP_SIZE_HASH_TABLE)
      { 
        ETG_TRACE_USR4(("dp_tclDpUndef::s32LoadPoolProperties() no pool property stored"));
        Vs32ReturnCode=DP_S32_ERR_NO_POOL;
	      memset(ptGetPointerShmHashTable(),0,DP_SIZE_HASH_TABLE);
	      ptGetPointerShmManage()->s32SizeFile=(tS32) DP_SIZE_HASH_TABLE;
      }
      else
      { /*get pool properties and saved into shared memory*/      
        memmove(ptGetPointerShmHashTable(),Vpu8Buffer,Vs32ReadBytes);	  
	      /*set size of file*/
	      ptGetPointerShmManage()->s32SizeFile=Vs32ReadBytes;
	      /*trace*/
	      ETG_TRACE_USR4(("dp_tclDpUndef::s32LoadPoolProperties() get pool properties and saved to shared memory (size: %d)",Vs32ReadBytes));
	      /* prepare pool properties*/
	      vPreparePoolProperties();	  
      }
	    delete[] Vpu8Buffer;
        Vpu8Buffer=NULL; /*Bug 123144: Safe handling*/
	    //fprintf(stderr, "s32LoadPoolProperties: size:%d\n",ptGetPointerShmManage()->s32SizeFile);
	    ETG_TRACE_USR4(("dp_tclDpUndef::s32LoadPoolProperties() size file:%d\n",ptGetPointerShmManage()->s32SizeFile));
    }
  }
  return(Vs32ReturnCode);
}

/******************************************************************************
* void dp_tclDpUndef::s32GetHash(void)
*
* DESCRIPTION: get hash for the pool
*
* PARAMETERS:  PtsPoolName: pool name
*
* RETURNS: void  
*****************************************************************************/
tS32  dp_tclDpUndef::s32GetHash(const tChar* PtsPoolName)
{
  tS32             Vs32Result=DP_S32_ERR_NO_POOL;
  tU32             Vu32Offset;
  tU16             Vu16TableHashAddr=u16CalcHashPool(PtsPoolName);
  tDpShmHashTable* VptHashTable=ptGetPointerShmHashTable();
  tDpShmHashTable* VptHashTableCheck=VptHashTable+Vu16TableHashAddr;

  /*if hashvalue use or invalid = check name  => !=FREE*/
  //fprintf(stderr, "s32GetHash: Pool %s calc hashvalue %d\n",PtsPoolName,Vu16TableHashAddr);
  if(VptHashTableCheck->u8State!=DP_HASH_FREE)
  {
    tU16  Vu16TableHashAddrStart=Vu16TableHashAddr;
    do
	  {  
	     if(VptHashTableCheck->u8State==DP_HASH_USE)
	     { /* get offset */
           Vu32Offset=VptHashTableCheck->u32Offset;
	         /*get properties */
	         tDpUndefPoolProperty* VPoolProp=ptGetPointerShmPoolProperty(Vu32Offset);
	         if(VPoolProp!=NULL)
	         { /*compare name*/
	           if(strcmp(PtsPoolName,&VPoolProp->strPoolName[0])==0)
	           {
	             Vs32Result=Vu16TableHashAddr;
			        //fprintf(stderr, "s32GetHash: Pool %s hash address %d\n",PtsPoolName,Vs32Result);
             }
           }
       }
	     Vu16TableHashAddr++;
	     if(Vu16TableHashAddr==DP_MAX_NUMBER_POOLS_UNDEF_ELEMENT)
	     {
	       Vu16TableHashAddr=0;
	     }
	     //get new VptHashTableCheck 
	     VptHashTableCheck=VptHashTable+Vu16TableHashAddr;
       /*search until success or ptHashTableCheck->u8State!=DP_HASH_FREE or Vu16TableHashAddrStart reached*/ 
    }while((Vs32Result<0)&&(VptHashTableCheck->u8State!=DP_HASH_FREE)&&(Vu16TableHashAddrStart!=Vu16TableHashAddr));
  }/*end if*/
  return(Vs32Result);
}
/******************************************************************************
* void dp_tclDpUndef::s32InsertHash()
*
* DESCRIPTION: 
*
* PARAMETERS:  
*
* RETURNS: void  
*****************************************************************************/
tS32  dp_tclDpUndef::s32InsertHash(const tChar* PtsPoolName,tU32 Pu32Offset)
{
  tS32             Vs32Result=DP_S32_ERR_NO_POOL;
  tU16             Vu16TableHashAddr=u16CalcHashPool(PtsPoolName);
  tU16             Vu16TableHashAddrStart=Vu16TableHashAddr;
  tDpShmHashTable* VptHashTable=ptGetPointerShmHashTable();
  tDpShmHashTable* VptHashTableCheck;

  do
  { 
    VptHashTableCheck= VptHashTable+Vu16TableHashAddr;
    if((VptHashTableCheck->u8State==DP_HASH_FREE)||(VptHashTableCheck->u8State==DP_HASH_INVALID))
	  {
	    Vs32Result=Vu16TableHashAddr;
	    VptHashTableCheck->u8State=DP_HASH_USE;
	    VptHashTableCheck->u32Offset=Pu32Offset;
	  }
	  Vu16TableHashAddr++;
	  if(Vu16TableHashAddr==DP_MAX_NUMBER_POOLS_UNDEF_ELEMENT)
	  {
	    Vu16TableHashAddr=0;
    }
  }while((Vs32Result<0)&&(Vu16TableHashAddrStart!=Vu16TableHashAddr));
  if(Vu16TableHashAddr==Vu16TableHashAddrStart)
  { // table for hash value full
    FATAL_M_ASSERT_ALWAYS();		
  }
  return(Vs32Result);
}
/******************************************************************************
* void dp_tclDpUndef::u16CalcHashPool()
*
* DESCRIPTION: 
*
* PARAMETERS:  
*
* RETURNS: void  
*****************************************************************************/
tU16 dp_tclDpUndef::u16CalcHashPool(const tChar* PpcString) 
{
   unsigned int  VuiHashAdr;
   const unsigned char *VpcPointer;

   VuiHashAdr = 0;
   VpcPointer = (const unsigned char *) PpcString;
   while(*VpcPointer != '\0') 
   {
      VuiHashAdr = DP_HASH_M * VuiHashAdr + *VpcPointer;
      VpcPointer++;
   }
   return((tU16)VuiHashAdr % DP_MAX_NUMBER_POOLS_UNDEF_ELEMENT);  
}
// *****************************************************************************
// ********   methods for get pointer in shm  
// *****************************************************************************
/******************************************************************************
* tDpUndefPoolProperty* dp_tclDpUndef::ptGetPointerShmPoolProperty(tU32 Pu32Offset)
*
* DESCRIPTION: get pointer from shared memory
*
* PARAMETERS:  Pu32Offset: offset get from hash table
*
* RETURNS: pointer pool properties; NULL if not available 
*****************************************************************************/
tDpUndefPoolProperty* dp_tclDpUndef::ptGetPointerShmPoolProperty(tU32 Pu32Offset)
{
  tDpUndefPoolProperty* VpReturn=NULL;
  tU32                  Vu32OffsetInShm=Pu32Offset+_u32OffsetPool;
  
  if(_pvShmProperty==NULL)
  {
    NORMAL_M_ASSERT_ALWAYS();
  }
  else
  { /*check lenght of shared memory with lenght */  
    if((Vu32OffsetInShm+sizeof(tDpUndefPoolProperty)) <= _u32ShmLength)
    {
      VpReturn=(tDpUndefPoolProperty*) (void*)((tU8*)_pvShmProperty+Vu32OffsetInShm);	
    }
  }
  return(VpReturn);
}

/******************************************************************************
* tU32  dp_tclDpUndef::u32PushBackProperty(tDpUndefPoolProperty* PpPoolProperty)
*
* DESCRIPTION: 
*
* PARAMETERS:  
*
* RETURNS: 
*****************************************************************************/
tU32  dp_tclDpUndef::u32PushBackProperty(tDpUndefPoolProperty* PpPoolProperty)
{
  tU32   Vu32Offset=DP_INVALID_OFFSET;
  tU32   Vu32SizeFile=(tU32)ptGetPointerShmManage()->s32SizeFile;

  if(Vu32SizeFile+sizeof(tDpShmManage)+sizeof(tDpUndefPoolProperty)<_u32ShmLength)
  {
    Vu32Offset=Vu32SizeFile-DP_SIZE_HASH_TABLE;//without sizeof(tDpShmManage)
    tDpUndefPoolProperty* VtsProperty=ptGetPointerShmPoolProperty(Vu32Offset);
	  if(VtsProperty!=NULL)
	  {
        PpPoolProperty->bValid=TRUE;
        memmove(VtsProperty,PpPoolProperty,sizeof(tDpUndefPoolProperty));
	    ptGetPointerShmManage()->s32SizeFile+=sizeof(tDpUndefPoolProperty);
	  }
  }  
  return(Vu32Offset);
}
/******************************************************************************
* tS32 dp_tclDpUndef::s32SavePoolProperties(void)
*
* DESCRIPTION: save pool properties from shared memory to file
*
* PARAMETERS:  Void
*
* RETURNS: success oor error code
*****************************************************************************/
tS32 dp_tclDpUndef::s32SavePoolProperties() 
{
  tS32          Vs32ReturnCode=DP_S32_NO_ERR;
  tChar         VstrName[DP_MAX_LEN_FILE] = { 0 };

  /*save properties to file */
  if(_poPersMemAccessHdl==NULL)
  {
    Vs32ReturnCode=DP_S32_ERR_UNKNOWN;
    NORMAL_M_ASSERT_ALWAYS();
  }
  else
  {
    OSAL_szStringNCopy( (tString)VstrName, DP_UNDEF_FILE_NAME_POOL_PROPERTIES,DP_MAX_LEN_FILE);
    VstrName[DP_MAX_LEN_FILE-1]='\0';
    if(ptGetPointerShmManage()->s32SizeFile > ((tS32)DP_SIZE_HASH_TABLE))
    {
      if(_poPersMemAccessHdl->bWriteFfsData(VstrName,(tU8*)ptGetPointerShmHashTable(),ptGetPointerShmManage()->s32SizeFile,
	                                        eDpLocation_FILE_SYSTEM_SECURE,DP_UNDEF_VERSION_FILE_PROPERTIES,TRUE)==FALSE)
      { 
        Vs32ReturnCode=DP_S32_ERR_UNKNOWN;
	      ETG_TRACE_ERR(("s32SavePoolProperties(): error -> pool properties not saved to file '%s'.", VstrName));
      }
    }
    else
    {
      Vs32ReturnCode=DP_S32_ERR_NO_POOL;
    }
  }
  return(Vs32ReturnCode);
}
/******************************************************************************
* tS32 dp_tclDpUndef::u32GetFreePoolId()
*
* DESCRIPTION: save pool properties from shared memory to file
*
* PARAMETERS:  Void
*
* RETURNS: success or error code
*****************************************************************************/
tU32 dp_tclDpUndef::u32GetFreePoolId()
{
  tU32                   Vu32PoolId=(tU32) CCA_C_U16_APP_UNDEFINED_BEGIN;
  tS32                   Vs32Size=(tS32)DP_SIZE_HASH_TABLE;
  TListPoolId            VtListPoolId;
  TListPoolId::iterator  VtDpPos;
  tBool                  VbFound=FALSE;
  tU32                   Vu32PoolIdList; 

  /*first fill list with poolid*/
  tDpUndefPoolProperty* VtpShmPoolProperty=ptGetPointerShmPoolProperty(0);
  if(VtpShmPoolProperty!=NULL)
  {
    VtListPoolId.clear();
    while(Vs32Size<ptGetPointerShmManage()->s32SizeFile)
    {
      if((VtpShmPoolProperty->bValid==TRUE)&&(VtpShmPoolProperty->u32PoolId>=Vu32PoolId))
	    {
          VtListPoolId.push_back(VtpShmPoolProperty->u32PoolId>>16);
	    }
	    VtpShmPoolProperty++;
	    Vs32Size+=sizeof(tDpUndefPoolProperty);
    };
    /*sort list*/
    VtListPoolId.sort();
    /*get free id*/
    for (VtDpPos=VtListPoolId.begin(); VtDpPos!=VtListPoolId.end() && !VbFound; ++VtDpPos)
    {
      Vu32PoolIdList=*VtDpPos; 
      if(Vu32PoolId < Vu32PoolIdList)
	    {
	      VbFound=TRUE;
	    }
	    else
	    {
	      Vu32PoolId=*VtDpPos;
	  	  Vu32PoolId++;
	    }
    }  
    /*if the end of the list*/
    if((VbFound==FALSE)&&(VtListPoolId.size()!=0))
    { /*increment id*/
      Vu32PoolId++;
	    /*assert if id reach the max*/
	    if (Vu32PoolId>=CCA_C_U16_APP_UNDEFINED_END)
	    {
	      FATAL_M_ASSERT_ALWAYS();		
	    }
    }
  }
  return(Vu32PoolId);
}
/******************************************************************************
* tS32 dp_tclDpUndef::vPreparePoolProperties()
*
* DESCRIPTION: prepare pool properties in shared memory
*              clear name process located after load properties
*              remove invalid properties
*
* PARAMETERS:  Void
*
* RETURNS: success or error code
*****************************************************************************/
void dp_tclDpUndef::vPreparePoolProperties(void)
{
  tDpUndefPoolProperty*  VptPoolProp=ptGetPointerShmPoolProperty(0);
  tU32                   Vu32Size=_u32OffsetPool;
  tU32                   Vu32SizeShm=sizeof(tDpShmManage)+ptGetPointerShmManage()->s32SizeFile;
  tDpUndefPoolProperty*  VptPoolPropNext;
  tBool                  VbChange=FALSE;
  tU32                   Vu32CopySize;
  tS32                   Vs32HashValue;

  if(VptPoolProp!=NULL)
  {
    do
	  { //get hash from poolname 
	    Vs32HashValue=s32GetHash(VptPoolProp->strPoolName);
      if(Vs32HashValue<0)
	    { //check state
	      if(VptPoolProp->bValid!=FALSE)
		    {//error trace pool found, for which no hash value found
	        ETG_TRACE_FATAL(("dp_tclDpUndef::vPreparePoolProperties() for pool '%s' no hash value found \n",VptPoolProp->strPoolName));
		      //fprintf(stderr, "dp_tclDpUndef::vPreparePoolProperties() for pool '%s' no hash value found \n",VptPoolProp->strPoolName);
		    }
		    /* delete pool property*/
        VptPoolPropNext=VptPoolProp;
        VptPoolPropNext++;		
        Vu32CopySize=Vu32SizeShm-Vu32Size-sizeof(tDpUndefPoolProperty);		
		    if (Vu32CopySize!=0)
		    {
		      memmove(VptPoolProp,VptPoolPropNext,Vu32CopySize);		 
		    }
        Vu32SizeShm-=sizeof(tDpUndefPoolProperty);
		    VbChange=TRUE;
	    }
	    else
	    { //save new offset in hash table
	      tDpShmHashTable*       VptHashTable=ptGetPointerShmHashTable()+Vs32HashValue;
		    VptHashTable->u32Offset=Vu32Size-_u32OffsetPool;
	      //delete strProcessNameLocated 
	      memset(VptPoolProp->strProcessNameLocated,0,DP_MAX_LEN_NAME_PROCESS);
		    VptPoolProp++;
	      Vu32Size+=sizeof(tDpUndefPoolProperty);
	    }     
    }while(Vu32Size < (tU32) Vu32SizeShm);
	  /*check if pool properties change*/
	  if(VbChange==TRUE)
	  {
	    ptGetPointerShmManage()->s32SizeFile=Vu32SizeShm-sizeof(tDpShmManage);
	    s32SavePoolProperties();	   
	  }
  }
}
/******************************************************************************
* tS32 dp_tclDpUndef::bCheckElementShouldBeShared()
*
* DESCRIPTION: 
*
* PARAMETERS:
*
* RETURNS: 
*****************************************************************************/
tBool  dp_tclDpUndef::bCheckElementShouldBeShared(tDpUndefPoolProperty* PtPoolProperty,
										  EDpUndefCall PeFunctionCall,
										  dp_tclBaseElement* PpDpElement)
{
  #ifdef DP_DATAPOOL_UNIT_TEST
  tBool                     VbElementShared=TRUE; // => only for test set to TRUE, because read access from file not possible
  #else
  tBool                     VbElementShared=FALSE;
  #endif
  tDpUndefElementProperty*  VptDpUndefElementProperty = NULL;
  /*read not default properties for this pool */
  tS32 Vs32SizeRead=s32GetElementProperties(PtPoolProperty->u32PoolId,PtPoolProperty->eLocation,&VptDpUndefElementProperty);
  if(Vs32SizeRead>0)
  { /* init all elements which not default */
    tDpUndefElementProperty* VtPosElementPropertiesPos = VptDpUndefElementProperty;
	  tU32                     Vu32Inc;
	  tU32                     Vu32MaxNumber=(((tU32)Vs32SizeRead)/sizeof(tDpUndefElementProperty));
    for (Vu32Inc=0;Vu32Inc < Vu32MaxNumber;Vu32Inc++)
    { 
	    if(strncmp(VtPosElementPropertiesPos->strElementName,PpDpElement->pcGetElementName(),DP_MAX_LEN_NAME_ELEMENT)==0)
	    {
	      if((PeFunctionCall==eDpUndefSet)&&((VtPosElementPropertiesPos->u16AccessType&0x0200)==0x0200))
		    {
		      VbElementShared=TRUE;
		    }
		    if((PeFunctionCall==eDpUndefGet)&&((VtPosElementPropertiesPos->u16AccessType&0x0400)==0x0400))
		    {
		      VbElementShared=TRUE;
		    }
		    if((PeFunctionCall==eDpUndefDelete)&&((VtPosElementPropertiesPos->u16AccessType&0x0400)==0x0400))
		    {
		      VbElementShared=TRUE;
		    }
	      break; //element found
	    }
	    /*increment pointer*/
	    VtPosElementPropertiesPos++;
    }/*end for*/
	  delete[] VptDpUndefElementProperty;
      VptDpUndefElementProperty=NULL; /*Bug 123144: Safe handling*/
  }
  return(VbElementShared);
}
/******************************************************************************
* tS32 dp_tclDpUndef::s32CalcHash()
*
* DESCRIPTION: 
*
* PARAMETERS:
*
* RETURNS: 
*****************************************************************************/
tS32  dp_tclDpUndef::s32CalcHash(tChar* str)
{
   tS32 Vs32Hash = 0;
   tS32 n = strlen(str);
   for (int i=0; i<n; i++) 
   {
      Vs32Hash = 5*Vs32Hash + str[i];
   }
   return Vs32Hash;
}
#endif

//EOF