/**
 * @copyright    (C) 2012 - 2016 Robert Bosch GmbH.
 *               The reproduction, distribution and utilization of this file as well as the
 *               communication of its contents to others without express authorization is prohibited.
 *               Offenders will be held liable for the payment of damages.
 *               All rights reserved in the event of the grant of a patent, utility model or design.
 * @brief        semaphore handling for the datapool
 * @addtogroup   Datapool semaphore
 * @{
 */
#include <fcntl.h>	
#include <unistd.h>
#include <sys/mman.h>
#include <sys/stat.h>
#include <errno.h>
#include <grp.h>

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

#define OSAL_S_IMPORT_INTERFACE_GENERIC		   
#include "osal_if.h"

#define DP_S_IMPORT_INTERFACE_DPSEM
#include "dp_if.h"
#include "pdd.h"

#define DP_SEM_TRACE_CLASS              (tU16)(TR_COMP_DATAPOOL + 0x07) 

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS DP_SEM_TRACE_CLASS
#include "trcGenProj/Header/dpSem.cpp.trc.h"
#endif

/******************************************************************************* 
|defines and macros 
|------------------------------------------------------------------------------*/


/******************************************************************************/
/* static  variable                                                           */
/******************************************************************************/


/******************************************************************************
* dp_tclDpSem::dp_tclDpSem() 
*
* DESCRIPTION: constructor   
*
* HISTORY: Created : 2014 12 16
*****************************************************************************/
dp_tclDpSem::dp_tclDpSem()
{
 
}
/******************************************************************************
* dp_tclDpSem::~dp_tclDpSem() 
*
* DESCRIPTION: destructor 
*
* HISTORY: Created : 2014 12 16
*****************************************************************************/
dp_tclDpSem::~dp_tclDpSem() 
{

}
/******************************************************************************
* void dp_tclDpSem::vCreateSemShm(void)
*
* DESCRIPTION: create semaphore for shared memory
*
* PARAMETERS:  PptSem: pointer semaphore
*
* RETURNS: void  
*
* HISTORY: Created : 2014 12 16
*****************************************************************************/
void  dp_tclDpSem::vCreateSemShm(sem_t* PptSem)
{
  if(sem_init(PptSem, 1/*shared*/, 1) < 0)
  {/* trace error*/        
	  ETG_TRACE_ERR(("dp_tclDpSem::vCreateSemShm() sem_init() fails"));
  }
  else
  {
    ETG_TRACE_USR4(("dp_tclDpSem::vCreateSemShm() success"));
  }
}
/******************************************************************************
* void dp_tclDpSem::vDestroySemShm(void)
*
* DESCRIPTION: destroy semaphore for shared memory
*
* PARAMETERS:  PptSem: pointer semaphore
*
* RETURNS: void  
*
* HISTORY: Created : 2014 12 16
*****************************************************************************/
void  dp_tclDpSem::vDestroySemShm(sem_t* PptSem)
{
  tS32   Vs32Value;
 
  if(sem_getvalue(PptSem,&Vs32Value)==0)
  { /* trace out value*/
    ETG_TRACE_USR4(("dp_tclDpSem::vDestroySemShm() value semaphore %d",Vs32Value));	
    /* if value=0;semaphore waits => no destroy !!!*/
    if(Vs32Value==0)
    {
       ETG_TRACE_ERR(("dp_tclDpSem::vDestroySemShm() semaphore waits"));
    }
    else
    {   /*destroy semaphore*/
        if(sem_destroy(PptSem) < 0)
        {/* trace error*/        
          ETG_TRACE_ERR(("dp_tclDpSem::vDestroySemShm() function sem_destroy() fails"));
        }
        else
        {
          ETG_TRACE_USR4(("dp_tclDpSem::vDestroySemShm() success"));
        }
    }
  }
  else
  {
	  ETG_TRACE_ERR(("dp_tclDpSem::vDestroySemShm() function sem_getvalue() fails"));
  }
}
/******************************************************************************
* tS32  dp_tclDpSem::s32LockSemShm(sem_t* PptSem)
*
* DESCRIPTION: lock semaphore for shared memory
*
* PARAMETERS:  PptSem: pointer semaphore
*
* RETURNS: void  
*
* HISTORY: Created : 2014 12 16
*****************************************************************************/
tS32  dp_tclDpSem::s32LockSemShm(sem_t* PptSem)
{
  tS32   Vs32Value;
  tS32   Vs32Result=DP_S32_NO_ERR;
  
  /* get value semaphore*/
  if(sem_getvalue(PptSem,&Vs32Value)==0)
  { /* trace out value*/
    ETG_TRACE_USR4(("dp_tclDpSem::s32LockSemShm() value semaphore %d",Vs32Value));	
    /* lock semaphore*/
    if(sem_wait(PptSem)==0)  
	  { /* no error */
	    ETG_TRACE_USR4(("dp_tclDpSem::s32LockSemShm() success"));	
    }/*end if*/
	  else
  	{
      Vs32Result=DP_S32_ERR_UNKNOWN; 
	    ETG_TRACE_ERR(("dp_tclDpSem::s32LockSemShm() function sem_wait() fails"));
	    FATAL_M_ASSERT_ALWAYS();
	  }
  }
  else
  {
    Vs32Result=DP_S32_ERR_UNKNOWN; 
    ETG_TRACE_ERR(("dp_tclDpSem::s32LockSemShm() function sem_getvalue() fails"));
    FATAL_M_ASSERT_ALWAYS();
  }
  return(Vs32Result);
}
/******************************************************************************
* tS32  dp_tclDpSem::s32UnLockSemShm(sem_t* PptSem)
*
* DESCRIPTION: unlock semaphore for shared memory
*
* PARAMETERS:  PptSem: pointer semaphore
*
* RETURNS: void  
*
* HISTORY: Created : 2014 12 16
*****************************************************************************/
tS32  dp_tclDpSem::s32UnLockSemShm(sem_t* PptSem)
{
  tS32   Vs32Value;
  tS32   Vs32Result=DP_S32_NO_ERR;

  /* get value semaphore*/
  if(sem_getvalue(PptSem,&Vs32Value)==0)
  { /* trace out value*/
    ETG_TRACE_USR4(("dp_tclDpSem::s32UnLockSemShm() value semaphore %d",Vs32Value));
	  /* post semaphore*/
    if(sem_post(PptSem)==0)  
	  { /* no error => get pointer*/
	    ETG_TRACE_USR4(("dp_tclDpSem::s32UnLockSemShm() success"));	
    }/*end if*/
	  else
	  {
      Vs32Result=DP_S32_ERR_UNKNOWN; 
	    ETG_TRACE_ERR(("dp_tclDpSem::s32UnLockSemShm() function sem_post() fails"));
	    FATAL_M_ASSERT_ALWAYS();
	  }
  }
  else
  {
    Vs32Result=DP_S32_ERR_UNKNOWN;  
    ETG_TRACE_ERR(("dp_tclDpSem::s32UnLockSemShm() function sem_getvalue() fails"));
    FATAL_M_ASSERT_ALWAYS();
  }
  return(Vs32Result);
}
/******************************************************************************
* tS32  dp_tclDpSem::s32GetAccessRight()
*
* DESCRIPTION: get access rights from PDD
*
* HISTORY: Created : 2014 12 16
*****************************************************************************/
tS32   dp_tclDpSem::s32GetAccessRight()
{
  return(PDD_ACCESS_RIGTHS);
}
/******************************************************************************
* FUNCTION: dp_tclDpSem::s32ChangeGroupAccess()
*
* DESCRIPTION: change group access to eco_pdd
*
* HISTORY:Created : 2014 12 16
*****************************************************************************/
tS32   dp_tclDpSem::s32ChangeGroupAccess(teDPKindRes PeKindRes,tS32 Ps32Id,const char* PStrResource)
{
  tS32   Vs32Result=DP_S32_NO_ERR;
  /*  The getgrnam() function returns a pointer to a structure containing the group id*/
  struct group *VtGroup=getgrnam(PDD_ACCESS_RIGTHS_GROUP_NAME);
  if(VtGroup==NULL)
  { 
    //Vs32Result=DP_S32_ERR_UNKNOWN;  
	  //ETG_TRACE_ERR(("dp_tclDpSem::s32ChangeGroupAccess() getgrnam()  fails"));
  }
  else
  { 
    if(PeKindRes==DP_KIND_RES_SEM)
	  {
	    if(chown(PStrResource, (uid_t)-1, VtGroup->gr_gid)==-1)
	    {	
		    Vs32Result=DP_S32_ERR_UNKNOWN;  	 	  
		    ETG_TRACE_ERR(("dp_tclDpSem::s32ChangeGroupAccess() chown() fails for resource %s",PStrResource));
	    }
	    else
	    {
        if(chmod(PStrResource, PDD_ACCESS_RIGTHS) == -1)
        {		
          Vs32Result=DP_S32_ERR_UNKNOWN;  
		      ETG_TRACE_ERR(("dp_tclDpSem::s32ChangeGroupAccess() fchmod() fails for resource %s",PStrResource));
        }
	    }
	  }
	  else
	  {/*set access right for vPdd_ShmId to group eco_pdd*/
	    if(fchown(Ps32Id, (uid_t) -1, VtGroup->gr_gid)!=0)
	    {	
        Vs32Result=DP_S32_ERR_UNKNOWN;  
	      ETG_TRACE_ERR(("dp_tclDpSem::s32ChangeGroupAccess() fchown() fails for resource %s",PStrResource));
	    }
	    else
	    {
	      if(fchmod(Ps32Id, PDD_ACCESS_RIGTHS) == -1)
        {
          Vs32Result=DP_S32_ERR_UNKNOWN;  
	        ETG_TRACE_ERR(("dp_tclDpSem::s32ChangeGroupAccess() fchmod() fails for resource %s",PStrResource));
        }
	    }
    }
  }
  return(Vs32Result);
}


//EOF